

program torques

! SL 12/2009:
! This program reads 4D (lon-lat-alt-time) fields directly from CAM outputs
! without regrid
!
! it computes:
! dmass -- 4D -- mass of each cell
! osam  -- 4D -- specific angular momentum (omega term)
! rsam  -- 4D -- specific angular momentum (zonal wind term)
! oaam  -- 1D -- integrated angular momentum (omega term)
! raam  -- 1D -- integrated angular momentum (zonal wind term)
! tmou  -- 2D -- Mountain torque fonction of lat
! tbls  -- 2D -- Surface friction torque IF duvdf is present 
!                                       or if simple friction
! tdyn  -- 2D -- Dynamics torque IF dudyn is present 
!
! Minimal requirements and dependencies:
! The dataset must include the following data:
! - surface pressure and surface geopotential
! - zonal wind
! Optional: dudyn, duvdf

implicit none

include "netcdf.inc" ! NetCDF definitions

character (len=128) :: infile ! input file name (name_P.nc)
character (len=128) :: outfile ! output file name

character (len=64) :: text ! to store some text
integer infid ! NetCDF input file ID
integer outfid ! NetCDF output file ID
integer lon_dimid,lat_dimid,alt_dimid,time_dimid ! NetCDF dimension IDs
integer lon_varid,lat_varid,alt_varid,time_varid, tmpvarid
integer              :: datashape1d ! shape of 1D datasets
integer,dimension(2) :: datashape2d ! shape of 2D datasets
integer,dimension(3) :: datashape3d ! shape of 3D datasets
integer,dimension(4) :: datashape4d ! shape of 4D datasets

real :: miss_val=9.99e+33 ! special "missing value" to specify missing data
real,parameter :: miss_val_def=9.99e+33 ! default value for "missing value"
real :: pi
real,dimension(:),allocatable :: lon ! longitude
integer lonlength ! # of grid points along longitude
real,dimension(:),allocatable :: lat ! latitude
real,dimension(:),allocatable :: latrad ! latitude in radian
integer latlength ! # of grid points along latitude
real,dimension(:),allocatable :: plev ! Pressure levels (Pa)
integer altlength ! # of grid point along presnivs (of input datasets)
real,dimension(:),allocatable :: time ! time
integer timelength ! # of points along time
real :: p0 ! GCM reference pressure
real,dimension(:,:,:),allocatable :: ps ! surface pressure
real,dimension(:,:,:),allocatable :: phis3d ! surface geopotential
real,dimension(:,:),allocatable :: phis ! surface geopotential
real,dimension(:),allocatable :: aps,bps ! hybrid vertical coordinates
real,dimension(:,:,:,:),allocatable :: press ! GCM atmospheric pressure 
real,dimension(:),allocatable :: pp ! Pressure levels (Pa)
real,dimension(:,:,:,:),allocatable :: temp ! atmospheric temperature
real,dimension(:,:,:,:),allocatable :: vitu ! zonal wind (in m/s)
real,dimension(:,:,:,:),allocatable :: vitv ! meridional wind (in m/s)
real,dimension(:,:,:,:),allocatable :: vitw ! vertical wind (in Pa/s, then converted in m/s)
real,dimension(:,:,:,:),allocatable :: duvdf  ! Friction in BL
real,dimension(:,:,:,:),allocatable :: dudyn  ! Dynamics 

real,dimension(:,:,:,:),allocatable :: grav ! gravity field (m s-2)
real,dimension(:,:,:,:),allocatable :: dmass ! mass in cell (kg)
real,dimension(:,:,:,:),allocatable :: osam ! planetary rotation specific ang (m2/s)
real,dimension(:,:,:,:),allocatable :: rsam ! zonal wind specific ang (m2/s)
real,dimension(:),allocatable :: oaam ! planetary rotation total ang (kg m2/s)
real,dimension(:),allocatable :: raam ! zonal wind total ang (kg m2/s)
real,dimension(:,:),allocatable :: tmou ! mountain torque (kg m2/s2)
real,dimension(:,:),allocatable :: tdyn ! dynamics torque (kg m2/s2)
real,dimension(:,:),allocatable :: tbls ! friction torque (kg m2/s2)

! for angular momentum budget normalisation
real,parameter :: hadley=1.e18
real,parameter :: hadday=1.e25

integer ierr,ierr1,ierr2 ! NetCDF routines return codes
integer i,j,ilon,ilat,ilev,itim ! for loops
integer idlsurf ! for option ideal surface
logical :: flag_duvdf,flag_dudyn,lmdflag

real :: deltalat,deltalon ! lat and lon intervals in radians
real,dimension(:,:,:,:),allocatable :: deltap ! pressure thickness of each layer (Pa)
real :: tmpp ! temporary pressure
real :: dz ! altitude diff
real :: signe ! orientation of lon axis for mountain torque computation

include "planet.h"

!===============================================================================
! 1. Input parameters
!===============================================================================

pi = 2.*asin(1.)

write(*,*) ""
write(*,*) "You are working on the atmosphere of ",planet

!===============================================================================
! 1.1 Input file
!===============================================================================

write(*,*) ""
write(*,*) " Program valid for Venus CAM or LMD output files"
write(*,*) "Enter input file name:"

read(*,'(a128)') infile
write(*,*) ""

! open input file

ierr = NF_OPEN(infile,NF_NOWRITE,infid)
if (ierr.ne.NF_NOERR) then
   write(*,*) 'ERROR: Pb opening file ',trim(infile)
   stop ""
endif

!===============================================================================
! 1.2 Get grids in lon,lat,alt(pressure),time
!===============================================================================

call get_iddim(infid,lat_varid,latlength,lon_varid,lonlength,&
                     alt_varid,altlength,time_varid,timelength,lmdflag )

allocate(lon(lonlength))
ierr=NF_GET_VAR_REAL(infid,lon_varid,lon)
if (ierr.ne.NF_NOERR) stop "Error: Failed reading longitude"
if(lon(1).gt.lon(2)) then
  signe=-1.
else
  signe=1.
endif

allocate(lat(latlength))
ierr=NF_GET_VAR_REAL(infid,lat_varid,lat)
if (ierr.ne.NF_NOERR) stop "Error: Failed reading lat"

allocate(latrad(latlength))
latrad = lat*pi/180.

! Lat, lon pressure intervals
deltalat = abs(latrad(2)-latrad(1))
deltalon = abs(lon(2)-lon(1))*pi/180.

allocate(plev(altlength))
ierr=NF_GET_VAR_REAL(infid,alt_varid,plev)
if (ierr.ne.NF_NOERR) stop "Error: Failed reading pressure levels"
if(.not.lmdflag) then
! in CAM files, pressure in mbar and reversed...
  call reverselev(altlength,plev)
  plev=plev*100.  ! convertion to Pa
endif

allocate(time(timelength))
ierr=NF_GET_VAR_REAL(infid,time_varid,time)
if (ierr.ne.NF_NOERR) stop "Error: Failed reading time"

! Time axis IN PLANET DAYS

if(.not.lmdflag) then
! in CAM files, time in Earth days...
!   => seconds
  time=time*86400.
endif
time=time/localday

!===============================================================================
! 1.3 Get output file name
!===============================================================================
write(*,*) ""
!write(*,*) "Enter output file name"
!read(*,*) outfile
outfile=infile(1:len_trim(infile)-3)//"_LTRQ.nc"
write(*,*) "Output file name is: "//trim(outfile)



!===============================================================================
! 2.1 Store needed fields 
!===============================================================================

!===============================================================================
! 2.1.1 Surface pressure and geopotential
!===============================================================================
allocate(ps(lonlength,latlength,timelength))
allocate(phis3d(lonlength,latlength,timelength))
allocate(phis(lonlength,latlength))

text="PS"
call get_var3d(infid,lonlength,latlength,timelength,text,ps,ierr1,ierr2)
if (ierr1.ne.NF_NOERR) then
  write(*,*) "  looking for psol instead... "
  text="psol"
  call get_var3d(infid,lonlength,latlength,timelength,text,ps,ierr1,ierr2)
  if (ierr1.ne.NF_NOERR) stop "Error: Failed to get psol ID"
endif
if (ierr2.ne.NF_NOERR) stop "Error: Failed reading surface pressure"
if((.not.lmdflag).and.(planet.eq."Venus")) call reverse3d(lonlength,latlength,timelength,ps)

text="PHIS"
call get_var3d(infid,lonlength,latlength,timelength,text,phis3d,ierr1,ierr2)
if (ierr1.ne.NF_NOERR) then
  write(*,*) " Failed to get PHIS ID (3d), looking for phis (2d) instead... "
  text="phis"
  call get_var2d(infid,lonlength,latlength,text,phis,ierr1,ierr2)
  if (ierr1.ne.NF_NOERR) stop "Error: Failed to get phis ID"
  if (ierr2.ne.NF_NOERR) stop "Error: Failed reading surface geopotential"
else
  if (ierr2.ne.NF_NOERR) stop "Error: Failed reading surface geopotential"
  phis(:,:)=phis3d(:,:,1)
  if((.not.lmdflag).and.(planet.eq."Venus")) call reverse2d(lonlength,latlength,phis)
endif

if(lmdflag) then
  p0=1.
else
  ierr=NF_INQ_VARID(infid,"P0",tmpvarid)
  if (ierr.ne.NF_NOERR) then
    write(*,*) "Failed to get P0 ID, used psref=",psref
    p0=psref
  else
    ierr=NF_GET_VAR_REAL(infid,tmpvarid,p0)
    if (ierr.ne.NF_NOERR) then
      write(*,*) "Failed reading reference pressure, used psref=",psref
      p0=psref
    endif
  endif
endif

!===============================================================================
! 2.1.3 Winds
!===============================================================================
allocate(vitu(lonlength,latlength,altlength,timelength))

! zonal wind vitu / U (in m/s)
if(lmdflag) then
  text="vitu"
else
  text="U"
endif

call get_var4d(infid,lonlength,latlength,altlength,timelength,text,vitu,ierr1,ierr2)
if (ierr1.ne.NF_NOERR) stop "Error: Failed to get U ID"
if (ierr2.ne.NF_NOERR) stop "Error: Failed reading zonal wind"

if(.not.lmdflag) call reverse4d(lonlength,latlength,altlength,timelength,vitu)

!===============================================================================
! 2.1.5 Friction in Boundary layer
!===============================================================================
allocate(duvdf(lonlength,latlength,altlength,timelength))

if(lmdflag) then
  text="duvdf"
else
  text="DUVDF"
endif
call get_var4d(infid,lonlength,latlength,altlength,timelength,text,duvdf,ierr1,ierr2)
if (ierr1.ne.NF_NOERR) then
  write(*,*) "Failed to get duvdf ID"
  flag_duvdf = .false.

! IDEAL FRICTION ?
write(*,*) ""
write(*,*) " Is the friction at surface ideal ? 0 for no, 1 for yes"
write(*,*) " duvdf = -u/(86400*30) in surface layer"
write(*,*) " timescale hard-coded: 30 Edays"
read(*,'(i1)') idlsurf
!write(*,*) ""
!write(*,*) " ASSUMED YES ! "
!idlsurf=1
write(*,*) ""

if (idlsurf.eq.1) then
  flag_duvdf = .true.
  duvdf = 0.
  ilev=1
do ilon=1,lonlength
 do ilat=1,latlength
  do itim=1,timelength
   duvdf(ilon,ilat,ilev,itim) = vitu(ilon,ilat,ilev,itim) * (-1.)/(86400.*30.)
  enddo
 enddo
enddo
endif

else
  if (ierr2.ne.NF_NOERR) stop "Error: Failed reading duvdf"
  if(.not.lmdflag) call reverse4d(lonlength,latlength,altlength,timelength,duvdf)
  flag_duvdf = .true.
endif

!===============================================================================
! 2.1.7 Dynamics (includes the mountain torque...)
!===============================================================================
allocate(dudyn(lonlength,latlength,altlength,timelength))

if(lmdflag) then
  text="dudyn"
else
  text="DUDYN"
endif
call get_var4d(infid,lonlength,latlength,altlength,timelength,text,dudyn,ierr1,ierr2)
if (ierr1.ne.NF_NOERR) then
  write(*,*) "Failed to get dudyn ID"
  flag_dudyn = .false.
else
  if (ierr2.ne.NF_NOERR) stop "Error: Failed reading dudyn"
  if(.not.lmdflag) call reverse4d(lonlength,latlength,altlength,timelength,dudyn)
  flag_dudyn = .true.
endif

!===============================================================================
! 2.2 Computations 
!===============================================================================

!===============================================================================
! 2.2.1 Pressure intervals
!===============================================================================
allocate(deltap(lonlength,latlength,altlength,timelength))
allocate(press(lonlength,latlength,altlength,timelength))
allocate(pp(altlength))

if(lmdflag) then  ! LMD vs CAM
  text="pres"
  call get_var4d(infid,lonlength,latlength,altlength,timelength,text,press,ierr1,ierr2)
  if (ierr1.ne.NF_NOERR) then
    stop "Error: Failed to get pres ID"
  else
    if (ierr2.ne.NF_NOERR) stop "Error: Failed reading pres"
  endif

else  ! LMD vs CAM
  ! hybrid coordinate aps/hyam
  text="hyam"
  ierr=NF_INQ_VARID(infid,text,tmpvarid)
  if (ierr.ne.NF_NOERR) then
    stop "Error: Could not find aps/hyam coordinates"
  else
    allocate(aps(altlength))
    ierr=NF_GET_VAR_REAL(infid,tmpvarid,aps)
    if (ierr.ne.NF_NOERR) then
      stop "Error: Failed reading aps/hyam"
    endif
    call reverselev(altlength,aps)
  endif

  ! hybrid coordinate hybm
  text="hybm"
  ierr=NF_INQ_VARID(infid,text,tmpvarid)
  if (ierr.ne.NF_NOERR) then
    stop "Error: Failed to get bps/hybm ID"
  else
    allocate(bps(altlength))
    ierr=NF_GET_VAR_REAL(infid,tmpvarid,bps)
    if (ierr.ne.NF_NOERR) then
      stop "Error: Failed reading bps/hybm"
    endif
    call reverselev(altlength,bps)
  endif

  aps=aps*p0

  do itim=1,timelength
    do ilev=1,altlength
      do ilat=1,latlength
        do ilon=1,lonlength
          press(ilon,ilat,ilev,itim)=aps(ilev)+bps(ilev)*ps(ilon,ilat,itim)
        enddo
      enddo
    enddo
  enddo

endif  ! LMD vs CAM

do itim=1,timelength

do ilon=1,lonlength
 do ilat=1,latlength
  tmpp=ps(ilon,ilat,itim)
  pp=press(ilon,ilat,:,itim)
  deltap(ilon,ilat,1,itim)= tmpp - exp((log(pp(1))+log(pp(2)))/2.) ! initialization, rectified with ps below
do ilev=2,altlength-1
  deltap(ilon,ilat,ilev,itim)= exp((log(pp(ilev-1))+log(pp(ilev)))/2.)-&
                     exp((log(pp(ilev))+log(pp(ilev+1)))/2.)
enddo
deltap(ilon,ilat,altlength,itim)=exp((log(pp(altlength-1))+log(pp(altlength)))/2.)
 enddo
enddo

enddo ! timelength

!===============================================================================
! 2.2.2 Mass in cells
!===============================================================================
allocate(grav(lonlength,latlength,altlength,timelength))
allocate(dmass(lonlength,latlength,altlength,timelength))

do itim=1,timelength

do ilon=1,lonlength
 do ilat=1,latlength
  do ilev=1,altlength
    if (deltap(ilon,ilat,ilev,itim).ne.miss_val) then
! Need to be consistent with GCM computations
      grav(ilon,ilat,ilev,itim) = g0
     dmass(ilon,ilat,ilev,itim) = &
               a0*cos(latrad(ilat))*deltalon &
             * a0*deltalat &
             * deltap(ilon,ilat,ilev,itim)/grav(ilon,ilat,ilev,itim)
    else
      grav(ilon,ilat,ilev,itim) = miss_val
     dmass(ilon,ilat,ilev,itim) = miss_val
    endif
  enddo
 enddo
enddo

enddo ! timelength

!===============================================================================
! 2.2.3 Specific angular momentum
!===============================================================================
allocate(osam(lonlength,latlength,altlength,timelength))
allocate(rsam(lonlength,latlength,altlength,timelength))

do itim=1,timelength

do ilon=1,lonlength
 do ilat=1,latlength
  do ilev=1,altlength
      osam(ilon,ilat,ilev,itim) = &
         a0*a0* cos(latrad(ilat))*cos(latrad(ilat)) * omega
      rsam(ilon,ilat,ilev,itim) = vitu(ilon,ilat,ilev,itim) &
       * a0*cos(latrad(ilat))
  enddo
 enddo
enddo

enddo ! timelength

!===============================================================================
! 2.2.4 Total angular momentum
!===============================================================================
allocate(oaam(timelength))
allocate(raam(timelength))

do itim=1,timelength

oaam(itim) = 0.
raam(itim) = 0.
do ilon=1,lonlength
 do ilat=1,latlength
  do ilev=1,altlength
      oaam(itim) = oaam(itim) &
       + osam(ilon,ilat,ilev,itim)/ hadday * dmass(ilon,ilat,ilev,itim)
      raam(itim) = raam(itim) &
       + rsam(ilon,ilat,ilev,itim)/ hadday * dmass(ilon,ilat,ilev,itim)
  enddo
 enddo
enddo
if (oaam(itim).eq.0.) then
  oaam(itim) = miss_val
  raam(itim) = miss_val
endif

enddo ! timelength

!===============================================================================
! 2.2.5 Mountain, friction, dynamics and GW torques
!===============================================================================
allocate(tmou(latlength,timelength))
if (flag_dudyn)  allocate(tdyn(latlength,timelength))
if (flag_duvdf)  allocate(tbls(latlength,timelength))

do itim=1,timelength

tmou(:,itim) = 0.
 do ilat=2,latlength-1
do ilon=1,lonlength
      if (ilon.eq.lonlength) then
         dz = (phis(   1,ilat)     -phis(ilon,ilat))/g0
       tmpp = (ps(     1,ilat,itim)  +ps(ilon,ilat,itim))/2.
      else
         dz = (phis(ilon+1,ilat)   -phis(ilon,ilat))/g0
       tmpp = (ps(ilon+1,ilat,itim)  +ps(ilon,ilat,itim))/2.
      endif
      tmou(ilat,itim) = tmou(ilat,itim) + a0*a0* deltalat*cos(latrad(ilat)) &
       * signe*dz*tmpp &
       / hadley
enddo
 enddo

if (flag_dudyn) then
tdyn(:,itim) = 0.
 do ilat=1,latlength
do ilon=1,lonlength
  do ilev=1,altlength
      tdyn(ilat,itim) = tdyn(ilat,itim) + dudyn(ilon,ilat,ilev,itim) &
       * a0*cos(latrad(ilat)) * dmass(ilon,ilat,ilev,itim) &
       / hadley
  enddo
enddo
  if (tdyn(ilat,itim).eq.0.) tdyn(ilat,itim) = miss_val
 enddo
endif

if (flag_duvdf) then
tbls(:,itim) = 0.
 do ilat=1,latlength
do ilon=1,lonlength
  do ilev=1,altlength
      tbls(ilat,itim) = tbls(ilat,itim) + duvdf(ilon,ilat,ilev,itim) &
       * a0*cos(latrad(ilat)) * dmass(ilon,ilat,ilev,itim) &
       / hadley
  enddo
enddo
  if (tbls(ilat,itim).eq.0.) tbls(ilat,itim) = miss_val
 enddo
endif

enddo ! timelength

print*,"End of computations"

!===============================================================================
! 3. Create output file 
!===============================================================================

! Create output file
ierr=NF_CREATE(outfile,NF_CLOBBER,outfid)
if (ierr.ne.NF_NOERR) then
  write(*,*)"Error: could not create file ",outfile
  stop
endif

!===============================================================================
! 3.1. Define and write dimensions
!===============================================================================

call write_dim(outfid,lonlength,latlength,altlength,timelength, &
    lon,lat,plev,time,lon_dimid,lat_dimid,alt_dimid,time_dimid)

!===============================================================================
! 3.2. Define and write variables
!===============================================================================

! 1D Variables

datashape1d=time_dimid
 
call write_var1d(outfid,datashape1d,timelength,&
                "oaam      ", "Total rotation ang  ","E25kgm2s-1",miss_val,&
                 oaam )

call write_var1d(outfid,datashape1d,timelength,&
                "raam      ", "Total wind ang      ","E25kgm2s-1",miss_val,&
                 raam )

! 2D variables

datashape2d(1)=lat_dimid
datashape2d(2)=time_dimid

call write_var2d(outfid,datashape2d,latlength,timelength,&
                "tmou      ", "Mountain torque     ","E18kgm2s-2",miss_val,&
                 tmou )

if (flag_dudyn) then
call write_var2d(outfid,datashape2d,latlength,timelength,&
                "tdyn      ", "Dynamics torque     ","E18kgm2s-2",miss_val,&
                 tdyn )
endif

if (flag_duvdf) then
call write_var2d(outfid,datashape2d,latlength,timelength,&
                "tbls      ", "Friction torque     ","E18kgm2s-2",miss_val,&
                 tbls )
endif

datashape2d(1)=lon_dimid
datashape2d(2)=lat_dimid

call write_var2d(outfid,datashape2d,lonlength,latlength,&
                "phis      ", "Surface geopot      ","m2 s-2    ",miss_val,&
                 phis )

! 3D variables

datashape3d(1)=lon_dimid
datashape3d(2)=lat_dimid
datashape3d(3)=time_dimid

call write_var3d(outfid,datashape3d,lonlength,latlength,timelength,&
                "ps        ", "Surface pressure    ","Pa        ",miss_val,&
                 ps )

datashape3d(1)=lat_dimid
datashape3d(2)=alt_dimid
datashape3d(3)=time_dimid

!!!! Close output file
ierr=NF_CLOSE(outfid)
if (ierr.ne.NF_NOERR) then
  write(*,*) 'Error, failed to close output file ',outfile
endif


end program

subroutine reverse4d(nlon,nlat,np,nt,newf)
!==============================================================================
! Purpose:
! reverse lat, lon and vertical axis on a 3D+time field
!==============================================================================
implicit none
!==============================================================================
! Arguments:
!==============================================================================
integer,intent(in) :: nlon ! longitude size
integer,intent(in) :: nlat ! latitude size
integer,intent(in) :: np   ! vertical size
integer,intent(in) :: nt   ! time size
real,intent(inout),dimension(nlon,nlat,np,nt) :: newf ! field to be reversed

!==============================================================================
! Local variables:
!==============================================================================
integer :: i,j,k,l
real,dimension(nlon,nlat,np,nt) :: tmp,tmp2 ! reversed fields
include "planet.h"

! Vertical axis:
    do k=1,np
          tmp(:,:,k,:)=newf(:,:,np+1-k,:)
    enddo
! horizontal dimensions
if(planet.eq."Venus") then
  do l=1,nt
      do i=1,nlat
        do j=1,nlon
          tmp2(j,i,:,l)=tmp(nlon+1-j,nlat+1-i,:,l)
        enddo
      enddo
  enddo
else
  tmp2=tmp
endif

newf=tmp2

end subroutine reverse4d

!===============================================================================

subroutine reverse2d(nlon,nlat,newf)
!==============================================================================
! Purpose:
! reverse lat and lon on a 2D field
!==============================================================================
implicit none
!==============================================================================
! Arguments:
!==============================================================================
integer,intent(in) :: nlon ! longitude size
integer,intent(in) :: nlat ! latitude size
real,intent(inout),dimension(nlon,nlat) :: newf ! field to be reversed

!==============================================================================
! Local variables:
!==============================================================================
integer :: i,j
real,dimension(nlon,nlat) :: tmp ! reversed field

      do i=1,nlat
        do j=1,nlon
          tmp(j,i)=newf(nlon+1-j,nlat+1-i)
        enddo
      enddo
newf=tmp

end subroutine reverse2d

!===============================================================================

subroutine reverse3d(nlon,nlat,nt,newf)
!==============================================================================
! Purpose:
! reverse lat and lon on a 2D+time field
!==============================================================================
implicit none
!==============================================================================
! Arguments:
!==============================================================================
integer,intent(in) :: nlon ! longitude size
integer,intent(in) :: nlat ! latitude size
integer,intent(in) :: nt ! time size
real,intent(inout),dimension(nlon,nlat,nt) :: newf ! field to be reversed

!==============================================================================
! Local variables:
!==============================================================================
integer :: i,j,l
real,dimension(nlon,nlat,nt) :: tmp ! reversed field

  do l=1,nt
      do i=1,nlat
        do j=1,nlon
          tmp(j,i,l)=newf(nlon+1-j,nlat+1-i,l)
        enddo
      enddo
  enddo
newf=tmp

end subroutine reverse3d

!===============================================================================

subroutine reverselev(np,newf)
!==============================================================================
! Purpose:
! reverse vertical pressure axis 
!==============================================================================
implicit none
!==============================================================================
! Arguments:
!==============================================================================
integer,intent(in) :: np   ! vertical size
real,intent(inout),dimension(np) :: newf ! field to be reversed

!==============================================================================
! Local variables:
!==============================================================================
integer :: k
real,dimension(np) :: tmp ! reversed field

    do k=1,np
          tmp(k)=newf(np+1-k)
    enddo
newf=tmp

end subroutine reverselev
