      SUBROUTINE updatereffrad(ngrid,nlayer,
     &                pplev,pplay,
     &                rice,nuice,
     &                reffrad,nueffrad)

       IMPLICIT NONE
c=======================================================================
c   subject:
c   --------
c   Subroutine designed to update the aerosol size distribution used by
c     the radiative transfer scheme. This size distribution is assumed
c     to be a log-normal distribution, with effective radius "reffrad" and
c     variance "nueffrad".
c   At firstcall, "rice" and "nuice" are not known, because
c     the H2O ice microphysical scheme is called after the radiative
c     transfer in physiq.F. That's why we call watercloud at firstcall
c     in physiq.F before calling updatereffrad. Of course, resulting
c     tendencies are not used to compute the water cycle.
c   See callradite.F for more information.
c
c   author:   
c   ------
c   J.-B. Madeleine (2009)
c
c=======================================================================
c
c    Declarations :
c    -------------
c
#include "dimensions.h"
#include "dimphys.h"
#include "callkeys.h"
#include "dimradmars.h"
#include "tracer.h"

c-----------------------------------------------------------------------
c     Inputs:
c     ------

      INTEGER :: ngrid,nlayer
c     Pressure between each layer (Pa)
      REAL pplev(ngrid,nlayer+1)
c     Pressure in the middle of each layer (Pa)
      REAL pplay(ngrid,nlayer)
c     Estimated ice crystal radius (m)
      REAL :: rice(ngridmx,nlayermx)
c     Estimated effective variance of the size distribution (n.u.)
      REAL :: nuice(ngridmx,nlayermx)

c     Outputs:
c     -------

c     Aerosol effective radius used for radiative transfer (meter)
      REAL :: reffrad(ngridmx,nlayermx,naerkind)
c     Aerosol effective variance used for radiative transfer (n.u.)
      REAL :: nueffrad(ngridmx,nlayermx,naerkind)

c     Local variables:
c     ---------------

      INTEGER :: ig,l                          ! 3D grid indices
      INTEGER :: iaer                          ! Aerosol index

c     Hyperbolic tangent function coefficients used to describe
c       dust size vertical profile (reff=f(pplay))
      REAL, PARAMETER :: dustcoefa = 0.10003736
      REAL, PARAMETER :: dustcoefb = 0.77463902
      REAL, PARAMETER :: dustcoefc = 3.1483479
      REAL, PARAMETER :: dustcoefd = 2.4331281

c     Local saved variables:
c     ---------------------

c==================================================================

      DO iaer = 1, naerkind ! Loop on aerosol kind
        aerkind: SELECT CASE (iaer)
c==================================================================
        CASE(1) aerkind                             ! Dust (iaer=1)
c==================================================================

c         Particule size distribution computed below will be used
c           only if there are more than one particule size in the
c           ASCII file which contains the scattering parameters
c           (see callradite.F for more information).
c         Dust particle size distribution is deduced from dust 
c           transport simulations. The resulting mean vertical size
c           distribution is fitted with a function of the form
c           f(x)= a+b*[1+tanh( (log(x)-c)/d )], f being the dust
c           particle radius and x the atmospheric pressure
c           in a given layer. Then it is used below to define a
c           "background" dust size distribution.
c         Effective variance is set to 0.3.

          DO l=1,nlayer
            DO ig=1,ngrid
              reffrad(ig,l,iaer)=(log(pplay(ig,l)) - dustcoefc) /
     &                           dustcoefd
              reffrad(ig,l,iaer)=max(( (1.E0 + 
     &                                 tanh(reffrad(ig,l,iaer))) * 
     &                                 dustcoefb + dustcoefa ) *
     &                                 1E-6,1.E-9)
              nueffrad(ig,l,iaer)=0.3
            ENDDO
          ENDDO

c==================================================================
        CASE(2) aerkind               ! Water ice crystals (iaer=2)
c==================================================================
          DO l=1,nlayer
            DO ig=1,ngrid
              reffrad(ig,l,iaer)=rice(ig,l)
              nueffrad(ig,l,iaer)=nuice(ig,l)
            ENDDO
          ENDDO
c==================================================================
        END SELECT aerkind
      ENDDO ! iaer (loop on aerosol kind)

      RETURN
      END 
