      SUBROUTINE initracer(qsurf,co2ice)

       IMPLICIT NONE
c=======================================================================
c   subject:
c   --------
c   Initialization related to tracer 
c   (transported dust, water, chemical species, ice...)
c
c   Name of the tracer
c
c   Test of dimension :
c   Initialize COMMON tracer in tracer.h, using tracer names provided
c   by the dynamics in "advtrac.h"
c
c   Old conventions: (not used any more)
c
c   If water=T : q(iq=nqmx) is the water mass mixing ratio
c     and q(iq=nqmx-1) is the ice mass mixing ratio

c   If there is transported dust, it uses iq=1 to iq=dustbin
c   If there is no transported dust : dustbin=0
c   If doubleq=T : q(iq=1) is the dust mass mixing ratio
c                  q(iq=2) is the dust number mixing ratio 

c   If (photochem.or.thermochem) there is "ncomp" chemical species (ncomp
c   is set in aeronomars/chimiedata.h) using the ncomp iq values starting at
c      iq=nqchem_min = dustbin+1   (nqchem_min is defined in inifis.F)
c   
c
c   author: F.Forget
c   ------
c    Modifs: Franck Montmessin, Sebastien Lebonnois (june 2003)
c            Ehouarn Millour (oct. 2008) identify tracers by their names
c=======================================================================


#include "dimensions.h"
#include "dimphys.h"
#include "comcstfi.h"
#include "callkeys.h"
#include "tracer.h"
#include "advtrac.h"
#include "comgeomfi.h"
#include "watercap.h"
#include "chimiedata.h"  

      real qsurf(ngridmx,nqmx)       ! tracer on surface (e.g.  kg.m-2)
      real co2ice(ngridmx)           ! co2 ice mass on surface (e.g.  kg.m-2)
      integer iq,ig,count
      real r0_lift , reff_lift
      logical :: oldnames ! =.true. if old tracer naming convention (q01,...)
      character(len=20) :: txt ! to store some text

c-----------------------------------------------------------------------
c  radius(nqmx)      ! aerosol particle radius (m)
c  rho_q(nqmx)       ! tracer densities (kg.m-3)
c  qext(nqmx)        ! Single Scat. Extinction coeff at 0.67 um
c  alpha_lift(nqmx)  ! saltation vertical flux/horiz flux ratio (m-1)
c  alpha_devil(nqmx) ! lifting coeeficient by dust devil
c  rho_dust          ! Mars dust density
c  rho_ice           ! Water ice density
c  doubleq           ! if method with mass (iq=1) and number(iq=2) mixing ratio
c  varian            ! Characteristic variance of log-normal distribution
c-----------------------------------------------------------------------

      integer :: nqchem_start

! Initialization: get tracer names from the dynamics and check if we are
!                 using 'old' tracer convention ('q01',q02',...)
!                 or new convention (full tracer names)
      ! check if tracers have 'old' names
      oldnames=.false.
      count=0
      do iq=1,nqmx
        txt=" "
        write(txt,'(a1,i2.2)') 'q',iq
        if (txt.eq.tnom(iq)) then
          count=count+1
        endif
      enddo ! of do iq=1,nqmx
      
      if (count.eq.nqmx) then
        write(*,*) "initracer: tracers seem to follow old naming ",
     &             "convention (q01,q02,...)"
        write(*,*) "   => will work for now ... "
        write(*,*) "      but you should run newstart to rename them"
        oldnames=.true.
      endif

      ! copy/set tracer names
      if (oldnames) then ! old names (derived from iq & option values)
        ! 1. dust:
        if (dustbin.ne.0) then ! transported dust
          do iq=1,dustbin
            txt=" "
            write(txt,'(a4,i2.2)') 'dust',iq
            noms(iq)=txt
            mmol(iq)=100.
          enddo
          ! special case if doubleq
          if (doubleq) then
            if (dustbin.ne.2) then
              write(*,*) 'initracer: error expected dustbin=2'
            else
              noms(1)='dust_mass'   ! dust mass mixing ratio
              noms(2)='dust_number' ! dust number mixing ratio
            endif
          endif
        endif
        ! 2. water & ice
        if (water) then
          noms(nqmx)='h2o_vap'
          mmol(nqmx)=18.
          noms(nqmx-1)='h2o_ice'
          mmol(nqmx-1)=18.
        endif
        ! 3. Chemistry
        if (photochem .or. callthermos) then
          if (doubleq) then
            nqchem_start=3
          else
            nqchem_start=dustbin+1
          end if
          do iq=nqchem_start,nqchem_start+ncomp-1
            noms(iq)=nomchem(iq-nqchem_start+1)
            mmol(iq)=mmolchem(iq-nqchem_start+1)
          enddo
        endif ! of if (photochem .or. callthermos)
        ! 4. Other tracers ????
        if ((dustbin.eq.0).and.(.not.water)) then
          noms(1)='co2'
          mmol(1)=44
          if (nqmx.eq.2) then
            noms(nqmx)='Ar_N2'
            mmol(nqmx)=30
          endif
        endif
      else
        ! copy tracer names from dynamics
        do iq=1,nqmx
          noms(iq)=tnom(iq)
        enddo
        ! mmol(:) array is initialized later (see below)
      endif ! of if (oldnames)

      ! special modification when using 'old' tracers:
      ! qsurf(nqmx) was h2o ice whereas q(nqmx) was water vapour
      ! and (if iceparty) q(nqmx-1) was null whereas q(nqmx-1) was water ice
      if (oldnames.and.water) then
        write(*,*)'initracer: moving surface water ice to index ',nqmx-1
        qsurf(1:ngridmx,nqmx-1)=qsurf(1:ngridmx,nqmx)
        qsurf(1:ngridmx,nqmx)=0
      endif 
      
c------------------------------------------------------------
c     Test Dimensions tracers
c------------------------------------------------------------

      if(photochem.or.thermochem) then
          if (water) then
              if ((dustbin+ncomp+2).ne.nqmx) then
                 print*,'initracer: tracer dimension problem:'
                 print*,"(dustbin+ncomp+2).ne.nqmx"
                 print*,"ncomp: ",ncomp
                 print*,"dustbin: ",dustbin
                 print*,"nqmx: ",nqmx
                 print*,'Change ncomp in chimiedata.h'
               endif
          else
              if ((dustbin+ncomp+1).ne.nqmx) then
                 print*,'initracer: tracer dimension problem:'
                 print*,"(dustbin+ncomp+1).ne.nqmx"
                 print*,"ncomp: ",ncomp
                 print*,"dustbin: ",dustbin
                 print*,"nqmx: ",nqmx
                 print*,'Change ncomp in chimiedata.h'
                 STOP
               endif
            endif
      endif

c------------------------------------------------------------
c         NAME and molar mass of the tracer 
c------------------------------------------------------------

   
! Identify tracers by their names: (and set corresponding values of mmol)
      ! 0. initialize tracer indexes to zero:
      do iq=1,nqmx
        igcm_dustbin(iq)=0
      enddo
      igcm_dust_mass=0
      igcm_dust_number=0
      igcm_h2o_vap=0
      igcm_h2o_ice=0
      igcm_co2=0
      igcm_co=0
      igcm_o=0
      igcm_o1d=0
      igcm_o2=0
      igcm_o3=0
      igcm_h=0
      igcm_h2=0
      igcm_oh=0
      igcm_ho2=0
      igcm_h2o2=0
      igcm_n2=0
      igcm_ar=0
      igcm_ar_n2=0

      ! 1. find dust tracers
      count=0
      if (dustbin.gt.0) then
        do iq=1,nqmx
          txt=" "
          write(txt,'(a4,i2.2)')'dust',count+1
          if (noms(iq).eq.txt) then
            count=count+1
            igcm_dustbin(count)=iq
            mmol(iq)=100.
          endif
        enddo !do iq=1,nqmx
      endif ! of if (dustbin.gt.0)
      if (doubleq) then
        do iq=1,nqmx
          if (noms(iq).eq."dust_mass") then
            igcm_dust_mass=iq
            count=count+1
          endif
          if (noms(iq).eq."dust_number") then
            igcm_dust_number=iq
            count=count+1
          endif
        enddo
      endif ! of if (doubleq)
      ! 2. find chemistry and water tracers
      do iq=1,nqmx
        if (noms(iq).eq."co2") then
          igcm_co2=iq
          mmol(igcm_co2)=44.
          count=count+1
        endif
        if (noms(iq).eq."co") then
          igcm_co=iq
          mmol(igcm_co)=28.
          count=count+1
        endif
        if (noms(iq).eq."o") then
          igcm_o=iq
          mmol(igcm_o)=16.
          count=count+1
        endif
        if (noms(iq).eq."o1d") then
          igcm_o1d=iq
          mmol(igcm_o1d)=16.
          count=count+1
        endif
        if (noms(iq).eq."o2") then
          igcm_o2=iq
          mmol(igcm_o2)=32.
          count=count+1
        endif
        if (noms(iq).eq."o3") then
          igcm_o3=iq
          mmol(igcm_o3)=48.
          count=count+1
        endif
        if (noms(iq).eq."h") then
          igcm_h=iq
          mmol(igcm_h)=1.
          count=count+1
        endif
        if (noms(iq).eq."h2") then
          igcm_h2=iq
          mmol(igcm_h2)=2.
          count=count+1
        endif
        if (noms(iq).eq."oh") then
          igcm_oh=iq
          mmol(igcm_oh)=17.
          count=count+1
        endif
        if (noms(iq).eq."ho2") then
          igcm_ho2=iq
          mmol(igcm_ho2)=33.
          count=count+1
        endif
        if (noms(iq).eq."h2o2") then
          igcm_h2o2=iq
          mmol(igcm_h2o2)=34.
          count=count+1
        endif
        if (noms(iq).eq."n2") then
          igcm_n2=iq
          mmol(igcm_n2)=28.
          count=count+1
        endif
        if (noms(iq).eq."ar") then
          igcm_ar=iq
          mmol(igcm_ar)=40.
          count=count+1
        endif
        if (noms(iq).eq."h2o_vap") then
          igcm_h2o_vap=iq
          mmol(igcm_h2o_vap)=18.
          count=count+1
        endif
        if (noms(iq).eq."h2o_ice") then
          igcm_h2o_ice=iq
          mmol(igcm_h2o_ice)=18.
          count=count+1
        endif
        ! Other stuff: e.g. for simulations using co2 + neutral gaz
        if (noms(iq).eq."Ar_N2") then
          igcm_ar_n2=iq
          mmol(igcm_ar_n2)=30.
          count=count+1
        endif
      enddo ! of do iq=1,nqmx
!      count=count+nbqchem
      
      ! check that we identified all tracers:
      if (count.ne.nqmx) then
        write(*,*) "initracer: found only ",count," tracers"
        write(*,*) "               expected ",nqmx
        do iq=1,count
          write(*,*)'      ',iq,' ',trim(noms(iq))
        enddo
        stop
      else
        write(*,*) "initracer: found all expected tracers, namely:"
        do iq=1,nqmx
          write(*,*)'      ',iq,' ',trim(noms(iq))
        enddo
      endif

      ! if water cycle but iceparty=.false., there will nevertheless be
      ! water ice at the surface (iceparty is not used anymore, but this
      ! part is still relevant, as we want to stay compatible with the
      ! older versions).
      if (water.and.(igcm_h2o_ice.eq.0)) then
        igcm_h2o_ice=igcm_h2o_vap ! so that qsurf(i_h2o_ice) is identified
                                  ! even though there is no q(i_h2o_ice)
      else
       ! surface ice qsurf(i_h2o_ice) was loaded twice by phyetat0,
       ! as qsurf(i_h2o_vap) & as qsurf(i_h2o_ice), so to be clean:
       if (igcm_h2o_vap.ne.0) then
         qsurf(1:ngridmx,igcm_h2o_vap)=0
       endif
      endif

c------------------------------------------------------------
c     Initialisation tracers ....
c------------------------------------------------------------
      call zerophys(nqmx,rho_q)

      rho_dust=2500.  ! Mars dust density (kg.m-3)
      rho_ice=920.    ! Water ice density (kg.m-3)

      if (doubleq) then
c       "doubleq" technique 
c       -------------------
c      (transport of mass and number mixing ratio)
c       iq=1: Q mass mixing ratio, iq=2: N number mixing ratio

        if( (nqmx.lt.2).or.(water.and.(nqmx.lt.3)) ) then
          write(*,*)'initracer: nqmx is too low : nqmx=', nqmx
          write(*,*)'water= ',water,' doubleq= ',doubleq   
        end if

        varian=0.637    ! Characteristic variance    
        qext(igcm_dust_mass)=3.04   ! reference extinction at 0.67 um for ref dust
        qext(igcm_dust_number)=3.04 ! reference extinction at 0.67 um for ref dust
        rho_q(igcm_dust_mass)=rho_dust
        rho_q(igcm_dust_number)=rho_dust

c       Intermediate calcul for computing geometric mean radius r0
c       as a function of mass and number mixing ratio Q and N
c       (r0 = (r3n_q * Q/ N)
        r3n_q = exp(-4.5*varian**2)*(3./4.)/(pi*rho_dust)

c       Intermediate calcul for computing effective radius reff
c       from geometric mean radius r0
c       (reff = ref_r0 * r0)
        ref_r0 = exp(2.5*varian**2)
        
c       lifted dust :
c       '''''''''''
        reff_lift = 3.e-6      !  Effective radius of lifted dust (m)
        alpha_devil(igcm_dust_mass)=9.e-9   !  dust devil lift mass coeff
        alpha_lift(igcm_dust_mass)=3.0e-15  !  Lifted mass coeff

        r0_lift = reff_lift/ref_r0
        alpha_devil(igcm_dust_number)=r3n_q*
     &                        alpha_devil(igcm_dust_mass)/r0_lift**3
        alpha_lift(igcm_dust_number)=r3n_q*
     &                        alpha_lift(igcm_dust_mass)/r0_lift**3

c       Not used:
        radius(igcm_dust_mass) = 0.
        radius(igcm_dust_number) = 0.

      else

       if (dustbin.gt.1) then
        print*,'ATTENTION:',
     $   ' properties of dust need input in initracer !!!'
        stop

       else if (dustbin.eq.1) then

c       This will be used for 1 dust particle size:
c       ------------------------------------------
        radius(igcm_dustbin(1))=3.e-6
        Qext(igcm_dustbin(1))=3.04
        alpha_lift(igcm_dustbin(1))=0.0e-6
        alpha_devil(igcm_dustbin(1))=7.65e-9
        qextrhor(igcm_dustbin(1))=(3./4.)*Qext(igcm_dustbin(1))
     &                         /(rho_dust*radius(igcm_dustbin(1)))
        rho_q(igcm_dustbin(1))=rho_dust

       endif
      end if    ! (doubleq)

c     Initialization for photochemistry:
c     ---------------------------------
      if (photochem) then
      ! initialize chemistry+water (water will be correctly initialized below)
      ! by initializing everything which is not dust ...
        do iq=1,nqmx
          txt=noms(iq)
          if (txt(1:4).ne."dust") then
            radius(iq)=0.
            Qext(iq)=0.
            alpha_lift(iq) =0.
            alpha_devil(iq)=0.
            qextrhor(iq)= 0.
         endif
        enddo ! do iq=1,nqmx
      endif

c     Initialization for water vapor
c     ------------------------------
      if(water) then
         radius(igcm_h2o_vap)=0.
         Qext(igcm_h2o_vap)=0.
         alpha_lift(igcm_h2o_vap) =0.
         alpha_devil(igcm_h2o_vap)=0.
         qextrhor(igcm_h2o_vap)= 0.

c       "Dryness coefficient" controlling the evaporation and
c        sublimation from the ground water ice (close to 1)
c        HERE, the goal is to correct for the fact
c        that the simulated permanent water ice polar caps
c        is larger than the actual cap and the atmospheric
c        opacity not always realistic.

         do ig=1,ngridmx
           if (ngridmx.ne.1) watercaptag(ig)=.false.
           dryness(ig) = 1.
         enddo

         IF (caps) THEN
c Perennial H20 north cap defined by watercaptag=true (allows surface to be
c hollowed by sublimation in vdifc).
         do ig=1,ngridmx
           if (lati(ig)*180./pi.gt.84) then
             if (ngridmx.ne.1) watercaptag(ig)=.true.
             dryness(ig) = 1.
c Use the following cap definition for high spatial resolution (latitudinal bin <= 5 deg)
c	      if (lati(ig)*180./pi.lt.85.and.long(ig).ge.0) then
c	        if (ngridmx.ne.1) watercaptag(ig)=.true.
c               dryness(ig) = 1.
c	      endif
c             if (lati(ig)*180./pi.ge.85) then
c               if (ngridmx.ne.1) watercaptag(ig)=.true.
c               dryness(ig) = 1.
c	      endif
           endif  ! (lati>80 deg)
         end do ! (ngridmx)
        ENDIF ! (caps)

         if(water.and.(nqmx.ge.2)) then
           radius(igcm_h2o_ice)=3.e-6
           rho_q(igcm_h2o_ice)=rho_ice
           Qext(igcm_h2o_ice)=0.
           alpha_lift(igcm_h2o_ice) =0.
           alpha_devil(igcm_h2o_ice)=0.
           qextrhor(igcm_h2o_ice)= (3./4.)*Qext(igcm_h2o_ice) 
     $       / (rho_ice*radius(igcm_h2o_ice))
         elseif(water.and.(nqmx.lt.2)) then
            write(*,*) 'nqmx is too low : nqmx=', nqmx
            write(*,*) 'water= ',water
         endif

      end if  ! (water)

c     Output for records:
c     ~~~~~~~~~~~~~~~~~~
      write(*,*)
      Write(*,*) '******** initracer : dust transport parameters :'
      write(*,*) 'alpha_lift = ', alpha_lift
      write(*,*) 'alpha_devil = ', alpha_devil
      write(*,*) 'radius  = ', radius
      if(doubleq) then
        write(*,*) 'reff_lift (um) =  ', reff_lift
        write(*,*) 'size distribution variance  = ', varian
        write(*,*) 'r3n_q , ref_r0 : ', r3n_q , ref_r0
      end if
      write(*,*) 'Qext  = ', qext 
      write(*,*)

      end
