MODULE inifis_mod
IMPLICIT NONE

CONTAINS

  SUBROUTINE inifis(ngrid,nlayer,nq, &
             day_ini,pdaysec,nday,ptimestep, &
             plat,plon,parea, &
             prad,pg,pr,pcpp)

  use init_print_control_mod, only: init_print_control
  use radinc_h, only: ini_radinc_h, naerkind
  use radcommon_h, only: ini_radcommon_h
  use radii_mod, only: radfixed
  use datafile_mod, only: datadir,hazeprop_file,hazerad_file,hazemmr_file,hazedens_file, &
                          config_mufi, mugasflux_file, aersprop_file, aerfprop_file, clddprop_file
  use comdiurn_h, only: sinlat, coslat, sinlon, coslon
  use comgeomfi_h, only: totarea, totarea_planet
  use comsoil_h, only: ini_comsoil_h, nsoilmx, lay1_soil, alpha_soil
  use time_phylmdz_mod, only: diagfi_output_rate,startfi_output_rate,slow_diagfi, &
                              init_time, daysec, dtphys
  use comcstfi_mod, only: rad, cpp, g, r, rcp, &
                          mugaz, pi, avocado
  use planetwide_mod, only: planetwide_sumval
  use nonoro_gwd_mix_mod, only: calljliu_gwimix
  use callkeys_mod
  use nonoro_gwd_ran_mod, only: ini_nonoro_gwd_ran, &
                                end_nonoro_gwd_ran
  use nonoro_gwd_mix_mod, only: ini_nonoro_gwd_mix, &
                                end_nonoro_gwd_mix
                                
  use surfdat_h
  use wstats_mod, only: callstats
  use writediagsoil_mod, only: diagsoil
  use ioipsl_getin_p_mod, only : getin_p
  use mod_phys_lmdz_para, only : is_parallel, is_master, bcast

!=======================================================================
!
!   purpose:
!   -------
!
!   Initialisation for the physical parametrisations of the LMD
!   Generic Model.
!
!   author: Frederic Hourdin 15 / 10 /93
!   -------
!   modified: Sebastien Lebonnois 11/06/2003 (new callphys.def)
!             Ehouarn Millour (oct. 2008) tracers are now identified
!              by their names and may not be contiguously
!              stored in the q(:,:,:,:) array
!             E.M. (june 2009) use getin routine to load parameters
!
!
!   arguments:
!   ----------
!
!   input:
!   ------
!
!    ngrid                 Size of the horizontal grid.
!                          All internal loops are performed on that grid.
!    nlayer                Number of vertical layers.
!    pdayref               Day of reference for the simulation
!    pday                  Number of days counted from the North. Spring
!                          equinoxe.
!
!=======================================================================
!
!-----------------------------------------------------------------------
!   declarations:
!   -------------
  use datafile_mod, only: datadir
  use ioipsl_getin_p_mod, only: getin_p
  IMPLICIT NONE



  REAL,INTENT(IN) :: prad,pg,pr,pcpp,pdaysec,ptimestep
  INTEGER,INTENT(IN) :: nday
  INTEGER,INTENT(IN) :: ngrid,nlayer,nq
  REAL,INTENT(IN) :: plat(ngrid),plon(ngrid),parea(ngrid)
  integer,intent(in) :: day_ini

  INTEGER ig
  CHARACTER(len=20) :: rname="inifis" ! routine name, for messages

  EXTERNAL iniorbit,orbite
  EXTERNAL SSUM
  REAL SSUM

  ! deprecated parameters
  REAL :: ecritphy ! to check that this obsolete flag is no longer used...
  logical aerohaze

  ! Initialize flags lunout, prt_level, debug (in print_control_mod)
  CALL init_print_control

  ! initialize constants in comcstfi_mod
  rad=prad
  cpp=pcpp
  g=pg
  r=pr
  rcp=r/cpp
  mugaz=8.314*1000./pr ! dummy init
  pi=2.*asin(1.)
  avocado = 6.02214179e23   ! added by RW

  ! Initialize some "temporal and calendar" related variables
#ifndef MESOSCALE
  CALL init_time(day_ini,pdaysec,nday,ptimestep)
#endif


  ! read in some parameters from "run.def" for physics,
  ! or shared between dynamics and physics.
  ecritphy=-666 ! dummy default value
  call getin_p("ecritphy",ecritphy) ! frequency of outputs in physics,
                                    ! in dynamical steps
  if (ecritphy/=-666) then
    call abort_physic(rname, &
         "Error: parameter ecritphy is obsolete! Remove it! "//&
         "And use diagfi_output_rate instead",1)
  endif

  ! do we read a startphy.nc file? (default: .true.)
  call getin_p("startphy_file",startphy_file)

! --------------------------------------------------------------
!  Reading the "callphys.def" file controlling some key options
! --------------------------------------------------------------

  IF (is_master) THEN
    ! check if 'callphys.def' file is around
    inquire(file="callphys.def",exist=iscallphys)
    write(*,*) trim(rname)//": iscallphys=",iscallphys
  ENDIF
  call bcast(iscallphys)

  IF(iscallphys) THEN

     if (is_master) then
       write(*,*)
       write(*,*)'--------------------------------------------'
       write(*,*)trim(rname)//': Parameters for the physics (callphys.def)'
       write(*,*)'--------------------------------------------'
     endif

     if (is_master) write(*,*) trim(rname)//&
       ": Directory where external input files are:"
     ! default 'datadir' is set in "datadir_mod"
     call getin_p("datadir",datadir) ! default path
     if (is_master) write(*,*) trim(rname)//": datadir = ",trim(datadir)

     if (is_master) write(*,*) "Haze optical properties datafile"
     hazeprop_file="optprop_tholins_fractal050nm.dat"  ! default file
     call getin_p("hazeprop_file",hazeprop_file)
     if (is_master) write(*,*) trim(rname)//" hazeprop_file = ",trim(hazeprop_file)

     if (is_master) write(*,*) "Haze radii datafile"
     hazerad_file="hazerad.txt"  ! default file
     call getin_p("hazerad_file",hazerad_file)
     if (is_master) write(*,*) trim(rname)//" hazerad_file = ",trim(hazerad_file)

     if (is_master) write(*,*) "Haze mmr datafile"
     hazemmr_file="None"  ! default file
     call getin_p("hazemmr_file",hazemmr_file)
     if (is_master) write(*,*) trim(rname)//" hazemmr_file = ",trim(hazemmr_file)
     if (is_master) write(*,*) "Haze density datafile"
     hazedens_file="None"  ! default file
     call getin_p("hazedens_file",hazedens_file)
     if (is_master) write(*,*) trim(rname)//" hazedens_file = ",trim(hazedens_file)

     if (is_master) write(*,*) trim(rname)//&
       ": Run with or without tracer transport ?"
     tracer=.false. ! default value
     call getin_p("tracer",tracer)
     if (is_master) write(*,*) trim(rname)//": tracer = ",tracer

     if (is_master) write(*,*) trim(rname)//&
       ": Output rate for diagfi.nc file (in physics steps) ?"
     diagfi_output_rate=24 !default value
     call getin_p("diagfi_output_rate",diagfi_output_rate)
     if (is_master) write(*,*) trim(rname)//": diagfi_output_rate = ",&
                               diagfi_output_rate

     if ((is_master).and.(ngrid.eq.1)) write(*,*) trim(rname)//&
       ": Open/close diagfi.nc at output rate (rather than once) ?"
     slow_diagfi =.false. ! default value, open/close diagfi once (1D only)
 	   call getin_p("slow_diagfi",slow_diagfi) ! if true, open/close the diagfi.nc
 	                                           ! at output frequency instead of once
 	                                           ! (slower but required for debug)
 	   if ((is_master).and.(ngrid.eq.1)) write(*,*) trim(rname)//&
 	     ": slow_diagfi = ", slow_diagfi
     
     if (is_master) write(*,*) trim(rname)//&
       ": Output rate for start/startfi.nc files (in physics steps) ?"
     startfi_output_rate=0 !default value
     call getin_p("startfi_output_rate",startfi_output_rate)
     if (is_master) write(*,*) trim(rname)//": startfi_output_rate = ",&
                               startfi_output_rate

     if (is_master) write(*,*) trim(rname)//&
       ": Run with or without atm mass update "//&
       " due to tracer evaporation/condensation?"
     mass_redistrib=.false. ! default value
     call getin_p("mass_redistrib",mass_redistrib)
     if (is_master) write(*,*) trim(rname)//&
       ": mass_redistrib = ",mass_redistrib

     if (is_master) then
       write(*,*) trim(rname)//": Diurnal cycle ?"
       write(*,*) trim(rname)//&
       ": (if diurnal=false, diurnal averaged solar heating)"
     endif
     diurnal=.true. ! default value
     call getin_p("diurnal",diurnal)
     if (is_master) write(*,*) trim(rname)//": diurnal = ",diurnal

     if (is_master) then
       write(*,*) trim(rname)//": Seasonal cycle ?"
       write(*,*) trim(rname)//&
       ": (if season=false, Ls stays constant, to value ", &
       "set in 'start'"
     endif
     season=.true. ! default value
     call getin_p("season",season)
     if (is_master) write(*,*) trim(rname)//": season = ",season

     evol1d=.false. ! default value
     call getin_p("evol1d",evol1d)
     if (is_master) write(*,*) trim(rname)//": evol1d = ",evol1d

     if (is_master) then
       write(*,*) trim(rname)//": Fast mode (nogcm) ?"
     endif
     fast=.false. ! default value
     call getin_p("fast",fast)
     if (is_master) write(*,*) trim(rname)//": fast = ",fast

     if (is_master) write(*,*) trim(rname)//"Run with Triton orbit ?"
     triton=.false. ! default value
     call getin_p("triton",triton)
     if (is_master) write(*,*) trim(rname)//" triton = ",triton
     convergeps=.false. ! default value
     call getin_p("convergeps",convergeps)
     if (is_master) write(*,*) trim(rname)//" convergeps = ",convergeps

     conservn2=.false. ! default value
     call getin_p("conservn2",conservn2)
     if (is_master) write(*,*) trim(rname)//" conservn2 = ",conservn2
     conservch4=.false. ! default value
     call getin_p("conservch4",conservch4)
     if (is_master) write(*,*) trim(rname)//" conservch4 = ",conservch4

     if (is_master) write(*,*) trim(rname)//"KBO runs (eris, makemake) ?"
     kbo=.false. ! default value
     call getin_p("kbo",kbo)
     if (is_master) write(*,*) trim(rname)//" kbo = ",kbo

     if (is_master) write(*,*) trim(rname)//"Specific paleo run ?"
     paleo=.false. ! default value
     call getin_p("paleo",paleo)
     if (is_master) write(*,*) trim(rname)//" paleo = ",paleo

     if (is_master) write(*,*) trim(rname)//"paleoclimate step (Earth years) "
     paleoyears=10000. ! default value
     call getin_p("paleoyears",paleoyears)
     if (is_master) write(*,*) trim(rname)//" paleoyears = ",paleoyears

     if (is_master) write(*,*) trim(rname)//&
       ": No seasonal cycle: initial day to lock the run during restart"
     noseason_day=0.0 ! default value
     call getin_p("noseason_day",noseason_day)
     if (is_master) write(*,*) trim(rname)//": noseason_day=", noseason_day

     if (is_master) write(*,*) trim(rname)//&
       ": Write some extra output to the screen ?"
     lwrite=.false. ! default value
     call getin_p("lwrite",lwrite)
     if (is_master) write(*,*) trim(rname)//": lwrite = ",lwrite

     if (is_master) write(*,*) trim(rname)//&
       ": Save statistics in file stats.nc ?"
     callstats=.true. ! default value
     call getin_p("callstats",callstats)
     if (is_master) write(*,*) trim(rname)//": callstats = ",callstats

     if (is_master) write(*,*) trim(rname)//&
       ": Write sub-surface fields in file diagsoil.nc ?"
     diagsoil=.false. ! default value
     call getin_p("diagsoil",diagsoil)
     if (is_master) write(*,*) trim(rname)//" diagsoil = ",diagsoil

     if (is_master) write(*,*) trim(rname)//&
       ": Test energy conservation of model physics ?"
     enertest=.false. ! default value
     call getin_p("enertest",enertest)
     if (is_master) write(*,*) trim(rname)//": enertest = ",enertest

     if (is_master) write(*,*) trim(rname)//": call radiative transfer ?"
     callrad=.true. ! default value
     call getin_p("callrad",callrad)
     if (is_master) write(*,*) trim(rname)//": callrad = ",callrad

     if (is_master) write(*,*) trim(rname)//&
       ": call correlated-k radiative transfer ?"
     corrk=.true. ! default value
     call getin_p("corrk",corrk)
     if (is_master) write(*,*) trim(rname)//": corrk = ",corrk

     if (is_master) write(*,*) trim(rname)//&
       ": call simple fixed temperature profile"
     fixed_temp_prof=.false. ! default value
     call getin_p("fixed_temp_prof",fixed_temp_prof)
     if (is_master) write(*,*) trim(rname)//": fixed_temp_prof = ",fixed_temp_prof
     tau_temp=1. ! default value
     call getin_p("tau_temp",tau_temp)

     if (is_master) write(*,*) trim(rname)//&
       ": prohibit calculations outside corrk T grid?"
     strictboundcorrk=.true. ! default value
     call getin_p("strictboundcorrk",strictboundcorrk)
     if (is_master) write(*,*) trim(rname)//&
       ": strictboundcorrk = ",strictboundcorrk

     if (is_master) write(*,*) trim(rname)//&
       ": prohibit calculations outside CIA T grid?"
     strictboundcia=.true. ! default value
     call getin_p("strictboundcia",strictboundcia)
     if (is_master) write(*,*) trim(rname)//&
       ": strictboundcia = ",strictboundcia

     if (is_master) write(*,*) trim(rname)//&
       ": Minimum atmospheric temperature for Planck function integration ?"
     tplanckmin=30.0 ! default value
     call getin_p("tplanckmin",tplanckmin)
     if (is_master) write(*,*) trim(rname)//": tplanckmin = ",tplanckmin

     if (is_master) write(*,*) trim(rname)//&
       ": Maximum atmospheric temperature for Planck function integration ?"
     tplanckmax=1500.0 ! default value
     call getin_p("tplanckmax",tplanckmax)
     if (is_master) write(*,*) trim(rname)//": tplanckmax = ",tplanckmax

     if (is_master) write(*,*) trim(rname)//&
       ": Temperature step for Planck function integration ?"
     dtplanck=0.1 ! default value
     call getin_p("dtplanck",dtplanck)
     if (is_master) write(*,*) trim(rname)//": dtplanck = ",dtplanck

     if (is_master) write(*,*) trim(rname)//&
       ": call gaseous absorption in the visible bands?"//&
       " (matters only if callrad=T)"
     callgasvis=.false. ! default value
     call getin_p("callgasvis",callgasvis)
     if (is_master) write(*,*) trim(rname)//": callgasvis = ",callgasvis

     if (is_master) write(*,*) trim(rname)//&
       ": call continuum opacities in radiative transfer ?"//&
       " (matters only if callrad=T)"
     continuum=.true. ! default value
     call getin_p("continuum",continuum)
     if (is_master) write(*,*) trim(rname)//": continuum = ",continuum

     if (is_master) write(*,*) trim(rname)//&
       ": call turbulent vertical diffusion ?"
     calldifv=.false. ! default value
     call getin_p("calldifv",calldifv)
     if (is_master) write(*,*) trim(rname)//": calldifv = ",calldifv
     vertdiff=.true. ! default value
     call getin_p("vertdiff",vertdiff)
     if (is_master) write(*,*) trim(rname)//": vertdiff = ",vertdiff

     if (is_master) write(*,*) trim(rname)//": use turbdiff instead of vdifc ?"
     UseTurbDiff=.false. ! default value
     call getin_p("UseTurbDiff",UseTurbDiff)
     if (is_master) write(*,*) trim(rname)//": UseTurbDiff = ",UseTurbDiff

     if (is_master) write(*,*) trim(rname)//": call convective adjustment ?"
     calladj=.false. ! default value
     call getin_p("calladj",calladj)
     if (is_master) write(*,*) trim(rname)//": calladj = ",calladj

     if (is_master) write(*,*) trim(rname)//&
     ": Radiative timescale for Newtonian cooling (in Earth days)?"
     tau_relax=30. ! default value
     call getin_p("tau_relax",tau_relax)
     if (is_master) write(*,*) trim(rname)//": tau_relax = ",tau_relax
     tau_relax=tau_relax*24*3600 ! convert Earth days --> seconds

     if (is_master) write(*,*)trim(rname)//&
       ": call thermal conduction in the soil ?"
     callsoil=.true. ! default value
     call getin_p("callsoil",callsoil)
     if (is_master) write(*,*) trim(rname)//": callsoil = ",callsoil

     if (is_master) write(*,*)trim(rname)//&
       ": call orographic gravity waves ?"
     calllott=.false. ! default value
     call getin_p("calllott",calllott)
     if (is_master) write(*,*)trim(rname)//": calllott = ",calllott

     if (is_master) write(*,*)trim(rname)//&
       ": call  non-orographic gravity waves ?"
     calllott_nonoro=.false. ! default value
     call getin_p("calllott_nonoro",calllott_nonoro)
     if (is_master) write(*,*)trim(rname)//&
       ": calllott_nonoro = ",calllott_nonoro

     if (is_master) write(*,*)trim(rname)//&       
       ": call jliu's non-orogrphic GW-induced turbulence"
     calljliu_gwimix=.false. ! default value
     call getin_p("calljliu_gwimix",calljliu_gwimix)
     if (is_master) write(*,*)trim(rname)//&    
       ": calljliu_gwimix = ",calljliu_gwimix

     if (is_master) write(*,*)trim(rname)//&
       ": Rad transfer is computed every iradia", &
       " physical timestep"
     iradia=1 ! default value
     call getin_p("iradia",iradia)
     if (is_master) write(*,*)trim(rname)//": iradia = ",iradia

     if (is_master) write(*,*)trim(rname)//": Rayleigh scattering ?"
     rayleigh=.false.
     call getin_p("rayleigh",rayleigh)
     if (is_master) write(*,*)trim(rname)//": rayleigh = ",rayleigh

     if (is_master) write(*,*) trim(rname)//&
       ": Use blackbody for stellar spectrum ?"
     stelbbody=.false. ! default value
     call getin_p("stelbbody",stelbbody)
     if (is_master) write(*,*) trim(rname)//": stelbbody = ",stelbbody

     if (is_master) write(*,*) trim(rname)//": Stellar blackbody temperature ?"
     stelTbb=5800.0 ! default value
     call getin_p("stelTbb",stelTbb)
     if (is_master) write(*,*) trim(rname)//": stelTbb = ",stelTbb

     if (is_master) write(*,*) trim(rname)//": Output mean OLR in 1D?"
     meanOLR=.false.
     call getin_p("meanOLR",meanOLR)
     if (is_master) write(*,*) trim(rname)//": meanOLR = ",meanOLR

     if (is_master) write(*,*)trim(rname)//": Output spectral OLR in 3D?"
     specOLR=.false.
     call getin_p("specOLR",specOLR)
     if (is_master) write(*,*)trim(rname)//": specOLR = ",specOLR

     if (is_master) write(*,*)trim(rname)//": Operate in kastprof mode?"
     kastprof=.false.
     call getin_p("kastprof",kastprof)
     if (is_master) write(*,*)trim(rname)//": kastprof = ",kastprof

     if (is_master) write(*,*)trim(rname)//&
       ": Uniform absorption in radiative transfer?"
     graybody=.false.
     call getin_p("graybody",graybody)
     if (is_master) write(*,*)trim(rname)//": graybody = ",graybody

! Soil model
     if (is_master) write(*,*)trim(rname)//&
       ": Number of sub-surface layers for soil scheme?"
     ! default value of nsoilmx set in comsoil_h
     call getin_p("nsoilmx",nsoilmx)
     if (is_master) write(*,*)trim(rname)//": nsoilmx=",nsoilmx

     if (is_master) write(*,*)trim(rname)//&
       ": Thickness of topmost soil layer (m)?"
     ! default value of lay1_soil set in comsoil_h
     call getin_p("lay1_soil",lay1_soil)
     if (is_master) write(*,*)trim(rname)//": lay1_soil = ",lay1_soil

     if (is_master) write(*,*)trim(rname)//&
       ": Coefficient for soil layer thickness distribution?"
     ! default value of alpha_soil set in comsoil_h
     call getin_p("alpha_soil",alpha_soil)
     if (is_master) write(*,*)trim(rname)//": alpha_soil = ",alpha_soil

     if (is_master) write(*,*)trim(rname)//&
       "Geothermal flux (W) at the bottom layer"
     fluxgeo=0. ! default value
     call getin_p("fluxgeo",fluxgeo)
     if (is_master) write(*,*)trim(rname)//" fluxgeo = ",fluxgeo

     if (is_master) write(*,*)trim(rname)//&
       "Assymetry flux (W) at the bottom layer"
     assymflux=.false. ! default value
     call getin_p("assymflux",assymflux)
     if (is_master) write(*,*)trim(rname)//" assymflux = ",assymflux

     if (is_master) write(*,*)trim(rname)//&
       "Geothermal flux (W) for assymetry"
     fluxgeo2=fluxgeo ! default value
     call getin_p("fluxgeo2",fluxgeo2)
     if (is_master) write(*,*)trim(rname)//" fluxgeo2 = ",fluxgeo2

     if (is_master) write(*,*)trim(rname)//&
       "Warm patch of flux"
     patchflux=0 ! default value
     call getin_p("patchflux",patchflux)
     if (is_master) write(*,*)trim(rname)//" patchflux = ",patchflux

! Chemistry in the thermosphere
     if (is_master) write(*,*) trim(rname)//": Use deposition ?"
     depos=.false.         ! default value
     call getin_p("depos",depos)
     if (is_master) write(*,*) trim(rname)//": depos = ",depos

     if (is_master) write(*,*)trim(rname)//": Production of haze ?"
     haze=.false. ! default value
     call getin_p("haze",haze)
     if (is_master) write(*,*)trim(rname)//": haze = ",haze



      if (is_master) write(*,*)trim(rname)// "call thermal conduction ?"
      callconduct=.false. ! default value
      call getin_p("callconduct",callconduct)
      if (is_master) write(*,*)trim(rname)// " callconduct = ",callconduct

      if (is_master) write(*,*)trim(rname)// "call phitop ?"
      phitop=0. ! default value
      call getin_p("phitop",phitop)
      if (is_master) write(*,*)trim(rname)// " phitop = ",phitop

      if (is_master) write(*,*)trim(rname)// "call molecular viscosity ?"
      callmolvis=.false. ! default value
      call getin_p("callmolvis",callmolvis)
      if (is_master) write(*,*)trim(rname)// " callmolvis = ",callmolvis

      if (is_master) write(*,*)trim(rname)// "call molecular diffusion ?"
      callmoldiff=.false. ! default value
      call getin_p("callmoldiff",callmoldiff)
      if (is_master) write(*,*)trim(rname)// " callmoldiff = ",callmoldiff

! Global1D mean and solar zenith angle

     if (ngrid.eq.1) then
      PRINT*, 'Simulate global averaged conditions ?'
      global1d = .false. ! default value
      call getin_p("global1d",global1d)
      write(*,*) "global1d = ",global1d

      ! Test of incompatibility : if global1d is true, there should not be any diurnal cycle.
      if (global1d.and.diurnal) then
         call abort_physic(rname,'if global1d is true, diurnal must be set to false',1)
      endif

      if (global1d) then
         PRINT *,'Solar Zenith angle (deg.) ?'
         PRINT *,'(assumed for averaged solar flux S/4)'
         szangle=60.0  ! default value
         call getin_p("szangle",szangle)
         write(*,*) "szangle = ",szangle
      endif
   endif ! of if (ngrid.eq.1)

! Test of incompatibility:
! if kastprof used, we must be in 1D
     if (kastprof.and.(ngrid.gt.1)) then
       call abort_physic(rname,'kastprof can only be used in 1D!',1)
     endif

     if (is_master) write(*,*)trim(rname)//&
       ": Stratospheric temperature for kastprof mode?"
     Tstrat=167.0
     call getin_p("Tstrat",Tstrat)
     if (is_master) write(*,*)trim(rname)//": Tstrat = ",Tstrat

     if (is_master) write(*,*)trim(rname)//": Remove lower boundary?"
     nosurf=.false.
     call getin_p("nosurf",nosurf)
     if (is_master) write(*,*)trim(rname)//": nosurf = ",nosurf

! Tests of incompatibility:
     if (nosurf.and.callsoil) then
       if (is_master) then
         write(*,*)trim(rname)//'nosurf not compatible with soil scheme!'
         write(*,*)trim(rname)//'... got to make a choice!'
       endif
       call abort_physic(rname,"nosurf not compatible with soil scheme!",1)
     endif

     if (is_master) write(*,*)trim(rname)//": Add an internal heat flux?", &
                   "... matters only if callsoil=F"
     intheat=0.
     call getin_p("intheat",intheat)
     if (is_master) write(*,*)trim(rname)//": intheat = ",intheat


     if (is_master) write(*,*)trim(rname)//": Test physics timescale in 1D?"
     testradtimes=.false.
     call getin_p("testradtimes",testradtimes)
     if (is_master) write(*,*)trim(rname)//": testradtimes = ",testradtimes

! Test of incompatibility:
! if testradtimes used, we must be in 1D
     if (testradtimes.and.(ngrid.gt.1)) then
       call abort_physic(rname,'testradtimes can only be used in 1D!',1)
     endif

     if (is_master) write(*,*)trim(rname)//": Default planetary temperature?"
     tplanet=215.0
     call getin_p("tplanet",tplanet)
     if (is_master) write(*,*)trim(rname)//": tplanet = ",tplanet

     if (is_master) write(*,*)trim(rname)//": Which star?"
     startype=1 ! default value = Sol
     call getin_p("startype",startype)
     if (is_master) write(*,*)trim(rname)//": startype = ",startype

     if (is_master) write(*,*)trim(rname)//": Value of stellar flux at 1 AU?"
     Fat1AU=1356.0 ! default value = Sol today
     call getin_p("Fat1AU",Fat1AU)
     if (is_master) write(*,*)trim(rname)//": Fat1AU = ",Fat1AU


! TRACERS:

     if (is_master) write(*,*)trim(rname)//": fixed radii for Cloud particles?"
     radfixed=.false. ! default value
     call getin_p("radfixed",radfixed)
     if (is_master) write(*,*)trim(rname)//": radfixed = ",radfixed

     if(kastprof)then
        radfixed=.true.
     endif

      if (is_master) write(*,*)trim(rname)//&
       "Number of radiatively active aerosols:"
     naerkind=0 ! default value
     call getin_p("naerkind",naerkind)
     if (is_master) write(*,*)trim(rname)//": naerkind = ",naerkind


!***************************************************************
         !! TRACERS options

     if (is_master)write(*,*)trim(rname)//&
      "call N2 condensation ?"
     n2cond=.true. ! default value
     call getin_p("n2cond",n2cond)
     if (is_master)write(*,*)trim(rname)//&
      " n2cond = ",n2cond

     if (is_master)write(*,*)trim(rname)//&
      "call N2 cloud condensation ?"
     condensn2=.false. ! default value
     call getin_p("condensn2",condensn2)
     if (is_master)write(*,*)trim(rname)//&
      "condensn2 = ",condensn2

     if (is_master)write(*,*)trim(rname)//&
      "call no N2 frost formation?"
     no_n2frost=.false. ! default value
     call getin_p("no_n2frost",no_n2frost)
     if (is_master)write(*,*)trim(rname)//&
      "no_n2frost = ",no_n2frost

     if (is_master)write(*,*)trim(rname)//&
      "N2 condensation subtimestep?"
     nbsub=20 ! default value
     call getin_p("nbsub",nbsub)
     if (is_master)write(*,*)trim(rname)//&
      " nbsub = ",nbsub

     if (is_master)write(*,*)trim(rname)//&
      "Gravitationnal sedimentation ?"
     sedimentation=.true. ! default value
     call getin_p("sedimentation",sedimentation)
     if (is_master)write(*,*)trim(rname)//&
      " sedimentation = ",sedimentation

     if (is_master)write(*,*)trim(rname)//&
      "Compute glacial flow ?"
     glaflow=.false. ! default value
     call getin_p("glaflow",glaflow)
     if (is_master)write(*,*)trim(rname)//&
      " glaflow = ",glaflow

     if (is_master)write(*,*)trim(rname)//&
      "Compute methane cycle ?"
     methane=.false. ! default value
     call getin_p("methane",methane)
     if (is_master)write(*,*)trim(rname)//&
      " methane = ",methane
     condmetsurf=.true. ! default value
     call getin_p("condmetsurf",condmetsurf)
     if (is_master)write(*,*)trim(rname)//&
      " condmetsurf = ",condmetsurf
     if (is_master)write(*,*)trim(rname)//&
      "call no CH4 frost formation?"
     no_ch4frost=.false. ! default value
     call getin_p("no_ch4frost",no_ch4frost)
     if (is_master)write(*,*)trim(rname)//&
      " no_ch4frost = ",no_ch4frost

     if (is_master)write(*,*)trim(rname)//&
      "Compute methane clouds ?"
     metcloud=.false. ! default value
     call getin_p("metcloud",metcloud)
     if (is_master)write(*,*)trim(rname)//&
      " metcloud = ",metcloud

     if (is_master)write(*,*)trim(rname)//&
      "Compute CO cycle ?"
     carbox=.false. ! default value
     call getin_p("carbox",carbox)
     if (is_master)write(*,*)trim(rname)//&
      " carbox = ",carbox
     condcosurf=.true. ! default value
     call getin_p("condcosurf",condcosurf)
     if (is_master)write(*,*)trim(rname)//&
      " condcosurf = ",condcosurf

     if (is_master)write(*,*)trim(rname)//&
      "Compute CO clouds ?"
     monoxcloud=.false. ! default value
     call getin_p("monoxcloud",monoxcloud)
     if (is_master)write(*,*)trim(rname)//&
      " monoxcloud = ",monoxcloud

     if (is_master)write(*,*)trim(rname)//&
     "atmospheric redistribution (s):"
     tau_n2=1. ! default value
     call getin_p("tau_n2",tau_n2)
     if (is_master)write(*,*)trim(rname)//&
     " tau_n2 = ",tau_n2
     tau_ch4=1.E7 ! default value
     call getin_p("tau_ch4",tau_ch4)
     if (is_master)write(*,*)trim(rname)//&
     " tau_ch4 = ",tau_ch4
     tau_co=1. ! default value
     call getin_p("tau_co",tau_co)
     if (is_master)write(*,*)trim(rname)//&
     " tau_co = ",tau_co
     tau_prechaze=1. ! default value
     call getin_p("tau_prechaze",tau_prechaze)
     if (is_master)write(*,*)trim(rname)//&
     " tau_prechaze = ",tau_prechaze

     if (is_master)write(*,*)trim(rname)//&
      "Day fraction for limited cold trap in SP?"
     dayfrac=0. ! default value
     call getin_p("dayfrac",dayfrac)
     if (is_master)write(*,*)trim(rname)//&
      " dayfrac = ",dayfrac
     thresh_non2=0. ! default value
     call getin_p("thresh_non2",thresh_non2)
     if (is_master)write(*,*)trim(rname)//&
      " thresh_non2 = ",thresh_non2

    ! use Pluto.old routines
     if (is_master) write(*,*) trim(rname)//": use vdifc from old Pluto ?"
     oldplutovdifc=.false. ! default value
     call getin_p("oldplutovdifc",oldplutovdifc)
     if (is_master) write(*,*) trim(rname)//": oldplutovdifc = ",oldplutovdifc

     if (is_master) write(*,*) trim(rname)//&
       ": call pluto.old correlated-k radiative transfer ?"
     oldplutocorrk=.false. ! default value
     call getin_p("oldplutocorrk",oldplutocorrk)
     if (is_master) write(*,*) trim(rname)//": oldplutocorrk = ",oldplutocorrk

     if (is_master) write(*,*) trim(rname)//&
       ": call pluto.old sedimentation ?"
     oldplutosedim=.false. ! default value
     call getin_p("oldplutosedim",oldplutosedim)
     if (is_master) write(*,*) trim(rname)//": oldplutosedim = ",oldplutosedim

!***************************************************************
     !! Haze options

     ! Microphysical moment model
     ! ~~~~~~~~~~~~~~~~~~~~~~~~~~
     if (is_master) write(*,*) "Run with or without microphysics?"
     callmufi=.false. ! default value
     call getin_p("callmufi",callmufi)
     if (is_master) write(*,*)" callmufi = ",callmufi

     if (is_master) write(*,*) "Run with or without microphysical clouds?"
     callmuclouds=.false. ! default value
     call getin_p("callmuclouds",callmuclouds)
     if (is_master) write(*,*)" callmuclouds = ",callmuclouds

     ! sanity check
     if (callmufi.and.(.not.tracer)) then
       print*,"You are running microphysics without tracer"
       print*,"Please start again with tracer =.true."
       stop
     endif
     if (callmuclouds.and.(.not.callmufi)) then
       print*,"You are running microphysical clouds without microphysics"
       print*,"Please start again with callmufi =.true."
       stop
     endif

     if (is_master) write(*,*) "Path to microphysical config file?"
     config_mufi='datagcm/microphysics/config.cfg' ! default value
     call getin_p("config_mufi",config_mufi)
     if (is_master) write(*,*) trim(rname)//" config_mufi = ",config_mufi

     if (is_master) write(*,*) "Condensable gas fluxes datafile"
     mugasflux_file='None' ! default file
     call getin_p("mugasflux_file",mugasflux_file)
     if (is_master) write(*,*) trim(rname)//" mugasflux_file = ",trim(mugasflux_file)

     if (is_master) write(*,*) "Spherical aerosol optical properties datafile"
     aersprop_file="optprop_rannou_r2-200nm_nu003.dat"  ! default file
     call getin_p("aersprop_file",aersprop_file)
     if (is_master) write(*,*) trim(rname)//" aersprop_file = ",trim(aersprop_file)

     if (is_master) write(*,*) "Fractal aerosol optical properties datafile"
     aerfprop_file="optprop_rannou_fractal_r010nm_N1_1e4_d2.dat"  ! default file
     call getin_p("aerfprop_file",aerfprop_file)
     if (is_master) write(*,*) trim(rname)//" aerfprop_file = ",trim(aerfprop_file)

     if (is_master) write(*,*) "Cloud drop optical properties datafile"
     clddprop_file="optprop_rannou_r2-200nm_nu003.dat"  ! default file
     call getin_p("clddprop_file",clddprop_file)
     if (is_master) write(*,*) trim(rname)//" clddprop_file = ",trim(clddprop_file)

     if (is_master) write(*,*) "Use haze production from CH4 photolysis or production rate?"
     call_haze_prod_pCH4=.false. ! default value
     call getin_p("call_haze_prod_pCH4",call_haze_prod_pCH4)
     if (is_master) write(*,*)" call_haze_prod_pCH4 = ",call_haze_prod_pCH4

     if (is_master) write(*,*) "Pressure level of aerosols production (Pa)?"
     haze_p_prod=1.0e-2 ! default value
     call getin_p("haze_p_prod",haze_p_prod)
     if (is_master) write(*,*)" haze_p_prod = ",haze_p_prod

     if (is_master) write(*,*) "Aerosol production rate (kg.m-2.s-1)?"
     haze_tx_prod=9.8e-14 ! default value
     call getin_p("haze_tx_prod",haze_tx_prod)
     if (is_master) write(*,*)" haze_tx_prod = ",haze_tx_prod

     if (is_master) write(*,*) "Equivalent radius production (m)?"
     haze_rc_prod=1.0e-9 ! default value
     call getin_p("haze_rc_prod",haze_rc_prod)
     if (is_master) write(*,*)" haze_rc_prod = ",haze_rc_prod

     if (is_master) write(*,*) "Monomer radius (m)?"
     haze_rm=1.0e-8 ! default value
     call getin_p("haze_rm",haze_rm)
     if (is_master) write(*,*)" haze_rm = ",haze_rm

     if (is_master) write(*,*) "Aerosol's fractal dimension?"
     haze_df=2.0 ! default value
     call getin_p("haze_df",haze_df)
     if (is_master) write(*,*)" haze_df = ",haze_df

     if (is_master) write(*,*) "Aerosol density (kg.m-3)?"
     haze_rho=800.0 ! default value
     call getin_p("haze_rho",haze_rho)
     if (is_master) write(*,*)" haze_rho = ",haze_rho

     if (is_master) write(*,*) "Radius of air molecule (m)?"
     air_rad=1.75e-10 ! default value
     call getin_p("air_rad",air_rad)
     if (is_master) write(*,*)" air_rad = ",air_rad

     ! Conversion precursor to haze
     ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     if (is_master) write(*,*) "Time constant of conversion in aerosol (s)?"
     tcon_ch4=1.e7 ! default value
     call getin_p("tcon_ch4",tcon_ch4)
     if (is_master) write(*,*)" tcon_ch4 = ",tcon_ch4

     if (is_master) write(*,*) "Ratio CH4 photolysis?"
     k_ch4=1. ! default value
     call getin_p("k_ch4",k_ch4)
     if (is_master) write(*,*)" k_ch4 = ",k_ch4

     if (is_master) write(*,*) "Nitrogen contribution ratio N/C?"
     ncratio_ch4=0.5 ! default value
     call getin_p("ncratio_ch4",ncratio_ch4)
     if (is_master) write(*,*)" ncratio_ch4 = ",ncratio_ch4

     ! Variables for aerosol absorption
     ! ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     if (is_master) write(*,*) "Sph. aer. absorption correction in VI?"
     Fabs_aers_VI=1. ! default value
     call getin_p("Fabs_aers_VI",Fabs_aers_VI)
     if (is_master) write(*,*)" Fabs_aers_VI = ",Fabs_aers_VI

     if (is_master) write(*,*) "Fra. aer. absorption correction in VI?"
     Fabs_aerf_VI=1. ! default value
     call getin_p("Fabs_aerf_VI",Fabs_aerf_VI)
     if (is_master) write(*,*)" Fabs_aerf_VI = ",Fabs_aerf_VI

     if (is_master) write(*,*) "Sph. aer. absorption correction in IR?"
     Fabs_aers_IR=1. ! default value
     call getin_p("Fabs_aers_IR",Fabs_aers_IR)
     if (is_master) write(*,*)" Fabs_aers_IR = ",Fabs_aers_IR

     if (is_master) write(*,*) "Fra. aer. absorption correction in IR?"
     Fabs_aerf_IR=1. ! default value
     call getin_p("Fabs_aerf_IR",Fabs_aerf_IR)
     if (is_master) write(*,*)" Fabs_aerf_IR = ",Fabs_aerf_IR

     if (is_master) write(*,*) "Cloud drop absorption correction in VI?"
     Fabs_cldd_VI=1. ! default value
     call getin_p("Fabs_cldd_VI",Fabs_cldd_VI)
     if (is_master) write(*,*)" Fabs_cldd_VI = ",Fabs_cldd_VI

     if (is_master) write(*,*) "Cloud drop absorption correction in IR?"
     Fabs_cldd_IR=1. ! default value
     call getin_p("Fabs_cldd_IR",Fabs_cldd_IR)
     if (is_master) write(*,*)" Fabs_cldd_IR = ",Fabs_cldd_IR

     ! Pluto haze model
     ! ~~~~~~~~~~~~~~~~
     if (is_master)write(*,*)trim(rname)//&
     "Production of haze ?"
     haze=.false. ! default value
     call getin_p("haze",haze)
     if (is_master)write(*,*)trim(rname)//&
     " haze = ",haze
     hazeconservch4=.false. ! conservch4, by default value ch4 is photolyzed
     call getin_p("hazeconservch4",hazeconservch4)
     if (is_master)write(*,*)trim(rname)//&
     "hazconservch4 = ",hazeconservch4
     if (is_master) write(*,*) "Production of haze with CH4 fix profile?"
     haze_ch4proffix=.false. ! default value
     call getin_p("haze_ch4proffix",haze_ch4proffix)
     if (is_master) write(*,*)" haze_ch4proffix = ",haze_ch4proffix
     if (is_master)write(*,*)trim(rname)//&
     "Production of haze (fast version) ?"
     fasthaze=.false. ! default value
     call getin_p("fasthaze",fasthaze)
     if (is_master)write(*,*)trim(rname)//&
     "fasthaze = ",fasthaze

     if (is_master)write(*,*)trim(rname)//&
     "Add source of haze ?"
     source_haze=.false. ! default value
     call getin_p("source_haze",source_haze)
     if (is_master)write(*,*)trim(rname)//&
     " source_haze = ",source_haze
     mode_hs=0 ! mode haze source default value
     call getin_p("mode_hs",mode_hs)
     if (is_master)write(*,*)trim(rname)//&
     " mode_hs",mode_hs
     kfix=1 ! default value
     call getin_p("kfix",kfix)
     if (is_master)write(*,*)trim(rname)//&
     "levels kfix",kfix
     fracsource=0.1 ! default value
     call getin_p("fracsource",fracsource)
     if (is_master)write(*,*)trim(rname)//&
     " fracsource",fracsource
     latsource=30. ! default value
     call getin_p("latsource",latsource)
     if (is_master)write(*,*)trim(rname)//&
     " latsource",latsource
     lonsource=180. ! default value
     call getin_p("lonsource",lonsource)
     if (is_master)write(*,*)trim(rname)//&
     " lonsource",lonsource

     if (is_master)write(*,*)trim(rname)//&
     "Radiatively active haze ?"
     optichaze=.false. ! default value
     call getin_p("optichaze",optichaze)
     if (is_master)write(*,*)trim(rname)//&
     "optichaze = ",optichaze

     aerohaze=.false. ! default value
     call getin_p("aerohaze",aerohaze)
     if (aerohaze) then
      if (is_master) write(*,*)trim(rname)//": aerohaze is deprecated.",&
      "it is now called optichaze=.true."
      call abort_physic(rname,"aerohaze is deprecated. It is now called optichaze",1)
     endif

     if (is_master)write(*,*)trim(rname)//&
     "Haze monomer radius ?"
     rad_haze=10.e-9 ! default value
     call getin_p("rad_haze",rad_haze)
     if (is_master)write(*,*)trim(rname)//&
     "rad_haze = ",rad_haze

     if (is_master)write(*,*)trim(rname)//&
     "fractal particle ?"
     fractal=.false. ! default value
     call getin_p("fractal",fractal)
     if (is_master)write(*,*)trim(rname)//&
     "fractal = ",fractal
     nb_monomer=10 ! default value
     call getin_p("nb_monomer",nb_monomer)
     if (is_master)write(*,*)trim(rname)//&
     "nb_monomer = ",nb_monomer

     if (is_master)write(*,*)trim(rname)//&
     "fixed gaseous CH4 mixing ratio for rad transfer ?"
    ch4fix=.false. ! default value
    call getin_p("ch4fix",ch4fix)
    if (is_master)write(*,*)trim(rname)//&
     " ch4fix = ",ch4fix
    if (is_master)write(*,*)trim(rname)//&
     "fixed gaseous CH4 mixing ratio for rad transfer ?"
    vmrch4fix=0.5 ! default value
    call getin_p("vmrch4fix",vmrch4fix)
    if (is_master)write(*,*)trim(rname)//&
     " vmrch4fix = ",vmrch4fix
    vmrch4_proffix=.false. ! default value
    call getin_p("vmrch4_proffix",vmrch4_proffix)
    if (is_master)write(*,*)trim(rname)//&
     " vmrch4_proffix = ",vmrch4_proffix

    if (is_master)write(*,*)trim(rname)//&
     "call specific cooling for radiative transfer ?"
    cooling=.false.  ! default value
    call getin_p("cooling",cooling)
    if (is_master)write(*,*)trim(rname)//&
     " cooling = ",cooling

    if (is_master)write(*,*)trim(rname)//&
     "NLTE correction for methane heating rates?"
    nlte=.false.  ! default value
    call getin_p("nlte",nlte)
    if (is_master)write(*,*)trim(rname)//&
     " nlte = ",nlte
    strobel=.true.  ! default value
    call getin_p("strobel",strobel)
    if (is_master)write(*,*)trim(rname)//&
     " strobel = ",strobel

     if (is_master)write(*,*)trim(rname)//&
     "fixed radius profile from txt file ?"
     haze_radproffix=.false. ! default value
     call getin_p("haze_radproffix",haze_radproffix)
     if (is_master)write(*,*)trim(rname)//&
     "haze_radproffix = ",haze_radproffix
     if (is_master)write(*,*)trim(rname)//&
     "fixed MMR profile from txt file ?"
     haze_proffix=.false. ! default value
     call getin_p("haze_proffix",haze_proffix)
     if (is_master)write(*,*)trim(rname)//&
     "haze_proffix = ",haze_proffix

     if (is_master)write(*,*)trim(rname)//&
     "Number mix ratio of haze particles for co clouds:"
     Nmix_co=100000. ! default value
     call getin_p("Nmix_co",Nmix_co)
     if (is_master)write(*,*)trim(rname)//&
     " Nmix_co = ",Nmix_co

     if (is_master)write(*,*)trim(rname)//&
     "Number mix ratio of haze particles for ch4 clouds:"
     Nmix_ch4=100000. ! default value
     call getin_p("Nmix_ch4",Nmix_ch4)
     if (is_master)write(*,*)trim(rname)//&
     " Nmix_ch4 = ",Nmix_ch4

!***************************************************************
     !! Surface properties

!*********** N2 *********************************

     if (is_master)write(*,*)trim(rname)//&
      "Mode for changing N2 albedo"
     mode_n2=0 ! default value
     call getin_p("mode_n2",mode_n2)
     if (is_master)write(*,*)trim(rname)//&
      " mode_n2 = ",mode_n2
     thres_n2ice=1. ! default value
     call getin_p("thres_n2ice",thres_n2ice)
     if (is_master)write(*,*)trim(rname)//&
      " thres_n2ice = ",thres_n2ice

     if (is_master)write(*,*)trim(rname)//&
      "Diff of N2 albedo with thickness"
     deltab=0. ! default value
     call getin_p("deltab",deltab)
     if (is_master)write(*,*)trim(rname)//&
      " deltab = ",deltab

     if (is_master)write(*,*)trim(rname)//&
      "albedo N2 beta "
     alb_n2b=0.7 ! default value
     call getin_p("alb_n2b",alb_n2b)
     if (is_master)write(*,*)trim(rname)//&
      " alb_n2b = ",alb_n2b

     if (is_master)write(*,*)trim(rname)//&
      "albedo N2 alpha "
     alb_n2a=0.7 ! default value
     call getin_p("alb_n2a",alb_n2a)
     if (is_master)write(*,*)trim(rname)//&
      " alb_n2a = ",alb_n2a

     if (is_master)write(*,*)trim(rname)//&
      "emis N2 beta "
     emis_n2b=0.7 ! default value
     call getin_p("emis_n2b",emis_n2b)
     if (is_master)write(*,*)trim(rname)//&
      " emis_n2b = ",emis_n2b

     if (is_master)write(*,*)trim(rname)//&
      "emis N2 alpha "
     emis_n2a=0.7 ! default value
     call getin_p("emis_n2a",emis_n2a)
     if (is_master)write(*,*)trim(rname)//&
      " emis_n2a = ",emis_n2a

!*********** CH4 *********************************

     if (is_master)write(*,*)trim(rname)//&
      "Mode for changing CH4 albedo"
     mode_ch4=0 ! default value
     call getin_p("mode_ch4",mode_ch4)
     if (is_master)write(*,*)trim(rname)//&
      " mode_ch4 = ",mode_ch4
     feedback_met=0 ! default value
     call getin_p("feedback_met",feedback_met)
     if (is_master)write(*,*)trim(rname)//&
      " feedback_met = ",feedback_met
     thres_ch4ice=1. ! default value, mm
     call getin_p("thres_ch4ice",thres_ch4ice)
     if (is_master)write(*,*)trim(rname)//&
      " thres_ch4ice = ",thres_ch4ice
     fdch4_latn=-1.
     fdch4_lats=0.
     fdch4_lone=0.
     fdch4_lonw=-1.
     fdch4_depalb=0.5
     fdch4_finalb=0.95
     fdch4_maxalb=0.99
     fdch4_ampl=200.
     fdch4_maxice=100.
     call getin_p("fdch4_latn",fdch4_latn)
     call getin_p("fdch4_lats",fdch4_lats)
     call getin_p("fdch4_lone",fdch4_lone)
     call getin_p("fdch4_lonw",fdch4_lonw)
     call getin_p("fdch4_depalb",fdch4_depalb)
     call getin_p("fdch4_finalb",fdch4_finalb)
     call getin_p("fdch4_maxalb",fdch4_maxalb)
     call getin_p("fdch4_maxice",fdch4_maxice)
     call getin_p("fdch4_ampl",fdch4_ampl)
     if (is_master)write(*,*)trim(rname)//&
      " Values for albedo feedback = ",fdch4_latn,&
     fdch4_lats,fdch4_lone,fdch4_lonw,fdch4_depalb,&
     fdch4_finalb,fdch4_maxalb,fdch4_maxice,fdch4_ampl

     if (is_master)write(*,*)trim(rname)//&
      "Latitude for diff albedo methane"
     metlateq=25. ! default value
     call getin_p("metlateq",metlateq)
     if (is_master)write(*,*)trim(rname)//&
      " metlateq = ",metlateq

     if (is_master)write(*,*)trim(rname)//&
      "Ls1 and Ls2 for change of ch4 albedo"
     metls1=-1. ! default value
     metls2=-2. ! default value
     call getin_p("metls1",metls1)
     call getin_p("metls2",metls2)

     if (is_master)write(*,*)trim(rname)//&
      "albedo CH4 "
     alb_ch4=0.5 ! default value
     call getin_p("alb_ch4",alb_ch4)
     if (is_master)write(*,*)trim(rname)//&
      " alb_ch4 = ",alb_ch4

     if (is_master)write(*,*)trim(rname)//&
      "albedo equatorial CH4 "
     alb_ch4_eq=alb_ch4 ! default value
     call getin_p("alb_ch4_eq",alb_ch4_eq)
     if (is_master)write(*,*)trim(rname)//&
      " alb_ch4_eq = ",alb_ch4_eq

     if (is_master)write(*,*)trim(rname)//&
      "albedo south hemis CH4 "
     alb_ch4_s=alb_ch4 ! default value
     call getin_p("alb_ch4_s",alb_ch4_s)
     if (is_master)write(*,*)trim(rname)//&
      " alb_ch4_s = ",alb_ch4_s

     if (is_master)write(*,*)trim(rname)//&
      "emis CH4 "
     emis_ch4=0.5 ! default value
     call getin_p("emis_ch4",emis_ch4)
     if (is_master)write(*,*)trim(rname)//&
      " emis_ch4 = ",emis_ch4

     if (is_master)write(*,*)trim(rname)//&
      "CH4 lag for n2 sublimation limitation"
     ch4lag=.false. ! default value
     latlag=-90. ! default value
     vmrlag=1. ! default value
     call getin_p("ch4lag",ch4lag)
     call getin_p("latlag",latlag)
     call getin_p("vmrlag",vmrlag)
     if (is_master)write(*,*)trim(rname)//&
      " ch4lag = ",ch4lag
     if (is_master)write(*,*)trim(rname)//&
      " latlag = ",latlag
     if (is_master)write(*,*)trim(rname)//&
      " vmrlag = ",vmrlag

     if (is_master)write(*,*)trim(rname)//&
      "Max temperature for surface ?"
     tsurfmax=.false. ! default value
     albmin_ch4=0.3 ! default value
     call getin_p("tsurfmax",tsurfmax)
     call getin_p("albmin_ch4",albmin_ch4)
     if (is_master)write(*,*)trim(rname)//&
      " tsurfmax = ",tsurfmax
     if (is_master)write(*,*)trim(rname)//&
      " albmin_ch4 = ",albmin_ch4


     if (is_master)write(*,*)trim(rname)//&
     "fixed gaseous CH4 mixing ratio for rad transfer ?"
     ch4fix=.false. ! default value
     call getin_p("ch4fix",ch4fix)
     if (is_master)write(*,*)trim(rname)//&
       " ch4fix = ",ch4fix
     if (is_master)write(*,*)trim(rname)//&
       "fixed gaseous CH4 mixing ratio for rad transfer ?"
     vmrch4fix=0.5 ! default value
     call getin_p("vmrch4fix",vmrch4fix)
     if (is_master)write(*,*)trim(rname)//&
        " vmrch4fix = ",vmrch4fix
     vmrch4_proffix=.false. ! default value
     call getin_p("vmrch4_proffix",vmrch4_proffix)
     if (is_master)write(*,*)trim(rname)//&
        " vmrch4_proffix = ",vmrch4_proffix


!*********** CO *********************************

     if (is_master)write(*,*)trim(rname)//&
      "albedo CO "
     alb_co=0.4 ! default value
     call getin_p("alb_co",alb_co)
     if (is_master)write(*,*)trim(rname)//&
      " alb_co = ",alb_co
     thres_coice=1. ! default value, mm
     call getin_p("thres_coice",thres_coice)
     if (is_master)write(*,*)trim(rname)//&
      " thres_coice = ",thres_coice

     if (is_master)write(*,*)trim(rname)//&
      "emis CO "
     emis_co=0.4 ! default value
     call getin_p("emis_co",emis_co)
     if (is_master)write(*,*)trim(rname)//&
      " emis_co = ",emis_co

!*********** THOLINS *********************************
     if (is_master)write(*,*)trim(rname)//&
      "Mode for changing tholins albedo/emis"
     mode_tholins=0 ! default value
     call getin_p("mode_tholins",mode_tholins)
     if (is_master)write(*,*)trim(rname)//&
      " mode_tholins = ",mode_tholins

     if (is_master)write(*,*)trim(rname)//&
      "albedo tho "
     alb_tho=0.1 ! default value
     call getin_p("alb_tho",alb_tho)
     if (is_master)write(*,*)trim(rname)//&
      " alb_tho = ",alb_tho

     if (is_master)write(*,*)trim(rname)//&
      "albedo tho eq"
     alb_tho_eq=0.1 ! default value
     call getin_p("alb_tho_eq",alb_tho_eq)
     if (is_master)write(*,*)trim(rname)//&
      " alb_tho_eq = ",alb_tho_eq

     if (is_master)write(*,*)trim(rname)//&
      "emis tholins "
     emis_tho=1. ! default value
     call getin_p("emis_tho",emis_tho)
     if (is_master)write(*,*)trim(rname)//&
      " emis_tho = ",emis_tho

     if (is_master)write(*,*)trim(rname)//&
      "emis tholins eq"
     emis_tho_eq=1. ! default value
     call getin_p("emis_tho_eq",emis_tho_eq)
     if (is_master)write(*,*)trim(rname)//&
      " emis_tho_eq = ",emis_tho_eq

     if (is_master)write(*,*)trim(rname)//&
      "Latitude for diff albedo tholins"
     tholateq=25. ! default value
     call getin_p("tholateq",tholateq)
     if (is_master)write(*,*)trim(rname)//&
      " tholateq = ",tholateq
     tholatn=-1.
     tholats=0.
     tholone=0.
     tholonw=-1.
     alb_tho_spe=0.1 ! default value
     emis_tho_spe=1. ! default value
     call getin_p("  tholatn",tholatn)
     call getin_p("  tholats",tholats)
     call getin_p("  tholone",tholone)
     call getin_p("  tholonw",tholonw)
     if (is_master)write(*,*)trim(rname)//&
      " Values for special tholins albedo = ",tholatn,&
        tholats,tholone,tholonw,alb_tho_spe,emis_tho_spe

     if (is_master)write(*,*)trim(rname)//&
      "Specific albedo"
     spelon1=-180. ! default value
     spelon2=180. ! default value
     spelat1=-90. ! default value
     spelat2=90. ! default value
     specalb=.false. ! default value
     if (is_master)write(*,*)trim(rname)//&
      "albedo/emis spe "
     albspe=0.1 ! default value
     emispe=1. ! default value
     call getin_p("spelon1",spelon1)
     call getin_p("spelon2",spelon2)
     call getin_p("spelat1",spelat1)
     call getin_p("spelat2",spelat2)
     call getin_p("specalb",specalb)
     call getin_p("albspe",albspe)
     call getin_p("emispe",emispe)

     if (is_master)write(*,*)trim(rname)//&
      " specific = ",specalb

!********************** TI *****************************

     if (is_master)write(*,*)trim(rname)//&
      "Change TI with time"
     changeti=.false. ! default value
     call getin_p("changeti",changeti)
     if (is_master)write(*,*)trim(rname)//&
      " changeti = ",changeti
     changetid=.false. ! default value for diurnal TI
     call getin_p("changetid",changetid)
     if (is_master)write(*,*)trim(rname)//&
      " changetid = ",changetid

     if (is_master)write(*,*)trim(rname)//&
      "IT N2 "
     ITN2=800. ! default value
     call getin_p("ITN2",ITN2)
     if (is_master)write(*,*)trim(rname)//&
      " ITN2 = ",ITN2
     ITN2d=20. ! default value
     call getin_p("ITN2d",ITN2d)
     if (is_master)write(*,*)trim(rname)//&
      " ITN2d = ",ITN2d

     if (is_master)write(*,*)trim(rname)//&
      "IT CH4"
     ITCH4=800. ! default value
     call getin_p("ITCH4",ITCH4)
     if (is_master)write(*,*)trim(rname)//&
      " ITCH4 = ",ITCH4
     ITCH4d=20. ! default value
     call getin_p("ITCH4d",ITCH4d)
     if (is_master)write(*,*)trim(rname)//&
      " ITCH4d = ",ITCH4d

     if (is_master)write(*,*)trim(rname)//&
      "IT H2O"
     ITH2O=800. ! default value
     call getin_p("ITH2O",ITH2O)
     if (is_master)write(*,*)trim(rname)//&
      " ITH2O = ",ITH2O
     ITH2Od=20. ! default value
     call getin_p("ITH2Od",ITH2Od)
     if (is_master)write(*,*)trim(rname)//&
      " ITH2Od = ",ITH2Od

!************** COOLING ***************

     alpha_top=5e-11 ! default value
     call getin_p("alpha_top",alpha_top)
     if (is_master)write(*,*)trim(rname)//&
      " alpha_top = ",alpha_top
     pref=0.02 ! default value
     call getin_p("pref",pref)
     if (is_master)write(*,*)trim(rname)//&
      " pref = ",pref
     deltap=0.1 ! default value
     call getin_p("deltap",deltap)
     if (is_master)write(*,*)trim(rname)//&
      " deltap = ",deltap

!=================================

     if (is_master) write(*,*)trim(rname)//&
       ": Is the variable gas species radiatively active?"
     Tstrat=167.0
     varactive=.false.
     call getin_p("varactive",varactive)
     if (is_master) write(*,*)trim(rname)//": varactive = ",varactive

     if (is_master) write(*,*)trim(rname)//&
       ": Is the variable gas species distribution set?"
     varfixed=.false.
     call getin_p("varfixed",varfixed)
     if (is_master) write(*,*)trim(rname)//": varfixed = ",varfixed

! Test of incompatibility:
! if varactive, then varfixed should be false
     if (varactive.and.varfixed) then
       call abort_physic(rname,'if varactive, varfixed must be OFF!',1)
     endif

     if (is_master) write(*,*)trim(rname)//": Gravitationnal sedimentation ?"
     sedimentation=.false. ! default value
     call getin_p("sedimentation",sedimentation)
     if (is_master) write(*,*)trim(rname)//": sedimentation = ",sedimentation

     if (is_master) write(*,*)trim(rname)//": Spectral Dependant albedo ?"
     albedo_spectral_mode=.false. ! default value
     call getin_p("albedo_spectral_mode",albedo_spectral_mode)
     if (is_master) write(*,*)trim(rname)//&
     ": albedo_spectral_mode = ",albedo_spectral_mode

     if (is_master) then
       write(*,*)trim(rname)//": Snow albedo ?"
       write(*,*)trim(rname)//": If albedo_spectral_mode=.true., then this "
       write(*,*)trim(rname)//": corresponds to the 0.5 microns snow albedo."
     endif
     albedosnow=0.5         ! default value
     call getin_p("albedosnow",albedosnow)
     if (is_master) write(*,*)trim(rname)//": albedosnow = ",albedosnow

     if (is_master) write(*,*)trim(rname)//": N2 ice albedo ?"
     albedon2ice=0.5       ! default value
     call getin_p("albedon2ice",albedon2ice)
     if (is_master) write(*,*)trim(rname)//": albedon2ice = ",albedon2ice

     if (is_master) write(*,*)trim(rname)//": Maximum ice thickness ?"
     maxicethick=2.0         ! default value
     call getin_p("maxicethick",maxicethick)
     if (is_master) write(*,*)trim(rname)//": maxicethick = ",maxicethick

     if (is_master) write(*,*)trim(rname)//": Minimum eddy mix coeff in 1D ?"
     kmixmin=1.0e-2         ! default value
     call getin_p("kmixmin",kmixmin)
     if (is_master) write(*,*)trim(rname)//": kmixmin = ",kmixmin
     kmix_proffix=.false.  ! default value
     call getin_p("kmix_proffix",kmix_proffix)
     if (is_master) write(*,*)trim(rname)//": kmix_proffix = ",kmix_proffix

     if (is_master) write(*,*)'Predefined Cp from dynamics is ',cpp,'J kg^-1 K^-1'
     if (is_master) write(*,*)'Predefined Mg from dynamics is ',mugaz,'amu'

     force_cpp=.false. ! default value
     call getin_p("force_cpp",force_cpp)
     if (force_cpp) then
      if (is_master) write(*,*)trim(rname)//": force_cpp = ",force_cpp
      if (is_master) write(*,*)trim(rname)//": force_cpp is deprecated.",&
      "Set cpp_mugaz_mode=1 in callfis to emulate force_cpp=.true."
      call abort_physic(rname,"Anyway, you need to set force_cpp=.false. to continue.",1)
     endif

     if (is_master) write(*,*)trim(rname)//&
     ": where do you want your cpp/mugaz value to come from?",&
     "=> 0: dynamics (3d), 1: forced in callfis (1d), 2: computed from gases.def (1d)?"
     cpp_mugaz_mode = 0 ! default value
     call getin_p("cpp_mugaz_mode",cpp_mugaz_mode)
     if (is_master) write(*,*)trim(rname)//": cpp_mugaz_mode = ",cpp_mugaz_mode


! Test of incompatibility:

     if ((.not.tracer).and.(haze)) then
       call abort_physic(rname, 'if haze are on, tracers must be on!', 1)
     endif
     if ((callmufi).and.(haze)) then
       call abort_physic(rname, 'if haze are on, microphysics should be deactivated!', 1)
     endif
     if ((haze).and.(naerkind.gt.1)) then
      call abort_physic(rname, 'if haze are on, naerkind must be equal to 1!', 1)
     endif
     if ((callmufi).and..not.(naerkind.gt.1)) then
      call abort_physic(rname, 'if microphysics is on, naerkind must be > 1!', 1)
     endif
     if ((callmufi).and..not.(callmuclouds).and.(naerkind.gt.2)) then
      call abort_physic(rname, 'Warning: here microphysical clouds are on, naerkind must be = 2!', 1)
     endif
     if ((callmufi).and.(callmuclouds).and..not.(naerkind.gt.2)) then
      call abort_physic(rname, 'if microphysical clouds are on, naerkind must be > 2!', 1)
     endif
     if (.not.(callmufi.or.haze).and.(optichaze)) then
      call abort_physic(rname, 'if microphysics and haze are off, optichaze must be deactivated!', 1)
     endif
     if ((callmufi.and.call_haze_prod_pCH4).and..not.(methane)) then
      call abort_physic(rname, 'if haze production from CH4 photolysis is on, methane must be activated!', 1)
     endif
     if (haze_proffix.and.sedimentation) then
         call abort_physic(rname, 'if haze profile is set, sedimentation must be deactivated', 1)
     endif
     if (callmolvis.and..not.callconduct) then
         call abort_physic(rname, 'if callmolvis is set, callconduct must be true', 1)
     endif
     if (glaflow.and..not.fast) then
         call abort_physic(rname, 'if glaflow is set, fast must be true', 1)
     endif
     if (paleo.and..not.fast) then
         call abort_physic(rname, 'if paleo is set, fast must be true', 1)
     endif
     if ((haze_proffix.or.haze_radproffix).and..not.optichaze) then
         call abort_physic(rname, 'for now, haze/rad_proffix only works with optichaze=T', 1)
     endif
     ! if (carbox.and.condcosurf.and.no_n2frost) then
     !   call abort_physic(rname, "CO surface condensation and no_n2frost are both active which may not be relevant", 1)
     ! end if

     if ((cpp_mugaz_mode >= 1).and.(is_master).and.(ngrid>1)) then
        write(*,*)'    !!!!! Be aware that having different values for cpp and mugaz in the dynamics and physics'
        write(*,*)'    in 3D can result in very pathological behavior. You have been warned !!!!!'
      else if ((cpp_mugaz_mode == 0).and.(is_master).and.(ngrid==1)) then
        ! for this specific 1D error we will remove run.def before aborting JL22
        call system("rm -rf run.def")
        call abort_physic(rname,"cpp_mugaz_mode must be >= 1 in 1d",1)
     endif

     if (cpp_mugaz_mode == 1) then
       mugaz = -99999.
       if (is_master) write(*,*)trim(rname)//&
         ": MEAN MOLECULAR MASS in g mol-1 ?"
       call getin_p("mugaz",mugaz)
       IF (mugaz.eq.-99999.) THEN
         call abort_physic(rname,"mugaz must be set if cpp_mugaz_mode = 1",1)
       ENDIF
       cpp = -99999.
       if (is_master) write(*,*)trim(rname)//&
         ": SPECIFIC HEAT CAPACITY in J K-1 kg-1 ?"
       call getin_p("cpp",cpp)
       IF (cpp.eq.-99999.) THEN
           PRINT *, "cpp must be set if cpp_mugaz_mode = 1"
           STOP
       ENDIF
       if (is_master) write(*,*)'New Cp from callfis is ',cpp,'J kg^-1 K^-1'
       if (is_master) write(*,*)'New Mg from callfis is ',mugaz,'amu'

     endif ! of if (cpp_mugaz_mode == 1)
     call su_gases
     call calc_cpp_mugaz

     if (is_master) then
       PRINT*,'--------------------------------------------'
       PRINT*
       PRINT*
     endif
  ELSE
     call abort_physic(rname,'Cannot read file callphys.def. Is it here ?',1)
  ENDIF ! of IF(iscallphys)

  if (is_master) then
    PRINT*
    PRINT*,'inifis: daysec',daysec
    PRINT*
    PRINT*,'inifis: The radiative transfer is computed:'
    PRINT*,'           each ',iradia,' physical time-step'
    PRINT*,'        or each ',iradia*dtphys,' seconds'
    PRINT*
  endif

!-----------------------------------------------------------------------
!     Some more initialization:
!     ------------------------

  ! Initializations for comgeomfi_h
#ifndef MESOSCALE
  totarea=SSUM(ngrid,parea,1)
  call planetwide_sumval(parea,totarea_planet)

  !! those are defined in comdiurn_h.F90
  IF (.not.ALLOCATED(sinlat)) ALLOCATE(sinlat(ngrid))
  IF (.not.ALLOCATED(coslat)) ALLOCATE(coslat(ngrid))
  IF (.not.ALLOCATED(sinlon)) ALLOCATE(sinlon(ngrid))
  IF (.not.ALLOCATED(coslon)) ALLOCATE(coslon(ngrid))

  DO ig=1,ngrid
     sinlat(ig)=sin(plat(ig))
     coslat(ig)=cos(plat(ig))
     sinlon(ig)=sin(plon(ig))
     coslon(ig)=cos(plon(ig))
  ENDDO
#endif
  ! initialize variables in radinc_h
  call ini_radinc_h(nlayer,tplanckmin,tplanckmax,dtplanck)

  ! initialize variables and allocate arrays in radcommon_h
  call ini_radcommon_h(naerkind)

  ! allocate "comsoil_h" arrays
  call ini_comsoil_h(ngrid)

  ! allocate arrays in "nonoro_gwd_ran_mod"
  call end_nonoro_gwd_ran
  call ini_nonoro_gwd_ran(ngrid,nlayer)

  ! allocate arrays in "nonoro_gwd_mix_mod"
  call end_nonoro_gwd_mix
  call ini_nonoro_gwd_mix(ngrid,nlayer,nq)

  END SUBROUTINE inifis

END MODULE inifis_mod
