MODULE display
!-----------------------------------------------------------------------
! NAME
!     display
!
! DESCRIPTION
!     Contains wrappers to print information.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use, intrinsic :: iso_fortran_env, only: stdout => output_unit
use, intrinsic :: iso_fortran_env, only: stderr => error_unit
use, intrinsic :: iso_fortran_env, only: stdin => input_unit
use numerics,                      only: dp, di, li, k4, minieps

! DECLARATION
! -----------
implicit none

! VARIABLES
! ---------
character(128) :: dir = ' '      ! Current directory
character(32)  :: logname = ' '  ! User name
character(32)  :: hostname = ' ' ! Machine/station name

contains
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

!=======================================================================
SUBROUTINE print_ini()
!-----------------------------------------------------------------------
! NAME
!     print_ini
!
! DESCRIPTION
!     Print the initial display for the PEM.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! LOCAL VARIABLES
! ---------------
character(8)              :: date
character(10)             :: time
character(5)              :: zone
integer(di), dimension(8) :: values

! CODE
! ----
write(stdout,*) '  *    .          .   +     .    *        .  +      .    .       .      '
write(stdout,*) '           +         _______  ________  ____    ____      *           + '
write(stdout,*) ' +   .        *     |_   __ \|_   __  ||_   \  /   _|          .       *'
write(stdout,*) '          .     .     | |__) | | |_ \_|  |   \/   |  *        *      .  '
write(stdout,*) '       .              |  ___/  |  _| _   | |\  /| |      .        .     '
write(stdout,*) '.  *          *      _| |_    _| |__/ | _| |_\/_| |_                  * '
write(stdout,*) '            +       |_____|  |________||_____||_____|   +     .         '
write(stdout,*) '  .      *          .   *       .   +       *          .        +      .'

! Some user info
call date_and_time(date,time,zone,values)
call getcwd(dir)
call getlog(logname)
call hostnm(hostname)
write(stdout,*)
write(stdout,*) '********* PEM information *********'
write(stdout,*)                     '+ User     : '//trim(logname)
write(stdout,*)                     '+ Machine  : '//trim(hostname)
write(stdout,*)                     '+ Directory: '//trim(dir)
write(stdout,'(a,i2,a,i2,a,i4)')   ' + Date     : ',values(3),'/',values(2),'/',values(1)
write(stdout,'(a,i2,a,i2,a,i2,a)') ' + Time     : ',values(5),':',values(6),':',values(7)
write(stdout,*)
write(stdout,*) '********* Initialization *********'

END SUBROUTINE print_ini
!=======================================================================

!=======================================================================
SUBROUTINE print_end(n_yr_run,dur_secs,r_plnt2earth_yr,pem_ini_date,n_yr_sim)
!-----------------------------------------------------------------------
! NAME
!     print_end
!
! DESCRIPTION
!     Print the ending display for the PEM.
!
! AUTHORS & DATE
!     JB Clement, 01/2026
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use utility,         only: format_duration, int2str
use workflow_status, only: i_pem_run

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), intent(in) :: r_plnt2earth_yr, pem_ini_date, n_yr_sim, n_yr_run, dur_secs

! CODE
! ----
write(stdout,*)
write(stdout,*) '****** PEM final information *******'
write(*,'(a,f16.4,a)')         ' + The run PEM #'//int2str(i_pem_run)//' achieved ', n_yr_run, ' Planetary years, completed in '//format_duration(dur_secs)//'.'
write(*,'(a,f16.4,a,f16.4,a)') ' + The workflow has achieved ', n_yr_sim, ' Planetary years =', n_yr_sim*r_plnt2earth_yr, ' Earth years.'
write(*,'(a,f16.4,a)')         ' + The reached date is now ', (pem_ini_date + n_yr_sim)*r_plnt2earth_yr, ' Earth years.'
write(*,*)                      '+ PEM: so far, so good!'
write(stdout,*) '************************************'
call but_why(n_yr_run)

END SUBROUTINE print_end
!=======================================================================

!=======================================================================
SUBROUTINE print_msg(message,frmt)
!-----------------------------------------------------------------------
! NAME
!     print_msg
!
! DESCRIPTION
!     Print a simple message (string).
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
character(*), intent(in)           :: message
character(*), intent(in), optional :: frmt

! CODE
! ----
if (present(frmt)) then
    write(stdout,frmt) message
else
    write(stdout,*) message
end if

END SUBROUTINE print_msg
!=======================================================================

!=======================================================================
SUBROUTINE print_err(message,frmt)
!-----------------------------------------------------------------------
! NAME
!     print_err
!
! DESCRIPTION
!     Print a simple message (string).
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
character(*), intent(in)           :: message
character(*), intent(in), optional :: frmt

! CODE
! ----
if (present(frmt)) then
    write(stderr,frmt) message
else
    write(stderr,*) message
end if

END SUBROUTINE print_err
!=======================================================================

!=======================================================================
SUBROUTINE but_why(n_yr_run)
!-----------------------------------------------------------------------
! NAME
!     but_why
!
! DESCRIPTION
!     Print a cool surprise.
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!     \(^o^)/
!     A rare egg may be hidden somewhere in this module... True trainers
!     will know better than to consult an AI to hatch it.
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), intent(in) :: n_yr_run

! LOCAL VARIABLES
! ---------------
integer(di), dimension(5,50), parameter  :: exeggcute = transpose(reshape([ &
                                                        87,104,97,116,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        68,69,65,68,66,69,65,84,45,80,76,65,78,69,84,32,105,115,32,101,118,111,108,118,105,110,103,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        84,97,45,100,97,32,46,46,46,32,84,97,45,100,97,32,46,46,46,32,84,97,45,100,97,32,46,46,46,32,84,97,32,116,97,45,100,97,32,116,97,45,100,97,33,0,0,0,0,0,0, &
                                                        67,111,110,103,114,97,116,117,108,97,116,105,111,110,115,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        89,111,117,114,32,68,69,65,68,66,69,65,84,45,80,76,65,78,69,84,32,101,118,111,108,118,101,100,32,105,110,116,111,32,70,65,66,85,76,79,85,83,45,80,76,65,78,69,84,33 &
                                                        ],[50,5]))
integer(di), dimension(20,42), parameter :: exeggutor = transpose(reshape([ &
                                                        73,32,119,97,110,110,97,32,98,101,32,116,104,101,32,118,101,114,121,32,98,101,115,116,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        76,105,107,101,32,110,111,32,109,111,100,101,108,32,101,118,101,114,32,119,97,115,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        84,111,32,98,117,105,108,100,32,116,104,101,109,32,114,105,103,104,116,44,32,116,104,97,116,226,128,153,115,32,109,121,32,114,101,97,108,32,116,101,115,116,0,0, &
                                                        84,111,32,115,105,109,117,108,97,116,101,32,116,104,101,109,32,105,115,32,109,121,32,99,97,117,115,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        73,32,119,105,108,108,32,116,114,97,118,101,108,32,103,114,105,100,115,32,97,110,100,32,108,97,110,100,115,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        70,114,111,109,32,99,111,114,101,32,116,111,32,115,107,121,32,115,111,32,119,105,100,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        84,101,97,99,104,32,112,108,97,110,101,116,115,32,104,111,119,32,116,111,32,117,110,100,101,114,115,116,97,110,100,0,0,0,0,0,0,0,0,0,0,0, &
                                                        84,104,101,32,112,104,121,115,105,99,115,32,100,101,101,112,32,105,110,115,105,100,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        80,76,65,78,69,84,83,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        71,111,116,116,97,32,109,111,100,101,108,32,116,104,101,109,32,97,108,108,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        73,116,226,128,153,115,32,115,99,105,101,110,99,101,32,97,110,100,32,99,111,100,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        73,32,107,110,111,119,32,105,116,226,128,153,115,32,109,121,32,99,97,108,108,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        80,76,65,78,69,84,83,33,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        70,114,111,109,32,100,117,115,116,32,116,111,32,103,108,111,119,105,110,103,32,115,112,104,101,114,101,115,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        87,105,116,104,32,116,105,109,101,32,97,110,100,32,114,101,115,111,108,118,101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, &
                                                        73,226,128,153,108,108,32,101,118,111,108,118,101,32,116,104,101,109,32,116,104,114,111,117,103,104,32,121,101,97,114,115,0,0,0,0,0,0,0,0,0,0, &
                                                        32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,226,128,148,32,74,66,67,0,0,0,0 &
                                                        ],[42,20]))
real(dp),                      parameter :: first_gen = 151._dp
character(256), dimension(59)            :: why_not
character(256), dimension(11)            :: surprise
integer(di)                              :: i

! CODE
! ----
why_not(37) = '⠀⠀⠀⠀⠈⢧⣭⠡⣤⢿⣿⣟⣷⡀⠀⢧⣻⡇⠘⣿⣟⡇⠀⠀⠀⠀⠀⠀⠀⠙⢿⣿⢿⣿⣿⣷⣤⣈⠙⠿⠿⣿⣿⣿⣥⣯⣤⣾⡿⠚⠁⠀⢻⣷⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(12) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠹⡀⠀⠀⠀⠀⣼⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣻⣽⣿⣿⣿⣧⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(55) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠙⠓⠦⣤⣀⠀⠉⠉⠉⠀⠀⠉⠉⠉⠍⠀⠀⣀⣤⠴⠀⠉⠛⠂⢿⣷⣦⣌⣙⠛⠻⠿⠿⠿⠷⠙⠛⢁⣴⣿⣷⡇⠀'
why_not(4)  = '⠀⠀⠀⠀⢀⡴⠚⡉⠍⢉⣉⣉⡁⠒⢂⠄⣀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(29) = '⠀⠀⢀⣠⣶⠶⣶⣶⣶⣶⣤⣄⣀⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(46) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠘⡎⢇⢻⣿⣇⠘⠆⠱⢄⠀⠙⢿⣦⣈⠻⣿⣮⣛⣷⣄⠑⠶⣶⣦⣐⣮⣽⣿⣿⣿⣯⣶⣾⠻⡿⢟⡹⠃⠀⢁⠀⠀⠀⣠⣶⠇⡀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(8)  = '⠀⠀⠀⠀⠀⠱⡠⢀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣠⣴⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣷⣄⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(21) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠁⠉⠛⠋⠛⢟⡚⠍⠉⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠴⡀⠀⠀⠀⠀'
why_not(3)  = ' '
why_not(41) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⢰⡄⠻⣿⣄⠙⣿⣷⡹⣦⡈⠻⣿⣷⡌⠙⢿⣷⣦⣀⠂⢄⣀⡀⠀⠀⠀⡉⠲⠶⠶⠟⠻⠿⣷⣾⣿⣿⣱⠟⠉⠉⠐⢀⣴⢯⣷⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(18) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠺⣑⠮⡝⢷⡞⡘⢎⠼⡙⢬⠓⡬⠣⢄⠩⢐⠡⡘⡰⢭⣞⡿⠋⠀⠀⠀⠀⠀⠀⠀⠢⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(52) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠳⣄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⠂⠀⠀⠀⠀⠀⠀⠛⢿⣿⣷⡽⡢⠙⢿⣿⣿⣿⣶⣤⣀⠀⠬⢄⣀⣀⣠⠔⢁⣾⣿⠇⣰⡄⢿⢀⡏⡇'
why_not(9)  = '⠀⠀⠀⠀⠀⠀⠳⡡⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣠⣾⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣆⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(33) = '⠀⠸⡏⣾⣿⡄⢻⣿⣿⢱⣿⠁⡔⠋⠉⠉⠐⠀⣩⡴⠋⢹⠁⢸⣿⣜⢿⡌⢻⣦⣿⣷⣄⠙⠦⣽⣿⣿⣿⣿⣿⣿⣿⣿⣿⡿⢦⣄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(1)  = trim(confusion(exeggcute(1,:)))
why_not(58) = trim(confusion(exeggcute(4,:)))
why_not(26) = ' '
why_not(14) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⢄⢸⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣟⢿⣻⣿⡿⣿⣾⢷⣯⣿⣿⣿⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(47) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢻⠈⠢⡙⢿⢷⣤⣄⡀⠑⠠⡀⠛⢿⣷⣌⠛⢿⣷⣝⢿⣦⡈⠙⠛⠛⠛⠒⠒⠂⠀⠀⠈⠛⠛⠉⡠⠒⡉⠉⣦⣴⡴⠋⡾⣀⢿⣄⠀⠀⠀⠀⠀⠀⠀'
why_not(6)  = '⠀⠀⠀⠀⢹⢸⠘⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⠰⠠⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(40) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠙⢿⣧⡈⢿⣿⣝⢷⣄⠙⢧⡀⣈⠻⣿⣿⣵⠀⠈⠛⠦⡀⠀⠀⠀⠀⠈⠉⠉⠀⠀⠀⣠⢤⣦⣄⣸⣿⡿⣫⣽⣿⣯⠟⠀⣸⣾⡆⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(24) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠓⠢⢖⡠⠄⢀⠠⠤⠤⠤⠀⢂⠤⢁⡞⠀⠀⠀⠀'
why_not(35) = '⠀⠀⠘⣇⢻⣿⣇⠸⣿⣿⣿⡀⠇⠀⠀⡼⣿⣿⡇⠀⠀⠀⡀⠀⠻⣟⢿⣿⣿⣤⡀⠘⢿⣿⣿⣿⣿⣤⣄⣀⠀⠀⠀⠠⠄⠀⣀⣠⣿⣿⣿⣇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(10) = '⠀⠀⠀⠀⠀⠀⠀⠱⡁⡂⠀⠀⠀⠀⠀⠀⠀⢀⣼⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣷⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(59) = trim(confusion(exeggcute(5,:)))
why_not(16) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢻⣟⣾⡁⡧⣧⢵⣮⣵⣥⣼⣧⣥⢯⡾⣥⣤⡟⣯⢤⡛⢤⢎⣙⡽⣯⣿⠇⠀⠐⠠⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(44) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⣿⣿⣦⡀⠀⠀⠀⠙⢛⣦⡈⠻⣷⣎⡻⣦⡈⠻⣿⣿⣶⣯⣝⣿⣿⣿⣖⣲⣶⣤⣤⣤⣄⣤⣤⣴⣶⣶⣶⠻⣉⣠⡴⠟⠁⡰⣿⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(2)  = trim(confusion(exeggcute(2,:)))
why_not(31) = '⠀⣼⣿⣼⣿⠏⢠⣾⣿⣿⣿⣿⣿⣷⣶⣬⣝⡛⢷⣦⣄⠀⠀⠀⠀⣀⣤⢤⣴⣶⡶⠛⣛⢫⣿⣶⣾⣶⣦⣤⣀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(49) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠘⣯⠀⠀⢀⠰⣤⠙⣿⣷⣤⣀⠀⠀⠀⠀⠉⢳⣶⣌⠻⢿⣷⣽⡿⣦⣄⠀⠀⠀⣀⣠⣤⣤⣥⡴⣾⠿⠒⠁⣠⣾⠃⢂⢻⣿⣧⡙⣿⣧⠀⠀⠀⠀'
why_not(19) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠙⠶⣩⠖⣭⠻⣷⣨⡑⣢⢉⠔⡡⢊⢄⡃⣖⣡⢗⡯⠞⠁⠀⠀⠀⠀⠀⠀⠀⠀⠀⠑⡀⠀⠀⠀⠀⠀⠀'
why_not(27) = trim(confusion(exeggcute(3,:)))
why_not(53) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠳⣄⡈⠐⠠⣑⠢⢤⣄⣀⠁⠒⠒⠲⠦⠄⠀⠀⠀⠈⠛⢿⣷⣦⣌⠛⢿⣾⣿⣻⣿⣿⣿⣶⣦⡤⠤⠒⣐⣴⣿⡏⣿⡇⢸⣿⡆⠀'
why_not(7)  = '⠀⠀⠀⠀⠸⡄⡜⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣀⣥⣤⣶⣶⣶⣶⣶⣶⣦⣤⣀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(42) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⣿⠱⠀⠈⠻⣷⣌⠻⣿⣮⡻⣦⡈⠻⣿⣦⡈⠻⢿⣿⣿⣶⣍⣙⠷⠾⣿⣿⣷⣶⣶⣾⣿⠿⠿⠋⡩⠽⠛⠒⠁⠀⢀⣨⣴⢏⣻⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(25) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠉⠓⠒⠳⠬⠭⡥⠴⠞⠋⠀⠀⠀⠀⠀'
why_not(57) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠉⠙⠛⠛⠛⠉⠉⠁⠀⠀⠀'
why_not(11) = '⠀⠀⠀⠀⠀⠀⠀⠀⠘⢆⠀⠀⠀⠀⠀⠀⢀⣾⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(34) = '⠀⠀⢻⡸⣿⣷⠈⣿⣿⣟⣿⠀⠀⠀⠀⠀⣠⣾⣇⠀⠀⠈⠀⠸⣿⣿⣿⣾⡢⠙⠻⣿⣿⣿⣦⣈⠙⠛⠿⢶⣭⣍⠉⠯⠭⠔⠃⠈⣻⣧⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(20) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠙⠶⣋⢮⣓⢷⡴⡍⢾⣔⣫⢦⣽⡶⠟⠋⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠒⡀⠀⠀⠀⠀⠀'
why_not(45) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⢸⢷⠻⣯⢦⠀⠀⠀⠀⠙⢿⣦⡈⠻⣿⣮⡻⣦⣀⠙⠪⠝⠛⠿⢿⣿⣿⣿⣿⣿⣿⣿⣿⡿⢟⣻⣿⣿⣷⠿⢛⠉⠀⠀⠈⠀⡿⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(5)  = '⠀⠀⠀⠀⣼⢁⠎⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠈⠒⠠⢀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(50) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⢧⠀⠈⠢⣙⢷⡜⢧⡉⠛⠿⣦⡀⠀⢄⠈⠛⢧⡷⢆⡙⠻⣿⣿⣿⣿⣦⣄⡉⠉⠉⠉⠀⠈⠀⣠⠴⠋⡴⠁⠀⠀⡄⣿⣿⣧⠘⣿⡄⠀⠀⠀'
why_not(28) = ' '
why_not(17) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⢻⡽⣹⣷⢮⡓⠮⣝⢾⣹⠽⣞⢧⡝⢶⣡⠛⡤⢃⠎⡱⢊⢶⣹⣿⠟⠀⠀⠀⠀⠐⢀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(39) = '⠀⠀⠀⠀⠀⠀⠀⠹⣿⡄⣒⠈⣿⣯⡻⣦⡈⢻⣎⠈⠻⣿⣿⣦⣄⠈⠢⣤⡀⠀⠀⠀⠀⠀⢌⡙⠒⠀⠈⠙⠛⠛⠛⠛⠛⠉⣠⣤⣦⣤⣤⣿⠟⠀⢿⡄⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(13) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠘⢆⠄⠀⢠⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣽⣿⡽⣾⣿⣿⣿⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(56) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠉⠙⠛⠒⠲⠒⠶⠒⠒⠛⠋⠉⠁⠀⠀⠀⠀⠀⠀⠀⠈⠙⠻⠿⢿⣿⣶⣶⣶⣶⣶⡿⠿⣿⡿⠋⠀⠀'
why_not(22) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠉⠚⢴⡠⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠐⢠⢃⠀⠀⠀⠀'
why_not(32) = '⠀⢹⢻⢻⣿⠀⣿⣿⣿⡿⣭⣶⠾⠟⠺⠿⢿⣿⣷⣦⠝⣫⡴⢺⣿⢿⡇⢻⣿⢿⣧⠀⢿⣮⡿⣿⣿⣿⣿⣿⣿⣿⣷⣦⣄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(54) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠙⠦⣄⡀⠉⠓⠢⣭⣿⣷⣶⣾⣿⣟⣛⣓⣀⣀⣀⣠⡌⠙⠻⣿⣦⣤⠙⠛⠿⣶⣭⣿⣛⡿⠿⠿⠿⠟⣫⣾⡿⠁⣼⡿⡇⠀'
why_not(15) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠹⣷⡟⢿⣏⡟⡿⣻⢿⡫⡟⣿⣯⡟⣿⢭⣯⡏⣦⡛⣽⢝⡻⣝⣿⣷⣿⣿⡧⠃⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(36) = '⠀⠀⠀⠘⣇⠻⣿⣄⠻⣿⣿⣷⡈⠀⠞⡁⣿⡜⣿⣶⢦⠀⠐⠀⠀⠙⢮⡛⢿⣿⣿⣷⣦⣌⡙⠻⢿⣿⣿⣿⣿⣿⣶⣶⣾⣿⣿⣿⣿⡿⡩⢷⡄⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(48) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⣷⡀⠹⡀⠙⠲⣬⣛⢦⡀⠀⠀⠀⠈⠻⣷⣄⠙⠻⣿⣮⡻⣶⣄⡉⠀⠈⠃⠀⠀⠀⠀⠀⠀⠀⠰⠏⢂⣼⠟⠋⠀⣼⢡⣿⣦⠻⣮⣳⡀⠀⠀⠀⠀'
why_not(23) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⠑⠣⣄⡄⡀⢀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⡈⠰⢸⠀⠀⠀⠀'
why_not(38) = '⠀⠀⠀⠀⠀⠈⢻⣷⡉⣤⠻⣿⣿⣿⣄⠈⣿⡿⡌⢾⣿⣧⡀⠀⠀⡀⠀⠀⠀⠀⠀⠙⠷⣮⣙⠛⠿⣿⣿⣿⣿⣿⣿⣿⠿⠟⣋⡀⠀⠀⠁⣠⣾⣿⣇⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(30) = '⠀⢠⣿⢟⣵⣿⡿⠛⠛⠛⠛⠛⠻⠿⣷⣦⣤⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(43) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⣿⣄⠁⠀⠀⠉⢿⣧⡈⠻⣿⣮⡻⣦⡈⠻⣿⣦⣕⡿⢿⣶⣭⣉⡛⠒⠦⠄⠉⠉⠑⠛⠻⠗⠒⠀⢀⣤⠶⢀⣴⡾⣿⠟⣡⠏⣸⠀⠀⠀⠀⠀⠀⠀⠀⠀'
why_not(51) = '⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠈⣷⡄⠀⠀⠉⠁⠀⠙⣦⠀⠂⡍⠑⠀⠉⠀⠀⠉⢻⠓⣶⡄⠈⠹⣿⣿⣿⣿⣶⣤⠀⠀⠐⠉⠀⢠⡞⠁⠀⠀⠀⡇⢸⣿⡟⣧⢱⢸⣿⡆⠀'

surprise(10) = '⠀⠀⠀⠘⣿⣿⣿⣿⣿⣿⣿⣷⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡆⠀'
surprise(9)  = '⠀⠀⠀⢿⣶⣿⣿⣿⣿⣿⡻⣿⡿⣿⣿⣿⣿⣶⣶⣾⣿⣿⠀⠀'
surprise(3)  = '⠀⠘⢿⣿⣿⣿⣿⣦⣀⣀⣀⣄⣀⣀⣠⣀⣤⣶⣿⣿⣿⣿⣿⠇'
surprise(11) = '⠀⠀⠀⠀⣼⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡇⠀'
surprise(5)  = '⠀⠀⠀⠀⣰⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣟⠋⠀⠀⠀'
surprise(8)  = '⠀⠀⠀⡁⠀⠈⣿⣿⣿⣿⢟⣛⡻⣿⣿⣿⣟⠀⠀⠈⣿⡇⠀⠀'
surprise(6)  = '⠀⠀⠀⢠⣿⣿⡏⠆⢹⣿⣿⣿⣿⣿⣿⠒⠈⣿⣿⣿⣇⠀⠀⠀'
surprise(4)  = '⠀⠀⠈⠻⣿⢿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⣿⡿⠋⠀'
surprise(7)  = '⠀⠀⠀⣼⣿⣿⣷⣶⣿⣿⣛⣻⣿⣿⣿⣶⣾⣿⣿⣿⣿⡀⠀⠀'
surprise(2)  = '⠀⣿⣿⣿⣷⣤⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⣀⣤⣶⣾⣿'
surprise(1)  = '⢰⣶⣤⡀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⠀⢀'

if (gottacatch_emall_()) then
    write(stdout,*)
    if (abs(n_yr_run - first_gen) < minieps) then
        do i = 1,size(surprise)
            write(stdout,*) trim(surprise(i))
        end do
        write(stdout,*) ' '
        do i = 1,size(exeggutor,1)
            write(stdout,*) trim(confusion(exeggutor(i,:)))
        end do
    else
        do i = 1,size(why_not)
            write(stdout,*) trim(why_not(i))
        end do
    end if
end if

END SUBROUTINE but_why
!=======================================================================

!=======================================================================
FUNCTION gottacatch_emall_() RESULT(flag)
!-----------------------------------------------------------------------
! NAME
!     gottacatch_emall_
!
! DESCRIPTION
!     Find the egg.
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!     Oh? An egg is about to hatch!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! LOCAL VARIABLES
! ---------------
integer(di), dimension(13), parameter :: egg = [80,82,79,70,69,83,83,79,82,95,79,65,75]
integer(di), dimension(43), parameter :: who = [79,104,63,32,73,116,32,115,101,101,109,115,32,121,111,117,39,114,101,32,110,111,116,32,116,104,101,32,114,105,103,104,116,32,112,101,114,115,111,110,46,46,46]
character(20)                         :: why_yes
integer(di)                           :: ierr
logical(k4)                           :: flag

! CODE
! ----
flag = .false.
call get_environment_variable(trim(confusion(egg)),why_yes,status = ierr)
if (ierr == 0) then
    flag = trim(why_yes) == 'yes'
    if (.not. flag) then
        write(stdout,*)
        write(stdout,*) trim(confusion(who))
    end if
end if

END FUNCTION gottacatch_emall_
!=======================================================================

!=======================================================================
FUNCTION confusion(code) RESULT(msg)
!-----------------------------------------------------------------------
! NAME
!     confusion
!
! DESCRIPTION
!     PEM used confusion!
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!     USER is confused! It hurt itself in its confusion!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
integer(di), dimension(:), intent(in) :: code

! LOCAL VARIABLES
! ---------------
integer(di)               :: i
character(:), allocatable :: msg

! CODE
! ----
msg = ''
do i = 1,size(code)
    if (code(i) /= 0) msg = msg//achar(code(i))
end do

END FUNCTION confusion
!=======================================================================

END MODULE display
