MODULE frost
!-----------------------------------------------------------------------
! NAME
!     frost
!
! DESCRIPTION
!     Module for managing frost variables.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use numerics, only: dp, di

! DECLARATION
! -----------
implicit none

! VARIABLES
! ---------
! Different types of frost retained by the PEM to give back to the PCM at the end
real(dp), dimension(:,:), allocatable :: h2o_frost4PCM
real(dp), dimension(:,:), allocatable :: co2_frost4PCM

! PARAMETERS
! ----------
! Different types of frost in the PCM at the beginning [Pa]
real(dp), dimension(:,:), allocatable, protected :: h2ofrost_PCM
real(dp), dimension(:,:), allocatable, protected :: co2frost_PCM

contains
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

!=======================================================================
SUBROUTINE ini_frost()
!-----------------------------------------------------------------------
! NAME
!     ini_frost
!
! DESCRIPTION
!     Initialize the parameters of module 'frost'.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use geometry, only: ngrid, nslope

! DECLARATION
! -----------
implicit none

! CODE
! ----
if (.not. allocated(h2ofrost_PCM)) allocate(h2ofrost_PCM(ngrid,nslope))
if (.not. allocated(co2frost_PCM)) allocate(co2frost_PCM(ngrid,nslope))
if (.not. allocated(h2o_frost4PCM)) allocate(h2o_frost4PCM(ngrid,nslope))
if (.not. allocated(co2_frost4PCM)) allocate(co2_frost4PCM(ngrid,nslope))
h2ofrost_PCM(:,:) = 0._dp
co2frost_PCM(:,:) = 0._dp
h2o_frost4PCM(:,:) = 0._dp
co2_frost4PCM(:,:) = 0._dp

END SUBROUTINE ini_frost
!=======================================================================

!=======================================================================
SUBROUTINE end_frost()
!-----------------------------------------------------------------------
! NAME
!     end_frost
!
! DESCRIPTION
!     Deallocate frost arrays.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! CODE
! ----
if (allocated(h2ofrost_PCM)) deallocate(h2ofrost_PCM)
if (allocated(co2frost_PCM)) deallocate(co2frost_PCM)
if (allocated(h2o_frost4PCM)) deallocate(h2o_frost4PCM)
if (allocated(co2_frost4PCM)) deallocate(co2_frost4PCM)

END SUBROUTINE end_frost
!=======================================================================

!=======================================================================
SUBROUTINE set_h2ofrost_PCM(h2ofrost_PCM_in)
!-----------------------------------------------------------------------
! NAME
!     set_h2ofrost_PCM
!
! DESCRIPTION
!     Setter for 'h2ofrost_PCM'.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), dimension(:,:), intent(in) :: h2ofrost_PCM_in

! CODE
! ----
h2ofrost_PCM(:,:) = h2ofrost_PCM_in(:,:)

END SUBROUTINE set_h2ofrost_PCM
!=======================================================================

!=======================================================================
SUBROUTINE set_co2frost_PCM(co2frost_PCM_in)
!-----------------------------------------------------------------------
! NAME
!     set_co2frost_PCM
!
! DESCRIPTION
!     Setter for 'co2frost_PCM'.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), dimension(:,:), intent(in) :: co2frost_PCM_in

! CODE
! ----
co2frost_PCM(:,:) = co2frost_PCM_in(:,:)

END SUBROUTINE set_co2frost_PCM
!=======================================================================

!=======================================================================
SUBROUTINE compute_frost4PCM(minPCM_h2ofrost,minPCM_co2frost)
!-----------------------------------------------------------------------
! NAME
!     compute_frost4PCM
!
! DESCRIPTION
!     Compute the frost to give back to the PCM (metamorphism).
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!     Frost for the PEM is the extra part of the PCM frost above the
!     yearly minimum.
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use display, only: print_msg

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), dimension(:,:), intent(in) :: minPCM_h2ofrost, minPCM_co2frost

! CODE
! ----
call print_msg('> Computing frost to give back to the PCM (metamorphism)')

h2o_frost4PCM(:,:) = 0._dp
co2_frost4PCM(:,:) = 0._dp
where (h2ofrost_PCM(:,:) > 0._dp) h2o_frost4PCM(:,:) = h2ofrost_PCM(:,:) - minPCM_h2ofrost(:,:)
where (co2frost_PCM(:,:) > 0._dp) co2_frost4PCM(:,:) = co2frost_PCM(:,:) - minPCM_co2frost(:,:)

END SUBROUTINE compute_frost4PCM
!=======================================================================

END MODULE frost
