MODULE info
!-----------------------------------------------------------------------
! NAME
!     info
!
! DESCRIPTION
!     Handles counters for PCM/PEM coupled runs and duration information
!     of the simulation metadata.
!
! AUTHORS & DATE
!     JB Clement, 2023-2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use numerics, only: dp, di, k4

! DECLARATION
! -----------
implicit none

! PARAMETERS
! ----------
character(14), parameter :: infofile_name = 'launchPEM.info'

! VARIABLES
! ---------
integer(di) :: iPCM, iPEM, nPCM, nPCM_ini ! Data about the chained simulation of PCM/PEM runs

contains
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

!=======================================================================
SUBROUTINE read_info()
!-----------------------------------------------------------------------
! NAME
!    read_info
!
! DESCRIPTION
!    Read the file "launchPEM.info" to get the number of simulated
!    years.
!
! AUTHORS & DATE
!    JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use stoppage,  only: stop_clean
use evolution, only: n_yr_sim, nmax_yr_sim, set_r_plnt2earth_yr
use display,   only: print_msg

! DECLARATION
! -----------
implicit none

! LOCAL VARIABLES
! ---------------
logical(k4) :: here
real(dp)    :: tmp
integer(di) :: ierr, funit

! CODE
! ----
inquire(file = infofile_name,exist = here)
if (.not. here) call stop_clean(__FILE__,__LINE__,'cannot find required file "'//infofile_name//'"! It should be created by the launching script.',1)
call print_msg('> Reading "'//infofile_name//'"')
open(newunit = funit,file = infofile_name,status = 'old',form = 'formatted',action = 'read',iostat = ierr)
if (ierr /= 0) call stop_clean(__FILE__,__LINE__,'error opening file "'//infofile_name//'"!',ierr)
read(funit,*) n_yr_sim, nmax_yr_sim, tmp, iPCM, iPEM, nPCM, nPCM_ini
call set_r_plnt2earth_yr(tmp)
close(funit)

END SUBROUTINE read_info
!=======================================================================

!=======================================================================
SUBROUTINE update_info(n_yr_run,stopPEM,n_yr_sim,nmax_yr_sim)
!-----------------------------------------------------------------------
! NAME
!    update_info
!
! DESCRIPTION
!    Update the first line of "launchPEM.info" to count the number of
!    simulated years. Write in "launchPEM.info" the reason why the PEM
!    stopped and the number of simulated years.
!
! AUTHORS & DATE
!    R. Vandemeulebrouck
!    JB Clement, 2023-2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use evolution, only: r_plnt2earth_yr, pem_ini_date
use utility,   only: int2str, nb_digits
use stoppage,  only: stop_clean
use display,   only: print_msg

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
integer(di), intent(in) :: stopPEM     ! Reason to stop
real(dp),    intent(in) :: n_yr_run    ! # of years
real(dp),    intent(in) :: n_yr_sim    ! Current simulated year
real(dp),    intent(in) :: nmax_yr_sim ! Maximum number of years to be simulated

! LOCAL VARIABLES
! ---------------
logical(k4)   :: here
integer(di)   :: cstat, ierr, funit
character(20) :: fch1, fch2, fch3

! CODE
! ----
call print_msg('> Updating "'//infofile_name//'"')
inquire(file = infofile_name,exist = here)
if (.not. here) call stop_clean(__FILE__,__LINE__,'cannot find required file "'//infofile_name//'"! It should be created by the launching script.',1)

! Modify the header (first line)
write(fch1,'(f'//int2str(nb_digits(n_yr_sim) + 5)//'.4)') n_yr_sim
write(fch2,'(f'//int2str(nb_digits(nmax_yr_sim) + 5)//'.4)') nmax_yr_sim
write(fch3,'(f6.4)') r_plnt2earth_yr ! 4 digits to the right of the decimal point to respect the precision of year in "launch_pem.sh"
call execute_command_line('sed -i "1s/.*/'//trim(fch1)//' '//trim(fch2)//' '//trim(fch3)//' '//int2str(iPCM)//' '//int2str(iPEM + 1)//' '//int2str(nPCM)//' '//int2str(nPCM_ini)//'/" launchPEM.info',cmdstat = cstat)
if (cstat > 0) then
    call stop_clean(__FILE__,__LINE__,'command execution failed!',1)
else if (cstat < 0) then
    call stop_clean(__FILE__,__LINE__,'command execution not supported!',1)
end if

! Add the information of current PEM run at the end of file
open(newunit = funit,file = infofile_name,status = "old",position = "append",action = "write",iostat = ierr)
if (ierr /= 0) call stop_clean(__FILE__,__LINE__,'error opening file "'//infofile_name//'"!',ierr)
! Date, Number of years done by the PEM run, Number of years done by the chainded simulation, Code of the stopping criterion
write(funit,'(f20.4,f20.4,f20.4,i20)') pem_ini_date + n_yr_sim, n_yr_run, n_yr_sim, stopPEM
close(funit)

END SUBROUTINE update_info
!=======================================================================

END MODULE info
