MODULE evolution
!-----------------------------------------------------------------------
! NAME
!     evolution
!
! DESCRIPTION
!     Time parameters for the evolution of the simulation.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use numerics, only: dp, di

! DECLARATION
! -----------
implicit none

! PARAMETERS
! ----------
real(dp), protected :: r_plnt2earth_yr ! Conversion ratio from Planetary years to Earth years
real(dp), protected :: pem_ini_date    ! Initial year (in Planetary years) of the simulation of the PEM defined in "run.def"
real(dp), protected :: dt              ! Time step in Planetary years
real(dp), protected :: nmax_yr_run     ! Maximum number of Planetary years of a PEM run if no stopping criterion is reached

! VARIABLES
! ---------
real(dp)    :: n_yr_run    ! Number of simulated Planetary years of the PEM run
real(dp)    :: n_yr_sim    ! Number of simulated Planetary years of the chained simulations
real(dp)    :: nmax_yr_sim ! Maximum number of Planetary years of the chained simulations
integer(di) :: idt         ! Number of timesteps of the PEM run

contains
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

!=======================================================================
SUBROUTINE set_r_plnt2earth_yr(r_plnt2earth_yr_in)
!-----------------------------------------------------------------------
! NAME
!     set_r_plnt2earth_yr
!
! DESCRIPTION
!     Setter for 'r_plnt2earth_yr'.
!
! AUTHORS & DATE
!     JB Clement, 12/2025
!
! NOTES
!
!-----------------------------------------------------------------------

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
real(dp), intent(in) :: r_plnt2earth_yr_in

! CODE
! ----
r_plnt2earth_yr = r_plnt2earth_yr_in

END SUBROUTINE set_r_plnt2earth_yr
!=======================================================================

!=======================================================================
SUBROUTINE set_evolution_config(pem_ini_earth_date,dt_in,nmax_yr_run_in)
!-----------------------------------------------------------------------
! NAME
!     set_evolution_config
!
! DESCRIPTION
!     Setter for 'evolution' configuration parameters.
!
! AUTHORS & DATE
!     JB Clement, 02/2026
!
! NOTES
!
!-----------------------------------------------------------------------

! DEPENDENCIES
! ------------
use utility, only: real2str, int2str
use display, only: print_msg

! DECLARATION
! -----------
implicit none

! ARGUMENTS
! ---------
integer(di), intent(in) :: pem_ini_earth_date
real(dp),    intent(in) :: dt_in, nmax_yr_run_in

! CODE
! ----
pem_ini_date = real(pem_ini_earth_date,dp)/r_plnt2earth_yr
dt = dt_in
nmax_yr_run = nmax_yr_run_in
call print_msg('pem_ini_earth_date = '//int2str(pem_ini_earth_date)//' | pem_ini_date = '//real2str(pem_ini_date))
call print_msg('dt                 = '//real2str(dt))
call print_msg('nmax_yr_run        = '//real2str(nmax_yr_run))

END SUBROUTINE set_evolution_config
!=======================================================================

END MODULE evolution
