MODULE parse_args_mod

!***********************************************************************
! DESCRIPTION:
!    Provides a subroutine to parse command-line options.
!    Recognizes:
!       --help
!       --version [file]
!       --jobid <id>
!       --add-sso
!***********************************************************************

#ifndef MESOSCALE
use pgrm_version_mod,  only: print_pgrm_version
#endif
use job_id_mod,        only: get_job_id
use job_timelimit_mod, only: get_job_timelimit

implicit none

logical :: add_sso_fields = .false. ! Default: do not include SSO fields

!=======================================================================
contains
!=======================================================================

SUBROUTINE parse_args()

implicit none

!---- Arguments

!---- Variables
integer        :: narg, i, eq_pos
character(256) :: arg, key, vlu, jobid

!---- Code
narg = command_argument_count() ! Get the number of command-line arguments
if (narg == 0) return ! No option: normal/default case, nothing to do

i = 1
do while (i <= narg)
    call get_command_argument(i,arg) ! Read the argument given to the program

    eq_pos = index(arg,'=')
    if (eq_pos > 0) then ! Handle "--keyword=value" style
        key = strip(arg(:eq_pos - 1))
        vlu = strip(arg(eq_pos + 1:))
    else ! Handle "--keyword [value]" style
        key = strip(arg)
        vlu = ''
        if (i < narg) then
            call get_command_argument(i + 1,arg) ! Read the argument given to the program
            if (len_trim(adjustl(arg)) > 0 .and. arg(1:2) /= '--') then
                vlu = strip(arg)
                i = i + 1 ! To skip the value argument next time
            endif
        endif
    endif

    select case (to_lower(key))
        case('--help')
            call print_usage()
            call exit(0)

#ifndef MESOSCALE
        case ('--version')
            if (len_trim(adjustl(vlu)) > 0) then
                call print_pgrm_version(vlu)
            else
                call print_pgrm_version()
            endif
            call exit(0)
#endif

        case ('--add-sso')
            add_sso_fields = .true.
            write(*,*) 'SSO fields will be included in "start_archive.nc"'

        case ('--auto-exit')
            call get_job_id(jobid)
            call get_job_timelimit(jobid)

        case default
            write(*,*) 'Error: unknown option "', key, '".'
            call exit(1)
    end select

    i = i + 1
enddo

END SUBROUTINE parse_args
!=======================================================================

SUBROUTINE print_usage()
    write(*,*)
    write(*,*) 'Usage: program [options]'
    write(*,*) '  --help              Show this help message and exit'
    write(*,*) '  --version [file]    Print program version and exit (optional output file)'
    write(*,*) '  --add-sso           Add SSO fields to "start_archive.nc" (only available for Mars start2archive)'
    write(*,*) '  --auto-exit         Enable automatic termination before reaching the job time limit (only available for the PEM)'
    write(*,*)
END SUBROUTINE print_usage
!=======================================================================

PURE FUNCTION strip(s) RESULT(t)

implicit none

!---- Arguments
character(*), intent(in) :: s
character(len(s)) :: t

!---- Variables

!---- Code
t = trim(adjustl(s))

END FUNCTION
!=======================================================================

PURE FUNCTION to_lower(s) RESULT(low)

implicit none

!---- Arguments
character(*), intent(in) :: s
character(len(s)) :: low

!---- Variables
integer :: i

!---- Code
low = s
do i = 1,len(s)
    if (iachar(s(i:i)) >= iachar('A') .and. iachar(s(i:i)) <= iachar('Z')) low(i:i) = achar(iachar(s(i:i)) + 32)
enddo

END FUNCTION to_lower


END MODULE parse_args_mod
