module surfdat_h

  ! arrays are allocated in conf_phys
  real, allocatable    :: albedodat(:) ! albedo of bare ground
  real, allocatable    :: phisfi(:) ! geopotential at ground level
  real, dimension(2)   :: albedice ! default albedo for ice (1:Northern hemisphere 2:Southern hemisphere)
  real, dimension(2)   :: emisice ! ice emissivity (1:Northern hemisphere 2:Southern hemisphere)
  real                 :: emissiv ! emissivity of bare ground
  logical              :: TESicealbedo ! use TES ice cap albedoes (if set to .true.)
  logical, allocatable :: watercaptag(:) ! flag for water ice surface
  real,    allocatable :: dryness(:)

!$OMP THREADPRIVATE(albedodat, phisfi,albedice,emisice,emissiv,TESicealbedo,     &
!$OMP                watercaptag,dryness)

  logical :: temptag !temp tag for water caps

!$OMP THREADPRIVATE(temptag)

  real              :: albedo_h2o_cap ! water cap albedo
  real              :: albedo_h2o_frost ! water frost albedo
  real              :: inert_h2o_ice ! water ice thermal inertia
  real              :: frost_albedo_threshold ! water frost thickness on the ground (kg.m^-2, ie mm)
  real              :: frost_metam_threshold ! water frost threshold before conversion to ice (kg.m^-2, ie mm)
  real              :: TESice_Ncoef ! coefficient for TES ice albedo in Northern hemisphere
  real              :: TESice_Scoef ! coefficient for TES ice albedo in Southern hemisphere
  real              :: iceradius(2) , dtemisice(2)
  real, allocatable :: zmea(:),zstd(:),zsig(:),zgam(:),zthe(:)
  real, allocatable :: hmons(:),summit(:),base(:)
  real, allocatable :: z0(:) ! surface roughness length (m)
  real              :: z0_default ! default (constant over planet) surface roughness (m)

  logical :: old_wsublimation_scheme    ! TEMPORARY : TO USE THE OLD WATER SUBLIMATION SCHEME (i.e., using Cd instead of Ch), true by default

!$OMP THREADPRIVATE(albedo_h2o_cap,albedo_h2o_frost,inert_h2o_ice,               &
!$OMP                frost_albedo_threshold,frost_metam_threshold,TESice_Ncoef,  &
!$OMP                TESice_Scoef,iceradius,dtemisice,                           &
!$OMP                zmea,zstd,zsig,zgam,zthe,hmons,summit,base,z0,z0_default, &
!$OMP                old_wsublimation_scheme)

  !! mountain top dust flows
  real,    allocatable :: alpha_hmons(:) ! sub-grid scale mountain mesh fraction
  real,    allocatable :: hsummit(:) ! mountain height above the GCM surface
  logical, allocatable :: contains_mons(:) ! is there a mountain in the grid mesh ?

!$OMP THREADPRIVATE(alpha_hmons,hsummit,contains_mons)

  !! variables
  real, allocatable  :: tsurf(:,:)   ! Surface temperature (K)
  real, allocatable  :: emis(:,:)    ! Thermal IR surface emissivity
  real, allocatable  :: capcal(:,:) ! surface heat capacity (J m-2 K-1)
  real, allocatable  :: fluxgrd(:,:) ! surface conduction flux (W.m-2)
  real, allocatable  :: qsurf(:,:,:) ! tracer on surface (e.g. kg.m-2)
  real, allocatable  :: watercap(:,:) ! Surface water ice (kg.m-2)
  real, allocatable  :: perennial_co2ice(:,:) ! Perennial CO2 ice (kg.m-2)
  real, dimension(2) :: albedo_perennialco2   ! Albedo for perennial co2 ice (-)
!$OMP THREADPRIVATE(tsurf,emis,capcal,fluxgrd,qsurf,watercap,perennial_co2ice)

contains

  subroutine ini_surfdat_h(ngrid,nq,nslope)

  implicit none
  integer,intent(in) :: ngrid  ! number of atmospheric columns
  integer,intent(in) :: nq     ! number of tracers
  integer,intent(in) :: nslope ! number of sub-grid scale slopes

    allocate(albedodat(ngrid))
    allocate(phisfi(ngrid))
    allocate(watercaptag(ngrid))
    allocate(dryness(ngrid))
    allocate(zmea(ngrid))
    allocate(zstd(ngrid))
    allocate(zsig(ngrid))
    allocate(zgam(ngrid))
    allocate(zthe(ngrid))
    allocate(z0(ngrid))
    allocate(qsurf(ngrid,nq,nslope))
    allocate(tsurf(ngrid,nslope))
    allocate(watercap(ngrid,nslope))
    allocate(perennial_co2ice(ngrid,nslope))
    allocate(emis(ngrid,nslope))
    allocate(capcal(ngrid,nslope))
    allocate(fluxgrd(ngrid,nslope))
    allocate(hmons(ngrid))
    allocate(summit(ngrid))
    allocate(base(ngrid))
    allocate(alpha_hmons(ngrid))
    allocate(hsummit(ngrid))
    allocate(contains_mons(ngrid))

  end subroutine ini_surfdat_h


  subroutine end_surfdat_h

  implicit none

    if (allocated(albedodat))        deallocate(albedodat)
    if (allocated(phisfi))           deallocate(phisfi)
    if (allocated(watercaptag))      deallocate(watercaptag)
    if (allocated(dryness))          deallocate(dryness)
    if (allocated(zmea))             deallocate(zmea)
    if (allocated(zstd))             deallocate(zstd)
    if (allocated(zsig))             deallocate(zsig)
    if (allocated(zgam))             deallocate(zgam)
    if (allocated(zthe))             deallocate(zthe)
    if (allocated(z0))               deallocate(z0)
    if (allocated(qsurf))            deallocate(qsurf)
    if (allocated(tsurf))            deallocate(tsurf)
    if (allocated(watercap))         deallocate(watercap)
    if (allocated(perennial_co2ice))  deallocate(perennial_co2ice)
    if (allocated(emis))             deallocate(emis)
    if (allocated(capcal))           deallocate(capcal)
    if (allocated(fluxgrd))          deallocate(fluxgrd)
    if (allocated(hmons))            deallocate(hmons)
    if (allocated(summit))           deallocate(summit)
    if (allocated(base))             deallocate(base)
    if (allocated(alpha_hmons))      deallocate(alpha_hmons)
    if (allocated(hsummit))          deallocate(hsummit)
    if (allocated(contains_mons))    deallocate(contains_mons)

  end subroutine end_surfdat_h

  subroutine ini_surfdat_h_slope_var(ngrid,nq,nslope)

  implicit none
  integer,intent(in) :: ngrid ! number of atmospheric columns
  integer,intent(in) :: nq ! number of tracers
  integer,intent(in) :: nslope ! number of sub-grid scale slope
    allocate(qsurf(ngrid,nq,nslope))
    allocate(tsurf(ngrid,nslope))
    allocate(watercap(ngrid,nslope))
    allocate(perennial_co2ice(ngrid,nslope))
    allocate(emis(ngrid,nslope))
    allocate(capcal(ngrid,nslope))
    allocate(fluxgrd(ngrid,nslope))

  end subroutine ini_surfdat_h_slope_var

  subroutine end_surfdat_h_slope_var

  implicit none

    if (allocated(qsurf))            deallocate(qsurf)
    if (allocated(tsurf))            deallocate(tsurf)
    if (allocated(watercap))         deallocate(watercap)
    if (allocated(perennial_co2ice)) deallocate(perennial_co2ice)
    if (allocated(emis))             deallocate(emis)
    if (allocated(capcal))           deallocate(capcal)
    if (allocated(fluxgrd))          deallocate(fluxgrd)

  end subroutine end_surfdat_h_slope_var

end module surfdat_h
