MODULE pemetat0_mod

implicit none

!=======================================================================
contains
!=======================================================================

SUBROUTINE pemetat0(filename,ngrid,nsoil_PCM,nsoil_PEM,nslope,timelen,timestep,TI_PEM,tsoil_PEM,icetable_depth,icetable_thickness,ice_porefilling, &
                    tsurf_avg_yr1,tsurf_avg_yr2,q_co2,q_h2o,ps_timeseries,ps_avg_global,d_h2oice,d_co2ice,co2_ice,h2o_ice,                         &
                    watersurf_avg,watersoil_avg,m_co2_regolith_phys,deltam_co2_regolith_phys,m_h2o_regolith_phys,deltam_h2o_regolith_phys,layerings_map)

use iostart_PEM,                only: open_startphy, close_startphy, get_field, get_var, inquire_dimension, inquire_dimension_length
use comsoil_h_PEM,              only: soil_pem, layer_PEM, fluxgeo, inertiedat_PEM, ini_huge_h2oice, depth_breccia, depth_bedrock, index_breccia, index_bedrock
use comsoil_h,                  only: inertiedat
use adsorption_mod,             only: regolith_adsorption, adsorption_pem
use ice_table_mod,              only: computeice_table_equilibrium, icetable_equilibrium, icetable_dynamic
use constants_marspem_mod,      only: alpha_clap_h2o, beta_clap_h2o, TI_breccia, TI_bedrock
use soil_thermalproperties_mod, only: update_soil_thermalproperties
use abort_pem_mod,              only: abort_pem
use compute_soiltemp_mod,       only: ini_tsoil_pem, compute_tsoil_pem
use layering_mod,               only: layering, array2stratif, nb_str_max, layering_algo, add_stratum, ini_layering
use callkeys_mod,               only: startphy_file
use glaciers_mod,               only: rho_co2ice, rho_h2oice
use comcstfi_h,                 only: r, mugaz, pi
use surfdat_h,                  only: watercaptag, perennial_co2ice, qsurf
use metamorphism,               only: frost4PCM, iPCM_h2ofrost, iPCM_co2frost
use tracers,                    only: mmol

implicit none

character(*),                   intent(in) :: filename      ! name of the startfi_PEM.nc
integer,                        intent(in) :: ngrid         ! # of physical grid points
integer,                        intent(in) :: nsoil_PCM     ! # of vertical grid points in the PCM
integer,                        intent(in) :: nsoil_PEM     ! # of vertical grid points in the PEM
integer,                        intent(in) :: nslope        ! # of sub-grid slopes
integer,                        intent(in) :: timelen       ! # time samples
real,                           intent(in) :: timestep      ! time step [s]
real,                           intent(in) :: ps_avg_global ! mean average pressure on the planet [Pa]
real, dimension(ngrid,nslope),  intent(in) :: tsurf_avg_yr1 ! surface temperature at the first year of PCM call [K]
real, dimension(ngrid,nslope),  intent(in) :: tsurf_avg_yr2 ! surface temperature at the second year of PCM call [K]
real, dimension(ngrid,timelen), intent(in) :: q_co2         ! MMR tracer co2 [kg/kg]
real, dimension(ngrid,timelen), intent(in) :: q_h2o         ! MMR tracer h2o [kg/kg]
real, dimension(ngrid,timelen), intent(in) :: ps_timeseries ! surface pressure [Pa]
real, dimension(ngrid,nslope),  intent(in) :: d_h2oice      ! tendencies on h2o ice
real, dimension(ngrid,nslope),  intent(in) :: d_co2ice      ! tendencies on co2 ice
real, dimension(ngrid,nslope),  intent(in) :: watersurf_avg ! surface water ice density, yearly averaged (kg/m^3)
! outputs
real, dimension(ngrid),                  intent(out) :: deltam_co2_regolith_phys ! mass of co2 that is exchanged due to adsorption desorption [kg/m^2]
real, dimension(ngrid),                  intent(out) :: deltam_h2o_regolith_phys ! mass of h2o that is exchanged due to adsorption desorption [kg/m^2]
real, dimension(ngrid,nslope),           intent(out) :: h2o_ice                  ! h2o ice amount [kg/m^2]
real, dimension(ngrid,nslope),           intent(out) :: co2_ice                  ! co2 ice amount [kg/m^2]
type(layering), dimension(ngrid,nslope), intent(inout) :: layerings_map       ! Layerings
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: TI_PEM              ! soil (mid-layer) thermal inertia in the PEM grid [SI]
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: tsoil_PEM           ! soil (mid-layer) temperature [K]
real, dimension(ngrid,nslope),           intent(inout) :: icetable_depth      ! Ice table depth [m]
real, dimension(ngrid,nslope),           intent(inout) :: icetable_thickness  ! Ice table thickness [m]
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: ice_porefilling     ! Subsurface ice pore filling [m3/m3]
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: m_co2_regolith_phys ! mass of co2 adsorbed [kg/m^2]
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: m_h2o_regolith_phys ! mass of h2o adsorbed [kg/m^2]
real, dimension(ngrid,nsoil_PEM,nslope), intent(inout) :: watersoil_avg       ! surface water ice density, yearly averaged (kg/m^3)
! local
real, dimension(ngrid,nsoil_PEM,nslope) :: tsoil_startpem    ! soil temperature saved in the start [K]
real, dimension(ngrid,nsoil_PEM,nslope) :: TI_startPEM       ! soil thermal inertia saved in the start [SI]
logical                                 :: found, found2     ! check if variables are found in the start
integer                                 :: iloop, ig, islope ! index for loops
real                                    :: delta             ! Depth of the interface regolith-breccia, breccia -bedrock [m]
character(2)                            :: num               ! intermediate string to read PEM start sloped variables
logical                                 :: startpem_file     ! boolean to check if we read the startfile or not
real, dimension(:,:,:,:), allocatable   :: stratif_array     ! Array for layerings

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!
!!! Purpose: read start_pem. Need a specific iostart_PEM
!!!
!!! Order: 0. Previous year of the PEM run
!!!           Ice initialization
!!!        1. Thermal Inertia
!!!        2. Soil Temperature
!!!        3. Ice table
!!!        4. Mass of CO2 & H2O adsorbed
!!!
!!! /!\ This order must be respected!
!!! Author: LL
!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!0.1 Check if the start_PEM exist.

inquire(file = filename,exist = startpem_file)

write(*,*)'Is there a "startpem" file?',startpem_file

!0.2 Set to default values
icetable_depth = -1. ! by default, no ice table
icetable_thickness = -1.
ice_porefilling = 0.
!1. Run
if (startpem_file) then
    write(*,*)'There is a "startpem.nc"!'
    ! open pem initial state file:
    call open_startphy(filename)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! H2O ice
    h2o_ice(:,:) = 0.
    call get_field("h2o_ice",h2o_ice,found)
    if (.not. found) then
        write(*,*)'Pemetat0: failed loading <h2o_ice>'
        write(*,*)'will reconstruct the values from watercaptag'
        write(*,*)'with default value ''ini_huge_h2oice'''
        do ig = 1,ngrid
            if (watercaptag(ig)) h2o_ice(ig,:) = ini_huge_h2oice + qsurf(ig,iPCM_h2ofrost,:) - frost4PCM(ig,:)%h2o
        enddo
    endif

    ! CO2 ice
    co2_ice(:,:) = 0.
    call get_field("co2_ice",co2_ice,found)
    if (.not. found) then
        write(*,*)'Pemetat0: failed loading <co2_ice>'
        write(*,*)'will reconstruct the values from perennial_co2ice'
        co2_ice(:,:) = perennial_co2ice(:,:) + qsurf(:,iPCM_co2frost,:) - frost4PCM(:,:)%co2
    endif

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    ! Layerings
    nb_str_max = 68
    if (layering_algo) then
        found = inquire_dimension("nb_str_max")
        if (.not. found) then
            write(*,*) 'Pemetat0: failed loading <nb_str_max>!'
            write(*,*) '''nb_str_max'' is set to 68 (sub-surface layers)!'
        else
            nb_str_max = int(inquire_dimension_length('nb_str_max'))
        endif
        allocate(stratif_array(ngrid,nslope,nb_str_max,6))
        stratif_array = 0.
        do islope = 1,nslope
            write(num,'(i2.2)') islope
            call get_field('stratif_slope'//num//'_top_elevation',stratif_array(:,islope,:,1),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_top_elevation>!'
                exit
            endif
            call get_field('stratif_slope'//num//'_h_co2ice',stratif_array(:,islope,:,2),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_h_co2ice>!'
                exit
            endif
            call get_field('stratif_slope'//num//'_h_h2oice',stratif_array(:,islope,:,3),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_h_h2oice>!'
                exit
            endif
            call get_field('stratif_slope'//num//'_h_dust',stratif_array(:,islope,:,4),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_h_dust>!'
                exit
            endif
            call get_field('stratif_slope'//num//'_h_pore',stratif_array(:,islope,:,5),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_h_pore>!'
                exit
            endif
            call get_field('stratif_slope'//num//'_poreice_volfrac',stratif_array(:,islope,:,6),found)
            if (.not. found) then
                write(*,*) 'Pemetat0: failed loading <stratif_slope'//num//'_poreice_volfrac>!'
                exit
            endif
        enddo ! islope
        if (.not. found) then
            write(*,*) 'So the layerings are initialized with sub-surface strata.'
            write(*,*) 'Ice is added with ''ini_huge_h2oice'' where ''watercaptag'' is true and otherwise with ''perennial_co2ice'' found in the PCM.'
            do ig = 1,ngrid
                if (watercaptag(ig)) then
                    do islope = 1,nslope
                        call ini_layering(layerings_map(ig,islope))
                        call add_stratum(layerings_map(ig,islope),1.05*ini_huge_h2oice/rho_h2oice,0.,ini_huge_h2oice/rho_h2oice,0.05*ini_huge_h2oice/rho_h2oice,0.,0.)
                    enddo
                else
                    do islope = 1,nslope
                        call ini_layering(layerings_map(ig,islope))
                        if (perennial_co2ice(ig,islope) > 0.) call add_stratum(layerings_map(ig,islope),1.05*perennial_co2ice(ig,islope)/rho_co2ice,perennial_co2ice(ig,islope)/rho_co2ice,0.,0.05*perennial_co2ice(ig,islope)/rho_co2ice,0.,0.)
                    enddo
                endif
            enddo
        else
            call array2stratif(stratif_array,ngrid,nslope,layerings_map)
        endif
        deallocate(stratif_array)
    endif

    if (soil_pem) then

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!1. Thermal Inertia
! a. General case
        do islope = 1,nslope
            write(num,'(i2.2)') islope
            call get_field("TI_PEM_slope"//num,TI_startPEM(:,:,islope),found)
            if (.not. found) then
                write(*,*)'PEM settings: failed loading < TI_PEM_slope'//num//'>'
                write(*,*)'will reconstruct the values of TI_PEM'

                do ig = 1,ngrid
                    if (TI_PEM(ig,index_breccia,islope) < TI_breccia) then
                        !!! transition
                        delta = depth_breccia
                        TI_PEM(ig,index_breccia + 1,islope) = sqrt((layer_PEM(index_breccia + 1) - layer_PEM(index_breccia))/ &
                                                                   (((delta - layer_PEM(index_breccia))/(TI_PEM(ig,index_breccia,islope)**2)) + &
                                                                   ((layer_PEM(index_breccia + 1) - delta)/(TI_breccia**2))))
                        do iloop = index_breccia + 2,index_bedrock
                            TI_PEM(ig,iloop,islope) = TI_breccia
                        enddo
                    else ! we keep the high TI values
                        do iloop = index_breccia + 1,index_bedrock
                            TI_PEM(ig,iloop,islope) = TI_PEM(ig,index_breccia,islope)
                        enddo
                    endif ! TI PEM and breccia comparison
                    !! transition
                    delta = depth_bedrock
                    TI_PEM(ig,index_bedrock + 1,islope) = sqrt((layer_PEM(index_bedrock + 1) - layer_PEM(index_bedrock))/ &
                                                               (((delta - layer_PEM(index_bedrock))/(TI_PEM(ig,index_bedrock,islope)**2)) + &
                                                               ((layer_PEM(index_bedrock + 1) - delta)/(TI_bedrock**2))))
                    do iloop = index_bedrock + 2,nsoil_PEM
                        TI_PEM(ig,iloop,islope) = TI_bedrock
                    enddo
                enddo
            else ! found
                do iloop = nsoil_PCM + 1,nsoil_PEM
                    TI_PEM(:,iloop,islope) = TI_startPEM(:,iloop,islope) ! 1st layers can change because of the presence of ice at the surface, so we don't change it here.
                enddo
            endif ! not found
        enddo ! islope

        write(*,*) 'PEMETAT0: TI done'

! b. Special case for inertiedat, inertiedat_PEM
        call get_field("inertiedat_PEM",inertiedat_PEM,found)
        if (.not. found) then
            do iloop = 1,nsoil_PCM
                inertiedat_PEM(:,iloop) = inertiedat(:,iloop)
            enddo
        !!! zone de transition
            delta = depth_breccia
            do ig = 1,ngrid
                if (inertiedat_PEM(ig,index_breccia) < TI_breccia) then
                    inertiedat_PEM(ig,index_breccia + 1) = sqrt((layer_PEM(index_breccia + 1) - layer_PEM(index_breccia))/ &
                                                                (((delta - layer_PEM(index_breccia))/(inertiedat(ig,index_breccia)**2)) + &
                                                                ((layer_PEM(index_breccia + 1) - delta)/(TI_breccia**2))))

                    do iloop = index_breccia + 2,index_bedrock
                        inertiedat_PEM(ig,iloop) = TI_breccia
                    enddo
                else
                    do iloop = index_breccia + 1,index_bedrock
                        inertiedat_PEM(ig,iloop) = inertiedat_PEM(ig,nsoil_PCM)
                    enddo
                endif ! comparison ti breccia
            enddo ! ig

            !!! zone de transition
            delta = depth_bedrock
            do ig = 1,ngrid
                inertiedat_PEM(ig,index_bedrock + 1) = sqrt((layer_PEM(index_bedrock + 1) - layer_PEM(index_bedrock))/ &
                                                            (((delta - layer_PEM(index_bedrock))/(inertiedat_PEM(ig,index_bedrock)**2))+ &
                                                            ((layer_PEM(index_bedrock + 1) - delta)/(TI_bedrock**2))))
            enddo

            do iloop = index_bedrock + 2,nsoil_PEM
                do ig = 1,ngrid
                    inertiedat_PEM(ig,iloop) = TI_bedrock
                enddo
            enddo
        endif ! not found

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!2. Soil Temperature
        do islope = 1,nslope
            write(num,fmt='(i2.2)') islope
            call get_field("tsoil_PEM_slope"//num,tsoil_startpem(:,:,islope),found)
            if (.not. found) then
                write(*,*)'PEM settings: failed loading <tsoil_PEM_slope'//num//'>'
                write(*,*)'will reconstruct the values of Tsoil'
                call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
                call compute_tsoil_pem(ngrid,nsoil_PEM,.true.,TI_PEM(:,:,islope),timestep,tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
            else
! predictor-corrector: due to changes of surface temperature in the PCM, the tsoil_startpem is adapted firstly
!                      for PCM year 1 and then for PCM year 2 in order to build the evolution of the profile at depth
                call compute_tsoil_pem(ngrid,nsoil_PEM,.true.,TI_PEM(:,:,islope),timestep,tsurf_avg_yr1(:,islope),tsoil_startpem(:,:,islope))
                call compute_tsoil_pem(ngrid,nsoil_PEM,.false.,TI_PEM(:,:,islope),timestep,tsurf_avg_yr1(:,islope),tsoil_startpem(:,:,islope))
                call compute_tsoil_pem(ngrid,nsoil_PEM,.false.,TI_PEM(:,:,islope),timestep,tsurf_avg_yr2(:,islope),tsoil_startpem(:,:,islope))

                tsoil_PEM(:,nsoil_PCM + 1:nsoil_PEM,islope) = tsoil_startpem(:,nsoil_PCM + 1:nsoil_PEM,islope)
            endif !found

            watersoil_avg(:,nsoil_PCM + 1:nsoil_PEM,islope) = exp(beta_clap_h2o/tsoil_PEM(:,nsoil_PCM + 1:nsoil_PEM,islope) + alpha_clap_h2o)/tsoil_PEM(:,nsoil_PCM + 1:nsoil_PEM,islope)*mmol(igcm_h2o_vap)/(mugaz*r)
        enddo ! islope
        write(*,*) 'PEMETAT0: TSOIL done'

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!3. Ice Table
        if (icetable_equilibrium) then
            call get_field("icetable_depth",icetable_depth,found)
            if (.not. found) then
                write(*,*)'PEM settings: failed loading <icetable_depth>'
                write(*,*)'will reconstruct the values of the ice table given the current state'
                call computeice_table_equilibrium(ngrid,nslope,nsoil_PEM,watercaptag,watersurf_avg,watersoil_avg,TI_PEM(:,1,:),icetable_depth,icetable_thickness)
                call update_soil_thermalproperties(ngrid,nslope,nsoil_PEM,d_h2oice,h2o_ice,ps_avg_global,icetable_depth,icetable_thickness,ice_porefilling,icetable_equilibrium,icetable_dynamic,TI_PEM)
                do islope = 1,nslope
                    call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
                enddo
            endif
            write(*,*) 'PEMETAT0: ICE TABLE done'
        else if (icetable_dynamic) then
            call get_field("ice_porefilling",ice_porefilling,found)
            if (.not. found) write(*,*)'PEM settings: failed loading <ice_porefilling>'
            call get_field("icetable_depth",icetable_depth,found)
            if (.not. found) then
                write(*,*)'PEM settings: failed loading <icetable_depth>'
                write(*,*)'will reconstruct the values of the ice table given the current state'
                call update_soil_thermalproperties(ngrid,nslope,nsoil_PEM,d_h2oice,h2o_ice,ps_avg_global,icetable_depth,icetable_thickness,ice_porefilling,icetable_equilibrium,icetable_dynamic,TI_PEM)
                do islope = 1,nslope
                    call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
                enddo
            endif
            write(*,*) 'PEMETAT0: ICE TABLE done'
        endif

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!4. CO2 & H2O Adsorption
        if (adsorption_pem) then
            do islope = 1,nslope
                write(num,fmt='(i2.2)') islope
                call get_field("mco2_reg_ads_slope"//num,m_co2_regolith_phys(:,:,islope),found)
                if (.not. found) then
                    m_co2_regolith_phys = 0.
                    exit
                endif
            enddo
            do islope = 1,nslope
                write(num,fmt='(i2.2)') islope
                call get_field("mh2o_reg_ads_slope"//num,m_h2o_regolith_phys(:,:,islope),found2)
                if (.not. found2) then
                    m_h2o_regolith_phys = 0.
                    exit
                endif
            enddo

            call regolith_adsorption(ngrid,nslope,nsoil_PEM,timelen,d_h2oice,d_co2ice,h2o_ice,co2_ice,tsoil_PEM,TI_PEM,ps_timeseries,q_co2,q_h2o, &
                                     m_h2o_regolith_phys,deltam_h2o_regolith_phys,m_co2_regolith_phys,deltam_co2_regolith_phys)

            if (.not. found) deltam_co2_regolith_phys = 0.
            if (.not. found2) deltam_h2o_regolith_phys = 0.

            write(*,*) 'PEMETAT0: CO2 & H2O adsorption done'
        endif ! adsorption_pem
    endif ! soil_pem

    call close_startphy

else ! No startpem, let's build all by hand

    write(*,*)'There is no "startpem.nc"!'

    ! h2o ice
    h2o_ice(:,:) = 0.
    write(*,*)'So ''h2o_ice'' is initialized with default value ''ini_huge_h2oice'' where ''watercaptag'' is true.'
    do ig = 1,ngrid
        if (watercaptag(ig)) h2o_ice(ig,:) = ini_huge_h2oice + qsurf(ig,iPCM_h2ofrost,:) - frost4PCM(ig,:)%h2o
    enddo

    ! co2 ice
    write(*,*)'So ''co2_ice'' is initialized with ''perennial_co2ice'' found in the PCM.'
    co2_ice(:,:) = perennial_co2ice(:,:) + qsurf(:,iPCM_co2frost,:) - frost4PCM(:,:)%co2

    ! Layerings
    nb_str_max = 68
    if (layering_algo) then
        write(*,*)'So the layerings are initialized with sub-surface strata.'
        write(*,*)'Ice is added with ''ini_huge_h2oice'' where ''watercaptag'' is true and otherwise with ''perennial_co2ice'' found in the PCM.'
        do ig = 1,ngrid
            if (watercaptag(ig)) then
                do islope = 1,nslope
                    call ini_layering(layerings_map(ig,islope))
                    call add_stratum(layerings_map(ig,islope),1.05*ini_huge_h2oice/rho_h2oice,0.,ini_huge_h2oice/rho_h2oice,0.05*ini_huge_h2oice/rho_h2oice,0.,0.)
                enddo
            else
                do islope = 1,nslope
                    call ini_layering(layerings_map(ig,islope))
                    if (perennial_co2ice(ig,islope) > 0.) call add_stratum(layerings_map(ig,islope),1.05*perennial_co2ice(ig,islope)/rho_co2ice,perennial_co2ice(ig,islope)/rho_co2ice,0.,0.05*perennial_co2ice(ig,islope)/rho_co2ice,0.,0.)
                enddo
            endif
        enddo
    endif

    if (soil_pem) then

!a) Thermal inertia
        do islope = 1,nslope
            do ig = 1,ngrid
                if (TI_PEM(ig,index_breccia,islope) < TI_breccia) then
                    !!! transition
                    delta = depth_breccia
                    TI_PEM(ig,index_breccia + 1,islope) = sqrt((layer_PEM(index_breccia + 1) - layer_PEM(index_breccia))/                   &
                                                               (((delta - layer_PEM(index_breccia))/(TI_PEM(ig,index_breccia,islope)**2)) + &
                                                               ((layer_PEM(index_breccia + 1) - delta)/(TI_breccia**2))))
                    do iloop = index_breccia + 2,index_bedrock
                        TI_PEM(ig,iloop,islope) = TI_breccia
                    enddo
                else ! we keep the high ti values
                    do iloop = index_breccia + 1,index_bedrock
                        TI_PEM(ig,iloop,islope) = TI_PEM(ig,index_breccia,islope)
                    enddo
                endif
                !! transition
                delta = depth_bedrock
                TI_PEM(ig,index_bedrock + 1,islope) = sqrt((layer_PEM(index_bedrock + 1) - layer_PEM(index_bedrock))/                   &
                                                           (((delta - layer_PEM(index_bedrock))/(TI_PEM(ig,index_bedrock,islope)**2)) + &
                                                           ((layer_PEM(index_bedrock + 1) - delta)/(TI_breccia**2))))
                do iloop = index_bedrock + 2,nsoil_PEM
                    TI_PEM(ig,iloop,islope) = TI_bedrock
                enddo
            enddo
        enddo

        do iloop = 1,nsoil_PCM
            inertiedat_PEM(:,iloop) = inertiedat(:,iloop)
        enddo
        !!! zone de transition
        delta = depth_breccia
        do ig = 1,ngrid
            if (inertiedat_PEM(ig,index_breccia) < TI_breccia) then
                inertiedat_PEM(ig,index_breccia + 1) = sqrt((layer_PEM(index_breccia + 1) - layer_PEM(index_breccia))/                &
                                                            (((delta - layer_PEM(index_breccia))/(inertiedat(ig,index_breccia)**2)) + &
                                                            ((layer_PEM(index_breccia + 1) - delta)/(TI_breccia**2))))
                do iloop = index_breccia + 2,index_bedrock
                    inertiedat_PEM(ig,iloop) = TI_breccia
                enddo
            else
                do iloop = index_breccia + 1,index_bedrock
                    inertiedat_PEM(ig,iloop) = inertiedat_PEM(ig,index_breccia)
                enddo
            endif
        enddo

        !!! zone de transition
        delta = depth_bedrock
        do ig = 1,ngrid
            inertiedat_PEM(ig,index_bedrock + 1) = sqrt((layer_PEM(index_bedrock + 1) - layer_PEM(index_bedrock))/                    &
                                                        (((delta - layer_PEM(index_bedrock))/(inertiedat_PEM(ig,index_bedrock)**2)) + &
                                                        ((layer_PEM(index_bedrock + 1) - delta)/(TI_bedrock**2))))
        enddo

        do iloop = index_bedrock + 2,nsoil_PEM
            do ig = 1,ngrid
                inertiedat_PEM(ig,iloop) = TI_bedrock
            enddo
        enddo

        write(*,*) 'PEMETAT0: TI done'

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!b) Soil temperature
        do islope = 1,nslope
            call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
            call compute_tsoil_pem(ngrid,nsoil_PEM,.true.,TI_PEM(:,:,islope),timestep,tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))

! First raw initialization
            watersoil_avg(:,nsoil_PCM + 1:nsoil_PEM,islope) = exp(beta_clap_h2o/tsoil_PEM(:,nsoil_PCM + 1:nsoil_PEM,islope) + alpha_clap_h2o)/tsoil_PEM(:,nsoil_PCM + 1:nsoil_PEM,islope)*mmol%h2o/(mugaz*r)
        enddo !islope
        write(*,*) 'PEMETAT0: TSOIL done'

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!c) Ice table
        if (icetable_equilibrium) then
            call computeice_table_equilibrium(ngrid,nslope,nsoil_PEM,watercaptag,watersurf_avg,watersoil_avg,TI_PEM(:,1,:),icetable_depth,icetable_thickness)
            call update_soil_thermalproperties(ngrid,nslope,nsoil_PEM,d_h2oice,h2o_ice,ps_avg_global,icetable_depth,icetable_thickness,ice_porefilling,icetable_equilibrium,icetable_dynamic,TI_PEM)
            do islope = 1,nslope
                call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
            enddo
            write(*,*) 'PEMETAT0: ICE TABLE done'
        else if (icetable_dynamic) then
            call update_soil_thermalproperties(ngrid,nslope,nsoil_PEM,d_h2oice,h2o_ice,ps_avg_global,icetable_depth,icetable_thickness,ice_porefilling,icetable_equilibrium,icetable_dynamic,TI_PEM)
            do islope = 1,nslope
                call ini_tsoil_pem(ngrid,nsoil_PEM,TI_PEM(:,:,islope),tsurf_avg_yr2(:,islope),tsoil_PEM(:,:,islope))
            enddo
            write(*,*) 'PEMETAT0: ICE TABLE done'
        endif

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!d) Regolith adsorbed
        if (adsorption_pem) then
            m_co2_regolith_phys = 0.
            m_h2o_regolith_phys = 0.
            call regolith_adsorption(ngrid,nslope,nsoil_PEM,timelen,d_h2oice,d_co2ice,h2o_ice,co2_ice,tsoil_PEM,TI_PEM,ps_timeseries,q_co2,q_h2o, &
                                     m_h2o_regolith_phys,deltam_h2o_regolith_phys, m_co2_regolith_phys,deltam_co2_regolith_phys)
            deltam_co2_regolith_phys = 0.
            deltam_h2o_regolith_phys = 0.
        endif

        write(*,*) 'PEMETAT0: CO2 & H2O adsorption done'
    endif !soil_pem
endif ! of if (startphy_file)

if (soil_pem) then ! Sanity check
    do ig = 1,ngrid
        do islope = 1,nslope
            do iloop = 1,nsoil_PEM
                if (isnan(tsoil_PEM(ig,iloop,islope))) call abort_pem("PEM - pemetat0","NaN detected in Tsoil",1)
            enddo
        enddo
    enddo
endif !soil_pem

END SUBROUTINE pemetat0

END MODULE pemetat0_mod
