MODULE profile_temp_mod

implicit none

CONTAINS

SUBROUTINE profile_temp(nlev,zkm,temp)

use ioipsl_getincom, only: getin ! To use 'getin'
use comcstfi_h,      only: pi
      
IMPLICIT NONE

!=======================================================================
!     Subroutine utilisee dans le modele 1-D "testphys1d"
!     pour l'initialisation du profil atmospherique
!=======================================================================
!
!   differents profils d'atmospheres. T=f(z)
!   entree:
!     nlev    nombre de niveaux
!     zkm     alititudes en km
!     ichoice choix de l'atmosphere:
!             1 Temperature constante
!             2 profil Savidjari
!             3 Lindner (profil polaire)
!             4 Inversion pour Francois
!             5 Seiff (moyen)
!             6 T constante + perturbation gauss (level) (christophe 10/98)
!             7 T constante + perturbation gauss    (km) (christophe 10/98)
!             8 Lecture du profile dans un fichier ASCII (profile_temp)
!     tref    temperature de reference
!     isin    ajout d'une perturbation (isin=1)
!     pic     pic perturbation gauss pour ichoice = 6 ou 7
!     largeur largeur de la perturbation gauss pour ichoice = 6 ou 7
!     hauteur hauteur de la perturbation gauss pour ichoice = 6 ou 7
!
!   sortie:
!     temp    temperatures en K
!     
!=======================================================================
!-----------------------------------------------------------------------
!   declarations:
!   -------------

!   arguments:
!   ----------

INTEGER,               intent(in) :: nlev
REAL, dimension(nlev), intent(in) :: zkm
REAL, dimension(nlev), intent(out) :: temp

!   local:
!   ------

INTEGER :: il,ichoice,iseiff,isin,iter,ierr
INTEGER, PARAMETER :: nseiff = 37
REAL :: tref,t1,t2,t3,ww
REAL, dimension(nseiff) :: tseiff
DATA tseiff/214. ,213.8,213.4,212.4,209.3,205. ,201.4,197.8, &
            194.6,191.4,188.2,185.2,182.5,180. ,177.5,175  , &
            172.5,170. ,167.5,164.8,162.4,160. ,158. ,156. , &
            154.1,152.2,150.3,148.7,147.2,145.7,144.2,143. , &
            142. ,141. ,140  ,139.5,139./
REAL :: pic,largeur
REAL :: hauteur,tmp

!-----------------------------------------------------------------------
!   read input profile type:
!--------------------------

ichoice=1 ! default value for ichoice
call getin("ichoice",ichoice)
tref=200 ! default value for tref
call getin("tref",tref)
isin=0 ! default value for isin (=0 means no perturbation)
call getin("isin",isin)
pic=26.522 ! default value for pic
call getin("pic",pic)
largeur=10 ! default value for largeur
call getin("largeur",largeur)
hauteur=30 ! default value for hauteur
call getin("hauteur",hauteur)

!-----------------------------------------------------------------------
!   ichoice=1 temperature constante:
!   --------------------------------

IF(ichoice.EQ.1) THEN
    DO il=1,nlev
        temp(il)=tref
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=2 savidjari:
!   --------------------

ELSE IF(ichoice.EQ.2) THEN
    DO il=1,nlev
        temp(il)=AMAX1(219.-2.5*zkm(il),140.)
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=3 Lindner:
!   ------------------

ELSE IF(ichoice.EQ.3) THEN
    DO il=1,nlev
        IF(zkm(il).LT.2.5) THEN
            temp(il)=150.+30.*zkm(il)/2.5
        ELSE IF(zkm(il).LT.5.) THEN
            temp(il)=180.
        ELSE
            temp(il)=AMAX1(180.-2.*(zkm(il)-5.),130.)
        ENDIF
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=4 Inversion pour Francois:
!   ------------------

ELSE IF(ichoice.EQ.4) THEN
    DO il=1,nlev
        IF(zkm(il).LT.20.) THEN
            temp(il)=135.
        ELSE
            temp(il)=AMIN1(135.+5.*(zkm(il)-20.),200.)
        ENDIF
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=5 Seiff:
!   ----------------

ELSE IF(ichoice.EQ.5) THEN
    DO il=1,nlev
        iseiff=INT(zkm(il)/2.)+1
        IF(iseiff.LT.nseiff-1) THEN
            temp(il)=tseiff(iseiff)+(zkm(il)-2.*(iseiff-1))*(tseiff(iseiff+1)-tseiff(iseiff))/2.
        ELSE
            temp(il)=tseiff(nseiff)
        ENDIF
    ENDDO
! IF(ichoice.EQ.6) THEN
!    DO iter=1,6
!        t2=temp(1)
!        t3=temp(2)
!        DO il=2,nlev-1
!            t1=t2
!            t2=t3
!            t3=temp(il+1)
!            ww=tanh(zkm(il)/20.)
!            ww=ww*ww*ww
!            temp(il)=t2+ww*.5*(t1-2.*t2+t3)
!        ENDDO
!    ENDDO
!ENDIF

!-----------------------------------------------------------------------
!   ichoice=6 
!   ---------

ELSE IF(ichoice.EQ.6) THEN
    DO il=1,nlev
        tmp=il-pic
        temp(il)=tref + hauteur*exp(-tmp*tmp/largeur/largeur)
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=7
!   ---------

ELSE IF(ichoice.EQ.7) THEN
    DO il=1,nlev
        tmp=zkm(il)-pic
        temp(il)=tref + hauteur*exp(-tmp*tmp*4/largeur/largeur)
    ENDDO

!-----------------------------------------------------------------------
!   ichoice=8
!   ---------

ELSE IF(ichoice.GE.8) THEN
    OPEN(11,file='profile_temp',status='old',form='formatted',iostat=ierr)
    if (ierr == 0) then
        DO il=1,nlev
            READ (11,*) temp(il)
        ENDDO
    else
        error stop 'File "profile_temp" not found!'
    endif
    CLOSE(11)

!-----------------------------------------------------------------------
ENDIF

!-----------------------------------------------------------------------
!   rajout eventuel d'une perturbation:
!   -----------------------------------

IF (isin.EQ.1) THEN
    DO il=1,nlev
!        if (nlev.EQ.501) then
!            if (zkm(il).LE.70.5) then
!                temp(il)=temp(il)+(1.-1000./(1000+zkm(il)*zkm(il)))*( 6.*SIN(zkm(il)*pi/6.)+9.*SIN(zkm(il)*pi/10.3) )
!            endif
!        else
        temp(il)=temp(il)+(1.-1000./(1000+zkm(il)*zkm(il)))*( 6.*SIN(zkm(il)*pi/6.)+9.*SIN(zkm(il)*pi/10.3) )
!        endif
    ENDDO
ENDIF

!-----------------------------------------------------------------------
!   Ecriture du profil de temperature dans un fichier profile_temp.out
!   ------------------------------------------------------------------

OPEN(12,file='profile_temp.out',status='unknown',form='formatted')
    DO il=1,nlev
        write(12,*) temp(il)
    ENDDO
CLOSE(12)

END SUBROUTINE profile_temp

END MODULE profile_temp_mod
