#!/bin/bash
################################################################
### Launching script for a chained simulation of 1D PCM runs ###
################################################################
set -e
trap 'echo -e "\033[31mError: an issue occurred in the script on line $LINENO! Please review the command and try again.\033[0m"' ERR


################################################################
# Modify here the parameters for the simulation
###############################################
# Path to the arch.env to source:
source ../trunk/LMDZ.COMMON/arch.env

# Set the number of Martian years to be simulated:
n_myear=100

# Name of executable for the PCM:
exePCM="testphys1d_32_phymars_seq.e"
################################################################


echo "The launching script is starting!"
echo "The output file is \"launch.log\"."
exec > launch.log 2>&1

# Save the current value of LC_NUMERIC and set it to a locale that uses a dot as the decimal separator
OLD_LC_NUMERIC=$LC_NUMERIC
LC_NUMERIC=en_US.UTF-8

# To end the launching script with error
errlaunch() {
    # Restore the previous value of LC_NUMERIC
    LC_NUMERIC=$OLD_LC_NUMERIC

    date
    echo "Error: an issue occured in the launching script!"
    exit 1
}


# Check if the set-up is correct
dir=`pwd`
machine=`hostname`
address=`whoami`
if [ -v n_myear ]; then
    if [ $(echo "$n_myear <= 0." | bc -l) -eq 1 ]; then
        echo "Error: 'n_myear' must be > 0!"
        errlaunch
    fi
else
    echo "Error: the number of years to be simulated is not set!"
    errlaunch
fi
if [ ! -f "$exePCM" ]; then
    echo "Error: file \"$exePCM\" does not exist in $dir!"
    errlaunch
fi
if [ ! -d "logs" ]; then
    mkdir logs
fi
if [ ! -d "starts" ]; then
    mkdir starts
fi
if [ ! -d "diags" ]; then
    mkdir diags
fi

# Initialization
dir=`pwd`
machine=`hostname`
address=`whoami`
echo "This is a chained simulation for PCM 1D runs in $dir on $machine."
echo "Number of years to be simulated: $n_myear Martian years."
i_myear=0
iPCM=1
if [ -f "startfi.nc" ]; then
    cp startfi.nc starts/
fi
if [ -f "start1D.txt" ]; then
    cp start1D.txt starts/
fi

# Main loop to to run PCM year by year
while [ $i_myear -lt $n_myear ]; do
    # Run the PCM
    echo "Run $iPCM: year $iPCM/$n_myear..."
    ./$exePCM > run.log 2>&1
    if [ ! -f "restartfi.nc" ] || ! (tail -n 100 run.log | grep -iq "everything is cool!"); then # Check if it ended abnormally
        echo "Error: the run $iPCM crashed!"
        errlaunch
    fi
    # Copy data files and prepare the next run
    mv run.log logs/run${iPCM}.log
    if [ -f "diagfi.nc" ]; then
        mv diagfi.nc diags/diagfi${iPCM}.nc
    fi
    if [ -f "diagsoil.nc" ]; then
        mv diagsoil.nc diags/diagsoil${iPCM}.nc
    fi
    if [ -f "stats.nc" ]; then
        mv stats.nc diags/stats${iPCM}.nc
    fi
    if [ -f "Xdiurnalave.nc" ]; then
        mv Xdiurnalave.nc diags/Xdiurnalave${iPCM}.nc
    fi
    cp restartfi.nc starts/startfi${iPCM}.nc
    mv restartfi.nc startfi.nc
    if [ -f "restart1D.txt" ]; then
        cp restart1D.txt starts/restart1D${iPCM}.txt
        mv restart1D.txt start1D.txt
    fi
    ((iPCM++))
    ((i_myear++))
    echo "Done!"
done

# Restore the previous value of LC_NUMERIC
LC_NUMERIC=$OLD_LC_NUMERIC

date
echo "Success: the launching script completed normally!"
exit 0
