      SUBROUTINE initracer(ngrid,nq)

      use surfdat_h, ONLY: dryness
      USE tracer_h
      USE callkeys_mod, only: water
      USE recombin_corrk_mod, ONLY: ini_recombin
      USE mod_phys_lmdz_para, only: is_master, bcast
      use generic_cloud_common_h
      IMPLICIT NONE
!=======================================================================
!   subject:
!   --------
!   Initialization related to tracer 
!   (transported dust, water, chemical species, ice...)
!
!   Name of the tracer
!
!   Test of dimension :
!   Initialize COMMON tracer in tracer.h, using tracer names provided
!   by the argument nametrac
!
!   author: F.Forget
!   ------
!            Ehouarn Millour (oct. 2008) identify tracers by their names
!            Y Jaziri & J. Vatant d'Ollone (2020) : Modern traceur.def
!            L Teinturier (2022): Tracer names are now read here instead of 
!                                  inside interfaces 
!=======================================================================

      integer,intent(in) :: ngrid,nq

      character(len=500) :: tracline   ! to read traceur.def lines
      ! character(len=30)  :: txt        ! to store some text
      integer :: blank      !to store the index of 1st blank when reading tracers names
      integer iq,ig,count,ierr

!-----------------------------------------------------------------------
!  radius(nq)      ! aerosol particle radius (m)
!  rho_q(nq)       ! tracer densities (kg.m-3)
!  qext(nq)        ! Single Scat. Extinction coeff at 0.67 um
!  alpha_lift(nq)  ! saltation vertical flux/horiz flux ratio (m-1)
!  alpha_devil(nq) ! lifting coeeficient by dust devil
!  rho_dust          ! Mars dust density
!  rho_ice           ! Water ice density
!  doubleq           ! if method with mass (iq=1) and number(iq=2) mixing ratio
!  varian            ! Characteristic variance of log-normal distribution
!-----------------------------------------------------------------------

      if (is_master) then ! only the master proc/thread needs do this

        moderntracdef=.false. ! For modern traceur.def (default false, old type)

        open(407, form = 'formatted', status = 'old', &
             file = 'traceur.def', iostat=ierr)
        if (ierr /=0) then
          ! call abort_physic('initracer',&
          ! 'Problem in opening traceur.def',1)
          print*,'Problem in opening traceur.def'
          stop
        end if
!! - Modif. by JVO and YJ for modern planetary traceur.def ---------------
        READ(407,'(A)') tracline
        IF (trim(tracline).ne.'#ModernTrac-v1') THEN ! Test modern traceur.def
          READ(tracline,*) nqtot ! Try standard traceur.def
        ELSE
         moderntracdef = .true.
         DO
           READ(407,'(A)',iostat=ierr) tracline
           IF (ierr==0) THEN
             IF (index(tracline,'#').ne.1) THEN ! Allows arbitary number of comments lines in the header
               READ(tracline,*) nqtot
               ! Temporary not implemented solution
               if (nqtot/=nq) then
      !            call abort_physic('initracer','Different number of '// &
      ! 'tracers in dynamics and physics not managed yet',1)
                print*,'!= nbr oftracers in dynamics and physics not managed yet'
                stop
               endif
               EXIT
             ENDIF
           ELSE ! If pb, or if reached EOF without having found nqtot
      !        call abort_physic('initracer','Unable to read numbers '// &
      !  'of tracers in traceur.def',1)
            print*,"unable to read numbers of tracer in tracer.def"
            stop
           ENDIF
         ENDDO
        ENDIF ! if modern or standard traceur.def
       
       endif ! of if (is_master)
       
       ! share the information with other procs/threads (if any)
       CALL bcast(nqtot)
       CALL bcast(moderntracdef)
       
!! -----------------------------------------------------------------------
       !! For the moment number of tracers in dynamics and physics are equal
       nqtot=nq
       !! we allocate once for all arrays in common in tracer_h.F90
       !! (supposedly those are not used before call to initracer)
       IF (.NOT.ALLOCATED(noms))           ALLOCATE(noms(nq))
       IF (.NOT.ALLOCATED(mmol))           ALLOCATE(mmol(nq))
       IF (.NOT.ALLOCATED(aki))            ALLOCATE(aki(nqtot))
       IF (.NOT.ALLOCATED(cpi))            ALLOCATE(cpi(nqtot))
       IF (.NOT.ALLOCATED(radius))         ALLOCATE(radius(nq))
       IF (.NOT.ALLOCATED(rho_q))          ALLOCATE(rho_q(nq))
       IF (.NOT.ALLOCATED(qext))           ALLOCATE(qext(nq))
       IF (.NOT.ALLOCATED(alpha_lift))     ALLOCATE(alpha_lift(nq))
       IF (.NOT.ALLOCATED(alpha_devil))    ALLOCATE(alpha_devil(nq))
       IF (.NOT.ALLOCATED(qextrhor))       ALLOCATE(qextrhor(nq))
       IF (.NOT.ALLOCATED(igcm_dustbin))   ALLOCATE(igcm_dustbin(nq))
       IF (.NOT.ALLOCATED(is_chim))        ALLOCATE(is_chim(nqtot))
       IF (.NOT.ALLOCATED(is_rad))         ALLOCATE(is_rad(nqtot))
       IF (.NOT.ALLOCATED(is_recomb))      ALLOCATE(is_recomb(nqtot))
       IF (.NOT.ALLOCATED(is_recomb_qset)) THEN
         ALLOCATE(is_recomb_qset(nqtot))
       ENDIF
       IF (.NOT.ALLOCATED(is_recomb_qotf)) THEN
         ALLOCATE(is_recomb_qotf(nqtot))
       ENDIF
       IF (.NOT. allocated(is_condensable)) allocate(is_condensable(nq)) !LT
       IF (.NOT. allocated(is_rgcs)) allocate(is_rgcs(nq)) !LT
       IF (.NOT. allocated(constants_mass)) allocate(constants_mass(nq))
       IF (.NOT. allocated(constants_delta_vapH)) allocate(constants_delta_vapH(nq))
       IF (.NOT. allocated(constants_Tref)) allocate(constants_Tref(nq))
       IF (.NOT. allocated(constants_Pref)) allocate(constants_Pref(nq))
       IF (.NOT. allocated(constants_epsi_generic)) allocate(constants_epsi_generic(nq))
       IF (.NOT. allocated(constants_RLVTT_generic)) allocate(constants_RLVTT_generic(nq))
       IF (.NOT. allocated(constants_metallicity_coeff)) allocate(constants_metallicity_coeff(nq))
       IF (.NOT. allocated(constants_RCPV_generic)) allocate(constants_RCPV_generic(nq))
       IF (.NOT.ALLOCATED(half_life))         ALLOCATE(half_life(nqtot))
       IF (.NOT.ALLOCATED(top_prod))         ALLOCATE(top_prod(nqtot))
       IF (.NOT.ALLOCATED(bot_prod))         ALLOCATE(bot_prod(nqtot))

       !! initialization
       alpha_lift(:)     = 0.
       alpha_devil(:)    = 0.
       mmol(:)           = 0.
       aki(:)            = 0.
       cpi(:)            = 0.
       is_chim(:)        = 0
       is_rad(:)         = 0
       is_recomb(:)      = 0
       is_recomb_qset(:) = 0
       is_recomb_qotf(:) = 0
       half_life(:)      = 0.
       top_prod(:)       = 0.
       bot_prod(:)       = 0.
       
       ! Added by JVO 2017 : these arrays are handled later
       ! -> initialization is the least we can do, please !!!
       radius(:)=0.
       qext(:)=0.

       ! For condensable tracers, by Lucas Teinturier and Noé Clément (2022)

       is_condensable(:)= 0
       is_rgcs(:) = 0
       constants_mass(:)=0
       constants_delta_vapH(:)=0
       constants_Tref(:)=0
       constants_Pref(:)=0
       constants_epsi_generic(:)=0
       constants_RLVTT_generic(:)=0
       constants_metallicity_coeff(:)=0
       constants_RCPV_generic(:)=0

       rho_q(:) = 0. !need to be init here if we want to read it from modern traceur with get_tracdat


! Initialization: Read tracers names from traceur.def
        do iq=1,nq 
          if (is_master) read(407,'(A)') tracline
          call bcast(tracline)
          blank = index(tracline,' ') ! Find position of 1st blank in tracline
          noms(iq) = tracline(1:blank) !ensure that in Modern-trac case, noms is actually the name and not all properties
        enddo

! Identify tracers by their names: (and set corresponding values of mmol)
      ! 0. initialize tracer indexes to zero:
      ! NB: igcm_* indexes are commons in 'tracer.h'
      do iq=1,nq
        igcm_dustbin(iq)=0
      enddo
      igcm_dust_mass=0
      igcm_dust_number=0
      igcm_h2o_vap=0
      igcm_h2o_ice=0
      igcm_co2=0
      igcm_co=0
      igcm_o=0
      igcm_o1d=0
      igcm_o2=0
      igcm_o3=0
      igcm_h=0
      igcm_h2=0
      igcm_oh=0
      igcm_ho2=0
      igcm_h2o2=0
      igcm_n2=0
      igcm_n=0
      igcm_n2d=0
      igcm_no=0
      igcm_no2=0
      igcm_ar=0
      igcm_ar_n2=0
      igcm_co2_ice=0

      igcm_ch4=0
      igcm_ch3=0
      igcm_ch=0
      igcm_3ch2=0
      igcm_1ch2=0
      igcm_cho=0
      igcm_ch2o=0
      igcm_ch3o=0
      igcm_c=0
      igcm_c2=0
      igcm_c2h=0
      igcm_c2h2=0
      igcm_c2h3=0
      igcm_c2h4=0
      igcm_c2h6=0
      igcm_ch2co=0
      igcm_ch3co=0
      igcm_hcaer=0

      ! 1. find dust tracers
      count=0

      ! 2. find chemistry and water tracers
      do iq=1,nq
        if (noms(iq).eq."co2") then
          igcm_co2=iq
          mmol(igcm_co2)=44.010
          count=count+1
!          write(*,*) 'co2: count=',count
        endif
        if (noms(iq).eq."co2_ice") then
          igcm_co2_ice=iq
          mmol(igcm_co2_ice)=44.010
          count=count+1
!          write(*,*) 'co2_ice: count=',count
        endif
        if (noms(iq).eq."h2o_vap") then
          igcm_h2o_vap=iq
          mmol(igcm_h2o_vap)=18.01528
          count=count+1
!          write(*,*) 'h2o_vap: count=',count
        endif
        if (noms(iq).eq."h2o_ice") then
          igcm_h2o_ice=iq
          mmol(igcm_h2o_ice)=18.01528
          count=count+1
!          write(*,*) 'h2o_ice: count=',count
        endif
        if (noms(iq).eq."co") then
          igcm_co=iq
          mmol(igcm_co)=28.010
          count=count+1
        endif
        if (noms(iq).eq."o") then
          igcm_o=iq
          mmol(igcm_o)=15.9994
          count=count+1
        endif
        if (noms(iq).eq."o1d") then
          igcm_o1d=iq
          mmol(igcm_o1d)=15.9994
          count=count+1
        endif
        if (noms(iq).eq."o2") then
          igcm_o2=iq
          mmol(igcm_o2)=31.998
          count=count+1
        endif
        if (noms(iq).eq."o3") then
          igcm_o3=iq
          mmol(igcm_o3)=47.9982
          count=count+1
        endif
        if (noms(iq).eq."h") then
          igcm_h=iq
          mmol(igcm_h)=1.00784
          count=count+1
        endif
        if (noms(iq).eq."h2") then
          igcm_h2=iq
          mmol(igcm_h2)=2.016
          count=count+1
        endif
        if (noms(iq).eq."oh") then
          igcm_oh=iq
          mmol(igcm_oh)=17.008
          count=count+1
        endif
        if (noms(iq).eq."ho2") then
          igcm_ho2=iq
          mmol(igcm_ho2)=33.0067
          count=count+1
        endif
        if (noms(iq).eq."h2o2") then
          igcm_h2o2=iq
          mmol(igcm_h2o2)=34.0147
          count=count+1
        endif
        if (noms(iq).eq."n2") then
          igcm_n2=iq
          mmol(igcm_n2)=28.0134
          count=count+1
        endif
        if (noms(iq).eq."ch4") then
          igcm_ch4=iq
          mmol(igcm_ch4)=16.0425
          count=count+1
        endif
        if (noms(iq).eq."ar") then
          igcm_ar=iq
          mmol(igcm_ar)=39.948
          count=count+1
        endif
        if (noms(iq).eq."n") then
          igcm_n=iq
          mmol(igcm_n)=14.0067
          count=count+1
        endif
        if (noms(iq).eq."no") then
          igcm_no=iq
          mmol(igcm_no)=30.0061
          count=count+1
        endif
        if (noms(iq).eq."no2") then
          igcm_no2=iq
          mmol(igcm_no2)=46.0055 
          count=count+1
        endif
        if (noms(iq).eq."n2d") then
          igcm_n2d=iq
          mmol(igcm_n2d)=28.0134
          count=count+1
        endif
        if (noms(iq).eq."ch3") then
          igcm_ch3=iq
          mmol(igcm_ch3)=15.034
          count=count+1
        endif
        if (noms(iq).eq."ch") then
          igcm_ch=iq
          mmol(igcm_ch)=13.0186
          count=count+1
        endif
        if (noms(iq).eq."3ch2") then
          igcm_3ch2=iq
          mmol(igcm_3ch2)=14.0266
          count=count+1
        endif
        if (noms(iq).eq."1ch2") then
          igcm_1ch2=iq
          mmol(igcm_1ch2)=14.0266
          count=count+1
        endif
        if (noms(iq).eq."cho") then
          igcm_cho=iq
          mmol(igcm_cho)=29.018
          count=count+1
        endif
        if (noms(iq).eq."ch2o") then
          igcm_ch2o=iq
          mmol(igcm_ch2o)=30.031
          count=count+1
        endif
        if (noms(iq).eq."ch3o") then
          igcm_ch3o=iq
          mmol(igcm_ch3o)=31.0339
          count=count+1
        endif
        if (noms(iq).eq."c") then
          igcm_c=iq
          mmol(igcm_c)=12.0107
          count=count+1
        endif
        if (noms(iq).eq."c2") then
          igcm_c2=iq
          mmol(igcm_c2)=24.0214 
          count=count+1
        endif
        if (noms(iq).eq."c2h") then
          igcm_c2h=iq
          mmol(igcm_c2h)=25.0293
          count=count+1
        endif
        if (noms(iq).eq."c2h2") then
          igcm_c2h2=iq
          mmol(igcm_c2h2)=26.038
          count=count+1
        endif
        if (noms(iq).eq."c2h3") then
          igcm_c2h3=iq
          mmol(igcm_c2h3)=27.0452
          count=count+1
        endif
        if (noms(iq).eq."c2h4") then
          igcm_c2h4=iq
          mmol(igcm_c2h4)=28.053
          count=count+1
        endif
        if (noms(iq).eq."c2h6") then
          igcm_c2h6=iq
          mmol(igcm_c2h6)=30.069
          count=count+1
        endif
        if (noms(iq).eq."ch2co") then
          igcm_ch2co=iq
          mmol(igcm_ch2co)=42.0367
          count=count+1
        endif
        if (noms(iq).eq."ch3co") then
          igcm_ch3co=iq
          mmol(igcm_ch3co)=43.0446
          count=count+1
        endif
        if (noms(iq).eq."hcaer") then
          igcm_hcaer=iq
          mmol(igcm_hcaer)=50.
          count=count+1
        endif

      enddo ! of do iq=1,nq

      ! 3. find condensable traceurs different from h2o and co2
      do iq=1,nq
        if ((index(noms(iq),"vap") .ne. 0) .and. (index(noms(iq),"h2o") .eq. 0) .and. (index(noms(iq),"co2") .eq. 0)) then
          count=count+1
        endif
        if ((index(noms(iq),"ice") .ne. 0) .and. (index(noms(iq),"h2o") .eq. 0) .and. (index(noms(iq),"co2") .eq. 0)) then
          count=count+1
        endif

      enddo ! of do iq=1,nq
      
      ! check that we identified all tracers:
      if (count.ne.nq) then
        write(*,*) "initracer: found only ",count," tracers"
        write(*,*) "               expected ",nq
        do iq=1,count
          write(*,*)'      ',iq,' ',trim(noms(iq))
        enddo
!        stop
      else
        write(*,*) "initracer: found all expected tracers, namely:"
        do iq=1,nq
          write(*,*)'      ',iq,' ',trim(noms(iq))
        enddo
      endif

      ! Get data of tracers. Need to rewind traceur.def first
      if (is_master) then
       rewind(407)
       do 
        read(407,'(A)') tracline
        if (index(tracline,"#") .ne. 1) then 
          exit
        endif
       enddo
      endif
      do iq=1,nqtot
        if (is_master) read(407,'(A)') tracline
        call bcast(tracline)
        call get_tracdat(iq, tracline)
      enddo

      if (is_master) close(407)

      ! Get specific data of condensable tracers
      do iq=1,nq
        if((is_condensable(iq)==1)) then
                write(*,*) "There is a specie which is condensable, for generic condensation : ", noms(iq)
                write(*,*) 'looking specie parameters for : ',noms(iq)(1:len(trim(noms(iq)))-4)
                call specie_parameters_table(noms(iq)(1:len(trim(noms(iq)))-4))
                constants_mass(iq)=m
                constants_delta_vapH(iq)=delta_vapH
                constants_Tref(iq)=Tref
                constants_Pref(iq)=Pref
                constants_epsi_generic(iq)=epsi_generic
                constants_RLVTT_generic(iq)=RLVTT_generic
                constants_metallicity_coeff(iq)=metallicity_coeff
                constants_RCPV_generic(iq)=RCPV_generic
        else
                write(*,*) "This tracer is not condensable, for generic condensation :  : ", noms(iq)
                write(*,*) "We keep condensable constants at zero"
        endif !(is_condensable(iq)==1) .and. (index(noms(iq),"vap") .ne. 0))
      enddo ! iq=1,nq

      ! Calculate number of species in the chemistry
      nesp = sum(is_chim)
      write(*,*) 'Number of species in the chemistry nesp = ',nesp

      ! Calculate number of generic tracers
      ngt = sum(is_condensable)
      write(*,*) 'Number of generic tracer is  ngt = ',ngt

      ! Calculate number of radiative generic condensable species
      n_rgcs = sum(is_rgcs)
      write(*,*)'Number of Radiative Generic Condensable Species is n_rgcs = ',n_rgcs
      if (n_rgcs> ngt/2) then 
        write(*,*) 'You have more Radiative Generic Condensable Species than Generic Condensable Species'
        write(*,*)'This is not possible: check your Modern traceur.def'
        call abort_physic("initracer, issue with # of RGCS and GCS",1)
      endif

!     Processing modern traceur options
      if(moderntracdef) then
        call ini_recombin
      endif

!------------------------------------------------------------
!     Initialisation tracers ....
!------------------------------------------------------------
      ! rho_q(1:nq)=0

      rho_dust=2500.  ! Mars dust density (kg.m-3)
      rho_ice=920.    ! Water ice density (kg.m-3)
      rho_co2=1620.   ! CO2 ice density (kg.m-3)

!     Initialization for water vapor
!     ------------------------------
      if(water) then
        radius(igcm_h2o_vap)=0.
        Qext(igcm_h2o_vap)=0.
        alpha_lift(igcm_h2o_vap) =0.
        alpha_devil(igcm_h2o_vap)=0.
        qextrhor(igcm_h2o_vap)= 0.

         !! this is defined in surfdat_h.F90
         IF (.not.ALLOCATED(dryness)) ALLOCATE(dryness(ngrid))

         do ig=1,ngrid
           dryness(ig) = 1.
         enddo


           radius(igcm_h2o_ice)=3.e-6
           rho_q(igcm_h2o_ice)=rho_ice
           Qext(igcm_h2o_ice)=0.
!           alpha_lift(igcm_h2o_ice) =0.
!           alpha_devil(igcm_h2o_ice)=0.
           qextrhor(igcm_h2o_ice)= (3./4.)*Qext(igcm_h2o_ice) &
            / (rho_ice*radius(igcm_h2o_ice))


      end if  ! (water)


!
!     some extra (possibly redundant) sanity checks for tracers:
!     ---------------------------------------------------------
       if (water) then
       ! verify that we indeed have h2o_vap and h2o_ice tracers
         if (igcm_h2o_vap.eq.0) then
           write(*,*) "initracer: error !!"
           write(*,*) "  cannot use water option without ",&
                     "an h2o_vap tracer !"
           stop
         endif
         if (igcm_h2o_ice.eq.0) then
           write(*,*) "initracer: error !!"
           write(*,*) "  cannot use water option without ",&
                     "an h2o_ice tracer !"
           stop
         endif
       endif


!     Output for records:
!     ~~~~~~~~~~~~~~~~~~
      write(*,*)
      Write(*,*) '******** initracer : dust transport parameters :'
      write(*,*) 'alpha_lift = ', alpha_lift
      write(*,*) 'alpha_devil = ', alpha_devil
      write(*,*) 'radius  = ', radius
      write(*,*) 'Qext  = ', qext 
      write(*,*)

      contains

      subroutine get_tracdat(iq,tracline)
        !-------------------ADDING NEW OPTIONS-------------------
        ! Duplicate if sentence for adding new options
        ! Do not forget to add the new variables in tracer_h.F90
        ! Do not forget to allocate and initialize the new variables above
        ! Please update list of options in "LMDZ.GENERIC/deftank/traceur.def.modern"
        !-------------------------------------------------------
          use tracer_h
          implicit none
          integer,           intent(in) :: iq       ! tracer index
          character(len=500),intent(in) :: tracline ! traceur.def lines with parameters
  
          read(tracline,*) noms(iq)
          ! JVO 20 : We should add a sanity check aborting when duplicates in names !
          write(*,*)"initracer: iq=",iq,"noms(iq)=",trim(noms(iq))
          ! option mmol
          if (index(tracline,'mmol='   ) /= 0) then
              read(tracline(index(tracline,'mmol=')+len('mmol='):),*)&
                  mmol(iq)
              write(*,*) ' Parameter value (traceur.def) : mmol=', &
                  mmol(iq)
          else
              write(*,*) ' Parameter value (default)     : mmol=', &
                 mmol(iq)
          end if
          ! option aki 
          if (index(tracline,'aki='   ) /= 0) then
              read(tracline(index(tracline,'aki=')+len('aki='):),*) &
                  aki(iq)
              write(*,*) ' Parameter value (traceur.def) : aki=', &
                  aki(iq)
          else
              write(*,*) ' Parameter value (default)     : aki=', &
                  aki(iq)
          end if
          ! option cpi 
          if (index(tracline,'cpi='   ) /= 0) then
              read(tracline(index(tracline,'cpi=')+len('cpi='):),*) &
                  cpi(iq)
              write(*,*) ' Parameter value (traceur.def) : cpi=', &
                  cpi(iq)
          else
              write(*,*) ' Parameter value (default)     : cpi=', &
                  cpi(iq)
          end if
          ! option is_chim 
          if (index(tracline,'is_chim='   ) /= 0) then
          read(tracline(index(tracline,'is_chim=')+len('is_chim='):),*) &
                  is_chim(iq)
              write(*,*) ' Parameter value (traceur.def) : is_chim=', &
                  is_chim(iq)
          else
              write(*,*) ' Parameter value (default)     : is_chim=', &
                  is_chim(iq)
          end if
          ! option is_rad 
          if (index(tracline,'is_rad=') /= 0) then
              read(tracline(index(tracline,'is_rad=') &
              +len('is_rad='):),*) is_rad(iq)
              write(*,*) ' Parameter value (traceur.def) : is_rad=', &
              is_rad(iq)
          else
              write(*,*) ' Parameter value (default)     : is_rad=',  &
              is_rad(iq)
          end if
          ! option is_recomb
          if (index(tracline,'is_recomb=') /= 0) then
              read(tracline(index(tracline,'is_recomb=') &
              +len('is_recomb='):),*) is_recomb(iq)
              write(*,*) ' Parameter value (traceur.def) : is_recomb=', &
              is_recomb(iq)
          else
              write(*,*) ' Parameter value (default)     : is_recomb=', &
              is_recomb(iq)
          end if
          ! option is_recomb_qset
          if (index(tracline,'is_recomb_qset=') /= 0) then
              read(tracline(index(tracline,'is_recomb_qset=') &
              +len('is_recomb_qset='):),*) is_recomb_qset(iq)
              write(*,*) ' Parameter value (traceur.def) :'// &
              ' is_recomb_qset=', &
              is_recomb_qset(iq)
          else
              write(*,*) ' Parameter value (default)     :'// &
              ' is_recomb_qset=', &
              is_recomb_qset(iq)
          endif
          ! option is_recomb_qotf
          if (index(tracline,'is_recomb_qotf=') /= 0) then
              read(tracline(index(tracline,'is_recomb_qotf=') &
              +len('is_recomb_qotf='):),*) is_recomb_qotf(iq)
              write(*,*) ' Parameter value (traceur.def) :'// &
              ' is_recomb_qotf=', &
              is_recomb_qotf(iq)
          else
              write(*,*) ' Parameter value (default)     :'// &
              ' is_recomb_qotf=',  &
              is_recomb_qotf(iq)
          end if
          !option is_condensable (LT)
          if (index(tracline,'is_condensable=') /=0) then 
            read(tracline(index(tracline,'is_condensable=') &
              +len('is_condensable='):),*) is_condensable(iq)
            write(*,*) ' Parameter value (traceur.def) :'// &
              ' is_condensable=', is_condensable(iq)       
          else
              write(*,*) ' Parameter value (default)     :'// &
              ' is_condensable=', is_condensable(iq)       
          endif
          !option radius
          if (index(tracline,'radius=') .ne. 0) then 
            read(tracline(index(tracline,'radius=') &
              +len('radius='):),*) radius(iq)
            write(*,*)'Parameter value (traceur.def) :'// &
              "radius=",radius(iq), "m"
          else
            write(*,*) ' Parameter value (default)     :'// &
            ' radius=', radius(iq)," m"     
          endif
          !option rho
          if (index(tracline,'rho=') .ne. 0) then 
            read(tracline(index(tracline,'rho=') &
              +len('rho='):),*) rho_q(iq)
            write(*,*)'Parameter value (traceur.def) :'// &
              "rho=",rho_q(iq)
          else
            write(*,*) ' Parameter value (default)     :'// &
              ' rho=', rho_q(iq)       
          endif
          !option is_rgcs 
          if (index(tracline,'is_rgcs') .ne. 0) then 
            read(tracline(index(tracline,'is_rgcs=') &
              +len('is_rgcs='):),*) is_rgcs(iq)
            write(*,*)'Parameter value (traceur.def) :'// &
              'is_rgcs=',is_rgcs(iq)
          else
            write(*,*)'Parameter value (default) : '// &
              'is_rgcs = ',is_rgcs(iq)
          end if
             ! option top_prod 
              if (index(tracline,'top_prod=') .ne. 0) then
                  read(tracline(index(tracline,'top_prod=') &
                  +len('top_prod='):),*) top_prod(iq)
                  write(*,*) ' Parameter value (traceur.def) : top_prod=', &
                  top_prod(iq)
              else
                  write(*,*) ' Parameter value (default)     : top_prod=',  &
                  top_prod(iq)
              end if
              ! option bot_prod 
              if (index(tracline,'bot_prod=') .ne. 0) then
                  read(tracline(index(tracline,'bot_prod=') &
                  +len('bot_prod='):),*) bot_prod(iq)
                  write(*,*) ' Parameter value (traceur.def) : bot_prod=', &
                  bot_prod(iq)
              else
                  write(*,*) ' Parameter value (default)     : bot_prod=',  &
                  bot_prod(iq)
              end if
      end subroutine get_tracdat

      end

