MODULE job_timelimit_mod

!***********************************************************************
! DESCRIPTION:
!    Retrieves the time limit (in seconds) for a given job ID via
!    SLURM (squeue) or PBS/TORQUE (qstat).
!***********************************************************************

implicit none

real            :: timelimit = 86400. ! Time limit for the job: 86400 s = 24 h by default
real, parameter :: antetime  = 3600.  ! Anticipation time to prevent reaching the job time limit: 3600 s = 1 h by default
logical         :: timewall = .false. ! Flag to enable the program self-termination before timeout

!=======================================================================
contains
!=======================================================================

SUBROUTINE get_job_timelimit(arg)

implicit none

!---- Arguments
character(*), intent(in) :: arg

!---- Variables
logical        :: num_str
character(256) :: chtimelimit ! Time limit for the job outputted by the SLURM or PBS/TORQUE command
integer        :: i, cstat, days, hours, minutes, seconds
character(1)   :: sep

!---- Code
! Check that the job ID is numeric
num_str = .true.
do i = 1,len_trim(arg)
    if (arg(i:i) < '0' .or. arg(i:i) > '9') then
        num_str = .false.
        exit
    endif
enddo
if (.not. num_str) error stop 'Error: job ID must be numeric.'

! Try SLURM (squeue)
call execute_command_line('squeue -j '//trim(adjustl(arg))//' -h --Format TimeLimit > tmp_cmdout.txt',cmdstat = cstat)
if (cstat /= 0) then
    ! On failure, try PBS/TORQUE (qstat)
    call execute_command_line('qstat -f '//trim(adjustl(arg))//' | grep "Walltime" | awk ''{print $3}'' > tmp_cmdout.txt',cmdstat = cstat)
    if (cstat > 0) then
        error stop 'Error: command execution failed!'
    else if (cstat < 0) then
        error stop 'Error: command execution not supported (neither SLURM nor PBS/TORQUE is installed)!'
    endif
endif

! Read the output
open(1,file = 'tmp_cmdout.txt',status = 'old')
read(1,'(a)') chtimelimit
close(1)
chtimelimit = trim(adjustl(chtimelimit))

! Remove temporary file
call execute_command_line('rm tmp_cmdout.txt',cmdstat = cstat)
if (cstat > 0) then
    error stop 'Error: command execution failed!'
else if (cstat < 0) then
    error stop 'Error: command execution not supported!'
endif

! Parse D-HH:MM:SS, HH:MM:SS or MM:SS
if (index(chtimelimit,'-') > 0) then ! Format "D-HH:MM:SS"
    read(chtimelimit,'(i1,a1,i2,a1,i2,a1,i2)') days, sep, hours, sep, minutes, sep, seconds
    timelimit = days*86400 + hours*3600 + minutes*60 + seconds
else if (index(chtimelimit,':') > 0 .and. len_trim(chtimelimit) > 5) then ! Format "HH:MM:SS"
    read(chtimelimit,'(i2,a1,i2,a1,i2)') hours, sep, minutes, sep, seconds
    timelimit = hours*3600 + minutes*60 + seconds
else ! Format "MM:SS"
    read(chtimelimit,'(i2,a1,i2)') minutes, sep, seconds
    timelimit = minutes*60 + seconds
endif

timewall = .true.

END SUBROUTINE get_job_timelimit

END MODULE job_timelimit_mod
