MODULE iostart

PRIVATE
    INTEGER,SAVE :: nid_start 
    INTEGER,SAVE :: nid_restart
!$OMP THREADPRIVATE(nid_start,nid_restart)    

    INTEGER,SAVE :: idim1,idim2,idim3,idim4
!$OMP THREADPRIVATE(idim1,idim2,idim3,idim4)
    INTEGER,PARAMETER :: length=100
    
    INTERFACE get_field
      MODULE PROCEDURE Get_field_r1,Get_field_r2,Get_field_r3
    END INTERFACE get_field
    
    INTERFACE get_var
      MODULE PROCEDURE get_var_r0,Get_var_r1,Get_var_r2,Get_var_r3
    END INTERFACE get_var

    INTERFACE put_field
      MODULE PROCEDURE put_field_r1,put_field_r2,put_field_r3
    END INTERFACE put_field

    INTERFACE put_var
      MODULE PROCEDURE put_var_r0,put_var_r1,put_var_r2,put_var_r3
    END INTERFACE put_var

    PUBLIC get_field,get_var,put_field,put_var
    PUBLIC open_startphy,close_startphy,open_restartphy,close_restartphy
    
CONTAINS

  SUBROUTINE open_startphy(filename)
  USE netcdf, ONLY: NF90_OPEN, nf90_strerror, NF90_NOWRITE, NF90_NOERR
  USE mod_phys_lmdz_para, ONLY: is_master
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN) :: filename
    INTEGER          :: ierr

    IF (is_master) THEN
      ierr = NF90_OPEN (filename, NF90_NOWRITE,nid_start)
      IF (ierr.NE.NF90_NOERR) THEN
        write(*,*)'open_startphy: problem opening file '//trim(filename)
        write(*,*)trim(nf90_strerror(ierr))
        CALL abort_physic("open_startphy","Cannot open file",1)
      ENDIF
    ENDIF
   
  END SUBROUTINE open_startphy

  SUBROUTINE Close_startphy
  USE netcdf, ONLY: NF90_CLOSE
  USE mod_phys_lmdz_para, ONLY: is_master
  IMPLICIT NONE
    INTEGER          :: ierr

    IF (is_master) THEN
        ierr = NF90_CLOSE (nid_start)
    ENDIF

  END SUBROUTINE close_startphy


  FUNCTION Inquire_Field(Field_name)
  USE netcdf, ONLY: NF90_INQ_VARID, NF90_NOERR
  USE mod_phys_lmdz_para, only: is_master, bcast
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN) :: Field_name
    LOGICAL :: inquire_field
    INTEGER :: varid
    INTEGER :: ierr
    
    IF (is_master) THEN
      ierr=NF90_INQ_VARID(nid_start,Field_name,varid)
      IF (ierr==NF90_NOERR) THEN
        Inquire_field=.TRUE.
      ELSE
        Inquire_field=.FALSE.
      ENDIF
    ENDIF

    CALL bcast(Inquire_field)

  END FUNCTION Inquire_Field
  
 
  SUBROUTINE Get_Field_r1(field_name,field,found)
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN)    :: Field_name
    REAL,INTENT(INOUT)               :: Field(:)
    LOGICAL,INTENT(OUT),OPTIONAL   :: found 

    IF (PRESENT(found)) THEN
      CALL Get_field_rgen(field_name,field,1,found)
    ELSE
      CALL Get_field_rgen(field_name,field,1)
    ENDIF
      
  END SUBROUTINE Get_Field_r1
  
  SUBROUTINE Get_Field_r2(field_name,field,found)
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN)    :: Field_name
    REAL,INTENT(INOUT)               :: Field(:,:)
    LOGICAL,INTENT(OUT),OPTIONAL   :: found 

    IF (PRESENT(found)) THEN
      CALL Get_field_rgen(field_name,field,size(field,2),found)
    ELSE
      CALL Get_field_rgen(field_name,field,size(field,2))
    ENDIF

      
  END SUBROUTINE Get_Field_r2
  
  SUBROUTINE Get_Field_r3(field_name,field,found)
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN)    :: Field_name
    REAL,INTENT(INOUT)               :: Field(:,:,:)
    LOGICAL,INTENT(OUT),OPTIONAL   :: found 

    IF (PRESENT(found)) THEN
      CALL Get_field_rgen(field_name,field,size(field,2)*size(field,3),found)
    ELSE
      CALL Get_field_rgen(field_name,field,size(field,2)*size(field,3))
    ENDIF
      
  END SUBROUTINE Get_Field_r3
  
  SUBROUTINE Get_field_rgen(field_name,field,field_size,found)
  USE netcdf, ONLY: NF90_INQ_VARID, NF90_GET_VAR, NF90_NOERR
  USE dimphy, ONLY: klon ! number of columns on local grid
  USE geometry_mod, ONLY: ind_cell_glo
  USE mod_grid_phy_lmdz, ONLY: klon_glo ! number of columns on global grid
  USE mod_phys_lmdz_para, ONLY: is_master, bcast, scatter, gather
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN) :: Field_name
    INTEGER,INTENT(IN) :: field_size
    REAL,INTENT(OUT) :: field(klon,field_size)
    LOGICAL,OPTIONAL,INTENT(OUT) :: found
    
    REAL    :: field_glo(klon_glo,field_size) ! field on global grid
    REAL :: field_glo_tmp(klon_glo,field_size)
    INTEGER :: ind_cell_glo_glo(klon_glo) ! cell indexes on global grid
    LOGICAL :: tmp_found
    INTEGER :: varid
    INTEGER :: ierr, i
    
    ! gather columns indexes on global grid
    CALL gather(ind_cell_glo,ind_cell_glo_glo)

    IF (is_master) THEN
  
      ierr=NF90_INQ_VARID(nid_start,Field_name,varid)
      
      IF (ierr==NF90_NOERR) THEN
        CALL body(field_glo_tmp)
        tmp_found=.TRUE.
      ELSE
        tmp_found=.FALSE.
      ENDIF
    
    ENDIF ! of IF (is_master)
    
    CALL bcast(tmp_found)

    IF (tmp_found) THEN
      IF (is_master) THEN
        ! reorder columns according to ind_cell_glo(:) indexes
        DO i=1,klon_glo
          field_glo(i,:)=field_glo_tmp(ind_cell_glo_glo(i),:)
        ENDDO
      ENDIF
      CALL scatter(field_glo,field)
    ENDIF
    
    IF (PRESENT(found)) THEN
      found=tmp_found
    ELSE
      IF (.NOT. tmp_found) THEN
        PRINT*, 'get_field_rgen: Field <'//field_name//'> not found'
        CALL abort_physic("get_field_rgen","Failed to find field",1)
      ENDIF
    ENDIF
 
    
    CONTAINS
     
     SUBROUTINE body(field_glo)
       REAL :: field_glo(klon_glo*field_size)
         ierr=NF90_GET_VAR(nid_start,varid,field_glo)
         IF (ierr/=NF90_NOERR) THEN
           ! La variable exist dans le fichier mais la lecture a echouee. 
           PRINT*, 'get_field_rgen: Failed reading <'//field_name//'>'
           ! It looks like the following is for Earth physics only...
           !IF (field_name=='CLWCON' .OR. field_name=='RNEBCON' .OR. field_name=='RATQS') THEN
           !   ! Essaye de lire le variable sur surface uniqument, comme fait avant
           !   field_glo(:)=0.
           !   ierr=NF90_GET_VAR(nid_start,varid,field_glo(1:klon_glo))
           !   IF (ierr/=NF90_NOERR) THEN
           !      PRINT*, 'phyetat0: Lecture echouee aussi en 2D pour <'//field_name//'>'
           !      CALL abort
           !   ELSE
           !      PRINT*, 'phyetat0: La variable <'//field_name//'> lu sur surface seulement'!, selon ancien format, le reste mis a zero'
           !   END IF
           !ELSE
          CALL abort_physic("get_field_rgen","Failed to read field",1)
           !ENDIF
         ENDIF

     END SUBROUTINE body

  END SUBROUTINE Get_field_rgen
  

  SUBROUTINE get_var_r0(var_name,var,found)
  IMPLICIT NONE  
    CHARACTER(LEN=*),INTENT(IN)  :: var_name
    REAL,INTENT(INOUT)             :: var
    LOGICAL,OPTIONAL,INTENT(OUT) :: found

    REAL                         :: varout(1)
    
    IF (PRESENT(found)) THEN
      CALL Get_var_rgen(var_name,varout,size(varout),found)
    ELSE
      CALL Get_var_rgen(var_name,varout,size(varout))
    ENDIF
    var=varout(1)
 
  END SUBROUTINE get_var_r0

  SUBROUTINE get_var_r1(var_name,var,found)
  IMPLICIT NONE  
    CHARACTER(LEN=*),INTENT(IN)  :: var_name
    REAL,INTENT(INOUT)             :: var(:)
    LOGICAL,OPTIONAL,INTENT(OUT) :: found
    
    IF (PRESENT(found)) THEN
      CALL Get_var_rgen(var_name,var,size(var),found)
    ELSE
      CALL Get_var_rgen(var_name,var,size(var))
    ENDIF
  
  END SUBROUTINE get_var_r1

  SUBROUTINE get_var_r2(var_name,var,found)
  IMPLICIT NONE  
    CHARACTER(LEN=*),INTENT(IN)  :: var_name
    REAL,INTENT(OUT)             :: var(:,:)
    LOGICAL,OPTIONAL,INTENT(OUT) :: found
    
    IF (PRESENT(found)) THEN
      CALL Get_var_rgen(var_name,var,size(var),found)
    ELSE
      CALL Get_var_rgen(var_name,var,size(var))
    ENDIF
  
  END SUBROUTINE get_var_r2

  SUBROUTINE get_var_r3(var_name,var,found)
  IMPLICIT NONE  
    CHARACTER(LEN=*),INTENT(IN)  :: var_name
    REAL,INTENT(INOUT)             :: var(:,:,:)
    LOGICAL,OPTIONAL,INTENT(OUT) :: found
    
    IF (PRESENT(found)) THEN
      CALL Get_var_rgen(var_name,var,size(var),found)
    ELSE
      CALL Get_var_rgen(var_name,var,size(var))
    ENDIF
  
  END SUBROUTINE get_var_r3

  SUBROUTINE Get_var_rgen(var_name,var,var_size,found)
  USE netcdf, ONLY: NF90_INQ_VARID, NF90_GET_VAR, NF90_NOERR
  USE mod_phys_lmdz_para, ONLY: is_master, bcast
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN) :: var_name
    INTEGER,INTENT(IN) :: var_size
    REAL,INTENT(OUT) :: var(var_size)
    LOGICAL,OPTIONAL,INTENT(OUT) :: found
    
    LOGICAL :: tmp_found
    INTEGER :: varid
    INTEGER :: ierr
    
    IF (is_master) THEN
  
      ierr=NF90_INQ_VARID(nid_start,var_name,varid)
      
      IF (ierr==NF90_NOERR) THEN
        ierr=NF90_GET_VAR(nid_start,varid,var)
        IF (ierr/=NF90_NOERR) THEN
          PRINT*, 'phyetat0: Failed loading <'//trim(var_name)//'>'
          CALL abort_physic("get_var_rgen","Failed to read variable",1)
        ENDIF
        tmp_found=.TRUE.
      ELSE
        tmp_found=.FALSE.
      ENDIF
    
    ENDIF
    
    CALL bcast(tmp_found)

    IF (tmp_found) THEN
      CALL bcast(var)
    ENDIF
    
    IF (PRESENT(found)) THEN
      found=tmp_found
    ELSE
      IF (.NOT. tmp_found) THEN
        PRINT*, 'phyetat0: Variable <'//trim(var_name)//'> not found'
        CALL abort_physic("get_var_rgen","Failed to read variable",1)
      ENDIF
    ENDIF

  END SUBROUTINE Get_var_rgen


  SUBROUTINE open_restartphy(filename)
  USE netcdf, ONLY: NF90_CREATE, NF90_CLOBBER, NF90_NOERR, nf90_strerror, &
                    NF90_PUT_ATT, NF90_GLOBAL, NF90_DEF_DIM, &
                    NF90_ENDDEF
  USE mod_phys_lmdz_para,ONLY: is_master
  USE mod_grid_phy_lmdz, ONLY: klon_glo
  USE dimphy, ONLY: klev, klevp1
  IMPLICIT NONE
    CHARACTER(LEN=*),INTENT(IN) :: filename
    INTEGER                     :: ierr
    
    IF (is_master) THEN
      ierr = NF90_CREATE(filename, NF90_CLOBBER, nid_restart)
      IF (ierr/=NF90_NOERR) THEN
          write(*,*)'open_restartphy: problem creating file '//trim(filename)
          write(*,*)trim(nf90_strerror(ierr))
          CALL abort_physic("open_restartphy","Failed creating file",1)
      ENDIF

      ierr = NF90_PUT_ATT (nid_restart, NF90_GLOBAL, "title","Fichier redemarrage physique")

      ierr = NF90_DEF_DIM (nid_restart, "index", length, idim1)
      ierr = NF90_DEF_DIM (nid_restart, "points_physiques", klon_glo, idim2)
      ierr = NF90_DEF_DIM (nid_restart, "horizon_vertical", klon_glo*klev, idim3)
      ierr = NF90_DEF_DIM (nid_restart, "horizon_klevp1", klon_glo*klevp1, idim4)

      ierr = NF90_ENDDEF(nid_restart)
    ENDIF

  END SUBROUTINE open_restartphy
  
  SUBROUTINE close_restartphy
  USE netcdf, ONLY: NF90_CLOSE
  USE mod_phys_lmdz_para, ONLY: is_master
  IMPLICIT NONE
    INTEGER          :: ierr

    IF (is_master) THEN
      ierr = NF90_CLOSE (nid_restart)
    ENDIF
 
  END SUBROUTINE close_restartphy

  
  SUBROUTINE put_field_r1(field_name,title,field)
  IMPLICIT NONE
  CHARACTER(LEN=*),INTENT(IN)    :: field_name
  CHARACTER(LEN=*),INTENT(IN)    :: title
  REAL,INTENT(IN)                :: field(:)
  
    CALL put_field_rgen(field_name,title,field,1)
  
  END SUBROUTINE put_field_r1

  SUBROUTINE put_field_r2(field_name,title,field)
  IMPLICIT NONE
  CHARACTER(LEN=*),INTENT(IN)    :: field_name
  CHARACTER(LEN=*),INTENT(IN)    :: title
  REAL,INTENT(IN)                :: field(:,:)
  
    CALL put_field_rgen(field_name,title,field,size(field,2))
  
  END SUBROUTINE put_field_r2

  SUBROUTINE put_field_r3(field_name,title,field)
  IMPLICIT NONE
  CHARACTER(LEN=*),INTENT(IN)    :: field_name
  CHARACTER(LEN=*),INTENT(IN)    :: title
  REAL,INTENT(IN)                :: field(:,:,:)
  
    CALL put_field_rgen(field_name,title,field,size(field,2)*size(field,3))
  
  END SUBROUTINE put_field_r3
  
  SUBROUTINE put_field_rgen(field_name,title,field,field_size)
  USE netcdf, ONLY: NF90_REDEF, NF90_ENDDEF, NF90_DEF_VAR, NF90_PUT_ATT, &
                    NF90_PUT_VAR, NF90_FLOAT, NF90_DOUBLE
  USE dimphy, ONLY: klon, klev, klevp1
  USE mod_grid_phy_lmdz, ONLY: klon_glo
  USE mod_phys_lmdz_para, ONLY: is_master, gather
  USE geometry_mod, ONLY: ind_cell_glo

  IMPLICIT NONE
  CHARACTER(LEN=*),INTENT(IN)    :: field_name
  CHARACTER(LEN=*),INTENT(IN)    :: title
  INTEGER,INTENT(IN)             :: field_size
  REAL,INTENT(IN)                :: field(klon,field_size)
  
  REAL :: field_glo(klon_glo,field_size)
  REAL :: field_glo_tmp(klon_glo,field_size)
  INTEGER :: ind_cell_glo_glo(klon_glo) ! cell indexes on global grid
  
  INTEGER                        :: ierr
  INTEGER                        :: nvarid
  INTEGER                        :: idim
  INTEGER :: i
   
    ! gather indexes on global grid
    CALL gather(ind_cell_glo,ind_cell_glo_glo)
    ! gather field on master
    CALL gather(field,field_glo_tmp)
    
    IF (is_master) THEN
      ! reorder columns
      DO i=1,klon_glo
        field_glo(ind_cell_glo_glo(i),:)=field_glo_tmp(i,:)
      ENDDO
      
      IF (field_size==1) THEN
        idim=idim2
      ELSE IF (field_size==klev) THEN
        idim=idim3
      ELSE IF (field_size==klevp1) THEN
        idim=idim4
      ELSE
        WRITE(*,*)"put_field_rgen error: issue with dimensions"
        CALL abort_physic("put_field_rgen","wrong field dimensions",1)
      ENDIF
         
      ierr = NF90_REDEF (nid_restart)
#ifdef NC_DOUBLE
      ierr = NF90_DEF_VAR (nid_restart, field_name, NF90_DOUBLE,(/ idim /),nvarid)
#else
      ierr = NF90_DEF_VAR (nid_restart, field_name, NF90_FLOAT,(/ idim /),nvarid)
#endif
      IF (LEN_TRIM(title) > 0) ierr = NF90_PUT_ATT (nid_restart,nvarid,"title", title)
      ierr = NF90_ENDDEF(nid_restart)
      ierr = NF90_PUT_VAR(nid_restart,nvarid,RESHAPE(field_glo,(/klon_glo*field_size/)))
    ENDIF
    
   END SUBROUTINE put_field_rgen  
  
   SUBROUTINE put_var_r0(var_name,title,var)
   IMPLICIT NONE
     CHARACTER(LEN=*),INTENT(IN) :: var_name
     CHARACTER(LEN=*),INTENT(IN) :: title
     REAL,INTENT(IN)             :: var
     REAL                        :: varin(1)
     
     varin(1)=var
     
     CALL put_var_rgen(var_name,title,varin,size(varin))

  END SUBROUTINE put_var_r0


   SUBROUTINE put_var_r1(var_name,title,var)
   IMPLICIT NONE
     CHARACTER(LEN=*),INTENT(IN) :: var_name
     CHARACTER(LEN=*),INTENT(IN) :: title
     REAL,INTENT(IN)             :: var(:)
     
     CALL put_var_rgen(var_name,title,var,size(var))

  END SUBROUTINE put_var_r1
 
  SUBROUTINE put_var_r2(var_name,title,var)
   IMPLICIT NONE
     CHARACTER(LEN=*),INTENT(IN) :: var_name
     CHARACTER(LEN=*),INTENT(IN) :: title
     REAL,INTENT(IN)             :: var(:,:)
     
     CALL put_var_rgen(var_name,title,var,size(var))

  END SUBROUTINE put_var_r2     
  
  SUBROUTINE put_var_r3(var_name,title,var)
   IMPLICIT NONE
     CHARACTER(LEN=*),INTENT(IN) :: var_name
     CHARACTER(LEN=*),INTENT(IN) :: title
     REAL,INTENT(IN)             :: var(:,:,:)
     
     CALL put_var_rgen(var_name,title,var,size(var))

  END SUBROUTINE put_var_r3

  SUBROUTINE put_var_rgen(var_name,title,var,var_size)
  USE netcdf, ONLY: NF90_REDEF, NF90_DEF_VAR, NF90_ENDDEF, NF90_PUT_VAR, &
                    NF90_PUT_ATT, NF90_FLOAT, NF90_DOUBLE
  USE mod_phys_lmdz_para, ONLY: is_master
  IMPLICIT NONE
     CHARACTER(LEN=*),INTENT(IN) :: var_name
     CHARACTER(LEN=*),INTENT(IN) :: title
     INTEGER,INTENT(IN)          :: var_size
     REAL,INTENT(IN)             :: var(var_size)
     
     INTEGER :: ierr
     INTEGER :: nvarid
         
    IF (is_master) THEN

      IF (var_size/=length) THEN
        WRITE(*,*)"put_var_rgen error: issue with dimensions"
        CALL abort_physic("put_var_rgen","wrong field dimensions",1)
      ENDIF
      
      ierr = NF90_REDEF (nid_restart)

#ifdef NC_DOUBLE
      ierr = NF90_DEF_VAR (nid_restart, var_name, NF90_DOUBLE,(/ idim1 /),nvarid)
#else
      ierr = NF90_DEF_VAR (nid_restart, var_name, NF90_FLOAT,(/ idim1 /),nvarid)
#endif
      IF (LEN_TRIM(title)>0) ierr = NF90_PUT_ATT (nid_restart,nvarid,"title", title)
      ierr = NF90_ENDDEF(nid_restart)
     
      ierr = NF90_PUT_VAR(nid_restart,nvarid,var)

    ENDIF
    
  END SUBROUTINE put_var_rgen     
    
END MODULE iostart
