#!/bin/bash
################################################################
### Launching script for a chained simulation of 1D PCM runs ###
################################################################

echo "The launching script is starting!"
echo "The output file is \"loglaunch.txt\"."
if [ "$1" = "bg" ]; then
    date
else
    nohup "$0" bg > loglaunch.txt 2>&1 &
    exit 1
fi

# A few parameters that might need be changed depending on your setup:
# Path to the arch.env to source:
source ../trunk/LMDZ.COMMON/arch.env

# Save the current value of LC_NUMERIC and set it to a locale that uses a dot as the decimal separator
OLD_LC_NUMERIC=$LC_NUMERIC
LC_NUMERIC=en_US.UTF-8


################################################################
#---------- Modify here the number of years to be simulated ------------
## set the number of Martian years:
n_myears=1000

#------------------ Modify here the name of PCM exe --------------------
## fill in the name of executable for PCM:
exePCM="testphys1d_29_phymars_seq.e"
################################################################


#------ Check if files/directories necessary for the script exist ------
dir=`pwd`
machine=`hostname`
address=`whoami`
if [ ! -f "$exePCM" ]; then
    echo "Error: file \"$exePCM\" does not exist in $dir!"
    exit 1
fi
if [ ! -d "out_PCM" ]; then
    mkdir out_PCM
fi
if [ ! -d "starts" ]; then
    mkdir starts
fi
if [ ! -d "diags" ]; then
    mkdir diags
fi

#---------------------------- Initialization ---------------------------
dir=`pwd`
machine=`hostname`
address=`whoami`
echo "This is a chained simulation for PCM runs in $dir on $machine."
echo "Number of years to be simulated: $n_myears Martian years."
myear=686.9725      # Number of Earth days in Martian year
eyear=365.256363004 # Number of days in Earth year
convert_years=$(echo "$myear/$eyear" | bc -l)
convert_years=$(printf "%.4f" $convert_years) # Rounding to the 4th decimal to respect the precision of Martian year
i_myear=0
iPCM=1
cp startfi.nc starts/
if [ -f "star1D.nc" ]; then
    cp star1D.txt starts/
fi

#---------------- Main loop to to run PCM year by year -----------------
while [ $i_myear -lt $n_myears ]; do
    # Run the PCM
    echo "Run PCM $iPCM: year $iPCM/$n_myears..."
    ./$exePCM > out_runPCM${iPCM} 2>&1
    if [ ! -f "restartfi.nc" ]; then # Check if run ended abnormally
        echo "Error: the run PCM $iPCM crashed!"
        exit 1
    fi
    # Copy data files and prepare the next run
    mv out_runPCM${iPCM} out_PCM/run${iPCM}
    mv diagfi.nc diags/diagfi${iPCM}.nc
    if [ -f "diagsoil.nc" ]; then
        mv diagsoil.nc diags/diagsoil${iPCM}.nc
    fi
    if [ -f "stats.nc" ]; then
        mv stats.nc diags/stats${iPCM}.nc
    fi
    if [ -f "Xdiurnalave.nc" ]; then
        mv Xdiurnalave.nc diags/Xdiurnalave${iPCM}.nc
    fi
    cp restartfi.nc starts/startfi${iPCM}.nc
    mv restartfi.nc startfi.nc
    if [ -f "restart1D.txt" ]; then
        cp restart1D.txt starts/restart1D${iPCM}.txt
        mv restart1D.txt start1D.txt
    fi
    ((iPCM++))
    ((i_myear++))
    echo "Done!"
done

# Restore the previous value of LC_NUMERIC
LC_NUMERIC=$OLD_LC_NUMERIC

date
echo "The launching script ended."
