!==================================================================
module radii_mod
!==================================================================
!  module to centralize the radii calculations for aerosols
!==================================================================
      
!     N2 cloud properties (initialized in inifis)
      real,save :: Nmix_n2 ! Number mixing ratio of N2 ice particles
!$OMP THREADPRIVATE(Nmix_n2)

      ! flag to specify if we assume a constant fixed radius for particles
      logical,save :: radfixed ! initialized in inifis
!$OMP THREADPRIVATE(radfixed)

contains


!==================================================================
   subroutine su_aer_radii(ngrid,nlayer,reffrad,nueffrad)
!==================================================================
!     Purpose
!     -------
!     Compute the effective radii of liquid and icy water particles
!     Jeremy Leconte (2012)
!     Extended to dust, N2, NH3, 2-lay,Nlay,auroral aerosols by ??
!     Added Radiative Generic Condensable Species effective radii
!     calculations  (Lucas Teinturier 2022)
!
!     Authors
!     -------
!     Jeremy Leconte (2012)
!
!==================================================================
      use mod_phys_lmdz_para, only : is_master
      use ioipsl_getin_p_mod, only: getin_p
      use radinc_h, only: naerkind
      use aerosol_mod, only: iaero_back2lay, iaero_n2, iaero_dust, &
                             iaero_h2so4, iaero_nh3, iaero_nlay, &
                             iaero_aurora, iaero_generic, i_rgcs_ice
      use callkeys_mod, only: size_nh3_cloud, nlayaero, aeronlay_size, &
                              aeronlay_nueff,aerogeneric
      use tracer_h, only: radius, nqtot, is_rgcs
      Implicit none

      integer,intent(in) :: ngrid
      integer,intent(in) :: nlayer

      real, intent(out) :: reffrad(ngrid,nlayer,naerkind)      !aerosols radii (K)
      real, intent(out) :: nueffrad(ngrid,nlayer,naerkind)     !variance     

      logical, save :: firstcall=.true.
!$OMP THREADPRIVATE(firstcall)
      integer :: iaer, ia , iq, i_rad  

      do iaer=1,naerkind
!     these values will change once the microphysics gets to work
!     UNLESS tracer=.false., in which case we should be working with
!     a fixed aerosol layer, and be able to define reffrad in a 
!     .def file. To be improved!
!                |-> Done in th n-layer aerosol case (JVO 20)

         if(iaer.eq.iaero_n2)then ! N2 ice
            reffrad(1:ngrid,1:nlayer,iaer) = 1.e-4
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif

         if(iaer.eq.iaero_dust)then ! dust
            reffrad(1:ngrid,1:nlayer,iaer) = 1.e-5
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif
 
         if(iaer.eq.iaero_h2so4)then ! H2SO4 ice
            reffrad(1:ngrid,1:nlayer,iaer) = 1.e-6
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif
            
         if(iaer.eq.iaero_back2lay)then ! Two-layer aerosols
            reffrad(1:ngrid,1:nlayer,iaer) = 2.e-6
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif


	 if(iaer.eq.iaero_nh3)then ! Nh3 cloud
            reffrad(1:ngrid,1:nlayer,iaer) = size_nh3_cloud
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif

         do ia=1,nlayaero
            if(iaer.eq.iaero_nlay(ia))then ! N-layer aerosols
               reffrad(1:ngrid,1:nlayer,iaer) = aeronlay_size(ia)
               nueffrad(1:ngrid,1:nlayer,iaer) = aeronlay_nueff(ia) 
            endif
         enddo

	 if(iaer.eq.iaero_aurora)then ! Auroral aerosols
            reffrad(1:ngrid,1:nlayer,iaer) = 3.e-7
            nueffrad(1:ngrid,1:nlayer,iaer) = 0.1 
         endif

         do ia=1,aerogeneric     ! Radiative Generic Condensable Species
            if (iaer .eq. iaero_generic(ia)) then 
               i_rad = i_rgcs_ice(ia)
               reffrad(1:ngrid,1:nlayer,iaer)=radius(i_rad)
               nueffrad(1:ngrid,1:nlayer,iaer) = 0.1
            endif 
         enddo  ! generic radiative condensable aerosols
         
      enddo ! iaer=1,naerkind

   end subroutine su_aer_radii
!==================================================================


!==================================================================
   subroutine n2_reffrad(ngrid,nlayer,nq,pq,reffrad)
!==================================================================
!     Purpose
!     -------
!     Compute the effective radii of n2 ice particles !AF24: copied from CO2
!
!     Authors
!     -------
!     Jeremy Leconte (2012)
!
!==================================================================
      USE tracer_h, only:igcm_n2_ice,rho_n2
      use comcstfi_mod, only: pi
      Implicit none

      integer,intent(in) :: ngrid,nlayer,nq

      real, intent(in) :: pq(ngrid,nlayer,nq) !tracer mixing ratios (kg/kg)
      real, intent(out) :: reffrad(ngrid,nlayer)      !n2 ice particles radii (m)

      integer :: ig,l
      real :: zrad   
      real,external :: CBRT            
            
      

      if (radfixed) then
         reffrad(1:ngrid,1:nlayer) = 5.e-5 ! N2 ice
      else
         do l=1,nlayer
            do ig=1,ngrid
               zrad = CBRT( 3*pq(ig,l,igcm_n2_ice)/(4*Nmix_n2*pi*rho_n2) )
               reffrad(ig,l) = min(max(zrad,1.e-6),100.e-6)
            enddo
         enddo      
      end if

   end subroutine n2_reffrad
!==================================================================



!==================================================================
   subroutine dust_reffrad(ngrid,nlayer,reffrad)
!==================================================================
!     Purpose
!     -------
!     Compute the effective radii of dust particles
!
!     Authors
!     -------
!     Jeremy Leconte (2012)
!
!==================================================================
      Implicit none

      integer,intent(in) :: ngrid
      integer,intent(in) :: nlayer

      real, intent(out) :: reffrad(ngrid,nlayer)      !dust particles radii (m)
            
      reffrad(1:ngrid,1:nlayer) = 2.e-6 ! dust

   end subroutine dust_reffrad
!==================================================================


!==================================================================
   subroutine h2so4_reffrad(ngrid,nlayer,reffrad)
!==================================================================
!     Purpose
!     -------
!     Compute the effective radii of h2so4 particles
!
!     Authors
!     -------
!     Jeremy Leconte (2012)
!
!==================================================================
      Implicit none

      integer,intent(in) :: ngrid
      integer,intent(in) :: nlayer

      real, intent(out) :: reffrad(ngrid,nlayer)      !h2so4 particle radii (m)
                
      reffrad(1:ngrid,1:nlayer) = 1.e-6 ! h2so4

   end subroutine h2so4_reffrad
!==================================================================

!==================================================================
   subroutine back2lay_reffrad(ngrid,reffrad,nlayer,pplev)
!==================================================================
!     Purpose
!     -------
!     Compute the effective radii of particles in a 2-layer model
!
!     Authors
!     -------
!     Sandrine Guerlet (2013)
!
!==================================================================
      use callkeys_mod, only: pres_bottom_tropo,pres_top_tropo,size_tropo,  &
                              pres_bottom_strato,size_strato
 
      Implicit none

      integer,intent(in) :: ngrid

      real, intent(out) :: reffrad(ngrid,nlayer)      ! particle radii (m)
      REAL,INTENT(IN) :: pplev(ngrid,nlayer+1) ! inter-layer pressure (Pa)
      INTEGER,INTENT(IN) :: nlayer ! number of atmospheric layers
      REAL :: expfactor
      INTEGER l,ig
            
      reffrad(:,:)=1e-6  !!initialization, not important
          DO ig=1,ngrid
            DO l=1,nlayer-1
              IF (pplev(ig,l) .le. pres_bottom_tropo .and. pplev(ig,l) .ge. pres_top_tropo) THEN
                reffrad(ig,l) = size_tropo
              ELSEIF (pplev(ig,l) .lt. pres_top_tropo .and. pplev(ig,l) .gt. pres_bottom_strato) THEN
                expfactor=log(size_strato/size_tropo) / log(pres_bottom_strato/pres_top_tropo)
                reffrad(ig,l)= size_tropo*((pplev(ig,l)/pres_top_tropo)**expfactor)
              ELSEIF (pplev(ig,l) .le. pres_bottom_strato) then
                reffrad(ig,l) = size_strato
              ENDIF
            ENDDO
          ENDDO

   end subroutine back2lay_reffrad
!==================================================================

end module radii_mod
!==================================================================
