#!/bin/bash
#####################################################################
### Launching script for a chained simulation of PEM and GCM runs ###
#####################################################################

echo "The launching script is starting!"
echo "The output file is \"loglaunch.txt\"."
if [ "$1" = "bg" ]; then
    date
else
    nohup "$0" bg > loglaunch.txt 2>&1 &
    exit 1
fi

# A few parameters that might need be changed depending on your setup:
# Path to the arch.env to source:
source ../trunk/LMDZ.COMMON/arch.env

# Save the current value of LC_NUMERIC and set it to a locale that uses a dot as the decimal separator
OLD_LC_NUMERIC=$LC_NUMERIC
LC_NUMERIC=en_US.UTF-8

#------- Modify here the number of Earth years to be simulated ---------
## set the number of years:
n_earth_years=1000000

#---------------- Modify here the number of GCM calls ------------------
## set the number of GCM calls between each PEM call:
nGCM=2

#------------------ Modify here the name of PEM exe --------------------
## fill in the name of executable for PEM:
exePEM=pem_29_phymars_seq.e

#-------------- Modify here the name of reshape XIOS exe ---------------
## fill in the name of executable for reshape XIOS:
exeReshape=reshape_XIOS_output_64x48x29_phymars_seq.e

#------------------------------ Parameters -----------------------------
myear=686.9725      # Number of Earth days in Martian year
eyear=365.256363004 # Number of days in Earth year

#------ Check if files/directories necessary for the script exist ------
dir=`pwd`
machine=`hostname`
address=`whoami`
if [ ! -f "exeGCM.sh" ]; then
    echo "Error: file \"exeGCM.sh\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "run_PEM.def" ]; then
    echo "Error: file \"run_PEM.def\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "run_GCM.def" ]; then
    echo "Error: file \"run_GCM.def\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "context_lmdz_physics.xml" ]; then
    echo "Error: file \"context_lmdz_physics.xml\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "field_def_physics_mars.xml" ]; then
    echo "Error: file \"field_def_physics_mars.xml\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "file_def_physics_mars.xml" ]; then
    echo "Error: file \"file_def_physics_mars.xml\" does not exist in $dir!"
    exit 1
fi
if [ ! -f "iodef.xml" ]; then
    echo "Error: file \"iodef.xml\" does not exist in $dir!"
    exit 1
fi
if [ ! -d "out_GCM" ]; then
    mkdir out_GCM
fi
if [ ! -d "out_PEM" ]; then
    mkdir out_PEM
fi
if [ ! -d "starts" ]; then
    mkdir starts
fi
if [ ! -d "diags" ]; then
    mkdir diags
fi

#---------------------------- Initialization ---------------------------
dir=`pwd`
machine=`hostname`
address=`whoami`
echo "This is a chained simulation for PEM and GCM runs in $dir on $machine."
convert_years=$(echo "$myear/$eyear" | bc -l)
convert_years=$(printf "%.4f" $convert_years) # Rounding to the 4th decimal to respect the precision of Martian year
n_myear=$(echo "($n_earth_years/$convert_years + 0.999999)/1" | bc) # Ceiling of n_earth_years/convert_years
echo "Number of years to simulate: $n_earth_years Earth years = $n_myear Martian years."
i_myear=0
iPEM=1
((iGCM = ($iPEM - 1)*$nGCM + 1))
cp startfi.nc starts/
if [ -f "start.nc" ]; then
    cp start.nc starts/
fi

# Create a temporary file to manage years of the chained simulation
echo $i_myear $n_myear $convert_years > tmp_PEMyears.txt

#---------------------- Main loop to call PEM/GCM ----------------------
while [ $i_myear -lt $n_myear ]; do
    #--- Loop to run GCM year by year
    cp run_GCM.def run.def
    for ((i = 1; i <= $nGCM; i++)); do
        echo "Run GCM $iGCM: call $i/$nGCM..."
        sed -i "s/#SBATCH --job-name=runGCM.*/#SBATCH --job-name=runGCM${iGCM}/" exeGCM.sh
        sed -i "s/out_runGCM[0-9]\+/out_runGCM${iGCM}/" exeGCM.sh
        sbatch -W exeGCM.sh
        if [ ! -f "restartfi.nc" ]; then # Check if run ended abnormally
            echo "Error: the run GCM $iGCM has crashed!"
            exit 1
        fi
        # Copy data files and prepare the next run
        mv out_runGCM${iGCM} out_GCM/run${iGCM}
        mv diagfi.nc diags/diagfi${iGCM}.nc
        if [ -f "diagsoil.nc" ]; then
            mv diagsoil.nc diags/diagsoil${iGCM}.nc
        fi
        if [ -f "stats.nc" ]; then
            mv stats.nc diags/stats${iGCM}.nc
        fi
        cp Xdiurnalave.nc diags/data2reshape${iGCM}.nc
        mv Xdiurnalave.nc data2reshape${i}.nc
        cp restartfi.nc starts/startfi${iGCM}.nc
        mv restartfi.nc startfi.nc
        if [ -f "restart.nc" ]; then
            cp restart.nc starts/restart${iGCM}.nc
            mv restart.nc start.nc
        elif [ -f "restart1D.txt" ]; then
            cp restart1D.txt starts/restart1D${iGCM}.txt
            mv restart1D.txt start1D.txt
        fi
        ((iGCM++))
        ((i_myear++))
        echo "Done!"
    done
    echo $i_myear $n_myear $convert_years > tmp_PEMyears.txt
    #--- Reshaping GCM data with XIOS
    echo "Reshaping GCM data with XIOS..."
    ./$exeReshape
    for file in datareshaped*; do
        mv $file ${file/reshaped/_GCM_Y}
    done
    echo "Done!"
    #--- Running PEM
    echo "Run PEM $iPEM..."
    cp run_PEM.def run.def
    mv startfi.nc startfi_evol.nc
    if [ -f "start.nc" ]; then
        mv start.nc start_evol.nc
    elif [ -f "start1D.txt" ]; then
        mv start1D.txt start1D_evol.txt
    fi
    ./$exePEM > out_runPEM${iPEM} 2>&1
    if [ ! -f "restartfi_evol.nc" ]; then # Check if run ended abnormally
        echo "Error: the run PEM $iPEM has crashed!"
        exit 1
    fi
    # Copy data files and prepare the next run
    mv out_runPEM${iPEM} out_PEM/run${iPEM}
    mv diagpem.nc diags/diagpem_${iPEM}.nc
    cp restartpem.nc starts/startpem_${iPEM}.nc
    mv restartpem.nc startpem.nc
    cp restartfi_evol.nc starts/startfi_postPEM${iPEM}.nc
    mv restartfi_evol.nc startfi.nc
    if [ -f "restart_evol.nc" ]; then
        cp restart_evol.nc starts/restart_postPEM${iPEM}.nc
        mv restart_evol.nc start.nc
    elif [ -f "restart1D_evol.txt" ]; then
        cp restart1D_evol.txt starts/restart1D_postPEM${iPEM}.txt
        mv restart1D_evol.txt start1D.txt
    fi
    ((iPEM++))
    read i_myear n_myear convert_years < tmp_PEMyears.txt
    echo "Done!"
done

# Delete the temporary file to manage years of the chained simulation
rm tmp_PEMyears.txt

# Restore the previous value of LC_NUMERIC
LC_NUMERIC=$OLD_LC_NUMERIC

#----------------- Preparation for relaunch if needed ------------------
#echo "Reinitializing starting files..."
#cp starts/startfi.nc .
#for file in profiles/*_0; do
#    cp $file ${file/_0/}
#done
#echo "Done!"

date
echo "The launching script has terminated."
