     subroutine interpolateH2H2(wn,temp,pres,abcoef,firstcall)
      implicit none

!==================================================================
!     
!     Purpose
!     -------
!     Calculates the H2-H2 CIA opacity, using a lookup table from
!     Borysow (2002)
!
!     Authors
!     -------
!     R. Wordsworth (2009)
!     
!==================================================================

#include "datafile.h"

      ! input
      double precision wn                 ! wavenumber             (cm^-1)
      double precision temp               ! temperature            (Kelvin)
      double precision pres               ! pressure               (Pascals)

      ! output
      double precision abcoef             ! absorption coefficient (m^-1)

      integer nS,nT
      parameter(nS=1649)
      parameter(nT=14)

      double precision amagat
      double precision wn_arr(nS)
      double precision temp_arr(nT)
      double precision abs_arr(nS,nT)
      double precision data_tmp(nT/2+1)

      integer k
      logical firstcall

      save wn_arr, temp_arr, abs_arr

      character*100 dt_file
      integer strlen,ios

      amagat=(273.15/temp)*(pres/101325.0)

      if(firstcall)then

!     1.1 Open the ASCII files

         ! cold
         dt_file=datafile(1:LEN_TRIM(datafile))//'/continuum_data/H2H2_CIA_LT.dat'
         open(33,file=dt_file,form='formatted',status='old',iostat=ios)
         if (ios.ne.0) then        ! file not found
            write(*,*) 'Error from interpolateH2H2.for'
            write(*,*) 'Data file could not be found:'
            write(*,*) dt_file
            call abort
         else
            do k=1,nS
               read(33,*) data_tmp
               wn_arr(k)=data_tmp(1)
               abs_arr(k,1:7)=data_tmp(2:8)
            end do
         endif
         close(33)

         ! hot
         dt_file=datafile(1:LEN_TRIM(datafile))//'/continuum_data/H2H2_CIA_HT.dat'
         open(34,file=dt_file,form='formatted',status='old',iostat=ios)
         if (ios.ne.0) then        ! file not found
            write(*,*) 'Error from interpolateH2H2.for'
            write(*,*) 'Data file could not be found:'
            write(*,*) dt_file
            call abort
         else
            do k=1,nS
               read(34,*) data_tmp
               wn_arr(k)=data_tmp(1)
               ! wn_arr is identical
               abs_arr(k,8:14)=data_tmp(2:8)
            end do
         endif
         close(34)

         temp_arr(1)  = 60.
         temp_arr(2)  = 100.
         temp_arr(3)  = 150.
         temp_arr(4)  = 200.
         temp_arr(5)  = 250.
         temp_arr(6)  = 300.
         temp_arr(7)  = 350.
         temp_arr(8)  = 400.
         temp_arr(9)  = 500.
         temp_arr(10) = 600.
         temp_arr(11) = 700.
         temp_arr(12) = 800.
         temp_arr(13) = 900.
         temp_arr(14) = 1000.


         print*,'At wavenumber ',wn,' cm^-1'
         print*,'   temperature ',temp,' K'
         print*,'   pressure ',pres,' Pa'

         call bilinearH2H2(wn_arr,temp_arr,abs_arr,wn,temp,abcoef)

         print*,'the absorption is ',abcoef,' cm^-1 amg^-2'

         abcoef=abcoef*100.0*amagat**2 ! convert to m^-1

         print*,'We have ',amagat,' amagats'
         print*,'So the absorption is ',abcoef,' m^-1'

      else
  
         call bilinearH2H2(wn_arr,temp_arr,abs_arr,wn,temp,abcoef)
         abcoef=abcoef*100.0*amagat**2 ! convert to m^-1
         !print*,'The absorption is ',abcoef,' m^-1'

         ! unlike for Rayleigh scattering, we do not currently weight by the BB function
         ! however our bands are normally thin, so this is no big deal.

      endif

      return
    end subroutine interpolateH2H2


!-------------------------------------------------------------------------
      subroutine bilinearH2H2(x_arr,y_arr,f2d_arr,x_in,y_in,f)
!     Necessary for interpolation of continuum data

      implicit none

      integer nX,nY,i,j,a,b
      parameter(nX=1649)
      parameter(nY=14)

      real*8 x_in,y_in,x,y,x1,x2,y1,y2
      real*8 f,f11,f12,f21,f22,fA,fB
      real*8 x_arr(nX)
      real*8 y_arr(nY)
      real*8 f2d_arr(nX,nY)
      
      integer strlen
      character*100 label
      label='subroutine bilinear'

      x=x_in
      y=y_in

!     1st check we're within the wavenumber range
      if ((x.lt.x_arr(2)).or.(x.gt.x_arr(nX-2))) then
         f=0.0D+0
         return
      else
         
!     in the x (wavenumber) direction 1st
         i=1
 10      if (i.lt.(nX+1)) then
            if (x_arr(i).gt.x) then
               x1=x_arr(i-1)
               x2=x_arr(i)
               a=i-1
               i=9999
            endif
            i=i+1
            goto 10
         endif
      endif
      
      if ((y.lt.y_arr(1)).or.(y.gt.y_arr(nY))) then
         write(*,*) 'Warning from bilinear.for:'
         write(*,*) 'Outside continuum temperature range!'
         if(y.lt.y_arr(1))then
            y=y_arr(1)+0.01
         endif
         if(y.gt.y_arr(nY))then
            y=y_arr(nY)-0.01
         endif
      else

!     in the y (temperature) direction 2nd
         j=1
 20      if (j.lt.(nY+1)) then
            if (y_arr(j).gt.y) then
               y1=y_arr(j-1)
               y2=y_arr(j)
               b=j-1
               j=9999
            endif
            j=j+1
            goto 20
         endif
      endif
      
      f11=f2d_arr(a,b)
      f21=f2d_arr(a+1,b)
      f12=f2d_arr(a,b+1)
      f22=f2d_arr(a+1,b+1)
      
!     1st in x-direction
      fA=f11*(x2-x)/(x2-x1)+f21*(x-x1)/(x2-x1)
      fB=f12*(x2-x)/(x2-x1)+f22*(x-x1)/(x2-x1)
      
!     then in y-direction
      f=fA*(y2-y)/(y2-y1)+fB*(y-y1)/(y2-y1)
      
      return
    end subroutine bilinearH2H2
