MODULE chemistrydata_mod
!--------------------------------------------
!     data for photochemistry
!--------------------------------------------
IMPLICIT NONE
!--------------------------------------------
!     dimensions of photolysis lookup table
!--------------------------------------------

integer, parameter :: nd    = 13  ! species
integer, parameter :: nz    = 143 ! altitude
integer, parameter :: nozo  = 7   ! ozone
integer, parameter :: nsza  = 27  ! solar zenith angle
integer, parameter :: ntemp = 4   ! temperature
integer, parameter :: ntau  = 8   ! dust

!--------------------------------------------

! tabulated solar zenith angles
real,parameter :: szatab(nsza) = [ 0.,  5., 10., 15., 20., 25., &
                                  30., 35., 40., 45., 50., 55., &
                                  60., 65., 70., 75., 80., 82., &
                                  84., 86., 88., 90., 91., 92., &
                                  93., 94., 95. ]

! tabulated opacities
real,parameter :: tautab(ntau)=[0., 0.2, 0.4, 0.6, 0.8, 1., 2., 4.]


real,save,protected :: jphot(ntemp,nsza,nz,nozo,ntau,nd)
!$OMP THREADPRIVATE(jphot)
real,save,protected :: colairtab(nz)
!$OMP THREADPRIVATE(colairtab)
real,save,protected :: table_ozo(nozo)
!$OMP THREADPRIVATE(table_ozo)

CONTAINS

  subroutine read_phototable

!***********************************************************************
!
!   subject:
!   --------
!
!   read photolysis lookup table
!
!   VERSION: 8/10/2014
!
!   Author:   Franck Lefevre
!
!   Arguments:
!   ----------
!
!***********************************************************************

  use ioipsl_getin_p_mod, only: getin_p
  use datafile_mod, only: datadir
  use mod_phys_lmdz_para, only: is_master
  use mod_phys_lmdz_transfert_para, only: bcast

  implicit none

!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
!     local:
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

  integer :: fic, ij, iozo, isza, itemp, iz, itau, ierr
  real    :: xsza

  character(len = 128) :: phototable ! photolysis table file name

!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! set photolysis table input file name
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

  phototable = "jmars.20140930" ! default

! look for a " phototable= ..." option in def files

  call getin_p("phototable",phototable)

  fic = 81
      
  if (is_master) then ! only the master needs to open file and read data

    open(fic, form = 'formatted', status = 'old',                &
           file =trim(datadir)//"/"//trim(phototable),iostat=ierr)

    if (ierr /= 0) THEN
        write(*,*)'Error : cannot open photolysis lookup table ', trim(phototable)
        write(*,*)'It should be in :',trim(datadir),'/'
        write(*,*)'1) You can change this directory in callphys.def'
        write(*,*)'   with:'
        write(*,*)'   datadir=/path/to/the/directory'
        write(*,*)'2) You can change the input phototable file name in'
        write(*,*)'   callphys.def with:'
        write(*,*)'   phototable=filename'
        call abort_physic("read_phototable","missing "//trim(phototable)//"file",1)
    end if

!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! read photolys table
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

    print*, 'read photolysis lookup table ',trim(phototable)

    do itau = 1,ntau
       do itemp = 1,ntemp
          do iozo = 1,nozo
             do isza = 1,nsza
                do iz = nz,1,-1
                   read(fic,*) colairtab(iz), xsza, table_ozo(iozo)
                   read(fic,'(7e11.4)') (jphot(itemp,isza,iz,iozo,itau,ij), ij= 1,nd)
                   do ij = 1,nd
                      if (jphot(itemp,isza,iz,iozo,itau,ij) == 1.e-30) then
                         jphot(itemp,isza,iz,iozo,itau,ij) = 0.
                      end if
                   end do
                end do
             end do
          end do
       end do
    end do

    print*, 'lookup table...ok'
    close(fic)

  endif ! of if (is_master)
    
  ! broadcast the information to all cores
  call bcast(colairtab)
  call bcast(table_ozo)
  call bcast(jphot)

end subroutine read_phototable

END MODULE chemistrydata_mod
