      Subroutine aeropacity(ngrid,nlayer,pplay,pplev,pt, &
         aerosol,reffrad,nueffrad,QREFvis3d,tau_col)

       use radinc_h, only : L_TAUMAX,naerkind
       use aerosol_mod
                  
       implicit none
#include "YOMCST.h"

!==================================================================
!     
!     Purpose
!     -------
!     Compute aerosol optical depth in each gridbox.
!     
!     Authors
!     ------- 
!     F. Forget
!     F. Montmessin (water ice scheme) 
!     update J.-B. Madeleine (2008)
!     dust removal, simplification by Robin Wordsworth (2009)
!     Generic n-layer aerosol - J. Vatant d'Ollone (2020)
!
!     Input
!     ----- 
!     ngrid             Number of horizontal gridpoints
!     nlayer            Number of layers
!     pplev             Pressure (Pa) at each layer boundary
!     reffrad(ngrid,nlayer,naerkind)         Aerosol effective radius
!     QREFvis3d(ngrid,nlayer,naerkind) \ 3d extinction coefficients
!     QREFir3d(ngrid,nlayer,naerkind)  / at reference wavelengths
!
!     Output
!     ------
!     aerosol            Aerosol optical depth in layer l, grid point ig
!     tau_col            Total column optical depth at grid point ig
!
!=======================================================================

      INTEGER,INTENT(IN) :: ngrid  ! number of atmospheric columns
      INTEGER,INTENT(IN) :: nlayer ! number of atmospheric layers
      REAL,INTENT(IN) :: pplay(ngrid,nlayer) ! mid-layer pressure (Pa)
      REAL,INTENT(IN) :: pplev(ngrid,nlayer+1) ! inter-layer pressure (Pa)
      REAL,INTENT(IN) :: pt(ngrid,nlayer) ! mid-layer temperatre (K)
      REAL,INTENT(OUT) :: aerosol(ngrid,nlayer,naerkind) ! aerosol optical depth
      REAL,INTENT(IN) :: reffrad(ngrid,nlayer,naerkind) ! aerosol effective radius
      REAL,INTENT(IN) :: nueffrad(ngrid,nlayer,naerkind) ! aerosol effective variance
      REAL,INTENT(IN) :: QREFvis3d(ngrid,nlayer,naerkind) ! extinction coefficient in the visible
      REAL,INTENT(OUT):: tau_col(ngrid) !column integrated visible optical depth

      real aerosol0, obs_tau_col_aurora, pm
      real pcloud_deck, cloud_slope

      real dp_strato(ngrid)
      real dp_tropo(ngrid)
      real dp_layer(ngrid)

      INTEGER l,ig,iq,iaer,ia,ll

      LOGICAL,SAVE :: firstcall=.true.
!$OMP THREADPRIVATE(firstcall)
      REAL CBRT
      EXTERNAL CBRT

      ! for fixed dust profiles
      real topdust, expfactor, zp
      REAL taudusttmp(ngrid) ! Temporary dust opacity used before scaling
      REAL tauh2so4tmp(ngrid) ! Temporary h2so4 opacity used before scaling

      real CLFtot

      !  for venus clouds
      real :: p_bot2,p_bot,p_top,p_top2,h_bot2,h_bot,h_top,h_top2
      real :: mode_dens,h_lay,nmode1,nmode2,nmode2p,nmode3,nmodeuv

! First call only
      IF (firstcall) THEN

      ! identify tracers
        write(*,*) "Active aerosols found in aeropacity:"

        if (iaero_venus1.ne.0) then
          print*,'iaero_venus1= ',iaero_venus1
        endif
        if (iaero_venus2.ne.0) then
          print*,'iaero_venus2= ',iaero_venus2
        endif
        if (iaero_venus2p.ne.0) then
          print*,'iaero_venus2p= ',iaero_venus2p
        endif
        if (iaero_venus3.ne.0) then
          print*,'iaero_venus3= ',iaero_venus3
        endif
        if (iaero_venusUV.ne.0) then
          print*,'iaero_venusUV= ',iaero_venusUV
        endif

        firstcall=.false.
      ENDIF ! of IF (firstcall)


!     ---------------------------------------------------------

!==================================================================
!         Venus clouds (4 modes)
!   S. Lebonnois (jan 2016)
!==================================================================
! distributions from Haus et al, 2016
! mode             1      2      2p     3
! r (microns)     0.30   1.05   1.40   3.65
! sigma           1.56   1.29   1.23   1.28
! reff (microns)  0.49   1.23   1.56   4.25
! nueff           0.21   0.067  0.044  0.063
! (nueff=exp(ln^2 sigma)-1)
!
! p_bot <=> zb ; p_top <=> zb+zc ; h_bot <=> Hlo ; h_top <=> Hup
! p<p_top: N(i+1)=N(i)*(p(i+1)/p(i))**(h_lay(i)/h_top)      
!    h_lay=R(T(i)+T(i+1))/2/g  (in m)
!    R=8.31/mu (mu in kg/mol)
! p>p_bot: N(i-1)=N(i)*(p(i)/p(i-1))**(h_lay(i)/h_bot)      
! N is in m-3
!
! dTau = Qext*[pi*reff**2*exp(-3*ln(1+nueff))]*N*h_lay*(-dp)/p

! Mode 1
      if (iaero_venus1 .ne.0) then
          iaer=iaero_venus1

!       1. Initialization
          aerosol(1:ngrid,1:nlayer,iaer)=0.0
! two scales heights for below and above max density (cf Haus et al 16)
          p_bot2 = 2.0e5 ! Pa   2.0e5
            h_bot2 = 5.0e3 ! m
          p_bot  = 1.2e5   !    1.2e5
            h_bot  = 1.0e3
          nmode1 = 1.935e8 ! m-3
          p_top  = 1.0e4
            h_top  = 3.5e3
          p_top2 = 4.5e2
            h_top2 = 2.0e3

!       2. Opacity calculation

          DO ig=1,ngrid
! determine the approximate middle of the mode layer
           ll=1
           DO l=1,nlayer-1    ! high p to low p
             IF (pplay(ig,l) .gt. (p_top+p_bot)/2) ll=l
           ENDDO
! from there go down and up for profile N
           mode_dens = nmode1  ! m-3
           DO l=ll,1,-1
             h_lay=RD*(pt(ig,l+1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .le. p_bot) THEN
               mode_dens = nmode1  ! m-3
             ELSEIF (pplay(ig,l) .gt. p_bot .and. pplay(ig,l) .le. p_bot2) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot)
             ELSEIF (pplay(ig,l) .gt. p_bot2) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot2)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
           mode_dens = nmode1  ! m-3
           DO l=ll+1,nlayer-1
             h_lay=RD*(pt(ig,l-1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .gt. p_top) THEN
               mode_dens = nmode1  ! m-3
             ELSEIF (pplay(ig,l) .le. p_top .and. pplay(ig,l) .gt. p_top2) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top)
             ELSEIF (pplay(ig,l) .le. p_top2) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top2)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
          ENDDO

      end if ! mode 1

! Mode 2
      if (iaero_venus2 .ne.0) then
          iaer=iaero_venus2

!       1. Initialization
          aerosol(1:ngrid,1:nlayer,iaer)=0.0
          p_bot = 1.0e4
            h_bot = 3.0e3
          nmode2 = 1.00e8 ! m-3
          p_top = 8.0e3
            h_top = 3.5e3
          p_top2 = 4.5e2
            h_top2 = 2.0e3

!       2. Opacity calculation

          DO ig=1,ngrid
! determine the approximate middle of the mode layer
           ll=1
           DO l=1,nlayer-1    ! high p to low p
             IF (pplay(ig,l) .gt. (p_top+p_bot)/2) ll=l
           ENDDO
! from there go down and up for profile N
           mode_dens = nmode2  ! m-3
           DO l=ll,1,-1
             h_lay=RD*(pt(ig,l+1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .le. p_bot) THEN
               mode_dens = nmode2  ! m-3
             ELSEIF (pplay(ig,l) .gt. p_bot) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
           mode_dens = nmode2  ! m-3
           DO l=ll+1,nlayer-1
             h_lay=RD*(pt(ig,l-1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .gt. p_top) THEN
               mode_dens = nmode2  ! m-3
             ELSEIF (pplay(ig,l) .le. p_top .and. pplay(ig,l) .gt. p_top2) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top)
             ELSEIF (pplay(ig,l) .le. p_top2) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top2)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
          ENDDO

      end if ! mode 2

! Mode 2p
      if (iaero_venus2p .ne.0) then
          iaer=iaero_venus2p

!       1. Initialization
          aerosol(1:ngrid,1:nlayer,iaer)=0.0
          p_bot = 1.2e5  ! 1.2e5
            h_bot = 0.1e3
          nmode2p = 5.0e7 ! m-3
          p_top = 2.3e4
            h_top = 1.0e3

!       2. Opacity calculation

          DO ig=1,ngrid
! determine the approximate middle of the mode layer
           ll=1
           DO l=1,nlayer-1    ! high p to low p
             IF (pplay(ig,l) .gt. (p_top+p_bot)/2) ll=l
           ENDDO
! from there go down and up for profile N
           mode_dens = nmode2p  ! m-3
           DO l=ll,1,-1
             h_lay=RD*(pt(ig,l+1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .le. p_bot) THEN
               mode_dens = nmode2p  ! m-3
             ELSEIF (pplay(ig,l) .gt. p_bot) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
           mode_dens = nmode2p  ! m-3
           DO l=ll+1,nlayer-1
             h_lay=RD*(pt(ig,l-1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .gt. p_top) THEN
               mode_dens = nmode2p  ! m-3
             ELSEIF (pplay(ig,l) .le. p_top) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
          ENDDO

      end if ! mode 2p

! Mode 3
      if (iaero_venus3 .ne.0) then
          iaer=iaero_venus3

!       1. Initialization
          aerosol(1:ngrid,1:nlayer,iaer)=0.0
          p_bot = 1.2e5  ! 1.2e5
            h_bot = 0.5e3
          nmode3 = 1.4e7 ! m-3
          p_top = 3.9e4
            h_top = 1.0e3

!       2. Opacity calculation

          DO ig=1,ngrid
! determine the approximate middle of the mode layer
           ll=1
           DO l=1,nlayer-1    ! high p to low p
             IF (pplay(ig,l) .gt. (p_top+p_bot)/2) ll=l
           ENDDO
! from there go down and up for profile N
           mode_dens = nmode3  ! m-3
           DO l=ll,1,-1
             h_lay=RD*(pt(ig,l+1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .le. p_bot) THEN
               mode_dens = nmode3  ! m-3
             ELSEIF (pplay(ig,l) .gt. p_bot) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
           mode_dens = nmode3  ! m-3
           DO l=ll+1,nlayer-1
             h_lay=RD*(pt(ig,l-1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .gt. p_top) THEN
               mode_dens = nmode3  ! m-3
             ELSEIF (pplay(ig,l) .le. p_top) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*                       &
              RPI*(reffrad(ig,l,iaer))**2*exp(-3.*log(1+nueffrad(ig,l,iaer)))* & 
              mode_dens*h_lay*(pplev(ig,l)-pplev(ig,l+1))/pplay(ig,l)
!             aerosol(ig,l,iaer) = mode_dens
           ENDDO
          ENDDO

      end if ! mode 3 

! UV absorber
      if (iaero_venusUV .ne.0) then
          iaer=iaero_venusUV

!       1. Initialization
          aerosol(1:ngrid,1:nlayer,iaer)=0.0
          p_bot = 3.3e4  ! 58 km
            h_bot = 1.0e3
          nmodeuv = 1.00e7 !m-3
          p_top = 3.7e3 ! 70 km
            h_top = 1.0e3

!       2. Opacity calculation

          DO ig=1,ngrid
! determine the approximate middle of the mode layer
           ll=1
           DO l=1,nlayer-1    ! high p to low p
             IF (pplay(ig,l) .gt. (p_top+p_bot)/2) ll=l
           ENDDO
! from there go down and up for profile N
           mode_dens = nmodeuv ! m-3
           DO l=ll,1,-1
             h_lay=RD*(pt(ig,l+1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .le. p_bot) THEN
               mode_dens = nmodeuv ! m-3
             ELSEIF (pplay(ig,l) .gt. p_bot) THEN
               mode_dens = mode_dens*(pplay(ig,l+1)/pplay(ig,l))**(h_lay/h_bot)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
! normalized to 0.35 microns (peak of absorption)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*mode_dens
           ENDDO
           mode_dens = nmodeuv ! m-3
           DO l=ll+1,nlayer-1
             h_lay=RD*(pt(ig,l-1)+pt(ig,l))/2./RG
             IF (pplay(ig,l) .gt. p_top) THEN
               mode_dens = nmodeuv ! m-3
             ELSEIF (pplay(ig,l) .le. p_top) THEN
               mode_dens = mode_dens*(pplay(ig,l)/pplay(ig,l-1))**(h_lay/h_top)
             ENDIF
             mode_dens=max(1.e-30,mode_dens)
! normalized to 0.35 microns (peak of absorption)
             aerosol(ig,l,iaer) = QREFvis3d(ig,l,iaer)*mode_dens
           ENDDO
          ENDDO

!       3. Re-normalize to Haus et al 2015 total column optical depth
         tau_col(:)=0.0
         DO l=1,nlayer
          DO ig=1,ngrid
               tau_col(ig) = tau_col(ig) &
                     + aerosol(ig,l,iaer)
            ENDDO
         ENDDO
         DO ig=1,ngrid
           DO l=1,nlayer-1
                aerosol(ig,l,iaer)=aerosol(ig,l,iaer)*0.205/tau_col(ig)
           ENDDO
         ENDDO

      end if ! UV absorber 

!==================================================================
!     if (is_master) then
!      ig=10
!      do l=1,nlayer
!         write(82,*) l,pplay(ig,l),aerosol(ig,l,1)
!         write(83,*) l,pplay(ig,l),aerosol(ig,l,2)
!         write(84,*) l,pplay(ig,l),aerosol(ig,l,3)
!         write(85,*) l,pplay(ig,l),aerosol(ig,l,4)
!         write(86,*) l,pplay(ig,l),aerosol(ig,l,5)
!      enddo
!     endif   
!      stop         
!==================================================================

! --------------------------------------------------------------------------
! Column integrated visible optical depth in each point (used for diagnostic)

      tau_col(:)=0.0
      do iaer = 1, naerkind
         do l=1,nlayer
            do ig=1,ngrid
               tau_col(ig) = tau_col(ig) + aerosol(ig,l,iaer)
            end do
         end do
      end do

      do ig=1,ngrid
         do l=1,nlayer
            do iaer = 1, naerkind
               if(aerosol(ig,l,iaer).gt.1.e3)then
                  print*,'WARNING: aerosol=',aerosol(ig,l,iaer)
                  print*,'at ig=',ig,',  l=',l,', iaer=',iaer
                  print*,'QREFvis3d=',QREFvis3d(ig,l,iaer)
                  print*,'reffrad=',reffrad(ig,l,iaer)
               endif
            end do
         end do
      end do

      do ig=1,ngrid
         if(tau_col(ig).gt.1.e3)then
            print*,'WARNING: tau_col=',tau_col(ig)
            print*,'at ig=',ig
            print*,'aerosol=',aerosol(ig,:,:)
            print*,'QREFvis3d=',QREFvis3d(ig,:,:)
            print*,'reffrad=',reffrad(ig,:,:)
         endif
      end do
      return
    end subroutine aeropacity
      
