      SUBROUTINE callradite(icount,ngrid,nlayer,nq,zday,ls,pq,albedo,
     $     emis,mu0,pplev,pplay,pt,tsurf,fract,dist_sol,igout,
     $     dtlw,dtsw,fluxsurf_lw,fluxsurf_sw,fluxtop_lw,fluxtop_sw,
     &     tauref,tau,aerosol,rice,nuice)

       IMPLICIT NONE
c=======================================================================
c   subject:
c   --------
c   Subroutine designed to call the main canonic
c   radiative transfer subroutine "lwmain" et "swmain"
c   to compute radiative heating and cooling rate and
c   radiative fluxes to the surface.
c
c   These calculations are only valid on the part of the atmosphere
c   where Local Thermal Equilibrium (NLTE) is verified. In practice
c   The calculations are only performed for the first "nlaylte"
c   parameters (nlaylte is calculated by subroutine "nlthermeq"
c   and stored in common "yomlw.h").
c
c   The purpose of this subroutine is to:
c      1) Make some initial calculation at first call
c      2) Compute the 3D scattering parameters depending on the
c        size distribution of the different tracers (added by JBM)
c      3) call "lwmain" and "swmain"
c
c
c   authors:   
c   ------
c   Francois Forget / Christophe Hourdin / J.-B. Madeleine (2009)
c
c
c   3D scattering scheme user's guide (J.-B. Madeleine)
c   ---------------------------------
c
c   This routine has been modified to take into account 3D, time
c   dependent scattering properties of the aerosols.
c---- The look-up tables that contain the scattering parameters
c   of a given tracer, for different sizes, are read by SUAER.F90.
c   The names of the corresponding ASCII files have to be set in
c   this subroutine (file_id variable), and files must be in the
c   directory specified in datafile.h. Be careful: discretization of
c   the particle size in the ASCII files MUST BE DONE using a
c   volume-ratio size distribution. This means that the volume of
c   particles in a size bin equals the volume of particles in the next
c   size bin multiplied by a constant volume ratio "Vrat". In other
c   words: v_(i) = Vrat * v_(i-1), and radius in the ASCII files must
c   obey the equation r_(i) = Vrat^(1/3) * r_(i-1). For a given number
c   of size bins "Nb", Vrat = (r_max/r_min)^(3/(Nb-1)).
c---- SUAER.F90 is in charge of reading the ASCII files and averaging
c   the scattering parameters in each GCM channel, using the three last
c   equations of Forget et al. 1998 (GRL 25, No.7, p.1105-1108).
c---- These look-up tables, loaded during the firstcall, are then
c   constantly used by the subroutine "aeroptproperties.F" to compute,
c   online, the 3D scattering parameters, based on the size distribution
c   (reffrad and nueffrad) of the different tracers, in each grid box.
c   This 3D size distribution is loaded by the "updatereffrad.F"
c   subroutine. A log-normal distribution is then assumed in
c   "aeroptproperties.F".
c---- The optical depth at the visible reference wavelength (set in
c   SUAER.F90, after the file_id variable) is then computed by
c   the subroutine "aeropacity.F", by using the size and spatial
c   distribution of the corresponding tracer. This connection has to
c   be implemented in "aeropacity.F" when adding a new tracer. To do so,
c   one can use equation 2 of Forget et al. 1998 (Icarus 131, p.302-316).
c---- The resulting variables "aerosol", "QVISsQREF3d", "omegaVIS3d" and
c   "gVIS3d" (same in the infrared) are finally used by lwmain.F and 
c   swmain.F to solve the radiative transfer equation.
c
c   changes:
c   -------
c
c   > SRL 7/2000
c   
c   This version has been modified to only calculate radiative tendencies
c   over layers 1..NFLEV (set in dimradmars.h).  Returns zero for higher
c   layers, if any.
c   In other routines, nlayermx -> nflev.
c   Routines affected: lwflux, lwi, lwmain, lwxb, lwxd, lwxn.
c
c   > J.-B. Madeleine 09W30
c
c   Removed the variable's splitting, which is now obsolete.
c
c   ----------
c   Here, solar band#1 is spectral interval between "long1vis" and "long2vis"
c   set in dimradmars.h 
c   Here, solar band#2 is spectral interval between "long2vis" and "long3vis"
c   set in dimradmars.h 
c
c   input:
c   ----- 
c   icount                counter of call to subroutine physic by gcm
c   ngrid                 number of gridpoint of horizontal grid
c   nlayer                Number of layer
c   nq                    Number of tracer
c   ls                    Solar longitude (Ls) , radian
c   zday                  Date (time since Ls=0, in martian days)
c   pq(ngrid,nlayer,nq)   Advected fields
c
c   albedo (ngrid,2)      hemispheric surface albedo
c                         albedo (i,1) : mean albedo for solar band#1 
c                                        (see below)
c                         albedo (i,2) : mean albedo for solar band#2
c                                        (see below)
c   emis                  Thermal IR surface emissivity (no unit)
c   mu0(ngridmx)           cos of solar zenith angle
c                           (=1 when sun at zenith)
c   pplay(ngrid,nlayer)    pressure (Pa) in the middle of each layer
c   pplev(ngrid,nlayer+1)  pressure (Pa) at boundaries of each layer
c   pt(ngrid,nlayer)       atmospheric temperature in each layer (K)
c   tsurf(ngrid)           surface temperature (K)
c   fract(ngridmx)         day fraction of the time interval 
c                          =1 during the full day ; =0 during the night
c   declin                 latitude of subsolar point
c   dist_sol               sun-Mars distance (AU)
c   igout                  coordinate of analysed point for debugging
c   reffrad(ngrid,nlayer,naerkind)  Aerosol effective radius
c   nueffrad(ngrid,nlayer,naerkind) Aerosol effective variance

c
c  output:
c  -------
c dtlw (ngrid,nlayer)       longwave (IR) heating rate (K/s)
c dtsw(ngrid,nlayer)        shortwave (Solar) heating rate (K/s)
c fluxsurf_lw(ngrid)        surface downward flux tota LW (thermal IR) (W.m-2)
c fluxsurf_sw(ngrid,1)      surface downward flux SW for solar band#1 (W.m-2)
c fluxsurf_sw(ngrid,2)      surface downward flux SW for solar band#2 (W.m-2)
c
c fluxtop_lw(ngrid)         outgoing upward flux tota LW (thermal IR) (W.m-2)
c fluxtop_sw(ngrid,1)       outgoing upward flux SW for solar band#1 (W.m-2)
c fluxtop_sw(ngrid,2)       outgoing upward flux SW for solar band#2 (W.m-2)

c   tauref       Prescribed mean column optical depth at 700 Pa 
c   tau          Column total visible dust optical depth at each point
c   aerosol(ngrid,nlayer,naerkind)    aerosol extinction optical depth
c                         at reference wavelength "longrefvis" set
c                         in dimradmars.h , in each layer, for one of
c                         the "naerkind" kind of aerosol optical
c                         properties.

c=======================================================================
c
c    Declarations :
c    -------------
c
#include "dimensions.h"
#include "dimphys.h"
#include "dimradmars.h"
#include "comcstfi.h"
#include "callkeys.h"
#include "yomlw.h"


c-----------------------------------------------------------------------
c    Input/Output
c    ------------
      INTEGER icount        
      INTEGER ngrid,nlayer,nq 
      INTEGER igout

      REAL pq(ngrid,nlayer,nq)
      REAL albedo(ngrid,2),emis(ngrid)
      REAL ls,zday

      REAL pplev(ngrid,nlayer+1),pplay(ngrid,nlayer)
      REAL pt(ngrid,nlayer)
      REAL tsurf(ngrid)
      REAL dist_sol,mu0(ngrid),fract(ngrid)
      REAL dtlw(ngridmx,nlayermx),dtsw(ngridmx,nlayermx)
      REAL fluxsurf_lw(ngridmx), fluxtop_lw(ngridmx)
      REAL fluxsurf_sw(ngridmx,2), fluxtop_sw(ngridmx,2)

      REAL tauref(ngrid), tau(ngrid,naerkind)
      REAL aerosol(ngrid,nlayer,naerkind)
      REAL rice(ngridmx,nlayermx)    ! Estimated ice crystal radius (m)
      REAL nuice(ngridmx,nlayermx)   ! Estimated effective variance
c
c    Local variables :
c    -----------------

      INTEGER j,l,ig,n

      real  cste_mars ! solar constant on Mars (Wm-2)
      REAL ptlev(ngridmx,nlayermx+1)
      REAL dp(ngrid,nflev)
      REAL dt0(ngrid)

c     Thermal IR net radiative budget (W m-2)

      REAL netrad(ngrid,nflev) 
      REAL fluxd_sw(ngrid,nflev+1,2)
      REAL fluxu_sw(ngrid,nflev+1,2)

c     Aerosol size distribution
      REAL :: reffrad(ngrid,nlayer,naerkind)
      REAL :: nueffrad(ngrid,nlayer,naerkind)
c     Aerosol optical properties
      REAL :: QVISsQREF3d(ngridmx,nlayermx,nsun,naerkind)
      REAL :: omegaVIS3d(ngridmx,nlayermx,nsun,naerkind)
      REAL :: gVIS3d(ngridmx,nlayermx,nsun,naerkind)

      REAL :: QIRsQREF3d(ngridmx,nlayermx,nir,naerkind)
      REAL :: omegaIR3d(ngridmx,nlayermx,nir,naerkind)
      REAL :: gIR3d(ngridmx,nlayermx,nir,naerkind)

      REAL :: QREFvis3d(ngridmx,nlayermx,naerkind)
      REAL :: QREFir3d(ngridmx,nlayermx,naerkind)

      REAL :: omegaREFvis3d(ngridmx,nlayermx,naerkind)
      REAL :: omegaREFir3d(ngridmx,nlayermx,naerkind)

c   local saved variables
c   ---------------------

      real pview(ngridmx)
      save pview
      
      real zco2   ! volume fraction of CO2 in Mars atmosphere
      DATA zco2/0.95/
      SAVE zco2

      LOGICAL firstcall
      DATA firstcall/.true./
      SAVE firstcall

c----------------------------------------------------------------------

c     Initialisation
c     --------------

      IF (firstcall) THEN

         DO ig=1,ngrid
            pview(ig)=1.66     ! cosecant of viewing angle
         ENDDO
         gcp = g/cpp

c        Logical tests for radiatively active water-ice clouds:
         IF ( (activice.AND.(.NOT.water)).OR.
     &        (activice.AND.(naerkind.LT.2)) ) THEN
           WRITE(*,*) 'If activice is TRUE, water has to be set'
           WRITE(*,*) 'to TRUE, and "naerkind" must be at least'
           WRITE(*,*) 'equal to 2 in dimradmars.h.'
           CALL ABORT
         ELSE IF ( (.NOT.activice).AND.(naerkind.GT.1) ) THEN
           WRITE(*,*) 'naerkind is greater than unity, but'
           WRITE(*,*) 'activice has not been set to .true.'
           WRITE(*,*) 'in callphys.def; this is not logical!'
           CALL ABORT
         ENDIF

c        Loading the optical properties in external look-up tables:
         CALL SUAER
         CALL SULW

         firstcall=.false.
      END IF

c     Computing aerosol optical properties and opacity
c     ------------------------------------------------

c     Updating aerosol size distributions:
      CALL updatereffrad(ngrid,nlayer,
     &                pplev,pplay,
     &                rice,nuice,
     &                reffrad,nueffrad)

c     Computing 3D scattering parameters:
      CALL aeroptproperties(ngrid,nlayer,reffrad,nueffrad,
     &                      QVISsQREF3d,omegaVIS3d,gVIS3d,
     &                      QIRsQREF3d,omegaIR3d,gIR3d,
     &                      QREFvis3d,QREFir3d,
     &                      omegaREFvis3d,omegaREFir3d)

c     Computing aerosol optical depth in each layer:
      CALL aeropacity(ngrid,nlayer,nq,zday,pplay,pplev,ls,pq,
     &      tauref,tau,aerosol,reffrad,
     &      QREFvis3d,QREFir3d,omegaREFvis3d,omegaREFir3d)

        do l=1,nlaylte
         do ig = 1, ngrid
c         Thickness of each layer (Pa) :
          dp(ig,l)= pplev(ig,l) - pplev(ig,l+1)
         enddo
        enddo

c       Intermediate  levels: (computing tlev)
c       ---------------------------------------
c       Extrapolation for the air temperature above the surface
        DO ig=1, ngrid
              ptlev(ig,1)=pt(ig,1)+
     s        (pplev(ig,1)-pplay(ig,1))*
     s        (pt(ig,1)-pt(ig,2))/(pplay(ig,1)-pplay(ig,2))

              dt0(ig) = tsurf(ig) - ptlev(ig,1)
        ENDDO

        DO l=2,nlaylte
         DO ig=1, ngrid
               ptlev(ig,l)=0.5*(pt(ig,l-1)+pt(ig,l))
         ENDDO
        ENDDO

        DO ig=1, ngrid
           ptlev(ig,nlaylte+1)=pt(ig,nlaylte)
        ENDDO


c       Longwave ("lw") radiative transfer (= thermal infrared)
c       -------------------------------------------------------
        call lwmain (icount,ngrid,nflev
     .        ,dp,dt0,emis,pplev,ptlev,pt
     .        ,aerosol,dtlw
     .        ,fluxsurf_lw,fluxtop_lw
     .        ,netrad
     &        ,QIRsQREF3d,omegaIR3d,gIR3d)

c       Shortwave ("sw") radiative transfer (= solar radiation)
c       -------------------------------------------------------
c          Mars solar constant (W m-2)
c          1370 W.m-2 is the solar constant at 1 AU.
           cste_mars=1370./(dist_sol*dist_sol)

           call swmain ( ngrid, nflev, 
     S     cste_mars, albedo, 
     S     mu0, dp, pplev, aerosol, fract,
     S     dtsw, fluxd_sw, fluxu_sw,
     &     QVISsQREF3d,omegaVIS3d,gVIS3d)

c       ------------------------------------------------------------

        do ig = 1, ngrid
          fluxsurf_sw(ig,1) = fluxd_sw(ig,1,1)
          fluxsurf_sw(ig,2) = fluxd_sw(ig,1,2)
          fluxtop_sw(ig,1) = fluxu_sw(ig,nlaylte+1,1)
          fluxtop_sw(ig,2) = fluxu_sw(ig,nlaylte+1,2)
        enddo

c     Zero tendencies for any remaining layers between nlaylte and nlayer
      if (nlayer.gt.nlaylte) then
         do l = nlaylte+1, nlayer
            do ig = 1, ngrid
               dtlw(ig, l) = 0.
               dtsw(ig, l) = 0.
            enddo
         enddo
      endif

c     Output for debugging if lwrite=T
c     --------------------------------
c     Write all nlayer layers, even though only nlaylte layers may have
c     non-zero tendencies.

         IF(lwrite) THEN
            PRINT*,'Diagnotique for the radiation'
            PRINT*,'albedo, emissiv, mu0,fract,fluxsurf_lw,fluxsurf_sw'
            PRINT*,albedo(igout,1),emis(igout),mu0(igout),
     s           fract(igout), fluxsurf_lw(igout),
     $     fluxsurf_sw(igout,1)+fluxsurf_sw(igout,2)
            PRINT*,'Tlay Tlev Play Plev dT/dt SW dT/dt LW (K/s)'
            PRINT*,'daysec',daysec
            DO l=1,nlayer
               PRINT*,pt(igout,l),ptlev(igout,l),
     s         pplay(igout,l),pplev(igout,l),
     s         dtsw(igout,l),dtlw(igout,l)
            ENDDO
         ENDIF


      return
      end 

