MODULE comchem_h

! ----------------------------------------------------------------------------
! Purpose : Stores data relative to chemistry in the GCM and upper chemistry
! ------- 
!           Please note that upper fields ykim_up are in molar fraction !
!
!          NB : For newstart there is a specific comchem_newstart_h module.
!
! Author : Jan Vatant d'Ollone (2017-18)
! ------
!
! NB : A given order is assumed for the 44 chemistry tracers :
! --   H, H2, CH, CH2s, CH2, CH3, CH4, C2, C2H, C2H2, C2H3, C2H4, C2H5, 
!      C2H6, C3H3, C3H5, C3H6, C3H7, C4H, C4H3, C4H4, C4H2s, CH2CCH2,
!      CH3CCH, C3H8, C4H2, C4H6, C4H10, AC6H6, C3H2, C4H5, AC6H5, N2,
!      N4S, CN, HCN, H2CN, CHCN, CH2CN, CH3CN, C3N, HC3N, NCCN, C4N2
        
! ----------------------------------------------------------------------------

IMPLICIT NONE  

   !! Hard-coded number of chemical species for Titan chemistry
   INTEGER, PARAMETER :: nkim = 44

   !! Hard-coded chemical species for Titan chemistry
   CHARACTER(len=10), DIMENSION(nkim), PARAMETER  :: cnames = &
     (/"H         ", "H2        ", "CH        ", "CH2s      ", "CH2       ", "CH3       ", &
       "CH4       ", "C2        ", "C2H       ", "C2H2      ", "C2H3      ", "C2H4      ", &
       "C2H5      ", "C2H6      ", "C3H3      ", "C3H5      ", "C3H6      ", "C3H7      ", & 
       "C4H       ", "C4H3      ", "C4H4      ", "C4H2s     ", "CH2CCH2   ", "CH3CCH    ", &
       "C3H8      ", "C4H2      ", "C4H6      ", "C4H10     ", "AC6H6     ", "C3H2      ", &
       "C4H5      ", "AC6H5     ", "N2        ", "N4S       ", "CN        ", "HCN       ", &
       "H2CN      ", "CHCN      ", "CH2CN     ", "CH3CN     ", "C3N       ", "HC3N      ", &
       "NCCN      ", "C4N2      "/)
   !! Hard-coded chemical species for Titan chemistry + "HV" specie for the photochem module.
   CHARACTER(len=10), DIMENSION(nkim+1), PARAMETER  :: nomqy_c = &
     (/"H         ", "H2        ", "CH        ", "CH2s      ", "CH2       ", "CH3       ", &
       "CH4       ", "C2        ", "C2H       ", "C2H2      ", "C2H3      ", "C2H4      ", &
       "C2H5      ", "C2H6      ", "C3H3      ", "C3H5      ", "C3H6      ", "C3H7      ", & 
       "C4H       ", "C4H3      ", "C4H4      ", "C4H2s     ", "CH2CCH2   ", "CH3CCH    ", &
       "C3H8      ", "C4H2      ", "C4H6      ", "C4H10     ", "AC6H6     ", "C3H2      ", &
       "C4H5      ", "AC6H5     ", "N2        ", "N4S       ", "CN        ", "HCN       ", &
       "H2CN      ", "CHCN      ", "CH2CN     ", "CH3CN     ", "C3N       ", "HC3N      ", &
       "NCCN      ", "C4N2      ", "HV        "/)
   !! Hard-coded chemical species molar mass (g.mol-1), shares the same indexing than cnames.
   REAL, DIMENSION(nkim), PARAMETER               :: cmmol = (/ &
       1.01   , 2.0158, 13.02, 14.03, 14.03, 15.03, 16.04  , 24.02, 25.03, 26.04  , 27.05  , &
       28.05  , 29.06 , 30.07, 39.06, 41.07, 42.08, 43.09  , 49.05, 51.07, 52.08  , 50.06  , &
       40.07  , 40.07 , 44.11, 50.06, 54.09, 58.13, 78.1136, 38.05, 53.07, 77.1136, 28.0134, &
       14.01  , 26.02 , 27.04, 28.05, 39.05, 40.04, 41.05  , 50.04, 51.05, 52.04  , 76.1   /)
   
   ! !!!!!!!!!!!!!!!!!!!!
   !  Upper chemistry 
   ! !!!!!!!!!!!!!!!!!!!!
   
   ! !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   ! These parameters as well as nkim above, MUST match titan.h in chimtitan
   INTEGER, PARAMETER :: nd_kim   = 54     ! Number of photodissociations
   INTEGER, PARAMETER :: nr_kim   = 377    ! Number of reactions in chemistry scheme
   INTEGER, PARAMETER :: nlrt_kim = 650    ! For the UV rad. transf., 650 levels of 2 km
   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   
   INTEGER, SAVE :: nlaykim_up   ! Number of upper atm. layers for chemistry from GCM top to 4.5E-5 Pa (1300km)
   INTEGER, SAVE :: nlaykim_tot  ! Number of total layers for chemistry from surface to 4.5E-5 Pa (1300km)
!$OMP_THREADPRIVATE(nlaykim_up,nlay_kim_tot)

  ! NB : For the startfile we use nlaykim_up grid (upper atm) and for outputs we use nlaykim_tot grid (all layers)

   REAL,SAVE,ALLOCATABLE,DIMENSION(:) :: preskim  ! Pressure (Pa) of upper chemistry (mid)-layers
!$OMP_THREADPRIVATE(preskim)

   REAL,SAVE,ALLOCATABLE,DIMENSION(:,:) :: zlay_kim  ! Altitude (km) of all chemistry (mid)-layers
!$OMP_THREADPRIVATE(zlay_kim)

   REAL,SAVE,ALLOCATABLE,DIMENSION(:,:,:) :: ykim_up ! Upper chemistry fields (mol/mol)
!$OMP_THREADPRIVATE(ykim_up)

CONTAINS

  SUBROUTINE ini_comchem_h(ngrid)
  
  IMPLICIT NONE
  
    include "dimensions.h"
  
    INTEGER,INTENT(IN) :: ngrid ! number of atmospheric columns
  
    nlaykim_tot = nlaykim_up + llm
  
    IF (.NOT.allocated(preskim)) ALLOCATE(preskim(nlaykim_up))
    IF (.NOT.allocated(zlay_kim)) ALLOCATE(zlay_kim(ngrid,nlaykim_tot))
    IF (.NOT.allocated(ykim_up)) ALLOCATE(ykim_up(nkim,ngrid,nlaykim_up))
  
  END SUBROUTINE ini_comchem_h


END MODULE comchem_h
