! Copyright Université Reims Champagnne-Ardenne (2010-2015)
! contributor: Jérémie Burgalat
! 
! jeremie.burgalat@univ-reims.fr
! 
! This software is a computer program whose purpose is to compute multi-variate
! linear interpolation.
! 
! This software is governed by the CeCILL-B license under French law and
! abiding by the rules of distribution of free software.  You can  use, 
! modify and/ or redistribute the software under the terms of the CeCILL-B
! license as circulated by CEA, CNRS and INRIA at the following URL
! "http://www.cecill.info". 
! 
! As a counterpart to the access to the source code and  rights to copy,
! modify and redistribute granted by the license, users are provided only
! with a limited warranty  and the software's author,  the holder of the
! economic rights,  and the successive licensors  have only  limited
! liability. 
! 
! In this respect, the user's attention is drawn to the risks associated
! with loading,  using,  modifying and/or developing or reproducing the
! software by the user in light of its specific status of free software,
! that may mean  that it is complicated to manipulate,  and  that  also
! therefore means  that it is reserved for developers  and  experienced
! professionals having in-depth computer knowledge. Users are therefore
! encouraged to load and test the software's suitability as regards their
! requirements in conditions enabling the security of their systems and/or 
! data to be ensured and,  more generally, to use and operate it in the 
! same conditions as regards security. 
! 
! The fact that you are presently reading this means that you have had
! knowledge of the CeCILL-B license and that you accept its terms.

!! file: datasets.F90
!! summary: Dataset module definition file
!! author: J. Burgalat
!! date: 2014

#if HAVE_CONFIG_H
#include "config.h"
#endif

MODULE DATASETS 
  !! dataset definitions module
  !!
  !! This module defines simple derived types that encapsulate data set variables. For a N-dimension 
  !! dataset, N vectors of \(n_{i}\) elements and a single N-dimensional array of 
  !! \(\prod_{i}^{N} n_{i}\) elements are defined.
  !!
  !! If the data set is to be used within [[lintdset(module)]] module then each coordinate values of the 
  !! dataset must be sorted either in ascending or descending order with no duplicates. The module does 
  !! not provide functionnality to sort and to check such requirements.
  !!
  !! The module also provides two interfaces to initialize data sets from input data file which can be 
  !! a NetCDF file (if NetCDF support is enabled at compilation time) or an ASCII file. In the latter 
  !! case, the file must contain a header which is formatted as follows:
  !!
  !! - The first line must contain only one value which is the number of coordinates (__N__)
  !! - The second line must contain all the dimension sizes (that is __N__ values)
  !! - Each other lines should contain __N+1__ columns with, for the __N__ first columns, the 
  !!   coordinates values and finally the point value in the last column.
  !! 
  !! @note
  !! Note that for ASCII file, data must be ordered so first dimensions vary first. Same requirement 
  !! is needed for NetCDF file but in most cases, it is implicitly done (if dimensions are ordered).
  USE LINT_PREC
#if HAVE_NC_FTN
  USE NETCDF
#endif
  IMPLICIT NONE

  PRIVATE 

  PUBLIC :: read_dset, clear_dset, is_in, debug

  LOGICAL :: debug = .false.  !! A control flag to enable verbose mode

#if HAVE_NC_FTN
  LOGICAL, PUBLIC, PARAMETER ::nc_supported = .true. !! NetCDF input files are supported
#else
  LOGICAL, PUBLIC, PARAMETER ::nc_supported = .false. !! NetCDF input files are not supported
#endif

  !> Initialize a data set from either an ASCII or a NetCDF file
  !!
  !! Whatever the kind of file, this interface assumes that you know the dimensionnality of the 
  !! extracted variable. If file content is not compatible with the kind of data set given as 
  !! output argument, or if some I/O error occured, the dataset is cleared.
  !! @note 
  !! Netcdf reader interface is available only if the library has been compiled with NetCDF support.
  INTERFACE read_dset 
#if HAVE_NC_FTN
    MODULE PROCEDURE ncdf_rd_1d,ncdf_rd_2d,ncdf_rd_3d,ncdf_rd_4d,ncdf_rd_5d
#if HAVE_NC4_FTN
    MODULE PROCEDURE ncdf4_rd_1d,ncdf4_rd_2d,ncdf4_rd_3d,ncdf4_rd_4d,ncdf4_rd_5d
#endif
#endif
    MODULE PROCEDURE ascii_rd_1d,ascii_rd_2d,ascii_rd_3d,ascii_rd_4d,ascii_rd_5d
  END INTERFACE

  !> Clear the given data set
  INTERFACE clear_dset 
    MODULE PROCEDURE clr_1d_set,clr_2d_set,clr_3d_set,clr_4d_set,clr_5d_set
  END INTERFACE

  !> Check if given point is within the data set
  INTERFACE is_in 
    MODULE PROCEDURE is_in_1d,is_in_2d,is_in_3d,is_in_4d,is_in_5d
  END INTERFACE

  !> Private interface to netcdf informations getters
#if HAVE_NC_FTN
  INTERFACE get_nc_info
    MODULE PROCEDURE get_nc3_info
#if HAVE_NC4_FTN
    MODULE PROCEDURE get_nc4_info 
#endif
  END INTERFACE
#endif

  TYPE, PUBLIC :: DSET1D
    !! A 1D data set
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE :: x    !! X coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE :: data !! Tabulated function's value at each coordinate
  END TYPE DSET1D

  TYPE, PUBLIC :: DSET2D
    !! A 2D data set
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE   :: x    !! X coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE   :: y    !! Y coordinate tabulated values
    REAL(kind=wp), DIMENSION(:,:), ALLOCATABLE :: data !! Tabulated function's value at each coordinate
  END TYPE DSET2D

  TYPE, PUBLIC :: DSET3D
    !! A 3D data set
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE     :: x    !! X coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE     :: y    !! Y coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE     :: z    !! Z coordinate tabulated values
    REAL(kind=wp), DIMENSION(:,:,:), ALLOCATABLE :: data !! Tabulated function's value at each coordinate
  END TYPE DSET3D

  TYPE, PUBLIC :: DSET4D
    !! A 4D data set
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE       :: x    !! X coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE       :: y    !! Y coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE       :: z    !! Z coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE       :: t    !! T coordinate tabulated values
    REAL(kind=wp), DIMENSION(:,:,:,:), ALLOCATABLE :: data !! Tabulated function's value at each coordinate
  END TYPE DSET4D

  TYPE, PUBLIC :: DSET5D
    !! A 5D data set
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE         :: x    !! X coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE         :: y    !! Y coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE         :: z    !! Z coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE         :: t    !! T coordinate tabulated values
    REAL(kind=wp), DIMENSION(:), ALLOCATABLE         :: w    !! W coordinate tabulated values
    REAL(kind=wp), DIMENSION(:,:,:,:,:), ALLOCATABLE :: data !! Tabulated function's value at each coordinate
  END TYPE DSET5D

  CONTAINS


  FUNCTION is_in_1d(set,x) RESULT(ret)
    !! Check if point is in the 1D data set
    TYPE(DSET1D), INTENT(in)  :: set !! Dataset object to search in
    REAL(kind=wp), INTENT(in) :: x   !! coordinate of the point to check
    LOGICAL :: ret                   !! .true. if the point is in the data set, .false. otherwise
    REAL(kind=wp) :: l,u
    ret=.true.
    l  = set%x(1) ; u= set%x(size(set%x))
    IF ((x>=l.EQV.x<=u).OR.(x<=l.EQV.x>=u)) ret=.false.
    RETURN
  END FUNCTION is_in_1d

  FUNCTION is_in_2d(set,x,y) RESULT(ret)
    !! Check if point is in the 2D data set
    TYPE(DSET2D), INTENT(in)  :: set !! Dataset object to search in
    REAL(kind=wp), INTENT(in) :: x   !! X coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: y   !! Y coordinate of the point to check
    LOGICAL :: ret                   !! .true. if the point is in the data set, .false. otherwise
    REAL(kind=wp) :: l,u
    ret=.false.
    l  = set%x(1) ; u= set%x(size(set%x))
    IF ((x>l.AND.x>u).OR.(x<l.AND.x<u)) RETURN
    IF ((x>l.AND.x>u).OR.(x<l.AND.x<u)) RETURN 
    l  = set%y(1) ; u= set%y(size(set%y))
    IF ((y>l.AND.y>u).OR.(y<l.AND.y<u)) RETURN
    ret=.true.
    RETURN
  END FUNCTION is_in_2d

  FUNCTION is_in_3d(set,x,y,z) RESULT(ret)
    !! Check if point is in the 3D data set
    TYPE(DSET3D), INTENT(in)  :: set !! Dataset object to search in
    REAL(kind=wp), INTENT(in) :: x   !! X coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: y   !! Y coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: z   !! Z coordinate of the point to check
    LOGICAL :: ret                   !! .true. if the point is in the data set, .false. otherwise
    REAL(kind=wp) :: l,u
    ret=.false.
    l  = set%x(1) ; u= set%x(size(set%x))
    IF ((x>l.AND.x>u).OR.(x<l.AND.x<u)) RETURN 
    l  = set%y(1) ; u= set%y(size(set%y))
    IF ((y>l.AND.y>u).OR.(y<l.AND.y<u)) RETURN
    l  = set%z(1) ; u= set%z(size(set%z))
    IF ((z>l.AND.z>u).OR.(z<l.AND.z<u)) RETURN
    ret=.true.
    RETURN
  END FUNCTION is_in_3d

  FUNCTION is_in_4d(set,x,y,z,t) RESULT(ret)
    !! Check if point is in the 4D data set
    TYPE(DSET4D), INTENT(in)  :: set !! Dataset object to search in
    REAL(kind=wp), INTENT(in) :: x   !! X coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: y   !! Y coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: z   !! Z coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: t   !! T coordinate of the point to check
    LOGICAL :: ret                   !! .true. if the point is in the data set, .false. otherwise
    REAL(kind=wp) :: l,u
    ret=.false.
    l  = set%x(1) ; u= set%x(size(set%x))
    IF ((x>l.AND.x>u).OR.(x<l.AND.x<u)) RETURN 
    l  = set%y(1) ; u= set%y(size(set%y))
    IF ((y>l.AND.y>u).OR.(y<l.AND.y<u)) RETURN
    l  = set%z(1) ; u= set%z(size(set%z))
    IF ((z>l.AND.z>u).OR.(z<l.AND.z<u)) RETURN
    l  = set%t(1) ; u= set%t(size(set%t))
    IF ((t>l.AND.t>u).OR.(t<l.AND.t<u)) RETURN
    ret=.true.
    RETURN
  END FUNCTION is_in_4d

  FUNCTION is_in_5d(set,x,y,z,t,w) RESULT(ret)
    !! Check if point is in the 4D data set
    TYPE(DSET5D), INTENT(in)  :: set !! Dataset object to search in
    REAL(kind=wp), INTENT(in) :: x   !! X coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: y   !! Y coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: z   !! Z coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: t   !! T coordinate of the point to check
    REAL(kind=wp), INTENT(in) :: w   !! W coordinate of the point to check
    LOGICAL :: ret                   !! .true. if the point is in the data set, .false. otherwise
    REAL(kind=wp) :: l,u
    ret=.false.
    l  = set%x(1) ; u= set%x(size(set%x))
    IF ((x>l.AND.x>u).OR.(x<l.AND.x<u)) RETURN 
    l  = set%y(1) ; u= set%y(size(set%y))
    IF ((y>l.AND.y>u).OR.(y<l.AND.y<u)) RETURN
    l  = set%z(1) ; u= set%z(size(set%z))
    IF ((z>l.AND.z>u).OR.(z<l.AND.z<u)) RETURN
    l  = set%t(1) ; u= set%t(size(set%t))
    IF ((t>l.AND.t>u).OR.(t<l.AND.t<u)) RETURN
    l  = set%w(1) ; u= set%w(size(set%w))
    IF ((w>l.AND.w>u).OR.(w<l.AND.w<u)) RETURN
    ret=.true.
    RETURN
  END FUNCTION is_in_5d

  FUNCTION ascii_header(path,ds,dp) RESULT(ret)
    !! Read ASCII file header
    !! 
    !! The method assumes the header is on two lines :
    !!
    !! - the first line must contain a single value which is the number of
    !!   dimensions.
    !! - the second must contain N values with the size of each dimensions (N
    !!   referring to the first line number).
    !!
    !! The file remains open in the logical unit 666 unless some error occured. 
    CHARACTER(len=*), INTENT(in)       :: path !! Path of the ASCII file to read
    INTEGER, INTENT(out), DIMENSION(:) :: ds   !! Size of each dimensions
    INTEGER, INTENT(out), DIMENSION(:) :: dp   !! Product of each lower dimension size (1 for 1st)
    LOGICAL :: ret                             !! .true. if no error occured, .false. otherwise
    INTEGER           :: nd,i,e
    CHARACTER(len=15) :: i2s
    INQUIRE(666,OPENED=ret)
    IF (ret) THEN
      WRITE(*,*) 'ERROR: LUN 666 already used...'
      ret = .false. ; RETURN
    ENDIF
    ret = .false.
    OPEN(666,file=TRIM(path))
    READ(666,*) nd
    IF (nd /= SIZE(ds)) THEN
      WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
      WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be "//TRIM(i2s)//"D"
      RETURN
    ENDIF
    READ(666,*,iostat=e) ds
    IF (e /= 0) THEN
      WRITE(*,'(a)') 'ERROR: Cannot get dimensions size'
      CLOSE(666) ; RETURN
    ENDIF
    dp(1) = 1
    DO i=2,nd
      dp(i)=PRODUCT(ds(:i-1))
    ENDDO
    ret = .true.
  END FUNCTION ascii_header

#if HAVE_NC_FTN
  FUNCTION nc_get_dim(fid,did,var,values,ds) RESULT(ret)
    !! Get dimensions information of a NetCDF variable 
    !!
    !! The method gets the values and size of a given dimension.
    INTEGER, INTENT(in)                                   :: fid
      !! Id of the NetCDF file
    INTEGER, INTENT(in)                                   :: did
      !! Id of the NetCDF dimension 
    CHARACTER(len=*), INTENT(in)                          :: var
      !! Name of the related NetCDF variable
    REAL(kind=wp), INTENT(out), DIMENSION(:), ALLOCATABLE :: values
      !! Values of the dimension
    INTEGER, INTENT(out)                                  :: ds 
      !! Size of __values__
    LOGICAL :: ret
      !! .true. if no error(s) occured, .false. otherwise
    INTEGER                      :: vid,err
    CHARACTER(len=NF90_MAX_NAME) :: dn
    CHARACTER(len=15)            :: i2s
    ret = .false.
    ! --- Get dimension informations
    IF (NF90_INQUIRE_DIMENSION(fid,did,dn,ds) /= NF90_NOERR) THEN
      IF (debug) THEN
        WRITE(i2s,*) did ; i2s=TRIM(i2s)
        WRITE(*,'(a)') "ERROR:"//TRIM(var)//": Cannot find "// &
                 "dimension #"//TRIM(i2s)//" name"
      ENDIF
      err = NF90_CLOSE(fid) ; RETURN
    ENDIF
    IF (ds < 2) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(dn)//": Invalid dimension "// &
                 "size (<2)"
      err = NF90_CLOSE(fid) ; RETURN
    ENDIF
    IF (NF90_INQ_VARID(fid,TRIM(dn),vid) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(dn)//": Cannot get ID"
      err = NF90_CLOSE(fid) ; RETURN
    ENDIF
    ALLOCATE(values(ds))
    IF (NF90_GET_VAR(fid,vid,values) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(dn)//": Cannot get values"
      err = NF90_CLOSE(fid) ; RETURN
    ENDIF
    ret = .true.
  END FUNCTION nc_get_dim

  FUNCTION get_nc3_info(path,variable,ncid,vid,dimids) RESULT(ret)
    !! Get variable informations from NetCDF file
    !! 
    !! The method attempts to read the given NetCDF file header and retrieves
    !! variable's informations from it:
    !!
    !! - parent file ID
    !! - variable ID
    !! - variable's dimensions ID
    !! - variable's dimensions sizes
    !!
    !! If no error occured (i.e. the function has returned .true.) then the
    !! NetCDF file remains open. In other case, the file is closed.
    CHARACTER(len=*), INTENT(in)                    :: path
      !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in)                    :: variable
      !! Name of the NetCDF variable to extract 
    INTEGER, INTENT(out)                            :: ncid
      !! NetCDF file ID 
    INTEGER, INTENT(out)                            :: vid
      !! NetCDF Variable ID
    INTEGER, INTENT(out), DIMENSION(:), ALLOCATABLE :: dimids
      !! Id of the variable's dimensions. __dimids__ is not allocated on error.
    LOGICAL :: ret
      !! .true. if no errors occured, .false. otherwise
    INTEGER :: fid,ty,nc,err
    ret = .false.
    ! Opens file
    IF (NF90_OPEN(TRIM(path),NF90_NOWRITE,fid) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'ERROR: Cannot open '//trim(path)
      RETURN
    ENDIF
    ncid = fid
    ! Searches for variable
    IF (NF90_INQ_VARID(ncid,TRIM(variable),vid) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot find '//TRIM(variable)//' in '//trim(path)
      nc = NF90_CLOSE(fid)
      RETURN
    ENDIF
    ! Get variable infos
    ! 1st call to get type and number of dimensions)
    IF (NF90_INQUIRE_VARIABLE(ncid,vid,xtype=ty,ndims=nc) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot access to '//TRIM(variable)//' informations'
      nc = NF90_CLOSE(fid)
      RETURN
    ELSE
      ! Checks type
      IF (ty == NF90_CHAR) THEN
        WRITE(*,'(a)') 'Inconsistent variable type (should be numeric)'
        nc = NF90_CLOSE(fid)
        RETURN
      ENDIF
      ALLOCATE(dimids(nc))
    ENDIF
    ! Gets variable's dimensions informations
    ! first get dimensions id
    IF (NF90_INQUIRE_VARIABLE(ncid,vid,dimids=dimids) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot access to '//TRIM(variable)//' informations'
      nc = NF90_CLOSE(fid)
      DEALLOCATE(dimids)
      RETURN
    ENDIF
    ret = .true. 
  END FUNCTION get_nc3_info

#if HAVE_NC4_FTN 
  FUNCTION get_nc4_info(path,variable,group,ncid,vid,dimids) RESULT(ret)
    !! Get variable informations from NetCDF4 file
    !!
    !! The method attempts to read the given NetCDF file header and retrieves
    !! variable's informations from it :
    !!
    !! - parent group/file ID
    !! - variable ID
    !! - variable's dimensions ID
    !! - variable's dimensions sizes
    !!
    !! If no error occured (i.e. the function has returned .true.) then the
    !! NetCDF file remains open. In other case, the file is closed.
    CHARACTER(len=*), INTENT(in)                    :: path
      !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in)                    :: variable
      !! Name of the NetCDF variable to extract 
    CHARACTER(len=*), INTENT(in)                    :: group
      !! Fullname of the variable's parent group (should be set to empty string for root group)
    INTEGER, INTENT(out)                            :: ncid
      !! NetCDF group ID 
    INTEGER, INTENT(out)                            :: vid
      !! NetCDF Variable ID
    INTEGER, INTENT(out), DIMENSION(:), ALLOCATABLE :: dimids
      !! Id of the variable's dimensions. On error, __dimids__ is not allocated. 
    LOGICAL :: ret
      !! .true. if no errors occured, .false. otherwise
    INTEGER :: fid,ty,nc,err
    ret = .false.
    ! Opens file
    IF (NF90_OPEN(TRIM(path),NF90_NOWRITE,fid) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'ERROR: Cannot open '//trim(path)
      RETURN
    ENDIF
    ! Searches for group based on its fullname
    IF (LEN_TRIM(group) == 0) THEN
      nc = NF90_NOERR ; ncid = fid
    ELSE
      nc = NF90_INQ_GRP_FULL_NCID(fid,TRIM(group),ncid)
    ENDIF
    SELECT CASE(nc)
      ! NF90_ENOGRP is missing from Netcdf-Fortran4.2 : its value=-125
      CASE(-125)
        WRITE(*,'(a)') TRIM(group)//' does not exist in '//TRIM(path)
        nc = NF90_CLOSE(fid) ; RETURN
      CASE(NF90_ENOTNC4,NF90_ESTRICTNC3)
        WRITE(*,'(a)') TRIM(path)//' is not a NetCDF-4 file with "group" feature'
        nc = NF90_CLOSE(fid) ; RETURN
      CASE(NF90_EHDFERR)
        WRITE(*,'(a)') "Too bad, an HDF5 error has been reported..."
        nc = NF90_CLOSE(fid) ; RETURN
    END SELECT
    ! Searches for variable
    IF (NF90_INQ_VARID(ncid,TRIM(variable),vid) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot find '//TRIM(variable)//' in '//trim(path)
      nc = NF90_CLOSE(fid)
      RETURN
    ENDIF
    ! Get variable infos
    ! 1st call to get type and number of dimensions)
    IF (NF90_INQUIRE_VARIABLE(ncid,vid,xtype=ty,ndims=nc) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot access to '//TRIM(variable)//' informations'
      nc = NF90_CLOSE(fid)
      RETURN
    ELSE
      ! Checks type
      IF (ty == NF90_CHAR) THEN
        WRITE(*,'(a)') 'Inconsistent variable type (should be numeric)'
        nc = NF90_CLOSE(fid)
        RETURN
      ENDIF
      ALLOCATE(dimids(nc))
    ENDIF
    ! Gets variable's dimensions informations
    ! first get dimensions id
    IF (NF90_INQUIRE_VARIABLE(ncid,vid,dimids=dimids) /= NF90_NOERR) THEN
      WRITE(*,'(a)') 'Cannot access to '//TRIM(variable)//' informations'
      nc = NF90_CLOSE(fid)
      DEALLOCATE(dimids)
      RETURN
    ENDIF
    ret = .true. 
  END FUNCTION get_nc4_info
#endif
#endif

  !-------------------------
  ! NetCDF data file readers
  !-------------------------

#if HAVE_NC_FTN


  FUNCTION ncdf_rd_1d(path,variable,set) RESULT(ret)
    !! Read a 1D data set from a NetCDF file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET1D), INTENT(out)    :: set      !! Output dataset object
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=NF90_MAX_NAME)       :: dn
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,variable,fi,vi,di)) RETURN
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 1) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get coordinates values 
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf_rd_1d

  FUNCTION ncdf_rd_2d(path,variable,set) RESULT(ret)
    !! Read a 2D data set from a NetCDF file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET2D), INTENT(out)    :: set      !! Output dataset object
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 2) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf_rd_2d

  FUNCTION ncdf_rd_3d(path,variable,set) RESULT(ret)
    !! Read a 3D data set from a NetCDF file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET3D), INTENT(out)    :: set      !! Output dataset object
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 3) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf_rd_3d

  FUNCTION ncdf_rd_4d(path,variable,set) RESULT(ret)
    !! Read a 4D data set from a NetCDF file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET4D), INTENT(out)    :: set      !! Output dataset object
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 4) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ T coordinate
    IF (.NOT.nc_get_dim(fi,di(4),variable,set%t,ds(4))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3),ds(4)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf_rd_4d

  FUNCTION ncdf_rd_5d(path,variable,set) RESULT(ret)
    !! Read a 5D data set from a NetCDF file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET5D), INTENT(out)    :: set      !! Output dataset object
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,variable,fi,vi,di)) RETURN
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 5) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ T coordinate
    IF (.NOT.nc_get_dim(fi,di(4),variable,set%t,ds(4))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ W coordinate
    IF (.NOT.nc_get_dim(fi,di(5),variable,set%w,ds(5))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3),ds(4),ds(5)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf_rd_5d

  ! NetCDF4 with groups !
#if HAVE_NC4_FTN 

  FUNCTION ncdf4_rd_1d(path,group,variable,set) RESULT(ret)
    !! Read a 1D data set from a NetCDF4 file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: group    !! Parent group of the variable (empty for root)
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET1D), INTENT(out)    :: set      !! Output dataset
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=NF90_MAX_NAME)       :: dn
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,group,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 1) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get coordinates values 
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf4_rd_1d

  FUNCTION ncdf4_rd_2d(path,group,variable,set) RESULT(ret)
    !! Read a 2D data set from a NetCDF4 file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: group    !! Parent group of the variable (empty for root)
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET2D), INTENT(out)    :: set      !! Output dataset
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,group,variable,fi,vi,di)) rETURN
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 2) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf4_rd_2d

  FUNCTION ncdf4_rd_3d(path,group,variable,set) RESULT(ret)
    !! Read a 3D data set from a NetCDF4 file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: group    !! Parent group of the variable (empty for root)
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET3D), INTENT(out)    :: set      !! Output dataset
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,group,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 3) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf4_rd_3d

  FUNCTION ncdf4_rd_4d(path,group,variable,set) RESULT(ret)
    !! Read a 4D data set from a NetCDF4 file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: group    !! Parent group of the variable (empty for root)
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET4D), INTENT(out)    :: set      !! Output dataset
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,group,variable,fi,vi,di)) RETURN 
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 4) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ T coordinate
    IF (.NOT.nc_get_dim(fi,di(4),variable,set%t,ds(4))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3),ds(4)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf4_rd_4d

  FUNCTION ncdf4_rd_5d(path,group,variable,set) RESULT(ret)
    !! Read a 5D data set from a NetCDF4 file
    CHARACTER(len=*), INTENT(in) :: path     !! Path of the NetCDF file
    CHARACTER(len=*), INTENT(in) :: group    !! Parent group of the variable (empty for root)
    CHARACTER(len=*), INTENT(in) :: variable !! Name of the NetCDF variable to extract 
    TYPE(DSET5D), INTENT(out)    :: set      !! Output dataset
    LOGICAL :: ret                           !! .true. if no errors occured, .false. otherwise
    INTEGER                            :: fi,vi,nd
    INTEGER, DIMENSION(:), ALLOCATABLE :: di,ds
    CHARACTER(len=15)                  :: i2s
    ret = .false.
    ! --- Reset the data set
    CALL clear_dset(set)
    ! --- Check NetCDF file info
    IF (.NOT.get_nc_info(path,group,variable,fi,vi,di)) RETURN
    ! --- Check dimension size
    nd = SIZE(di)
    IF (nd /= 5) THEN
      IF (debug) THEN
        WRITE(i2s,*) nd ; i2s=ADJUSTL(i2s)
        WRITE(*,'(a)') "ERROR: Incompatible size, DSET should be"//TRIM(i2s)//"D"
      ENDIF
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Get dimensions informations
    ALLOCATE(ds(nd))
    ! ------ X coordinate
    IF (.NOT.nc_get_dim(fi,di(1),variable,set%x,ds(1))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Y coordinate
    IF (.NOT.nc_get_dim(fi,di(2),variable,set%y,ds(2))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ Z coordinate
    IF (.NOT.nc_get_dim(fi,di(3),variable,set%z,ds(3))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ T coordinate
    IF (.NOT.nc_get_dim(fi,di(4),variable,set%t,ds(4))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! ------ W coordinate
    IF (.NOT.nc_get_dim(fi,di(5),variable,set%w,ds(5))) THEN
      CALL clear_dset(set) ; RETURN
    ENDIF
    ! --- Read data
    ALLOCATE(set%data(ds(1),ds(2),ds(3),ds(4),ds(5)))
    IF (NF90_GET_VAR(fi,vi,set%data) /= NF90_NOERR) THEN
      IF (debug) WRITE(*,'(a)') "ERROR:"//TRIM(variable)//": Cannot get values"
      nd = NF90_CLOSE(fi) ; CALL clear_dset(set) ; RETURN
    ENDIF
    nd = NF90_CLOSE(fi)
    ret = .true.
    RETURN
  END FUNCTION ncdf4_rd_5d
#endif
#endif

  !------------------------
  ! ASCII data file readers
  !------------------------

  FUNCTION ascii_rd_1d(path,set) RESULT(ret)
    !! Read a 1D data set from a ASCII file
    CHARACTER(len=*), INTENT(in) :: path !! Path of the ASCII file to read
    TYPE(dset1d), INTENT(out)    :: set  !! output 1D dataset
    LOGICAL :: ret                       !! .true. if no error occured, .false. otherwise
    INTEGER                     :: i,e
    REAL(kind=wp), DIMENSION(2) :: vl
    INTEGER, DIMENSION(1)       :: cc,ds,dp
    ret = .false.
    CALL clear_dset(set)
    IF (.NOT.ascii_header(path,ds,dp)) RETURN
    ALLOCATE(set%x(ds(1)), &
             set%data(ds(1)))
    cc(:) = 1
    DO i=1,ds(1)
      READ(666,*) vl(:)
      set%x(cc(1)) = vl(1)
      set%data(cc(1)) = vl(2)
      cc(1) = cc(1) + 1
    ENDDO
    READ(666,*,iostat=e) vl(1) 
    IF (e == 0) THEN
      IF (debug) WRITE(*,*) 'ERROR: Extra value found...'
      ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
    ENDIF
    ret = .true.
    CLOSE(666)
    RETURN
  END FUNCTION ascii_rd_1d

  FUNCTION ascii_rd_2d(path,set) RESULT(ret)
    !! Read a 2D data set from a ASCII file
    CHARACTER(len=*), INTENT(in) :: path !! Path of the ASCII file to read
    TYPE(dset2d), INTENT(out)    :: set  !! output 2D dataset
    LOGICAL :: ret                       !! .true. if no error occured, .false. otherwise
    INTEGER                     :: i,e
    REAL(kind=wp), DIMENSION(3) :: vl 
    INTEGER, DIMENSION(2)       :: cc,ds,dp
    ret = .false.
    CALL clear_dset(set)
    IF (.NOT.ascii_header(path,ds,dp)) RETURN
    ALLOCATE(set%x(ds(1)),set%y(ds(2)), &
             set%data(ds(1),ds(2)))
    cc(:) = 1
    DO i=1,PRODUCT(ds) 
      READ(666,*,iostat=e) vl     ! Reads the line
      IF (e /= 0) THEN
        IF (debug) WRITE(*,'(a)') 'ERROR: Malformed file, line: ',i+2
        ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
      ENDIF
      ! --- X coordinate
      set%x(cc(1)) = vl(1)
      ! --- Y coordinate
      set%y(cc(2)) = vl(2)
      ! --- Data
      set%data(cc(1),cc(2)) = vl(3)
      ! - Update counters
      IF (mod(i,dp(1))==0) cc(1) = cc(1)+1 ; IF (cc(1) > ds(1)) cc(1)=1
      IF (mod(i,dp(2))==0) cc(2) = cc(2)+1 ; IF (cc(2) > ds(2)) cc(2)=1
    ENDDO
    READ(666,*,iostat=e) vl(1)
    IF (e == 0) THEN
      IF (debug) WRITE(*,'(a)') 'ERROR: Extra value found'
      ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
    ENDIF
    ret = .true.
    CLOSE(666)
    RETURN
  END FUNCTION ascii_rd_2d

  FUNCTION ascii_rd_3d(path,set) RESULT(ret)
    !! Read a 3D data set from a ASCII file
    CHARACTER(len=*), INTENT(in) :: path !! Path of the ASCII file to read
    TYPE(dset3d), INTENT(out)    :: set  !! output 3D dataset
    LOGICAL :: ret                       !! .true. if no error occured, .false. otherwise
    INTEGER                     :: i,e
    REAL(kind=wp), DIMENSION(4) :: vl 
    INTEGER, DIMENSION(3)       :: cc,ds,dp
    ret = .false.
    CALL clear_dset(set)
    IF (.NOT.ascii_header(path,ds,dp)) RETURN
    ALLOCATE(set%x(ds(1)),set%y(ds(2)),set%z(ds(3)), &
             set%data(ds(1),ds(2),ds(3)))
    cc(:) = 1
    DO i=1,PRODUCT(ds) 
      READ(666,*,iostat=e) vl     ! Reads the line
      IF (e /= 0) THEN
        IF (debug) WRITE(*,'(a)') 'ERROR: Malformed file, line: ',i+2
        ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
      ENDIF
      ! --- X coordinate
      set%x(cc(1)) = vl(1)
      ! --- Y coordinate
      set%y(cc(2)) = vl(2)
      ! --- Z coordinate
      set%z(cc(3)) = vl(3)
      ! --- Data
      set%data(cc(1),cc(2),cc(3)) = vl(4)
      ! - Update counters
      IF (mod(i,dp(1))==0) cc(1) = cc(1)+1 ; IF (cc(1) > ds(1)) cc(1)=1
      IF (mod(i,dp(2))==0) cc(2) = cc(2)+1 ; IF (cc(2) > ds(2)) cc(2)=1
      IF (mod(i,dp(3))==0) cc(3) = cc(3)+1 ; IF (cc(3) > ds(3)) cc(3)=1
    ENDDO
    READ(666,*,iostat=e) vl(1)
    IF (e == 0) THEN
      IF (debug) WRITE(*,'(a)') 'ERROR: Extra value found'
      ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
    ENDIF
    ret = .true.
    CLOSE(666)
    RETURN
  END FUNCTION ascii_rd_3d

  FUNCTION ascii_rd_4d(path,set) RESULT(ret)
    !! Read a 4D data set from a ASCII file
    CHARACTER(len=*), INTENT(in) :: path !! Path of the ASCII file to read
    TYPE(dset4d), INTENT(out)    :: set  !! output 4D dataset
    LOGICAL :: ret                       !! .true. if no error occured, .false. otherwise
    INTEGER                     :: i,e
    REAL(kind=wp), DIMENSION(5) :: vl 
    INTEGER, DIMENSION(4)       :: cc,ds,dp
    ret = .false.
    CALL clear_dset(set)
    IF (.NOT.ascii_header(path,ds,dp)) RETURN
    ALLOCATE(set%x(ds(1)),set%y(ds(2)),set%z(ds(3)),set%t(ds(4)), &
             set%data(ds(1),ds(2),ds(3),ds(4)))
    cc(:) = 1
    DO i=1,PRODUCT(ds) 
      READ(666,*,iostat=e) vl     ! Reads the line
      IF (e /= 0) THEN
        IF (debug) WRITE(*,'(a)') 'ERROR: Malformed file, line: ',i+2
        ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
      ENDIF
      ! --- X coordinate
      set%x(cc(1)) = vl(1)
      ! --- Y coordinate
      set%y(cc(2)) = vl(2)
      ! --- Z coordinate
      set%z(cc(3)) = vl(3)
      ! --- T coordinate
      set%t(cc(4)) = vl(4)
      ! --- Data
      set%data(cc(1),cc(2),cc(3),cc(4)) = vl(5)
      ! - Update counters
      IF (mod(i,dp(1))==0) cc(1) = cc(1)+1 ; IF (cc(1) > ds(1)) cc(1)=1
      IF (mod(i,dp(2))==0) cc(2) = cc(2)+1 ; IF (cc(2) > ds(2)) cc(2)=1
      IF (mod(i,dp(3))==0) cc(3) = cc(3)+1 ; IF (cc(3) > ds(3)) cc(3)=1
      IF (mod(i,dp(4))==0) cc(4) = cc(4)+1 ; IF (cc(4) > ds(4)) cc(4)=1
    ENDDO
    READ(666,*,iostat=e) vl(1)
    IF (e == 0) THEN
      IF (debug) WRITE(*,'(a)') 'ERROR: Extra value found'
      ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
    ENDIF
    CLOSE(666)
    ret = .true.
    RETURN
  END FUNCTION ascii_rd_4d

  FUNCTION ascii_rd_5d(path,set) RESULT(ret)
    !! Read a 5D data set from a ASCII file
    CHARACTER(len=*), INTENT(in) :: path !! Path of the ASCII file to read
    TYPE(dset5d), INTENT(out)    :: set  !! output 5D dataset
    LOGICAL :: ret                       !! .true. if no error occured, .false. otherwise
    INTEGER                     :: i,e
    REAL(kind=wp), DIMENSION(6) :: vl 
    INTEGER, DIMENSION(5)       :: cc,ds,dp
    ret = .false.
    CALL clear_dset(set)
    IF (.NOT.ascii_header(path,ds,dp)) RETURN
    ALLOCATE(set%x(ds(1)),set%y(ds(2)),set%z(ds(3)),set%t(ds(4)),set%w(ds(5)), &
             set%data(ds(1),ds(2),ds(3),ds(4),ds(5)))
    cc(:) = 1
    DO i=1,PRODUCT(ds) 
      READ(666,*,iostat=e) vl     ! Reads the line
      IF (e /= 0) THEN
        IF (debug) WRITE(*,'(a)') 'ERROR: Malformed file, line: ',i+2
        ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
      ENDIF
      ! --- X coordinate
      set%x(cc(1)) = vl(1)
      ! --- Y coordinate
      set%y(cc(2)) = vl(2)
      ! --- Z coordinate
      set%z(cc(3)) = vl(3)
      ! --- T coordinate
      set%t(cc(4)) = vl(4)
      ! --- W coordinate
      set%w(cc(5)) = vl(5)
      ! --- Data
      set%data(cc(1),cc(2),cc(3),cc(4),cc(5)) = vl(6)
      ! - Update counters
      IF (mod(i,dp(1)) == 0) cc(1)=cc(1)+1 ; IF (cc(1) > ds(1)) cc(1)=1
      IF (mod(i,dp(2)) == 0) cc(2)=cc(2)+1 ; IF (cc(2) > ds(2)) cc(2)=1
      IF (mod(i,dp(3)) == 0) cc(3)=cc(3)+1 ; IF (cc(3) > ds(3)) cc(3)=1
      IF (mod(i,dp(4)) == 0) cc(4)=cc(4)+1 ; IF (cc(4) > ds(4)) cc(4)=1
      IF (mod(i,dp(5)) == 0) cc(5)=cc(5)+1 ; IF (cc(5) > ds(5)) cc(5)=1
    ENDDO
    READ(666,*,iostat=e) vl(1)
    IF (e == 0) THEN
      IF (debug) WRITE(*,'(a)') 'ERROR: Extra value found'
      ret = .false. ; call clear_dset(set) ; CLOSE(666) ; RETURN
    ENDIF
    CLOSE(666)
    ret = .true.
    RETURN
  END FUNCTION ascii_rd_5d

  SUBROUTINE clr_1d_set(set)
    !! Clear the given 1D data set
    TYPE(DSET1D), INTENT(inout) :: set !! dataset object to clear
    IF (ALLOCATED(set%data)) DEALLOCATE(set%data)
    IF (ALLOCATED(set%x))    DEALLOCATE(set%x)
    RETURN
  END SUBROUTINE clr_1d_set

  SUBROUTINE clr_2d_set(set)
    !! Clear the given 2D data set
    TYPE(DSET2D), INTENT(inout) :: set !! dataset object to clear
    IF (ALLOCATED(set%data)) DEALLOCATE(set%data)
    IF (ALLOCATED(set%x))    DEALLOCATE(set%x)
    IF (ALLOCATED(set%y))    DEALLOCATE(set%y)
    RETURN
  END SUBROUTINE clr_2d_set

  SUBROUTINE clr_3d_set(set)
    !! Clear the given 3D data set
    TYPE(DSET3D), INTENT(inout) :: set !! dataset object to clear
    IF (ALLOCATED(set%data)) DEALLOCATE(set%data)
    IF (ALLOCATED(set%x))    DEALLOCATE(set%x)
    IF (ALLOCATED(set%y))    DEALLOCATE(set%y)
    IF (ALLOCATED(set%z))    DEALLOCATE(set%z)
    RETURN
  END SUBROUTINE clr_3d_set

  SUBROUTINE clr_4d_set(set)
    !! Clear the given 4D data set
    TYPE(DSET4D), INTENT(inout) :: set !! dataset object to clear
    IF (ALLOCATED(set%data)) DEALLOCATE(set%data)
    IF (ALLOCATED(set%x))    DEALLOCATE(set%x)
    IF (ALLOCATED(set%y))    DEALLOCATE(set%y)
    IF (ALLOCATED(set%z))    DEALLOCATE(set%z)
    IF (ALLOCATED(set%t))    DEALLOCATE(set%t)
    RETURN
  END SUBROUTINE clr_4d_set

  SUBROUTINE clr_5d_set(set)
    !! Clear the given 5D data set
    TYPE(DSET5D), INTENT(inout) :: set !! dataset object to clear 
    IF (ALLOCATED(set%data)) DEALLOCATE(set%data)
    IF (ALLOCATED(set%x))    DEALLOCATE(set%x)
    IF (ALLOCATED(set%y))    DEALLOCATE(set%y)
    IF (ALLOCATED(set%z))    DEALLOCATE(set%z)
    IF (ALLOCATED(set%t))    DEALLOCATE(set%t)
    IF (ALLOCATED(set%w))    DEALLOCATE(set%w)
    RETURN
  END SUBROUTINE clr_5d_set

END MODULE DATASETS
