!******************************************************************************
!*     venus_SAS_composition SUBROUTINE
!*     modified from
!*     PROGRAM PSC_MODEL_E
!*     by A. Mttnen
!*     subroutine for LMDZ+photochemistry VENUS
!*     by A. Stolzenbach
!*
!*     Input/Output files:
!*     -------------------
!*
!----------------------------------------------------------------------------
      SUBROUTINE new_cloud_venus(deltaT,NDTOT,
     + MEDIAN,GSTDEV,tt,
     + pp,ppwv,
     + mr_wv,mr_sa,
     + niv,
     + mrt_wv,mrt_sa,
     + WSA,
     + PSSA,SATPSSA,
     + RHOSASL)


!      USE real16
      USE chemparam_mod
      IMPLICIT NONE

!     Niveau (correspond pression, altitude fixe des couches nuageuses)
      INTEGER niv


!     Aerosol and PSC variables:
      REAL 
     +        WSA,WWV
!      REAL  RMIN,RMAX
!----------------------------------------------------------------------------
!     Ambient air state variables:
      REAL 
     +        tt,pp,
     +        mr_wv,mr_sa,
     +        PPWV,PPSA,
     +        PSSA,SATPSSA

!----------------------------------------------------------------------------
!     Physical constants:
      REAL  MH2O,MH2SO4,MAIR,CWV,CSA !,CNA,MHNO3
      PARAMETER(
     +          MH2O=18.0153d-3,
!     +          MHNO3=63.01d-3,
     +          MH2SO4=98.078d-3,
!     +          MAIR=28.9644d-3,
!AM Venus
     +          MAIR=43.45d-3,
     +          CWV=MAIR/MH2O,
!     +          CNA=MAIR/MHNO3,
     +          CSA=MAIR/MH2SO4)
!     Thermodynamic functions:
      REAL  ROSAS 
!AM 
!     Mathematical constants:
      REAL  PI
      PARAMETER(PI=3.1415926536)
      
!----------------------------------------------------------------------------
!     Time variables:
	REAL deltaT
!----------------------------------------------------------------------------
!     Auxilary variables:
      REAL  
     +     NDTOT,MEDIAN,GSTDEV,
     +     mrt_wv,mrt_sa,
     +     NH2SO4,NH2O,
     +     MASS, 
     +     X0,X1,X2,X3,X4,X5,X6,
     +     RHOSASL,MSAL,
     +     waterps,condmass,RMH2S4
      REAL  H2SO4,H2SO4_liq,H2O_liq
      REAL  RSTDEV
      REAL  RMEDRA
      REAL  R2SO4
      REAL  DENSO4
      REAL  CONHS4
      REAL  H2O
      REAL  ACTSO4
      REAL  CONCM
      REAL  mrsa_conc
      REAL  RNLOG

! >>> Program starts here:


!     mass of an H2SO4 molecule (g)
      RMH2S4=98.078/(6.02214129d23)

!AM Venus
!Here we call a subroutine that contains a nucleation parametrisation for stratosphere and 
!Venus and use that for calculating the number density of liquid sulfate aerosols. These 
!aerosols will then be given an equilibrium composition for the given size distribution
! calculates binary nucleation rate using revised theory, stauffer+binder&stauffer kinetics
 ! and   noppel hydrate correction
 ! t     temperature [K]
 ! rehu    relative humidity %/100 which means 100%=1
 ! rhoa  concentration of h2so4 vapour [1/m^3] 
 ! x     mole fraction in the core of the critical cluster  
 ! nwtot total number of water molecules in the critical cluster
 ! natot total number of h2so4 molecules in the critical cluster
 ! rc    radius of the critical cluster core [m]
 ! jnuc  nucleation rate [1/m^3s] 

  ! Hanna Vehkamki and Markku Kulmala and Ismo Napari 
  ! and Kari E. J. Lehtinen and Claudia Timmreck and Madis Noppel and Ari Laaksonen, 2002,
  ! An improved parameterization for sulfuric acid/water nucleation rates for tropospheric 
  !and stratospheric conditions, () J. Geophys. Res., 107, pp. 4622-4631

 
             IF (niv.GE.cloudmin .AND. niv.LE.cloudmax) THEN

              CALL WGTGV(MEDIAN,TT,PPWV,WSA,RHOSASL,MSAL)          

              R2SO4=WSA*100.
!     R2SO4 -> activity coeff (ACTSO4)
              CALL STRAACT(R2SO4,ACTSO4)
!              write(*,*) 'R2SO4,ACTSO4 ',R2SO4,ACTSO4
!     R2SO4, T -> aerosol density (R2SO4)

              DENSO4=ROSAS(TT,WSA)
!	units g/cm3 required by the following routines
              DENSO4=DENSO4*1.d-3            

              CONCM= (PP)/(1.3806488D-23*TT) !air number density, molec/m3? CHECK UNITS!
              CONCM=CONCM*1.d-6 !in molec./cm3

		  NH2SO4=mrt_sa*CONCM
		  NH2O=mrt_wv*CONCM

              CALL CALNLOG_SAT(ACTSO4,NH2SO4,NH2O,WSA,DENSO4,GSTDEV,
     +             MEDIAN,TT,RNLOG,H2SO4_liq,H2O_liq,
     +		 PSSA,SATPSSA)

		  
!		  NDTOT nbr # pour 1cm3
              NDTOT=RNLOG

!      IF ((NDTOT.GT.1.0d+3).OR.
!     & ((niv.GT.45).AND.(mr_wv.GT.1.0e-6))) THEN
!      PRINT*,'PROBLEME GENERAL AVEC CES PUTAINS DE ROUTINES'
!      PRINT*,'H2SO4COND',H2SO4_liq/CONCM,'H2SO4',mr_sa
!      PRINT*,'DND2',SATPSSA*1.0d-6/(1.38D-23*TT)
!      PRINT*,'NH2O',NH2O,'NH2SO4',NH2SO4
!      PRINT*,'H2OCOND',H2O_liq/CONCM,'H2O',mr_wv
!      PRINT*,'H2SO4tot',mrt_sa,'H2Otot',mrt_wv
!      PRINT*,'MEDIAN',MEDIAN,'GSTDEV',GSTDEV
!      PRINT*,'NBRTOT',NDTOT,'level',niv,'WSA',WSA
!      STOP
!      ENDIF
      
      
          mr_wv=mrt_wv-H2O_liq/CONCM
          mr_sa=mrt_sa-H2SO4_liq/CONCM
          
          
!          Problmes quand on a condense tout, on peut obtenir des -1e-24
!		aprs la soustraction et conversion de ND  VMR
          IF (mr_wv.LT.0.0) THEN
          mr_wv=0.0d0
          END IF
          
          IF (mr_sa.LT.0.0) THEN
          mr_sa=0.0d0
          END IF
	
	
             ELSE
!             PRINT*,'**** NDTOT OUT CLOUD ****'
             NDTOT=0.0d0
             WSA=0.0d0
             PSSA=0.0d0
             SATPSSA=0.0d0
             RHOSASL=0.0d0
!             write(*,*) 'NDTOT = 0.0!!'
             END IF
             
      END


******************************************************************************
*     SUBROUTINE WGTGV(RADIUS,TAIR,PPWV,WSAS,RHOSAS,MSA)
******************************************************************************
*
*     This subroutine calculates the acid mass fraction, density, and
*     mass of sulfuric acid in a single aerosol droplet of a specified 
*     radius in equilibrium with ambient water vapor partial pressure 
*     and temperature.
*
*     The calculation is performed by iteration of
*        ln(PPWV) - [(2Mh2o sigma)/(R T r rho) - ln(ph2osa)] = 0
*     using the secant method. Vapor pressures by Gmitro and Vermeulen
*     (PWVSAS_GV) are used.  
*
*     Input/output variables:
*     REAL(KIND=4)  RADIUS,TAIR,PPWV,WSAS,RHOSAS,MSA
*
*     Input:       
*         RADIUS:  m         Radius of aerosol droplet
*         TAIR:    K         Temperature of ambient air 
*         PPWV:    Pa        Partial pressure of ambient water vapor 
*
*     Output:
*         WSAS:              mass fraction of sulfuric acid. [0.1;1]
*         RHOSAS:  kg/m**3   Density of sulfuric acid solution droplet
*         MSA:     kg        Mass of sulfuric acid in droplet
*          CALL WGTGV(PTSIZE(25,1),TAIR,PPPWV,WSA,MSA1)
      SUBROUTINE WGTGV(RADIUS,TAIR,PPWV,WSAS,RHOSAS,MSA)
!
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      REAL   RADIUS,TAIR,PPWV,WSAS,RHOSAS,MSA
!
!     Physical constants:
      REAL   MH2O, RGAS
      PARAMETER(
!       Molar weight of water (kg/mole)
     +          MH2O=18.0153d-3,
!       Universal gas constant (J/(mole K))
     +          RGAS=8.31441d0)
!
!     Mathematical constants:
      REAL   PI
      PARAMETER(PI=3.1415926536d0)

!
!     External functions needed:
      REAL   PWVSAS_GV,STSAS,ROSAS
!     PWVSAS_GV:      Natural logaritm of water vapor pressure over
!                  sulfuric acid solution
!     STSAS:       Surface tension of sulfuric acid solution
!     ROSAS:       Density of sulfuric acid solution
!
!     Auxiliary local variables:
      REAL   DELW,DELLP,C1,C2,W0,W1,W2,F0,F1,WGUESS,LPPWV,RO
      INTEGER ITERAT,MAXITE
      REAL   WMIN
      PARAMETER(
!         Minimum H2SO4 weight fraction:
     +    WMIN=0.1D0,
!         Relative error on iterated weight fraction:
     +        DELW=0.001D0,
!         Relative error on iterated ln(pressure):
     +        DELLP=0.0001D0,
!         Guess of sulfuric acid mass fraction:
     +        WGUESS=0.7D0,
!         Maximum iteration number:
     +        MAXITE=20)

!
      PARAMETER(
     +        C1=2.0d0*MH2O/RGAS,
     +        C2=4.0d0*PI/3.0d0)
!

!----------------------------------------------------------------------------
!      write(*,*) 'in wgtgv, tair, radius, ppwv ', 
!     + tair, radius, ppwv

      W0=WGUESS
      LPPWV=DLOG(PPWV)
!      write(*,*) lppwv
      RO=ROSAS(TAIR,W0)
      F0=LPPWV-C1*STSAS(TAIR,W0)/(TAIR*RADIUS*RO)-PWVSAS_GV(TAIR,W0)
!      write(*,*) 'st pwvsas_gv ', STSAS(TAIR,W0), PWVSAS_GV(TAIR,W0)
!      write(*,*) 'F0, RO ', F0, RO
      W1=W0*1.01D0
      ITERAT=0
!----------------------------------------------------------------------------
10    RO=ROSAS(TAIR,W1)
      F1=LPPWV-C1*STSAS(TAIR,W1)/(TAIR*RADIUS*RO)-PWVSAS_GV(TAIR,W1)
!      write(*,*) 'st pwvsas_gv ', STSAS(TAIR,W1), PWVSAS_GV(TAIR,W1)
!      write(*,*) 'F1, RO ', F1, RO
      IF(ABS(F1-F0).LT.DELLP) THEN
          WSAS=W1
!          write(*,*) 'wsas1 in wgtgv ', WSAS
          RHOSAS=RO
          MSA=C2*WSAS*RHOSAS*RADIUS**3
      ELSE
          W2=MAX(0.0D0,MIN((F1*W0-F0*W1)/(F1-F0),1.0D0))
!          write(*,*) 'w2 max ', w2
          ITERAT=ITERAT+1
          IF(ABS(W2-W1).LT.DELW*ABS(W2).OR.ABS(F1).LT.DELLP.OR.
     +             ITERAT.GT.MAXITE) THEN
              WSAS=W2
!              write(*,*) 'wsas2 in wgtgv ', WSAS
              RHOSAS=RO
              MSA=C2*WSAS*RHOSAS*RADIUS**3
          ELSE
              W0=W1
              W1=W2
!              write(*,*) 'w0, w1, endloop wgtgv ', W0, W1
              F0=F1
              GOTO 10
          ENDIF
      ENDIF
      IF(WSAS.LT.WMIN) THEN
          WSAS=WMIN
          RHOSAS=ROSAS(TAIR,WMIN)
      ENDIF

      if(wsas .eq. 1.0) then
         wsas=0.999999d0
      endif


!----------------------------------------------------------------------------
      RETURN 
      END


!*****************************************************************************
!*     REAL FUNCTION ROSAS(TAIR,WSA)                                         
      REAL	FUNCTION ROSAS(TAIR,WSA)
!*****************************************************************************
!*
!*     Density of liquid sulfuric acid solution.
!*
!*     Source: John H.Perry (ed.):Chemical Engineers Handbook,
!*                                McGraw-Hill, New York 1963, p. 3-79 & 3-80
!*
!*     The original data set in temp. range 0 ! to 20 ! and weight pct.
!*     0 to 100 % has been fitted with a polynomium of two variables
!*     of order 5 in W and lineary in T. Fit quality better than 0.5 %
!*
!*     Input:  TAIR: Temperature  (K)
!*             WSA:  Weight fraction of H2SO4  [0;1] 
!*     Output: Density of sulfuric acid solution  (kg/m**3)
!*
!
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER I
      REAL   TAIR,WSA
      REAL, DIMENSION(6) :: C
      REAL, DIMENSION(6) :: A
      REAL, DIMENSION(6) :: B
      REAL, DIMENSION(6) :: D
      DATA (A(I),I=1,6)/
     # 1.00190D+03, 5.50496D+02, 1.54093D+03,-4.89219D+03, 7.56555D+03,
     #-3.92739D+03/
      DATA (B(I),I=1,6)/
     # 1.98378D+01, 1.02256D+03,-1.48665D+03,-7.24651D+02, 3.68348D+03,
     #-2.22159D+03/
      DATA (D(I),I=1,6)/
     #-6.97011E-02,-3.59886D+00, 5.24992D+00, 2.54047D+00,-1.29355D+01,
     # 7.80553D+00/
!C
      DO I=1,6
              C(I)=A(I)+B(I)+D(I)*TAIR
      ENDDO

      ROSAS=C(1)+WSA*(C(2)+WSA*(C(3)+WSA*(C(4)+WSA*(C(5)+WSA*C(6)))))
      
      RETURN
      END function rosas

*****************************************************************************
*     REAL FUNCTION STSAS(TAIR,WSA)                                         *
!    REAL FUNCTION STSAS(TAIR,WSA)
*****************************************************************************
*
*     Surface tension of sulfuric acid solution/vapor.
*
*     Source: Tabazadeh et al. JGR, 102,23845,1997
*             Sabinina & Terpugov: Z. Phys. Chem. A173 ,237, 1935.
*
*
*     Input:  TAIR: Temperature (K)
*             WSA:  Weight fraction of H2SO4  [0;1]
*     Output: Surface tension of sulfuric acid solution (N/m)
*
!    IMPLICIT NONE
!    REAL(KIND=4)  TAIR,WSA,W
!    W=WSA*100.0d0
!    STSAS=1.0d-3*(142.35d0-0.96525d0*W-TAIR*(0.22954d0-0.0033948d0*W))
!    RETURN
!    END
*****************************************************************************
*     REAL FUNCTION STSAS(TAIR,WSA)                                         *
      REAL     FUNCTION STSAS(TAIR,WSA)
*****************************************************************************
*
*     Surface tension of sulfuric acid solution/vapor.
*
*     Source: Tabazadeh et al. submitted,1999
*             Myhre et al., J. Chem. Eng. Data 43,617,1998.
*
*
*     Input:  TAIR: Temperature (K)
*             WSA:  Weight fraction of H2SO4  [0;1]
*     Output: Surface tension of sulfuric acid solution (N/m)
*
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      REAL   TAIR,WSA,W,T,S180,S220,S260
!
      W=DBLE(WSA)*100.0D0
      T=DBLE(TAIR)
      IF(W.LT.40.0D0) THEN
          T=DMAX1(180.0D0,DMIN1(T,260.0D0))
          S220=(((((8.969257061D-7*W-1.145573827D-4)*W+5.415260617D-3)
     +          *W-1.050692123D-1)*W+5.312072092D-1)*W+82.01197792D0)
          IF(T.LE.220.0D0) THEN
            S180=(((((1.736789787D-6*W-1.912224154D-4)*W+7.485866933D-3)
     +          *W-1.103647657D-1)*W+9.541966318D-2)*W+85.75507114D0)
!            STSAS=REAL(1.0D-3*(S220+(5.5D0-0.025D0*T)*(S180-S220)))
            STSAS=1.0D-3*(S220+(5.5D0-0.025D0*T)*(S180-S220))
          ELSE IF(T.GT.220.0D0) THEN
            S260=(((((2.095358048D-7*W-2.384669516D-5)*W+8.87979880D-4)
     +          *W-9.682499074D-3)*W-6.9631232740D-3)*W+77.40682664D0)
!            STSAS=REAL(1.0D-3*(S260+(6.5D0-0.025D0*T)*(S220-S260)))
            STSAS=1.0D-3*(S260+(6.5D0-0.025D0*T)*(S220-S260))
          ENDIF
      ELSE
!          STSAS=1.0d-3*
!     +        REAL(142.35D0-0.96525D0*W-TAIR*(0.22954D0-0.0033948D0*W))
          STSAS=1.0d-3*
     +        142.35D0-0.96525D0*W-TAIR*(0.22954D0-0.0033948D0*W)
      ENDIF

      RETURN
      END
*****************************************************************************
*     REAL FUNCTION PWVSAS_GV(TAIR,WSA)                                        
      REAL FUNCTION PWVSAS_GV(TAIR,WSA)
*****************************************************************************
*
*     Natural logaritm of saturated water vapor pressure over plane
*     sulfuric acid solution.
*
*     Source: J.I.Gmitro & T.Vermeulen: A.I.Ch.E.J.  10,740,1964.
*             W.F.Giauque et al.: J. Amer. Chem. Soc. 82,62,1960.
*
*     The formula of Gmitro & Vermeulen for saturation pressure
*     is used:
*                 ln(p) = A ln(298/T) + B/T + C + DT
*     with values of A,B,C and D given by Gmitro & Vermeulen,
*     and calculated from partial molal properties given by Giauque et al.
*     
*     
*
*     Input:  TAIR: Temperature (K)
*             WSA:  Weight fraction of H2SO4  [0;1] 
*     Output: Natural logaritm of water vapor pressure 
*             over sulfuric acid solution   ( ln(Pa) )
*
*
*     External functions needed for calculation of partial molal 
*     properties of pure components at 25 ! as function of W.
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      REAL  CPH2O,ALH2O,FFH2O,LH2O
*     CPH2O:  Partial molal heat capacity of sulfuric acid solution.
*     ALH2O:  Temparature derivative of CPH2O
*     FFH2O:  Partial molal free energy of sulfuric acid solution.
*     LH2O:   Partial molal enthalpy of sulfuric acid
*
!
!
      REAL   TAIR,WSA
      REAL   ADOT,BDOT,CDOT,DDOT
      REAL   RGAS,MMHGPA
      REAL   K1,K2
      REAL   A,B,C,D,CP,L,F,ALFA
!     Physical constants given by Gmitro & Vermeulen:
      PARAMETER(
     +        ADOT=-3.67340,
     +        BDOT=-4143.5,
     +        CDOT=10.24353,
     +        DDOT=0.618943d-3)
      PARAMETER(
!     Gas constant (cal/(deg mole)):
     +        RGAS=1.98726,
!     Natural logarith of conversion factor between atm. and Pa:     
     +     MMHGPA=11.52608845, 
     +     K1=298.15,
     +     K2=K1*K1/2.0)
!
!
      CP=CPH2O(WSA)
      F=-FFH2O(WSA)
      L=-LH2O(WSA)
      ALFA=ALH2O(WSA)
!
      A=ADOT+(CP-K1*ALFA)/RGAS
      B=BDOT+(L-K1*CP+K2*ALFA)/RGAS
      C=CDOT+(CP+(F-L)/K1)/RGAS
      D=DDOT-ALFA/(2.0d0*RGAS)
!
!     WRITE(*,*) 'TAIR= ',TAIR,'  WSA= ',WSA
!     WRITE(*,*) 'CPH2O(WSA)= ',CP
!     WRITE(*,*) 'ALFAH2O(WSA)= ',ALFA
!     WRITE(*,*) 'FFH2O(WSA)= ',F
!     WRITE(*,*) 'LH2O(WSA)= ',L
!
      PWVSAS_GV=A*DLOG(K1/TAIR)+B/TAIR+C+D*TAIR+MMHGPA
      RETURN
      END
*******************************************************************************
*     REAL FUNCTION CPH2O(W)
      REAL FUNCTION CPH2O(W)
*******************************************************************************
*
*     Relative partial molal heat capacity of water (cal/(deg mole) in 
*     sulfuric acid solution, as a function of H2SO4 weight fraction [0;1],
*     calculated by cubic spline fitting.
*
*     Source: Giauque et al.: J. Amer. Chem. Soc. 82,62,1960.
*
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER NPOINT,I
      PARAMETER(NPOINT=109)
      REAL   W,WTAB(NPOINT),CPHTAB(NPOINT),
     +              Y2(NPOINT),YWORK(NPOINT),CPH
      LOGICAL FIRST
      DATA (WTAB(I),I=1,NPOINT)/
     +0.00000,0.08932,0.09819,0.10792,0.11980,0.13461,0.15360,0.16525,
     +0.17882,0.19482,0.21397,0.23728,0.26629,0.27999,0.29517,0.31209,
     +0.33107,0.35251,0.36430,0.37691,0.39043,0.40495,0.42059,0.43749,
     +0.44646,0.45580,0.46555,0.47572,0.48634,0.49745,0.50908,0.52126,
     +0.53405,0.54747,0.56159,0.57646,0.58263,0.58893,0.59537,0.60195,
     +0.60868,0.61557,0.62261,0.62981,0.63718,0.64472,0.65245,0.66037,
     +0.66847,0.67678,0.68530,0.69404,0.70300,0.71220,0.72164,0.73133,
     +0.73628,0.74129,0.74637,0.75152,0.75675,0.76204,0.76741,0.77286,
     +0.77839,0.78399,0.78968,0.79545,0.80130,0.80724,0.81327,0.81939,
     +0.82560,0.83191,0.83832,0.84482,0.85143,0.85814,0.86495,0.87188,
     +0.87892,0.88607,0.89334,0.90073,0.90824,0.91588,0.92365,0.93156,
     +0.93959,0.94777,0.95610,0.96457,0.97319,0.98196,0.99090,0.99270,
     +0.99452,0.99634,0.99725,0.99817,0.99835,0.99853,0.99872,0.99890,
     +0.99908,0.99927,0.99945,0.99963,0.99982/
      DATA (CPHTAB(I),I=1,NPOINT)/
     + 17.996, 17.896, 17.875, 17.858, 17.840, 17.820, 17.800, 17.791,
     + 17.783, 17.777, 17.771, 17.769, 17.806, 17.891, 18.057, 18.248,
     + 18.429, 18.567, 18.613, 18.640, 18.660, 18.660, 18.642, 18.592,
     + 18.544, 18.468, 18.348, 18.187, 17.995, 17.782, 17.562, 17.352,
     + 17.162, 16.993, 16.829, 16.657, 16.581, 16.497, 16.405, 16.302,
     + 16.186, 16.053, 15.901, 15.730, 15.540, 15.329, 15.101, 14.853,
     + 14.586, 14.296, 13.980, 13.638, 13.274, 12.896, 12.507, 12.111,
     + 11.911, 11.711, 11.514, 11.320, 11.130, 10.940, 10.760, 10.570,
     + 10.390, 10.200, 10.000, 9.8400, 9.7600, 9.7900, 9.9500, 10.310,
     + 10.950, 11.960, 13.370, 15.060, 16.860, 18.550, 20.000, 21.170,
     + 22.030, 22.570, 22.800, 22.750, 22.420, 21.850, 21.120, 20.280,
     + 19.360, 18.350, 17.220, 15.940, 14.490, 12.840, 10.800, 9.8000,
     + 7.8000, 3.8000,0.20000,-5.4000,-7.0000,-8.8000,-10.900,-13.500,
     +-17.000,-22.000,-29.000,-40.000,-59.000/
      DATA FIRST/.TRUE./
      SAVE FIRST,WTAB,CPHTAB,Y2
!
      IF(FIRST) THEN
          FIRST=.FALSE.
          CALL SPLINE(WTAB,CPHTAB,NPOINT,YWORK,Y2)
      ENDIF
      CALL SPLINT(WTAB,CPHTAB,Y2,NPOINT,W,CPH)
      CPH2O=CPH
      RETURN
      END
!
*******************************************************************************
      REAL  FUNCTION FFH2O(W)
*     REAL FUNCTION FFH2O(W)
*******************************************************************************
*
*     Relative partial molal free energy water (cal/mole) in 
*     sulfuric acid solution, as a function of H2SO4 weight fraction [0;1],
*     calculated by cubic spline fitting.
*
*     Source: Giauque et al.: J. Amer. Chem. Soc. 82,62,1960.
*
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER NPOINT,I
      PARAMETER(NPOINT=110)
      REAL   W,WTAB(NPOINT),FFTAB(NPOINT),
     +              Y2(NPOINT),YWORK(NPOINT),FF
      LOGICAL FIRST
      DATA (WTAB(I),I=1,NPOINT)/
     +0.00000,0.08932,0.09819,0.10792,0.11980,0.13461,0.15360,0.16525,
     +0.17882,0.19482,0.21397,0.23728,0.26629,0.27999,0.29517,0.31209,
     +0.33107,0.35251,0.36430,0.37691,0.39043,0.40495,0.42059,0.43749,
     +0.44646,0.45580,0.46555,0.47572,0.48634,0.49745,0.50908,0.52126,
     +0.53405,0.54747,0.56159,0.57646,0.58263,0.58893,0.59537,0.60195,
     +0.60868,0.61557,0.62261,0.62981,0.63718,0.64472,0.65245,0.66037,
     +0.66847,0.67678,0.68530,0.69404,0.70300,0.71220,0.72164,0.73133,
     +0.73628,0.74129,0.74637,0.75152,0.75675,0.76204,0.76741,0.77286,
     +0.77839,0.78399,0.78968,0.79545,0.80130,0.80724,0.81327,0.81939,
     +0.82560,0.83191,0.83832,0.84482,0.85143,0.85814,0.86495,0.87188,
     +0.87892,0.88607,0.89334,0.90073,0.90824,0.91588,0.92365,0.93156,
     +0.93959,0.94777,0.95610,0.96457,0.97319,0.98196,0.99090,0.99270,
     +0.99452,0.99634,0.99725,0.99817,0.99835,0.99853,0.99872,0.99890,
     +0.99908,0.99927,0.99945,0.99963,0.99982, 1.0000/
      DATA (FFTAB(I),I=1,NPOINT)/
     +0.00000, 22.840, 25.810, 29.250, 33.790, 39.970, 48.690, 54.560,
     + 61.990, 71.790, 85.040, 103.70, 130.70, 145.20, 163.00, 184.50,
     + 211.50, 245.60, 266.40, 290.10, 317.40, 349.00, 385.60, 428.40,
     + 452.50, 478.80, 507.50, 538.80, 573.30, 611.60, 653.70, 700.50,
     + 752.60, 810.60, 875.60, 948.60, 980.60, 1014.3, 1049.7, 1087.1,
     + 1126.7, 1168.7, 1213.5, 1261.2, 1312.0, 1366.2, 1424.3, 1486.0,
     + 1551.8, 1622.3, 1697.8, 1778.5, 1864.9, 1956.8, 2055.8, 2162.0,
     + 2218.0, 2276.0, 2337.0, 2400.0, 2466.0, 2535.0, 2607.0, 2682.0,
     + 2760.0, 2842.0, 2928.0, 3018.0, 3111.0, 3209.0, 3311.0, 3417.0,
     + 3527.0, 3640.0, 3757.0, 3878.0, 4002.0, 4130.0, 4262.0, 4397.0,
     + 4535.0, 4678.0, 4824.0, 4973.0, 5128.0, 5287.0, 5454.0, 5630.0,
     + 5820.0, 6031.0, 6268.0, 6541.0, 6873.0, 7318.0, 8054.0, 8284.0,
     + 8579.0, 8997.0, 9295.0, 9720.0, 9831.0, 9954.0, 10092., 10248.,
     + 10423., 10618., 10838., 11099., 11460., 12014./
      DATA FIRST/.TRUE./
      SAVE FIRST,WTAB,FFTAB,Y2
!
      IF(FIRST) THEN
          FIRST=.FALSE.
          CALL SPLINE(WTAB,FFTAB,NPOINT,YWORK,Y2)
      ENDIF
      CALL SPLINT(WTAB,FFTAB,Y2,NPOINT,W,FF)
      FFH2O=FF
      RETURN
      END
!
*******************************************************************************
      REAL FUNCTION LH2O(W)
*     REAL FUNCTION LH2O(W)
*******************************************************************************
*
*     Relative partial molal heat content of water (cal/mole) in 
*     sulfuric acid solution, as a function of H2SO4 weight fraction [0;1],
*     calculated by cubic spline fitting.
*
*     Source: Giauque et al.: J. Amer. Chem. Soc. 82,62,1960.
*
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER NPOINT,I
      PARAMETER(NPOINT=110)
      REAL   W,WTAB(NPOINT),LTAB(NPOINT),
     +              Y2(NPOINT),YWORK(NPOINT),L
      LOGICAL FIRST
      DATA (WTAB(I),I=1,NPOINT)/
     +0.00000,0.08932,0.09819,0.10792,0.11980,0.13461,0.15360,0.16525,
     +0.17882,0.19482,0.21397,0.23728,0.26629,0.27999,0.29517,0.31209,
     +0.33107,0.35251,0.36430,0.37691,0.39043,0.40495,0.42059,0.43749,
     +0.44646,0.45580,0.46555,0.47572,0.48634,0.49745,0.50908,0.52126,
     +0.53405,0.54747,0.56159,0.57646,0.58263,0.58893,0.59537,0.60195,
     +0.60868,0.61557,0.62261,0.62981,0.63718,0.64472,0.65245,0.66037,
     +0.66847,0.67678,0.68530,0.69404,0.70300,0.71220,0.72164,0.73133,
     +0.73628,0.74129,0.74637,0.75152,0.75675,0.76204,0.76741,0.77286,
     +0.77839,0.78399,0.78968,0.79545,0.80130,0.80724,0.81327,0.81939,
     +0.82560,0.83191,0.83832,0.84482,0.85143,0.85814,0.86495,0.87188,
     +0.87892,0.88607,0.89334,0.90073,0.90824,0.91588,0.92365,0.93156,
     +0.93959,0.94777,0.95610,0.96457,0.97319,0.98196,0.99090,0.99270,
     +0.99452,0.99634,0.99725,0.99817,0.99835,0.99853,0.99872,0.99890,
     +0.99908,0.99927,0.99945,0.99963,0.99982, 1.0000/
      DATA (LTAB(I),I=1,NPOINT)/
     +0.00000, 5.2900, 6.1000, 7.1800, 8.7800, 11.210, 15.290, 18.680,
     + 23.700, 31.180, 42.500, 59.900, 89.200, 106.70, 128.60, 156.00,
     + 190.40, 233.80, 260.10, 290.00, 324.00, 362.50, 406.50, 456.10,
     + 483.20, 512.40, 543.60, 577.40, 613.80, 653.50, 696.70, 744.50,
     + 797.20, 855.80, 921.70, 995.70, 1028.1, 1062.3, 1098.3, 1136.4,
     + 1176.7, 1219.3, 1264.7, 1313.0, 1364.3, 1418.9, 1477.3, 1539.9,
     + 1607.2, 1679.7, 1757.9, 1842.7, 1934.8, 2035.4, 2145.5, 2267.0,
     + 2332.0, 2401.0, 2473.0, 2550.0, 2631.0, 2716.0, 2807.0, 2904.0,
     + 3007.0, 3118.0, 3238.0, 3367.0, 3507.0, 3657.0, 3821.0, 3997.0,
     + 4186.0, 4387.0, 4599.0, 4819.0, 5039.0, 5258.0, 5476.0, 5694.0,
     + 5906.0, 6103.0, 6275.0, 6434.0, 6592.0, 6743.0, 6880.0, 7008.0,
     + 7133.0, 7255.0, 7376.0, 7497.0, 7618.0, 7739.0, 7855.0, 7876.0,
     + 7905.0, 7985.0, 8110.0, 8415.0, 8515.0, 8655.0, 8835.0, 9125.0,
     + 9575.0, 10325., 11575., 13500., 15200., 16125./
      DATA FIRST/.TRUE./
      SAVE FIRST,WTAB,LTAB,Y2
!
      IF(FIRST) THEN
          FIRST=.FALSE.
          CALL SPLINE(WTAB,LTAB,NPOINT,YWORK,Y2)
      ENDIF
      CALL SPLINT(WTAB,LTAB,Y2,NPOINT,W,L)
      LH2O=L
      RETURN
      END
*******************************************************************************
      REAL FUNCTION ALH2O(W)
*     REAL FUNCTION ALH2O(W)
*******************************************************************************
*
*     Relative partial molal temperature derivative of heat capacity (water) 
*     in sulfuric acid solution, (cal/deg**2), calculated by 
*     cubic spline fitting.
*
*     Source: Giauque et al.: J. Amer. Chem. Soc. 82,62,1960.
*
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER NPOINT,I
      PARAMETER(NPOINT=96)
      REAL   W,WTAB(NPOINT),ATAB(NPOINT),
     +              Y2(NPOINT),YWORK(NPOINT),A
      LOGICAL FIRST
      DATA (WTAB(I),I=1,NPOINT)/
     +0.29517,0.31209,
     +0.33107,0.35251,0.36430,0.37691,0.39043,0.40495,0.42059,0.43749,
     +0.44646,0.45580,0.46555,0.47572,0.48634,0.49745,0.50908,0.52126,
     +0.53405,0.54747,0.56159,0.57646,0.58263,0.58893,0.59537,0.60195,
     +0.60868,0.61557,0.62261,0.62981,0.63718,0.64472,0.65245,0.66037,
     +0.66847,0.67678,0.68530,0.69404,0.70300,0.71220,0.72164,0.73133,
     +0.73628,0.74129,0.74637,0.75152,0.75675,0.76204,0.76741,0.77286,
     +0.77839,0.78399,0.78968,0.79545,0.80130,0.80724,0.81327,0.81939,
     +0.82560,0.83191,0.83832,0.84482,0.85143,0.85814,0.86495,0.87188,
     +0.87892,0.88607,0.89334,0.90073,0.90824,0.91588,0.92365,0.93156,
     +0.93959,0.94777,0.95610,0.96457,0.97319,0.98196,0.99090,0.99270,
     +0.99452,0.99634,0.99725,0.99817,0.99835,0.99853,0.99872,0.99890,
     +0.99908,0.99927,0.99945,0.99963,0.99982, 1.0000/
      DATA (ATAB(I),I=1,NPOINT)/
     + 0.0190, 0.0182, 0.0180, 0.0177, 0.0174, 0.0169, 0.0167, 0.0164,
     + 0.0172, 0.0212, 0.0239, 0.0264, 0.0276, 0.0273, 0.0259, 0.0238,
     + 0.0213, 0.0190, 0.0170, 0.0155, 0.0143, 0.0133, 0.0129, 0.0124,
     + 0.0120, 0.0114, 0.0106, 0.0097, 0.0084, 0.0067, 0.0047, 0.0024,
     +-0.0002,-0.0031,-0.0063,-0.0097,-0.0136,-0.0178,-0.0221,-0.0263,
     +-0.0303,-0.0340,-0.0352,-0.0360,-0.0362,-0.0356,-0.0343,-0.0321,
     +-0.0290,-0.0251,-0.0201,-0.0137,-0.0058, 0.0033, 0.0136, 0.0254,
     + 0.0388, 0.0550, 0.0738, 0.0962, 0.1198, 0.1300, 0.1208, 0.0790,
     + 0.0348, 0.0058,-0.0102,-0.0211,-0.0292,-0.0350,-0.0390,-0.0418,
     +-0.0432,-0.0436,-0.0429,-0.0411,-0.0384,-0.0346,-0.0292,-0.0220,
     +-0.0130,-0.0110,-0.0080,-0.0060,-0.0040,-0.0030,-0.0030,-0.0020,
     +-0.0020,-0.0020,-0.0020,-0.0010,-0.0010, 0.0000, 0.0000, 0.0000/
      DATA FIRST/.TRUE./
      SAVE FIRST,WTAB,ATAB,Y2
!
      IF(FIRST) THEN
          FIRST=.FALSE.
          CALL SPLINE(WTAB,ATAB,NPOINT,YWORK,Y2)
      ENDIF
      CALL SPLINT(WTAB,ATAB,Y2,NPOINT,MAX(WTAB(1),W),A)
      ALH2O=A
      RETURN
      END
!******************************************************************************
      SUBROUTINE SPLINE(X,Y,N,WORK,Y2)
!******************************************************************************
!     Routine to calculate 2.nd derivatives of tabulated function
!     Y(i)=Y(Xi), to be used for cubic spline calculation.
!
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER N,I
      REAL   X(N),Y(N),WORK(N),Y2(N)
      REAL   SIG,P,QN,UN,YP1,YPN

!AM Venus: Let's check the values
!      write(*,*) 'In spline, N ', N

      YP1=(Y(2)-Y(1))/(X(2)-X(1))
      YPN=(Y(N)-Y(N-1))/(X(N)-X(N-1))
      IF(YP1.GT.99.0E+30) THEN
          Y2(1)=0.0
          WORK(1)=0.0
      ELSE
          Y2(1)=-0.5d0
          WORK(1)=(3.0d0/(X(2)-X(1)))*((Y(2)-Y(1))/(X(2)-X(1))-YP1)
      ENDIF
      DO I=2,N-1
!         write(*,*) 'In spline, I ', I
          SIG=(X(I)-X(I-1))/(X(I+1)-X(I-1))
          P=SIG*Y2(I-1)+2.0d0
          Y2(I)=(SIG-1.0d0)/P
          WORK(I)=(6.0d0*((Y(I+1)-Y(I))/(X(I+1)-X(I))-(Y(I)-Y(I-1))
     +             /(X(I)-X(I-1)))/(X(I+1)-X(I-1))-SIG*WORK(I-1))/P
      ENDDO
      IF(YPN.GT.99.0E+30) THEN
          QN=0.0
          UN=0.0
      ELSE
          QN=0.5d0
          UN=(3.0d0/(X(N)-X(N-1)))*(YPN-(Y(N)-Y(N-1))/(X(N)-X(N-1)))
      ENDIF
      Y2(N)=(UN-QN*WORK(N-1))/(QN*Y2(N-1)+1.0d0)
      DO I=N-1,1,-1
!         write(*,*) 'In spline, I ', I
          Y2(I)=Y2(I)*Y2(I+1)+WORK(I)
      ENDDO
!
      RETURN
      END

!******************************************************************************
      SUBROUTINE SPLINT(XA,YA,Y2A,N,X,Y)
!******************************************************************************
!     Cubic spline calculation
!
!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

      INTEGER KLO,KHI,N,K
      REAL   XA(N),YA(N),Y2A(N)
      REAL   X,Y,H,A,B
!
      KLO=1
      KHI=N
 1    IF(KHI-KLO.GT.1) THEN
          K=(KHI+KLO)/2
          IF(XA(K).GT.X) THEN
              KHI=K
          ELSE
              KLO=K
          ENDIF
          GOTO 1
      ENDIF
      H=XA(KHI)-XA(KLO)
      A=(XA(KHI)-X)/H
      B=(X-XA(KLO))/H
      Y=A*YA(KLO)+B*YA(KHI)+
     +        ((A**3-A)*Y2A(KLO)+(B**3-B)*Y2A(KHI))*(H**2)/6.0d0
!
      RETURN
      END
!******************************************************************
      SUBROUTINE CALNLOG_SAT(ACTSO4,H2SO4,H2O,WFSA,DENSO4,RSTDEV,
     + RMEDRA,T,RNLOG,H2SO4COND,H2OCOND,
     + acidps,satpacid)

!     DERIVE NO (TOTAL NUMBER OF AEROSOL PARTICLES CONCENTRATION) 
!     FROM TOTAL H2SO4 AND RMOD/SIGMA OF AEROSOL LOG-NORMAL 
!                                       SIZE DISTRIBTUION
!     ASSUMING ALL THE H2SO4 ABOVE MIXTURE SAT PRESSURE IS CONDENSED
!    ---------------------------------------------------------------
!     INPUT:
!     ACTSO4: H2SO4 activity 
!     H2SO4: #/cm3 of total H2SO4
!	H2O  : #/cm3 of total H2O
!     WFSA: aerosol H2SO4 weight fraction (fraction)
!     DENSO4: aerosol volumic mass (gr/cm3 = aerosol mass/aerosol volume)
!!       for total mass, almost same result with ro=1.67 gr/cm3
!     RSTDEV: standard deviation of aerosol distribution (no unit)
!     RMEDRA: median radius (m)
!     RMEDR : median radius converti en cm
!     T: temperature (K)
!
!     OUTPUT: 
!     RNLOG: total number of aerosol particles (VMR)
!     RNLOG is in the same units as H2SO4
!     if H2SO4 is in number density (for example, molec/cm3), 
!          RNLOG (number of particles/cm3), etc...
!	H2OCOND
!	H2SO4COND

!      USE real16
      IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

!      INTEGER NOLAT,NOLEV
      REAL  H2SO4, H2O, WFSA, ACTSO4
      REAL  H2OCOND, H2SO4COND
      REAL  RSTDEV, RMEDRA
      REAL  DENSO4,  T
      REAL  RNLOG,RMEDR
!     working variables
      INTEGER I,J
      REAL  RMH2S4,PI1,RMTOT
      REAL  DND2,pstand,lpar,acidps
      REAL  x1, satpacid
      REAL , DIMENSION(2):: act
!
!     masse of an H2SO4 molecule (g)
      RMH2S4=98.078/(6.02214129e23)
      
!     3/4*PI
      PI1    =3./(4.0*4.0*ATAN(1.0))

      RMEDR=RMEDRA*1.e2 !AM: this needs to be in cm!
      
      pstand=1.01325e5 !Pa  1 atm pressure

        x1=(WFSA/98.08)/(WFSA/98.08 + ((1.-WFSA)/18.0153))

        call zeleznik(x1,t,act)

!pure acid satur vapor pressure
        lpar= -11.695+DLOG(pstand) ! Zeleznik
        acidps=1/360.15-1.0/t+0.38/545.
     + *(1.0+DLOG(360.15/t)-360.15/t)
        acidps = 10156.0*acidps +lpar
        acidps = DEXP(acidps)    !Pa

!acid sat.vap.pres over mixture (flat surface):
        satpacid=act(2)*acidps ! Pa 

!       Conversion from Pa to N.D #/m3
        DND2=satpacid/(1.3806488e-23*T)
!       Conversion from N.D #/m3 TO #/cm3
        DND2=DND2*1.d-6
	        
!	H2SO4COND N.D #/cm3 condensee ssi H2SO4>H2SO4sat
	IF (H2SO4.GE.DND2) THEN
	H2SO4COND=H2SO4-DND2
!	calcul de H2O cond correspondant a H2SO4 cond
	H2OCOND=H2SO4COND*98.078*(1.0-WFSA)/(18.0153*WFSA)

!     RMTOT: = Mass of H2SO4 satur per cm3 of air/ Mass of sulfuric acid part of droplet solution per cm3 
!	RMTOT=M_distrib/rho_droplet
	
	RMTOT=H2SO4COND*RMH2S4/(DENSO4*WFSA)

!
!       RNLOG: total number of aerosol particles per cm3
        RNLOG= RMTOT*EXP(-4.5*DLOG(RSTDEV)*DLOG(RSTDEV)) 
     +    *PI1/( RMEDR*RMEDR*RMEDR)

!	Si on a H2SO4<H2SO4sat on ne condense rien et NDTOT=0
	ELSE
	H2SO4COND=0.0d0
	H2OCOND=0.0d0
	RNLOG=0.0d0
	END IF
	     	
!	Test si H2O en defaut H2Ocond>H2O dispo
	IF (H2OCOND.GT.H2O) THEN
	
!	On peut alors condenser tout le H2O dispo
	H2OCOND=H2O
!	On met alors egalement a jour le H2SO4 cond correspondant au H2O cond
	H2SO4COND=H2OCOND*18.0153*WFSA/(98.078*(1.0-WFSA))
	
!     RMTOT: = Mass of H2SO4 satur per cm3 of air/ Mass of sulfuric acid part of droplet solution per cm3 
!	RMTOT=Volume of aerosol cm3 /cm3 of air
!	Volume of aerosol/cm3 air
	
	RMTOT=H2SO4COND*RMH2S4/(DENSO4*WFSA)

!
!       RNLOG: total number of aerosol particles per cm3
        RNLOG= RMTOT*EXP(-4.5*DLOG(RSTDEV)*DLOG(RSTDEV)) 
     +    *PI1/( RMEDR*RMEDR*RMEDR)
      
      END IF
     	

		
      RETURN
      END
      
!****************************************************************
      SUBROUTINE STRAACT(R2SO4,ACTSO4)

!     H2SO4 ACTIVITY (GIAUQUE) AS A FUNCTION OF H2SO4 WP
!    ----------------------------------------
!     INPUT:
!	R2SO4: percent (%) of WSA (Weight fraction of Sulfuric Acid) 
!
!     OUTPUT: 
!     ACTSO4: H2SO4 activity (percent)
!       USE real16
       IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

	   
!       INTEGER NOLAT,NOLEV
       REAL  R2SO4,     ACTSO4
	  
!      Working variables	  	   
       INTEGER NN,I,J,JX,JX1
       REAL  TC,TB,TA,XT
       PARAMETER (NN=109)
       REAL, DIMENSION(NN) :: XC, X

!      H2SO4 activity 
       DATA X/
     *   0.0,0.25,0.78,1.437,2.19,3.07,4.03,5.04,6.08
     *  ,7.13,8.18,14.33,18.59,28.59,39.17,49.49
     *  ,102.4,157.8,215.7,276.9,341.6,409.8,481.5,556.6
     *  ,635.5,719.,808.,902.,1000.,1103.,1211.,1322.,1437.,1555.
     *  ,1677.,1800.,1926.,2054.,2183.,2312.,2442.,2572.,2701.,2829.
     *  ,2955.,3080.,3203.,3325.,3446.,3564.,3681.,3796.,3910.,4022.
     *  ,4134.,4351.,4564.,4771.,4974.,5171.,5364.,5551.,5732.,5908.
     *  ,6079.,6244.,6404.,6559.,6709.,6854.,6994.,7131.,7264.,7393.
     *  ,7520.,7821.,8105.,8373.,8627.,8867.,9093.,9308.,9511.,9703.
     *  ,9885.,10060.,10225.,10535.,10819.,11079.,11318.,11537.
     *  ,11740.,12097.,12407.,12676.,12915.,13126.,13564.,13910.
     *  ,14191.,14423.,14617.,14786.,10568.,15299.,15491.,15654.
     *  ,15811./
!      H2SO4 weight fraction (percent)
       DATA XC/
     *   100.0,99.982,99.963,99.945,99.927,99.908,99.890,99.872
     *  ,99.853,99.835,99.817,99.725,99.634,99.452,99.270
     *  ,99.090,98.196,97.319,96.457,95.610,94.777,93.959,93.156
     *  ,92.365,91.588,90.824,90.073,89.334,88.607,87.892,87.188
     *  ,86.495,85.814,85.143,84.482,83.832,83.191,82.560,81.939
     *  ,81.327,80.724,80.130,79.545,78.968,78.399,77.839,77.286
     *  ,76.741,76.204,75.675,75.152,74.637,74.129,73.628,73.133
     *  ,72.164,71.220,70.300,69.404,68.530,67.678,66.847,66.037
     *  ,65.245,64.472,63.718,62.981,62.261,61.557,60.868,60.195
     *  ,59.537,58.893,58.263,57.646,56.159,54.747,53.405,52.126
     *  ,50.908,49.745,48.634,47.572,46.555,45.580,44.646,43.749
     *  ,42.059,40.495,39.043,37.691,36.430,35.251,33.107,31.209
     *  ,29.517,27.999,26.629,23.728,21.397,19.482,17.882,16.525
     *  ,15.360,13.461,11.980,10.792,9.819,8.932/


!     HERE LINEAR INTERPOLATIONS
        XT=R2SO4
        CALL POSACT(XT,XC,NN,JX)
        JX1=JX+1
        IF(JX.EQ.0) THEN
          ACTSO4=0.0 
        ELSE IF(JX.GE.NN) THEN
          ACTSO4=15811.0 
        ELSE 
          TC=XT-XC(JX)
          TB=X(JX1)-X(JX)
          TA=XC(JX1)-XC(JX)
          TA=TB/TA
          ACTSO4=X(JX)+(TA*TC)
        ENDIF
10    CONTINUE

      RETURN
      END
!********************************************************************
       SUBROUTINE POSACT(XT,X_ARR,N,JX)
	   
!      POSITION OF XT IN THE ARRAY X
!    ---------------------------------------------
!       USE real16
       IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       INTEGER N
       REAL  XT
       REAL, DIMENSION(N) :: X_ARR
!      Working variables	  	   
       INTEGER JX,I
	   
       IF(XT.GT.X_ARR(1)) THEN
         JX=0
       ELSE
         DO 10 I=1,N
           IF (XT.GT.X_ARR(I)) GO TO 20
 10      CONTINUE
 20      JX=I
       ENDIF
	   
       RETURN
       END

       SUBROUTINE Zeleznik(x,T,act)

  !+++++++++++++++++++++++++++++++++++++++++++++++++++
  !     Water and sulfuric acid activities in liquid
  !     aqueous solutions.
  !     Frank J. Zeleznik, Thermodynnamic properties
  !     of the aqueous sulfuric acid system to 220K-350K,
  !     mole fraction 0,...,1
  !     J. Phys. Chem. Ref. Data, Vol. 20, No. 6,pp.1157, 1991
  !+++++++++++++++++++++++++++++++++++++++++++++++++++ 

!         USE real16
         IMPLICIT NONE
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

         REAL  x,T, activitya, activityw
         REAL , DIMENSION(2):: act
!         REAL x,T, activitya, activityw
!         REAL, DIMENSION(2):: act


!         write(*,*) 'x, T ', x, T

         act(2)=activitya(x,T) 
         act(1)=activityw(x,T)

!         write(*,*) 'act ', act

       END SUBROUTINE Zeleznik

!start of functions related to zeleznik activities

       FUNCTION m111(T)
!       USE real16       
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  T,m111
       m111=-23.524503387D0 
     &    +0.0406889449841D0*T 
     &    -0.151369362907D-4*T**2+2961.44445015D0/T 
     &    +0.492476973663D0*dlog(T)
       END FUNCTION m111

       FUNCTION m121(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  m121,T
       m121=1114.58541077D0-1.1833078936D0*T 
     &    -0.00209946114412D0*T**2-246749.842271D0/T 
     &    +34.1234558134D0*dlog(T)
       END FUNCTION m121

       FUNCTION m221(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  m221,T
       m221=-80.1488100747D0-0.0116246143257D0*T 
     &    +0.606767928954D-5*T**2+3092.72150882D0/T 
     &    +12.7601667471D0*dlog(T)
       END FUNCTION m221

       FUNCTION m122(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  m122,T
       m122=888.711613784D0-2.50531359687D0*T 
     &    +0.000605638824061D0*T**2-196985.296431D0/T 
     &    +74.550064338D0*dlog(T)
       END FUNCTION m122

       FUNCTION e111(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e111,T
       e111=2887.31663295D0-3.32602457749D0*T 
     &    -0.2820472833D-2*T**2-528216.112353D0/T 
     &    +0.68699743564D0*dlog(T)
       END FUNCTION e111

       FUNCTION e121(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e121,T
       e121=-370.944593249D0-0.690310834523D0*T 
     &    +0.56345508422D-3*T**2-3822.52997064D0/T 
     &    +94.2682037574D0*dlog(T)
       END FUNCTION e121

       FUNCTION e211(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e211,T
       e211=38.3025318809D0-0.0295997878789D0*T 
     &    +0.120999746782D-4*T**2-3246.97498999D0/T 
     &    -3.83566039532D0*dlog(T)
       END FUNCTION e211

       FUNCTION e221(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e221,T
       e221=2324.76399402D0-0.141626921317D0*T 
     &    -0.00626760562881D0*T**2-450590.687961D0/T 
     &    -61.2339472744D0*dlog(T)
       END FUNCTION e221

       FUNCTION e122(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e122,T
       e122=-1633.85547832D0-3.35344369968D0*T 
     &    +0.00710978119903D0*T**2+198200.003569D0/T 
     &    +246.693619189D0*dlog(T)
       END FUNCTION e122

       FUNCTION e212(T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  e212,T
       e212=1273.75159848D0+1.03333898148D0*T 
     &    +0.00341400487633D0*T**2+195290.667051D0/T 
     &    -431.737442782D0*dlog(T)
       END FUNCTION e212

       FUNCTION lnAa(x1,T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  lnAa,T,x1 
     &          ,m111,m121,m221,m122 
     &            ,e111,e121,e211,e122,e212,e221
       lnAa=-( 
     &    (2*m111(T)+e111(T)*(2*dlog(x1)+1))*x1 
     &    +(2*m121(T)+e211(T)*dlog(1-x1)+e121(T)*(dlog(x1)+1))*(1-x1) 
     &    -(m111(T)+e111(T)*(dlog(x1)+1))*x1*x1 
     &    -(2*m121(T)+e121(T)*(dlog(x1)+1)+e211(T)*(dlog(1-x1)+1) 
     &    -(2*m122(T)+e122(T)*dlog(x1) 
     &           +e212(T)*dlog(1-x1))*(1-x1))*x1*(1-x1) 
     &    -(m221(T)+e221(T)*(dlog(1-x1)+1))*(1-x1)**2 
     &    -x1*(1-x1)*( 
     &                  (6*m122(T)+e122(T)*(3*dlog(x1)+1) 
     &                          +e212(T)*(3*dlog(1-x1)+1) 
     &                   )*x1*(1-x1) 
     &                -(2*m122(T)+e122(T)*(dlog(x1)+1) 
     &                                   +e212(T)*dlog(1-x1) 
     &                    )*(1-x1)) 
     &     )
       END FUNCTION lnAa

       FUNCTION lnAw(x1,T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  lnAw,T,x1 
     &          ,m111,m121,m221,m122 
     &            ,e111,e121,e211,e122,e212,e221
       lnAw=-( 
     &  (2*m121(T)+e121(T)*dlog(x1)+e211(T)*(dlog(1-x1)+1))*x1 
     &  +(2*m221(T)+e221(T)*(2*dlog(1-x1)+1))*(1-x1) 
     &  -(m111(T)+e111(T)*(dlog(x1)+1))*x1*x1 
     & -(2*m121(T)+e121(T)*(dlog(x1)+1) 
     &            +e211(T)*(dlog(1-x1)+1))*x1*(1-x1) 
     &        -(m221(T)+e221(T)*(dlog(1-x1)+1))*(1-x1)**2 
     &   +x1*(2*m122(T)+e122(T)*dlog(x1)+e212(T)*dlog(1-x1))*x1*(1-x1) 
     &  +x1*(1-x1)*((2*m122(T)+e122(T)*dlog(x1) 
     &                        +e212(T)*(dlog(1-x1)+1))*x1 
     &               -(6*m122(T)+e122(T)*(3*dlog(x1)+1) 
     &                        +e212(T)*(3*dlog(1-x1)+1))*(1-x1)*x1) 
     &     )
       END FUNCTION lnAw

       FUNCTION activitya(xal,T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  lnAa,T,xal,activitya 
!       &          ,m111,m121,m221,m122 &
!       &            ,e111,e121,e211,e122,e212,e221

!       write(*,*) 'in activitya ', xal, T
       activitya=DEXP(lnAa(xal,T)-lnAa(1.D0-1.D-12,T))
       END FUNCTION activitya

       FUNCTION activityw(xal,T)
!       USE real16
!      integer, parameter :: sp_k = kind(1.0) ! Default Type of REAL
!      integer, parameter :: real_8 = selected_real_kind(2*precision
!     + (1.0_sp_k))

       REAL  lnAw,T,xal,activityw 
!       &          ,m111,m121,m221,m122 &
!       &            ,e111,e121,e211,e122,e212,e221
       activityw=DEXP(lnAw(xal,T)-lnAw(1.D-12,T))
       END FUNCTION activityw

! end of functions related to zeleznik activities
