!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! L. Fita, LMD. May 2014. Module to compute lidar retrievals (values on the vertical 
!   column) at each time-step. Based on wrf_ts.F
! This routine prints out the current value of variables at all specified
!   time series locations that are within the current patch.
!
! Michael G. Duda -- 25 August 2005
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
SUBROUTINE calc_lidar_locations( grid )

   USE module_domain, ONLY : domain, get_ijk_from_grid, domain_clock_get
   USE module_configure, ONLY : model_config_rec, grid_config_rec_type, model_to_grid_config_rec
   USE module_dm, ONLY : wrf_dm_min_real
   USE module_llxy
   USE module_state_description

   IMPLICIT NONE

   ! Arguments
   TYPE (domain), INTENT(INOUT) :: grid

   ! Externals
   LOGICAL, EXTERNAL :: wrf_dm_on_monitor
! L. Fita, LMD. May 2014
! Already defined as function by wrf_timeseries.F
   INTEGER, EXTERNAL :: get_unused_unit

   ! Local variables
   INTEGER :: nlidarloc_temp
   INTEGER :: i, k, iunit
   REAL :: lidar_rx, lidar_ry, lidar_xlat, lidar_xlong, lidar_hgt
   REAL :: known_lat, known_lon
   CHARACTER (LEN=132) :: message
   TYPE (PROJ_INFO) :: lidar_proj
   TYPE (grid_config_rec_type) :: config_flags

   INTEGER :: ids, ide, jds, jde, kds, kde,        &
              ims, ime, jms, jme, kms, kme,        &
              ips, ipe, jps, jpe, kps, kpe,        &
              imsx, imex, jmsx, jmex, kmsx, kmex,  &
              ipsx, ipex, jpsx, jpex, kpsx, kpex,  &
              imsy, imey, jmsy, jmey, kmsy, kmey,  &
              ipsy, ipey, jpsy, jpey, kpsy, kpey
   CHARACTER (LEN=50)                                    :: SimStartTime

   IF ( grid%nlidarloc .LE. 0 ) RETURN

#if ((EM_CORE == 1) && (DA_CORE != 1))
   IF ( grid%dfi_stage == DFI_FST ) THEN
#endif
      CALL get_ijk_from_grid ( grid ,                               &
                               ids, ide, jds, jde, kds, kde,        &
                               ims, ime, jms, jme, kms, kme,        &
                               ips, ipe, jps, jpe, kps, kpe,        &
                               imsx, imex, jmsx, jmex, kmsx, kmex,  &
                               ipsx, ipex, jpsx, jpex, kpsx, kpex,  &
                               imsy, imey, jmsy, jmey, kmsy, kmey,  &
                               ipsy, ipey, jpsy, jpey, kpsy, kpey )
   
      CALL model_to_grid_config_rec ( grid%id , model_config_rec , config_flags )
   
      ! Set up map transformation structure
      CALL map_init(lidar_proj)
   
      IF (ips <= 1 .AND. 1 <= ipe .AND. &
          jps <= 1 .AND. 1 <= jpe) THEN
         known_lat = grid%xlat(1,1)
         known_lon = grid%xlong(1,1)
      ELSE
         known_lat = 9999.
         known_lon = 9999.
      END IF
      known_lat = wrf_dm_min_real(known_lat)
      known_lon = wrf_dm_min_real(known_lon)
   
      ! Mercator
      IF (config_flags%map_proj == PROJ_MERC) THEN
         CALL map_set(PROJ_MERC, lidar_proj,               &
                      truelat1 = config_flags%truelat1, &
                      lat1     = known_lat,             &
                      lon1     = known_lon,             &
                      knowni   = 1.,                    &
                      knownj   = 1.,                    &
                      dx       = config_flags%dx)
   
      ! Lambert conformal
      ELSE IF (config_flags%map_proj == PROJ_LC) THEN
         CALL map_set(PROJ_LC, lidar_proj,                  &
                      truelat1 = config_flags%truelat1,  &
                      truelat2 = config_flags%truelat2,  &
                      stdlon   = config_flags%stand_lon, &
                      lat1     = known_lat,              &
                      lon1     = known_lon,              &
                      knowni   = 1.,                     &
                      knownj   = 1.,                     &
                      dx       = config_flags%dx)
   
      ! Polar stereographic
      ELSE IF (config_flags%map_proj == PROJ_PS) THEN
         CALL map_set(PROJ_PS, lidar_proj,                  &
                      truelat1 = config_flags%truelat1,  &
                      stdlon   = config_flags%stand_lon, &
                      lat1     = known_lat,              &
                      lon1     = known_lon,              &
                      knowni   = 1.,                     &
                      knownj   = 1.,                     &
                      dx       = config_flags%dx)
   
#if (EM_CORE == 1)
      ! Cassini (global ARW)
      ELSE IF (config_flags%map_proj == PROJ_CASSINI) THEN
         CALL map_set(PROJ_CASSINI, lidar_proj,                            &
                      latinc   = grid%dy*360.0/(2.0*EARTH_RADIUS_M*PI), &
                      loninc   = grid%dx*360.0/(2.0*EARTH_RADIUS_M*PI), & 
                      lat1     = known_lat,                             &
                      lon1     = known_lon,                             &
                      lat0     = config_flags%pole_lat,                 &
                      lon0     = config_flags%pole_lon,                 &
                      knowni   = 1.,                                    &
                      knownj   = 1.,                                    &
                      stdlon   = config_flags%stand_lon)
#endif

      ! Rotated latitude-longitude
      ELSE IF (config_flags%map_proj == PROJ_ROTLL) THEN
         CALL map_set(PROJ_ROTLL, lidar_proj,                      &
! I have no idea how this should work for NMM nested domains
                      ixdim    = grid%e_we-1,                   &
                      jydim    = grid%e_sn-1,                   &
                      phi      = real(grid%e_sn-2)*grid%dy/2.0, &
                      lambda   = real(grid%e_we-2)*grid%dx,     &
                      lat1     = config_flags%cen_lat,          &
                      lon1     = config_flags%cen_lon,          &
                      latinc   = grid%dy,                       &
                      loninc   = grid%dx,                       &
                      stagger  = HH)
   
      END IF
   
      ! Determine lidar locations for domain
      IF (.NOT. grid%have_calculated_lidarlocs) THEN
         grid%have_calculated_lidarlocs = .TRUE.
         WRITE(message, '(A43,I3)') 'Computing lidar locations for domain ', grid%id
         CALL wrf_message(message)
   
         nlidarloc_temp = 0
         DO k=1,grid%nlidarloc
   
            IF (config_flags%map_proj == 0) THEN  ! For idealized cases, no map transformation needed
               lidar_rx = grid%latlidarloc(k)           ! NB: (x,y) = (lat,lon) rather than (x,y) = (lon,lat)
               lidar_ry = grid%lonlidarloc(k)
            ELSE
               CALL latlon_to_ij(lidar_proj, grid%latlidarloc(k), grid%lonlidarloc(k), lidar_rx, lidar_ry)
            END IF
            

            nlidarloc_temp = nlidarloc_temp + 1
            grid%ilidarloc(nlidarloc_temp) = NINT(lidar_rx)
            grid%jlidarloc(nlidarloc_temp) = NINT(lidar_ry)
            grid%id_lidarloc(nlidarloc_temp) = k
   
            ! Is point outside of domain (or on the edge of domain)?
            IF (grid%ilidarloc(nlidarloc_temp) < ids .OR. grid%ilidarloc(nlidarloc_temp) > ide .OR. &
                grid%jlidarloc(nlidarloc_temp) < jds .OR. grid%jlidarloc(nlidarloc_temp) > jde) THEN
               nlidarloc_temp = nlidarloc_temp - 1
   
            END IF
   
         END DO
   
         grid%next_lidar_time = 1
   
         grid%nlidarloc_domain = nlidarloc_temp
   
         DO k=1,grid%nlidarloc_domain
   
            ! If location is outside of patch, we need to get lat/lon of TS grid cell from another patch
            IF (grid%ilidarloc(k) < ips .OR. grid%ilidarloc(k) > ipe .OR. &
                grid%jlidarloc(k) < jps .OR. grid%jlidarloc(k) > jpe) THEN
               lidar_xlat  = 1.E30
               lidar_xlong = 1.E30
               lidar_hgt   = 1.E30
            ELSE
               lidar_xlat  = grid%xlat(grid%ilidarloc(k),grid%jlidarloc(k))
               lidar_xlong = grid%xlong(grid%ilidarloc(k),grid%jlidarloc(k))
#if (EM_CORE == 1)
               lidar_hgt   = grid%ht(grid%ilidarloc(k),grid%jlidarloc(k))
#endif
            END IF
#if DM_PARALLEL
            lidar_xlat  = wrf_dm_min_real(lidar_xlat)
            lidar_xlong = wrf_dm_min_real(lidar_xlong)
            lidar_hgt   = wrf_dm_min_real(lidar_hgt)
#endif
   
            IF ( wrf_dm_on_monitor() ) THEN

               iunit = get_unused_unit()
               IF ( iunit <= 0 ) THEN
                  CALL wrf_error_fatal('Error in calc_lidar_locations: could not find a free Fortran unit.')
               END IF
               CALL domain_clock_get(grid, simulationStartTimeStr = SimStartTime)

               WRITE(grid%lidar_filename(k),'(A)') TRIM(grid%namelidarloc(grid%id_lidarloc(k)))//'.LIDAR.d00'
               i = LEN_TRIM(grid%lidar_filename(k))
               WRITE(grid%lidar_filename(k)(i-1:i),'(I2.2)') grid%id
               OPEN(UNIT=iunit, FILE=TRIM(grid%lidar_filename(k)), FORM='FORMATTED', STATUS='REPLACE')
#if (EM_CORE == 1)
               WRITE(UNIT=iunit, &
                     FMT='(A26,I2,I3,A6,A2,F7.3,A1,F8.3,A3,I4,A1,I4,A3,F7.3,A1,F8.3,A2,F6.1,A32,A19)') &
                     grid%desclidarloc(grid%id_lidarloc(k))//' ', grid%id, grid%id_lidarloc(k), &
                     ' '//grid%namelidarloc(grid%id_lidarloc(k)), &
                     ' (', grid%latlidarloc(grid%id_lidarloc(k)), ',', grid%lonlidarloc(grid%id_lidarloc(k)), ') (', &
                     grid%ilidarloc(k), ',', grid%jlidarloc(k), ') (', &
                     lidar_xlat, ',', lidar_xlong, ') ', &
                     lidar_hgt,' meters. simulation start time: ',TRIM(SimStartTime)
#else
               WRITE(UNIT=iunit, &
                     FMT='(A26,I2,I3,A6,A2,F7.3,A1,F8.3,A3,I4,A1,I4,A3,F7.3,A1,F8.3,A2,F6.1,A32,A19)') &
                     grid%desclidarloc(grid%id_lidarloc(k))//' ', grid%id, grid%id_lidarloc(k), &
                     ' '//grid%namelidarloc(grid%id_lidarloc(k)), &
                     ' (', grid%latlidarloc(grid%id_lidarloc(k)), ',', grid%lonlidarloc(grid%id_lidarloc(k)), ') (', &
                     grid%ilidarloc(k), ',', grid%jlidarloc(k), ') (', &
                     lidar_xlat, ',', lidar_xlong, ') ', &
                     lidar_hgt,' meters. simulation start time: ',TRIM(SimStartTime)
#endif
               CLOSE(UNIT=iunit)
            END IF
         END DO
   
      END IF
#if ((EM_CORE == 1) && (DA_CORE != 1))
   END IF
#endif

END SUBROUTINE calc_lidar_locations


SUBROUTINE calc_lidar( grid )

   USE module_domain
   USE module_model_constants

   IMPLICIT NONE

   ! Arguments
   TYPE (domain), INTENT(INOUT) :: grid

   LOGICAL, EXTERNAL :: wrf_dm_on_monitor

   ! Local variables
   INTEGER :: i, k, mm, n, ix, iy, rc
   REAL :: xtime_minutes
   REAL, ALLOCATABLE, DIMENSION(:)                       :: earth_u, earth_v,        &
     output_t, output_qv, output_qc, output_qr, output_qs, output_qh, output_qi,     &
     output_qg
   REAL, ALLOCATABLE, DIMENSION(:) :: p8w

   ! Parameter lidar_model_level:  
       ! TRUE to output T, Q, and wind at lowest model level
       ! FALSE to output T and Q at 2-m and wind at 10-m diagnostic levels:
   LOGICAL, PARAMETER :: lidar_model_level = .FALSE.  

   IF ( grid%nlidarloc_domain .LE. 0 ) RETURN

#if ((EM_CORE == 1) && (DA_CORE != 1))
   IF ( grid%dfi_opt /= DFI_NODFI .AND. grid%dfi_stage /= DFI_FST ) RETURN
#endif

   n = grid%next_lidar_time

   ALLOCATE(p8w(grid%sm32:grid%em32))
   ALLOCATE(earth_u(grid%sm32:grid%em32))
   ALLOCATE(earth_v(grid%sm32:grid%em32))
   ALLOCATE(output_t(grid%sm32:grid%em32))
   ALLOCATE(output_qv(grid%sm32:grid%em32))
   ALLOCATE(output_qc(grid%sm32:grid%em32))
   ALLOCATE(output_qr(grid%sm32:grid%em32))
   ALLOCATE(output_qs(grid%sm32:grid%em32))
   ALLOCATE(output_qh(grid%sm32:grid%em32))
   ALLOCATE(output_qi(grid%sm32:grid%em32))
   ALLOCATE(output_qg(grid%sm32:grid%em32))

   DO i=1,grid%nlidarloc_domain

      ix = grid%ilidarloc(i)
      iy = grid%jlidarloc(i)
  
      IF (grid%sp31 <= ix .AND. ix <= grid%ep31 .AND. &
          grid%sp33 <= iy .AND. iy <= grid%ep33) THEN
          DO k=grid%sm32, grid%em32-1
            !
            ! Output from the lowest model computational level:
            !
#if (EM_CORE == 1)
            earth_u(k) = grid%u_2(ix,k,iy)*grid%cosa(ix,iy)-grid%v_2(ix,k,iy)*grid%sina(ix,iy)
            earth_v(k) = grid%v_2(ix,k,iy)*grid%cosa(ix,iy)+grid%u_2(ix,k,iy)*grid%sina(ix,iy)
            output_t(k) = grid%t_2(ix,k,iy) + 300.
#else
            earth_u(k) = grid%u_2(ix,k,iy)*grid%cosa(ix,iy)-grid%v_2(ix,k,iy)*grid%sina(ix,iy)
            earth_v(k) = grid%v_2(ix,k,iy)*grid%cosa(ix,iy)+grid%u_2(ix,k,iy)*grid%sina(ix,iy)
            output_t(k) = grid%t(ix,k,iy) + 300.
#endif
            output_qv(k) = grid%moist(ix,k,iy,P_QV)
            output_qc(k) = grid%moist(ix,k,iy,P_QC)
            output_qr(k) = grid%moist(ix,k,iy,P_QR)
            output_qs(k) = grid%moist(ix,k,iy,P_QS)
            output_qh(k) = grid%moist(ix,k,iy,P_QH)
            output_qi(k) = grid%moist(ix,k,iy,P_QI)
            output_qg(k) = grid%moist(ix,k,iy,P_QG)
          END DO

      
         CALL domain_clock_get( grid, minutesSinceSimulationStart=xtime_minutes )
         grid%lidar_hour(n,i) = xtime_minutes / 60.
         grid%lidar_z(n,:,i) = ( grid%ph_2(ix,:,iy) + grid%phb(ix,:,iy) ) / g
         grid%lidar_p(n,:,i) = grid%p(ix,:,iy) + grid%pb(ix,:,iy) * 0.01
         grid%lidar_u(n,:,i) = earth_u
         grid%lidar_v(n,:,i) = earth_v
         grid%lidar_w(n,:,i) = grid%w_2(ix,:,iy)
         grid%lidar_t(n,:,i) = output_t
         grid%lidar_qv(n,:,i) = output_qv
         grid%lidar_qc(n,:,i) = output_qc
         grid%lidar_qr(n,:,i) = output_qr
         grid%lidar_qs(n,:,i) = output_qs
         grid%lidar_qh(n,:,i) = output_qh
         grid%lidar_qi(n,:,i) = output_qi
         grid%lidar_qg(n,:,i) = output_qg
         grid%lidar_dens(n,:,i) = 1./grid%alt(ix,:,iy)
         grid%lidar_cldfra(n,:,i) = grid%cldfra(ix,:,iy)
         grid%lidar_drydens(n,i) = grid%mu_2(ix,iy) + grid%mub(ix,iy) * 0.01
         grid%lidar_psfc(n,i) = grid%psfc(ix,iy)
#if (EM_CORE == 1)
         grid%lidar_rainc(n,i)  = grid%rainc(ix,iy)
         grid%lidar_rainnc(n,i) = grid%rainnc(ix,iy)
#endif   
      ELSE
         grid%lidar_hour(n,i) = 1.E30
         grid%lidar_u(n,:,i) = 1.E30
         grid%lidar_v(n,:,i) = 1.E30
         grid%lidar_w(n,:,i) = 1.E30
         grid%lidar_t(n,:,i) = 1.E30
         grid%lidar_qv(n,:,i) = 1.E30
         grid%lidar_qc(n,:,i) = 1.E30
         grid%lidar_qr(n,:,i) = 1.E30
         grid%lidar_qs(n,:,i) = 1.E30
         grid%lidar_qh(n,:,i) = 1.E30
         grid%lidar_qi(n,:,i) = 1.E30
         grid%lidar_qg(n,:,i) = 1.E30
         grid%lidar_dens(n,:,i) = 1.E30
         grid%lidar_cldfra(n,:,i) = 1.E30
         grid%lidar_psfc(n,i) = 1.E30
#if (EM_CORE == 1)
         grid%lidar_rainc(n,i)  = 1.E30
         grid%lidar_rainnc(n,i) = 1.E30
#endif
   
      END IF
   END DO

   DEALLOCATE(p8w)
 
   grid%next_lidar_time = grid%next_lidar_time + 1

   IF ( grid%next_lidar_time > grid%lidar_buf_size ) CALL write_lidar(grid)

END SUBROUTINE calc_lidar


SUBROUTINE write_lidar( grid )

   USE module_domain, ONLY : domain
   USE module_dm, ONLY : wrf_dm_min_reals
   USE module_state_description

   IMPLICIT NONE

   ! Arguments
   TYPE (domain), INTENT(INOUT) :: grid

   LOGICAL, EXTERNAL :: wrf_dm_on_monitor
! L. Fita, LMD. May 2014
! Already defined as function by wrf_timeseries.F
   INTEGER, EXTERNAL :: get_unused_unit

   ! Local variables
   INTEGER :: i, n, k, ix, iy, iunit
   REAL, ALLOCATABLE, DIMENSION(:,:,:) :: lidar_buf

   IF ( grid%nlidarloc_domain .LE. 0 ) RETURN

#if ((EM_CORE == 1) && (DA_CORE != 1))
   IF ( grid%dfi_opt /= DFI_NODFI .AND. grid%dfi_stage /= DFI_FST ) RETURN
#endif

#ifdef DM_PARALLEL
   ALLOCATE(lidar_buf(grid%lidar_buf_size,grid%sm32:grid%em32,grid%max_lidar_locs))

   lidar_buf(:,1,:) = grid%lidar_hour(:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,1,:),grid%lidar_hour(:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_z(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_z(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_p(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_p(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_u(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_u(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_v(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_v(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_w(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_w(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_t(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_t(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qv(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qv(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qc(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qc(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qr(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qr(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qs(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qs(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qh(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qh(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qi(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qi(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_qg(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_qg(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_dens(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_dens(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,:,:) = grid%lidar_cldfra(:,:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,:,:),grid%lidar_cldfra(:,:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,1,:) = grid%lidar_drydens(:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,1,:),grid%lidar_drydens(:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,1,:) = grid%lidar_psfc(:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,1,:),grid%lidar_psfc(:,:),grid%lidar_buf_size*grid%max_lidar_locs)

#if (EM_CORE == 1)
   lidar_buf(:,1,:) = grid%lidar_rainc(:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,1,:),grid%lidar_rainc(:,:),grid%lidar_buf_size*grid%max_lidar_locs)

   lidar_buf(:,1,:) = grid%lidar_rainnc(:,:)
   CALL wrf_dm_min_reals(lidar_buf(:,1,:),grid%lidar_rainnc(:,:),grid%lidar_buf_size*grid%max_lidar_locs)
#endif

   DEALLOCATE(lidar_buf)
#endif

   IF ( wrf_dm_on_monitor() ) THEN

      iunit = get_unused_unit()
      IF ( iunit <= 0 ) THEN
         CALL wrf_error_fatal('Error in write_lidar: could not find a free Fortran unit.')
      END IF

      DO i=1,grid%nlidarloc_domain

         ix = grid%ilidarloc(i)
         iy = grid%jlidarloc(i)

         OPEN(UNIT=iunit, FILE=TRIM(grid%lidar_filename(i)), STATUS='unknown', POSITION='append', FORM='formatted')

         DO n=1,grid%next_lidar_time - 1
#if (EM_CORE == 1)
            WRITE(UNIT=iunit,FMT='(a8,1x,i2,f13.6,i5,i5,i5,1x,4(e13.5,1x))') &
                            'new_time',grid%id, grid%lidar_hour(n,i), &
                            grid%id_lidarloc(i), ix, iy, &
                            grid%lidar_psfc(n,i), &
                            grid%lidar_rainc(n,i), &
                            grid%lidar_rainnc(n,i), &
                            grid%lidar_drydens(n,i)
#else
            WRITE(UNIT=iunit,FMT='(a8,1x,i2,f13.6,i5,i5,i5,1x,2(e13.5,1x))') &
                            'new_time',grid%id, grid%lidar_hour(n,i), &
                            grid%id_lidarloc(i), ix, iy, &
                            grid%lidar_psfc(n,i), &
                            grid%lidar_drydens(n,i)
#endif
            WRITE(UNIT=iunit, FMT='(5x,a3,1x,15(a14,1x),a10)') 'k', 'z [m]', 'p [hPa]',   &
              'u [ms-1]', 'v [ms-1]', 'w [ms-1]', 't_pot [k]', 'qv [kgkg-1]',             &
              'qc [kgkg-1]', 'qr [kgkg-1]', 'qs [kgkg-1]', 'qh [kgkg-1]', 'qi [kgkg-1]',  &
              'qg [kgkg-1]', 'dens [kg m-3]', 'cldfra [1]', '__________'
            DO k=grid%sm32,grid%em32-1
              WRITE(UNIT=iunit,FMT='(5x,i3,1x,15(e13.5,1x))') k, &
                              grid%lidar_z(n,k,i), &
                              grid%lidar_p(n,k,i), &
                              grid%lidar_u(n,k,i), &
                              grid%lidar_v(n,k,i), &
                              grid%lidar_w(n,k,i), &
                              grid%lidar_t(n,k,i), &
                              grid%lidar_qv(n,k,i), &
                              grid%lidar_qc(n,k,i), &
                              grid%lidar_qr(n,k,i), &
                              grid%lidar_qs(n,k,i), &
                              grid%lidar_qh(n,k,i), &
                              grid%lidar_qi(n,k,i), &
                              grid%lidar_qg(n,k,i), &
                              grid%lidar_dens(n,k,i), &
                              grid%lidar_cldfra(n,k,i)
           END DO
         END DO

         CLOSE(UNIT=iunit)

      END DO

   END IF

   grid%next_lidar_time = 1

END SUBROUTINE write_lidar

! L. Fita, LMD. May 2014
! Already defined as function by wrf_timeseries.F
! No SUBROUTINE calc_p8w(grid, ix, iy, p8w, k_start, k_end)
