MODULE module_lmdz_phys
!
! Module to include LMDZ (http://lmdz.lmd.jussieu.fr/) physical packages in WRF
!
! L. Fita, Laboratoire Meterologie Dynamique, IPSL, UPMC, CNRS, Jussieu, Paris, France
!   July 2013
!
  CONTAINS

  SUBROUTINE call_lmdz_phys(wrf_grid, wrf_xtime, wrf_restart_alarm,                  &
     &        wrf_Lon, wrf_Lat,                                                      &
     &        wrf_T, wrf_U, wrf_V,                                                   &
     &        wrf_perP, wrf_baseP,                                                   &
     &        wbdyw, wrf_isrestart,                                                  &
     &        wrf_islowbdyin                                                         &
                  ! dimension arguments
     &             ,wids,wide, wjds,wjde, wkds,wkde                                  &
     &             ,wims,wime, wjms,wjme, wkms,wkme                                  &
     &             ,wips,wipe, wjps,wjpe, wkps,wkpe                                  &
     &             ,wi_start,wi_end                                                  &
     &             ,wj_start,wj_end                                                  &
     &             ,wkts, wkte                                                       &
     &             ,wnum_tiles                                                       &
     &             ,wnum3dm, wparfirstscal,                                          &
     &        wnx, wny, wnz,                                                         &
     &        wjulday, wgmt, wtime_step,                                             &
     &        wrf_fulleta, wrf_halfeta, wrf_dfulleta,                                &
     &        wrf_fullpres, wrf_pergeopot, wrf_basegeopot,                           &
     &        wrf_MOIST, wrf_w, wrf_ptop,                                            &
     &        wrf_permass, wrf_basemass,                                             &
     &        wrf_mut, wrf_muu, wrf_muv,                                             &
     &        wrf_Utend, wrf_Vtend, wrf_Ttend,                                       &
     &        wrf_MOISTtend, wrf_psfctend,                                           &
     &        wrf_qvid, wrf_qcid, wrf_qrid,                                          &
     &        wrf_qsid, wrf_qiid, wrf_qhid, wrf_qgid,                                &
     &        wrf_mapft, wrf_mapfu, wrf_mapfv, wrf_dx, wrf_dy,                       &
     &        wrf_dbg, landcat, soilcat,                                             &
     &        wrf_l_pbl, wrf_l_con, wrf_l_thermals, wrf_l_wake,                      & 
     &        wrf_nsoillayers,                                                       &
     &        icheck_p, jcheck_p, kcheck_p                                           &
     &                                                 )

! WRF modules
    USE module_model_constants
    USE module_domain_type

! LMDZ modules
    USE infotrac, ONLY: infotrac_init, type_trac
    USE comgeomphy
    USE control_mod
    USE indice_sol_mod
    USE phys_state_var_mod
!    USE fonte_neige_mod, ONLY: fonte_neige_init, fonte_neige_final
    USE fonte_neige_mod
    USE iostart, ONLY: get_var
    USE surface_data
    USE pbl_surface_mod
    USE mod_grid_phy_lmdz, only : klon_glo

! WRF+LMDZ 

! Used to initialize variables in an old fashion way... Now using 'lmdz_wrf_variables_mod'
    USE wrf_lmdz_mod
    USE lmdz_wrf_variables_mod
    USE NOread_limit_sub_variables
    USE output_lmdz_NOmodule

    IMPLICIT NONE

    INCLUDE "../lmdz/dimensions.h"
    INCLUDE "../lmdz/comvert.h"
    INCLUDE "../lmdz/temps.h"
    INCLUDE "../lmdz/dimsoil.h"

    TYPE(domain) , TARGET                                :: wrf_grid

    REAL, INTENT(IN)                                     :: wrf_xtime, wrf_Ptop,     &
        wgmt, wrf_dx, wrf_dy
    LOGICAL, INTENT(IN)                                  :: wrf_restart_alarm,       &
        wrf_isrestart, wrf_islowbdyin

    INTEGER, INTENT(IN)                                  :: wnx, wny, wnz, wbdyw

    INTEGER, INTENT(IN)                                  ::                          &
        &                               wids,wide, wjds,wjde, wkds,wkde,             &
                                        wims,wime, wjms,wjme, wkms,wkme,             &
                                        wips,wipe, wjps,wjpe, wkps,wkpe,             &
                                        wkts,wkte, wnum_tiles

    INTEGER, DIMENSION(wnum_tiles), INTENT(IN)           ::                          &
         & wi_start,wi_end,wj_start,wj_end
    INTEGER, INTENT(IN)                                  :: wparfirstscal, wnum3dm
    REAL, INTENT(IN)                                     :: wjulday
    REAL, INTENT(IN)                                     :: wtime_step
    INTEGER, INTENT(IN)                                  :: wrf_qvid, wrf_qcid,      &
         & wrf_qrid, wrf_qsid, wrf_qiid, wrf_qhid, wrf_qgid, wrf_dbg

    REAL, INTENT(IN), DIMENSION(wkms:wkme)               :: wrf_fulleta
    REAL, INTENT(IN), DIMENSION(wkms:wkme)               :: wrf_halfeta, wrf_dfulleta

    REAL, INTENT(IN), DIMENSION(wims:wime,wkms:wkme,wjms:wjme) ::                    &
        & wrf_T, wrf_perP, wrf_baseP, wrf_U, wrf_V

    REAL, INTENT(OUT), DIMENSION(wims:wime,wkms:wkme,wjms:wjme) ::                   &
        & wrf_Ttend, wrf_Utend, wrf_Vtend

    REAL, INTENT(IN), DIMENSION(wims:wime,wkms:wkme,wjms:wjme) ::                    &
        & wrf_fullpres, wrf_pergeopot, wrf_basegeopot, wrf_w

    REAL, INTENT(IN), DIMENSION(wims:wime,wjms:wjme)     :: wrf_Lon, wrf_Lat,        &
        & wrf_basemass, wrf_permass, wrf_mapft, wrf_mapfu, wrf_mapfv, wrf_mut,       &
        & wrf_muu, wrf_muv

    REAL, INTENT(OUT), DIMENSION(wims:wime,wjms:wjme)    :: wrf_psfctend

    REAL, INTENT(IN), DIMENSION(wims:wime,wkms:wkme,wjms:wjme,1:wnum3dm) ::          &
        & wrf_MOIST

    REAL, INTENT(OUT), DIMENSION(wims:wime,wkms:wkme,wjms:wjme,1:wnum3dm) ::         &
        & wrf_MOISTtend

    CHARACTER(LEN=256), INTENT(IN)                       :: landcat, soilcat
    INTEGER, INTENT(IN)                                  :: wrf_l_pbl, wrf_l_con,    &
      wrf_l_thermals, wrf_l_wake
    INTEGER, INTENT(IN)                                  :: wrf_nsoillayers
    INTEGER, INTENT(IN)                                  :: icheck_p, jcheck_p,      &
      kcheck_p
! Local
    INTEGER                                              :: ip, jp, kp, lp

    REAL, DIMENSION(wims:wime,wkms:wkme,wjms:wjme)       :: wrf_Temp,                &
       & wrf_Temptend, wrf_P, wrfMASSValues
    REAL, DIMENSION(wims:wime,wkms:wkme,wjms:wjme)       :: wrfUdestagg, wrfVdestagg,&
       & wrfUtenddestagg, wrfVtenddestagg
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw), wkms:wkme-1) ::         &
       & GeopotValues, TValues, TtendValues, HalfPressValues, WFluxMassValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw), wkms:wkme-1) ::         &  !! DE-STAGGED !!
       & UValues, UtendValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw), wkms:wkme-1) ::         &  !! DE-STAGGED !!
       & VValues, VtendValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw), wkms:wkme) ::           &  !! Z-STAGG !!
       & FullPressValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw)) ::                      &  
       & sfcGeopotValues
    REAL, ALLOCATABLE, DIMENSION(:,:,:)                  :: MixingRatioValues,       &
       & MixingRatiotendValues

    REAL, DIMENSION(wims:wime,wkms:wkme,wjms:wjme)       :: wrfWValues,              &
       & wrfPERMASSValues, wrfZFluxMassValues
    REAL, DIMENSION(wims:wime,wkms:wkme,wjms:wjme)       :: wrf_geopot, w3Dmat,&
       & wrf_geopot_half
    REAL, DIMENSION(wims:wime,wjms:wjme)                 :: wrf_sfcgeopot

    LOGICAL                                              :: lmdz_debut, lmdz_lafin

    INTEGER                                              :: k, ix, iy, iz, iq, ixy,  &
       & lmdzp, ilmdz, jlmdz, ir, ixx, iyy
    INTEGER                                              :: dimx, dimy, dimz, dimxy
    INTEGER                                              :: ddimx, ddimy, ddimxy, ddimxy2
    INTEGER                                              :: dim_klon_glo, dim_ngridmx
    INTEGER                                              :: im2, jm2, km2
    INTEGER                                              :: lmdzmixingratios
    CHARACTER (LEN=1024)                                 :: message, wrf_err_message

    INTEGER, PARAMETER                                   :: Nphykeys=20
    REAL, DIMENSION(Nphykeys)                            :: phykeys

    REAL, DIMENSION(2,(wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw),wkms:wkme-1) ::        &
        & lmdz_dudyn, l3Dmat
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw),3) :: lmdz_PVtheta
    REAL, DIMENSION(wkms:wkme-1)                         :: lmdzoutP
    REAL, DIMENSION((wime-wims-2*wbdyw),(wjme-wjms-2*wbdyw)) :: wrf_area
    REAL, DIMENSION((wime-wims-2*wbdyw),(wjme-wjms-2*wbdyw)) :: wrf_cu
    REAL, DIMENSION((wime-wims-2*wbdyw),(wjme-wjms-2*wbdyw)) :: wrf_cv
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw)) :: AreaValues,          &
        LonValues, LatValues, PsfctendValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw)) :: CUValues, CVValues
    REAL, DIMENSION((wime-wims-2*wbdyw)*(wjme-wjms-2*wbdyw)) :: runoff_lic
    LOGICAL                                              :: existsfile
    INTEGER                                              :: ierr
    CHARACTER(LEN=50)                                    :: errmsg, wrnmsg, subname
    CHARACTER(LEN=100)                                   :: defilen, defvarn, defstr
    INTEGER                                              :: funit, defint
    LOGICAL                                              :: is_used, defbool
    REAL, DIMENSION(20)                                  :: lmdz_clesphy0
    INTEGER                                              :: iflag_phys
    REAL                                                 :: defreal,ksoilmax
    REAL, DIMENSION(4)                                   :: ksoils
    REAL                                                 :: Pi
    CHARACTER(LEN=50)                                    :: LMDZvarmethod, Spt, Spt3D
    CHARACTER(LEN=4)                                     :: ipS, jpS, kpS
    CHARACTER(LEN=8)                                     :: lpS
    INTEGER                                              :: nlanduse
    REAL, DIMENSION(11)                                  :: ORCHIDEElevels
    INTEGER                                              :: Nsoillayers,begdataset,  &
      Nvars,Nrs,Ncs,Nwrfsoiltypes
    REAL                                                 :: soilTmass
    CHARACTER(LEN=50)                                    :: datasetname
    CHARACTER(LEN=100)                                   :: tablename
    REAL, ALLOCATABLE, DIMENSION(:,:,:)                  :: soilvals
    CHARACTER(LEN=50), ALLOCATABLE, DIMENSION(:)         :: soilvalns
    INTEGER                                              :: lpbl, lcon, lth, lwake 
    TYPE(WRFU_Alarm), TARGET, DIMENSION(MAX_WRF_ALARMS)  :: wrf_alarms
   
!!!!!!! Variables
! 
! wrf_xtime: WRF actual timestep (in minutes since simulation started)
! wrf_restart_alarm: WRF value indicating whether this is a restart time-step
! wrf_T: WRF 3D temperature
! wrf_perP: WRF 3D perturbation of the pressure
! wrf_baseP: WRF 3D base pressure
! wn[x/y/z]: WRF dimensions at each axis
! wbdyw: number of grid points in the sponge zone
! wids: start for i in domain
! wide: end for i in domain
! wjds: start for j in domain
! wjde: end for j in domain
! wkds: start for k in domain
! wkde: end for k in domain
! wims: start for i in memory
! wime: end for i in memory
! wjms: start for j in memory
! wjme: end for j in memory
! wips: start for i in patch
! wipe: end for i in patch
! wjps: start for j in patch
! wjpe: end for j in patch
! wkms: start for k in memory
! wkme: end for k in memory
! wi_start: start indices for i in tile
! wi_end: end indices for i in tile
! wj_start: start indices for j in tile
! wj_end: end indices for j in tile
! wkts: start for k in tile
! wkte: end for k in tile
! wnum_tiles: number of tiles
! wjulday: WRF julian day
! wgmt: WRF gmt hour (0, 1)
! wtime_step: WRF time-step (in seconds)
! wrf_dbg: level of debug from WRF
! wrf_isrestart: flag indicating whether simulation is beggining from a restart or 
!   not (zero time since simulation started )
!
! lmdz_debut: flag indicating if this is the first step for LMDZ
! lmdz_lafin: flag indicating if this is the last step for LMDZ
! wrf_fulleta: WRF full mass eta levels (w)
! wrf_dfulleta: WRF full mass high between eta levels (w)
! wrf_halfeta: WRF half mass eta levels (t, u, v, h)
! wrf_Ptop: WRF top pressure
! wrf_Lon: WRF longitudes
! wrf_Lat: WRF latitudes
! wrf_T: WRF potential temperature
! wrf_Ttend: WRF potential temperature tendency
! wrf_Temp: WRF temperature
! wrf_Temptend: WRF temperature tendency
! wrf_perP: WRF perturbation of pressure
! wrf_baseP: WRF base of pressure
! wrf_P: WRF pressure
! wrf_U: WRF x-wind speed
! wrf_Utend: WRF x-wind speed tendency
! wrf_V: WRF y-wind speed
! wrf_Vtend: WRF y-wind speed tendency
! wrf_fullpres: WRF pressure in full eta levels
! wrf_pergeopot: WRF perturbation of geopotential
! wrf_basegeopot: WRF base of geopotential
! wrf_geopot: WRF geopotential
! wrf_geopot_half: WRF geopotential at the half eta level
! wrf_psfctend: WRF surface pressure tendency
! wrf_sfcgeopot: WRF surface geopotential
! wrf_MOIST: WRF MOIST ARRAY (water spieces and tracers)
! wrf_MOISTtend: WRF MOIST tendency ARRAY (water spieces and tracers)
! wrf_qvid: WRF water vapor index of wrf_MOIST
! wrf_qcid: WRF cloud liquid water index of wrf_MOIST
! wrf_qrid: WRF rain liquid water index of wrf_MOIST
! wnum3dm: WRF number of water species
! wrf_w: WRF vertical velocity
! wrf_permass: WRF perturbation dry air mass in column
! wrf_basemass: WRF base dry air mass in column
! wrf_mut: WRF total dry air mass column on mass points
! wrf_muu: WRF total dry air mass column on u-wind grid points
! wrf_muv: WRF total dry air mass column on v-wind gid points
! wrfWValues: staggered WRF vertical velocity (in m s-1)
! wrfPERMASSValues: de-staggered WRF perturbation of dry mass column (in Pa)
! wrfMASSValues: de-staggered WRF perturbation of dry mass column (in kg)
! wrfZFluxMassValues: WRF vertical flux mass values (kg s-1)
! wrf_mapf[t/u/v]: WRF map factor for Temperature, u-wind and v-wind grid points
! wrf_d[x/y]: length of the grid spacing in the x and y direction (in m)
! wrf_area: WRF area of the grid cells
! wrf_c[u/v]: WRF covariant values for wind components
! landcat: land category used in the simulation
! nlanduse: number of soil types in the WRF landuse variable
!
! LonValues: 2D-LMDZ array of WRF longitudes (in rad)
! LatValues: 2D-LMDZ array of WRF latitudes (in rad)
! TValues: 2D-LMDZ array of WRF temperatures (in K)
! TtendValues: 2D-LMDZ tendency array of WRF temperatures (in K)
! HalfPressValues: 2D-LMDZ array of WRF pressure at half-mass levels (in Pa)
! UValues: 2D-LMDZ array of WRF x-wind speed (in m s-1)
! UtendValues: 2D-LMDZ array of WRF tendency x-wind speed (in m s-2)
! VValues: 2D-LMDZ array of WRF y-wind speed (in m s-1)
! VtendValues: 2D-LMDZ array of WRF tendency y-wind speed (in m s-2)
! PsfctendValues: 2D-LMDZ array of surface pressure tendency y-wind speed (in Pa s-1)
! FullPressValues: 2D-LMDZ array of WRF pressure at full-mass levels (in Pa)
! GeopotValues: 2D-LMDZ array of WRF geopotential (in m2 s-2)
! sfcGeopotValues: LMDZ array of WRF sfc geopotential (in m2 s-2)
! lmdzmixingratios: number of 'mixing ratios' to pass to the LMDZ physics
! MixingRatioValues: 2D-LMDZ array of Mixing Ratios (in kg kg-1)
! MixingRatiotendValues: 2D-LMDZ array of tendency Mixing Ratios (in kg kg-1 s-1)
! WFluxMassValues: 2Dvertical mass flux values (in kg m s-1)
! phykeys: LMDZ obsolete control I/O vector
! lmdz_dudyn: LMDZ obsolete vector
! lmdz_PVtheta: LMDZ obsolete PV values at given theta levels
! lmdzoutP: LMDZ output pressure values (using same eta levels, but imposing Psfc=1000 hPa)
! preff: refernce pressure for lmdzouP
! AreaValues: LMDZ area (m2)
! CUArea: LMDZ cu coeff. (u_covariant = cu * u)
! CVArea: LMDZ cv coeff. (v_covariant = cv * v)
! 
!!!!!!! Subroutines/Functions
! WRFmoist_LMDZmoist: Subroutine to transform from WRF moisture array to the LMDZ moisture array
! eta_to_pressure: Subroutine to transform from eta levels to pressure levels
! vect_mat: Subroutine to transform a LMDZ physic 1D vector to a 3D matrix
! mat_vect: Subroutine to transform a 3D matrix to a LMDZ physic 1D vector
! temp_to_theta: Subroutine to transform from temperatures to WRF potential temperatures
! theta_to_temp: Subroutine to transform from WRF potential temperatures to temperature
! physiq: Subroutine with the LMDZ physics

! CAPITAL variables stand for the name of the variables inside the subroutines/functions

!!!!!!! !!!!!! !!!!! !!!! !!! !! !
    WRITE(message, *)'  starting: module_lmdz_phys.F'
    CALL wrf_debug(75, message)

    PRINT *,' --------------------------------------------------------------------------------------------------'

! Basic definitions
!!
    errmsg='ERROR -- error -- ERROR -- error'
    wrnmsg='WARNING -- warning -- WARNING -- warning'
    subname='module_lmdz_phys'

    Pi = 2.*ASIN(1.)

    type_trac = 'lmdz'

    dimx=wime-wims
    dimy=wjme-wjms
    dimz=wkme-wkms
    dimxy=dimx*dimy
    dim_klon_glo = dimx*dimy
    dim_ngridmx = dimx*dimy
    ddimx = dimx-2*wbdyw
    ddimy = dimy-2*wbdyw
    ddimxy=ddimx*ddimy
    ddimxy2=ddimx*(ddimy/2-1)+ddimx/2

! Point to retrieve values for debugging
!    ip=ddimx/2
!    jp=ddimy/2
!    kp=dimz/2
!    ip=5
!    jp=11
    ip=icheck_p
    jp=jcheck_p
    kp=kcheck_p

    lp=ddimx*(jp-1)+ip
    WRITE(ipS,'(I4)')ip
    WRITE(jpS,'(I4)')jp
    WRITE(kpS,'(I4)')kp
    WRITE(lpS,'(I8)')lp
    Spt = TRIM(ipS) // ', '//TRIM(jpS)
    Spt3D = TRIM(ipS) // ', '//TRIM(jpS)//', '//TRIM(kpS)
    PRINT *,'  checking point ip: ',ip,' jp: ',jp,' kp: ',kp,' lp: ',lp

    PRINT *,'  Lluis in module_lmdz_phy itime_step= ', wrf_grid%itimestep,           &
      ' wrf_xtime: ',wrf_xtime,' wrf_rst: ', wrf_isrestart,' wrf_lwbdy: ',           &
      wrf_islowbdyin

    phykeys=0.
    lmdz_dudyn=0.
    lmdz_PVtheta=0.

! Checking of first/last time-step for LMDZ
!!
    lmdz_debut = .FALSE.
    lmdz_lafin = .FALSE.

    DO iy=1,ddimy
      DO ix=1,ddimx
        IF (wrf_Lon(ix,iy) < 0.) THEN
          LonValues(ddimx*(iy-1)+ix) = (360. + wrf_Lon(ix,iy))*Pi/180.
        ELSE
          LonValues(ddimx*(iy-1)+ix) = wrf_Lon(ix,iy)*Pi/180.
        END IF
        LatValues(ddimx*(iy-1)+ix) = wrf_Lat(ix,iy)*Pi/180.
        PsfctendValues(ddimx*(iy-1)+ix) = wrf_psfctend(ix,iy)
      END DO
    END DO

! Reshape can not be used? Different initial/ending indices !
!      LonValues = RESHAPE(wrf_Lon, (/ dimx*dimy/), ORDER=(/1/))
!      LatValues = RESHAPE(wrf_Lat, (/ dimx*dimy/), ORDER=(/1/))

    firststep: IF ( wrf_xtime == 0 .OR. wrf_isrestart) THEN
      lmdz_debut = .TRUE.

      DO funit=10,100
        INQUIRE(UNIT=funit, OPENED=is_used)
        IF (.NOT. is_used) EXIT
      END DO
      lmdz_clesphy0 = 0.

      CALL conf_gcm(funit, .TRUE., lmdz_clesphy0)
! L. Fita, wee need to provide the value
      annee_ref = anneeref
      day_ref = dayref

! Uploading LMDZ general configuration
! used instead call conf_gcm, since variables are not found in memory (or at least with a value)
      defilen = 'gcm.def'
      defvarn = 'iflag_phys'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      iflag_phys = defint

! Is not working it has an space...
!      defvarn = 'type_trac'
!      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
!      type_trac = TRIM(defstr)

! used instead call conf_gcm, since variables are not found in memory (or at least with a value)
      defilen = 'config.def'
      defvarn = 'type_ocean'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      type_ocean= TRIM(defstr)

! used instead call conf_gcm, since variables are not found in memory (or at least with a value)
! Looking for consistency for some namelist variables and *.def LMDZ values
      PRINT *,'  Checking consistency between namelist and LMDZ *.def values'
      defilen = 'physiq.def'
      defvarn = 'iflag_pbl'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      lpbl = defint
      defvarn = 'iflag_con'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      lcon = defint
      defvarn = 'iflag_thermals'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      lth = defint
      defvarn = 'iflag_wake'
      CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
      lwake = defint

      IF (wrf_l_pbl /= lpbl) THEN
        PRINT *,TRIM(errmsg)
        PRINT *,'  ' // TRIM(subname) // ' namelist LMDZ pbl value: ',               &
          wrf_l_pbl,' differs from iflag_pbl= ',lpbl ,' in physiq.def !!!!'
        WRITE(wrf_err_message,*)'  wrong namelist.input value !!'
        CALL wrf_error_fatal(TRIM(wrf_err_message))
      END IF
      IF (wrf_l_con /= lcon) THEN
        PRINT *,TRIM(errmsg)
        PRINT *,'  ' // TRIM(subname) // ' namelist LMDZ convection value: ',        &
          wrf_l_con,' differs from iflag_con= ',lcon ,' in physiq.def !!!!'
        WRITE(wrf_err_message,*)'  wrong namelist.input value !!'
        CALL wrf_error_fatal(TRIM(wrf_err_message))
      END IF
      IF (wrf_l_thermals /= lth) THEN
        PRINT *,TRIM(errmsg)
        PRINT *,'  ' // TRIM(subname) // ' namelist LMDZ thermals value: ',          &
          wrf_l_thermals,' differs from iflag_thermals= ',lth ,' in physiq.def !!!!'
        WRITE(wrf_err_message,*)'  wrong namelist.input value !!'
        CALL wrf_error_fatal(TRIM(wrf_err_message))
      END IF
      IF (wrf_l_wake /= lwake) THEN
        PRINT *,TRIM(errmsg)
        PRINT *,'  ' // TRIM(subname) // ' namelist LMDZ wakes value: ',             &
          wrf_l_wake,' differs from iflag_wake= ',lwake ,' in physiq.def !!!!'
        WRITE(wrf_err_message,*)'  wrong namelist.input value !!'
        CALL wrf_error_fatal(TRIM(wrf_err_message))
      END IF

! Initialization of the variables
!!

      INQUIRE(FILE='traceur.def', EXIST=existsfile)

! L. Fita, LMD. August, this would be much more nice if it is in config_flags structure....
      IF (existsfile) THEN
        DO funit=10,100
          INQUIRE(UNIT=funit, OPENED=is_used)
          IF (.NOT. is_used) EXIT
        END DO
        OPEN(UNIT=funit, FILE='traceur.def', STATUS='OLD')
        READ(funit,*)lmdzmixingratios
        CLOSE(UNIT=funit)
      ELSE
        WRITE(message, *)errmsg
        CALL wrf_debug(75, message)
        WRITE(wrf_err_message,*)'  required file traceur.def not found !!'
        CALL wrf_error_fatal(TRIM(wrf_err_message))
      END IF

      CALL infotrac_init

! Initializing LMDZ matrices...

      CALL Init_Phys_lmdz(1,ddimxy,dimz,1,(/ddimxy/))
      CALL InitComgeomphy()

      CALL phys_state_var_init(0)
!!        CALL neige_initialize()
!!        CALL limit_initialize()

!!        CALL pbl_initialize()

      PRINT *,'Lluis after initializing all LMDZ!'

      rlon=LonValues
      rlat=LatValues

      xtime0: IF (wrf_xtime == 0.) THEN

        nlanduse=UBOUND(wrf_grid%landusef, 2)
        CALL WRFlanduse_LMDZKsoil(landcat,wims,wime,wjms,wjme,ddimx,ddimy,nlanduse,  &
          wrf_grid%landusef,wrf_grid%landmask,wrf_grid%lter,wrf_grid%llic,           &
          wrf_grid%loce,wrf_grid%lsic)

! LANDMASK in LMDZ as fractions
        wrf_grid%lmsk = wrf_grid%lter + wrf_grid%llic
! Pseudo WRF-initialization

! LANDMASK in LMDZ as 1/0
!        wrf_grid%lmsk=0.
!        DO ix=1,ddimx
!          DO iy=1,ddimy
!            IF (wrf_grid%lter(ix,iy) >= 0.5) THEN
!              wrf_grid%lmsk(ix,iy)=1.
!              wrf_grid%lter(ix,iy)=1.
!              wrf_grid%llic(ix,iy)=0.
!              wrf_grid%loce(ix,iy)=0.
!              wrf_grid%lsic(ix,iy)=0.
!            ELSE
!              wrf_grid%lmsk(ix,iy)=0.
!              wrf_grid%lter(ix,iy)=0.
!              wrf_grid%llic(ix,iy)=0.
!              wrf_grid%loce(ix,iy)=1.
!              wrf_grid%lsic(ix,iy)=0.
!            END IF
!          END DO
!        END DO
! End of Pseudo WRF-initialization
!!!!! 

! Value is given at the first time-step, not any more, since it will be continuously 
!   used by LMDZ
! l. Fita, LMD. January 2014
! This should be in this way, but, because we can not use any scheme DZS=0., so 
!    direct met_em values (Registry modified) are used. soil layers are in [cm]!
!        DO iz=1,SIZE(wrf_grid%soil_layers(ip,:,jp))
!          wrf_grid%lwsol = wrf_grid%lwsol + wrf_grid%sm(:,iz,:)*                     &
!            wrf_grid%soil_layers(:,iz,:)/100.
!        END DO
        DO iz=1,4
          wrf_grid%ltksoil(:,iz,:)=wrf_grid%st(:,1,:)
        END DO
!
! Only using SST on that ocean grid points
        DO ix=1,ddimx
          DO iy=1,ddimy
            IF (wrf_grid%loce(ix,iy) > 0.) THEN
! Full SST grid points must have a SST value from the wrfinput (WRF 1/0 landmask)
!   but that LMDZ points with fractions of land/mask could not. Using the closest one
!   if it is an isolated grid point, using TSK instead
              IF (wrf_grid%sst(ix,iy) < 200.15) THEN
                wrf_grid%ltksoil(ix,3,iy) = -9.
                IF ( (ix > 1).AND.(ix < ddimx) .AND. (iy > 1).AND.(iy < ddimy) ) THEN
                  DO ixx=-1,1
                    DO iyy=-1,1
!                      PRINT *,ixx,iyy,wrf_grid%sst(ix+ixx,iy+iyy)
                      IF (wrf_grid%sst(ix+ixx,iy+iyy) > 200.15) THEN
                        wrf_grid%ltksoil(ix,3,iy) = wrf_grid%sst(ix+ixx,iy+iyy)
                        EXIT
                      END IF
                    END DO
                    IF ( wrf_grid%ltksoil(ix,3,iy) > 200.15) EXIT
                  END DO
                  IF ( wrf_grid%ltksoil(ix,3,iy) == -9.) wrf_grid%ltksoil(ix,3,iy) = &
                    wrf_grid%tsk(ix,iy)
                ELSE
                  wrf_grid%ltksoil(ix,3,iy) = wrf_grid%tsk(ix,iy)
                END IF
              ELSE
                wrf_grid%ltksoil(ix,3,iy) = wrf_grid%sst(ix,iy)
              END IF
              IF ( (wrf_grid%ltksoil(ix,3,iy) < 200.15) .OR.                         &
                (wrf_grid%ltksoil(ix,3,iy) > 370.) ) THEN
                PRINT *,TRIM(errmsg)
                WRITE(wrf_err_message,*) '  ' // TRIM(subname) //                    &
                  ' SST inconsistency at point: ', ix, ', ', iy,' with loce: ',      &
                  wrf_grid%loce(ix,iy),' has a tsoil(oce)= ',                        &
                  wrf_grid%ltksoil(ix,3,iy),' sst: ',wrf_grid%sst(ix,iy),' tsk: ',   &
                  wrf_grid%tsk(ix,iy)
                CALL wrf_error_fatal(TRIM(wrf_err_message))
              END IF
            END IF
          END DO
        END DO

!! Soil temperature, humidity, water-content...
! This should go on the namelist... later!
        ORCHIDEElevels = (/ 0.098, 0.1955, 0.5865, 1.3685, 2.9326, 6.0606, 12.3167,  &
          24.8289, 49.8534, 99.9022, 200. /)

        datasetname = soilcat
        tablename = 'SOILPARM.TBL'

        PRINT *,'  Lluis reading dataset "' // TRIM(datasetname) // '" from "' //    &
          TRIM(tablename) // '"'

        CALL table_characteristics(datasetname,tablename,begdataset,Nvars,Nrs,Ncs)

! L. Fita, LMD. Feburary 2014
! Again (as in the landuses case), the files content more categeories that the really ones used by the model...
        Nsoillayers=SIZE(wrf_grid%soil_layers(ip,:,jp))
        Nwrfsoiltypes=SIZE(wrf_grid%soilctop(ip,:,jp))
        IF (Nrs > Nwrfsoiltypes) THEN
          PRINT *,TRIM(wrnmsg)
          PRINT *,"  '" // TRIM(datasetname)//"' from '"//TRIM(tablename)//"' has ", &
            Nrs,' soil layers, meanwhile WRF data has ',Nwrfsoiltypes
          PRINT *,'  using only the first ',  Nwrfsoiltypes, ' of the dataset !!!!'
          Nrs=Nsoillayers
        ELSE IF (Nrs < Nwrfsoiltypes) THEN
          PRINT *,TRIM(errmsg)
          PRINT *,"  '" // TRIM(datasetname)//"' from '"//TRIM(tablename)//"' has ", &
            Nrs,' soil layers, meanwhile WRF data has ',Nwrfsoiltypes
          WRITE(wrf_err_message,*)'  wrong the dataset !!!!'
          CALL wrf_error_fatal(TRIM(wrf_err_message))
        END IF

        IF (ALLOCATED(soilvals)) DEALLOCATE(soilvals)
        ALLOCATE(soilvals(Nvars,Nrs,Ncs))

        IF (ALLOCATED(soilvalns)) DEALLOCATE(soilvalns)
        ALLOCATE(soilvalns(Nrs))

        CALL read_table(datasetname, tablename, begdataset, Nvars, Nrs, Ncs,         &
          soilvals, soilvalns)

        DO ix=1,ddimx
          DO iy=1,ddimy
!L. Fita, LMD. January 2014 Soil layers are in cm!


! NOTE: wrf_dx, wrf_dy should be matrix !!!
            CALL compute_TOTsoil_mass_water_values(datasetname, Nvars, Nrs, Ncs,     &
              soilvals, soilvalns, ix, iy, Nrs, wrf_grid%soilctop(ix,:,iy),          &
              wrf_grid%soilcbot(ix,:,iy), Nsoillayers, wrf_grid%sm(ix,:,iy),         &
              wrf_grid%soil_layers(ix,Nsoillayers:1:-1,iy)/100., wrf_dx, wrf_dy,     &
              soilTmass, wrf_grid%lwsol(ix,iy))

            wrf_grid%lwsol(ix,iy) =  wrf_grid%lwsol(ix,iy)/(wrf_dx*wrf_dy*1000.)

            CALL interpolate_layers(SIZE(wrf_grid%soil_layers(ix,:,iy)),             &
              wrf_grid%soil_layers(ix,SIZE(wrf_grid%soil_layers(ix,:,iy)):1:-1,iy),  &
              wrf_grid%st(ix,:,iy),11, ORCHIDEElevels,wrf_grid%lotter(ix,:,iy))
          END DO
        END DO

        wrf_grid%lotlic=wrf_grid%lotter
        wrf_grid%lotoce=wrf_grid%lotter
        wrf_grid%lotsic=wrf_grid%lotter

        DEALLOCATE(soilvals)
        DEALLOCATE(soilvalns)

! LMDZ does not distinguish among species, use it as a whole
        wrf_grid%lclwcon=SUM(wrf_MOIST,4)
        DO ix=wims,wime
          DO iz=wkms,wkme
            DO iy=wjms,wjme
              IF (wrf_grid%lclwcon(ix,iz,iy) /= 0.) THEN
                wrf_grid%lratqs=(wrf_MOIST(ix,iz,iy,wrf_qrid)+                       &
                wrf_MOIST(ix,iz,iy,wrf_qsid)+wrf_MOIST(ix,iz,iy,wrf_qiid)+           &
                wrf_MOIST(ix,iz,iy,wrf_qhid)+wrf_MOIST(ix,iz,iy,wrf_qgid))/          &
                wrf_grid%lclwcon(ix,iz,iy)
              END IF
            END DO
          END DO
        END DO 

!        wrf_grid%znt = wrf_grid%z0
!        wrf_grid%albedo = wrf_grid%albbck
        wrf_grid%lbils=0.
        wrf_grid%lfder=0.
        wrf_grid%lrnebcon=0.
        DO ix=1,ddimx
          DO iy=1,ddimy
            ksoils = (/ wrf_grid%lter(ix,iy), wrf_grid%llic(ix,iy),                  &
              wrf_grid%loce(ix,iy), wrf_grid%lsic(ix,iy) /)
            ksoilmax = MAXVAL(ksoils)
            DO iz = 1,4
              IF (ksoilmax == ksoils(iz)) wrf_grid%lnat(ix,iy) = iz
            END DO
          END DO
        END DO
        wrf_grid%lrads = wrf_grid%swdown

        wrf_grid%lcontrol(1) = wtime_step
        defilen = 'config.def'
        defvarn = 'nbapp_rad'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        wrf_grid%lcontrol(6) = defint
        wrf_grid%lcontrol(2) = NINT (86400./wtime_step/ FLOAT(defint) )
!L. Fita, LMD. February 2014: current value of atmospheric CO2 should be 
!!   co2_ppm0 + delta_co2_ppm, from 'carbon_cycle_mod.F90'
        defvarn = 'co2_ppm'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        wrf_grid%lcontrol(3) = defint
!!c co2_ppm0 : initial value of atmospheric CO2
        wrf_grid%lcontrol(16) = defint
        defvarn = 'solaire'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        wrf_grid%lcontrol(4) = defreal
        wrf_grid%lcontrol(5) = wrf_l_con

! Default value cycle_diurne = .T.
        wrf_grid%lcontrol(7) = 1.
        defvarn = 'soil_model'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        IF(defbool) wrf_grid%lcontrol(8) = 1.
        defilen = 'physiq.def'
        defvarn = 'new_oliq'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        IF(defbool) wrf_grid%lcontrol(9) = 1.
        defvarn = 'ok_orodr'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        IF(defbool) wrf_grid%lcontrol(10) = 1.
        defvarn = 'ok_orolf'
        CALL def_get_scalar_value(defilen, defvarn, defstr, defint, defreal, defbool)
        IF(defbool) wrf_grid%lcontrol(11) = 1.

        wrf_grid%lcontrol(13) = dayref + nday
        wrf_grid%lcontrol(14) = annee_ref
        wrf_grid%lcontrol(15) = wrf_grid%itimestep

! Let's split rugosity by Ksoils
!! Static roughness length
        wrf_grid%lrug=wrf_grid%znt

!! Dynamic roughness length (It should work if we use WRF schemes...)
!        wrf_grid%lrug=wrf_grid%znt
! First split by Ksoil types.
!   NOTE: in somehow we should read again the table 'VEGPARM.TBL' and recompute the
!     roughness length in case that appear a new soil type. (but what left below a 
!     melted glacier?)
!
        LMDZvarmethod='prod'
        wrf_grid%lrugksoil(:,1,:) = wrf_grid%znt

        CALL lmdz_varKsoil(wims,wime,wjms,wjme,dimx,dimy,wbdyw,wrf_grid%snow,        &
          LMDZvarmethod,wrf_grid%lter,wrf_grid%llic,wrf_grid%loce,wrf_grid%lsic,     &
          wrf_grid%lsnowksoil(:,1,:),wrf_grid%lsnowksoil(:,2,:),                     &
          wrf_grid%lsnowksoil(:,3,:),wrf_grid%lsnowksoil(:,4,:))
! All the iced soil types have the same roughtness length 0.001
        wrf_grid%lrugksoil(:,2,:)=0.001
        wrf_grid%lrugksoil(:,3,:)=0.0001
        wrf_grid%lrugksoil(:,4,:)=0.001

! Recomputig again the total roughness length
        CALL wrf_varKsoil(wims,wime,wjms,wjme,dimx,dimy,wbdyw,LMDZvarmethod,         &
          wrf_grid%lter,wrf_grid%llic,wrf_grid%loce,wrf_grid%lsic,                   &
          wrf_grid%lrugksoil(:,1,:),wrf_grid%lrugksoil(:,2,:),                       &
          wrf_grid%lrugksoil(:,3,:),wrf_grid%lrugksoil(:,4,:),wrf_grid%lrug)

! Air surface moisture for each soil type. First order?
        DO iz=1,4
          wrf_grid%lqksoil(:,iz,:)=wrf_grid%q2
        END DO

! Let's split albedo by Ksoils
        wrf_grid%lalbksoil=0.

        DO iz=1,4
!! Static roughness length (changing on time in wrflwbdy_d01)
          wrf_grid%lalbksoil(:,iz,:)=wrf_grid%albedo
!! Dynamic roughness length (It should work if we use WRF schemes...)
!          wrf_grid%lalbksoil(:,iz,:)=wrf_grid%albedo
        END DO
! First split by Ksoil types.
!   NOTE: in somehow we should read again the table 'VEGPARM.TBL' and recompute the
!     albedo in case that appear a new soil type. (but what left below a melted 
!     glacier?)
!
! All the iced soil types have the same albedo
! For snow/ice there is zmin=0.55, zmax=0.70, tacking the mean value: 0.625
        wrf_grid%lalbksoil(:,2,:)=0.625
        wrf_grid%lalbksoil(:,3,:)=0.08
        wrf_grid%lalbksoil(:,4,:)=0.625

! Recomputig again the total roughness length
        CALL wrf_varKsoil(wims,wime,wjms,wjme,dimx,dimy,wbdyw,LMDZvarmethod,         &
          wrf_grid%lter,wrf_grid%llic,wrf_grid%loce,wrf_grid%lsic,                   &
          wrf_grid%lalbksoil(:,1,:),wrf_grid%lalbksoil(:,2,:),                       &
          wrf_grid%lalbksoil(:,3,:),wrf_grid%lalbksoil(:,4,:),wrf_grid%albedo)
! Using the same values for the LW albedo
        wrf_grid%llwalbksoil = wrf_grid%lalbksoil

! Orography values just beacuse there is no other way right now (they should come 
!   from geogrid.exe)
        wrf_grid%lzmea = wrf_grid%ht
        wrf_grid%lzstd = wrf_grid%topostdv
        wrf_grid%lzsig = wrf_grid%slope
        wrf_grid%lzgam = SQRT((wrf_grid%toposlpx**2. + wrf_grid%toposlpy**2.))/      &
          (1. + wrf_grid%slope)
        wrf_grid%lzthe = ACOS(wrf_grid%toposlpx/SQRT(wrf_grid%toposlpx**2. +         &
          wrf_grid%toposlpy**2.))
        wrf_grid%lzpic = wrf_grid%ht*1.10
        wrf_grid%lzval = wrf_grid%ht*0.90
        wrf_grid%lzrugsrel = wrf_grid%lrugKsoil(:,1,:)

! R = 8.3144621 ([J mol/K] http://en.wikipedia.org/wiki/Gas_constant)
! mu = 28.97 (Molecular Mass of Air [kg/kmol] from http://www.engineeringtoolbox.com/molecular-mass-air-d_679.html)
! R / mu =? r_d

!        lmdz_Area = dLon * dLat * cos(lat) * Earth_rad

      END IF xtime0

!  Grids do not start/end at the same index
      wrf_area = 0.
      wrf_cu = 0.
      wrf_cv = 0.
      DO iy=1,ddimy
        DO ix=1,ddimx
          wrf_area(ix,iy) = wrf_dx*wrf_dy/wrf_mapft(ix,iy)
          wrf_cu(ix,iy) = wrf_dx/wrf_mapfu(ix,iy)
          wrf_cv(ix,iy) = wrf_dy/wrf_mapfv(ix,iy)
        END DO
      END DO

!      cu = dLon * Earth_rad
!      cv = dLat * Earth_rad * cos(Lat)

!      lmdz_cu(1) = cu(1)
!      lmdz_cv(1) = cv(1)
!      DO j=2,dimy
!        DO i=1,dimx
!          lmdz_cu((j-2)*dimx+1+i) = cu((j-1)*(dimx+1)+i)
!          lmdz_cv((j-2)*dimx+1+i) = cv((j-1)*(dimx+1)+i)
!        ENDDO
!      ENDDO
!      lmdz_cu(dim_ngridmx) = cu(dimxy+1)
!      lmdz_cv(dim_ngridmx) = cv(dimxy-dimx)

      AreaValues = RESHAPE(wrf_area, (/ ddimx*ddimy/), ORDER=(/1/))
      CUValues = RESHAPE(wrf_cu, (/ ddimx*ddimy/), ORDER=(/1/))
      CVValues = RESHAPE(wrf_cv, (/ ddimx*ddimy/), ORDER=(/1/))

      CALL lmdz_vars_init(ddimxy) 

      CALL load_lmdz(wrf_grid,ddimx,ddimy,dimz,wbdyw,ddimxy,4,11,rlon,rlat,zmasq,    &
        pctsrf,ftsol,ftsoil_rst,qsurf_rst,qsol_rst,falb1,falb2,evap_rst,snow_rst,    &
        radsol,solsw,sollw,fder_rst,rain_fall,snow_fall,agesno_rst,zmea,zstd,zgam,   &
        zthe,zpic,zval,rugoro,rugos_rst,restart_runoff,zmax0,f0,wake_s,wake_cstar,   &
        wake_pe,wake_fip,sst,albedo,t_ancien,q_ancien,u_ancien,v_ancien,clwcon,      &
        rnebcon,ratqs,ema_work1,ema_work2,wake_deltat,wake_deltaq,fm_therm,          &
        entr_therm,detr_therm,phy_glo)

      CALL pbl_surface_init(qsol_rst, fder_rst, snow_rst, qsurf_rst, evap_rst,       &
        rugos_rst, agesno_rst, ftsoil_rst)
      CALL fonte_neige_init(restart_runoff)

      DO ix=1,ddimx
        DO iy=1,ddimy
          ixy=ddimx*(iy-1)+ix
          IF (wrf_grid%lter(ix,iy) /= pctsrf(ixy,1)) PRINT *,'    ' //               &
            TRIM(subname) // ' ter DOES NOT COINCIDE! ', ix,', ',iy,' lter: ',       &
            wrf_grid%lter(ix,iy),' pctsrf: ',pctsrf(ixy,1)
          IF (wrf_grid%llic(ix,iy) /= pctsrf(ixy,2)) PRINT *,'    ' //               &
            TRIM(subname) // ' lic DOES NOT COINCIDE! ', ix,', ',iy,' llic: ',       &
            wrf_grid%llic(ix,iy),' pctsrf: ',pctsrf(ixy,2)
          IF (wrf_grid%loce(ix,iy) /= pctsrf(ixy,3)) PRINT *,'    ' //               &
            TRIM(subname) // ' oce DOES NOT COINCIDE! ', ix,', ',iy,' loce: ',       &
            wrf_grid%loce(ix,iy),' pctsrf: ',pctsrf(ixy,3)
          IF (wrf_grid%lsic(ix,iy) /= pctsrf(ixy,4)) PRINT *,'    ' //               &
            TRIM(subname) // ' sic DOES NOT COINCIDE! ', ix,', ',iy,' lsic: ',       &
            wrf_grid%lsic(ix,iy),' pctsrf: ',pctsrf(ixy,4)
        END DO
      END DO

      CALL NOread_limit_init(ddimxy, klon_glo)

      sst_glo = sst(1:ddimxy)
! In ocean_forced_mod is used as tsurf_limit = sst(knindex(1:knon))
      tsurf_limit = ftsol(:,3)

! In surf_land_bucket_mod is used as z0_new=rugos(knindex(1:knon),1) [z0_new == z0_limit]
      z0_limit = rugos_rst(:,1)
      alb_limit = albedo

      rug_glo = rugos_rst(:,is_ter)
      pct_glo(klon_glo,:) = pctsrf(klon_glo,:)

! It is not clear how to deal with 'limit.nc' times...
!      temps=wjulday

      CALL iniphysiq(ddimx*ddimy, dimz, 86400., wjulday, wtime_step,                    &
        & LatValues, LonValues, wrf_area, CUValues, CVValues, 1./reradius, g, r_d, cp,  &
        & iflag_phys)

! NO NEED to do this, using wrfinput_d[nn], wrfbdy_d[nn] and wrflowinput_d[nn]
! Checking for 'startphy.nc' presence
!!      INQUIRE(FILE='startphy.nc', EXIST=existsfile)
!!      IF (.NOT.existsfile) THEN
!!        WRITE(wrf_err_message,*)"  file 'startphy.nc' with initial conditions " //   &
!!          & ' for LMDZphys does not exist!!!!'
!!        CALL wrf_error_fatal(TRIM(wrf_err_message))
!!      END IF

      CALL init_ovars_lmdz_NOmodule(ddimxy,dimz,nbsrf)

      PRINT *,'  Lluis: END of firststep !!!!!'
    END IF firststep

    ip=icheck_p
    jp=jcheck_p
! WRF alarms
! I can not print them !!!!
!!
!    wrf_grid%alarms => wrf_alarms 
!    PRINT *,'  Lluis WRF alarms: time-step', wrf_grid%itimestep,' _______'
!    DO ix=1,3*INT(MAX_WRF_ALARMS/3),3
!      PRINT *,(ix+iy,': ',wrf_alarms(ix+iy), iy=0,2)
!    END DO
!    PRINT *,(iy,': ',wrf_alarms(iy), iy=3*INT(MAX_WRF_ALARMS/3)+1, MAX_WRF_ALARMS)

! What to do with this?
!!!    IF ( wrf_restart_alarm ) lmdz_lafin = .TRUE.

    INQUIRE(FILE='traceur.def', EXIST=existsfile)
    IF (existsfile) THEN
      DO funit=10,100
        INQUIRE(UNIT=funit, OPENED=is_used)
        IF (.NOT. is_used) EXIT
      END DO
      OPEN(UNIT=10, FILE='traceur.def', STATUS='OLD')
      READ(10,*)lmdzmixingratios
      CLOSE(UNIT=10)
    ELSE

      WRITE(message, *)errmsg
      CALL wrf_debug(75, message)
      WRITE(wrf_err_message,*)'  required file traceur.def not found !!'
      CALL wrf_error_fatal(TRIM(wrf_err_message))
    END IF

! Regarding if it is a lowbdy input time
!!
    PRINT *,'  ',wrf_grid%itimestep,' wrf_islowbdyin: ',wrf_islowbdyin
!    IF ((wrf_islowbdyin) .AND. .NOT.wrf_restart_alarm) THEN
    IF ((wrf_islowbdyin)) THEN
      PRINT *,'  LOWBDY time-step!!!!!!! '
      PRINT *,'    before: wsst ', wrf_grid%sst(ip,jp),' lsst: ',sst(lp)
      CALL get_lowbdy(ddimx, ddimy, dimz, wbdyw, ddimxy, 4, 11, klon_glo,            &
        wrf_grid, pctsrf, ftsol, rugos_rst, albedo, sst, tsurf_limit, z0_limit,      &
        alb_limit, rug_glo, pct_glo)
      PRINT *,'    after: wsst ', wrf_grid%sst(ip,jp),' lsst: ',sst(lp)
!      STOP
    END IF

    IF (ALLOCATED(MixingRatioValues)) DEALLOCATE(MixingRatioValues)
    ALLOCATE(MixingRatioValues(ddimxy,dimz,lmdzmixingratios), STAT=ierr)
    IF (ierr /= 0) THEN
      PRINT *,errmsg
      PRINT *,'  ' // subname // ': Problem allocating WRF moisture array!', ierr
    END IF

    IF (ALLOCATED(MixingRatiotendValues)) DEALLOCATE(MixingRatiotendValues)
    ALLOCATE(MixingRatiotendValues(ddimxy,dimz,lmdzmixingratios))

! Transforming from WRF theta temperature to LMDZ temperature (in K)
!!
    wrf_P = wrf_baseP + wrf_perP
    wrf_geopot = wrf_basegeopot + wrf_pergeopot
    wrf_geopot_half(:,wkms:wkme-1,:) = ( wrf_geopot(:,wkms:wkme-1,:) +       &
      wrf_geopot(:,wkms+1:wkme,:))/2.
    wrf_sfcgeopot = wrf_grid%ht*g
! de-staggering winds
!!

    wrfUdestagg(wims:wime-1,:,wjms:wjme-1) = (wrf_U(wims:wime-1,:,wjms:wjme-1) +     &
      wrf_U(wims+1:wime,:,wjms:wjme-1))/2.
    wrfVdestagg(wims:wime-1,:,wjms:wjme-1) = (wrf_V(wims:wime-1,:,wjms:wjme-1) +     &
      wrf_V(wims:wime-1,:,wjms+1:wjme))/2.
    wrfUtenddestagg(wims:wime-1,:,wjms:wjme-1) =                                     &
      (wrf_Utend(wims:wime-1,:,wjms:wjme-1) + wrf_Utend(wims+1:wime,:,wjms:wjme-1))/2.
    wrfVtenddestagg(wims:wime-1,:,wjms:wjme-1) =                                     &
      (wrf_Vtend(wims:wime-1,:,wjms:wjme-1) + wrf_Vtend(wims:wime-1,:,wjms+1:wjme))/2.

!! NO de-stagger in z
!!    wrfWValues = (wrf_w(:,wkms:wkme-1,:)+wrf_w(:,wkms+1:wkme,:))/2.
!!    wrfPERMASSValues = (wrf_permass(:,wkms:wkme-1,:)+wrf_permass(:,wkms+1:wkme,:))/2.
!!    DO k=wkms,wkme
!!      wrfMASSValues(:,k,:)=-(wrfPERMASSVAlues(:,k,:)+wrf_basemass)*wrf_dfulleta(k)/g
!!    END DO
    wrfWValues=wrf_w
    DO iz=wkms,wkme
      wrfMASSValues(:,iz,:) = -(wrf_permass+wrf_basemass)*wrf_dfulleta(iz)/g
    END DO
    wrfZFluxMassValues=wrfWValues*wrfMASSValues

!!    CALL theta_to_temp(THETAVALUES=wrf_T, PRESSVALUES=wrf_P, TEMPVALUES=wrf_Temp     &
!!                  ! Dimension arguments           
!!       &             ,IDS=wids,IDE=wide, JDS=wjds,JDE=wjde, KDS=wkds,KDE=wkde        &
!!       &             ,IMS=wims,IME=wime, JMS=wjms,JME=wjme, KMS=wkms,KME=wkme        &
!!       &             ,IPS=wips,IPE=wipe, JPS=wjps,JPE=wjpe, KPS=wkps,KPE=wkpe        &
!!       &             ,I_START=wi_start,I_END=wi_end                                  &
!!       &             ,J_START=wj_start,J_END=wj_end                                  &
!!       &             ,KTS=wkts, KTE=wkte                                             &
!!       &             ,NUM_TILES=wnum_tiles                                           &
!!       &                                                          )

    CALL theta_to_temp(wrf_T, wrf_P, wrf_Temp                                        &
                  ! Dimension arguments           
       &             ,wids,wide, wjds,wjde, wkds,wkde                                &
       &             ,wims,wime, wjms,wjme, wkms,wkme                                &
       &             ,wips,wipe, wjps,wjpe, wkps,wkpe                                &
       &             ,wi_start,wi_end                                                &
       &             ,wj_start,wj_end                                                &
       &             ,wkts, wkte                                                     &
       &             ,wnum_tiles                                                     &
       &                                                          )

!!    CALL theta_to_temp(THETAVALUES=wrf_Ttend, PRESSVALUES=wrf_P,                     &
!!       &              TEMPVALUES=wrf_Temptend                                        &
!!                  ! Dimension arguments           
!!       &             ,IDS=wids,IDE=wide, JDS=wjds,JDE=wjde, KDS=wkds,KDE=wkde        &
!!       &             ,IMS=wims,IME=wime, JMS=wjms,JME=wjme, KMS=wkms,KME=wkme        &
!!       &             ,IPS=wips,IPE=wipe, JPS=wjps,JPE=wjpe, KPS=wkps,KPE=wkpe        &
!!       &             ,I_START=wi_start,I_END=wi_end                                  &
!!       &             ,J_START=wj_start,J_END=wj_end                                  &
!!       &             ,KTS=wkts, KTE=wkte                                             &
!!       &             ,NUM_TILES=wnum_tiles                                           &
!!       &                                                          )

    CALL theta_to_temp(wrf_Ttend, wrf_P, wrf_Temptend                                  &
                  ! Dimension arguments            
       &             ,wids,wide, wjds,wjde, wkds,wkde                                  &
       &             ,wims,wime, wjms,wjme, wkms,wkme                                  &
       &             ,wips,wipe, wjps,wjpe, wkps,wkpe                                  &
       &             ,wi_start,wi_end                                                  &
       &             ,wj_start,wj_end                                                  &
       &             ,wkts, wkte                                                       &
       &             ,wnum_tiles                                                       &
       &                                                          )

    DO ix=1,ddimx
      DO iy=1,ddimy
        sfcGeopotValues(ddimx*(iy-1)+ix) = wrf_grid%ht(ix,iy)*g
      END DO 
    END DO

    CALL mat_vect(wrf_Temp, dimx, dimy, dimz, wbdyw, TValues)
    CALL mat_vect(wrf_Temptend, dimx, dimy, dimz, wbdyw, TtendValues)
    CALL mat_vect(wrfUdestagg, dimx, dimy, dimz, wbdyw, UValues)
    CALL mat_vect(wrfUtenddestagg, dimx, dimy, dimz, wbdyw, UtendValues)
    CALL mat_vect(wrfVdestagg, dimx, dimy, dimz, wbdyw, VValues)
    CALL mat_vect(wrfVtenddestagg, dimx, dimy, dimz, wbdyw, VtendValues)
!    CALL mat_vect(wrf_U, dimx, dimy, dimz, wbdyw, UValues)
!    CALL mat_vect(wrf_Utend, dimx, dimy, dimz, wbdyw, UtendValues)
!    CALL mat_vect(wrf_V, dimx, dimy, dimz, wbdyw, VValues)
!    CALL mat_vect(wrf_Vtend, dimx, dimy, dimz, wbdyw, VtendValues)
    CALL mat_vect(wrf_P, dimx, dimy, dimz, wbdyw, HalfPressValues)
    CALL mat_vect_zstagg(wrf_fullpres, dimx, dimy, dimz+1, wbdyw, FullPressValues)
    CALL mat_vect(wrf_geopot_half, dimx, dimy, dimz, wbdyw, GeopotValues)
    CALL mat_vect(wrfZFluxMassValues, dimx, dimy, dimz, wbdyw, WFluxMassValues)

    CALL WRFmoist_LMDZmoist(wrf_MOIST, dimx, dimy, dimz, wnum3dm, wbdyw, wrf_qvid,   &
       & wrf_qcid, wrf_qrid, lmdzmixingratios, MixingRatioValues)

    CALL WRFmoist_LMDZmoist(wrf_MOISTtend, dimx, dimy, dimz, wnum3dm, wbdyw, wrf_qvid,&
       & wrf_qcid, wrf_qrid, lmdzmixingratios, MixingRatiotendValues)

    preff=100000.
    CALL eta_to_pressure(wrf_halfeta, preff, wrf_Ptop, dimz, lmdzoutP)
! L. Fita, LMD October 2014. LMDZ geopotential values are relative to the surface 
    DO iz=1, dimz
      GeopotValues(:,iz) = GeopotValues(:,iz) - sfcGeopotValues(:)
    END DO

! We need to provide values to these variables in order to make it works
    presnivs=lmdzoutP

! Lluis Testing water species
    MixingRatioValues(:,:,2)=0.

    IF (wrf_dbg >= 200) THEN

!! Values at a given LMDZ matrix point
      lmdzp=lp
      jlmdz=INT(lmdzp/ddimx) + 1
      ilmdz=lmdzp-(jlmdz-1)*ddimx
!! Values at a given WRF 2D point
!      ilmdz=ip
!      jlmdz=jp
!      lmdzp=ddimx*(jlmdz-1)+ilmdz

      PRINT *,'  WRF+LMDZ lmdzp: ',lmdzp,' : ',ilmdz,jlmdz
      PRINT *,'k wrf_T wrf_Temp TValues_______________'
      DO iz=1,dimz
        PRINT *,iz, wrf_T(ilmdz,iz,jlmdz), wrf_Temp(ilmdz,iz,jlmdz), TValues(lmdzp,iz)
      END DO
      PRINT *,'k wrf_U Uvalues wrf_V VValues _______________'
      DO iz=1,dimz
        PRINT *,iz, wrf_U(ilmdz,iz,jlmdz), UValues(lmdzp,iz), wrf_V(ilmdz,iz,jlmdz), &
          VValues(lmdzp,iz)
      END DO
      PRINT *,'k wrf_fullpres FUllPressValues wrf_sfcgeopot hgt sfcGeopotValues______'
      DO iz=1,dimz+1
        PRINT *,iz,wrf_fullpres(ilmdz,iz,jlmdz), FullPressValues(lmdzp,iz),          &
          wrf_sfcgeopot(ilmdz,jlmdz), wrf_grid%ht(ilmdz,jlmdz), sfcGeopotValues(lmdzp)
      END DO
      PRINT *,'k wrf_halfpress HalfPressValues wrf_Geopot GeopotValues lmdz outP ' //&
        '_______________'
      DO iz=1,dimz
        PRINT *,iz, wrf_P(ilmdz,iz,jlmdz), HalfPressValues(lmdzp,iz),                &
          wrf_geopot(ilmdz,iz,jlmdz), GeopotValues(lmdzp,iz), lmdzoutP(iz)
      END DO
      PRINT *,'k wrf_MOIST MixingRatioValues _______________'
      DO iz=1,dimz
        PRINT *,iz,wrf_MOIST(ilmdz,iz,jlmdz,wrf_qvid), MixingRatioValues(lmdzp,iz,1),&
          wrf_MOIST(ix,iz,jlmdz,wrf_qcid), MixingRatioValues(lmdzp,iz,2)
      END DO
      PRINT *,'k wrf_mass wrf_W wrf_dfulleta wrfMASSValues wrfZFluxMassValues ' //   &
        'WFluxMassValues ____________'
      DO iz=1,dimz
        PRINT *,iz,wrf_permass(ilmdz,jlmdz)+wrf_basemass(ilmdz,jlmdz),               &
          wrf_W(ilmdz,iz,jlmdz), wrf_dfulleta(iz), wrfMASSValues(ilmdz,iz,jlmdz),    &
          wrfZFluxMassValues(ilmdz,iz,jlmdz), WFluxMassValues(lmdzp,iz)
      END DO
    END IF

    ip=icheck_p
    jp=jcheck_p

!! L. Fita, July 2013. 
!! Misteriously 'physiq' suborutine from LMDZ can not be called using keys
!!
!!    CALL physiq(NLON=dimxy, NLEV=dimz,                                               &
!!       &  DEBUT=lmdz_debut, LAFIN=lmdz_lafin,                                        &
!!       &  JD_CUR=wjulday, JH_CUR=wgmt, PDTPHYS=wtime_step,                           &
!!       &  PAPRS=FullPressValues, PPLAY=HalfPressValues,                              &
!!       &  PPHI=GeopotValues, PPHIS=wrf_sfcgeopot,                                    &
!!       &  PRESNIVS=lmdzoutP, CLESPHY0=phykeys,                                       &
!!       &  U=UValues, V=VValues, T=TValues, QX=MixingRatioValues,                     &
!!       &  FLXMASS_W=WFluxMassValues,                                                 &
!!       &  D_U=UtendValues, D_V=VtendValues, D_T=TtendValues,                         &
!!       &  D_QX=MixingRatiotendValues, D_PS=wrf_psfctend,                             &
!!       &  DUDYN=lmdz_dudyn, PVTETA=lmdz_PVtheta)

!!    PRINT *,'  Lluis: going to physiq'

    CALL physiq(ddimxy, dimz,                                                        &
       &  lmdz_debut, lmdz_lafin,                                                    &
       &  wjulday, wgmt, wtime_step,                                                 &
       &  FullPressValues, HalfPressValues,                                          &
       &  GeopotValues, sfcGeopotValues,                                             &
       &  lmdzoutP, phykeys,                                                         &
       &  UValues, VValues, TValues, MixingRatioValues,                              &
       &  WFluxMassValues,                                                           &
       &  UtendValues, VtendValues, TtendValues,                                     &
       &  MixingRatiotendValues, psfctendValues,                                     &
       &  lmdz_dudyn, lmdz_PVtheta)

    ip=icheck_p
    jp=jcheck_p
! Lluis Testing water species
!!    MixingRatiotendValues(:,:,1)=1000.*MixingRatiotendValues(:,:,1)
!!    MixingRatiotendValues(:,:,1)=MixingRatiotendValues(:,:,1)+MixingRatiotendValues(:,:,2)
    MixingRatiotendValues(:,:,2)=0.

!!    PRINT *,'  Lluis: outgoing from physiq'

!! Tvalues, [U/V/T]Values, are not given back just inputs
    CALL vect_mat(TtendValues, dimx, dimy, dimz, wbdyw, wrf_Temptend)
    CALL vect_mat(UtendValues, dimx, dimy, dimz, wbdyw, wrfUtenddestagg)
    CALL vect_mat(VtendValues, dimx, dimy, dimz, wbdyw, wrfVtenddestagg)
!    CALL vect_mat(UtendValues, dimx, dimy, dimz, wbdyw, wrf_Utend)
!    CALL vect_mat(VtendValues, dimx, dimy, dimz, wbdyw, wrf_Vtend)
!!    CALL vect_mat(TValues, dimx, dimy, dimz, wbdyw, wrf_Temp)
!!    CALL vect_mat(UValues, dimx, dimy, dimz, wbdyw, wrf_U)
!!    CALL vect_mat(VValues, dimx, dimy, dimz, wbdyw, wrf_V)
!!    CALL vect_mat(HalfPressValues, dimx, dimy, dimz, wbdyw, wrf_P)
!!    CALL vect_mat(FullPressValues, dimx, dimy, dimz, wbdyw, wrf_fullpres)
!!    CALL vect_mat(GeopotValues, dimx, dimy, dimz, wbdyw, wrf_geopot)

    DO iy=1,ddimy
      DO ix=1,ddimx
        wrf_psfctend(ix,iy) = PsfctendValues(ddimx*(iy-1)+ix)
      END DO
    END DO

!! No changes are suffered not need to give it back
!!    CALL temp_to_theta(TEMPVALUES=wrf_Temp, PRESSVALUES=wrf_P, THETAVALUES=wrf_T     &
!!                  ! Dimension arguments           
!!       &             ,IDS=wids,IDE=wide, JDS=wjds,JDE=wjde, KDS=wkds,KDE=wkde        &
!!       &             ,IMS=wims,IME=wime, JMS=wjms,JME=wjme, KMS=wkms,KME=wkme        &
!!       &             ,IPS=wips,IPE=wipe, JPS=wjps,JPE=wjpe, KPS=wkps,KPE=wkpe        &
!!       &             ,I_START=wi_start,I_END=wi_end                                  &
!!       &             ,J_START=wj_start,J_END=wj_end                                  &
!!       &             ,KTS=wkts, KTE=wkte                                             &
!!       &             ,NUM_TILES=wnum_tiles                                           &
!!       &                                                          )

    CALL temp_to_theta(TEMPVALUES=wrf_Temptend, PRESSVALUES=wrf_P,                   &
       &              THETAVALUES=wrf_Ttend                                          &
                  ! Dimension arguments           
       &             ,IDS=wids,IDE=wide, JDS=wjds,JDE=wjde, KDS=wkds,KDE=wkde        &
       &             ,IMS=wims,IME=wime, JMS=wjms,JME=wjme, KMS=wkms,KME=wkme        &
       &             ,IPS=wips,IPE=wipe, JPS=wjps,JPE=wjpe, KPS=wkps,KPE=wkpe        &
       &             ,I_START=wi_start,I_END=wi_end                                  &
       &             ,J_START=wj_start,J_END=wj_end                                  &
       &             ,KTS=wkts, KTE=wkte                                             &
       &             ,NUM_TILES=wnum_tiles                                           &
       &                                                          )

    wrf_Ttend = wrf_Ttend + 300.

! Re-staggering wind tendencies
!!

    wrf_Utend(wims,:,wjms:wjme-1) = wrfUtenddestagg(wims,:,wjms:wjme-1)
    wrf_Vtend(wims:wime-1,:,wjms) = wrfVtenddestagg(wims:wime-1,:,wjms)
    wrf_Utend(wims,:,wjme-1) = wrfUtenddestagg(wims,:,wjme-1)
    wrf_Vtend(wime-1,:,wjms) = wrfVtenddestagg(wime-1,:,wjms)
    wrf_Utend(wims+1:wime-1,:,wjms:wjme-1) =                                         &
       &  (wrfUtenddestagg(wims:wime-2,:,wjms:wjme-1) +                              &
       &   wrfUtenddestagg(wims+1:wime-1,:,wjms:wjme-1))/2.
    wrf_Vtend(wims:wime-1,:,wjms+1:wjme-1) =                                         &
       & (wrfVtenddestagg(wims:wime-1,:,wjms:wjme-2) +                               &
       &   wrfVtenddestagg(wims:wime-1,:,wjms+1:wjme-1))/2.
    wrf_Utend(wime,:,wjms:wjme-1) = wrfUtenddestagg(wime-1,:,wjms:wjme-1)
    wrf_Vtend(wims:wime-1,:,wjme) = wrfVtenddestagg(wims:wime-1,:,wjme-1)
    wrf_Utend(wime,:,wjme) = wrfUtenddestagg(wime-1,:,wjme-1)
    wrf_Vtend(wime,:,wjme) = wrfVtenddestagg(wime-1,:,wjme-1)

!!    wrf_Ttend = wrf_Temptend

!!    wrf_perP = wrf_P - wrf_baseP
!!    wrf_pergeopot = wrf_geopot - wrf_basegeopot

!!    CALL LMDZmoist_WRFmoist(MixingRatioValues, dimx, dimy, dimz, wnum3dm, wrf_qvid,  &
!!       & wrf_qcid, wrf_qrid, lmdzmixingratios, wrf_MOIST)

    CALL LMDZmoist_WRFmoist(MixingRatiotendValues, dimx, dimy, dimz, wnum3dm, wbdyw, &
       & wrf_qvid, wrf_qcid, wrf_qrid, lmdzmixingratios, wrf_MOISTtend)

    IF (wrf_dbg >= 200) THEN
      PRINT *,'  WRF+LMDZ lmdzp: ',lmdzp
      PRINT *,'k T tendencies _______________'
      DO iz=1,dimz
        PRINT *,iz, TtendValues(lmdzp,iz), wrf_Temptend(ilmdz,iz,jlmdz),             &
          wrf_Ttend(ilmdz,iz,jlmdz)
      END DO
      PRINT *,'k U tendencies V tendencies_______________'
      DO iz=1,dimz
        PRINT *,iz, UtendValues(lmdzp,iz), wrf_Utend(ilmdz,iz,jlmdz),                &
          VtendValues(lmdzp,iz), wrf_Vtend(ilmdz,iz,jlmdz)
      END DO
      PRINT *,'k MixingRatiotendValues wrf_MOISTtend _______________'
      DO iz=1,dimz
        PRINT *,iz, MixingRatiotendValues(lmdzp,iz,1),                               &
          wrf_MOISTtend(ilmdz,iz,jlmdz,wrf_qvid), MixingRatiotendValues(lmdzp,iz,2), &
          wrf_MOISTtend(ilmdz,iz,jlmdz,wrf_qcid)
      END DO
      PRINT *,'wrf_sfcgeopot psfc_tendencies sfcGeopotValues_______________'
      PRINT *,wrf_sfcgeopot(ilmdz,jlmdz), wrf_psfctend(ilmdz,jlmdz),                 &
        sfcGeopotValues(lmdzp)
    END IF

    PRINT *,'  Lluis psfctendValues ______________ '
    DO ix=1,4*INT(ddimxy/4),4
      PRINT *,(ix+iy,':',psfctendValues(ix+iy), iy=0,3)
    END DO
    PRINT *,(iy,':',psfctendValues(iy), iy=4*INT(ddimxy/4)+1,ddimxy)

! All the tendencies have to be given as decoupled from the dry air mass column! (see dyn_em/module_em.F:rk_update_scalar) 
!!
    DO iz=1,dimz
      wrf_Ttend(:,iz,:) = wrf_Ttend(:,iz,:)*wrf_mut
      wrf_Utend(:,iz,:) = wrf_Utend(:,iz,:)*wrf_muu
      wrf_Vtend(:,iz,:) = wrf_Vtend(:,iz,:)*wrf_muv
      DO iq=1,wnum3dm
        wrf_MOISTtend(:,iz,:,iq) = wrf_MOISTtend(:,iz,:,iq)*wrf_mut
      END DO
    END DO

    CALL get_lmdz(ddimx,ddimy,dimz,wbdyw,ddimxy,4,11,rlon,rlat,zmasq,                &
      pctsrf,ftsol,ftsoil,zxqsurf,qsurf,falb1,falb2,fevap,zxsnow,radsol,solsw,       &
      sollw,fder,rain_fall,snow_fall,agesno,zmea,zstd,zgam,zthe,zpic,zval,rugoro,    &
      frugs,zmax0,f0,wake_s,wake_cstar,wake_pe,wake_fip,sst,albedo,                  &
      t_ancien,q_ancien,u_ancien,v_ancien,clwcon,rnebcon,ratqs,ema_work1,ema_work2,  &
      wake_deltat,wake_deltaq,fm_therm,entr_therm,detr_therm,phy_glo,wrf_grid)
!    CALL fonte_neige_final(runoff_lic)     

! Some re-start variables
    PRINT *,'   Lluis lrunofflic: ',LBOUND(wrf_grid%lrunofflic), ':',                &
      UBOUND(wrf_grid%lrunofflic),' ruonff_lic: ',UBOUND(runoff_lic),' ddimx: ',     &
      ddimx,' ddimy: ',ddimy

!    PRINT *,'    iy ix ddimx*(iy-1)+ix runoff_lic wrf_grid%lrunofflic_______'
!    DO iy=1,ddimy
!      DO ix=1,ddimx
!        PRINT *,iy,ix,ddimx*(iy-1)+ix
!        wrf_grid%lrunofflic(ix,iy) = runoff_lic(ddimx*(iy-1)+ix)
!        PRINT *,iy,ix,ddimx*(iy-1)+ix,runoff_lic(ddimx*(iy-1)+ix),wrf_grid%lrunofflic(ix,iy)
!      END DO
!    END DO

!    STOP
!!    CALL get_lmdz(ddimx,ddimy,dimz,wbdyw,ddimxy,4,11,zmasq,pctsrf,ftsol,             &
!!        ftsoil,fqsurf,qsol_d,falb1,falb2,fevap,zxsnow,radsol,solsw,sollw,fder,       &
!!        rain_fall,snow_fall,agesno,zmea,zstd,zgam,zthe,zpic,zval,rugoro,             &
!!        frugs,run_off_lic,zmax0,f0,wake_s,wake_cstar,wake_pe,wake_fip,sst,           &
!!        albedo,t_ancien,q_ancien,u_ancien,v_ancien,clwcon,rnebcon,ratqs,             &
!!        ema_work1,ema_work2,wake_deltat,wake_deltaq,fm_therm,entr_therm,             &
!!        detr_therm,phy_glo,wrf_grid)

    PRINT *,'  Lluis after get '//TRIM(Spt)//' ltksoil: ',wrf_grid%ltksoil(ip,:,jp), &
      ' ftsol: ',ftsol(lp,:)

    PRINT *,'  Lluis after get '//TRIM(Spt)//' fqsurf: ',zxqsurf(lp),' qsurf_rst: ', &
        qsurf_rst(lp,:),' smois :',wrf_grid%smois(ip,:,jp),' lqksoil: ',             &
        wrf_grid%lqksoil(ip,:,jp)

! Recomputig again the total roughness length
    LMDZvarmethod = 'prod'

    CALL wrf_varKsoil(wims,wime,wjms,wjme,dimx,dimy,wbdyw,LMDZvarmethod,             &
      wrf_grid%lter,wrf_grid%llic,wrf_grid%loce,wrf_grid%lsic,                       &
      wrf_grid%lrugksoil(:,1,:),wrf_grid%lrugksoil(:,2,:),wrf_grid%lrugksoil(:,3,:), &
      wrf_grid%lrugksoil(:,4,:),wrf_grid%lrug)

    CALL wrf_varKsoil(wims,wime,wjms,wjme,dimx,dimy,wbdyw,LMDZvarmethod,             &
      wrf_grid%lter,wrf_grid%llic,wrf_grid%loce,wrf_grid%lsic,                       &
      wrf_grid%lalbksoil(:,1,:),wrf_grid%lalbksoil(:,2,:),wrf_grid%lalbksoil(:,3,:), &
      wrf_grid%lalbksoil(:,4,:),wrf_grid%albedo)

! L. Fita, LMD January 2014. 
!  It should work nicely in this way... but memory got crazy...
    CALL get_lmdz_out(ddimx, ddimy, dimz, wbdyw, ddimxy, nbsrf, nsoilmx,             &
       & lmdzmixingratios, 1, 2, wrf_l_pbl, wrf_l_con, wrf_l_thermals, wrf_l_wake,   &
       & 0, wrf_grid%itimestep, wtime_step, FullPressValues, HalfPressValues,        &
       & GeopotValues, sfcGeopotValues, lmdzoutP, MixingRatioValues,                 &
       & UtendValues, VtendValues, TtendValues, MixingRatiotendValues,               &
       & psfctendValues, wrf_grid)

    ip=icheck_p
    jp=jcheck_p

    CALL put_lmdz_in_WRFout(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,  &
       & wbdyw, ddimxy, ip, jp, lp, nbsrf, nsoilmx,                                  &
       & lmdzmixingratios, 1, 2, wrf_l_pbl, wrf_l_con, wrf_l_thermals, wrf_l_wake,   &
       & wrf_grid%itimestep, wtime_step, FullPressValues, HalfPressValues,           &
       & GeopotValues, sfcGeopotValues, lmdzoutP, MixingRatioValues,                 &
       & UtendValues, VtendValues, TtendValues, MixingRatiotendValues,               &
       & psfctendValues, ORCHIDEElevels, ftsoil, WRF_nsoillayers,                    &
       & wrf_grid%soil_layers/100., wrf_grid%zs, wrf_grid%dzs, wrf_grid%nest_pos,    &
       & wrf_grid%q2, wrf_grid%t2, wrf_grid%th2, wrf_grid%psfc, wrf_grid%u10,        &
       & wrf_grid%v10, wrf_grid%resm, wrf_grid%zetatop, wrf_grid%tslb,wrf_grid%smois,&
       & wrf_grid%sh2o, wrf_grid%smcrel, wrf_grid%sfcrunoff, wrf_grid%udrunoff,      &
       & wrf_grid%grdflx, wrf_grid%acgrdflx, wrf_grid%snow, wrf_grid%snowh,          &
       & wrf_grid%canwat, wrf_grid%sst, wrf_grid%sstsk, wrf_grid%lai,                &
       & wrf_grid%h_diabatic, wrf_grid%sina, wrf_grid%tsk, wrf_grid%tiso,            &
       & wrf_grid%max_msftx, wrf_grid%max_msfty, wrf_grid%rainc, wrf_grid%raincv,    &
       & wrf_grid%rainsh, wrf_grid%rainshv, wrf_grid%rainnc, wrf_grid%rainncv,       &
       & wrf_grid%snownc, wrf_grid%snowncv, wrf_grid%graupelnc, wrf_grid%graupelncv, &
       & wrf_grid%hailnc, wrf_grid%hailncv, wrf_grid%stepave_count, wrf_grid%cldfra, &
       & wrf_grid%swdown, wrf_grid%glw, wrf_grid%swnorm, wrf_grid%olr,               &
       & wrf_grid%emiss, wrf_grid%tmn, wrf_grid%ust, wrf_grid%pblh, wrf_grid%hfx,    &
       & wrf_grid%qfx, wrf_grid%lh, wrf_grid%achfx, wrf_grid%aclhf,                  &
       & wrf_grid%snowc, wrf_grid%sr, wrf_grid%potevp, wrf_grid%snopcx,              &
       & wrf_grid%soiltb)
!       & wrf_grid%save_topo_from_real,                                               &
!       & wrf_grid%avg_fuel_frac, wrf_grid%uah, wrf_grid%vah, wrf_grid%seed1,         &
!       & wrf_grid%seed2)

    ip=icheck_p
    jp=jcheck_p


! So it is done in a more larger to type and less elegant way...
!
!    CALL get_lmdz_out2D_i(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,    &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,      &
!       & wrf_l_wake, wrf_grid%itimestep, wtime_step, FullPressValues2,               &
!       & HalfPressValues2, GeopotValues2,                                            &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2, wrf_grid%lages_ter,                &
!       & wrf_grid%lages_lic, wrf_grid%lages_oce, wrf_grid%lages_sic, wrf_grid%laire, &
!       & wrf_grid%laireter, wrf_grid%lalb1, wrf_grid%lalb2, wrf_grid%lalbe_ter,      &
!       & wrf_grid%lalbe_lic, wrf_grid%lalbe_oce, wrf_grid%lalbe_sic, wrf_grid%lale,  &
!       & wrf_grid%lale_bl, wrf_grid%lale_wk, wrf_grid%lalp, wrf_grid%lalp_bl,        &
!       & wrf_grid%lalp_wk, wrf_grid%lbils, wrf_grid%lbils_diss, wrf_grid%lbils_ec,   &
!       & wrf_grid%lbils_enthalp, wrf_grid%lbils_kinetic, wrf_grid%lbils_latent,      &
!       & wrf_grid%lbils_tke, wrf_grid%lcape, wrf_grid%lcape_max, wrf_grid%lcdrh,     &
!       & wrf_grid%lcdrm, wrf_grid%lcldh, wrf_grid%lcldl, wrf_grid%lcldm,             &
!       & wrf_grid%lcldq, wrf_grid%lcldt, wrf_grid%lcontfracatm, wrf_grid%lcontfracor,&
!       & wrf_grid%ldthmin, wrf_grid%ldtsvdft, wrf_grid%ldtsvdfo, wrf_grid%ldtsvdfg,  &
!       & wrf_grid%ldtsvdfi)

!    CALL get_lmdz_out2D_ii(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,   &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,      &
!       & wrf_l_wake, wrf_grid%itimestep, wtime_step, FullPressValues2,               &
!       & HalfPressValues2, GeopotValues2,                                            &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2, wrf_grid%levap, wrf_grid%levap_ter,&
!       & wrf_grid%levap_lic,                                                         &
!       & wrf_grid%levap_oce, wrf_grid%levap_sic, wrf_grid%levappot_ter,              &
!       & wrf_grid%levappot_lic, wrf_grid%levappot_oce, wrf_grid%levappot_sic,        & 
!       & wrf_grid%lfbase, wrf_grid%lffonte, wrf_grid%lflat, wrf_grid%lflw_ter,       &
!       & wrf_grid%lflw_lic, wrf_grid%lflw_oce, wrf_grid%lflw_sic,                    &
!       & wrf_grid%lfqcalving, wrf_grid%lfqfonte, wrf_grid%lfract_ter,                &
!       & wrf_grid%lfract_lic, wrf_grid%lfract_oce, wrf_grid%lfract_sic,              & 
!       & wrf_grid%lfsnow, wrf_grid%lfsw_ter, wrf_grid%lfsw_lic, wrf_grid%lfsw_oce,   &
!       & wrf_grid%lfsw_sic, wrf_grid%lftime_th, wrf_grid%liwp, wrf_grid%llat_ter,    &
!       & wrf_grid%llat_lic, wrf_grid%llat_oce, wrf_grid%llat_sic, wrf_grid%llmaxth,  &
!       & wrf_grid%llwp, wrf_grid%lmsnow)

!    CALL get_lmdz_out2D_iii(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,  &
!       & wbdyw, ddimxy, ip, jp, lp, nbsrf, nsoilmx, lmdzmixingratios, 1, 2,          &
!       & wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,                             &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,wrf_grid%lndayrain,wrf_grid%lnettop,&
!       & wrf_grid%lpbase, wrf_grid%lphis, wrf_grid%lplcl, wrf_grid%lplfc,            &
!       & wrf_grid%lpluc, wrf_grid%lplul, wrf_grid%lplulst, wrf_grid%lplulth,         &
!       & wrf_grid%lpourc_ter, wrf_grid%lpourc_lic, wrf_grid%lpourc_oce,              &
!       & wrf_grid%lpourc_sic, wrf_grid%lprecip, wrf_grid%lprw, wrf_grid%lpsol,       &
!       & wrf_grid%lptop, wrf_grid%lqsat2m, wrf_grid%lqsol, wrf_grid%lqsurf,          &
!       & wrf_grid%lradsol, wrf_grid%lrh2m, wrf_grid%lrh2m_max, wrf_grid%lrh2m_min,   &
!       & wrf_grid%lrugs, wrf_grid%lrugs_ter, wrf_grid%lrugs_lic, wrf_grid%lrugs_oce, &
!       & wrf_grid%lrugs_sic, wrf_grid%lsens, wrf_grid%lsicf, wrf_grid%ls_lcl,        &
!       & wrf_grid%lslp, wrf_grid%lsnow, wrf_grid%lsnowl, wrf_grid%lsoll,             &
!       & wrf_grid%lsoll0, wrf_grid%lsollwdown, wrf_grid%lsols, wrf_grid%lsols0,      &
!       & wrf_grid%ls_pblh, wrf_grid%ls_pblt, wrf_grid%ls_therm)

!    CALL get_lmdz_out2D_iv(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,   &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,      &
!       & wrf_l_wake, wrf_grid%itimestep,                                             &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,  wrf_grid%lt2m,                    &
!       & wrf_grid%lt2m_max, wrf_grid%lt2m_min, wrf_grid%lt2m_ter, wrf_grid%lt2m_lic, &
!       & wrf_grid%lt2m_oce, wrf_grid%lt2m_sic, wrf_grid%ltaux, wrf_grid%ltauy,       &
!       & wrf_grid%ltaux_ter, wrf_grid%ltaux_lic, wrf_grid%ltaux_oce,                 &
!       & wrf_grid%ltaux_sic, wrf_grid%ltauy_ter, wrf_grid%ltauy_lic,                 & 
!       & wrf_grid%ltauy_oce, wrf_grid%ltauy_sic, wrf_grid%lt_oce_sic, wrf_grid%ltopl,&
!       & wrf_grid%ltopl0, wrf_grid%ltops, wrf_grid%ltops0, wrf_grid%ltpot,           &
!       & wrf_grid%ltpote, wrf_grid%ltsol, wrf_grid%ltsol_ter, wrf_grid%ltsol_lic,    &
!       & wrf_grid%ltsol_oce, wrf_grid%ltsol_sic)

!    CALL get_lmdz_out2D_v(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,    &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,      &
!       & wrf_l_wake, wrf_grid%itimestep,                                             &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,wrf_grid%lu10m, wrf_grid%lu10m_ter, &
!       & wrf_grid%lu10m_lic, wrf_grid%lu10m_oce, wrf_grid%lu10m_sic, wrf_grid%lue,   &
!       & wrf_grid%luq, wrf_grid%lustar, wrf_grid%lustar_ter, wrf_grid%lustar_lic,    &
!       & wrf_grid%lustar_oce, wrf_grid%lustar_sic, wrf_grid%lv10m,                   &
!       & wrf_grid%lv10m_ter, wrf_grid%lv10m_lic, wrf_grid%lv10m_oce,                 &
!       & wrf_grid%lv10m_sic, wrf_grid%lve, wrf_grid%lvq, wrf_grid%lwake_h,           &
!       & wrf_grid%lwake_s, wrf_grid%lwbeff, wrf_grid%lwbilo_ter, wrf_grid%lwbilo_lic,&
!       & wrf_grid%lwbilo_oce, wrf_grid%lwbilo_sic, wrf_grid%lwbils_ter,              &
!       & wrf_grid%lwbils_lic, wrf_grid%lwbils_oce, wrf_grid%lwbils_sic,              &
!       & wrf_grid%lweakinv, wrf_grid%lwind10m, wrf_grid%lwind10max, wrf_grid%lzmax_th)

!    DO iz=wkms, wkme-2
!      CALL get_lmdz_out3D_z_i(wims, wime, wjms, wjme, iz, ddimx, ddimy, dimz, wbdyw, &
!        & ddimxy, ip, jp, lp, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_con,     &
!        & wrf_l_thermals, wrf_l_wake,                                                &
!        & wrf_grid%itimestep,                                                        &
!        & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,             &
!        & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                             &
!        & UtendValues2, VtendValues2, TtendValues2,                                  &
!        & MixingRatiotendValues2, psfctendValues2,                                   &
!        & wrf_grid%la_th(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lbeta_prec(wims:wime,iz,wjms:wjme),                               &
!        & wrf_grid%ldmc(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%ldnwd(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%ldnwd0(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqajs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqcon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqdyn(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqeva(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqlsc(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqlscst(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%ldqlscth(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%ldqphy(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqthe(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqvdf(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldqwak(wims:wime,iz,wjms:wjme))

!      CALL get_lmdz_out3D_z_ii(wims, wime, wjms, wjme, iz, ddimx, ddimy, dimz,       &
!        & wbdyw, ddimxy, ip, jp, lp, nbsrf, nsoilmx, lmdzmixingratios, 1, 2,         &
!        & wrf_l_thermals, wrf_l_wake,                                                &
!        & wrf_grid%itimestep,                                                        &
!        & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,             &
!        & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                             &
!        & UtendValues2, VtendValues2, TtendValues2,                                  &
!        & MixingRatiotendValues2, psfctendValues2,                                   &
!        & wrf_grid%ldtajs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtcon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtdis(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtdyn(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtec(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%ldteva(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ld_th(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lcldemi(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%lcldtau(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%lclwcon(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%ldtlif(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtlsc(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtlschr(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%ldtlscst(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%ldtlscth(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%ldtlw0(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtlwr(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtoro(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtphy(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtsw0(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtswr(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtthe(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtvdf(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldtwak(wims:wime,iz,wjms:wjme))

!      CALL get_lmdz_out3D_z_iii(wims, wime, wjms, wjme, iz, ddimx, ddimy,            &
!        & dimz, wbdyw, ddimxy, ip, jp, lp, nbsrf, nsoilmx, lmdzmixingratios, 1, 2,   &
!        & wrf_l_con, wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,                 &
!        & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,             &
!        & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                             &
!        & UtendValues2, VtendValues2, TtendValues2,                                  &
!        & MixingRatiotendValues2, psfctendValues2,                                   &
!        & wrf_grid%lducon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldudyn(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldulif(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lduoro(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lduvdf(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldvcon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldvdyn(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldvlif(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldvoro(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%ldvvdf(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%le_th(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%levu(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lfl(wims:wime,iz,wjms:wjme),                                      &
!        & wrf_grid%lgeop(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lh2o(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%liwcon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lkz(wims:wime,iz,wjms:wjme),                                      &
!        & wrf_grid%lkz_max(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%llambda_th(wims:wime,iz,wjms:wjme),                               &
!        & wrf_grid%llwcon(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lma(wims:wime,iz,wjms:wjme),                                      &
!        & wrf_grid%lmc(wims:wime,iz,wjms:wjme),                                      &
!        & wrf_grid%lmcd(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lmass(wims:wime,iz,wjms:wjme))

!     CALL get_lmdz_out3D_z_iv(wims, wime, wjms, wjme, iz, ddimx, ddimy, dimz,        &
!        & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,     &
!        & wrf_l_wake, wrf_grid%itimestep,                                            &
!        & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,             &
!        & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                             &
!        & UtendValues2, VtendValues2, TtendValues2,                                  &
!        & MixingRatiotendValues2, psfctendValues2,                                   &
!        & wrf_grid%loliq(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lovap(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lovapinit(wims:wime,iz,wjms:wjme),                                &
!        & wrf_grid%lpaprs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lpr_con_i(wims:wime,iz,wjms:wjme),                                &
!        & wrf_grid%lpr_con_l(wims:wime,iz,wjms:wjme),                                &
!        & wrf_grid%lpres(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lpr_lsc_i(wims:wime,iz,wjms:wjme),                                &
!        & wrf_grid%lpr_lsc_l(wims:wime,iz,wjms:wjme),                                &
!        & wrf_grid%lptconv(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%lq_th(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lratqs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lre(wims:wime,iz,wjms:wjme),                                      &
!        & wrf_grid%lref_ice(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%lref_liq(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%lrhum(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lrld(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lrldcs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lrlu(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lrlucs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lrneb(wims:wime,iz,wjms:wjme),                                    &
!        & wrf_grid%lrnebcon(wims:wime,iz,wjms:wjme),                                 &
!        & wrf_grid%lrnebls(wims:wime,iz,wjms:wjme),                                  &
!        & wrf_grid%lrsd(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lrsdcs(wims:wime,iz,wjms:wjme),                                   &
!        & wrf_grid%lrsu(wims:wime,iz,wjms:wjme),                                     &
!        & wrf_grid%lrsucs(wims:wime,iz,wjms:wjme))

!     CALL get_lmdz_out3D_z_v(wims, wime, wjms, wjme, iz, ddimx, ddimy, dimz,         &
!       & wbdyw, ddimxy, ip, jp, lp, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, 1,       &
!       & wrf_l_con, wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,                  &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,                                    &
!       & wrf_grid%lsens_ter(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lsens_lic(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lsens_oce(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lsens_sic(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%ltemp(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%ltheta(wims:wime,iz,wjms:wjme),                                    &
!       & wrf_grid%ltke(wims:wime,iz,wjms:wjme),                                      &
!       & wrf_grid%ltke_max(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%ltke_ter(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%ltke_lic(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%ltke_oce(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%ltke_sic(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%ltnhus(wims:wime,iz,wjms:wjme),                                    &
!       & wrf_grid%ltnhusc(wims:wime,iz,wjms:wjme),                                   &
!       & wrf_grid%ltnhusscpbl(wims:wime,iz,wjms:wjme),                               &
!       & wrf_grid%ltnt(wims:wime,iz,wjms:wjme),                                      &
!       & wrf_grid%ltntc(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%ltntr(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%ltntscpbl(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lupwd(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%lvitu(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%lvitv(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%lvitw(wims:wime,iz,wjms:wjme),                                     &
!       & wrf_grid%lvprecip(wims:wime,iz,wjms:wjme),                                  &
!       & wrf_grid%lwake_omg(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lwdtraina(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lwdtrainm(wims:wime,iz,wjms:wjme),                                 &
!       & wrf_grid%lzfull(wims:wime,iz,wjms:wjme),                                    &
!       & wrf_grid%lzhalf(wims:wime,iz,wjms:wjme))

!    END DO

!    CALL get_lmdz_out3D_ii(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,   &
!      & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals,       &
!        & wrf_l_wake, wrf_grid%itimestep,      &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                              &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,                                    &
!       & wrf_grid%ldtajs,                                                            &
!       & wrf_grid%ldtcon, wrf_grid%ldtdis, wrf_grid%ldtdyn, wrf_grid%ldtec,          &
!       & wrf_grid%ldteva, wrf_grid%ld_th, wrf_grid%lcldemi, wrf_grid%lcldtau,        &
!       & wrf_grid%lclwcon, wrf_grid%ldtlif, wrf_grid%ldtlsc, wrf_grid%ldtlschr,      &
!       & wrf_grid%ldtlscst, wrf_grid%ldtlscth, wrf_grid%ldtlw0, wrf_grid%ldtlwr,     &
!       & wrf_grid%ldtoro, wrf_grid%ldtphy, wrf_grid%ldtsw0, wrf_grid%ldtswr,         &
!       & wrf_grid%ldtthe, wrf_grid%ldtvdf, wrf_grid%ldtwak)

!    CALL get_lmdz_out3D_iii(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,  &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,     &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                                &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,                                    &
!       & wrf_grid%lducon,                                                            &
!       & wrf_grid%ldudyn, wrf_grid%ldulif, wrf_grid%lduoro, wrf_grid%lduvdf,         &
!       & wrf_grid%ldvcon, wrf_grid%ldvdyn, wrf_grid%ldvlif, wrf_grid%ldvoro,         &
!       & wrf_grid%ldvvdf, wrf_grid%le_th, wrf_grid%levu, wrf_grid%lfl,wrf_grid%lgeop,&
!        & wrf_grid%lh2o, wrf_grid%liwcon, wrf_grid%lkz, wrf_grid%lkz_max,             &
!        & wrf_grid%llambda_th, wrf_grid%llwcon, wrf_grid%lma, wrf_grid%lmass)

!     CALL get_lmdz_out3D_iv(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,   &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,     &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                                &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,                                    &
!       & wrf_grid%loliq, wrf_grid%lovap,                                             &
!       & wrf_grid%lovapinit, wrf_grid%lpaprs, wrf_grid%lpr_con_i, wrf_grid%lpr_con_l,&
!       & wrf_grid%lpres, wrf_grid%lpr_lsc_i, wrf_grid%lpr_lsc_l, wrf_grid%lptconv,   &
!       & wrf_grid%lq_th, wrf_grid%lratqs, wrf_grid%lre, wrf_grid%lref_ice,           &
!       & wrf_grid%lref_liq, wrf_grid%lrhum, wrf_grid%lrld, wrf_grid%lrldcs,          &
!       & wrf_grid%lrlu, wrf_grid%lrlucs, wrf_grid%lrneb, wrf_grid%lrnebcon,          &
!       & wrf_grid%lrnebls, wrf_grid%lrsd, wrf_grid%lrsdcs, wrf_grid%lrsu,            &
!       & wrf_grid%lrsucs)

!    CALL get_lmdz_out3D_v(wims, wime, wjms, wjme, wkms, wkme, ddimx, ddimy, dimz,    &
!       & wbdyw, ddimxy, nbsrf, nsoilmx, lmdzmixingratios, 1, 2, wrf_l_thermals, wrf_l_wake, wrf_grid%itimestep,     &
!       & wtime_step, FullPressValues2, HalfPressValues2, GeopotValues2,              &
!       & sfcGeopotValues, lmdzoutP, MixingRatioValues2,                                &
!       & UtendValues2, VtendValues2, TtendValues2,                                   &
!       & MixingRatiotendValues2, psfctendValues2,                                    &
!       & wrf_grid%lsens_ter, wrf_grid%lsens_lic, wrf_grid%lsens_oce,                 &
!       & wrf_grid%lsens_sic, wrf_grid%ltemp, wrf_grid%ltheta, wrf_grid%ltke,         &
!       & wrf_grid%ltke_max, wrf_grid%ltke_ter, wrf_grid%ltke_lic, wrf_grid%ltke_oce, &
!       & wrf_grid%ltke_sic, wrf_grid%ltnhus, wrf_grid%ltnhusc, wrf_grid%ltnhusscpbl, &
!       & wrf_grid%ltnt, wrf_grid%ltntc, wrf_grid%ltntr, wrf_grid%ltntscpbl,          &
!       & wrf_grid%lupwd, wrf_grid%lvitu, wrf_grid%lvitv, wrf_grid%lvitw,             &
!       & wrf_grid%lwake_deltaq, wrf_grid%lwake_deltat, wrf_grid%lwake_omg,           &
!       & wrf_grid%lwdtraina, wrf_grid%lwdtrainm, wrf_grid%lzfull, wrf_grid%lzhalf)

    DEALLOCATE(MixingRatioValues)
    DEALLOCATE(MixingRatiotendValues)

    WRITE(message, *)'  exiting: module_lmdz_phys.F'
    CALL wrf_debug(75, message)

  END SUBROUTINE call_lmdz_phys

END MODULE module_lmdz_phys
