!! Fortran version of different diagnostics
! L. Fita. LMD May 2016
! gfortran module_generic.o module_ForDiagnosticsVars.o -c module_ForDiagnostics.F90
!
! f2py -m module_ForDiagnostics --f90exec=/usr/bin/gfortran-4.7 -c module_generic.F90 module_ForDiagnosticsVars.F90 module_ForDiagnostics.F90

MODULE module_ForDiagnostics

  USE module_definitions
  USE module_generic
  USE module_ForDiagnosticsVars

  CONTAINS

!!!!!!! Calculations
! compute_cllmh4D3: Computation of low, medium and high cloudiness from a 4D CLDFRA and pressure being 
!   3rd dimension the z-dim 
! compute_cllmh3D3: Computation of low, medium and high cloudiness from a 3D CLDFRA and pressure being 
!   3rd dimension the z-dim 
! compute_cllmh: Computation of low, medium and high cloudiness
! compute_clt4D3: Computation of total cloudiness from a 4D CLDFRA being 3rd dimension the z-dim
! compute_clt3D3: Computation of total cloudiness from a 3D CLDFRA being 3rd dimension the z-dim
! compute_clt: Computation of total cloudiness
! compute_psl_ptarget4d2: Compute sea level pressure using a target pressure. Similar to the Benjamin 
!   and Miller (1990). Method found in p_interp.F90
! compute_tv4d: 4D calculation of virtual temperaure
! VirtualTemp1D: Function for 1D calculation of virtual temperaure

!!!
! Calculations
!!!

  SUBROUTINE compute_cllmh4D2(cldfra4D, pres4D, cllmh4D2, d1, d2, d3, d4)
! Subroutine to compute the low, medium and high cloudiness following 'newmicro.F90' from LMDZ from a 4D CLDFRA and pressure 
!   where zdim is the 2nd dimension (thus, cldfra4D(d1,d2,d3,d4) --> cllmh(3,d1,d3,d4) 1: low, 2: medium, 3: high
! It should be properly done via an 'INTERFACE', but...

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3, d4
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(in)        :: cldfra4D, pres4D
    REAL(r_k), DIMENSION(3,d1,d3,d4), INTENT(out)        :: cllmh4D2

! Local
    INTEGER                                              :: i,j,k, zdim, Ndim

!!!!!!! Variables
! cldfra4D: 4D cloud fraction values [1]
! pres4D: 4D pressure values [Pa]
! Ndim: number of dimensions of the input data
! d[1-4]: dimensions of 'cldfra4D'
! zdim: number of the vertical-dimension within the matrix
! cltlmh4D2: low, medium, high cloudiness for the 4D cldfra and d2 being 'zdim'

    fname = 'compute_cllmh4D2'
    zdim = 2
    Ndim = 4

    DO i=1, d1
      DO j=1, d3
        DO k=1, d4
          cllmh4D2(:,i,j,k) = var_cllmh(cldfra4D(i,:,j,k), pres4D(i,:,j,k), d2)
        END DO
      END DO
    END DO
    
    RETURN 

  END SUBROUTINE compute_cllmh4D2

  SUBROUTINE compute_cllmh3D1(cldfra3D, pres3D, cllmh3D1, d1, d2, d3)
! Subroutine to compute the low, medium and high cloudiness following 'newmicro.F90' from LMDZ from a 3D CLDFRA and pressure 
!   where zdim is the 1st dimension (thus, cldfra3D(d1,d2,d3) --> cllmh(3,d2,d3) 1: low, 2: medium, 3: high
! It should be properly done via an 'INTERFACE', but...

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3
    REAL(r_k), DIMENSION(d1,d2,d3), INTENT(in)           :: cldfra3D, pres3D
    REAL(r_k), DIMENSION(3,d2,d3), INTENT(out)           :: cllmh3D1

! Local
    INTEGER                                              :: i,j,k, zdim, Ndim

!!!!!!! Variables
! cldfra3D: 3D cloud fraction values [1]
! pres3D: 3D pressure values [Pa]
! Ndim: number of dimensions of the input data
! d[1-3]: dimensions of 'cldfra3D'
! zdim: number of the vertical-dimension within the matrix
! cltlmh3D1: low, medium, high cloudiness for the 3D cldfra and d1 being 'zdim'

    fname = 'compute_cllmh3D1'
    zdim = 1
    Ndim = 3

    DO i=1, d1
      DO j=1, d2
        cllmh3D1(:,i,j) = var_cllmh(cldfra3D(:,i,j), pres3D(:,i,j), d1)
      END DO
    END DO
    
    RETURN 

  END SUBROUTINE compute_cllmh3D1

  SUBROUTINE compute_cllmh(cldfra1D, cldfra2D, cldfra3D, cldfra4D, pres1D, pres2D, pres3D, pres4D,    &
    Ndim, zdim, cllmh1D, cllmh2D1, cllmh2D2, cllmh3D1, cllmh3D2, cllmh3D3, cllmh4D1, cllmh4D2,        &
    cllmh4D3, cllmh4D4, d1, d2, d3, d4)
! Subroutine to compute the low, medium and high cloudiness following 'newmicro.F90' from LMDZ

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Ndim, d1, d2, d3, d4, zdim
    REAL(r_k), DIMENSION(d1), OPTIONAL, INTENT(in)       :: cldfra1D, pres1D
    REAL(r_k), DIMENSION(d1,d2), OPTIONAL, INTENT(in)    :: cldfra2D, pres2D
    REAL(r_k), DIMENSION(d1,d2,d3), OPTIONAL, INTENT(in) :: cldfra3D, pres3D
    REAL(r_k), DIMENSION(d1,d2,d3,d4), OPTIONAL,                                                      &
      INTENT(in)                                         :: cldfra4D, pres4D
    REAL(r_k), DIMENSION(3), OPTIONAL, INTENT(out)       :: cllmh1D
    REAL(r_k), DIMENSION(d1,3), OPTIONAL, INTENT(out)    :: cllmh2D1
    REAL(r_k), DIMENSION(d2,3), OPTIONAL, INTENT(out)    :: cllmh2D2
    REAL(r_k), DIMENSION(d2,d3,3), OPTIONAL, INTENT(out) :: cllmh3D1
    REAL(r_k), DIMENSION(d1,d3,3), OPTIONAL, INTENT(out) :: cllmh3D2
    REAL(r_k), DIMENSION(d1,d2,3), OPTIONAL, INTENT(out) :: cllmh3D3
    REAL(r_k), DIMENSION(d2,d3,d4,3), OPTIONAL,                                                       &
      INTENT(out)                                        :: cllmh4D1
    REAL(r_k), DIMENSION(d1,d3,d4,3), OPTIONAL,                                                       &
      INTENT(out)                                        :: cllmh4D2
    REAL(r_k), DIMENSION(d1,d2,d4,3), OPTIONAL,                                                       &
      INTENT(out)                                        :: cllmh4D3
    REAL(r_k), DIMENSION(d1,d2,d3,3), OPTIONAL,                                                       &
      INTENT(out)                                        :: cllmh4D4

! Local
    INTEGER                                              :: i,j,k

!!!!!!! Variables
! cldfra[1-4]D: cloud fraction values [1]
! pres[1-4]D: pressure values [Pa]
! Ndim: number of dimensions of the input data
! d[1-4]: dimensions of 'cldfra'
! zdim: number of the vertical-dimension within the matrix
! cllmh1D: low, medium and high cloudiness for the 1D cldfra
! cllmh2D1: low, medium and high cloudiness for the 2D cldfra and d1 being 'zdim'
! cllmh2D2: low, medium and high cloudiness for the 2D cldfra and d2 being 'zdim'
! cllmh3D1: low, medium and high cloudiness for the 3D cldfra and d1 being 'zdim'
! cllmh3D2: low, medium and high cloudiness for the 3D cldfra and d2 being 'zdim'
! cllmh3D3: low, medium and high cloudiness for the 3D cldfra and d3 being 'zdim'
! cllmh4D1: low, medium and high cloudiness for the 4D cldfra and d1 being 'zdim'
! cllmh4D2: low, medium and high cloudiness for the 4D cldfra and d2 being 'zdim'
! cllmh4D3: low, medium and high cloudiness for the 4D cldfra and d3 being 'zdim'
! cllmh4D4: low, medium and high cloudiness for the 4D cldfra and d4 being 'zdim'

    fname = 'compute_cllmh'

    SELECT CASE (Ndim)
      CASE (1)
        cllmh1D = var_cllmh(cldfra1D, pres1D, d1)
      CASE (2)
        IF (zdim == 1) THEN
          DO i=1, d2
            cllmh2D1(i,:) = var_cllmh(cldfra2D(:,i), pres2D(:,i), d1)
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            cllmh2D2(i,:) = var_cllmh(cldfra2D(:,i), pres2D(i,:), d2)
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2'
          STOP
        END IF
      CASE (3)
        IF (zdim == 1) THEN
          DO i=1, d2
            DO j=1, d3
              cllmh3D1(i,j,:) = var_cllmh(cldfra3D(:,i,j), pres3D(:,i,j), d1)
            END DO
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            DO j=1, d3
              cllmh3D2(i,j,:) = var_cllmh(cldfra3D(i,:,j), pres3D(i,:,j), d2)
            END DO
          END DO
        ELSE IF (zdim == 3) THEN
          DO i=1, d1
            DO j=1, d2
              cllmh3D3(i,j,:) = var_cllmh(cldfra3D(i,j,:), pres3D(i,j,:), d3)
            END DO
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2,3'
          STOP
        END IF
      CASE (4)
        IF (zdim == 1) THEN
          DO i=1, d2
            DO j=1, d3
              DO k=1, d4
                cllmh4D1(i,j,k,:) = var_cllmh(cldfra4D(:,i,j,k), pres4D(:,i,j,k), d1)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            DO j=1, d3
              DO k=1, d4
                cllmh4D2(i,j,k,:) = var_cllmh(cldfra4D(i,:,j,k), pres4D(i,:,j,k), d2)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 3) THEN
          DO i=1, d2
            DO j=1, d3
              DO k=1, d4
                cllmh4D3(i,j,k,:) = var_cllmh(cldfra4D(i,j,:,k), pres4D(i,j,:,k), d3)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 4) THEN
          DO i=1, d1
            DO j=1, d2
              DO k=1, d3
                cllmh4D4(i,j,k,:) = var_cllmh(cldfra4D(i,j,k,:), pres4D(i,j,k,:), d4)
              END DO
            END DO
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2,3,4'
          STOP
        END IF
      CASE DEFAULT
        PRINT *,TRIM(ErrWarnMsg('err'))
        PRINT *,'  ' // TRIM(fname) // ': Ndim:', Ndim,' not ready !!'
        STOP
      END SELECT

    RETURN 

  END SUBROUTINE compute_cllmh

  SUBROUTINE compute_clt4D2(cldfra4D, clt4D2, d1, d2, d3, d4)
! Subroutine to compute the total cloudiness following 'newmicro.F90' from LMDZ from a 4D CLDFRA
!   where zdim is the 2nd dimension (thus, cldfra4D(d1,d2,d3,d4) --> clt(d1,d3,d4)
! It should be properly done via an 'INTERFACE', but...

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3, d4
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(in)        :: cldfra4D
    REAL(r_k), DIMENSION(d1,d3,d4), INTENT(out)          :: clt4D2

! Local
    INTEGER                                              :: i,j,k, zdim, Ndim

!!!!!!! Variables
! cldfra4D: 4D cloud fraction values [1]
! Ndim: number of dimensions of the input data
! d[1-4]: dimensions of 'cldfra4D'
! zdim: number of the vertical-dimension within the matrix
! clt4D2: total cloudiness for the 4D cldfra and d2 being 'zdim'

    fname = 'compute_clt4D2'
    zdim = 2
    Ndim = 4

    DO i=1, d1
      DO j=1, d3
        DO k=1, d4
          clt4D2(i,j,k) = var_clt(cldfra4D(i,:,j,k), d2)
        END DO
      END DO
    END DO
    
    RETURN 

  END SUBROUTINE compute_clt4D2

  SUBROUTINE compute_clt3D1(cldfra3D, clt3D1, d1, d2, d3)
! Subroutine to compute the total cloudiness following 'newmicro.F90' from LMDZ from a 3D CLDFRA
!   where zdim is the 1st dimension (thus, cldfra4D(d1,d2,d3) --> clt(d2,d3)
! It should be properly done via an 'INTERFACE', but...

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3
    REAL(r_k), DIMENSION(d1,d2,d3), INTENT(in)           :: cldfra3D
    REAL(r_k), DIMENSION(d2,d3), INTENT(out)             :: clt3D1

! Local
    INTEGER                                              :: i,j,k, zdim, Ndim

!!!!!!! Variables
! cldfra3D: 3D cloud fraction values [1]
! Ndim: number of dimensions of the input data
! d[1-3]: dimensions of 'cldfra3D'
! zdim: number of the vertical-dimension within the matrix
! clt3D1: total cloudiness for the 3D cldfra and d1 being 'zdim'

    fname = 'compute_clt3D1'
    zdim = 1
    Ndim = 3

    DO i=1, d2
      DO j=1, d3
        clt3D1(i,j) = var_clt(cldfra3D(:,i,j), d1)
      END DO
    END DO
    
    RETURN 

  END SUBROUTINE compute_clt3D1

  SUBROUTINE compute_clt(cldfra1D, cldfra2D, cldfra3D, cldfra4D, Ndim, zdim, clt1D, clt2D1, clt2D2,   &
    clt3D1, clt3D2, clt3D3, clt4D1, clt4D2, clt4D3, clt4D4, d1, d2, d3, d4)
! Subroutine to compute the total cloudiness following 'newmicro.F90' from LMDZ

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: Ndim, d1, d2, d3, d4, zdim
    REAL(r_k), DIMENSION(d1), OPTIONAL, INTENT(in)       :: cldfra1D
    REAL(r_k), DIMENSION(d1,d2), OPTIONAL, INTENT(in)    :: cldfra2D
    REAL(r_k), DIMENSION(d1,d2,d3), OPTIONAL, INTENT(in) :: cldfra3D
    REAL(r_k), DIMENSION(d1,d2,d3,d4), OPTIONAL,                                                      &
      INTENT(in)                                         :: cldfra4D
    REAL(r_k), OPTIONAL, INTENT(out)                     :: clt1D
    REAL(r_k), DIMENSION(d1), OPTIONAL, INTENT(out)      :: clt2D1
    REAL(r_k), DIMENSION(d2), OPTIONAL, INTENT(out)      :: clt2D2
    REAL(r_k), DIMENSION(d2,d3), OPTIONAL, INTENT(out)   :: clt3D1
    REAL(r_k), DIMENSION(d1,d3), OPTIONAL, INTENT(out)   :: clt3D2
    REAL(r_k), DIMENSION(d1,d2), OPTIONAL, INTENT(out)   :: clt3D3
    REAL(r_k), DIMENSION(d2,d3,d4), OPTIONAL,INTENT(out) :: clt4D1
    REAL(r_k), DIMENSION(d1,d3,d4), OPTIONAL,INTENT(out) :: clt4D2
    REAL(r_k), DIMENSION(d1,d2,d4), OPTIONAL,INTENT(out) :: clt4D3
    REAL(r_k), DIMENSION(d1,d2,d3), OPTIONAL,INTENT(out) :: clt4D4

! Local
    INTEGER                                              :: i,j,k

!!!!!!! Variables
! cldfra[1-4]D: cloud fraction values [1]
! Ndim: number of dimensions of the input data
! d[1-4]: dimensions of 'cldfra'
! zdim: number of the vertical-dimension within the matrix
! clt1D: total cloudiness for the 1D cldfra
! clt2D1: total cloudiness for the 2D cldfra and d1 being 'zdim'
! clt2D2: total cloudiness for the 2D cldfra and d2 being 'zdim'
! clt3D1: total cloudiness for the 3D cldfra and d1 being 'zdim'
! clt3D2: total cloudiness for the 3D cldfra and d2 being 'zdim'
! clt3D3: total cloudiness for the 3D cldfra and d3 being 'zdim'
! clt4D1: total cloudiness for the 4D cldfra and d1 being 'zdim'
! clt4D2: total cloudiness for the 4D cldfra and d2 being 'zdim'
! clt4D3: total cloudiness for the 4D cldfra and d3 being 'zdim'
! clt4D4: total cloudiness for the 4D cldfra and d4 being 'zdim'

    fname = 'compute_clt'

    SELECT CASE (Ndim)
      CASE (1)
        clt1D = var_clt(cldfra1D, d1)
      CASE (2)
        IF (zdim == 1) THEN
          DO i=1, d2
            clt2D1(i) = var_clt(cldfra2D(:,i), d1)
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            clt2D2(i) = var_clt(cldfra2D(:,i), d2)
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2'
          STOP
        END IF
      CASE (3)
        IF (zdim == 1) THEN
          DO i=1, d2
            DO j=1, d3
              clt3D1(i,j) = var_clt(cldfra3D(:,i,j), d1)
            END DO
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            DO j=1, d3
              clt3D2(i,j) = var_clt(cldfra3D(i,:,j), d2)
            END DO
          END DO
        ELSE IF (zdim == 3) THEN
          DO i=1, d1
            DO j=1, d2
              clt3D3(i,j) = var_clt(cldfra3D(i,j,:), d3)
            END DO
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2,3'
          STOP
        END IF
      CASE (4)
        IF (zdim == 1) THEN
          DO i=1, d2
            DO j=1, d3
              DO k=1, d4
                clt4D1(i,j,k) = var_clt(cldfra4D(:,i,j,k), d1)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 2) THEN
          DO i=1, d1
            DO j=1, d3
              DO k=1, d4
                clt4D2(i,j,k) = var_clt(cldfra4D(i,:,j,k), d2)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 3) THEN
          DO i=1, d2
            DO j=1, d3
              DO k=1, d4
                clt4D3(i,j,k) = var_clt(cldfra4D(i,j,:,k), d3)
              END DO
            END DO
          END DO
        ELSE IF (zdim == 4) THEN
          DO i=1, d1
            DO j=1, d2
              DO k=1, d3
                clt4D4(i,j,k) = var_clt(cldfra4D(i,j,k,:), d4)
              END DO
            END DO
          END DO
        ELSE
          PRINT *,TRIM(ErrWarnMsg('err'))
          PRINT *,'  ' // TRIM(fname) // ': wrong zdim:', zdim,' for Ndim=', Ndim, ' !!'
          PRINT *,'    accepted values: 1,2,3,4'
          STOP
        END IF
      CASE DEFAULT
        PRINT *,TRIM(ErrWarnMsg('err'))
        PRINT *,'  ' // TRIM(fname) // ': Ndim:', Ndim,' not ready !!'
        STOP
      END SELECT

    RETURN 

  END SUBROUTINE compute_clt

  SUBROUTINE compute_psl_ptarget4d2(press, ps, hgt, ta, qv, ptarget, psl, d1, d2, d3, d4)
    ! Subroutine to compute sea level pressure using a target pressure. Similar to the Benjamin 
    !   and Miller (1990). Method found in p_interp.F90

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3, d4
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(in)        :: press, ta, qv
    REAL(r_k), DIMENSION(d1,d2,d4), INTENT(in)           :: ps
    REAL(r_k), DIMENSION(d1,d2), INTENT(in)              :: hgt
    REAL(r_k), INTENT(in)                                :: ptarget
    REAL(r_k), DIMENSION(d1,d2,d4), INTENT(out)          :: psl

! Local
    INTEGER                                              :: i, j, it
    INTEGER                                              :: kin
    INTEGER                                              :: kupper
    REAL(r_k)                                            :: dpmin, dp, tbotextrap,   &
      tvbotextrap, virtual
    ! Exponential related to standard atmosphere lapse rate r_d*gammav/g
    REAL(r_k), PARAMETER                                 :: expon=r_d*gammav/grav

!!!!!!! Variables
! press: Atmospheric pressure [Pa]
! ps: surface pressure [Pa]
! hgt: surface height
! ta: temperature [K]
! qv: water vapor mixing ratio
! dz: number of vertical levels
! psl: sea-level pressure

    fname = 'compute_psl_ptarget4d2'

    ! Minimal distance between pressures [Pa]
    dpmin=1.e4
    psl=0.

    DO i=1,d1
      DO j=1,d2
        IF (hgt(i,j) /= 0.) THEN
          DO it=1,d4

            ! target pressure to be used for the extrapolation [Pa] (defined in namelist.input)
            !   ptarget = 70000. default value

            ! We are below both the ground and the lowest data level.

            !      First, find the model level that is closest to a "target" pressure
            !        level, where the "target" pressure is delta-p less that the local
            !        value of a horizontally smoothed surface pressure field.  We use
            !        delta-p = 150 hPa here. A standard lapse rate temperature profile
            !        passing through the temperature at this model level will be used
            !        to define the temperature profile below ground.  This is similar
            !        to the Benjamin and Miller (1990) method, using  
            !        700 hPa everywhere for the "target" pressure.

            kupper = 0
            loop_kIN: DO kin=d3,1,-1
              kupper = kin
              dp=abs( press(i,j,kin,it) - ptarget )
              IF (dp .GT. dpmin) EXIT loop_kIN
              dpmin=min(dpmin,dp)
            ENDDO loop_kIN

            tbotextrap=ta(i,j,kupper,it)*(ps(i,j,it)/ptarget)**expon
            tvbotextrap=virtualTemp1D(tbotextrap,qv(i,j,kupper,it))

            psl(i,j,it) = ps(i,j,it)*((tvbotextrap+gammav*hgt(i,j))/tvbotextrap)**(1/expon)
          END DO
        ELSE
          psl(i,j,:) = ps(i,j,:)
        END IF
      END DO
    END DO

    RETURN

  END SUBROUTINE compute_psl_ptarget4d2

  SUBROUTINE compute_tv4d(ta,qv,tv,d1,d2,d3,d4)
! 4D calculation of virtual temperaure

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3, d4
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(in)        :: ta, qv
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(out)       :: tv

! Variables
! ta: temperature [K]
! qv: mixing ratio [kgkg-1]
! tv: virtual temperature

    tv = ta*(oneRK+(qv/epsilonv))/(oneRK+qv)

  END SUBROUTINE compute_tv4d

  FUNCTION VirtualTemp1D (ta,qv) result (tv)
! 1D calculation of virtual temperaure

    IMPLICIT NONE

    REAL(r_k), INTENT(in)                                :: ta, qv
    REAL(r_k)                                            :: tv

! Variables
! ta: temperature [K]
! qv: mixing ratio [kgkg-1]

    tv = ta*(oneRK+(qv/epsilonv))/(oneRK+qv)

  END FUNCTION VirtualTemp1D

! ---- BEGIN modified from module_diag_afwa.F ---- !

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    Theta
  !~
  !~ Description:
  !~    This function calculates potential temperature as defined by
  !~    Poisson's equation, given temperature and pressure ( hPa ).
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION Theta ( t, p )

    IMPLICIT NONE

     !~ Variable declaration
     !  --------------------
     REAL(r_k), INTENT ( IN )                            :: t
     REAL(r_k), INTENT ( IN )                            :: p
     REAL(r_k)                                           :: theta

     ! Using WRF values
     !REAL :: Rd ! Dry gas constant
     !REAL :: Cp ! Specific heat of dry air at constant pressure
     !REAL :: p00 ! Standard pressure ( 1000 hPa )
     REAL(r_k)                                           :: Rd, p00
  
     !Rd =  287.04
     !Cp = 1004.67
     !p00 = 1000.00

     Rd = r_d
     p00 = p1000mb/100.

     !~ Poisson's equation
     !  ------------------
     theta = t * ( (p00/p)**(Rd/Cp) )
  
  END FUNCTION Theta

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    Thetae
  !~
  !~ Description:
  !~    This function returns equivalent potential temperature using the 
  !~    method described in Bolton 1980, Monthly Weather Review, equation 43.
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION Thetae ( tK, p, rh, mixr )

    IMPLICIT NONE

     !~ Variable Declarations
     !  ---------------------
     REAL(r_k) :: tK        ! Temperature ( K )
     REAL(r_k) :: p         ! Pressure ( hPa )
     REAL(r_k) :: rh        ! Relative humidity
     REAL(r_k) :: mixr      ! Mixing Ratio ( kg kg^-1)
     REAL(r_k) :: te        ! Equivalent temperature ( K )
     REAL(r_k) :: thetae    ! Equivalent potential temperature
  
     ! Using WRF values
     !REAL, PARAMETER :: R  = 287.04         ! Universal gas constant (J/deg kg)
     !REAL, PARAMETER :: P0 = 1000.0         ! Standard pressure at surface (hPa)
     REAL(r_k)                                                :: R, p00, Lv
     !REAL, PARAMETER :: lv = 2.54*(10**6)   ! Latent heat of vaporization
                                            ! (J kg^-1)
     !REAL, PARAMETER :: cp = 1004.67        ! Specific heat of dry air constant
                                            ! at pressure (J/deg kg)
     REAL(r_k) :: tlc                            ! LCL temperature
  
     R = r_d
     p00 = p1000mb/100.
     lv = XLV

     !~ Calculate the temperature of the LCL
     !  ------------------------------------
     tlc = TLCL ( tK, rh )
  
     !~ Calculate theta-e
     !  -----------------
     thetae = (tK * (p00/p)**( (R/Cp)*(1.- ( (.28E-3)*mixr*1000.) ) ) )* &
                 exp( (((3.376/tlc)-.00254))*&
                    (mixr*1000.*(1.+(.81E-3)*mixr*1000.)) )
  
  END FUNCTION Thetae

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    The2T.f90
  !~
  !~ Description:
  !~    This function returns the temperature at any pressure level along a
  !~    saturation adiabat by iteratively solving for it from the parcel
  !~    thetae.
  !~
  !~ Dependencies:
  !~    function thetae.f90
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION The2T ( thetaeK, pres, flag ) result ( tparcel )

    IMPLICIT NONE
  
     !~ Variable Declaration
     !  --------------------
     REAL(r_k),    INTENT     ( IN ) :: thetaeK
     REAL(r_k),    INTENT     ( IN ) :: pres
     LOGICAL, INTENT ( INOUT )  :: flag
     REAL(r_k)                       :: tparcel
  
     REAL(r_k) :: thetaK
     REAL(r_k) :: tovtheta
     REAL(r_k) :: tcheck
     REAL(r_k) :: svpr, svpr2
     REAL(r_k) :: smixr, smixr2
     REAL(r_k) :: thetae_check, thetae_check2
     REAL(r_k) :: tguess_2, correction
  
     LOGICAL :: found
     INTEGER :: iter
  
     ! Using WRF values
     !REAL :: R     ! Dry gas constant
     !REAL :: Cp    ! Specific heat for dry air
     !REAL :: kappa ! Rd / Cp
     !REAL :: Lv    ! Latent heat of vaporization at 0 deg. C
     REAL(r_k)                                                :: R, kappa, Lv

     R = r_d
     Lv = XLV
     !R     = 287.04
     !Cp    = 1004.67
     Kappa = R/Cp
     !Lv    = 2.500E+6

     !~ Make initial guess for temperature of the parcel
     !  ------------------------------------------------
     tovtheta = (pres/100000.0)**(r/cp)
     tparcel  = thetaeK/exp(lv*.012/(cp*295.))*tovtheta

     iter = 1
     found = .false.
     flag = .false.

     DO
        IF ( iter > 105 ) EXIT

        tguess_2 = tparcel + REAL ( 1 )

        svpr   = 6.122 * exp ( (17.67*(tparcel-273.15)) / (tparcel-29.66) )
        smixr  = ( 0.622*svpr ) / ( (pres/100.0)-svpr )
        svpr2  = 6.122 * exp ( (17.67*(tguess_2-273.15)) / (tguess_2-29.66) )
        smixr2 = ( 0.622*svpr2 ) / ( (pres/100.0)-svpr2 )

        !  ------------------------------------------------------------------ ~!
        !~ When this function was orinially written, the final parcel         ~!
        !~ temperature check was based off of the parcel temperature and      ~!
        !~ not the theta-e it produced.  As there are multiple temperature-   ~!
        !~ mixing ratio combinations that can produce a single theta-e value, ~!
        !~ we change the check to be based off of the resultant theta-e       ~!
        !~ value.  This seems to be the most accurate way of backing out      ~!
        !~ temperature from theta-e.                                          ~!
        !~                                                                    ~!
        !~ Rentschler, April 2010                                             ~!
        !  ------------------------------------------------------------------  !

        !~ Old way...
        !thetaK = thetaeK / EXP (lv * smixr  /(cp*tparcel) )
        !tcheck = thetaK * tovtheta

        !~ New way
        thetae_check  = Thetae ( tparcel,  pres/100., 100., smixr  )
        thetae_check2 = Thetae ( tguess_2, pres/100., 100., smixr2 )

        !~ Whew doggies - that there is some accuracy...
        !IF ( ABS (tparcel-tcheck) < .05) THEN
        IF ( ABS (thetaeK-thetae_check) < .001) THEN
           found = .true.
           flag  = .true.
           EXIT
        END IF

        !~ Old
        !tparcel = tparcel + (tcheck - tparcel)*.3

        !~ New
        correction = ( thetaeK-thetae_check ) / ( thetae_check2-thetae_check )
        tparcel = tparcel + correction

        iter = iter + 1
     END DO

     !IF ( .not. found ) THEN
     !   print*, "Warning! Thetae to temperature calculation did not converge!"
     !   print*, "Thetae ", thetaeK, "Pressure ", pres
     !END IF

  END FUNCTION The2T

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    VirtualTemperature
  !~
  !~ Description:
  !~    This function returns virtual temperature given temperature ( K )
  !~    and mixing ratio.
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION VirtualTemperature ( tK, w ) result ( Tv )

    IMPLICIT NONE

     !~ Variable declaration
     real(r_k), intent ( in ) :: tK !~ Temperature
     real(r_k), intent ( in ) :: w  !~ Mixing ratio ( kg kg^-1 )
     real(r_k)                :: Tv !~ Virtual temperature

     Tv = tK * ( 1.0 + (w/0.622) ) / ( 1.0 + w )

  END FUNCTION VirtualTemperature

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~
  !~ Name:
  !~    SaturationMixingRatio
  !~
  !~ Description:
  !~    This function calculates saturation mixing ratio given the
  !~    temperature ( K ) and the ambient pressure ( Pa ).  Uses 
  !~    approximation of saturation vapor pressure.
  !~
  !~ References:
  !~    Bolton (1980), Monthly Weather Review, pg. 1047, Eq. 10
  !~
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION SaturationMixingRatio ( tK, p ) result ( ws )

    IMPLICIT NONE

    REAL(r_k), INTENT ( IN ) :: tK
    REAL(r_k), INTENT ( IN ) :: p
    REAL(r_k)                :: ws

    REAL(r_k) :: es

    es = 6.122 * exp ( (17.67*(tK-273.15))/ (tK-29.66) )
    ws = ( 0.622*es ) / ( (p/100.0)-es )

  END FUNCTION SaturationMixingRatio

  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  !~                                                                     
  !~ Name:                                                                
  !~    tlcl                                                               
  !~                                                                        
  !~ Description:                                                            
  !~    This function calculates the temperature of a parcel of air would have
  !~    if lifed dry adiabatically to it's lifting condensation level (lcl).  
  !~                                                                          
  !~ References:                                                              
  !~    Bolton (1980), Monthly Weather Review, pg. 1048, Eq. 22
  !~                                                                          
  !!!~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~!!!
  FUNCTION TLCL ( tk, rh )
    
    IMPLICIT NONE
 
    REAL(r_k), INTENT ( IN ) :: tK   !~ Temperature ( K )
    REAL(r_k), INTENT ( IN ) :: rh   !~ Relative Humidity ( % )
    REAL(r_k)                :: tlcl
    
    REAL(r_k) :: denom, term1, term2

    term1 = 1.0 / ( tK - 55.0 )
!! Lluis
!    IF ( rh > REAL (0) ) THEN
    IF ( rh > zeroRK ) THEN
      term2 = ( LOG (rh/100.0)  / 2840.0 )
    ELSE
      term2 = ( LOG (0.001/oneRK) / 2840.0 )
    END IF
    denom = term1 - term2
!! Lluis
!    tlcl = ( 1.0 / denom ) + REAL ( 55 ) 
    tlcl = ( oneRK / denom ) + 55*oneRK

  END FUNCTION TLCL

  FUNCTION var_cape_afwa1D(nz, tk, rhv, p, hgtv, sfc, cape, cin, zlfc, plfc, lidx, parcel) RESULT (ostat)
! Function to compute cape on a 1D column following implementation in phys/module_diag_afwa.F

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: nz, sfc
    REAL(r_k), DIMENSION(nz), INTENT(in)                 :: tk, rhv, p, hgtv
    REAL(r_k), INTENT(out)                               :: cape, cin, zlfc, plfc, lidx
    INTEGER                                              :: ostat
    INTEGER, INTENT(in)                                  :: parcel
  
    ! Local
    !~ Derived profile variables
    !  -------------------------
    REAL(r_k), DIMENSION(nz)                             :: rh, hgt, ws, w, dTvK, buoy
    REAL(r_k)                                            :: tlclK, plcl, nbuoy, pbuoy
  
    !~ Source parcel information
    !  -------------------------
    REAL(r_k)                                            :: srctK, srcrh, srcws, srcw, srcp,          &
      srctheta, srcthetaeK
    INTEGER                                              :: srclev
    REAL(r_k)                                            :: spdiff
   
    !~ Parcel variables
    !  ----------------
    REAL(r_k)                                            :: ptK, ptvK, tvK, pw
  
    !~ Other utility variables
    !  -----------------------
    INTEGER                                              :: i, j, k
    INTEGER                                              :: lfclev
    INTEGER                                              :: prcl
    INTEGER                                              :: mlev
    INTEGER                                              :: lyrcnt
    LOGICAL                                              :: flag
    LOGICAL                                              :: wflag
    REAL(r_k)                                            :: freeze
    REAL(r_k)                                            :: pdiff
    REAL(r_k)                                            :: pm, pu, pd
    REAL(r_k)                                            :: lidxu
    REAL(r_k)                                            :: lidxd
  
    REAL(r_k), PARAMETER                                 :: Rd = r_d
    REAL(r_k), PARAMETER                                 :: RUNDEF = -9.999E30

!!!!!!! Variables
! nz: Number of vertical levels
! sfc: Surface level in the profile
! tk: Temperature profile [K]
! rhv: Relative Humidity profile [1]
! rh: Relative Humidity profile [%]
! p: Pressure profile [Pa]
! hgtv: Height profile [m]
! hgt: Height profile [gpm]
! cape: CAPE [Jkg-1]
! cin: CIN [Jkg-1]
! zlfc: LFC Height [gpm]
! plfc: LFC Pressure [Pa]
! lidx: Lifted index
!   FROM: https://en.wikipedia.org/wiki/Lifted_index
!     lidx >= 6: Very Stable Conditions
!     6 > lidx > 1: Stable Conditions, Thunderstorms Not Likely
!     0 > lidx > -2: Slightly Unstable, Thunderstorms Possible, With Lifting Mechanism (i.e., cold front, daytime heating, ...)
!     -2 > lidx > -6: Unstable, Thunderstorms Likely, Some Severe With Lifting Mechanism
!     -6 > lidx: Very Unstable, Severe Thunderstorms Likely With Lifting Mechanism
! ostat: Function return status (Nonzero is bad)
! parcel:
!   Most Unstable = 1 (default)
!   Mean layer = 2
!   Surface based = 3
!~ Derived profile variables
!  -------------------------
! ws: Saturation mixing ratio
! w: Mixing ratio
! dTvK: Parcel / ambient Tv difference
! buoy: Buoyancy
! tlclK: LCL temperature [K]
! plcl: LCL pressure [Pa]
! nbuoy: Negative buoyancy
! pbuoy: Positive buoyancy
  
!~ Source parcel information
!  -------------------------
! srctK: Source parcel temperature [K]
! srcrh: Source parcel rh [%]
! srcws: Source parcel sat. mixing ratio
! srcw: Source parcel mixing ratio
! srcp: Source parcel pressure [Pa]
! srctheta: Source parcel theta [K]
! srcthetaeK: Source parcel theta-e [K]
! srclev: Level of the source parcel
! spdiff: Pressure difference
   
!~ Parcel variables
!  ----------------
! ptK: Parcel temperature [K]
! ptvK: Parcel virtual temperature [K]
! tvK: Ambient virtual temperature [K]
! pw: Parcel mixing ratio
  
!~ Other utility variables
!  -----------------------
! lfclev: Level of LFC
! prcl: Internal parcel type indicator
! mlev: Level for ML calculation
! lyrcnt: Number of layers in mean layer
! flag: Dummy flag
! wflag: Saturation flag
! freeze: Water loading multiplier
! pdiff: Pressure difference between levs 
! pm, pu, pd: Middle, upper, lower pressures
! lidxu: Lifted index at upper level
! lidxd: Lifted index at lower level

    fname = 'var_cape_afwa'  

    !~ Initialize variables
    !  --------------------
    rh = rhv*100.
    hgt = hgtv*g
    ostat = 0
    CAPE = zeroRK
    CIN = zeroRK
    ZLFC = RUNDEF
    PLFC = RUNDEF
  
    !~ Look for submitted parcel definition
    !~ 1 = Most unstable
    !~ 2 = Mean layer
    !~ 3 = Surface based
    !  -------------------------------------
    IF ( parcel > 3 .or. parcel < 1 ) THEN
       prcl = 1
    ELSE
       prcl =  parcel
    END IF
  
    !~ Initalize our parcel to be (sort of) surface based.  Because of
    !~ issues we've been observing in the WRF model, specifically with
    !~ excessive surface moisture values at the surface, using a true
    !~ surface based parcel is resulting a more unstable environment
    !~ than is actually occuring.  To address this, our surface parcel
    !~ is now going to be defined as the parcel between 25-50 hPa
    !~ above the surface. UPDATE - now that this routine is in WRF,
    !~ going to trust surface info. GAC 20140415
    !  ----------------------------------------------------------------
  
    !~ Compute mixing ratio values for the layer
    !  -----------------------------------------
    DO k = sfc, nz
      ws  ( k )   = SaturationMixingRatio ( tK(k), p(k) )
      w   ( k )   = ( rh(k)/100.0 ) * ws ( k )
    END DO
  
    srclev      = sfc
    srctK       = tK    ( sfc )
    srcrh       = rh    ( sfc )
    srcp        = p     ( sfc )
    srcws       = ws    ( sfc )
    srcw        = w     ( sfc )
    srctheta    = Theta ( tK(sfc), p(sfc)/100.0 )
   
      !~ Compute the profile mixing ratio.  If the parcel is the MU parcel,
      !~ define our parcel to be the most unstable parcel within the lowest
      !~ 180 mb.
      !  -------------------------------------------------------------------
      mlev = sfc + 1
      DO k = sfc + 1, nz
   
         !~ Identify the last layer within 100 hPa of the surface
         !  -----------------------------------------------------
         pdiff = ( p (sfc) - p (k) ) / REAL ( 100 )
         IF ( pdiff <= REAL (100) ) mlev = k

         !~ If we've made it past the lowest 180 hPa, exit the loop
         !  -------------------------------------------------------
         IF ( pdiff >= REAL (180) ) EXIT

         IF ( prcl == 1 ) THEN
            !IF ( (p(k) > 70000.0) .and. (w(k) > srcw) ) THEN
            IF ( (w(k) > srcw) ) THEN
               srctheta = Theta ( tK(k), p(k)/100.0 )
               srcw = w ( k )
               srclev  = k
               srctK   = tK ( k )
               srcrh   = rh ( k )
               srcp    = p  ( k )
            END IF
         END IF
   
      END DO
   
      !~ If we want the mean layer parcel, compute the mean values in the
      !~ lowest 100 hPa.
      !  ----------------------------------------------------------------
      lyrcnt =  mlev - sfc + 1
      IF ( prcl == 2 ) THEN
   
         srclev   = sfc
         srctK    = SUM ( tK (sfc:mlev) ) / REAL ( lyrcnt )
         srcw     = SUM ( w  (sfc:mlev) ) / REAL ( lyrcnt )
         srcrh    = SUM ( rh (sfc:mlev) ) / REAL ( lyrcnt )
         srcp     = SUM ( p  (sfc:mlev) ) / REAL ( lyrcnt )
         srctheta = Theta ( srctK, srcp/100. )
   
      END IF
   
      srcthetaeK = Thetae ( srctK, srcp/100.0, srcrh, srcw )
   
      !~ Calculate temperature and pressure of the LCL
      !  ---------------------------------------------
      tlclK = TLCL ( tK(srclev), rh(srclev) )
      plcl  = p(srclev) * ( (tlclK/tK(srclev))**(Cp/Rd) )
   
      !~ Now lift the parcel
      !  -------------------
   
      buoy  = REAL ( 0 )
      pw    = srcw
      wflag = .false.
      DO k  = srclev, nz
         IF ( p (k) <= plcl ) THEN
   
            !~ The first level after we pass the LCL, we're still going to
            !~ lift the parcel dry adiabatically, as we haven't added the
            !~ the required code to switch between the dry adiabatic and moist
            !~ adiabatic cooling.  Since the dry version results in a greater
            !~ temperature loss, doing that for the first step so we don't over
            !~ guesstimate the instability.
            !  ----------------------------------------------------------------
   
            IF ( wflag ) THEN
               flag  = .false.
   
               !~ Above the LCL, our parcel is now undergoing moist adiabatic
               !~ cooling.  Because of the latent heating being undergone as
               !~ the parcel rises above the LFC, must iterative solve for the
               !~ parcel temperature using equivalant potential temperature,
               !~ which is conserved during both dry adiabatic and
               !~ pseudoadiabatic displacements.
               !  --------------------------------------------------------------
               ptK   = The2T ( srcthetaeK, p(k), flag )
   
               !~ Calculate the parcel mixing ratio, which is now changing
               !~ as we condense moisture out of the parcel, and is equivalent
               !~ to the saturation mixing ratio, since we are, in theory, at
               !~ saturation.
               !  ------------------------------------------------------------
               pw = SaturationMixingRatio ( ptK, p(k) )
   
               !~ Now we can calculate the virtual temperature of the parcel
               !~ and the surrounding environment to assess the buoyancy.
               !  ----------------------------------------------------------
               ptvK  = VirtualTemperature ( ptK, pw )
               tvK   = VirtualTemperature ( tK (k), w (k) )
   
               !~ Modification to account for water loading
               !  -----------------------------------------
               freeze = 0.033 * ( 263.15 - pTvK )
               IF ( freeze > 1.0 ) freeze = 1.0
               IF ( freeze < 0.0 ) freeze = 0.0
   
               !~ Approximate how much of the water vapor has condensed out
               !~ of the parcel at this level
               !  ---------------------------------------------------------
               freeze = freeze * 333700.0 * ( srcw - pw ) / 1005.7
   
               pTvK = pTvK - pTvK * ( srcw - pw ) + freeze
               dTvK ( k ) = ptvK - tvK
               buoy ( k ) = g * ( dTvK ( k ) / tvK )
   
            ELSE
   
               !~ Since the theta remains constant whilst undergoing dry
               !~ adiabatic processes, can back out the parcel temperature
               !~ from potential temperature below the LCL
               !  --------------------------------------------------------
               ptK   = srctheta / ( 100000.0/p(k) )**(Rd/Cp)
   
               !~ Grab the parcel virtual temperture, can use the source
               !~ mixing ratio since we are undergoing dry adiabatic cooling
               !  ----------------------------------------------------------
               ptvK  = VirtualTemperature ( ptK, srcw )
   
               !~ Virtual temperature of the environment
               !  --------------------------------------
               tvK   = VirtualTemperature ( tK (k), w (k) )
   
               !~ Buoyancy at this level
               !  ----------------------
               dTvK ( k ) = ptvK - tvK
               buoy ( k ) = g * ( dtvK ( k ) / tvK )
   
               wflag = .true.
   
            END IF
   
         ELSE
   
            !~ Since the theta remains constant whilst undergoing dry
            !~ adiabatic processes, can back out the parcel temperature
            !~ from potential temperature below the LCL
            !  --------------------------------------------------------
            ptK   = srctheta / ( 100000.0/p(k) )**(Rd/Cp)
   
            !~ Grab the parcel virtual temperture, can use the source
            !~ mixing ratio since we are undergoing dry adiabatic cooling
            !  ----------------------------------------------------------
            ptvK  = VirtualTemperature ( ptK, srcw )
   
            !~ Virtual temperature of the environment
            !  --------------------------------------
            tvK   = VirtualTemperature ( tK (k), w (k) )
   
            !~ Buoyancy at this level
            !  ---------------------
            dTvK ( k ) = ptvK - tvK
            buoy ( k ) = g * ( dtvK ( k ) / tvK )
   
         END IF

         !~ Chirp
         !  -----
  !          WRITE ( *,'(I15,6F15.3)' )k,p(k)/100.,ptK,pw*1000.,ptvK,tvK,buoy(k)
   
      END DO
   
      !~ Add up the buoyancies, find the LFC
      !  -----------------------------------
      flag   = .false.
      lfclev = -1
      nbuoy  = REAL ( 0 )
      pbuoy = REAL ( 0 )
      DO k = sfc + 1, nz
         IF ( tK (k) < 253.15 ) EXIT
         CAPE = CAPE + MAX ( buoy (k), 0.0 ) * ( hgt (k) - hgt (k-1) )
         CIN  = CIN  + MIN ( buoy (k), 0.0 ) * ( hgt (k) - hgt (k-1) )
   
         !~ If we've already passed the LFC
         !  -------------------------------
         IF ( flag .and. buoy (k) > REAL (0) ) THEN
            pbuoy = pbuoy + buoy (k)
         END IF
   
         !~ We are buoyant now - passed the LFC
         !  -----------------------------------
         IF ( .not. flag .and. buoy (k) > REAL (0) .and. p (k) < plcl ) THEN
            flag = .true.
            pbuoy = pbuoy + buoy (k)
            lfclev = k
         END IF
   
         !~ If we think we've passed the LFC, but encounter a negative layer
         !~ start adding it up.
         !  ----------------------------------------------------------------
         IF ( flag .and. buoy (k) < REAL (0) ) THEN
            nbuoy = nbuoy + buoy (k)

            !~ If the accumulated negative buoyancy is greater than the
            !~ positive buoyancy, then we are capped off.  Got to go higher
            !~ to find the LFC. Reset positive and negative buoyancy summations
            !  ----------------------------------------------------------------
            IF ( ABS (nbuoy) > pbuoy ) THEN
               flag   = .false.
               nbuoy  = REAL ( 0 )
               pbuoy  = REAL ( 0 )
               lfclev = -1
            END IF
         END IF

      END DO

      !~ Calculate lifted index by interpolating difference between
      !~ parcel and ambient Tv to 500mb.
      !  ----------------------------------------------------------
      DO k = sfc + 1, nz

         pm = 50000.
         pu = p ( k )
         pd = p ( k - 1 )

         !~ If we're already above 500mb just set lifted index to 0.
         !~ --------------------------------------------------------
         IF ( pd .le. pm ) THEN
            lidx = zeroRK
            EXIT
   
         ELSEIF ( pu .le. pm .and. pd .gt. pm) THEN

            !~ Found trapping pressure: up, middle, down.
            !~ We are doing first order interpolation.  
            !  ------------------------------------------
            lidxu = -dTvK ( k ) * ( pu / 100000. ) ** (Rd/Cp)
            lidxd = -dTvK ( k-1 ) * ( pd / 100000. ) ** (Rd/Cp)
            lidx = ( lidxu * (pm-pd) + lidxd * (pu-pm) ) / (pu-pd)
            EXIT

         ENDIF

      END DO
   
      !~ Assuming the the LFC is at a pressure level for now
      !  ---------------------------------------------------
      IF ( lfclev > zeroRK ) THEN
         PLFC = p   ( lfclev )
         ZLFC = hgt ( lfclev )
      END IF
   
      IF ( PLFC /= PLFC .OR. PLFC < zeroRK ) THEN
         PLFC = -oneRK
         ZLFC = -oneRK
      END IF
   
      IF ( CAPE /= CAPE ) cape = zeroRK
   
      IF ( CIN  /= CIN  ) cin  = zeroRK

      !~ Chirp
      !  -----
  !       WRITE ( *,* ) ' CAPE: ', cape, ' CIN:  ', cin
  !       WRITE ( *,* ) ' LFC:  ', ZLFC, ' PLFC: ', PLFC
  !       WRITE ( *,* ) ''
  !       WRITE ( *,* ) ' Exiting buoyancy.'
  !       WRITE ( *,* ) ' ==================================== '
  !       WRITE ( *,* ) ''
   
    RETURN

  END FUNCTION var_cape_afwa1D

! ---- END modified from module_diag_afwa.F ---- !

  SUBROUTINE compute_cape_afwa4D(ta, hur, press, zg, hgt, cape, cin, zlfc, plfc, li, parcelmethod,    &
    d1, d2, d3, d4)
! Subroutine to use WRF phys/module_diag_afwa.F `buyoancy' subroutine to compute CAPE, CIN, ZLFC, PLFC, LI

    IMPLICIT NONE

    INTEGER, INTENT(in)                                  :: d1, d2, d3, d4, parcelmethod
    REAL(r_k), DIMENSION(d1,d2,d3,d4), INTENT(in)        :: ta, hur, press, zg
    REAL(r_k), DIMENSION(d1,d2), INTENT(in)              :: hgt
    REAL(r_k), DIMENSION(d1,d2,d4), INTENT(out)          :: cape, cin, zlfc, plfc, li
 
! Local
    INTEGER                                              :: i, j, it
    INTEGER                                              :: ofunc

!!!!!!! Variables
! ta: air temperature [K]
! hur: relative humidity [%]
! press: air pressure [Pa]
! zg: geopotential height [gpm]
! hgt: topographical height [m]
! cape: Convective available potential energy [Jkg-1]
! cin: Convective inhibition [Jkg-1]
! zlfc: height at the Level of free convection [m]
! plfc: pressure at the Level of free convection [Pa]
! li: lifted index [1]
! parcelmethod:
!   Most Unstable = 1 (default)
!   Mean layer = 2
!   Surface based = 3

    fname = 'compute_cape_afwa4D'

    DO i=1, d1
      DO j=1, d2
        DO it=1, d4
          ofunc = var_cape_afwa1D(d3, ta(i,j,:,it), hur(i,j,:,it), press(i,j,:,it), zg(i,j,:,it),     &
            1, cape(i,j,it), cin(i,j,it), zlfc(i,j,it), plfc(i,j,it), li(i,j,it), parcelmethod)
          zlfc(i,j,it) = zlfc(i,j,it)/g - hgt(i,j)
        END DO
      END DO
    END DO

    RETURN

  END SUBROUTINE compute_cape_afwa4D

END MODULE module_ForDiagnostics
