MODULE module_scientific
! Module of the scientific function/subroutines
  
!!!!!!! Functions & subroutines
! borders_matrixL: Subroutine to provide the borders of a logical array (interested in .TRUE.)
! FindMinimumR_K*: Function returns the location of the minimum in the section between Start and End.
! gridpoints_InsidePolygon: Subroutine to determine if a series of grid points are inside a polygon 
!   following ray casting algorithm
! look_clockwise_borders: Subroutine to look clock-wise for a next point within a collection of borders 
!   (limits of a region)
! paths_border: Subroutine to search the paths of a border field.
! path_properties: Subroutine to determine the properties of a path
! polygons: Subroutine to search the polygons of a border field. FORTRAN based. 1st = 1!
! polygons_t: Subroutine to search the polygons of a temporal series of boolean fields. FORTRAN based. 1st = 1!
! PrintQuantilesR_K: Subroutine to print the quantiles of values REAL(r_k)
! quantilesR_K: Subroutine to provide the quantiles of a given set of values of type real 'r_k'
! rand_sample: Subroutine to randomly sample a range of indices
! SortR_K*: Subroutine receives an array x() r_K and sorts it into ascending order.
! SwapR_K*: Subroutine swaps the values of its two formal arguments.

!!! *Functions/Subroutines to sort values adpated. The method used is usually referred to as "selection" method.
! from: http://www.cs.mtu.edu/~shene/COURSES/cs201/NOTES/chap08/sorting.f90

  USE module_definitions
  USE module_generic

  CONTAINS

SUBROUTINE polygons_t(dbg, dx, dy, dt, boolmatt, polys, Npoly)
! Subroutine to search the polygons of a temporal series of boolean fields. FORTRAN based. 1st = 1!

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx, dy, dt
  LOGICAL, DIMENSION(dx,dy,dt), INTENT(in)               :: boolmatt
  LOGICAL, INTENT(in)                                    :: dbg
  INTEGER, DIMENSION(dt), INTENT(out)                    :: Npoly
  INTEGER, DIMENSION(dx,dy,dt), INTENT(out)              :: polys

! Local
  INTEGER                                                :: i,it

!!!!!!! Variables
! dx,dy: spatial dimensions of the space
! boolmatt: boolean matrix tolook for the polygons (.TRUE. based)
! polys: found polygons
! Npoly: number of polygons found

  fname = 'polygons'

  IF (dbg) PRINT *,TRIM(fname)

  polys = -1
  Npoly = 0

  DO it=1,dt
    IF (ANY(boolmatt(:,:,it))) THEN
      IF (dbg) THEN
        PRINT *,'  it:', it, ' num. TRUE:', COUNT(boolmatt(:,:,it)), 'bool _______'
        DO i=1,dx
          PRINT *,boolmatt(i,:,it)
        END DO
      END IF
      CALL polygons(dbg, dx, dy, boolmatt(:,:,it), polys(:,:,it), Npoly(it))
    ELSE
      IF (dbg) THEN
        PRINT *,'  it:', it, " without '.TRUE.' values skipiing it!!"
      END IF
    END IF
  END DO

END SUBROUTINE polygons_t

SUBROUTINE polygons(dbg, dx, dy, boolmat, polys, Npoly)
! Subroutine to search the polygons of a boolean field. FORTRAN based. 1st = 1!

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx, dy
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: boolmat
  LOGICAL, INTENT(in)                                    :: dbg
  INTEGER, INTENT(out)                                   :: Npoly
  INTEGER, DIMENSION(dx,dy), INTENT(out)                 :: polys

! Local
  INTEGER                                                :: i, j, ip, ipp, Nppt
  INTEGER                                                :: ierr
  INTEGER, DIMENSION(:,:), ALLOCATABLE                   :: borders
  LOGICAL, DIMENSION(dx,dy)                              :: isborder, isbordery
  INTEGER, DIMENSION(:,:,:), ALLOCATABLE                 :: paths
  INTEGER                                                :: Npath
  INTEGER, DIMENSION(:), ALLOCATABLE                     :: Nptpaths
  INTEGER, DIMENSION(2)                                  :: xtrx, xtry, meanpth
  INTEGER                                                :: Nvertx, Npts
  INTEGER, DIMENSION(:,:), ALLOCATABLE                   :: vertxs, points
  LOGICAL, DIMENSION(:), ALLOCATABLE                     :: isin

!!!!!!! Variables
! dx,dy: spatial dimensions of the space
! boolmat: boolean matrix tolook for the polygons (.TRUE. based)
! polys: found polygons
! Npoly: number of polygons found

  fname = 'polygons'

  polys = -1

  Nppt = dx*dy/10
  Npts = dx*dy

  IF (ALLOCATED(borders)) DEALLOCATE(borders)
  ALLOCATE(borders(Nppt,2), STAT=ierr)
  msg = "Problems allocating matrix 'borders'"
  CALL ErrMsg(msg, fname, ierr)

  IF (ALLOCATED(paths)) DEALLOCATE(paths)
  ALLOCATE(paths(Nppt,Nppt,2), STAT=ierr)
  msg = "Problems allocating matrix 'paths'"
  CALL ErrMsg(msg, fname, ierr)

  IF (ALLOCATED(Nptpaths)) DEALLOCATE(Nptpaths)
  ALLOCATE(Nptpaths(Nppt), STAT=ierr)
  msg = "Problems allocating matrix 'Nptpaths'"
  CALL ErrMsg(msg, fname, ierr)

  ! Filling with the points of all the space
  IF (ALLOCATED(points)) DEALLOCATE(points)
  ALLOCATE(points(Npts,2), STAT=ierr)
  msg = "Problems allocating matrix 'points'"
  CALL ErrMsg(msg, fname, ierr)

  ip = 1
  DO i=1, dx
    DO j=1, dy
      points(ip,1) = i
      points(ip,2) = j
      ip = ip + 1
    END DO
  END DO

  CALL borders_matrixL(dx, dy, Nppt, boolmat, borders, isborder, isbordery)
  CALL paths_border(dbg, dx, dy, isborder, Nppt, borders, paths, Npath, Nptpaths)

  Npoly = Npath

  DO ip=1, Npath
    IF (ALLOCATED(vertxs)) DEALLOCATE(vertxs)
    ALLOCATE(vertxs(Nptpaths(ip),2))
    msg = "Problems allocating matrix 'vertxs'"
    CALL ErrMsg(msg, fname, ierr)

    IF (ALLOCATED(isin)) DEALLOCATE(isin)
    ALLOCATE(isin(Npts), STAT=ierr)
    msg = "Problems allocating matrix 'isin'"
    CALL ErrMsg(msg, fname, ierr)

    isin = .FALSE.

    IF (dbg) PRINT *, '  path:', ip, ' N pts:', Nptpaths(ip)

    CALL path_properties(dx, dy, boolmat, Nptpaths(ip), paths(ip,1:Nptpaths(ip),:), xtrx, xtry,       &
      meanpth, 'y', Nvertx, vertxs)

    IF (dbg) THEN
      PRINT *, '    properties  _______'
      PRINT *, '    x-extremes:', xtrx
      PRINT *, '    y-extremes:', xtry
      PRINT *, '    center mean:', meanpth
      PRINT *, '    y-vertexs:', Nvertx,' ________'
      DO i=1, Nvertx
        PRINT *,'      ',i,':',vertxs(i,:)
      END DO
    END IF
 
    CALL gridpoints_InsidePolygon(dx, dy, isbordery, Nptpaths(ip), paths(ip,1:Nptpaths(ip),:), Nvertx,&
      xtrx, xtry, vertxs, Npts, points, isin)

    ! Filling polygons
    ipp = 1
    DO i=1, dx
      DO j=1, dy
        IF (isin(ipp)) polys(i,j) = ip
        ipp = ipp + 1
      END DO
    END DO

    IF (dbg) THEN
      PRINT *,'  boolmat isborder isbordery polygon (',xtrx(1),',',xtry(1),')x(',xtrx(2),',',xtry(2), &
        ') _______' 
      DO i=xtrx(1), xtrx(2)
        PRINT *,i,':',boolmat(i,xtry(1):xtry(2)), ' border ', isborder(i,xtry(1):xtry(2)),            &
          ' isbordery ', isbordery(i,xtry(1):xtry(2)), ' polygon ', polys(i,xtry(1):xtry(2))
      END DO
    END IF

  END DO

  DEALLOCATE (borders)  
  DEALLOCATE (Nptpaths)
  DEALLOCATE (paths)
  DEALLOCATE (vertxs)
  DEALLOCATE (points)
  DEALLOCATE (isin)

  RETURN

END SUBROUTINE polygons

  SUBROUTINE path_properties(dx, dy, Lmat, Nptspth, pth, xxtrm, yxtrm, meanctr, axs, Nvrtx, vrtxs)
! Subroutine to determine the properties of a path:
!   extremes: minimum and maximum of the path along x,y axes
!   meancenter: center from the mean of the coordinates of the paths locations
!   vertexs: path point, without neighbours along a given axis

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx, dy, Nptspth
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: Lmat
  INTEGER, DIMENSION(Nptspth,2), INTENT(in)              :: pth
  CHARACTER, INTENT(in)                                  :: axs
  INTEGER, DIMENSION(2), INTENT(out)                     :: meanctr, xxtrm, yxtrm
  INTEGER, INTENT(out)                                   :: Nvrtx
  INTEGER, DIMENSION(Nptspth,2), INTENT(out)             :: vrtxs

! Local
  INTEGER                                                :: i, ip, jp
  INTEGER                                                :: neig1, neig2

!!!!!!! Variables
! dx,dy: size of the space
! Lmat: original matrix of logical values for the path
! Nptspth: number of points of the path
! pth: path coordinates (clockwise)
! axs: axis of finding the vertex
! [x/y]xtrm: minimum and maximum coordinates of the path
! meanctr: center from the mean of the coordinates of the path
! Nvrtx: Number of vertexs of the path
! vrtxs: coordinates of the vertexs

  fname = 'path_properties'

  vrtxs = -1
  Nvrtx = 0

  xxtrm = (/ MINVAL(pth(:,1)), MAXVAL(pth(:,1)) /)
  yxtrm = (/ MINVAL(pth(:,2)), MAXVAL(pth(:,2)) /)
  meanctr = (/ SUM(pth(:,1))/Nptspth, SUM(pth(:,2))/Nptspth /)

  IF (axs == 'x' .OR. axs == 'X') THEN
    ! Looking vertexs along x-axis
    DO i=1, Nptspth
      ip = pth(i,1)
      jp = pth(i,2)
      neig1 = 0
      neig2 = 0
      ! W-point 
      IF (ip == 1) THEN
        neig1 = -1
      ELSE
        IF (.NOT.Lmat(ip-1,jp)) neig1 = -1
      END IF
      ! E-point 
      IF (ip == dx) THEN
        neig2 = -1
      ELSE
        IF (.NOT.Lmat(ip+1,jp)) neig2 = -1
      END IF
    
      IF (neig1 == -1 .AND. neig2 == -1) THEN
        Nvrtx = Nvrtx + 1
        vrtxs(Nvrtx,:) = (/ip,jp/)
      END IF
    END DO
  ELSE IF (axs == 'y' .OR. axs == 'Y') THEN
    ! Looking vertexs along x-axis
    DO i=1, Nptspth
      ip = pth(i,1)
      jp = pth(i,2)

      neig1 = 0
      neig2 = 0
      ! S-point 
      IF (jp == 1) THEN
        neig1 = -1
      ELSE
        IF (.NOT.Lmat(ip,jp-1)) neig1 = -1
      END IF
      ! N-point 
      IF (jp == dy) THEN
        neig2 = -1
      ELSE
        IF (.NOT.Lmat(ip,jp+1)) neig2 = -1
      END IF

      IF (neig1 == -1 .AND. neig2 == -1) THEN
        Nvrtx = Nvrtx + 1
        vrtxs(Nvrtx,:) = (/ ip, jp /)
      END IF
    END DO
  ELSE
    msg = "Axis '" // axs // "' not available" // CHAR(10) // "  Available ones: 'x', 'X', 'y, 'Y'"
    CALL ErrMsg(msg, fname, -1)
  END IF

  RETURN

  END SUBROUTINE path_properties

  SUBROUTINE gridpoints_InsidePolygon(dx, dy, isbrdr, Npath, path, Nvrtx, xpathxtrm, ypathxtrm,       &
    vrtxs, Npts, pts, inside)
! Subroutine to determine if a series of grid points are inside a polygon following ray casting algorithm
! FROM: https://en.wikipedia.org/wiki/Point_in_polygon

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx,dy,Npath,Nvrtx,Npts
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: isbrdr
  INTEGER, DIMENSION(Npath,2), INTENT(in)                :: path
  INTEGER, DIMENSION(2), INTENT(in)                      :: xpathxtrm, ypathxtrm
  INTEGER, DIMENSION(Npath,2)                            :: vrtxs
  INTEGER, DIMENSION(Npts,2), INTENT(in)                 :: pts
  LOGICAL, DIMENSION(Npts), INTENT(out)                  :: inside

! Local
  INTEGER                                                :: i,j,ip,ix,iy
  INTEGER                                                :: Nintersecs, isvertex, ispath
  INTEGER                                                :: ierr
  LOGICAL, DIMENSION(:,:), ALLOCATABLE                   :: halo_brdr
  INTEGER                                                :: Nbrbrdr

!!!!!!! Variables
! dx,dy: space size
! Npath: number of points of the path of the polygon
! path: path of the polygon
! isbrdr: boolean matrix of the space wqith .T. on polygon border
! Nvrtx: number of vertexs of the path
! [x/y]pathxtrm extremes of the path
! vrtxs: vertexs of the path along y-axis
! Npts: number of points
! pts: points to look for
! inside: vector wether point is inside or not (coincident to a border is inside)

  fname = 'gridpoints_InsidePolygon'

  ! Creation of a 1-grid point larger matrix to deal with points reaching the limits
  IF (ALLOCATED(halo_brdr)) DEALLOCATE(halo_brdr)
  ALLOCATE(halo_brdr(dx+2,dy+2), STAT=ierr)
  msg = "Problems allocating matrix 'halo_brdr'"
  CALL ErrMsg(msg, fname, ierr)
  halo_brdr = .FALSE.

  DO i=1,dx
    halo_brdr(i+1,2:dy+1) = isbrdr(i,:)
  END DO

  inside = .FALSE.

  DO ip=1,Npts
    Nintersecs = 0
    ix = pts(ip,1)
    iy = pts(ip,2)
    ! Point might be outside path range...
    IF (ix >= xpathxtrm(1) .AND. ix <= xpathxtrm(2) .AND. iy >= ypathxtrm(1) .AND.                    &
      iy <= ypathxtrm(2)) THEN

      ! It is a border point?
      ispath = index_list_coordsI(Npath, path, (/ix,iy/))
      IF (isbrdr(ix,iy) .AND. (ispath /= -1)) THEN
        inside(ip) = .TRUE.
        CYCLE
      END IF

      ! Looking along y-axis
      ! Accounting for consecutives borders
      Nbrbrdr = 0
      DO j=MAX(1,ypathxtrm(1)-1),iy-1
        ! Only counting that borders that are not vertexs
        ispath = index_list_coordsI(Npath, path, (/ix,j/))
        isvertex = index_list_coordsI(Npath, vrtxs, (/ix,j/))

        IF (halo_brdr(ix+1,j+1) .AND. (ispath /= -1) .AND. (isvertex == -1) ) Nintersecs = Nintersecs + 1 
        IF (halo_brdr(ix+1,j+1) .AND. (ispath /= -1) .AND. (halo_brdr(ix+1,j+1) .EQV. halo_brdr(ix+1,j+2))) THEN
          Nbrbrdr = Nbrbrdr + 1
        ELSE
          ! Will remove that consecutive borders above 2
          IF (Nbrbrdr /= 0) THEN
            Nintersecs = Nintersecs - MAX(Nbrbrdr-1, 0)
            Nbrbrdr = 0
          END IF
        END IF
      END DO
      IF (MOD(Nintersecs,2) /= 0) inside(ip) = .TRUE.
    END IF

  END DO

  RETURN

END SUBROUTINE gridpoints_InsidePolygon

SUBROUTINE look_clockwise_borders(dx,dy,Nbrdrs,brdrs,gbrdr,isbrdr,ix,iy,dbg,xf,yf,iff)
! Subroutine to look clock-wise for a next point within a collection of borders (limits of a region)

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx, dy, Nbrdrs, ix, iy
  INTEGER, DIMENSION(Nbrdrs,2), INTENT(in)               :: brdrs
  LOGICAL, DIMENSION(Nbrdrs), INTENT(in)                 :: gbrdr
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: isbrdr
  LOGICAL, INTENT(in)                                    :: dbg
  INTEGER, INTENT(out)                                   :: xf, yf, iff

! Local
  INTEGER                                                :: isch
  CHARACTER(len=2), DIMENSION(8)                         :: Lclock
  INTEGER, DIMENSION(8,2)                                :: spt
  INTEGER                                                :: iif, jjf

!!!!!!! Variables
! dx, dy: 2D shape ot the space
! Nbrdrs: number of brdrs found in this 2D space
! brdrs: list of coordinates of the borders
! gbrdr: accounts for the use if the given border point
! isbrdr: accounts for the matrix of the point is a border or not
! ix,iy: coordinates of the point to start to find for
! xf,yf: coordinates of the found point
! iff: position of the border found within the list of borders

  fname = 'look_clockwise_borders'

  ! Looking clock-wise assuming that one starts from the westernmost point

  ! Label of the search
  lclock = (/ 'W ', 'NW', 'N ', 'NE', 'E ', 'SE', 'S ', 'SW' /)
  ! Transformation to apply
  !spt = (/ (/-1,0/), (/-1,1/), (/0,1/), (/1,1/), (/1,0/), (/1,-1/), (/0,-1/), (/-1,-1/) /)
  spt(:,1) = (/ -1, -1, 0, 1, 1, 1, 0, -1 /)
  spt(:,2) = (/ 0, 1, 1, 1, 0, -1, -1, -1 /)

  xf = -1
  yf = -1
  DO isch=1, 8
    ! clock-wise search
    IF (spt(isch,1) >= 0) THEN
      iif = MIN(dx,ix+spt(isch,1))
    ELSE
      iif = MAX(1,ix+spt(isch,1))
    END IF
    IF (spt(isch,2) >= 0) THEN
      jjf = MIN(dy,iy+spt(isch,2))
    ELSE
      jjf = MAX(1,iy+spt(isch,2))
    END IF
    iff = index_list_coordsI(Nbrdrs, brdrs,(/iif,jjf/))
    IF (iff > 0) THEN
      IF (dbg) PRINT *,'    ' // lclock(isch) // '-point:', iif,jjf, ':', iff, 'is',isbrdr(iif,jjf),  &
        'got',gbrdr(iff)
      IF (isbrdr(iif,jjf) .AND. .NOT.gbrdr(iff)) THEN
        xf = iif
        yf = jjf
        EXIT
      END IF
    END IF
  END DO

  RETURN

END SUBROUTINE look_clockwise_borders

SUBROUTINE borders_matrixL(dx,dy,dxy,Lmat,brdrs,isbrdr,isbrdry)
! Subroutine to provide the borders of a logical array (interested in .TRUE.)

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx,dy,dxy
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: Lmat
  INTEGER, DIMENSION(dxy,2), INTENT(out)                 :: brdrs
  LOGICAL, DIMENSION(dx,dy), INTENT(out)                 :: isbrdr, isbrdry

! Local
  INTEGER                                                :: i,j,ib

!!!!!!! Variables
! dx,dy: size of the space
! dxy: maximum number of border points
! Lmat: Matrix to look for the borders
! brdrs: list of coordinates of the borders
! isbrdr: matrix with .T./.F. wether the given matrix point is a border or not
! isbrdry: matrix with .T./.F. wether the given matrix point is a border or not only along y-axis

  fname = 'borders_matrixL'

  isbrdr = .FALSE.
  brdrs = -1
  ib = 1

  ! Starting with the borders. If a given point is TRUE it is a path-vertex
  ! Along y-axis
  DO i=1, dx
    IF (Lmat(i,1) .AND. .NOT.isbrdr(i,1)) THEN 
      brdrs(ib,1) = i
      brdrs(ib,2) = 1
      isbrdr(i,1) = .TRUE.
      ib=ib+1
    END IF
    IF (Lmat(i,dy) .AND. .NOT.isbrdr(i,dy)) THEN 
      brdrs(ib,1) = i
      brdrs(ib,2) = dy
      isbrdr(i,dy) = .TRUE.
      ib=ib+1
    END IF
  END DO
  ! Along x-axis
  DO j=1, dy
    IF (Lmat(1,j) .AND. .NOT.isbrdr(1,j)) THEN 
      brdrs(ib,1) = 1
      brdrs(ib,2) = j
      isbrdr(1,j) = .TRUE.
      ib=ib+1
     END IF
    IF (Lmat(dx,j) .AND. .NOT.isbrdr(dx,j)) THEN 
      brdrs(ib,1) = dx
      brdrs(ib,2) = j
      isbrdr(dx,j) = .TRUE.
      ib=ib+1
    END IF
  END DO

  isbrdry = isbrdr

  ! Border as that when looking on x-axis points with Lmat(i) /= Lmat(i+1)
  DO i=1, dx-1
    DO j=1, dy-1
      IF ( Lmat(i,j) .NEQV. Lmat(i+1,j) ) THEN 
        IF (Lmat(i,j) .AND. .NOT.isbrdr(i,j)) THEN
          brdrs(ib,1) = i
          brdrs(ib,2) = j
          isbrdr(i,j) = .TRUE.
          ib=ib+1
        ELSE IF (Lmat(i+1,j) .AND. .NOT.isbrdr(i+1,j)) THEN
          brdrs(ib,1) = i+1
          brdrs(ib,2) = j
          isbrdr(i+1,j) = .TRUE.
          ib=ib+1
        END IF
      END IF
      ! y-axis
      IF ( Lmat(i,j) .NEQV. Lmat(i,j+1) ) THEN 
        IF (Lmat(i,j) .AND. .NOT.isbrdr(i,j)) THEN
          brdrs(ib,1) = i
          brdrs(ib,2) = j
          isbrdr(i,j) = .TRUE.
          isbrdry(i,j) = .TRUE.
          ib=ib+1
        ELSE IF (Lmat(i,j+1) .AND. .NOT.isbrdr(i,j+1)) THEN
          brdrs(ib,1) = i
          brdrs(ib,2) = j+1
          isbrdr(i,j+1) = .TRUE.
          isbrdry(i,j+1) = .TRUE.
          ib=ib+1
        END IF
      END IF
    END DO        
  END DO

  DO i=1, dx-1
    DO j=1, dy-1
      ! y-axis
      IF ( Lmat(i,j) .NEQV. Lmat(i,j+1) ) THEN 
        IF (Lmat(i,j)) THEN
          isbrdry(i,j) = .TRUE.
        ELSE IF (Lmat(i,j+1)) THEN
          isbrdry(i,j+1) = .TRUE.
        END IF
      END IF
    END DO        
  END DO
  ! only y-axis adding bands of 2 grid points
  DO i=1, dx-1
    DO j=2, dy-2
      IF ( (Lmat(i,j) .EQV. Lmat(i,j+1)) .AND. (Lmat(i,j).NEQV.Lmat(i,j-1)) .AND. (Lmat(i,j).NEQV.Lmat(i,j+2)) ) THEN 
        IF (Lmat(i,j)) THEN
          isbrdry(i,j) = .TRUE.
          isbrdry(i,j+1) = .TRUE.
        END IF
      END IF
    END DO        
  END DO

  RETURN

END SUBROUTINE borders_matrixL

SUBROUTINE paths_border(dbg, dx, dy, isborder, Nppt, borders, paths, Npath, Nptpaths)
! Subroutine to search the paths of a border field.

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: dx, dy, Nppt
  LOGICAL, INTENT(in)                                    :: dbg
  LOGICAL, DIMENSION(dx,dy), INTENT(in)                  :: isborder
  INTEGER, DIMENSION(Nppt,2), INTENT(in)                 :: borders
  INTEGER, DIMENSION(Nppt,Nppt,2), INTENT(out)           :: paths
  INTEGER, INTENT(out)                                   :: Npath
  INTEGER, DIMENSION(Nppt), INTENT(out)                  :: Nptpaths

! Local
  INTEGER                                                :: i,j,k,ib
  INTEGER                                                :: ierr
  INTEGER                                                :: Nbrdr
  LOGICAL, DIMENSION(:), ALLOCATABLE                     :: gotbrdr, emptygotbrdr
  INTEGER                                                :: iipth, ipath, ip, Nptspath
  INTEGER                                                :: iib, jjb, iip, ijp, iif, jjf, iff
  LOGICAL                                                :: found, finishedstep

!!!!!!! Variables
! dx,dy: spatial dimensions of the space
! Nppt: possible number of paths and points that the paths can have
! isborder: boolean matrix which provide the borders of the polygon
! borders: coordinates of the borders of the polygon
! paths: coordinates of each found path
! Npath: number of paths found
! Nptpaths: number of points per path

  fname = 'paths_border'

  ! Sarting matrix
  paths = -1
  Npath = 0
  Nptspath = 0
  Nptpaths = -1

  ib=1
  finishedstep = .FALSE.

  ! Number of border points
  DO ib=1, Nppt
    IF (borders(ib,1) == -1 ) EXIT
  END DO
  Nbrdr = ib-1
    
  IF (dbg) THEN
    PRINT *,'    borders _______'
    DO i=1,Nbrdr
      PRINT *,'    ',i,':',borders(i,:)
    END DO
  END IF

  ! Matrix which keeps track if a border point has been located
  IF (ALLOCATED(gotbrdr)) DEALLOCATE(gotbrdr)
  ALLOCATE(gotbrdr(Nbrdr), STAT=ierr)
  msg = "Problems allocating matrix 'gotbrdr'"
  CALL ErrMsg(msg, fname, ierr)
  IF (ALLOCATED(emptygotbrdr)) DEALLOCATE(emptygotbrdr)
  ALLOCATE(emptygotbrdr(Nbrdr), STAT=ierr)
  msg = "Problems allocating matrix 'emptygotbrdr'"
  CALL ErrMsg(msg, fname, ierr)

  gotbrdr = .FALSE.
  emptygotbrdr = .FALSE.

  ! Starting the fun...
    
  ! Looking along the lines and when a border is found, starting from there in a clock-wise way
  iipth = 1
  ipath = 1    
  DO ib=1,Nbrdr
    iib = borders(iipth,1)
    jjb = borders(iipth,2)
    ! Starting new path
    newpath: IF (.NOT.gotbrdr(iipth)) THEN
      ip = 1
      Nptspath = 1
      paths(ipath,ip,:) = borders(iipth,:)
      gotbrdr(iipth) = .TRUE.
      ! Looking for following clock-wise search
      ! Not looking for W, because search starts from the W
      iip = iib
      ijp = jjb
      DO k=1,Nbrdr
        IF (dbg) PRINT *,ipath,'iip jip:', iip, ijp
        found = .FALSE.
        CALL look_clockwise_borders(dx,dy,Nppt,borders,gotbrdr,isborder,iip,ijp,.FALSE.,iif,jjf,iff)
        IF (iif /= -1) THEN
          ip=ip+1
          paths(ipath,ip,:) = (/ iif,jjf /)
          found = .TRUE.
          gotbrdr(iff) = .TRUE.
          iip = iif
          ijp = jjf
          Nptspath = Nptspath + 1          
        END IF

        IF (dbg) THEN
          PRINT *,iib,jjb,'    end of this round path:', ipath, '_____', gotbrdr
          DO i=1, Nptspath
            PRINT *,'      ',i,':',paths(ipath,i,:)
          END DO
        END IF
        ! If it is not found a next point, might be because it is a non-polygon related value
        IF (.NOT.found) THEN
          IF (dbg) PRINT *,'NOT FOUND !!!', gotbrdr
          ! Are still there available borders?  
          IF (ALL(gotbrdr) .EQV. .TRUE.) THEN
            finishedstep = .TRUE.
            Npath = ipath
            Nptpaths(ipath) = Nptspath
            EXIT
          ELSE
            Nptpaths(ipath) = Nptspath
            ! Let's have a look if the previous points in the path have already some 'non-located' neighbourgs
            DO i=Nptspath,1,-1
              iip = paths(ipath,i,1)
              ijp = paths(ipath,i,2)
              CALL look_clockwise_borders(dx,dy,Nppt,borders,gotbrdr,isborder,iip,ijp,.FALSE., iif,   &
                jjf,iff)
              IF (iif /= -1 .AND. iff /= -1) THEN
                IF (dbg) PRINT *,'    re-take path from point:', iif,',',jjf,' n-path:', iff
                found = .TRUE.
                iipth = index_list_coordsI(Nppt, borders, (/iip,ijp/))
                EXIT
              END IF
            END DO
            IF (.NOT.found) THEN
              ! Looking for the next available border point for the new path
              DO i=1,Nbrdr
                IF (.NOT.gotbrdr(i)) THEN
                  iipth = i
                  EXIT
                END IF
              END DO
              IF (dbg) PRINT *,'  Looking for next path starting at:', iipth, ' point:',              &
                borders(iipth,:)
              ipath=ipath+1
              EXIT
            END IF
          END IF
        ELSE
          IF (dbg) PRINT *,'  looking for next point...'
        END IF
        IF (finishedstep) EXIT
      END DO
    END IF newpath
  END DO
  Npath = ipath
  Nptpaths(ipath) = Nptspath
    
  DEALLOCATE (gotbrdr)
  DEALLOCATE (emptygotbrdr)

  RETURN

END SUBROUTINE paths_border

SUBROUTINE rand_sample(Nvals, Nsample, sample)
! Subroutine to randomly sample a range of indices

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: Nvals, Nsample
  INTEGER, DIMENSION(Nsample), INTENT(out)               :: sample

! Local
  INTEGER                                                :: i, ind, jmax
  REAL, DIMENSION(Nsample)                               :: randv
  CHARACTER(len=50)                                      :: fname
  LOGICAL                                                :: found
  LOGICAL, DIMENSION(Nvals)                              :: issampled
  CHARACTER(len=256)                                     :: msg
  CHARACTER(len=10)                                      :: IS1, IS2

!!!!!!! Variables
! Nvals: number of values
! Nsamples: number of samples
! sample: samnple 
  fname = 'rand_sample'

  IF (Nsample > Nvals) THEN
    WRITE(IS1,'(I10)')Nvals
    WRITE(IS2,'(I10)')Nsample
    msg = 'Sampling of ' // TRIM(IS1) // ' is too big for ' // TRIM(IS1) // 'values'
    CALL ErrMsg(msg, fname, -1)
  END IF

  ! Generation of random numbers always the same series during the whole program!
  CALL RANDOM_NUMBER(randv)

  ! Making sure that we do not repeat any value
  issampled = .FALSE.

  DO i=1, Nsample
    ! Generation of the index from the random numbers
    ind = MAX(INT(randv(i)*Nvals), 1)

    IF (.NOT.issampled(ind)) THEN
      sample(i) = ind
      issampled(ind) = .TRUE.
    ELSE
      ! Looking around the given index
      !PRINT *,' Index :', ind, ' already sampled!', issampled(ind)
      found = .FALSE.
      DO jmax=1, Nvals
        ind = MIN(ind+jmax, Nvals)
        IF (.NOT.issampled(ind)) THEN
          sample(i) = ind
          issampled(ind) = .TRUE.
          found = .TRUE.
          EXIT
        END IF
        ind = MAX(1, ind-jmax)
        IF (.NOT.issampled(ind)) THEN
          sample(i) = ind
          issampled(ind) = .TRUE.
          found = .TRUE.
          EXIT
        END IF
      END DO
      IF (.NOT.found) THEN
        msg = 'sampling could not be finished due to absence of available value!!'
        CALL ErrMsg(msg, fname, -1)
      END IF
    END IF

  END DO

  RETURN

END SUBROUTINE rand_sample

SUBROUTINE PrintQuantilesR_K(Nvals, vals, Nquants, qtvs, bspc)
! Subroutine to print the quantiles of values REAL(r_k)

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: Nvals, Nquants
  REAL(r_k), DIMENSION(Nvals), INTENT(in)                :: vals
  REAL(r_k), DIMENSION(Nquants), INTENT(in)              :: qtvs
  CHARACTER(len=1000), OPTIONAL                          :: bspc

! Local
  INTEGER                                                :: iq
  LOGICAL, DIMENSION(Nvals)                              :: search1, search2, search
  CHARACTER(len=6)                                       :: RS1
  CHARACTER(len=50)                                      :: fname
  CHARACTER(len=1000)                                    :: bspcS

!!!!!!! Variables
! vals: series of values
! qtvs: values of the quantiles
! bspc: base quantity of spaces

  fname = 'PrintQuantilesR_K'

  IF (PRESENT(bspc)) THEN
    bspcS = bspc
  ELSE
    bspcS = '      '
  END IF

  DO iq=1, Nquants-1

    WHERE (vals >= qtvs(iq)) 
      search1 = .TRUE.
    ELSEWHERE
      search1 = .FALSE.
    END WHERE

    WHERE (vals < qtvs(iq+1)) 
      search2 = .TRUE.
    ELSEWHERE
      search2 = .FALSE.
    END WHERE

    WHERE (search1 .AND. search2) 
      search = .TRUE.
    ELSEWHERE
      search = .FALSE.
    END WHERE

    WRITE(RS1, '(F6.2)')(iq)*100./(Nquants-1)
    PRINT *, TRIM(bspcS) // '[',iq,']', TRIM(RS1) // ' %:', qtvs(iq), 'N:', COUNT(search)

  END DO

  RETURN

END SUBROUTINE PrintQuantilesR_K

   INTEGER FUNCTION FindMinimumR_K(x, dsize, Startv, Endv)
! Function returns the location of the minimum in the section between Start and End.

      IMPLICIT NONE

      INTEGER, INTENT(in)                                :: dsize
      REAL(r_k), DIMENSION(dsize), INTENT(in)            :: x
      INTEGER, INTENT(in)                                :: Startv, Endv

! Local
      REAL(r_k)                                          :: Minimum
      INTEGER                                            :: Location
      INTEGER                                            :: i

      Minimum  = x(Startv)                               ! assume the first is the min
      Location = Startv                                  ! record its position
      DO i = Startv+1, Endv                              ! start with next elements
         IF (x(i) < Minimum) THEN                        !   if x(i) less than the min?
            Minimum  = x(i)                              !      Yes, a new minimum found
            Location = i                                 !      record its position
         END IF
      END DO

      FindMinimumR_K = Location                          ! return the position

   END FUNCTION  FindMinimumR_K

   SUBROUTINE SwapR_K(a, b)
! Subroutine swaps the values of its two formal arguments.

      IMPLICIT  NONE

      REAL(r_k), INTENT(INOUT)                           :: a, b
! Local
      REAL(r_k)                                          :: Temp

      Temp = a
      a    = b
      b    = Temp

   END SUBROUTINE  SwapR_K

   SUBROUTINE  SortR_K(x, Nx)
! Subroutine receives an array x() r_K and sorts it into ascending order.

      IMPLICIT NONE

      INTEGER, INTENT(IN)                                :: Nx
      REAL(r_k), DIMENSION(Nx), INTENT(INOUT)            :: x

! Local
      INTEGER                                            :: i
      INTEGER                                            :: Location

      DO i = 1, Nx-1                                     ! except for the last
         Location = FindMinimumR_K(x, Nx-i+1, i, Nx)     ! find min from this to last
         CALL  SwapR_K(x(i), x(Location))                ! swap this and the minimum
      END DO

   END SUBROUTINE  SortR_K

SUBROUTINE quantilesR_K(Nvals, vals, Nquants, quants)
! Subroutine to provide the quantiles of a given set of values of type real 'r_k'

  IMPLICIT NONE

  INTEGER, INTENT(in)                                    :: Nvals, Nquants
  REAL(r_k), DIMENSION(Nvals), INTENT(in)                :: vals
  REAL(r_k), DIMENSION(Nquants), INTENT(out)             :: quants

! Local
  INTEGER                                                :: i
  REAL(r_k)                                              :: minv, maxv
  REAL(r_k), DIMENSION(Nvals)                            :: sortedvals

!!!!!!! Variables
! Nvals: number of values
! Rk: kind of real of the values
! vals: values
! Nquants: number of quants
! quants: values at which the quantile start

  minv = MINVAL(vals)
  maxv = MAXVAL(vals)

  sortedvals = vals
  ! Using from: http://www.cs.mtu.edu/~shene/COURSES/cs201/NOTES/chap08/sorting.f90
  CALL SortR_K(sortedvals, Nvals)

  quants(1) = minv
  DO i=2, Nquants
    quants(i) = sortedvals(INT((i-1)*Nvals/Nquants))
  END DO

END SUBROUTINE quantilesR_K

END MODULE module_scientific
