!$Id: infotrac.F90 4301 2022-10-20 11:57:21Z dcugnet $
!
MODULE infotrac_phy

   USE       strings_mod, ONLY: msg, maxlen, cat, dispTable, num2str, strStack, strParse, strCount, strIdx, maxTableWidth
   USE readTracFiles_mod, ONLY: trac_type, readTracersFiles, tracers,  addPhase,  addKey, iH2O,  &
       isoSelect,  indexUpdate, isot_type, testTracersFiles, isotope,  delPhase,  getKey, tran0, &
       isoKeys, isoName, isoZone, isoPhas, processIsotopes,  isoCheck, itZonIso,  nbIso,         &
          niso,   ntiso,   nzone,   nphas, isoF=>isoFamilies,iqIsoPha, iqWIsoPha, ixIso, new2oldH2O
   IMPLICIT NONE

   PRIVATE

   !=== FOR TRACERS:
   PUBLIC :: init_infotrac_phy                             !--- Initialization of the tracers
   PUBLIC :: tracers, type_trac                            !--- Full tracers database, tracers type keyword
   PUBLIC :: nqtot, nbtr, nqo, nqCO2, nqtottr, nqtke       !--- Main dimensions
   PUBLIC :: conv_flg, pbl_flg                             !--- Convection & boundary layer activation keys
   PUBLIC :: new2oldH2O                                    !--- For backwards compatibility in phyetat0
   PUBLIC :: addPhase, delPhase                            !--- Add/remove the phase from the name of a tracer
   PUBLIC :: nbtr_bin, nbtr_sulgas                         !--- Number of aerosols bins and sulfur gases for StratAer model
   PUBLIC :: id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat

   !=== FOR ISOTOPES: General
   PUBLIC :: isot_type, nbIso                              !--- Derived type, full isotopes families database + nb of families
   PUBLIC :: isoSelect, ixIso, isoFamilies                 !--- Isotopes family selection tool + selected family index
   !=== FOR ISOTOPES: Specific to water
   PUBLIC :: iH2O                                          !--- Value of "ixIso" for "H2O" isotopes class
   PUBLIC :: ivap, iliq, isol, ibs, icf, irvc, itke
   !=== FOR ISOTOPES: Depending on the selected isotopes family
   PUBLIC :: isotope                                       !--- Selected isotopes database (argument of getKey)
   PUBLIC :: isoKeys, isoName, isoZone, isoPhas            !--- Isotopes keys & names, tagging zones names, phases
   PUBLIC ::    niso,   ntiso,   nzone,   nphas            !--- Number of   "   "
   PUBLIC :: itZonIso                                      !--- Index "it" in "isoName(1:niso)" = f(tagging idx, isotope idx)
   PUBLIC :: iqIsoPha                                      !--- Index "iq" in "qx"              = f(isotope idx,   phase idx)
   PUBLIC :: iqWIsoPha                                     !--- Same as iqIsoPha but with normal water phases
   PUBLIC :: isoCheck                                      !--- Run isotopes checking routines
   !=== FOR BOTH TRACERS AND ISOTOPES
   PUBLIC :: getKey                                        !--- Get a key from "tracers" or "isotope"

!=== CONVENTIONS FOR TRACERS NUMBERS:
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  | water in different |    water tagging      |  water isotopes | other tracers | additional tracers moments |
!  | phases: H2O_[glsrb]|      isotopes         |                 |               |  for higher order schemes  |
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  |                    |                       |                 |               |                            |
!  |<--     nqo      -->|<-- nqo*niso* nzone -->|<-- nqo*niso  -->|<--  nbtr   -->|<--        (nmom)        -->|         
!  |                    |                                         |                                            |
!  |                    |<-- nqo*niso*(nzone+1)  =   nqo*ntiso -->|<--    nqtottr = nbtr + nmom             -->|
!  |                                                                              = nqtot - nqo*(ntiso+1)      |
!  |                                                                                                           |
!  |<--                        nqtrue  =  nbtr + nqo*(ntiso+1)                 -->|                            |
!  |                                                                                                           |
!  |<--                        nqtot   =  nqtrue + nmom                                                     -->|
!  |                                                                                                           |
!  |-----------------------------------------------------------------------------------------------------------|
!  NOTES FOR THIS TABLE:
!  * Used "niso", "nzone" and "ntiso" are components of "isotopes(ip)" for water (isotopes(ip)%parent == 'H2O'),
!    since water is so far the sole tracers family, except passive CO2, removed from the main tracers table.
!  * For water, "nqo" is equal to the more general field "isotopes(ip)%nphas".
!  * "niso", "nzone", "ntiso", "nphas" are defined for other isotopic tracers families, if any.
!
!=== DERIVED TYPE EMBEDDING MOST OF THE TRACERS-RELATED QUANTITIES (LENGTH: nqtot)
!    Each entry is accessible using "%" sign.
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  |  entry      | Meaning                                              | Former name | Possible values        |
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  | name        | Name (short)                                         | tname       |                        |
!  | keys        | key/val pairs accessible with "getKey" routine       | /           |                        |
!  | gen0Name    | Name of the 1st generation ancestor                  | /           |                        |
!  | parent      | Name of the parent                                   | /           |                        |
!  | longName    | Long name (with adv. scheme suffix) for outputs      | ttext       |                        |
!  | type        | Type (so far: tracer or tag)                         | /           | tracer,tag             |
!  | phase       | Phases list ("g"as / "l"iquid / "s"olid              |             | [g|l|s|r|b]            |
!  |             |              "r"(cloud) / "b"lowing)                 | /           |                        |
!  | component   | Name(s) of the merged/cumulated section(s)           | /           | coma-separated names   |
!  | iGeneration | Generation (>=1)                                     | /           |                        |
!  | iqParent    | Index of the parent tracer                           | iqpere      | 1:nqtot                |
!  | iqDescen    | Indexes of the childs       (all generations)        | iqfils      | 1:nqtot                |
!  | nqDescen    | Number of the descendants   (all generations)        | nqdesc      | 1:nqtot                |
!  | nqChildren  | Number of childs            (1st generation only)    | nqfils      | 1:nqtot                |
!  | isInPhysics | Advected tracers from the main table kept in physics | /           | nqtottr .TRUE. values  |
!  | iso_iGroup  | Isotopes group index in isotopes(:)                  | /           | 1:nbIso                |
!  | iso_iName   | Isotope  name  index in isotopes(iso_iGroup)%trac(:) | iso_indnum  | 1:niso                 |
!  | iso_iZone   | Isotope  zone  index in isotopes(iso_iGroup)%zone(:) | zone_num    | 1:nzone                |
!  | iso_iPhase  | Isotope  phase index in isotopes(iso_iGroup)%phas(:) | phase_num   | 1:nphas                |
!  +-------------+------------------------------------------------------+-------------+------------------------+
!
!=== DERIVED TYPE EMBEDDING MOST OF THE ISOTOPES-RELATED QUANTITIES (LENGTH: nbIso, NUMBER OF ISOTOPES FAMILIES)
!    Each entry is accessible using "%" sign.
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  |  entry | length | Meaning                                          |    Former name     | Possible values |
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  | parent          | Parent tracer (isotopes family name)             |                    |                 |
!  | keys   | niso   | Isotopes keys/values pairs list + number         |                    |                 |
!  | trac   | ntiso  | Isotopes + tagging tracers list + number         | / | ntraciso       |                 |
!  | zone   | nzone  | Geographic tagging zones   list + number         | / | ntraceurs_zone |                 |
!  | phase  | nphas  | Phases                     list + number         |                    | [g|l|s|r|b] 1:5 |
!  | iqIsoPha        | Index in "qx"           = f(name(1:ntiso)),phas) | iqiso              | 1:nqtot         |
!  | iqWIsoPha       | Index in "qx"       = f(name(1:ntiso+nqo)),phas) |   /                | 1:nqtot         |
!  | itZonIso        | Index in "trac(1:ntiso)"= f(zone, name(1:niso))  | index_trac         | 1:ntiso         |
!  +-----------------+--------------------------------------------------+--------------------+-----------------+

   !=== INDICES FOR WATER
   INTEGER, SAVE :: ivap, iliq, isol, ibs, icf, irvc, itke
!$OMP THREADPRIVATE(ivap, iliq, isol, ibs, icf, irvc, itke)

   !=== DIMENSIONS OF THE TRACERS TABLES, TRACERS TYPE(S)
   INTEGER, SAVE :: nqtot                                       !--- Tracers nb in dynamics (incl. higher moments + H2O)
   INTEGER, SAVE :: nbtr                                        !--- Tracers nb in physics  (excl. higher moments + H2O)
   INTEGER, SAVE :: nqo                                         !--- Number of water phases
   INTEGER, SAVE :: nqtottr                                     !--- Number of tracers passed to phytrac (TO BE DELETED ?)
   INTEGER, SAVE :: nqCO2                                       !--- Number of tracers of CO2  (ThL)
   INTEGER, SAVE :: nqtke                                       !--- Number of TKE tracers
   CHARACTER(LEN=maxlen), SAVE :: type_trac                     !--- Keyword for tracers type(s)
!$OMP THREADPRIVATE(nqtot, nbtr, nqo, nqtottr, nqCO2, nqtke, type_trac)

   !=== VARIABLES FOR INCA
   INTEGER, SAVE, ALLOCATABLE :: conv_flg(:), pbl_flg(:)        !--- Convection / boundary layer activation (nbtr)
!$OMP THREADPRIVATE(conv_flg, pbl_flg)

   !=== LIST OF DEFINED ISOTOPES FAMILIES
   CHARACTER(LEN=maxlen), SAVE, ALLOCATABLE :: isoFamilies(:)   !--- Generation 0 tracer name for each isotopes family (nbIso)
!$OMP THREADPRIVATE(isoFamilies)

  !=== SPECIFIC TO STRATOSPHERIC AEROSOLS (CK/OB)
  INTEGER, SAVE ::  nbtr_bin, nbtr_sulgas         !--- number of aerosols bins and sulfur gases for StratAer model
!$OMP THREADPRIVATE(nbtr_bin, nbtr_sulgas)
  INTEGER, SAVE ::  id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat
!$OMP THREADPRIVATE(id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat)

CONTAINS

SUBROUTINE init_infotrac_phy
   USE iniprint_mod_h
   USE ioipsl_getin_p_mod, ONLY: getin_p
   USE CHEM_REP, ONLY: init_chem_rep_trac, init_chem_rep_phys, init_chem_rep_xjour, &
        pdel, d_q_rep, d_ql_rep, d_qi_rep, &
        iter, rsuntime, pdt_rep, daynum, rch42d, rn2o2d, rcfc112d, rcfc122d, &
        ptrop, ttrop, ztrop, gravit, itroprep, Z1, Z2, fac, B, &
        solaireTIME, ok_SUNTIME, ok_rtime2d, ndimozon
   USE CHEMINI_REPR, ONLY: CHEMINI_REP
   USE chemmain_rlong, ONLY: chemmain_rlong_1401
   USE lmdz_cppkeys_wrapper, ONLY: CPPKEY_INCA, CPPKEY_REPROBUS, CPPKEY_STRATAER
   USE mod_phys_lmdz_para, ONLY: is_master, is_omp_master
   IMPLICIT NONE
!==============================================================================================================================
!
!   Auteur:  P. Le Van /L. Fairhead/F.Hourdin
!   -------
!
!   Modifications:
!   --------------
!   05/94: F.Forget      Modif special traceur
!   02/02: M-A Filiberti Lecture de traceur.def
!   01/22: D. Cugnet     Nouveaux tracer.def et tracer_*.def + encapsulation (types trac_type et isot_type)
!
!   Objet:
!   ------
!   GCM LMD nouvelle grille
!
!==============================================================================================================================
!   ... modification de l'integration de q ( 26/04/94 ) ....
!------------------------------------------------------------------------------------------------------------------------------
! Declarations:


!------------------------------------------------------------------------------------------------------------------------------
! Local variables
   INTEGER, ALLOCATABLE :: hadv(:), vadv(:)                          !--- Horizontal/vertical transport scheme number
   INTEGER :: nqINCA
   CHARACTER(LEN=maxlen), ALLOCATABLE :: tnames(:)
   CHARACTER(LEN=2)      ::   suff(9)                                !--- Suffixes for schemes of order 3 or 4 (Prather)
   CHARACTER(LEN=3)      :: descrq(30)                               !--- Advection scheme description tags
   CHARACTER(LEN=maxlen) :: msg1, texp, ttp                          !--- Strings for messages and expanded tracers type
   INTEGER :: fType                                                  !--- Tracers description file type ; 0: none
                                                                     !--- 1/2/3: "traceur.def"/"tracer.def"/"tracer_*.def"
   INTEGER :: nqtrue                                                 !--- Tracers nb from tracer.def (no higher order moments)
   INTEGER :: iad                                                    !--- Advection scheme number
   INTEGER :: iq, jq, it, nt, im, nm                                 !--- Indexes and temporary variables
   LOGICAL :: lerr, lInit
   TYPE(trac_type), ALLOCATABLE, TARGET :: ttr(:)
   TYPE(trac_type), POINTER             :: t1, t(:)
   CHARACTER(LEN=maxlen),   ALLOCATABLE :: types_trac(:)             !--- Keywords for tracers type(s), parsed version
   CHARACTER(LEN=*), PARAMETER :: modname="init_infotrac_phy"
!------------------------------------------------------------------------------------------------------------------------------
! Initialization :
!------------------------------------------------------------------------------------------------------------------------------
   suff          = ['x ','y ','z ','xx','xy','xz','yy','yz','zz']
   descrq( 1:30) =  '   '
   descrq( 1: 2) = ['LMV','BAK']
   descrq(10:20) = ['VL1','VLP','FH1','FH2','VLH','   ','PPM','PPS','PPP','   ','SLP']
   descrq(30)    =  'PRA'

   CALL getin_p('type_trac',type_trac)

   CALL msg('type_trac = "'//TRIM(type_trac)//'"', modname, is_master)
   IF(strCount(type_trac, '|', nt)) CALL abort_physic(modname, 'Could''nt parse the "type_trac" string with delimiter "|"', 1)
   IF(nt >= 3) CALL abort_physic(modname, 'you need to modify type_trac, this version is not supported by lmdz', 1)
   IF(strParse(type_trac, '|', types_trac, n=nt)) CALL abort_physic(modname, "couldn't parse "//'"type_trac"', 1)
   IF(nt == 2) type_trac = types_trac(2) ! TO BE DELETED SOON

   CALL msg('type_trac = "'//TRIM(type_trac)//'"', modname)
   lInit = .NOT.ALLOCATED(tracers)

!##############################################################################################################################
   IF(lInit .AND. is_master) THEN                                    !=== SKIPED IF ALREADY DONE IN dyn3d_common/infotrac  ####
!##############################################################################################################################
   !--- MESSAGE ABOUT THE CHOSEN CONFIGURATION
   msg1 = 'For type_trac = "'//TRIM(type_trac)//'":'
   SELECT CASE(type_trac)
      CASE('inca'); CALL msg(TRIM(msg1)//' coupling with INCA chemistry model',        modname)
      CASE('inco'); CALL msg(TRIM(msg1)//' coupling jointly with INCA and CO2 cycle',  modname)
      CASE('repr'); CALL msg(TRIM(msg1)//' coupling with REPROBUS chemistry model',    modname)
      CASE('co2i'); CALL msg(TRIM(msg1)//' you have chosen to run with CO2 cycle',     modname)
      CASE('coag'); CALL msg(TRIM(msg1)//' tracers are treated for COAGULATION tests', modname)
      CASE('lmdz'); CALL msg(TRIM(msg1)//' tracers are treated in LMDZ only',          modname)
      CASE DEFAULT; CALL abort_physic(modname,'type_trac='//TRIM(type_trac)//' not possible yet.',1)
   END SELECT

   !--- COHERENCE TEST BETWEEN "type_trac" AND PREPROCESSING KEYS
   SELECT CASE(type_trac)
      CASE('inca', 'inco')
         IF(.NOT.CPPKEY_INCA)     CALL abort_physic(modname, 'You must add cpp key INCA and compile with INCA code', 1)
      CASE('repr')
         IF(.NOT.CPPKEY_REPROBUS) CALL abort_physic(modname, 'You must add cpp key REPROBUS and compile with REPROBUS code', 1)
      CASE('coag')
         IF(.NOT.CPPKEY_STRATAER) CALL abort_physic(modname, 'You must add cpp key StratAer and compile with StratAer code', 1)
   END SELECT
!##############################################################################################################################
   END IF
!##############################################################################################################################

!==============================================================================================================================
! 0) DETERMINE THE TYPE OF THE INPUT TRACERS DESCRIPTION FILE AND READ IT
!==============================================================================================================================
   texp = type_trac                                                  !=== EXPANDED (WITH "|" SEPARATOR) "type_trac"
   IF(texp == 'inco') texp = 'co2i|inca'
   IF(texp /= 'lmdz') texp = 'lmdz|'//TRIM(texp)
   IF(testTracersFiles(modname, texp, fType, lInit.AND.is_master)) CALL abort_physic(modname, 'problem with tracers file(s)',1)
   ttp = type_trac; IF(fType /= 1) ttp = texp
   !---------------------------------------------------------------------------------------------------------------------------
   IF(fType == 0) CALL abort_physic(modname, 'Missing "traceur.def", "tracer.def" or "tracer_<keyword>.def tracers file.',1)
   !---------------------------------------------------------------------------------------------------------------------------
   IF(fType == 1 .AND. ANY(['inca', 'inco'] == type_trac)) &         !=== FOUND OLD STYLE INCA "traceur.def"
      CALL abort_physic(modname, 'retro-compatibility with old-style INCA traceur.def files has been disabled.', 1)
   !---------------------------------------------------------------------------------------------------------------------------

!##############################################################################################################################
   IF(lInit) THEN
      IF(readTracersFiles(ttp, lRepr=type_trac == 'repr')) CALL abort_physic(modname, 'problem with tracers file(s)',1)
   END IF
   CALL msg('No tracers description file(s) reading needed: already done', modname, .NOT.lInit.AND.is_master)
!##############################################################################################################################

!==============================================================================================================================
! 1) Get various numbers: "nqtrue" (first order only tracers), "nqo" (water phases), 'nbtr' (tracers passed to physics), etc.
!==============================================================================================================================
   nqtrue = SIZE(tracers)                                                                               !--- "true" tracers
   nqo    =      COUNT(tracers(:)%component == 'lmdz' .AND. delPhase(tracers(:)%name)     == 'H2O')     !--- Water phases
   nqtke  =      COUNT(tracers(:)%component == 'lmdz' .AND. tracers(:)%name               == 'TKE')     !--- TKE tracers
   nbtr = nqtrue-COUNT(tracers(:)%component == 'lmdz' .AND. delPhase(tracers(:)%gen0Name) == 'H2O')     !--- Passed to phytrac
   nqCO2  =      COUNT( [type_trac == 'inco', type_trac == 'co2i'] )
   IF(CPPKEY_INCA) &
   nqINCA =      COUNT(tracers(:)%component == 'inca')
   IF(CPPKEY_REPROBUS) CALL Init_chem_rep_trac(nbtr, nqo, tracers(:)%name)     !--- Transfert the number of tracers to Reprobus

!==============================================================================================================================
! 2) Calculate nqtot, number of tracers needed (greater if advection schemes 20 or 30 have been chosen).
!==============================================================================================================================
   IF(getKey('hadv', hadv, ky=tracers(:)%keys)) CALL abort_physic(modname, 'missing key "hadv"', 1)
   IF(getKey('vadv', vadv, ky=tracers(:)%keys)) CALL abort_physic(modname, 'missing key "vadv"', 1)
   DO iq = 1, nqtrue
      IF( hadv(iq)<20 .OR. (ANY(hadv(iq)==[20,30]) .AND. hadv(iq)==vadv(iq)) ) CYCLE
      WRITE(msg1,'("The choice hadv=",i0,", vadv=",i0,a)')hadv(iq),vadv(iq),' for "'//TRIM(tracers(iq)%name)//'" is not available'
      CALL abort_physic(modname, TRIM(msg1), 1)
   END DO
   nqtot =    COUNT( hadv< 20 .AND. vadv< 20 ) &                     !--- No additional tracer
         +  4*COUNT( hadv==20 .AND. vadv==20 ) &                     !--- 3  additional tracers
         + 10*COUNT( hadv==30 .AND. vadv==30 )                       !--- 9  additional tracers

   !--- More tracers due to the choice of advection scheme => assign total number of tracers
   IF( nqtot /= nqtrue ) THEN
      CALL msg('The choice of advection scheme for one or more tracers makes it necessary to add tracers')
      CALL msg('The number of true tracers is '//TRIM(num2str(nqtrue)))
      CALL msg('The total number of tracers needed is '//TRIM(num2str(nqtot)))
   END IF

!##############################################################################################################################
   IF(lInit) THEN                                                    !=== SKIPED IF ALREADY DONE IN dyn3d_common/infotrac  ####
!##############################################################################################################################

!==============================================================================================================================
! 3) Determine the advection scheme ; needed to compute the full tracers list, the long names and nqtot
!==============================================================================================================================
   ALLOCATE(ttr(nqtot))
   jq = nqtrue+1
   DO iq = 1, nqtrue
      t1 => tracers(iq)

      !--- VERIFY THE CHOICE OF ADVECTION SCHEME
      iad = -1
      IF(hadv(iq)     ==    vadv(iq)    ) iad = hadv(iq)
      IF(hadv(iq)==10 .AND. vadv(iq)==16) iad = 11
      WRITE(msg1,'("Bad choice of advection scheme for ",a,": hadv = ",i0,", vadv = ",i0)')TRIM(t1%name), hadv(iq), vadv(iq)
      IF(iad == -1) CALL abort_physic(modname, msg1, 1)

      !--- SET FIELDS longName AND isInPhysics
      t1%longName   = t1%name; IF(iad > 0) t1%longName=TRIM(t1%name)//descrq(iad)
      t1%isInPhysics= iad >= 0 .AND. (delPhase(t1%gen0Name) /= 'H2O' .OR. t1%component /= 'lmdz')
      ttr(iq)       = t1

      !--- DEFINE THE HIGHER ORDER TRACERS, IF ANY
      nm = 0
      IF(iad == 20) nm = 3                                           !--- 2nd order scheme
      IF(iad == 30) nm = 9                                           !--- 3rd order scheme
      IF(nm == 0) CYCLE                                              !--- No higher moments
      ttr(jq+1:jq+nm)             = t1
      ttr(jq+1:jq+nm)%name        = [ (TRIM(t1%name)    //'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%gen0Name    = [ (TRIM(t1%name)    //'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%parent      = [ (TRIM(t1%parent)  //'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%longName    = [ (TRIM(t1%longName)//'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%isInPhysics = [ (.FALSE., im=1, nm) ]
      jq = jq + nm
   END DO

   DEALLOCATE(hadv, vadv)
   CALL MOVE_ALLOC(FROM=ttr, TO=tracers)

   !--- SET FIELDS iqParent, iqDescen, nqDescen, nqChildren, iGeneration
   IF(indexUpdate(tracers)) CALL abort_physic(modname, 'problem with tracers indices update', 1)

   !=== DETERMINE ISOTOPES RELATED PARAMETERS ; DEFINE THE EXPLICIT KEYS iso_i*
   IF(processIsotopes()) CALL abort_physic(modname, 'Problem when processing isotopes parameters', 1)
   !!SN do we really need to call isoSelect('H2O',.TRUE.) in processIsotopes and here ?
   iH2O = -1
   IF(nbIso /= 0) THEN
      IF(isoSelect('H2O', .TRUE.)) THEN
         IF(isoSelect(1, .TRUE.)) CALL abort_physic(modname, "Can't select the first isotopes family", 1)
      ELSE
         iH2O = ixIso; CALL getin_p('ok_iso_verif', isotope%check)
      END IF
   END IF

!##############################################################################################################################
   ELSE
!##############################################################################################################################
   DO iq = 1, nqtrue
      IF(hadv(iq)     ==    vadv(iq)    ) iad = hadv(iq)
      IF(hadv(iq)==10 .AND. vadv(iq)==16) iad = 11
      tracers(iq)%isInPhysics= iad >= 0 .AND. (delPhase(tracers(iq)%gen0Name) /= 'H2O' .OR. tracers(iq)%component /= 'lmdz')
   END DO
   tracers(nqtrue+1:nqtot)%isInPhysics = .FALSE.
!##############################################################################################################################
   END IF
!##############################################################################################################################
   isoFamilies = isoF(:)



   !--- Convection / boundary layer activation for all tracers
   IF(.NOT.ALLOCATED(conv_flg)) ALLOCATE(conv_flg(nbtr)); conv_flg(1:nbtr) = 1
   IF(.NOT.ALLOCATED( pbl_flg)) ALLOCATE( pbl_flg(nbtr));  pbl_flg(1:nbtr) = 1

   !--- Note: nqtottr can differ from nbtr when nmom/=0
   nqtottr = nqtot - COUNT(delPhase(tracers%gen0Name) == 'H2O' .AND. tracers%component == 'lmdz')
   IF(COUNT(tracers%iso_iName == 0) - COUNT(delPhase(tracers%name) == 'H2O' .AND. tracers%component == 'lmdz') /= nqtottr) &
      CALL abort_physic(modname, 'problem with the computation of nqtottr', 1)

   !--- Compute indices for water
   ivap = strIdx(tracers(:)%name, addPhase('H2O', 'g'))
   iliq = strIdx(tracers(:)%name, addPhase('H2O', 'l'))
   isol = strIdx(tracers(:)%name, addPhase('H2O', 's'))
   ibs  = strIdx(tracers(:)%name, addPhase('H2O', 'b'))
   icf  = strIdx(tracers(:)%name, addPhase('H2O', 'f'))
   irvc = strIdx(tracers(:)%name, addPhase('H2O', 'c'))
   !--- Compute indices for TKE when it is advected
   itke = strIdx(tracers(:)%name, 'TKE')
   IF (nqtke .GE. 1) THEN
       !--- For TKE, we force isInPhysics=.False. 
       tracers(itke)%isInPhysics = .FALSE.
   ENDIF

   IF(CPPKEY_STRATAER .AND. type_trac == 'coag') THEN
      nbtr_bin    = COUNT([(tracers(iq)%name(1:3)=='BIN', iq=1, nqtot)])
      nbtr_sulgas = COUNT([(tracers(iq)%name(1:3)=='GAS', iq=1, nqtot)])
      tnames = PACK(tracers(:)%name, MASK=tracers(:)%isInPhysics)
      id_BIN01_strat = strIdx(tnames, 'BIN01'   )
      id_OCS_strat   = strIdx(tnames, 'GASOCS'  )
      id_SO2_strat   = strIdx(tnames, 'GASSO2'  )
      id_H2SO4_strat = strIdx(tnames, 'GASH2SO4')
      id_TEST_strat  = strIdx(tnames, 'GASTEST' )
   END IF

   !=== DISPLAY THE RESULTS
   IF(.NOT.is_master) RETURN
   CALL msg('nqo    = '//TRIM(num2str(nqo)),    modname)
   CALL msg('nbtr   = '//TRIM(num2str(nbtr)),   modname)
   CALL msg('nqtrue = '//TRIM(num2str(nqtrue)), modname)
   CALL msg('nqtot  = '//TRIM(num2str(nqtot)),  modname)
   CALL msg('niso   = '//TRIM(num2str(niso)),   modname)
   CALL msg('ntiso  = '//TRIM(num2str(ntiso)),  modname)
   CALL msg('nqtke  = '//TRIM(num2str(nqtke)),  modname)
   CALL msg('nqCO2  = '//TRIM(num2str(nqCO2)),  modname, CPPKEY_INCA)
   CALL msg('nqINCA = '//TRIM(num2str(nqINCA)), modname, CPPKEY_INCA)
   t => tracers
   CALL msg('Information stored in '//TRIM(modname)//': ', modname)
   IF(dispTable('issssssssiiiiiiii', ['iq  ', 'name', 'lNam', 'g0Nm', 'prnt', 'type', 'phas', 'comp',      &
                              'isPh', 'iGen', 'iqPr', 'nqDe', 'nqCh', 'iGrp', 'iNam', 'iZon', 'iPha'],     &
      cat(t%name, t%longName,  t%gen0Name, t%parent, t%type, t%phase, t%component, num2str(t%isInPhysics)),&
      cat([(iq, iq=1, nqtot)], t%iGeneration, t%iqParent, t%nqDescen, t%nqChildren, t%iso_iGroup,          &
                  t%iso_iName, t%iso_iZone, t%iso_iPhase), nColMax=maxTableWidth, nHead=2, sub=modname))   &
      CALL abort_physic(modname, "problem with the tracers table content", 1)
   CALL msg('No isotopes identified.', modname, nbIso == 0)
   IF(nbIso == 0) RETURN
   DO it = 1, nbIso
      IF(isoSelect(it, .TRUE.)) CALL abort_physic(modname, 'Problem when selecting isotopes class', 1)
      CALL msg('For isotopes family "'//TRIM(isoFamilies(it))//'":', modname)
      CALL msg('  isoKeys%name = '//strStack(isoKeys%name), modname)
      CALL msg('  isoName = '//strStack(isoName), modname)
      CALL msg('  isoZone = '//strStack(isoZone), modname)
      CALL msg('  isoPhas = '//TRIM(isoPhas),     modname)
   END DO
   !!SN do we need to call isoSelect(iH2O, .TRUE.) three times ?
   IF(isoSelect(iH2O, .TRUE.)) lerr = isoSelect(1, .TRUE.)

   IF(CPPKEY_STRATAER .AND. type_trac == 'coag') THEN
      CALL msg('nbtr_bin       ='//TRIM(num2str(nbtr_bin      )), modname)
      CALL msg('nbtr_sulgas    ='//TRIM(num2str(nbtr_sulgas   )), modname)
      CALL msg('id_BIN01_strat ='//TRIM(num2str(id_BIN01_strat)), modname)
      CALL msg('id_OCS_strat   ='//TRIM(num2str(id_OCS_strat  )), modname)
      CALL msg('id_SO2_strat   ='//TRIM(num2str(id_SO2_strat  )), modname)
      CALL msg('id_H2SO4_strat ='//TRIM(num2str(id_H2SO4_strat)), modname)
      CALL msg('id_TEST_strat  ='//TRIM(num2str(id_TEST_strat )), modname)
   END IF

END SUBROUTINE init_infotrac_phy

END MODULE infotrac_phy
