! $Id: lmdz_cloud_optics_prop.f90 5797 2025-08-05 12:22:14Z aborella $
MODULE lmdz_cloud_optics_prop

CONTAINS

SUBROUTINE cloud_optics_prop(klon, klev, paprs, pplay, temp, radocond, picefra, pclf, pclc, &
    pcltau, pclemi, pch, pcl, pcm, pct, radocondwp, xflwp, xfiwp, xflwc, xfiwc, &
    mass_solu_aero, mass_solu_aero_pi, pcldtaupi, distcltop, temp_cltop, re, fl, reliq, reice, &
    reliq_pi, reice_pi, scdnc, cldncl, reffclwtop, lcc, reffclws, &
    reffclwc, cldnvi, lcc3d, lcc3dcon, lcc3dstra, icc3dcon, icc3dstra,  & 
    icefrac_optics, dNovrN, ptconv, rnebcon, ccwcon, &
    !--AB contrails
    contfravol, contfra, qice_cont, Nice_cont, pclc_cont, pcltau_nocont, &
    pclemi_nocont, pcltau_cont, pclemi_cont, pch_nocont, pct_cont, &
    xfiwp_cont, xfiwc_cont, reice_cont, &
    missing_val)

  USE lmdz_cloud_optics_prop_ini , ONLY : flag_aerosol, ok_cdnc
  USE lmdz_cloud_optics_prop_ini , ONLY : lunout
  USE lmdz_cloud_optics_prop_ini , ONLY : bl95_b0, bl95_b1
  USE lmdz_cloud_optics_prop_ini , ONLY : latitude_deg
  USE lmdz_cloud_optics_prop_ini , ONLY : iflag_t_glace
  USE lmdz_cloud_optics_prop_ini , ONLY : cdnc_max, cdnc_max_m3
  USE lmdz_cloud_optics_prop_ini , ONLY : cdnc_min, cdnc_min_m3
  USE lmdz_cloud_optics_prop_ini , ONLY : thres_tau, thres_neb
  USE lmdz_cloud_optics_prop_ini , ONLY : prmhc, prlmc
  USE lmdz_cloud_optics_prop_ini , ONLY : coef_froi, coef_chau
  USE lmdz_cloud_optics_prop_ini , ONLY : seuil_neb
  USE lmdz_cloud_optics_prop_ini , ONLY : t_glace_min_old, t_glace_max_old
  USE lmdz_cloud_optics_prop_ini , ONLY : k_ice0, df
  USE lmdz_cloud_optics_prop_ini , ONLY : rg, rd, rpi
  USE lmdz_cloud_optics_prop_ini , ONLY : rad_chau1, rad_chau2, iflag_rei
  USE lmdz_cloud_optics_prop_ini , ONLY : ok_icefra_lscp, rei_max, rei_min
  USE lmdz_cloud_optics_prop_ini , ONLY : rei_coef, rei_min_temp
  USE lmdz_cloud_optics_prop_ini , ONLY : zepsec, novlp, iflag_ice_thermo, ok_new_lscp
  USE lmdz_cloud_optics_prop_ini , ONLY : ok_plane_contrail, rho_ice
  USE lmdz_cloud_optics_prop_ini , ONLY : eff2vol_radius_contrails, rei_min_contrails
  



  IMPLICIT NONE
  ! ======================================================================
  ! Authors: Z.X. Li (LMD/CNRS) date: 19930910
  !          O.Boucher (LMD/CNRS) mise a jour en 201212
  !          I. Musat (LMD/CNRS) : prise en compte de la meme hypothese 
  !                              de recouvrement pour les nuages que pour 
  !                              le rayonnement rrtm via le parametre
  !                               novlp de radopt.h : 20160721
  !          L.Fairheard, E.Vignon, JB Madeleine, L. Raillard, A. Idelkadi
  !          M. Coulon-Decorzens: replayisation of the routine + cleaning
  !                               and commentaries
  !
  ! Aim: compute condensate optical properties,
  !      cloud optical depth and emissivity
  ! ======================================================================
  
  ! List of arguments
  !------------------

  ! input:
  INTEGER, INTENT(IN) :: klon, klev      ! number of horizontal and vertical grid points
  REAL, INTENT(IN) :: missing_val
  REAL, INTENT(IN) :: paprs(klon, klev+1)! pressure at bottom interfaces [Pa]
  REAL, INTENT(IN) :: pplay(klon, klev)  ! pressure at the middle of layers [Pa]
  REAL, INTENT(IN) :: temp(klon, klev)   ! temperature [K]
  REAL, INTENT(IN) :: radocond(klon, klev) ! cloud condensed water seen by radiation [kg/kg]
  REAL, INTENT(IN) :: picefra(klon,klev) ! ice fraction in clouds from large scale condensation scheme [-]
  REAL, INTENT(IN) :: rnebcon(klon,klev) ! convection cloud fraction [-]
  REAL, INTENT(IN) :: ccwcon(klon,klev)  ! condensed water from deep convection [kg/kg]
  ! jq for the aerosol indirect effect
  ! jq introduced by Johannes Quaas (quaas@lmd.jussieu.fr), 27/11/2003
  REAL, INTENT(IN) :: mass_solu_aero(klon, klev)    ! total mass concentration for all soluble aerosols [ug m-3]
  REAL, INTENT(IN) :: mass_solu_aero_pi(klon, klev) ! - (pre-industrial value)
  REAL, INTENT(IN)  :: dNovrN(klon)         ! enhancement factor for cdnc
  REAL, INTENT(OUT) :: distcltop(klon,klev) ! distance from large scale cloud top [m]
  REAL, INTENT(OUT) :: temp_cltop(klon,klev)!temperature at large scale cloud top [K]
  REAL, INTENT(IN) :: pclf(klon, klev)      ! cloud fraction for radiation [-]

  LOGICAL, INTENT(IN) :: ptconv(klon, klev) ! flag for grid points affected by deep convection

  ! inout:
  REAL, INTENT(INOUT) :: pclc(klon, klev) ! cloud cover for radiation [-]

  ! out:
  REAL, INTENT(OUT) :: pct(klon)      ! 2D total cloud cover [-]
  REAL, INTENT(OUT) :: pcl(klon)      ! 2D low cloud cover [-]
  REAL, INTENT(OUT) :: pcm(klon)      ! 2D mid cloud cover [-]
  REAL, INTENT(OUT) :: pch(klon)      ! 2D high cloud cover [-]
  REAL, INTENT(OUT) :: radocondwp(klon) ! total condensed water path (seen by radiation) [kg/m2]
  REAL, INTENT(OUT) :: xflwp(klon)    ! liquid water path (seen by radiation) [kg/m2]
  REAL, INTENT(OUT) :: xfiwp(klon)    ! ice water path (seen by radiation) [kg/m2]
  REAL, INTENT(OUT) :: xflwc(klon, klev) ! liquid water content seen by radiation [kg/kg]
  REAL, INTENT(OUT) :: xfiwc(klon, klev) ! ice water content seen by radiation [kg/kg]
  REAL, INTENT(OUT) :: re(klon, klev) ! cloud droplet effective radius multiplied by fl
  REAL, INTENT(OUT) :: fl(klon, klev) ! xliq * rneb, denominator to re; fraction of liquid water clouds
                                      ! introduced to avoid problems in the averaging of the output
                                      ! water clouds within a grid cell

  REAL, INTENT(OUT) :: pcltau(klon, klev) ! cloud optical depth [m]
  REAL, INTENT(OUT) :: pclemi(klon, klev) ! cloud emissivity [-]
  REAL, INTENT(OUT) :: pcldtaupi(klon, klev) ! pre-industrial value of cloud optical thickness, ie.
                                             ! values of optical thickness that does not account
                                             ! for aerosol effects on cloud droplet radius [m]

  REAL, INTENT(OUT) :: reliq(klon, klev)   ! liquid droplet effective radius [m] 
  REAL, INTENT(OUT) :: reice(klon, klev)   ! ice effective radius [m]
  REAL, INTENT(OUT) :: reliq_pi(klon, klev)! liquid droplet effective radius [m], pre-industrial
  REAL, INTENT(OUT) :: reice_pi(klon, klev)! ice effective radius [m], pre-industrial
  REAL, INTENT(OUT) :: scdnc(klon, klev)   ! cloud droplet number concentration, mean over the whole mesh [m-3]
  REAL, INTENT(OUT) :: cldncl(klon)        ! cloud droplet number concentration at top of cloud [m-3]
  REAL, INTENT(OUT) :: reffclwtop(klon)    ! effective radius of cloud droplet at top of cloud [m] 
  REAL, INTENT(OUT) :: lcc(klon)           ! liquid Cloud Content at top of cloud [kg/kg]
  REAL, INTENT(OUT) :: reffclws(klon, klev)! stratiform cloud droplet effective radius 
  REAL, INTENT(OUT) :: reffclwc(klon, klev)! convective cloud droplet effective radius 
  REAL, INTENT(OUT) :: cldnvi(klon)        ! column Integrated cloud droplet Number [/m2]
  REAL, INTENT(OUT) :: lcc3d(klon, klev)   ! cloud fraction for liquid part only [-]
  REAL, INTENT(OUT) :: lcc3dcon(klon, klev)! cloud fraction for liquid part only, convective clouds [-]
  REAL, INTENT(OUT) :: lcc3dstra(klon, klev)!cloud fraction for liquid part only, stratiform clouds [-]
  REAL, INTENT(OUT) :: icc3dcon(klon, klev)! cloud fraction for liquid part only, convective clouds [-]
  REAL, INTENT(OUT) :: icc3dstra(klon, klev)! cloud fraction for ice part only, stratiform clouds [-] 
  REAL, INTENT(INOUT) :: icefrac_optics(klon, klev)! ice fraction in clouds seen by radiation [-]

  !--AB for contrails. All these are used / outputed only if ok_plane_contrail=y
  REAL, INTENT(IN)  :: contfravol(klon, klev)    ! contrails volumic fraction [-]
  REAL, INTENT(IN)  :: contfra(klon, klev)       ! contrails fraction for radiation [-]
  REAL, INTENT(IN)  :: qice_cont(klon, klev)     ! contrails condensed water [kg/kg]
  REAL, INTENT(IN)  :: Nice_cont(klon, klev)     ! contrails ice crystals concentration [#/kg]
  REAL, INTENT(OUT) :: pch_nocont(klon)          ! 2D high cloud cover without contrails[-]
  REAL, INTENT(OUT) :: pct_cont(klon)            ! 2D total contrails cover[-]
  REAL, INTENT(OUT) :: xfiwp_cont(klon)          ! ice water path (seen by radiation) of contrails [kg/m2]
  REAL, INTENT(OUT) :: xfiwc_cont(klon, klev)    ! ice water content seen by radiation of contrails [kg/kg]
  REAL, INTENT(OUT) :: pclc_cont(klon, klev)     ! cloud fraction for radiation of contrails [-]
  REAL, INTENT(OUT) :: pcltau_nocont(klon, klev) ! cloud optical depth without contrails [-]
  REAL, INTENT(OUT) :: pclemi_nocont(klon, klev) ! cloud emissivity without contrails [-]
  REAL, INTENT(OUT) :: pcltau_cont(klon, klev)   ! contrails optical depth [-]
  REAL, INTENT(OUT) :: pclemi_cont(klon, klev)   ! contrails emissivity [-]
  REAL, INTENT(OUT) :: reice_cont(klon, klev)    ! ice effective radius of contrails [microns]
  !--AB

  ! Local variables
  !----------------

  LOGICAL, SAVE :: first = .TRUE.
  !$OMP THREADPRIVATE(first)
  INTEGER flag_max

  ! threshold PARAMETERs
  REAL phase3d(klon, klev)
  REAL tcc(klon), ftmp(klon), lcc_integrat(klon), height(klon)
  LOGICAL lo
  INTEGER i, k
  REAL radius


  REAL rel, tc, rei, iwc, dei, deimin, deimax
  REAL k_ice

  ! jq for the aerosol indirect effect
  ! jq introduced by Johannes Quaas (quaas@lmd.jussieu.fr), 27/11/2003
  REAL cdnc(klon, klev) ! cloud droplet number concentration [m-3]
  REAL cdnc_pi(klon, klev) ! cloud droplet number concentration [m-3] (pi value)
  REAL re_pi(klon, klev) ! cloud droplet effective radius [um] (pi value)

  ! IM cf. CR:parametres supplementaires
  REAL dzfice(klon,klev)
  REAL zclear(klon)
  REAL zcloud(klon)
  REAL zcloudh(klon)
  REAL zcloudm(klon)
  REAL zcloudl(klon)
  REAL rhodz(klon, klev) !--rho*dz pour la couche
  REAL zrho(klon, klev) !--rho pour la couche
  REAL dh(klon, klev) !--dz pour la couche
  REAL rad_chaud(klon, klev) !--rayon pour les nuages chauds
  REAL rad_chaud_pi(klon, klev) !--rayon pour les nuages chauds pre-industriels
  REAL zflwp_var, zfiwp_var
  REAL d_rei_dt
  REAL pclc_nocont(klon,klev)
  REAL pclf_nocont(klon,klev)
  REAL xfiwc_nocont(klon,klev)
  REAL mice_cont, rei_cont


  ! FH : 2011/05/24
  ! rei = ( rei_max - rei_min ) * T(°C) / 81.4 + rei_max
  ! to be used for a temperature in celcius T(°C) < 0
  ! rei=rei_min for T(°C) < -81.4
  ! Calcul de la pente de la relation entre rayon effective des cristaux
  ! et la température Pour retrouver les résultats numériques de la version d'origine,
  ! on impose 0.71 quand on est proche de 0.71
  d_rei_dt = (rei_max-rei_min)/81.4
  IF (abs(d_rei_dt-0.71)<1.E-4) d_rei_dt = 0.71

  ! Calculer l'epaisseur optique et l'emmissivite des nuages
  ! IM inversion des DO

  xflwp = 0.D0
  xfiwp = 0.D0
  xflwc = 0.D0
  xfiwc = 0.D0
  !--AB
  xfiwp_cont = 0.D0
  xfiwc_cont = 0.D0
  reice_cont = 1.

  reliq = 0.
  reice = 0.
  reliq_pi = 0.
  reice_pi = 0.

  IF ((.NOT. ok_new_lscp) .AND. iflag_t_glace.EQ.0) THEN
    DO k = 1, klev
      DO i = 1, klon
        ! -layer calculation
        rhodz(i, k) = (paprs(i,k)-paprs(i,k+1))/rg ! kg/m2
        zrho(i, k) = pplay(i, k)/temp(i, k)/rd ! kg/m3
        dh(i, k) = rhodz(i, k)/zrho(i, k) ! m
        ! -Fraction of ice in cloud using a linear transition
        icefrac_optics(i, k) = 1.0 - (temp(i,k)-t_glace_min_old)/(t_glace_max_old-t_glace_min_old)
        icefrac_optics(i, k) = min(max(icefrac_optics(i,k),0.0), 1.0)
        ! -IM Total Liquid/Ice water content
        xflwc(i, k) = (1.-icefrac_optics(i,k))*radocond(i, k)
        xfiwc(i, k) = icefrac_optics(i, k)*radocond(i, k)
      ENDDO
    ENDDO
  ELSE ! of IF (iflag_t_glace.EQ.0)
    DO k = 1, klev


!!$      IF (ok_new_lscp) THEN
!!$          CALL icefrac_lscp(klon,temp(:,k),iflag_ice_thermo,distcltop(:,k),temp_cltop(:,k),icefrac_optics(:,k),dzfice(:,k))
!!$      ELSE
!!$          CALL icefrac_lsc(klon,temp(:,k),pplay(:,k)/paprs(:,1),icefrac_optics(:,k))
!!$      ENDIF

      DO i = 1, klon

        IF ((.NOT. ptconv(i,k)) .AND. ok_new_lscp .AND. ok_icefra_lscp) THEN
        ! EV: take the ice fraction directly from the lscp code
        ! consistent only for non convective grid points
        ! critical for mixed phase clouds
            icefrac_optics(i,k)=picefra(i,k)
        ENDIF

        ! -layer calculation
        rhodz(i, k) = (paprs(i,k)-paprs(i,k+1))/rg ! kg/m2
        zrho(i, k) = pplay(i, k)/temp(i, k)/rd ! kg/m3
        dh(i, k) = rhodz(i, k)/zrho(i, k) ! m
        ! -IM Total Liquid/Ice water content
        xflwc(i, k) = (1.-icefrac_optics(i,k))*radocond(i, k)
        xfiwc(i, k) = icefrac_optics(i, k)*radocond(i, k)
      ENDDO
    ENDDO

    !--AB
    IF ( ok_plane_contrail ) THEN
      !--If contrails are activated, we diagnose iwc without contrails
      DO k = 1, klev
        DO i = 1, klon
          pclf_nocont(i,k) = MAX(0., pclf(i, k) - contfravol(i, k))
          pclc_nocont(i,k) = MAX(0., pclc(i, k) - contfra(i, k))
          xfiwc_nocont(i, k) = MAX(0., xfiwc(i, k) - qice_cont(i, k))
          xfiwc_cont(i,k) = qice_cont(i,k)
        ENDDO
      ENDDO
    ENDIF
  ENDIF






  IF (ok_cdnc) THEN

    ! --we compute cloud properties as a function of the aerosol load

    DO k = 1, klev
      DO i = 1, klon
        ! Formula "D" of Boucher and Lohmann, Tellus, 1995
        ! Cloud droplet number concentration (CDNC) is restricted
        ! to be within [20, 1000 cm^3]

        ! --pre-industrial case
        cdnc_pi(i, k) = 10.**(bl95_b0+bl95_b1*log(max(mass_solu_aero_pi(i,k), &
          1.E-4))/log(10.))*1.E6 !-m-3
        cdnc_pi(i, k) = min(cdnc_max_m3, max(cdnc_min_m3,cdnc_pi(i,k)))

      ENDDO
    ENDDO

    !--flag_aerosol=7 => MACv2SP climatology 
    !--in this case there is an enhancement factor
    IF (flag_aerosol .EQ. 7) THEN 

      !--present-day
      DO k = 1, klev
        DO i = 1, klon
          cdnc(i, k) = cdnc_pi(i,k)*dNovrN(i)
        ENDDO
      ENDDO

    !--standard case
    ELSE

      DO k = 1, klev
        DO i = 1, klon

          ! Formula "D" of Boucher and Lohmann, Tellus, 1995
          ! Cloud droplet number concentration (CDNC) is restricted
          ! to be within [20, 1000 cm^3]

          ! --present-day case
          cdnc(i, k) = 10.**(bl95_b0+bl95_b1*log(max(mass_solu_aero(i,k), &
            1.E-4))/log(10.))*1.E6 !-m-3
          cdnc(i, k) = min(cdnc_max_m3, max(cdnc_min_m3,cdnc(i,k)))

        ENDDO
      ENDDO

    ENDIF !--flag_aerosol

    !--computing cloud droplet size
    DO k = 1, klev
      DO i = 1, klon

        ! --present-day case
        rad_chaud(i, k) = 1.1*((radocond(i,k)*pplay(i, &
          k)/(rd*temp(i,k)))/(4./3*rpi*1000.*cdnc(i,k)))**(1./3.)
        rad_chaud(i, k) = max(rad_chaud(i,k)*1.E6, 5.)

        ! --pre-industrial case
        rad_chaud_pi(i, k) = 1.1*((radocond(i,k)*pplay(i, &
          k)/(rd*temp(i,k)))/(4./3.*rpi*1000.*cdnc_pi(i,k)))**(1./3.)
        rad_chaud_pi(i, k) = max(rad_chaud_pi(i,k)*1.E6, 5.)

        ! --pre-industrial case
        ! --liquid/ice cloud water paths:
        IF (pclc(i,k)<=seuil_neb) THEN

          pcldtaupi(i, k) = 0.0

        ELSE

          zflwp_var = 1000.*(1.-icefrac_optics(i,k))*radocond(i, k)/pclc(i, k)* &
            rhodz(i, k)
          zfiwp_var = 1000.*icefrac_optics(i, k)*radocond(i, k)/pclc(i, k)*rhodz(i, k)
          ! Calculation of ice cloud effective radius in micron
          IF (iflag_rei .EQ. 2) THEN
            ! in-cloud ice water content in g/m3
            IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclc(i,k) * zrho(i,k) * 1000.
            ELSE
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclf(i,k) * zrho(i,k) * 1000.
            ENDIF
            ! this formula is a simplified version of the Sun 2001 one (as in the IFS model,
            ! and which is activated for iflag_rei = 1).
            ! In particular, the term in temperature**2 has been simplified.
            ! The new coefs are tunable, and are by default set so that the results fit well
            ! to the Sun 2001 formula
            ! By default, rei_coef = 2.4 and rei_min_temp = 175.
            ! The ranges of these parameters are determined so that the RMSE between this
            ! formula and the one from Sun 2001 is less than 4 times the minimum RMSE
            ! The ranges are [1.9, 2.9] for rei_coef and [160., 185.] for rei_min_temp
            dei = rei_coef * (iwc**0.2445) * ( temp(i,k) - rei_min_temp )
            ! we clip the results
            deimin = 20. + 40. * COS(ABS(latitude_deg(i)) / 180. * RPI)
            deimax = 155.
            dei = MIN(MAX(dei, deimin), deimax)
            ! formula to convert effective diameter in effective radius
            rei = 3. * SQRT(3.) / 8. * dei
          ELSEIF (iflag_rei .EQ. 1) THEN
            ! when we account for precipitation in the radiation scheme,
            ! It is recommended to use the rei formula from Sun and Rikkus 1999 with a revision
            ! from Sun 2001 (as in the IFS model)
            ! in cloud ice water content in g/m3
            IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclc(i,k) * zrho(i,k) * 1000.
            ELSE
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclf(i,k) * zrho(i,k) * 1000.
            ENDIF
            dei=(1.2351+0.0105*(temp(i,k)-273.15))*(45.8966*(iwc**0.2214) + &
               & 0.7957*(iwc**0.2535)*(temp(i,k)-83.15))
            !deimax=155.0
            !deimin=20.+40*cos(abs(latitude_deg(i))/180.*RPI)
            !Etienne: deimax and deimin controled by rei_max and rei_min in physiq.def
            deimax=rei_max*2.0
            deimin=2.0*rei_min+40.*cos(abs(latitude_deg(i))/180.*RPI) 
            dei=min(dei,deimax)
            dei=max(dei,deimin)
            rei=3.*sqrt(3.)/8.*dei
           ELSE
            ! Default
            ! for ice clouds: as a function of the ambiant temperature
            ! [formula used by Iacobellis and Somerville (2000), with an
            ! asymptotical value of 3.5 microns at T<-81.4 C added to be
            ! consistent with observations of Heymsfield et al. 1986]:
            ! 2011/05/24 : rei_min = 3.5 becomes a free PARAMETER as well as
            ! rei_max=61.29
            tc = temp(i, k) - 273.15
            rei = d_rei_dt*tc + rei_max
            IF (tc<=-81.4) rei = rei_min
           ENDIF

          ! -- cloud optical thickness :
          ! [for liquid clouds, traditional formula,
          ! for ice clouds, Ebert & Curry (1992)]

          IF (zfiwp_var==0. .OR. rei<=0.) rei = 1.

          pcldtaupi(i, k) = 3.0/2.0*zflwp_var/rad_chaud_pi(i, k) + &
            zfiwp_var*(3.448E-03+2.431/rei)

        ENDIF

      ENDDO
    ENDDO

  ELSE !--not ok_cdnc

    ! -prescribed cloud droplet radius

    DO k = 1, min(3, klev)
      DO i = 1, klon
        rad_chaud(i, k) = rad_chau2
        rad_chaud_pi(i, k) = rad_chau2
      ENDDO
    ENDDO
    DO k = min(3, klev) + 1, klev
      DO i = 1, klon
        rad_chaud(i, k) = rad_chau1
        rad_chaud_pi(i, k) = rad_chau1
      ENDDO
    ENDDO

  ENDIF !--ok_cdnc

  ! --computation of cloud optical depth and emissivity
  ! --in the general case

  IF ( .NOT. ok_plane_contrail ) THEN

  DO k = 1, klev
    DO i = 1, klon

      IF (pclc(i,k)<=seuil_neb) THEN

        ! effective cloud droplet radius (microns) for liquid water clouds:
        ! For output diagnostics cloud droplet effective radius [um]
        ! we multiply here with f * xl (fraction of liquid water
        ! clouds in the grid cell) to avoid problems in the averaging of the
        ! output.
        ! In the output of IOIPSL, derive the REAL cloud droplet
        ! effective radius as re/fl

        fl(i, k) = seuil_neb*(1.-icefrac_optics(i,k))
        re(i, k) = rad_chaud(i, k)*fl(i, k)
        rel = 0.
        rei = 0.
        pclc(i, k) = 0.0
        pcltau(i, k) = 0.0
        pclemi(i, k) = 0.0

      ELSE

        ! -- liquid/ice cloud water paths:

        zflwp_var = 1000.*(1.-icefrac_optics(i,k))*radocond(i, k)/pclc(i, k)*rhodz(i, k)
        zfiwp_var = 1000.*icefrac_optics(i, k)*radocond(i, k)/pclc(i, k)*rhodz(i, k)

        ! effective cloud droplet radius (microns) for liquid water clouds:
        ! For output diagnostics cloud droplet effective radius [um]
        ! we multiply here with f Effective radius of cloud droplet at top of cloud (m)* xl (fraction of liquid water
        ! clouds in the grid cell) to avoid problems in the averaging of the
        ! output.
        ! In the output of IOIPSL, derive the REAL cloud droplet
        ! effective radius as re/fl

        fl(i, k) = pclc(i, k)*(1.-icefrac_optics(i,k))
        re(i, k) = rad_chaud(i, k)*fl(i, k)

        rel = rad_chaud(i, k)

        ! Calculation of ice cloud effective radius in micron


        IF (iflag_rei .EQ. 2) THEN
            ! in-cloud ice water content in g/m3
            IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclc(i,k) * zrho(i,k) * 1000.
            ELSE
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclf(i,k) * zrho(i,k) * 1000.
            ENDIF
            ! this formula is a simplified version of the Sun 2001 one (as in the IFS model,
            ! and which is activated for iflag_rei = 1).
            ! In particular, the term in temperature**2 has been simplified.
            ! The new coefs are tunable, and are by default set so that the results fit well
            ! to the Sun 2001 formula
            ! By default, rei_coef = 2.4 and rei_min_temp = 175.
            ! The ranges of these parameters are determined so that the RMSE between this
            ! formula and the one from Sun 2001 is less than 4 times the minimum RMSE
            ! The ranges are [1.9, 2.9] for rei_coef and [160., 185.] for rei_min_temp
            dei = rei_coef * (iwc**0.2445) * ( temp(i,k) - rei_min_temp )
            ! we clip the results
            deimin = 20. + 40. * COS(ABS(latitude_deg(i)) / 180. * RPI)
            deimax = 155.
            dei = MIN(MAX(dei, deimin), deimax)
            ! formula to convert effective diameter to effective radius
            rei = 3. * SQRT(3.) / 8. * dei

        ELSEIF (iflag_rei .EQ. 1) THEN
            ! when we account for precipitation in the radiation scheme,
            ! we use the rei formula from Sun and Rikkus 1999 with a revision
            ! from Sun 2001 (as in the IFS model)
            ! in cloud ice water content in g/m3
            IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclc(i,k) * zrho(i,k) * 1000.
            ELSE
              iwc = icefrac_optics(i,k) * radocond(i,k) / pclf(i,k) * zrho(i,k) * 1000.
            ENDIF
            dei=(1.2351+0.0105*(temp(i,k)-273.15))*(45.8966*(iwc**0.2214) + &
               &0.7957*(iwc**0.2535)*(temp(i,k)-83.15))
            !deimax=155.0
            !deimin=20.+40*cos(abs(latitude_deg(i))/180.*RPI)
            !Etienne: deimax and deimin controled by rei_max and rei_min in physiq.def
            deimax=rei_max*2.0
            deimin=2.0*rei_min+40.*cos(abs(latitude_deg(i))/180.*RPI) 
            dei=min(dei,deimax)
            dei=max(dei,deimin)
            rei=3.*sqrt(3.)/8.*dei
       
        ELSE
            ! Default
            ! for ice clouds: as a function of the ambiant temperature
            ! [formula used by Iacobellis and Somerville (2000), with an
            ! asymptotical value of 3.5 microns at T<-81.4 C added to be
            ! consistent with observations of Heymsfield et al. 1986]:
            ! 2011/05/24 : rei_min = 3.5 becomes a free PARAMETER as well as
            ! rei_max=61.29
            tc = temp(i, k) - 273.15
            rei = d_rei_dt*tc + rei_max
            IF (tc<=-81.4) rei = rei_min
        ENDIF
        ! -- cloud optical thickness :
        ! [for liquid clouds, traditional formula,
        ! for ice clouds, Ebert & Curry (1992)]

        IF (zflwp_var==0.) rel = 1.
        IF (zfiwp_var==0. .OR. rei<=0.) rei = 1.

        pcltau(i, k) = 3.0/2.0*(zflwp_var/rel) + zfiwp_var*(3.448E-03+2.431/ &
          rei)

        ! -- cloud infrared emissivity:
        ! [the broadband infrared absorption coefficient is PARAMETERized
        ! as a function of the effective cld droplet radius]
        ! Ebert and Curry (1992) formula as used by Kiehl & Zender (1995):

        k_ice = k_ice0 + 1.0/rei

        pclemi(i, k) = 1.0 - exp(-coef_chau*zflwp_var-df*k_ice*zfiwp_var)

      ENDIF

      reice(i, k) = rei

      xflwp(i) = xflwp(i) + xflwc(i, k)*rhodz(i, k)
      xfiwp(i) = xfiwp(i) + xfiwc(i, k)*rhodz(i, k)

    ENDDO
  ENDDO

  ELSE !--AB if contrails are activated

  DO k = 1, klev
    DO i = 1, klon

      IF (pclc(i,k)<=seuil_neb) THEN

        ! effective cloud droplet radius (microns) for liquid water clouds:
        ! For output diagnostics cloud droplet effective radius [um]
        ! we multiply here with f * xl (fraction of liquid water
        ! clouds in the grid cell) to avoid problems in the averaging of the
        ! output.
        ! In the output of IOIPSL, derive the REAL cloud droplet
        ! effective radius as re/fl

        fl(i, k) = seuil_neb*(1.-icefrac_optics(i,k))
        re(i, k) = rad_chaud(i, k)*fl(i, k)
        rel = 0.
        rei = 1.
        pclc(i, k) = 0.0
        pcltau(i, k) = 0.0
        pclemi(i, k) = 0.0

        !--AB contrails
        rei_cont = 1.
        pclc_nocont(i,k) = 0.
        pclc_cont(i,k) = 0.
        pcltau_cont(i,k) = missing_val
        pclemi_cont(i,k) = missing_val
        pcltau_nocont(i,k) = missing_val
        pclemi_nocont(i,k) = missing_val

      ELSE

        IF ( pclc_nocont(i,k) .GT. zepsec ) THEN
          ! -- liquid/ice cloud water paths:

          zflwp_var = 1000.*xflwc(i, k)/pclc(i, k)*rhodz(i, k)
          zfiwp_var = 1000.*xfiwc_nocont(i, k)/pclc_nocont(i, k)*rhodz(i, k)

          ! effective cloud droplet radius (microns) for liquid water clouds:
          ! For output diagnostics cloud droplet effective radius [um]
          ! we multiply here with f Effective radius of cloud droplet at top of cloud (m)* xl (fraction of liquid water
          ! clouds in the grid cell) to avoid problems in the averaging of the
          ! output.
          ! In the output of IOIPSL, derive the REAL cloud droplet
          ! effective radius as re/fl

          fl(i, k) = pclc_nocont(i, k)*(1.-icefrac_optics(i,k))
          re(i, k) = rad_chaud(i, k)*fl(i, k)

          rel = rad_chaud(i, k)

          ! Calculation of ice cloud effective radius in micron

          IF (iflag_rei .EQ. 2) THEN
              ! in-cloud ice water content in g/m3
              IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
                iwc = xfiwc_nocont(i, k) / pclc_nocont(i,k) * zrho(i,k) * 1000.
              ELSE
                iwc = xfiwc_nocont(i, k) / pclf_nocont(i,k) * zrho(i,k) * 1000.
              ENDIF
              ! this formula is a simplified version of the Sun 2001 one (as in the IFS model,
              ! and which is activated for iflag_rei = 1).
              ! In particular, the term in temperature**2 has been simplified.
              ! The new coefs are tunable, and are by default set so that the results fit well
              ! to the Sun 2001 formula
              ! By default, rei_coef = 2.4 and rei_min_temp = 175.
              ! The ranges of these parameters are determined so that the RMSE between this
              ! formula and the one from Sun 2001 is less than 4 times the minimum RMSE
              ! The ranges are [1.9, 2.9] for rei_coef and [160., 185.] for rei_min_temp
              dei = rei_coef * (iwc**0.2445) * ( temp(i,k) - rei_min_temp )
              ! we clip the results
              deimin = 20. + 40. * COS(ABS(latitude_deg(i)) / 180. * RPI)
              deimax = 155.
              dei = MIN(MAX(dei, deimin), deimax)
              ! formula to convert effective diameter to effective radius
              rei = 3. * SQRT(3.) / 8. * dei
              rei = MAX(rei, rei_min)
              
          ELSEIF (iflag_rei .EQ. 1) THEN
              ! when we account for precipitation in the radiation scheme,
              ! we use the rei formula from Sun and Rikkus 1999 with a revision
              ! from Sun 2001 (as in the IFS model)
              ! in cloud ice water content in g/m3
              IF ( ptconv(i,k) ) THEN
              !--Needed because pclf does not contain convective clouds (should be fixed...)
                iwc = xfiwc_nocont(i, k) / pclc_nocont(i,k) * zrho(i,k) * 1000.
              ELSE
                iwc = xfiwc_nocont(i, k) / pclf_nocont(i,k) * zrho(i,k) * 1000.
              ENDIF
              dei=(1.2351+0.0105*(temp(i,k)-273.15))*(45.8966*(iwc**0.2214) + &
                 &0.7957*(iwc**0.2535)*(temp(i,k)-83.15))
              !deimax=155.0
              !deimin=20.+40*cos(abs(latitude_deg(i))/180.*RPI)
              !Etienne: deimax and deimin controled by rei_max and rei_min in physiq.def
              deimax=rei_max*2.0
              deimin=2.0*rei_min+40.*cos(abs(latitude_deg(i))/180.*RPI) 
              dei=min(dei,deimax)
              dei=max(dei,deimin)
              rei=3.*sqrt(3.)/8.*dei
       
          ELSE
              ! Default
              ! for ice clouds: as a function of the ambiant temperature
              ! [formula used by Iacobellis and Somerville (2000), with an
              ! asymptotical value of 3.5 microns at T<-81.4 C added to be
              ! consistent with observations of Heymsfield et al. 1986]:
              ! 2011/05/24 : rei_min = 3.5 becomes a free PARAMETER as well as
              ! rei_max=61.29
              tc = temp(i, k) - 273.15
              rei = d_rei_dt*tc + rei_max
              IF (tc<=-81.4) rei = rei_min
          ENDIF
          ! -- cloud optical thickness :
          ! [for liquid clouds, traditional formula,
          ! for ice clouds, Ebert & Curry (1992)]


          IF (zflwp_var==0.) rel = 1.
          IF (zfiwp_var==0. .OR. rei<=0.) rei = 1.

          !--Diagnostics of clouds emissivity, optical depth and ice crystal radius
          !--without contrails
          pcltau_nocont(i,k) = 3.0/2.0*(zflwp_var/rel) + zfiwp_var*(3.448E-03+2.431/rei)
          ! -- cloud infrared emissivity:
          ! [the broadband infrared absorption coefficient is PARAMETERized
          ! as a function of the effective cld droplet radius]
          ! Ebert and Curry (1992) formula as used by Kiehl & Zender (1995):
          k_ice = k_ice0 + 1.0/rei
          pclemi_nocont(i, k) = 1.0 - exp(-coef_chau*zflwp_var-df*k_ice*zfiwp_var)

        ELSE

          rel = 1.
          rei = 1.

          !--Diagnostics of clouds emissivity, optical depth and ice crystal radius
          !--without contrails
          pclc_nocont(i,k) = 0.
          pclc(i,k) = contfra(i,k)
          pcltau_nocont(i, k) = 0.
          pclemi_nocont(i, k) = 0.

        ENDIF


        IF ( contfra(i,k) .GT. zepsec ) THEN
          pclc_cont(i,k) = contfra(i,k)
          mice_cont = qice_cont(i,k) / MAX(1e-10, Nice_cont(i,k))
          !--Volumic radius (in meters)
          rei_cont = (mice_cont / rho_ice * 3. / 4. / RPI)**(1./3.)
          !--Effective radius (in microns)
          rei_cont = MIN(200., MAX(rei_min_contrails, rei_cont / eff2vol_radius_contrails * 1e6))
          zfiwp_var = 1000.*xfiwc_cont(i, k)/pclc_cont(i, k)*rhodz(i, k)

          pcltau_cont(i, k) = zfiwp_var*(3.448E-03+2.431/rei_cont)
          ! -- cloud infrared emissivity:
          ! [the broadband infrared absorption coefficient is PARAMETERized
          ! as a function of the effective cld droplet radius]
          ! Ebert and Curry (1992) formula as used by Kiehl & Zender (1995):
          k_ice = k_ice0 + 1.0/rei_cont
          pclemi_cont(i, k) = 1.0 - exp(-df*k_ice*zfiwp_var)
        ELSE
          pclc_cont(i,k) = 0.
          rei_cont = 1.
          pcltau_cont(i, k) = 0.
          pclemi_cont(i, k) = 0.
        ENDIF

        pcltau(i,k) = (pclc_nocont(i,k)*pcltau_nocont(i,k) &
            & + pclc_cont(i,k)*pcltau_cont(i,k)) &
            & / pclc(i,k)
        pclemi(i,k) = (pclc_nocont(i,k)*pclemi_nocont(i,k) &
            & + pclc_cont(i,k)*pclemi_cont(i,k)) &
            & / pclc(i,k)

        IF ( pclc_nocont(i,k) .EQ. 0. ) THEN
          pcltau_nocont(i,k) = missing_val
          pclemi_nocont(i,k) = missing_val
        ENDIF

        IF ( pclc_cont(i,k) .EQ. 0. ) THEN
          pcltau_cont(i,k) = missing_val
          pclemi_cont(i,k) = missing_val
        ENDIF

      ENDIF

      reice(i, k) = rei
      reice_cont(i,k) = rei_cont

      xflwp(i) = xflwp(i) + xflwc(i, k)*rhodz(i, k)
      xfiwp(i) = xfiwp(i) + xfiwc(i, k)*rhodz(i, k)
      xfiwp_cont(i) = xfiwp_cont(i) + xfiwc_cont(i, k)*rhodz(i, k)

    ENDDO
  ENDDO

  ENDIF ! ok_plane_contrail

  ! --if cloud droplet radius is fixed, then pcldtaupi=pcltau

  IF (.NOT. ok_cdnc) THEN
    DO k = 1, klev
      DO i = 1, klon
        pcldtaupi(i, k) = pcltau(i, k)
        reice_pi(i, k) = reice(i, k)
      ENDDO
    ENDDO
  ENDIF

  DO k = 1, klev
    DO i = 1, klon
      reliq(i, k) = rad_chaud(i, k)
      reliq_pi(i, k) = rad_chaud_pi(i, k)
      reice_pi(i, k) = reice(i, k)
    ENDDO
  ENDDO

  ! COMPUTE CLOUD LIQUID PATH AND TOTAL CLOUDINESS
  ! IM cf. CR:test: calcul prenant ou non en compte le recouvrement
  ! initialisations

  DO i = 1, klon
    zclear(i) = 1.
    zcloud(i) = 0.
    zcloudh(i) = 0.
    zcloudm(i) = 0.
    zcloudl(i) = 0.
    pch(i) = 1.0
    pcm(i) = 1.0
    pcl(i) = 1.0
    radocondwp(i) = 0.0
  ENDDO

  ! --calculation of liquid water path

  DO k = klev, 1, -1
    DO i = 1, klon
      radocondwp(i) = radocondwp(i) + radocond(i, k)*rhodz(i, k)
    ENDDO
  ENDDO

  ! --calculation of cloud properties with cloud overlap
  ! choix de l'hypothese de recouvrement nuageuse via radopt.h (IM, 19.07.2016)
  ! !novlp=1: max-random
  ! !novlp=2: maximum
  ! !novlp=3: random


  IF (novlp==1) THEN
    DO k = klev, 1, -1
      DO i = 1, klon
        zclear(i) = zclear(i)*(1.-max(pclc(i,k),zcloud(i)))/(1.-min(real( &
          zcloud(i),kind=8),1.-zepsec))
        pct(i) = 1. - zclear(i)
        IF (paprs(i,k)<prmhc) THEN
          pch(i) = pch(i)*(1.-max(pclc(i,k),zcloudh(i)))/(1.-min(real(zcloudh &
            (i),kind=8),1.-zepsec))
          zcloudh(i) = pclc(i, k)
        ELSE IF (paprs(i,k)>=prmhc .AND. paprs(i,k)<prlmc) THEN
          pcm(i) = pcm(i)*(1.-max(pclc(i,k),zcloudm(i)))/(1.-min(real(zcloudm &
            (i),kind=8),1.-zepsec))
          zcloudm(i) = pclc(i, k)
        ELSE IF (paprs(i,k)>=prlmc) THEN
          pcl(i) = pcl(i)*(1.-max(pclc(i,k),zcloudl(i)))/(1.-min(real(zcloudl &
            (i),kind=8),1.-zepsec))
          zcloudl(i) = pclc(i, k)
        ENDIF
        zcloud(i) = pclc(i, k)
      ENDDO
    ENDDO
  ELSE IF (novlp==2) THEN
    DO k = klev, 1, -1
      DO i = 1, klon
        zcloud(i) = max(pclc(i,k), zcloud(i))
        pct(i) = zcloud(i)
        IF (paprs(i,k)<prmhc) THEN
          pch(i) = min(pclc(i,k), pch(i))
        ELSE IF (paprs(i,k)>=prmhc .AND. paprs(i,k)<prlmc) THEN
          pcm(i) = min(pclc(i,k), pcm(i))
        ELSE IF (paprs(i,k)>=prlmc) THEN
          pcl(i) = min(pclc(i,k), pcl(i))
        ENDIF
      ENDDO
    ENDDO
  ELSE IF (novlp==3) THEN
    DO k = klev, 1, -1
      DO i = 1, klon
        zclear(i) = zclear(i)*(1.-pclc(i,k))
        pct(i) = 1 - zclear(i)
        IF (paprs(i,k)<prmhc) THEN
          pch(i) = pch(i)*(1.0-pclc(i,k))
        ELSE IF (paprs(i,k)>=prmhc .AND. paprs(i,k)<prlmc) THEN
          pcm(i) = pcm(i)*(1.0-pclc(i,k))
        ELSE IF (paprs(i,k)>=prlmc) THEN
          pcl(i) = pcl(i)*(1.0-pclc(i,k))
        ENDIF
      ENDDO
    ENDDO
  ENDIF

  DO i = 1, klon
    pch(i) = 1. - pch(i)
    pcm(i) = 1. - pcm(i)
    pcl(i) = 1. - pcl(i)
  ENDDO

  !--AB we recompute high cloud cover with contrails only
  !--and without contrails
  IF ( ok_plane_contrail ) THEN
    DO i = 1, klon
      zclear(i) = 1.
      zcloud(i) = 0.
      zcloudh(i) = 0.
      pch_nocont(i) = 1.
      pct_cont(i) = 1.
    ENDDO

    DO k = klev, 1, -1
      DO i = 1, klon
        zclear(i) = zclear(i)*(1.-max(pclc_cont(i,k),zcloud(i)))/&
          (1.-min(real(zcloud(i),kind=8),1.-zepsec))
        pct_cont(i) = 1. - zclear(i)
        zcloud(i) = pclc_cont(i,k)
        IF (paprs(i,k)<prmhc) THEN
          pch_nocont(i) = pch_nocont(i)*(1.-max(pclc_nocont(i,k),zcloudh(i)))/(1.-min(real( &
            zcloudh(i),kind=8),1.-zepsec))
          zcloudh(i) = pclc_nocont(i, k)
        ENDIF
      ENDDO
    ENDDO

    DO i = 1, klon
      pch_nocont(i) = 1. - pch_nocont(i)
    ENDDO
  ENDIF ! ok_plane_contrail

  ! ========================================================
  ! DIAGNOSTICS CALCULATION FOR CMIP5 PROTOCOL
  ! ========================================================
  ! change by Nicolas Yan (LSCE)
  ! Cloud Droplet Number Concentration (CDNC) : 3D variable
  ! Fractionnal cover by liquid water cloud (LCC3D) : 3D variable
  ! Cloud Droplet Number Concentration at top of cloud (CLDNCL) : 2D variable
  ! Droplet effective radius at top of cloud (REFFCLWTOP) : 2D variable
  ! Fractionnal cover by liquid water at top of clouds (LCC) : 2D variable

  IF (ok_cdnc) THEN

    DO k = 1, klev
      DO i = 1, klon
        phase3d(i, k) = 1 - icefrac_optics(i, k)
        IF (pclc(i,k)<=seuil_neb) THEN
          lcc3d(i, k) = seuil_neb*phase3d(i, k)
        ELSE
          lcc3d(i, k) = pclc(i, k)*phase3d(i, k)
        ENDIF
        scdnc(i, k) = lcc3d(i, k)*cdnc(i, k) ! m-3
      ENDDO
    ENDDO

    DO i = 1, klon
      lcc(i) = 0.
      reffclwtop(i) = 0.
      cldncl(i) = 0.
      IF (novlp.EQ.3 .OR. novlp.EQ.1) tcc(i) = 1.
      IF (novlp.EQ.2) tcc(i) = 0.
    ENDDO

    DO i = 1, klon
      DO k = klev - 1, 1, -1 !From TOA down

          ! Test, if the cloud optical depth exceeds the necessary
          ! threshold:

        IF (pcltau(i,k)>thres_tau .AND. pclc(i,k)>thres_neb) THEN

          IF (novlp.EQ.2) THEN
            IF (first) THEN
              WRITE (*, *) 'Hypothese de recouvrement: MAXIMUM'
              first = .FALSE.
            ENDIF
            flag_max = -1.
            ftmp(i) = max(tcc(i), pclc(i,k))
          ENDIF

          IF (novlp.EQ.3) THEN
            IF (first) THEN
              WRITE (*, *) 'Hypothese de recouvrement: RANDOM'
              first = .FALSE.
            ENDIF
            flag_max = 1.
            ftmp(i) = tcc(i)*(1-pclc(i,k))
          ENDIF

          IF (novlp.EQ.1) THEN
            IF (first) THEN
              WRITE (*, *) 'Hypothese de recouvrement: MAXIMUM_ &
                &                                             &
                &                                          RANDOM'
              first = .FALSE.
            ENDIF
            flag_max = 1.
            ftmp(i) = tcc(i)*(1.-max(pclc(i,k),pclc(i,k+1)))/(1.-min(pclc(i, &
              k+1),1.-thres_neb))
          ENDIF
          ! Effective radius of cloud droplet at top of cloud (m)
          reffclwtop(i) = reffclwtop(i) + rad_chaud(i, k)*1.0E-06*phase3d(i, &
            k)*(tcc(i)-ftmp(i))*flag_max
          ! CDNC at top of cloud (m-3)
          cldncl(i) = cldncl(i) + cdnc(i, k)*phase3d(i, k)*(tcc(i)-ftmp(i))* &
            flag_max
          ! Liquid Cloud Content at top of cloud
          lcc(i) = lcc(i) + phase3d(i, k)*(tcc(i)-ftmp(i))*flag_max
          ! Total Cloud Content at top of cloud
          tcc(i) = ftmp(i)

        ENDIF ! is there a visible, not-too-small cloud?
      ENDDO ! loop over k

      IF (novlp.EQ.3 .OR. novlp.EQ.1) tcc(i) = 1. - tcc(i)

    ENDDO ! loop over i

    ! ! Convective and Stratiform Cloud Droplet Effective Radius (REFFCLWC
    ! REFFCLWS)
    DO i = 1, klon
      DO k = 1, klev
        ! Weight to be used for outputs: eau_liquide*couverture nuageuse
        lcc3dcon(i, k) = rnebcon(i, k)*phase3d(i, k)*ccwcon(i, k) ! eau liquide convective
        lcc3dstra(i, k) = pclc(i, k)*radocond(i, k)*phase3d(i, k)
        lcc3dstra(i, k) = lcc3dstra(i, k) - lcc3dcon(i, k) ! eau liquide stratiforme
        lcc3dstra(i, k) = max(lcc3dstra(i,k), 0.0)
        !FC pour la glace (CAUSES)
        icc3dcon(i, k) = rnebcon(i, k)*(1-phase3d(i, k))*ccwcon(i, k) !  glace convective
        icc3dstra(i, k)= pclc(i, k)*radocond(i, k)*(1-phase3d(i, k))
        icc3dstra(i, k) = icc3dstra(i, k) - icc3dcon(i, k) ! glace stratiforme
        icc3dstra(i, k) = max( icc3dstra(i, k), 0.0)
        !FC (CAUSES)

        ! Compute cloud droplet radius as above in meter
        radius = 1.1*((radocond(i,k)*pplay(i,k)/(rd*temp(i,k)))/(4./3*rpi*1000.* &
          cdnc(i,k)))**(1./3.)
        radius = max(radius, 5.E-6)
        ! Convective Cloud Droplet Effective Radius (REFFCLWC) : variable 3D
        reffclwc(i, k) = radius
        reffclwc(i, k) = reffclwc(i, k)*lcc3dcon(i, k)
        ! Stratiform Cloud Droplet Effective Radius (REFFCLWS) : variable 3D
        reffclws(i, k) = radius
        reffclws(i, k) = reffclws(i, k)*lcc3dstra(i, k)
      ENDDO !klev
    ENDDO !klon

    ! Column Integrated Cloud Droplet Number (CLDNVI) : variable 2D

    DO i = 1, klon
      cldnvi(i) = 0.
      lcc_integrat(i) = 0.
      height(i) = 0.
      DO k = 1, klev
        cldnvi(i) = cldnvi(i) + cdnc(i, k)*lcc3d(i, k)*dh(i, k)
        lcc_integrat(i) = lcc_integrat(i) + lcc3d(i, k)*dh(i, k)
        height(i) = height(i) + dh(i, k)
      ENDDO ! klev
      lcc_integrat(i) = lcc_integrat(i)/height(i)
      IF (lcc_integrat(i)<=1.0E-03) THEN
        cldnvi(i) = cldnvi(i)*lcc(i)/seuil_neb
      ELSE
        cldnvi(i) = cldnvi(i)*lcc(i)/lcc_integrat(i)
      ENDIF
    ENDDO ! klon

    DO i = 1, klon
      DO k = 1, klev
        IF (scdnc(i,k)<=0.0) scdnc(i, k) = 0.0
        IF (reffclws(i,k)<=0.0) reffclws(i, k) = 0.0
        IF (reffclwc(i,k)<=0.0) reffclwc(i, k) = 0.0
        IF (lcc3d(i,k)<=0.0) lcc3d(i, k) = 0.0
        IF (lcc3dcon(i,k)<=0.0) lcc3dcon(i, k) = 0.0
        IF (lcc3dstra(i,k)<=0.0) lcc3dstra(i, k) = 0.0
!FC (CAUSES)
        IF (icc3dcon(i,k)<=0.0) icc3dcon(i, k) = 0.0
        IF (icc3dstra(i,k)<=0.0) icc3dstra(i, k) = 0.0
!FC (CAUSES)
      ENDDO
      IF (reffclwtop(i)<=0.0) reffclwtop(i) = 0.0
      IF (cldncl(i)<=0.0) cldncl(i) = 0.0
      IF (cldnvi(i)<=0.0) cldnvi(i) = 0.0
      IF (lcc(i)<=0.0) lcc(i) = 0.0
    ENDDO

  ENDIF !ok_cdnc

  first=.false. !to be sure

  RETURN

END SUBROUTINE cloud_optics_prop

END MODULE lmdz_cloud_optics_prop
