MODULE strings_mod

  USE, INTRINSIC :: ISO_FORTRAN_ENV, ONLY: REAL64, REAL32

  IMPLICIT NONE

  PRIVATE
  PUBLIC :: maxlen, init_printout, msg, get_in, lunout, prt_level
  PUBLIC :: strLower, strHead, strStack,  strCount, strReduce,  strClean, strIdx
  PUBLIC :: strUpper, strTail, strStackm, strParse, strReplace, strFind, find, duplicate, cat
  PUBLIC :: dispTable, dispOutliers, dispNameList
  PUBLIC :: is_numeric, num2str, str2bool, str2int, str2real, str2dble
  PUBLIC :: reduceExpr, addQuotes, checkList, removeComment

  INTERFACE get_in;     MODULE PROCEDURE getin_s,  getin_i,  getin_r,  getin_l;  END INTERFACE get_in
  INTERFACE num2str;    MODULE PROCEDURE bool2str, int2str, real2str, dble2str;  END INTERFACE num2str
  INTERFACE  msg;       MODULE PROCEDURE        msg_1,                    msg_m; END INTERFACE  msg
  INTERFACE strHead;    MODULE PROCEDURE    strHead_1,                strHead_m; END INTERFACE strHead
  INTERFACE strTail;    MODULE PROCEDURE    strTail_1,                strTail_m; END INTERFACE strTail
  INTERFACE strClean;   MODULE PROCEDURE   strClean_1,               strClean_m; END INTERFACE strClean
  INTERFACE strReduce;  MODULE PROCEDURE  strReduce_1,              strReduce_2; END INTERFACE strReduce
  INTERFACE strIdx;     MODULE PROCEDURE     strIdx_1,                 strIdx_m; END INTERFACE strIdx
  INTERFACE strCount;   MODULE PROCEDURE  strCount_m1, strCount_11, strCount_1m; END INTERFACE strCount
  INTERFACE strReplace; MODULE PROCEDURE strReplace_1,             strReplace_m; END INTERFACE strReplace
  INTERFACE cat; MODULE PROCEDURE  horzcat_s00, horzcat_i00, horzcat_r00, horzcat_d00, &
                                   horzcat_s10, horzcat_i10, horzcat_r10, horzcat_d10, &
                                   horzcat_s11, horzcat_i11, horzcat_r11, horzcat_d11, &
                                   horzcat_s21, horzcat_i21, horzcat_r21;  END INTERFACE cat !horzcat_d21
  INTERFACE strFind;      MODULE PROCEDURE strFind_1, strFind_m;           END INTERFACE strFind
  INTERFACE find;         MODULE PROCEDURE strFind_1, strFind_m, intFind_1, intFind_m, booFind; END INTERFACE find
  INTERFACE duplicate;    MODULE PROCEDURE dupl_s, dupl_i, dupl_r, dupl_l; END INTERFACE duplicate
  INTERFACE dispOutliers; MODULE PROCEDURE dispOutliers_1, dispOutliers_2; END INTERFACE dispOutliers
  INTERFACE reduceExpr;   MODULE PROCEDURE   reduceExpr_1,   reduceExpr_m; END INTERFACE reduceExpr
  INTERFACE addQuotes;    MODULE PROCEDURE    addQuotes_1,    addQuotes_m; END INTERFACE addQuotes

  INTEGER, PARAMETER :: maxlen    = 256                    !--- Standard maximum length for strings
  INTEGER,      SAVE :: lunout    = 6                      !--- Printing unit  (default: 6, ie. on screen)
  INTEGER,      SAVE :: prt_level = 1                      !--- Printing level (default: 1, ie. print all)

CONTAINS

!==============================================================================================================================
SUBROUTINE init_printout(lunout_, prt_level_)
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: lunout_, prt_level_
  lunout    = lunout_
  prt_level = prt_level_
END SUBROUTINE init_printout
!==============================================================================================================================


!==============================================================================================================================
!=== Same as getin ; additional last argument: the default value.
!==============================================================================================================================
SUBROUTINE getin_s(nam, val, def)
  USE ioipsl, ONLY: getin
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)    :: nam
  CHARACTER(LEN=*), INTENT(INOUT) :: val
  CHARACTER(LEN=*), INTENT(IN)    :: def
  val = def; CALL getin(nam, val)
  IF(val/=def) WRITE(lunout,*)TRIM(nam)//' = '//TRIM(val)
END SUBROUTINE getin_s
!==============================================================================================================================
SUBROUTINE getin_i(nam, val, def)
  USE ioipsl, ONLY: getin
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)    :: nam
  INTEGER,          INTENT(INOUT) :: val
  INTEGER,          INTENT(IN)    :: def
  val = def; CALL getin(nam, val)
  IF(val/=def) WRITE(lunout,*)TRIM(nam)//' = '//TRIM(num2str(val))
END SUBROUTINE getin_i
!==============================================================================================================================
SUBROUTINE getin_r(nam, val, def)
  USE ioipsl, ONLY: getin
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)    :: nam
  REAL,             INTENT(INOUT) :: val
  REAL,             INTENT(IN)    :: def
  val = def; CALL getin(nam, val)
  IF(val/=def) WRITE(lunout,*)TRIM(nam)//' = '//TRIM(num2str(val))
END SUBROUTINE getin_r
!==============================================================================================================================
SUBROUTINE getin_l(nam, val, def)
  USE ioipsl, ONLY: getin
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)    :: nam
  LOGICAL,          INTENT(INOUT) :: val
  LOGICAL,          INTENT(IN)    :: def
  val = def; CALL getin(nam, val)
  IF(val.NEQV.def) WRITE(lunout,*)TRIM(nam)//' = '//TRIM(num2str(val))
END SUBROUTINE getin_l
!==============================================================================================================================


!==============================================================================================================================
!=== Display one or several messages, one each line, starting with the current routine name "modname".
!==============================================================================================================================
SUBROUTINE msg_1(str, modname, ll, unit)
  IMPLICIT NONE
  !--- Display a simple message "str". Optional parameters:
  !    * "modname": module name, displayed in front of the message (with ": " separator) if present.
  !    * "ll":      message trigger ; message is displayed only if ll==.TRUE.
  !    * "unit":    write unit (by default: "lunout")
  CHARACTER(LEN=*),           INTENT(IN) :: str
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: modname
  LOGICAL,          OPTIONAL, INTENT(IN) :: ll
  INTEGER,          OPTIONAL, INTENT(IN) :: unit
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen) :: subn
  INTEGER :: unt
  subn = '';    IF(PRESENT(modname)) subn = modname
  IF(PRESENT(ll)) THEN; IF(.NOT.ll) RETURN; END IF
  unt = lunout; IF(PRESENT(unit)) unt = unit
  IF(subn == '') WRITE(unt,'(a)') str                                          !--- Simple message
  IF(subn /= '') WRITE(unt,'(a)') TRIM(subn)//': '//str                        !--- Routine name provided
END SUBROUTINE msg_1
!==============================================================================================================================
SUBROUTINE msg_m(str, modname, ll, unit, nmax)
  IMPLICIT NONE
  !--- Same as msg_1 with multiple strings that are stacked (separator: coma) on up to "nmax" full lines.
  CHARACTER(LEN=*),           INTENT(IN) :: str(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: modname
  LOGICAL,          OPTIONAL, INTENT(IN) :: ll
  INTEGER,          OPTIONAL, INTENT(IN) :: unit
  INTEGER,          OPTIONAL, INTENT(IN) :: nmax
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen), ALLOCATABLE :: s(:)
  CHARACTER(LEN=maxlen) :: subn
  INTEGER :: unt, nmx, k
  LOGICAL :: l
  subn = '';    IF(PRESENT(modname)) subn = modname
  l   = .TRUE.; IF(PRESENT(ll))     l = ll
  unt = lunout; IF(PRESENT(unit)) unt = unit
  nmx = 128;    IF(PRESENT(nmax)) nmx = nmax
  s = strStackm(str, ', ', nmx)
  DO k=1,SIZE(s); CALL msg_1(s(k), subn,  l,   unt); END DO
END SUBROUTINE msg_m
!==============================================================================================================================


!==============================================================================================================================
!=== Lower/upper case conversion function. ====================================================================================
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION strLower(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: k
  out = str
  DO k=1,LEN_TRIM(str)
    IF(str(k:k)>='A' .AND. str(k:k)<='Z') out(k:k)=ACHAR(IACHAR(str(k:k))+32)
  END DO
END FUNCTION strLower
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION strUpper(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: k
  out = str
  DO k=1,LEN_TRIM(str)
    IF(str(k:k)>='a' .AND. str(k:k)<='z') out(k:k)=ACHAR(IACHAR(str(k:k))-32)
  END DO
END FUNCTION strUpper
!==============================================================================================================================


!==============================================================================================================================
!=== Extract the substring in front of the first (last if lBackward==TRUE) occurrence of "sep" in "str"        ================
!=== Examples for str='a_b_c' and sep='_' and the bash command with the same effect:                           ================
!===    * strHead(..,.FALSE.) = 'a'           ${str%%$sep*}                                                    ================
!===    * strHead(..,.TRUE.)  = 'a_b'         ${str%$sep*}                                                     ================
!==============================================================================================================================
CHARACTER(LEN=maxlen) FUNCTION strHead_1(str, sep, lBackward) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),           INTENT(IN) :: str
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  LOGICAL,          OPTIONAL, INTENT(IN) :: lBackward
!------------------------------------------------------------------------------------------------------------------------------
  IF(PRESENT(sep)) THEN
    IF(     PRESENT(lBackWard)) out = str(1:INDEX(str,sep,lBackWard)-1)
    IF(.NOT.PRESENT(lBackWard)) out = str(1:INDEX(str,sep)-1)
  ELSE
    IF(     PRESENT(lBackWard)) out = str(1:INDEX(str,'/',lBackWard)-1)
    IF(.NOT.PRESENT(lBackWard)) out = str(1:INDEX(str,'/')-1)
  END IF
  IF(out == '') out = str
END FUNCTION strHead_1
!==============================================================================================================================
FUNCTION strHead_m(str, sep, lBackward) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=maxlen),     ALLOCATABLE :: out(:)
  CHARACTER(LEN=*),           INTENT(IN) :: str(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  LOGICAL,          OPTIONAL, INTENT(IN) :: lBackward
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  IF(PRESENT(sep)) THEN
    IF(     PRESENT(lBackWard)) out = [(strHead_1(str(k), sep, lBackWard), k=1, SIZE(str))]
    IF(.NOT.PRESENT(lBackWard)) out = [(strHead_1(str(k), sep),            k=1, SIZE(str))]
  ELSE
    IF(     PRESENT(lBackWard)) out = [(strHead_1(str(k), '/', lBackWard), k=1, SIZE(str))]
    IF(.NOT.PRESENT(lBackWard)) out = [(strHead_1(str(k), '/'),            k=1, SIZE(str))]
  END IF
END FUNCTION strHead_m
!==============================================================================================================================
!=== Extract the substring following the first (last if lBackward==TRUE) occurrence of "sep" in "str"          ================
!=== Examples for str='a_b_c' and sep='_' and the bash command with the same effect:                           ================
!===    * strTail(str, '_', .FALSE.) = 'b_c'         ${str#*$sep}                                              ================
!===    * strTail(str, '_', .TRUE.)  = 'c'           ${str##*$sep}                                             ================
!==============================================================================================================================
CHARACTER(LEN=maxlen) FUNCTION strTail_1(str, sep, lBackWard) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),           INTENT(IN) :: str
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  LOGICAL,          OPTIONAL, INTENT(IN) :: lBackWard
!------------------------------------------------------------------------------------------------------------------------------
  IF(PRESENT(sep)) THEN
    IF(     PRESENT(lBackWard)) out = str(INDEX(str,sep,lBackWard)+LEN(sep):LEN_TRIM(str))
    IF(.NOT.PRESENT(lBackWard)) out = str(INDEX(str,sep)          +LEN(sep):LEN_TRIM(str))
  ELSE
    IF(     PRESENT(lBackWard)) out = str(INDEX(str,'/',lBackWard)+1:LEN_TRIM(str))
    IF(.NOT.PRESENT(lBackWard)) out = str(INDEX(str,'/')          +1:LEN_TRIM(str))
  END IF
  IF(out == '') out = str
END FUNCTION strTail_1
!==============================================================================================================================
FUNCTION strTail_m(str, sep, lBackWard) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=maxlen),     ALLOCATABLE :: out(:)
  CHARACTER(LEN=*),           INTENT(IN) :: str(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  LOGICAL,          OPTIONAL, INTENT(IN) :: lBackWard
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  IF(PRESENT(sep)) THEN
    IF(     PRESENT(lBackWard)) out = [(strTail_1(str(k), sep, lBackWard), k=1, SIZE(str))]
    IF(.NOT.PRESENT(lBackWard)) out = [(strTail_1(str(k), sep),            k=1, SIZE(str))]
  ELSE
    IF(     PRESENT(lBackWard)) out = [(strTail_1(str(k), '/', lBackWard), k=1, SIZE(str))]
    IF(.NOT.PRESENT(lBackWard)) out = [(strTail_1(str(k), '/'),            k=1, SIZE(str))]
  END IF
END FUNCTION strTail_m
!==============================================================================================================================


!==============================================================================================================================
!=== Concatenates the strings "str(:)" with separator "sep" into a single string using a separator (',' by default). ==========
!==============================================================================================================================
FUNCTION strStack(str, sep, mask) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=:),          ALLOCATABLE :: out
  CHARACTER(LEN=*),           INTENT(IN) :: str(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  LOGICAL,          OPTIONAL, INTENT(IN) :: mask(:)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=:), ALLOCATABLE :: s
  INTEGER :: is, i0
  IF(SIZE(str) == 0) THEN; out = ''; RETURN; END IF
  ALLOCATE(s, SOURCE=', '); IF(PRESENT(sep)) s=sep
  IF(PRESENT(mask)) THEN
    IF(ALL(.NOT.mask)) THEN; out = ''; RETURN; END IF
    i0 = 0; DO WHILE(.NOT.mask(i0+1)); i0 = i0+1; END DO
    out = str(i0); DO is=i0+1,SIZE(str, DIM=1); IF(.NOT.mask(is)) CYCLE; out = TRIM(out)//s//TRIM(str(is)); END DO
  ELSE
    out = str(1); DO is=2,SIZE(str, DIM=1); out = TRIM(out)//s//TRIM(str(is)); END DO
  END IF
END FUNCTION strStack
!==============================================================================================================================
!=== Concatenate the strings "str(:)" with separator "sep" into one or several lines of "nmax" characters max (for display) ===
!==============================================================================================================================
FUNCTION strStackm(str, sep, nmax) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=maxlen),     ALLOCATABLE :: out(:)
  CHARACTER(LEN=*),           INTENT(IN) :: str(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: sep
  INTEGER,          OPTIONAL, INTENT(IN) :: nmax
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen), ALLOCATABLE :: t(:)
  CHARACTER(LEN=maxlen) :: sp
  INTEGER :: is, ns, no, mx, n
  IF(SIZE(str) == 0) THEN; out = ['']; RETURN; END IF
  sp =', '; IF(PRESENT(sep )) sp = sep
  ns = 2  ; IF(PRESENT(sep )) ns = LEN(sep)
  mx = 256; IF(PRESENT(nmax)) mx = nmax
  no = 1; out = ['']
  DO is = 1, SIZE(str)
    n = LEN_TRIM(str(is)); IF(out(no)/='') n = n+ns+LEN_TRIM(out(no))          !--- Line length after "str(is)" inclusion
    IF(out(no) == '') THEN
      out(no) = str(is)                                                        !--- Empty new line: set to "str(is)"
    ELSE IF(n <= mx) THEN
      out(no) = TRIM(out(no))//sp(1:ns)//TRIM(str(is))                         !--- Append "str(is)" to the current line
    ELSE
      ALLOCATE(t(no+1)); t(1:no) = out; no=no+1; t(no) = str(is)               !--- Full line: "str(si)" put in next line
      CALL MOVE_ALLOC(FROM=t, TO=out)
    END IF
  END DO
END FUNCTION strStackm
!==============================================================================================================================


!==============================================================================================================================
!=== String cleaning: replace tabulation by spaces, remove NULL characters and comments. ======================================
!==============================================================================================================================
SUBROUTINE strClean_1(str)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(INOUT) :: str
  INTEGER :: k, n, m
  n = LEN(str)
  DO k = n, 1, -1
    m = IACHAR(str(k:k))
    IF(m==9) str(k:k) = ' '                           !--- Replace the tabulations with spaces
    IF(m==0) str(k:n) = str(k+1:n)//' '               !--- Remove the NULL characters
  END DO
  m = INDEX(str,'!')-1; IF(m==-1) m = LEN_TRIM(str)   !--- Remove end of line comment
  str = ADJUSTL(str(1:m))
END SUBROUTINE strClean_1
!==============================================================================================================================
SUBROUTINE strClean_m(str)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(INOUT) :: str(:)
  INTEGER :: k
  DO k = 1, SIZE(str); CALL strClean_1(str(k)); END DO
END SUBROUTINE strClean_m
!==============================================================================================================================


!==============================================================================================================================
!=== strReduce_1(str1)     : Remove duplicated elements of str1.    ===========================================================
!=== strReduce_2(str1,str2): Append str1 with new elements of str2. ===========================================================
!==============================================================================================================================
SUBROUTINE strReduce_1(str, nb)
  IMPLICIT NONE
  CHARACTER(LEN=*), ALLOCATABLE, INTENT(INOUT) :: str(:)
  INTEGER,          OPTIONAL,    INTENT(OUT)   :: nb
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen), ALLOCATABLE :: s1(:)
  INTEGER :: k, n, n1
  IF(PRESENT(nb)) nb = 0
  CALL MOVE_ALLOC(FROM=str, TO=s1); CALL strClean(s1)
  n1 = SIZE(s1, DIM=1)                                     !--- Total nb. of elements in "s1"
  n  = COUNT( [( ALL(s1(1:k-1)/=s1(k)), k=1, n1 )] )       !--- Nb of unique elements in "s1"
  ALLOCATE(str(n))
  IF(n==0) RETURN
  str(1) = s1(1)
  n=1; DO k=2,n1; IF(ANY(s1(1:k-1)==s1(k))) CYCLE; n=n+1; str(n)=s1(k); END DO
  IF(PRESENT(nb)) nb=n
END SUBROUTINE strReduce_1
!==============================================================================================================================
SUBROUTINE strReduce_2(str1, str2)
  IMPLICIT NONE
  CHARACTER(LEN=*),   ALLOCATABLE, INTENT(INOUT) :: str1(:)
  CHARACTER(LEN=*),                INTENT(IN)    :: str2(:)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen), ALLOCATABLE :: s1(:), s2(:)
  INTEGER :: k
  IF(SIZE(str2)==0) RETURN
  s2 = str2; CALL strClean(s2)
  IF(.NOT.ALLOCATED(s2)) RETURN
  IF(SIZE(s2) == 0) THEN; DEALLOCATE(s2); RETURN; END IF
  IF(.NOT.ALLOCATED(str1)) THEN
    str1 = s2
  ELSE IF(SIZE(str1)==0) THEN
    str1 = s2
  ELSE
    s1 = str1; CALL strClean(s1)
    str1 = [s1, PACK(s2, MASK= [( ALL(s1(:) /= s2(k)), k=1, SIZE(s2) )] ) ]
  END IF
END SUBROUTINE strReduce_2
!==============================================================================================================================


!==============================================================================================================================
!=== GET THE INDEX OF THE FIRST APPEARANCE IN THE STRING VECTOR "str(:)" OF THE STRING(s) "s[(:)]" ============================
!=== OPTIONALY: GET THE NUMBER OF FOUND ELEMENTS "n". NB: UNFOUND => INDEX=0                       ============================
!==============================================================================================================================
INTEGER FUNCTION strIdx_1(str, s) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str(:), s
  DO out = 1, SIZE(str); IF(str(out) == s) EXIT; END DO
  IF(out == 1+SIZE(str) .OR. SIZE(str)==0) out = 0
END FUNCTION strIdx_1
!==============================================================================================================================
FUNCTION strIdx_m(str, s, n) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: str(:), s(:)
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = [(strIdx_1(str(:), s(k)), k=1, SIZE(s))]
  IF(PRESENT(n)) n = COUNT(out(:)/=0)
END FUNCTION strIdx_m
!==============================================================================================================================


!==============================================================================================================================
!=== GET THE INDEX LIST OF THE ELEMENTS OF "str(:)" EQUAL TO "s" AND OPTIONALY, ITS LENGTH "n" ================================
!==============================================================================================================================
FUNCTION strFind_1(str, s, n) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: str(:), s
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = PACK( [(k, k=1, SIZE(str(:), DIM=1))], MASK = str(:) == s )
  IF(PRESENT(n)) n = SIZE(out(:), DIM=1)
END FUNCTION strFind_1
!==============================================================================================================================
FUNCTION strFind_m(str, s, n) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: str(:), s(:)
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = [(strFind_1(str, s(k)), k=1, SIZE(s))]
  IF(PRESENT(n)) n = SIZE(out(:), DIM=1)
END FUNCTION strFind_m
!==============================================================================================================================
FUNCTION intFind_1(i,j,n) RESULT(out)
  IMPLICIT NONE
  INTEGER,           INTENT(IN)  :: i(:), j
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = PACK( [(k, k=1, SIZE(i(:), DIM=1))], MASK = i(:) == j )
  IF(PRESENT(n)) n = SIZE(out(:), DIM=1)
END FUNCTION intFind_1
!==============================================================================================================================
FUNCTION intFind_m(i,j,n) RESULT(out)
  IMPLICIT NONE
  INTEGER,           INTENT(IN)  :: i(:), j(:)
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = [(intFind_1(i, j(k)), k=1, SIZE(j))]
  IF(PRESENT(n)) n = SIZE(out(:), DIM=1)
END FUNCTION intFind_m
!==============================================================================================================================
FUNCTION booFind(l,n) RESULT(out)
   IMPLICIT NONE
   LOGICAL,           INTENT(IN)  :: l(:)
  INTEGER, OPTIONAL, INTENT(OUT) :: n
  INTEGER,           ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k
  out = PACK( [(k, k=1, SIZE(l(:), DIM=1))], MASK = l(:) )
  IF(PRESENT(n)) n = SIZE(out(:), DIM=1)
END FUNCTION booFind
!==============================================================================================================================


!==============================================================================================================================
!=== DUPLICATE A VECTOR "v(:)" "n" times ======================================================================================
!==============================================================================================================================
SUBROUTINE dupl_s(v, n, vdup)
  CHARACTER(LEN=*),                   INTENT(IN)  :: v(:)
  INTEGER,                            INTENT(IN)  :: n
  CHARACTER(LEN=maxlen), ALLOCATABLE, INTENT(OUT) :: vdup(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: nv, i
  nv = SIZE(v)
  ALLOCATE(vdup(n*nv))
  DO i = 1, n; vdup(1+(i-1)*nv:i*nv) = v; END DO
END SUBROUTINE dupl_s
!==============================================================================================================================
SUBROUTINE dupl_i(v, n, vdup)
  INTEGER,              INTENT(IN)  :: v(:)
  INTEGER,              INTENT(IN)  :: n
  INTEGER, ALLOCATABLE, INTENT(OUT) :: vdup(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: nv, i
  nv = SIZE(v)
  ALLOCATE(vdup(n*nv))
  DO i = 1, n; vdup(1+(i-1)*nv:i*nv) = v; END DO
END SUBROUTINE dupl_i
!==============================================================================================================================
SUBROUTINE dupl_r(v, n, vdup)
  REAL,                 INTENT(IN)  :: v(:)
  INTEGER,              INTENT(IN)  :: n
  REAL,    ALLOCATABLE, INTENT(OUT) :: vdup(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: nv, i
  nv = SIZE(v)
  ALLOCATE(vdup(n*nv))
  DO i = 1, n; vdup(1+(i-1)*nv:i*nv) = v; END DO
END SUBROUTINE dupl_r
!==============================================================================================================================
SUBROUTINE dupl_l(v, n, vdup)
  LOGICAL,              INTENT(IN)  :: v(:)
  INTEGER,              INTENT(IN)  :: n
  LOGICAL, ALLOCATABLE, INTENT(OUT) :: vdup(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: nv, i
  nv = SIZE(v)
  ALLOCATE(vdup(n*nv))
  DO i = 1, n; vdup(1+(i-1)*nv:i*nv) = v; END DO
END SUBROUTINE dupl_l
!==============================================================================================================================


!==============================================================================================================================
!=== GET THE INDEX IN "rawList" OF THE 1ST APPEARANCE OF ONE OF THE "del(:)" SEPARATORS (0 IF NONE OF THEM ARE PRESENT)
!===  IF lSc == .TRUE.:  * SKIP HEAD SIGNS OR EXPONENTS SIGNS THAT SHOULD NOT BE CONFUSED WITH SEPARATORS
!===                     * THEN TEST WHETHER THE STRING FROM START TO THE FOUND SEPARATOR IS A CORRECTLY FORMATTED NUMBER
!==============================================================================================================================
LOGICAL FUNCTION strIdx_prv(rawList, del, ibeg, idx, idel, lSc) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: rawList                          !--- String in which delimiters have to be identified
  CHARACTER(LEN=*),  INTENT(IN)  :: del(:)                           !--- List of delimiters
  INTEGER,           INTENT(IN)  :: ibeg                             !--- Start index
  INTEGER,           INTENT(OUT) :: idx                              !--- Index of the first identified delimiter in "rawList"
  INTEGER,           INTENT(OUT) :: idel                             !--- Index of the identified delimiter (0 if idx==0)
  LOGICAL, OPTIONAL, INTENT(IN)  :: lSc                              !--- Care about nbs with front sign or in scient. notation
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: idx0                                                    !--- Used to display an identified non-numeric string
  lerr = .FALSE.
  idx = strIdx1(rawList, del, ibeg, idel)                            !--- idx/=0: del(idel) is at position "idx" in "rawList" 
  IF(.NOT.PRESENT(lSc))               RETURN                         !--- No need to check exceptions for numbers => finished
  IF(.NOT.        lSc )               RETURN                         !--- No need to check exceptions for numbers => finished

  !=== No delimiter found: the whole string must be a valid number
  IF(idx == 0) THEN                                                  !--- No element of "del" in "rawList"
    lerr = .NOT.is_numeric(rawList(ibeg:LEN_TRIM(rawList)))          !--- String must be a number
    IF(lerr) idx = LEN_TRIM(rawList); RETURN                         !--- Set idx so that rawList(ibeg:idx-1) = whole string
  END IF

  lerr = idx == 1 .AND. INDEX('+-',del(idel)) /= 0; IF(lerr) RETURN  !--- The front delimiter is different from +/-: error
  IF(    idx /= 1 .AND. is_numeric(rawList(ibeg:idx-1)))     RETURN  !--- The input string head is a valid number

  !=== The string part in front of the 1st delimiter is not a valid number: search for next delimiter index "idx"
  idx0 = idx ; idx = strIdx1(rawList, del, idx+1, idel)              !--- Keep start index because idx is recycled
  IF(idx == 0) THEN
    lerr = .NOT.is_numeric(rawList(ibeg:LEN_TRIM(rawList)))          !--- No other delimiter: whole string must be a valid numb
    IF(lerr) idx = idx0; RETURN
  END IF
  lerr = .NOT.is_numeric(rawList(ibeg:idx-1))

CONTAINS

!------------------------------------------------------------------------------------------------------------------------------
INTEGER FUNCTION strIdx1(str, del, ib, id) RESULT(i)
!--- Get the index of the first appereance of one of the delimiters "del(:)" in "str" starting from position "ib".
!--- "id" is the index in "del(:)" of the first delimiter found.
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: str, del(:)
  INTEGER,           INTENT(IN)  :: ib
  INTEGER,           INTENT(OUT) :: id
!------------------------------------------------------------------------------------------------------------------------------
  DO i = ib, LEN_TRIM(str); id = strIdx(del, str(i:i)); IF(id /= 0) EXIT; END DO
  IF(i > LEN_TRIM(str)) THEN; i = 0; id = 0; END IF
END FUNCTION strIdx1

END FUNCTION strIdx_prv
!==============================================================================================================================


!==============================================================================================================================
!=== Count the number of elements separated by "delimiter" in list "rawList". =================================================
!==============================================================================================================================
LOGICAL FUNCTION strCount_11(rawList, delimiter, nb, lSc) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: rawList
  CHARACTER(LEN=*),  INTENT(IN)  :: delimiter
  INTEGER,           INTENT(OUT) :: nb
  LOGICAL, OPTIONAL, INTENT(IN)  :: lSc
!------------------------------------------------------------------------------------------------------------------------------
  LOGICAL :: ll
  ll = .FALSE.; IF(PRESENT(lSc)) ll = lSc
  lerr = strCount_1m(rawList, [delimiter], nb, ll)
END FUNCTION strCount_11
!==============================================================================================================================
LOGICAL FUNCTION strCount_m1(rawList, delimiter, nb, lSc) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),     INTENT(IN)  :: rawList(:)
  CHARACTER(LEN=*),     INTENT(IN)  :: delimiter
  INTEGER, ALLOCATABLE, INTENT(OUT) :: nb(:)
  LOGICAL,    OPTIONAL, INTENT(IN)  :: lSc
!------------------------------------------------------------------------------------------------------------------------------
  LOGICAL :: ll
  INTEGER :: id
  ll  = .FALSE.; IF(PRESENT(lSc)) ll = lSc .AND. INDEX('+-', delimiter) /= 0
  lerr = .TRUE.
  ALLOCATE(nb(SIZE(rawList)))
  DO id = 1, SIZE(rawList)
    lerr = lerr .AND. strCount_1m(rawList(id), [delimiter], nb(id), ll)
  END DO
END FUNCTION strCount_m1
!==============================================================================================================================
LOGICAL FUNCTION strCount_1m(rawList, delimiter, nb, lSc) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(IN)  :: rawList
  CHARACTER(LEN=*),  INTENT(IN)  :: delimiter(:)
  INTEGER,           INTENT(OUT) :: nb
  LOGICAL, OPTIONAL, INTENT(IN)  :: lSc
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER              :: ib, ie, jd, nr
  LOGICAL              :: ll
  CHARACTER(LEN=1024)  :: r
  lerr = .FALSE.
  ll   = .FALSE.; IF(PRESENT(lSc)) ll = lSc
  r  = TRIM(ADJUSTL(rawList))
  nr = LEN_TRIM(r); IF(nr == 0) RETURN
  nb = 1; ib = 1
  DO
    lerr = strIdx_prv(r, delimiter, ib, ie, jd, ll)
    CALL msg('"'//TRIM(r(ib:ie-1))//'" is not numeric', ll = lerr); IF(lerr) RETURN
    IF(ie == 0 .OR. jd == 0) EXIT
    ib = ie + LEN(delimiter(jd))
    DO WHILE(r(ib:ib) == ' ' .AND. ib < nr); ib = ib + 1; END DO     !--- Skip spaces before next chain
    nb = nb + 1
  END DO
END FUNCTION strCount_1m
!==============================================================================================================================


!==============================================================================================================================
!=== Purpose: Parse "delimiter"-separated list "rawList" into the pair keys(:), vals(:).   ====================================
!===          Corresponding "vals" remains empty if the element does not contain "=" sign. ====================================
!==============================================================================================================================
LOGICAL FUNCTION strParse(rawList, delimiter, keys, n, vals) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),                             INTENT(IN)  :: rawList, delimiter
  CHARACTER(LEN=maxlen), ALLOCATABLE,           INTENT(OUT) :: keys(:)
  INTEGER,                            OPTIONAL, INTENT(OUT) :: n
  CHARACTER(LEN=maxlen), ALLOCATABLE, OPTIONAL, INTENT(OUT) :: vals(:)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=1024) :: r
  INTEGER :: nr, nk
  lerr = .FALSE.
  r  = TRIM(ADJUSTL(rawList))
  nr = LEN_TRIM(r); IF(nr == 0) THEN; keys = ['']; RETURN; END IF
  nk = countK()                                                      !--- COUNT THE ELEMENTS
  CALL parseK(keys)                                                  !--- PARSE THE KEYS
  IF(PRESENT(vals)) CALL parseV(vals)                                !--- PARSE <key>=<val> PAIRS
  IF(PRESENT(n)) n = nk                                              !--- RETURN THE NUMBER OF KEYS

CONTAINS

!------------------------------------------------------------------------------------------------------------------------------
INTEGER FUNCTION countK() RESULT(nkeys)
!--- Get the number of elements after parsing.
  IMPLICIT NONE
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: ib, ie, nl
  nkeys = 1; ib = 1; nl = LEN(delimiter)
  DO
    ie = INDEX(rawList(ib:nr), delimiter)+ib-1                       !--- Determine the next separator start index
    IF(ie == ib-1) EXIT
    ib = ie + nl
    DO WHILE(ANY([0, 9, 32] == IACHAR(r(ib:ib))) .AND. ib < nr)      !--- Skip blanks (ascii): NULL (0), TAB (9), SPACE (32)
      ib = ib + 1
    END DO     !--- Skip spaces before next chain
    nkeys = nkeys+1
  END DO
END FUNCTION countK

!------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE parseK(keys)
!--- Parse the string separated by "delimiter" from "rawList" into "keys(:)"
  IMPLICIT NONE
  CHARACTER(LEN=maxlen), ALLOCATABLE, INTENT(OUT) :: keys(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: ib, ie, ik
  ALLOCATE(keys(nk))
  ib = 1
  DO ik = 1, nk
    ie = INDEX(rawList(ib:nr), delimiter)+ib-1                       !--- Determine the next separator start index
    IF(ie == ib-1) EXIT
    keys(ik) = r(ib:ie-1)                                            !--- Get the ikth key
    ib = ie + LEN(delimiter)
    DO WHILE(r(ib:ib) == ' ' .AND. ib < nr); ib = ib + 1; END DO     !--- Skip spaces before next chain
  END DO
  keys(ik) = r(ib:nr)                                                !--- Get the last key
END SUBROUTINE parseK

!------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE parseV(vals)
!--- Parse the <key>=<val> pairs in "keys(:)" into "keys" and "vals"
  IMPLICIT NONE
  CHARACTER(LEN=maxlen), ALLOCATABLE, INTENT(OUT) :: vals(:)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen) :: key
  INTEGER :: ik, ix
  ALLOCATE(vals(nk))
  DO ik = 1, nk; key = keys(ik)
    vals(ik) = ''
    ix = INDEX(key, '='); IF(ix == 0) CYCLE                          !--- First "=" index in "key"
    vals(ik) = ADJUSTL(key(ix+1:LEN_TRIM(key)))
    keys(ik) = ADJUSTL(key(1:ix-1))
  END DO
END SUBROUTINE parseV

END FUNCTION strParse
!==============================================================================================================================
LOGICAL FUNCTION strParse_m(rawList, delimiter, keys, n, vals, lSc, id) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),                             INTENT(IN)  :: rawList, delimiter(:)
  CHARACTER(LEN=maxlen),           ALLOCATABLE, INTENT(OUT) :: keys(:)  !--- Parsed keys vector
  INTEGER,               OPTIONAL,              INTENT(OUT) :: n        !--- Length of the parsed vector
  CHARACTER(LEN=maxlen), OPTIONAL, ALLOCATABLE, INTENT(OUT) :: vals(:)  !--- Values for <name>=<value> keys
  LOGICAL,               OPTIONAL,              INTENT(IN)  :: lSc      !--- Take care about numbers in scientific notation
  INTEGER,               OPTIONAL, ALLOCATABLE, INTENT(OUT) :: id(:)    !--- Indexes of the separators in "delimiter(:)" vector
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=1024) :: r
  INTEGER :: nr, ik, nk, ib, ie, jd
  LOGICAL :: ll
  ll = .FALSE.; IF(PRESENT(lSc)) ll = lSc
  lerr = strCount_1m(rawList, delimiter, nk, ll)
  CALL msg("Couldn't parse list: non-numerical strings were found", ll=lerr); IF(lerr) RETURN

  !--- FEW ALLOCATIONS
  ALLOCATE(keys(nk))
  IF(PRESENT(vals)) ALLOCATE(vals(nk))
  IF(PRESENT(id))   ALLOCATE(id(nk-1))
  IF(PRESENT(n)) n = nk

  !--- PARSING
  r  = TRIM(ADJUSTL(rawList))
  nr = LEN_TRIM(r); IF(nr == 0) RETURN
  ib = 1
  DO ik = 1, nk-1
    lerr = strIdx_prv(r, delimiter, ib, ie, jd, ll)
    CALL msg('Non-numeric values found', ll=lerr); IF(lerr) RETURN
    keys(ik) = r(ib:ie-1)
    IF(PRESENT(vals)) CALL parseKeys(keys(ik), vals(ik))             !--- Parse a <key>=<val> pair
    IF(PRESENT(id  )) id(ik) = jd                                    !--- Index in "delimiter(:)" of the "ik"th delimiter
    ib = ie + LEN_TRIM( delimiter(jd) )                              !--- Length of the current delimiter
    DO WHILE(r(ib:ib) == ' ' .AND. ib < nr); ib = ib + 1; END DO     !--- Skip spaces before next chain
  END DO
  keys(nk) = r(ib:nr)
  IF(PRESENT(vals)) CALL parseKeys(keys(nk), vals(nk))               !--- Parse a <key>=<val> pair

CONTAINS

!------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE parseKeys(key, val)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(INOUT) :: key
  CHARACTER(LEN=*), INTENT(OUT)   :: val
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: ix
  ix = INDEX(key, '='); IF(ix == 0) RETURN                           !--- First "=" index in "key"
  val = ADJUSTL(key(ix+1:LEN_TRIM(key)))
  key = ADJUSTL(key(1:ix-1))
END SUBROUTINE parseKeys

END FUNCTION strParse_m    
!==============================================================================================================================


!==============================================================================================================================
!=== String substitution: replace "key" by "val" each time it appears in "str".
!==============================================================================================================================
SUBROUTINE strReplace_1(str, key, val, lsurr)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(INOUT) :: str        !--- Main string
  CHARACTER(LEN=*),  INTENT(IN)    :: key, val   !--- "key" will be replaced by "val"
  LOGICAL, OPTIONAL, INTENT(IN)    :: lsurr      !--- TRUE => key must be surrounded by special characters to be substituted
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: i0, ix, nk, ns
  LOGICAL :: lsur, lb, le
  lsur = .FALSE.; IF(PRESENT(lsurr)) lsur = lsurr
  nk = LEN_TRIM(key)
  i0 = 1
  DO
    ns = LEN_TRIM(str)
    ix = INDEX(str(i0:ns), TRIM(key))            !--- First appearance index of "key" in "s", starting from index "i0"
    IF(ix == 0) EXIT
    ix = ix + i0 -1
    IF(lsur) THEN                                !--- Key must be surrounded by special characters
      !--- lb=.TRUE.: key is at the very beginning of "str" or located after  a special character
      lb = ix     ==1;  IF(.NOT.lb) lb = INDEX('+-*/()^', str(ix-1 :ix-1 ))/=0
      !--- le=.TRUE.: key is at the very end       of "str" or located before a special character
      le = ix+nk-1==ns; IF(.NOT.le) le = INDEX('+-*/()^', str(ix+nk:ix+nk))/=0
      IF(.NOT.(lb.AND.le)) THEN; i0 = i0 + nk; CYCLE; END IF
    END IF
    str = str(1:ix-1)//TRIM(val)//str(ix+nk:ns)
  END DO
END SUBROUTINE strReplace_1
!==============================================================================================================================
SUBROUTINE strReplace_m(str, key, val, lsurr)
  IMPLICIT NONE
  CHARACTER(LEN=*),  INTENT(INOUT) :: str(:)     !--- Main strings vector
  CHARACTER(LEN=*),  INTENT(IN)    :: key, val   !--- "key" will be replaced by "val"
  LOGICAL, OPTIONAL, INTENT(IN)    :: lsurr      !--- TRUE => key must be surrounded by special characters to be substituted
  INTEGER :: k
  LOGICAL :: ll
  ll=.FALSE.; IF(PRESENT(lsurr)) ll=lsurr
  DO k=1, SIZE(str); CALL strReplace_1(str(k),key,val,ll); END DO
END SUBROUTINE strReplace_m
!==============================================================================================================================


!==============================================================================================================================
!=== Contatenate horizontally scalars/vectors of strings/integers/reals into a vector/array ===================================
!==============================================================================================================================
FUNCTION horzcat_s00(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),                   INTENT(IN) :: v0
  CHARACTER(LEN=*), OPTIONAL, TARGET, INTENT(IN) :: v1, v2, v3, v4, v5, v6, v7, v8, v9
  CHARACTER(LEN=maxlen), ALLOCATABLE :: out(:)
  CHARACTER(LEN=maxlen), POINTER     :: v
  INTEGER                            :: ncol, iv
  LOGICAL                            :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  ncol = 1+COUNT(pre)
  ALLOCATE(out(ncol))
  out(1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     out(iv) = v
  END DO
END FUNCTION horzcat_s00
!==============================================================================================================================
FUNCTION horzcat_s10(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),           INTENT(IN) :: v0(:), v1
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: v2, v3, v4, v5, v6, v7, v8, v9
  CHARACTER(LEN=maxlen), ALLOCATABLE :: out(:), tmp(:)
  INTEGER :: nc
  nc = SIZE(v0)
  tmp = horzcat_s00(v0(nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) THEN
!ym fix for nvidia compiler
!ym out = [v0(1:nc-1), tmp]
     out = v0(1:nc-1)
     out = [out , tmp]
  ENDIF
END FUNCTION horzcat_s10
!==============================================================================================================================
FUNCTION horzcat_s11(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),                   INTENT(IN) :: v0(:)
  CHARACTER(LEN=*), OPTIONAL, TARGET, INTENT(IN) :: v1(:), v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  CHARACTER(LEN=maxlen), ALLOCATABLE :: out(:,:)
  CHARACTER(LEN=maxlen), POINTER     :: v(:)
  INTEGER :: nrow, ncol, iv, n
  LOGICAL :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  nrow = SIZE(v0)
  ncol = 1+COUNT(pre)
  ALLOCATE(out(nrow, ncol))
  out(:,1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     n = SIZE(v, DIM=1)
     IF(n /= nrow) THEN; CALL msg("Can't concatenate vectors of differing lengths"); STOP; END IF
     out(:,iv) = v(:)
  END DO
END FUNCTION horzcat_s11
!==============================================================================================================================
FUNCTION horzcat_s21(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*),           INTENT(IN) :: v0(:,:), v1(:)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  CHARACTER(LEN=maxlen), ALLOCATABLE :: out(:,:), tmp(:,:), pk(:)
  INTEGER :: nc
  nc  = SIZE(v0, 2)
  tmp = horzcat_s11(v0(:,nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
!ym fix for nvidia compiler
!ym  IF(nc /= 1) out = RESHAPE([PACK(v0(:,1:nc-1), .TRUE.), PACK(tmp, .TRUE.)], SHAPE=[SIZE(s0, 1), nc + SIZE(tmp, 2)-1])
  IF(nc /= 1) THEN
     pk = PACK(v0(:,1:nc-1), .TRUE.)
     pk = [ pk, PACK(tmp, .TRUE.)]
     out = RESHAPE(pk, SHAPE=[SIZE(v0, 1), nc + SIZE(tmp, 2)-1])
  ENDIF
END FUNCTION horzcat_s21
!==============================================================================================================================
FUNCTION horzcat_i00(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  INTEGER,                   INTENT(IN) :: v0
  INTEGER, OPTIONAL, TARGET, INTENT(IN) :: v1, v2, v3, v4, v5, v6, v7, v8, v9
  INTEGER, ALLOCATABLE :: out(:)
  INTEGER, POINTER     :: v
  INTEGER              :: ncol, iv
  LOGICAL              :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  ncol = SIZE(pre)
  ALLOCATE(out(ncol))
  out(1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     out(iv) = v
  END DO
END FUNCTION horzcat_i00
!==============================================================================================================================
FUNCTION horzcat_i10(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  INTEGER,           INTENT(IN) :: v0(:), v1
  INTEGER, OPTIONAL, INTENT(IN) :: v2, v3, v4, v5, v6, v7, v8, v9
  INTEGER, ALLOCATABLE :: out(:), tmp(:)
  INTEGER :: nc
  nc = SIZE(v0)
  tmp = horzcat_i00(v0(nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = [v0(1:nc-1), tmp]
END FUNCTION horzcat_i10
!==============================================================================================================================
FUNCTION horzcat_i11(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  INTEGER,                   INTENT(IN) :: v0(:)
  INTEGER, OPTIONAL, TARGET, INTENT(IN) :: v1(:), v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  INTEGER, ALLOCATABLE :: out(:,:)
  INTEGER, POINTER     :: v(:)
  INTEGER              :: nrow, ncol, iv, n
  LOGICAL              :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  nrow = SIZE(v0)
  ncol = 1+COUNT(pre)
  ALLOCATE(out(nrow, ncol))
  out(:,1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     n = SIZE(v, DIM=1)
     IF(n /= nrow) THEN; CALL msg("Can't concatenate vectors of differing lengths"); STOP; END IF
     out(:,iv) = v(:)
  END DO
END FUNCTION horzcat_i11
!==============================================================================================================================
FUNCTION horzcat_i21(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  INTEGER,           INTENT(IN) :: v0(:,:), v1(:)
  INTEGER, OPTIONAL, INTENT(IN) :: v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  INTEGER, ALLOCATABLE :: out(:,:), tmp(:,:)
  INTEGER :: nc
  nc  = SIZE(v0, 2)
  tmp = horzcat_i11(v0(:,nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = RESHAPE([PACK(v0(:,1:nc-1), .TRUE.), PACK(tmp, .TRUE.)], SHAPE=[SIZE(v0, 1), nc + SIZE(tmp, 2)-1])
END FUNCTION horzcat_i21
!==============================================================================================================================
FUNCTION horzcat_r00(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL32),                   INTENT(IN) :: v0
  REAL(KIND=REAL32), OPTIONAL, TARGET, INTENT(IN) :: v1, v2, v3, v4, v5, v6, v7, v8, v9
  REAL(KIND=REAL32), ALLOCATABLE :: out(:)
  REAL(KIND=REAL32), POINTER :: v
  INTEGER           :: ncol, iv
  LOGICAL           :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  ncol = 1+COUNT(pre)
  ALLOCATE(out(ncol))
  out(1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     out(iv) = v
  END DO
END FUNCTION horzcat_r00
!==============================================================================================================================
FUNCTION horzcat_r10(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL32),           INTENT(IN) :: v0(:), v1
  REAL(KIND=REAL32), OPTIONAL, INTENT(IN) :: v2, v3, v4, v5, v6, v7, v8, v9
  REAL(KIND=REAL32), ALLOCATABLE :: out(:), tmp(:)
  INTEGER :: nc
  nc  = SIZE(v0)
  tmp = horzcat_r00(v0(nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = [v0(1:nc-1), tmp]
END FUNCTION horzcat_r10
!==============================================================================================================================
FUNCTION horzcat_r11(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL32),                   INTENT(IN) :: v0(:)
  REAL(KIND=REAL32), OPTIONAL, TARGET, INTENT(IN) :: v1(:), v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  REAL(KIND=REAL32), ALLOCATABLE :: out(:,:)
  REAL(KIND=REAL32), POINTER     :: v(:)
  INTEGER :: nrow, ncol, iv, n
  LOGICAL :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  nrow = SIZE(v0)
  ncol = 1+COUNT(pre)
  ALLOCATE(out(nrow, ncol))
  out(:,1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     n = SIZE(v, DIM=1)
     IF(n /= nrow) THEN; CALL msg("Can't concatenate vectors of differing lengths"); STOP; END IF
     out(:,iv) = v(:)
  END DO
END FUNCTION horzcat_r11
!==============================================================================================================================
FUNCTION horzcat_r21(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL32),           INTENT(IN) :: v0(:,:), v1(:)
  REAL(KIND=REAL32), OPTIONAL, INTENT(IN) :: v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  REAL(KIND=REAL32), ALLOCATABLE :: out(:,:), tmp(:,:)
  INTEGER :: nc
  nc  = SIZE(v0, 2)
  tmp = horzcat_r11(v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = RESHAPE([PACK(v0(:,1:nc-1), .TRUE.), PACK(tmp, .TRUE.)], SHAPE=[SIZE(v0, 1), nc + SIZE(tmp, 2)-1])
END FUNCTION horzcat_r21
!==============================================================================================================================
FUNCTION horzcat_d00(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL64),                   INTENT(IN) :: v0
  REAL(KIND=REAL64), OPTIONAL, TARGET, INTENT(IN) :: v1, v2, v3, v4, v5, v6, v7, v8, v9
  REAL(KIND=REAL64), ALLOCATABLE :: out(:)
  REAL(KIND=REAL64), POINTER     :: v
  INTEGER                        :: ncol, iv
  LOGICAL                        :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  ncol = 1+COUNT(pre)
  ALLOCATE(out(ncol))
  out(1) = v0
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
        CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
        CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     out(iv) = v
  END DO
END FUNCTION horzcat_d00
!==============================================================================================================================
FUNCTION horzcat_d10(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL64),           INTENT(IN) :: v0(:), v1
  REAL(KIND=REAL64), OPTIONAL, INTENT(IN) :: v2, v3, v4, v5, v6, v7, v8, v9
  REAL(KIND=REAL64), ALLOCATABLE :: out(:), tmp(:)
  INTEGER :: nc
  nc = SIZE(v0)
  tmp = horzcat_d00(v0(nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = [v0(1:nc-1), tmp]
END FUNCTION horzcat_d10
!==============================================================================================================================
FUNCTION horzcat_d11(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL64),                   INTENT(IN) :: v0(:)
  REAL(KIND=REAL64), OPTIONAL, TARGET, INTENT(IN) :: v1(:), v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  REAL(KIND=REAL64), ALLOCATABLE :: out(:,:)
  REAL(KIND=REAL64), POINTER     :: v(:)
  INTEGER :: nrow, ncol, iv, n
  LOGICAL :: pre(9)
!------------------------------------------------------------------------------------------------------------------------------
  pre(:) = [PRESENT(v1),PRESENT(v2),PRESENT(v3),PRESENT(v4),PRESENT(v5),PRESENT(v6),PRESENT(v7),PRESENT(v8),PRESENT(v9)]
  nrow = SIZE(v0)
  ncol = 1+COUNT(pre)
  ALLOCATE(out(nrow, ncol))
  DO iv = 2, ncol; IF(.NOT.pre(iv-1)) CYCLE
     SELECT CASE(iv-1)
       CASE(1); v=> v1; CASE(2); v=> v2; CASE(3); v=> v3; CASE(4); v=> v4; CASE(5); v=> v5
       CASE(6); v=> v6; CASE(7); v=> v7; CASE(8); v=> v8; CASE(9); v=> v9
     END SELECT
     n = SIZE(v, DIM=1)
     IF(n /= nrow) THEN; CALL msg("Can't concatenate vectors of differing lengths"); STOP; END IF
     out(:,iv) = v(:)
  END DO
END FUNCTION horzcat_d11
!==============================================================================================================================
FUNCTION horzcat_d21(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL64),           INTENT(IN) :: v0(:,:), v1(:)
  REAL(KIND=REAL64), OPTIONAL, INTENT(IN) :: v2(:), v3(:), v4(:), v5(:), v6(:), v7(:), v8(:), v9(:)
  REAL(KIND=REAL64), ALLOCATABLE :: out(:,:), tmp(:,:)
  INTEGER :: nc
  nc  = SIZE(v0, 2)
  tmp = horzcat_d11(v0(:,nc), v1, v2, v3, v4, v5, v6, v7, v8, v9)
  IF(nc == 1) out = tmp
  IF(nc /= 1) out = RESHAPE([PACK(v0(:,1:nc-1), .TRUE.), PACK(tmp, .TRUE.)], SHAPE=[SIZE(v0, 1), nc + SIZE(tmp, 2)-1])
END FUNCTION horzcat_d21
!==============================================================================================================================


!==============================================================================================================================
!--- Display a clean table composed of successive vectors of same length.
!=== The profile "p" describe in which order to pick up the columns from "s", "i" and "r" for display.
!===  * nRowMax lines are displayed (default: all lines)
!===  * nColMax characters (default: as long as needed) are displayed at most on a line. If the effective total length is
!===    higher, several partial tables are displayed ; the nHead (default: 1) first columns are included in each sub-table.
!==============================================================================================================================
 FUNCTION dispTable(p, titles, s, i, r, rFmt, nRowMax, nColMax, nHead, unit, sub) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),           INTENT(IN)  :: p             !--- DISPLAY MAP: s/i/r
  CHARACTER(LEN=*),           INTENT(IN)  :: titles(:)     !--- TITLES (ONE EACH COLUMN)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: s(:,:)        !--- STRINGS
  INTEGER,          OPTIONAL, INTENT(IN)  :: i(:,:)        !--- INTEGERS
  REAL,             OPTIONAL, INTENT(IN)  :: r(:,:)        !--- REALS
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: rFmt          !--- Format for reals
  INTEGER,          OPTIONAL, INTENT(IN)  :: nRowMax       !--- Display at most "nRowMax" rows
  INTEGER,          OPTIONAL, INTENT(IN)  :: nColMax       !--- Display at most "nColMax" characters each line
  INTEGER,          OPTIONAL, INTENT(IN)  :: nHead         !--- Head columns repeated for multiple tables display
  INTEGER,          OPTIONAL, INTENT(IN)  :: unit          !--- Output unit (default: screen)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: sub           !--- Subroutine name
  LOGICAL :: lerr
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=2048) :: row
  CHARACTER(LEN=maxlen)  :: rFm, el, subn
  CHARACTER(LEN=maxlen), ALLOCATABLE :: d(:,:)
  CHARACTER(LEN=1) :: s1, sp
  INTEGER :: is, ii, ir, it, k, nmx,  unt, ic, np
  INTEGER :: ns, ni, nr, nt, l, ncol, nHd, ib, l0
  INTEGER, ALLOCATABLE :: n(:), ncmx(:)
  INTEGER, PARAMETER   :: nm=1                             !--- Space between values & columns
  LOGICAL :: ls, li, lr
  subn = '';    IF(PRESENT(sub)) subn = sub
  rFm = '*';    IF(PRESENT(rFmt)) rFm = rFmt               !--- Specified format for reals
  unt = lunout; IF(PRESENT(unit)) unt = unit               !--- Specified output unit
  np = LEN_TRIM(p); ns = 0; ni = 0; nr = 0; ncol = 0
  ls = PRESENT(s); li = PRESENT(i); lr = PRESENT(r)
  lerr = .FALSE.; IF(.NOT.ANY([ls,li,lr])) RETURN          !--- Nothing to do
  sp = '|'                                                 !--- Separator

  !--- CHECK ARGUMENTS COHERENCE
  lerr = np /= SIZE(titles); CALL msg('display map "p" length and titles list mismatch', subn, lerr); IF(lerr) RETURN
  IF(ls) THEN
    ns = SIZE(s, 1); ncol = ncol + SIZE(s, 2); lerr = COUNT([(p(ic:ic)=='s', ic=1, np)]) /= SIZE(s, 2)
  END IF
  IF(li) THEN
    ni = SIZE(i, 1); ncol = ncol + SIZE(i, 2); lerr = COUNT([(p(ic:ic)=='i', ic=1, np)]) /= SIZE(i, 2)
  END IF
  IF(lr) THEN
    nr = SIZE(r, 1); ncol = ncol + SIZE(r, 2); lerr = COUNT([(p(ic:ic)=='r', ic=1, np)]) /= SIZE(r, 2)
  END IF
  CALL msg('display map "p" length and arguments number mismatch', subn, lerr); IF(lerr) RETURN
  lerr = ncol /= SIZE(titles); CALL msg('"titles" length and arguments number mismatch', subn, lerr); IF(lerr) RETURN
  lerr = ls.AND.li.AND.ns/=ni; CALL msg('string and integer arguments lengths mismatch', subn, lerr); IF(lerr) RETURN
  lerr = ls.AND.lr.AND.ns/=nr; CALL msg(   'string and real arguments lengths mismatch', subn, lerr); IF(lerr) RETURN
  lerr = li.AND.lr.AND.ni/=nr; CALL msg(  'integer and real arguments lengths mismatch', subn, lerr); IF(lerr) RETURN
  nmx = MAX(ns,ni,nr)+1; IF(PRESENT(nRowMax)) nmx = MIN(nmx,nRowMax+1)

  !--- Allocate the assembled quantities array
  ALLOCATE(d(nmx,ncol), n(ncol))

  !--- Assemble the vectors into a strings array in the order indicated by "pattern"
  is =  1; ii = 1; ir = 1
  DO ic = 1, ncol
    d(1,ic) = TRIM(titles(ic))
    SELECT CASE(p(ic:ic))
      CASE('s'); d(2:nmx,ic) =         s(:,is)     ; is = is + 1
      CASE('i'); d(2:nmx,ic) = num2str(i(:,ii)    ); ii = ii + 1
      CASE('r'); d(2:nmx,ic) = num2str(r(:,ir),rFm); ir = ir + 1
    END SELECT
  END DO
  CALL cleanZeros(d)
  DO ic = 1, ncol
    n(ic)=0; DO ir=1, nmx; n(ic)=MAX(n(ic), LEN_TRIM(d(ir,ic))); END DO
  END DO
  n(:) = n(:) + 2*nm

  !--- Build the vector of max column index in case the rows are too long (table must be displayed in multiple parts)
  nHd = 1; IF(PRESENT(nHead)) nHd = nHead
  IF(.NOT.PRESENT(nColMax)) THEN
    nt = 1; ncmx = [ncol]
  ELSE
    nt = 1; l0 = SUM(n(1:nHd)+1)+1
    IF(PRESENT(sub)) l0=l0+LEN_TRIM(subn)+1
    !--- Count the number of table parts
    l = l0; DO ic = nHd+1, ncol; l = l+n(ic)+1; IF(l>=nColMax) THEN; nt = nt+1; l = l0+n(ic)+1; END IF; END DO
    !--- Get the index of the last column for each table part
    ALLOCATE(ncmx(nt)); k = 1
    l = l0; DO ic = nHd+1, ncol; l = l+n(ic)+1; IF(l>=nColMax) THEN; ncmx(k) = ic-1; l = l0+n(ic)+1; k = k+1; END IF; END DO
    ncmx(nt) = ncol
  END IF
      
  !--- Display the strings array as a table
  DO it = 1, nt
    DO ir = 1, nmx; row = ''
      DO ic = 1, nHd; el = d(ir,ic)
        s1 = sp
        row = TRIM(row)//REPEAT(' ',nm)//TRIM(el)//REPEAT(' ',n(ic)-LEN_TRIM(el)-nm)//s1
      END DO
      ib = nHd+1; IF(it>1) ib = ncmx(it-1)+1
      DO ic = ib, ncmx(it); el = d(ir,ic)
        s1 = sp
        row = TRIM(row)//REPEAT(' ',nm)//TRIM(el)//REPEAT(' ',n(ic)-LEN_TRIM(el)-nm)//s1
      END DO
      nr = LEN_TRIM(row)-1                                           !--- Final separator removed
      CALL msg(row(1:nr), subn, unit=unt)
      IF(ir /= 1) CYCLE                                              !--- Titles only are underlined
      row=''; DO ic=1,nHd; row=TRIM(row)//REPEAT('-',n(ic))//'+'; END DO
      DO ic = ib,ncmx(it); row=TRIM(row)//REPEAT('-',n(ic))//'+'; END DO
      CALL msg(row(1:LEN_TRIM(row)-1), subn, unit=unt)
    END DO
    CALL msg('', subn, unit=unt)
  END DO

END FUNCTION dispTable
!==============================================================================================================================

!==============================================================================================================================
LOGICAL FUNCTION dispNamelist(unt, p, titles, s, i, r, rFmt, llast) RESULT(lerr)
  IMPLICIT NONE
  INTEGER,                    INTENT(IN)  :: unt           !--- Output unit
  CHARACTER(LEN=*),           INTENT(IN)  :: p             !--- DISPLAY MAP: s/i/r
  CHARACTER(LEN=*),           INTENT(IN)  :: titles(:)     !--- TITLES (ONE EACH COLUMN)
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: s(:,:)        !--- STRINGS
  INTEGER,          OPTIONAL, INTENT(IN)  :: i(:,:)        !--- INTEGERS
  REAL,             OPTIONAL, INTENT(IN)  :: r(:,:)        !--- REALS
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: rFmt          !--- Format for reals
  LOGICAL,          OPTIONAL, INTENT(IN)  :: llast         !--- Last variable: no final ','
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen)  :: rFm, el
  CHARACTER(LEN=maxlen), ALLOCATABLE :: d(:,:)
  CHARACTER(LEN=:),      ALLOCATABLE :: sp, row
  INTEGER :: is, ii, ir, nrow, ic
  INTEGER :: ns, ni, nr, ncol, np
  INTEGER, ALLOCATABLE :: n(:)
  LOGICAL :: ls, li, lr, la
  rFm = '*';    IF(PRESENT(rFmt)) rFm = rFmt               !--- Specified format for reals
  ls = PRESENT(s); li = PRESENT(i); lr = PRESENT(r)
  lerr = .FALSE.; IF(.NOT.ANY([ls,li,lr])) RETURN          !--- Nothing to do
  la = .FALSE.; IF(PRESENT(llast)) la = llast

  !--- CHECK ARGUMENTS COHERENCE
  ns = 0; ni = 0; nr = 0; np = LEN_TRIM(p); ncol = 0
  IF(ls) THEN; ns = SIZE(s, DIM=1); ncol = ncol + SIZE(s, DIM=2)
    lerr = COUNT([(p(ic:ic)=='s', ic=1, np)]) /= SIZE(s, DIM=2)
  END IF
  IF(li) THEN; ni = SIZE(i, DIM=1); ncol = ncol + SIZE(i, DIM=2)
    lerr = COUNT([(p(ic:ic)=='i', ic=1, np)]) /= SIZE(i, DIM=2)
  END IF
  IF(lr) THEN; nr = SIZE(r, DIM=1); ncol = ncol + SIZE(r, DIM=2)
    lerr = COUNT([(p(ic:ic)=='r', ic=1, np)]) /= SIZE(r, DIM=2)
  END IF
  CALL msg('display map "p" length and arguments number mismatch', ll=lerr); IF(lerr) RETURN
  lerr = ncol /= SIZE(titles); CALL msg('"titles" length and arguments number mismatch', ll=lerr); IF(lerr) RETURN
  lerr = ls.AND.li.AND.ns/=ni; CALL msg('string and integer arguments lengths mismatch', ll=lerr); IF(lerr) RETURN
  lerr = ls.AND.lr.AND.ns/=nr; CALL msg(   'string and real arguments lengths mismatch', ll=lerr); IF(lerr) RETURN
  lerr = li.AND.lr.AND.ni/=nr; CALL msg(  'integer and real arguments lengths mismatch', ll=lerr); IF(lerr) RETURN

  !--- Allocate the assembled quantities array
  nrow = MAX(ns,ni,nr)+1
  ALLOCATE(d(nrow,ncol), n(ncol))

  !--- Assemble the vectors into a strings array in the order indicated by "pattern"
  is =  1; ii = 1; ir = 1
  DO ic = 1, ncol
    d(1,ic) = TRIM(titles(ic))
    SELECT CASE(p(ic:ic))
      CASE('s'); d(2:nrow,ic) =         s(:,is)     ; is = is + 1
      CASE('i'); d(2:nrow,ic) = num2str(i(:,ii)    ); ii = ii + 1
      CASE('r'); d(2:nrow,ic) = num2str(r(:,ir),rFm); ir = ir + 1
    END SELECT
  END DO
  CALL cleanZeros(d)
  DO ic = 1, ncol
    n(ic) = 0; DO ir=1, nrow; n(ic)=MAX(n(ic), LEN_TRIM(d(ir,ic))); END DO
    IF(needQuotes(d(2,ic)) .AND. ic/=1) n(ic) = n(ic) + 2 !--- For quotes, using second line only
  END DO

  !--- Display the strings array as a table
  DO ir = 1, nrow
    row = ''; sp = '   '; IF(TRIM(d(ir,1)) /= '') sp = ' = '
    DO ic = 1, ncol
      el = d(ir,ic); IF(ic /= 1) el = addQuotes_1(el)
      row = row//TRIM(el)//REPEAT(' ',n(ic)-LEN_TRIM(el))//sp
      sp = '   '; IF(ic /= ncol-1) CYCLE
      IF(TRIM(d(MIN(ir+1,nrow),1)) /= '' .AND. (ir /= nrow .OR. .NOT.la)) sp = ' , '
    END DO
    WRITE(unt,'(a)')TRIM(row)
  END DO

  !--- End of section
  IF(la) THEN
    WRITE(unt,'(a)')'/'
    WRITE(unt,'(a)')
  END IF

END FUNCTION dispNameList
!==============================================================================================================================


!==============================================================================================================================
LOGICAL FUNCTION dispOutliers_1(ll, a, n, err_msg, nam, subn, nRowmax, nColMax, nHead, unit) RESULT(lerr)
  IMPLICIT NONE
! Display outliers list in tables
! If "nam" is supplied, it means the last index is for tracers => one table each tracer for rank > 2.
  LOGICAL,                    INTENT(IN)  :: ll(:)                   !--- Linearized mask of outliers
  REAL,                       INTENT(IN)  ::  a(:)                   !--- Linearized array of values
  INTEGER,                    INTENT(IN)  ::  n(:)                   !--- Profile before linearization
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: err_msg, nam(:), subn   !--- Error message, variables and calling subroutine names
  INTEGER,          OPTIONAL, INTENT(IN)  :: nRowMax                 !--- Maximum number of lines to display    (default: all)
  INTEGER,          OPTIONAL, INTENT(IN)  :: nColMax                 !--- Maximum number of characters per line (default: 2048)
  INTEGER,          OPTIONAL, INTENT(IN)  :: nHead                   !--- Number of front columns to duplicate  (default: 1)
  INTEGER,          OPTIONAL, INTENT(IN)  :: unit                    !--- Output unit                           (def: lunout)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen),      ALLOCATABLE :: ttl(:)
  LOGICAL,                    ALLOCATABLE :: m(:)
  INTEGER,                    ALLOCATABLE :: ki(:), kj(:)
  INTEGER                                 :: i, j, k, rk, rk1, ib, ie, itr, nm, unt, nRmx, nCmx, nHd, nv
  CHARACTER(LEN=maxlen)                   :: mes, sub, fm='(f12.9)', v, s
  CHARACTER(LEN=maxlen),      ALLOCATABLE :: vnm(:)

  lerr = ANY(ll); IF(.NOT.lerr) RETURN                               !--- No outliers -> finished

  mes = 'outliers found'; IF(PRESENT(err_msg)) mes = err_msg         !--- Error message
  vnm = ['a'];            IF(PRESENT(nam ))    vnm = nam             !--- Variables names
  sub = 'dispOutliers';   IF(PRESENT(subn))    sub = subn            !--- Calling subroutine name
  nRmx= SIZE(a);          IF(PRESENT(nRowMax)) nRmx=MIN(nRmx,nRowMax)!-- Maximum number of lines to print
  nCmx= 2048;             IF(PRESENT(nColMax)) nCmx=MIN(nCmx,nColMax)!-- Maximum number of characters each line
  nHd = 1;                IF(PRESENT(nHead))   nHd = nHead           !--- Number of front columns to duplicate
  unt = lunout;           IF(PRESENT(unit))    unt = unit            !--- Unit to print messages

  rk = SIZE(n); nv = SIZE(vnm)
  lerr = nv/=1 .AND. nv/=n(rk); CALL msg('SIZE(nam) /= 1 and /= last "n" element', sub, lerr); IF(lerr) RETURN
  lerr = SIZE(a) /=   SIZE(ll); CALL msg('ll" and "a" sizes mismatch',             sub, lerr); IF(lerr) RETURN
  lerr = SIZE(a) /= PRODUCT(n); CALL msg('profile "n" does not match "a" and "ll', sub, lerr); IF(lerr) RETURN
  CALL msg(mes, sub, unit=unt)

  !--- SCALAR CASE: single value to display
  IF(rk==1.AND.n(1)==1) THEN
    IF(ll(1)) WRITE(unt,'(a," = ",f12.9)')TRIM(nam(1)),a(1); RETURN
  END IF

  rk1 = rk; IF(nv==1) rk1 = rk-1                                    !--- Rank of each displayed table
  SELECT CASE(rk1)                                                  !--- Indices list
    CASE(1,2); ki = [ (i,i=1,n(1)) ]
    CASE(3);   ki = [((i,i=1,n(1)),j=1,n(2))]; kj = [((j,i=1,n(1)),j=1,n(2))]
    CASE DEFAULT; WRITE(unt,*)'Sorry: routine "dispOutliers" is limited to rank 3'; RETURN
  END SELECT

  !--- VECTOR CASE:  table " name | value " (known names)  /  )  /  " i | a(i) " (unknown names)
  IF(rk==1) THEN
    ALLOCATE(ttl(2)); ttl(2) = TRIM(vnm(1))//'(i)'; ttl(1) = 'i'
    IF(nv == 1) lerr = dispTable('sr', ttl,               s=cat(PACK(nam,ll)), r=cat(PACK(a,ll)), &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
    IF(nv /= 1) lerr = dispTable('ir', ['name ','value'], i=cat(PACK(ki,m)),   r=cat(PACK(a,ll)), &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
    CALL msg("can't display outliers table", sub, lerr, unt)
    RETURN
  END IF

  !--- OTHER CASES: one table for each tracer (last index)
  ttl = [(ACHAR(k), k = 105, 104+rk)]                                !--- Titles list ('i', 'j', 'k'...)
  s = strStack( ttl(1:rk-1) )                                        !--- Tracer name dummy indexes: (i, j, k, ...

  DO itr=1,n(rk)
    nm = PRODUCT(n(1:rk-1))                                          !--- number of elements per tracer
    ie = itr*nm; ib = ie-nm+1; m=ll(ib:ie)                           !--- section bounds for tracer "itr" ; outlayers mask
    IF(.NOT.ANY(m)) CYCLE                                            !--- no outlayers for tracer "itr"
    v = TRIM(vnm(MIN(itr,SIZE(vnm))))//'('//TRIM(s)                  !--- "<name>("
    IF(nv == 1) ttl(rk) = TRIM(v)//','//num2str(itr)//')'            !--- "<name>(i,j,itr)" (single name)
    IF(nv /= 1) ttl(rk) = TRIM(v)//')'                               !--- "<nam(itr)>(i,j)" (one name each table/itr index)
    IF(rk==2) lerr = dispTable('ir',  ttl, i=cat(PACK(ki,m)),            r=cat(PACK(a(ib:ie),m)), &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
    IF(rk==3) lerr = dispTable('iir', ttl, i=cat(PACK(ki,m),PACK(kj,m)), r=cat(PACK(a(ib:ie),m)), &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
    CALL msg("can't display outliers table", sub, lerr, unt)
    IF(lerr) RETURN
  END DO
END FUNCTION dispOutliers_1
!==============================================================================================================================
LOGICAL FUNCTION dispOutliers_2(ll, a, n, err_msg, nam, subn, nRowMax, nColMax, nHead, unit) RESULT(lerr)
  IMPLICIT NONE
! Display outliers list in tables
! If "nam" is supplied and, it means the last index is for tracers => one table each tracer for rank > 2.
  LOGICAL,                    INTENT(IN)  :: ll(:)                   !--- Linearized mask of outliers
  REAL,                       INTENT(IN)  ::  a(:,:)                 !--- Linearized arrays of values stacked along 2nd dim.
  INTEGER,                    INTENT(IN)  ::  n(:)                   !--- Profile before linearization
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN)  :: err_msg, nam(:), subn   !--- Error message, variables and calling subroutine names
  INTEGER,          OPTIONAL, INTENT(IN)  :: nRowMax                 !--- Maximum number of lines to display    (default: all)
  INTEGER,          OPTIONAL, INTENT(IN)  :: nColMax                 !--- Maximum number of characters per line (default: 2048)
  INTEGER,          OPTIONAL, INTENT(IN)  :: nHead                   !--- Number of front columns to duplicate  (default: 1)
  INTEGER,          OPTIONAL, INTENT(IN)  :: unit                    !--- Output unit                           (def: lunout)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen)                   :: mes, sub, fm='(f12.9)', prf
  CHARACTER(LEN=maxlen),      ALLOCATABLE :: ttl(:), vnm(:)
  INTEGER,                    ALLOCATABLE :: ki(:), kj(:), kl(:)
  INTEGER                                 :: i, j, k, rk, nv, unt, nRmx, nCmx, nHd
  REAL,                       ALLOCATABLE :: val(:,:)

  lerr = ANY(ll); IF(.NOT.lerr) RETURN                               !--- No outliers -> finished
  rk = SIZE(n); nv = SIZE(a,2)
  mes = 'outliers found';        IF(PRESENT(err_msg)) mes = err_msg  !--- Error message
  vnm = [(ACHAR(k+96),k=1,nv)];  IF(PRESENT(nam ))    vnm = nam      !--- Variables names
  sub = 'dispOutliers';          IF(PRESENT(subn))    sub = subn     !--- Calling subroutine name
  nRmx= SIZE(a);          IF(PRESENT(nRowMax)) nRmx=MIN(nRmx,nRowMax)!-- Maximum number of lines to print
  nCmx= 2048;             IF(PRESENT(nColMax)) nCmx=MIN(nCmx,nColMax)!-- Maximum number of characters each line
  nHd = 1;                IF(PRESENT(nHead))   nHd = nHead           !--- Number of front columns to duplicate
  unt = lunout;                  IF(PRESENT(unit))    unt = unit     !--- Unit to print messages
  lerr= SIZE(vnm) /= nv;         CALL msg('SIZE(nam) /= SIZE(a,2)',                  sub, lerr, unt); IF(lerr) RETURN
  lerr= SIZE(a,1) /= SIZE(ll);   CALL msg('"ll" and "a" sizes mismatch',             sub, lerr, unt); IF(lerr) RETURN
  lerr= SIZE(a,1) /= PRODUCT(n); CALL msg('profile "n" does not match "a" and "ll"', sub, lerr, unt); IF(lerr) RETURN

  SELECT CASE(rk)                                                   !--- Indices list
    CASE(0); IF(ll(1)) THEN; WRITE(unt,'(a,", ",a," = ",2f12.9)')TRIM(vnm(1)),TRIM(vnm(2)),a(1,1),a(1,2); RETURN; END IF
    CASE(1); ki = [  (i,i=1,n(1)) ]
    CASE(2); ki = [ ((i,i=1,n(1)),j=1,n(2))];           kj = [ ((j,i=1,n(1)),j=1,n(2))]
    CASE(3); ki = [(((i,i=1,n(1)),j=1,n(2)),k=1,n(3))]; kj = [(((j,i=1,n(1)),j=1,n(2)),k=1,n(3))]
             kl = [(((k,i=1,n(1)),j=1,n(2)),k=1,n(3))]
    CASE DEFAULT; WRITE(unt,*)'Sorry: routine "dispOutliers_2" is limited to rank 3'; RETURN
  END SELECT

  ttl = [(ACHAR(k), k = 105, 104+rk), vnm]                           !--- Titles list ('i', 'j', 'k'...'var1', 'var2', ...)
  prf = REPEAT('i',rk)//REPEAT('r',nv)                               !--- Profile
  ALLOCATE(val(COUNT(ll),nv)); DO k=1, nv; val(:,k) = PACK(a(:,k),ll); END DO
  IF(rk == 1) lerr = dispTable(prf, ttl, i = cat(PACK(ki,ll)),                         r = val, &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
  IF(rk == 2) lerr = dispTable(prf, ttl, i = cat(PACK(ki,ll),PACK(kj,ll)),             r = val, &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
  IF(rk == 3) lerr = dispTable(prf, ttl, i = cat(PACK(ki,ll),PACK(kj,ll),PACK(kl,ll)), r = val, &
                                 rFmt=fm, nRowMax=nRmx, nColMax=nCmx, nHead=nHd, unit=unt, sub=sub)
  CALL msg("can't display outliers table", sub, lerr, unt)
END FUNCTION dispOutliers_2
!==============================================================================================================================


!==============================================================================================================================
!=== Reduce an algebrical expression (basic operations and parenthesis) to a single number (string format) ====================
!==============================================================================================================================
LOGICAL FUNCTION reduceExpr_1(str, val) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),      INTENT(IN)  :: str
  CHARACTER(LEN=maxlen), INTENT(OUT) :: val
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen)              :: v
  CHARACTER(LEN=1024)                :: s, vv
  CHARACTER(LEN=1024), ALLOCATABLE :: vl(:)
  INTEGER,             ALLOCATABLE :: ip(:)
  INTEGER :: nl, nn, i, j, im, ix
  LOGICAL :: ll
  s = str

  !--- Check wether the parenthesis are correctly formed
  ll = strCount(s,'(',nl)
  ll = strCount(s,')',nn)
  lerr = nl /= nn
  CALL msg('Mismatching number of opening and closing parenthesis: '//TRIM(s), ll=lerr); IF(lerr) RETURN
  nl = 2*nl-1

  !--- Build vectors ; vl: parenthesis-separated elements ; ip: parenthesis types (1: opening, 2: closing)
  ALLOCATE(ip(nl-1),vl(nl))
  j = 1; im = 1
  DO i = 1, LEN_TRIM(str)
    ix = INDEX('()', str(i:i))
    IF(ix == 0) CYCLE
    ip(j) = ix
    vl(j) = str(im:i-1)
    j = j + 1; im = i + 1
  END DO
  vl(j) = str(im:LEN_TRIM(str))

  !--- Search for each opening/closing parenthesis pair
  DO WHILE(nl > 1)
    i = 1; DO WHILE(ip(i) /= 1 .OR. ip(i+1) /= 2); i = i + 1; END DO !IF(i > SIZE(ip)+1) EXIT;END DO
    lerr = reduceExpr_basic(vl(i+1), v); IF(lerr) RETURN
    v = TRIM(vl(i))//TRIM(v); IF(i+2<=nl) v=TRIM(v)//TRIM(vl(i+2))
    vv = v//REPEAT(' ',768)
    IF(i == 1) THEN;         ip =  ip(3:nl-1);                vl = [            vv, vl(4  :nl)]
    ELSE IF(i == nl-1) THEN; ip =  ip(1:nl-2);                vl = [vl(1:nl-3), vv            ]
    ELSE;                    ip = [ip(1: i-1), ip(i+2:nl-1)]; vl = [vl(1: i-1), vv, vl(i+3:nl)]; END IF
    nl = SIZE(vl)
  END DO
  lerr = reduceExpr_basic(vl(1), val)
END FUNCTION reduceExpr_1


!==============================================================================================================================
!=== Reduce a simple algebrical expression (basic operations, no parenthesis) to a single number (string format) ==============
!==============================================================================================================================
LOGICAL FUNCTION reduceExpr_basic(str, val) RESULT(lerr)
  IMPLICIT NONE
  CHARACTER(LEN=*),      INTENT(IN)  :: str
  CHARACTER(LEN=*),      INTENT(OUT) :: val
  REAL(KIND=REAL64),     ALLOCATABLE :: vl(:)
  INTEGER,               ALLOCATABLE :: id(:)
  CHARACTER(LEN=maxlen), ALLOCATABLE :: ky(:)
  CHARACTER(LEN=1),      ALLOCATABLE :: op(:)
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=1024) :: s
  REAL(KIND=REAL64) :: v, vm, vp
  INTEGER      :: i, ni, io
  lerr = .FALSE.
  IF(is_numeric(str)) THEN; val=TRIM(str); RETURN; END IF
  op = ['^','/','*','+','-']                                                   !--- List of recognized operations
  s = str
  lerr = strParse_m(s, op, ky, lSc=.TRUE., id = id)                            !--- Parse the values
  IF(lerr) RETURN                                                              !--- Problem with the parsing
  vl = str2dble(ky)                                                            !--- Conversion to doubles
  lerr = ANY(vl >= HUGE(1._REAL64))
  CALL msg('Some values are non-numeric in: '//TRIM(s), ll=lerr)
  IF(lerr) RETURN                                                              !--- Non-numerical values found
  DO io = 1, SIZE(op)                                                          !--- Loop on known operators (order matters !)
    DO i = SIZE(id), 1, -1                                                     !--- Loop on found operators
      ni = SIZE(id)
      IF(id(i) /= io) CYCLE                                                    !--- Current found operator is not op(io)
      vm = vl(i); vp = vl(i+1)                                                 !--- Couple of values used for current operation
      SELECT CASE(op(io))                                                      !--- Perform operation on the two values
        CASE('^'); v = vm**vp
        CASE('/'); v = vm/vp
        CASE('*'); v = vm*vp
        CASE('+'); v = vm+vp
        CASE('-'); v = vm-vp
      END SELECT
      IF(i == ni) THEN; vl = [vl(1:ni-1), v]; ELSE; vl = [vl(1:i-1), v, vl(i+2:ni+1)]; END IF
      IF(i == ni) THEN; id =  id(1:ni-1);     ELSE; id = [id(1:i-1),    id(i+1:ni  )]; END IF
    END DO
  END DO
  val = num2str(vl(1))

END FUNCTION reduceExpr_basic
!==============================================================================================================================

!==============================================================================================================================
FUNCTION reduceExpr_m(str, val) RESULT(lerr)
  IMPLICIT NONE
  LOGICAL,               ALLOCATABLE              :: lerr(:)
  CHARACTER(LEN=*),                   INTENT(IN)  :: str(:)
  CHARACTER(LEN=maxlen), ALLOCATABLE, INTENT(OUT) :: val(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: i
  ALLOCATE(lerr(SIZE(str)),val(SIZE(str)))
  lerr(:) = [(reduceExpr_1(str(i), val(i)), i=1, SIZE(str))]
END FUNCTION reduceExpr_m
!==============================================================================================================================


!==============================================================================================================================
!=== Check whether a string is a number or not ================================================================================
!==============================================================================================================================
ELEMENTAL LOGICAL FUNCTION is_numeric(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  REAL    :: x
  INTEGER :: e
  CHARACTER(LEN=12) :: fmt
  IF(TRIM(str) == '') THEN; out = .FALSE.; RETURN; END IF
  WRITE(fmt,'("(f",i0,".0)")') LEN_TRIM(str)
  READ(str,fmt,IOSTAT=e) x
  out = e==0 .AND. INDEX('Ee',str(LEN_TRIM(str):LEN_TRIM(str)))==0
END FUNCTION is_numeric
!==============================================================================================================================


!==============================================================================================================================
!=== Convert a string into a logical/integer integer or an integer/real into a string =========================================
!==============================================================================================================================
ELEMENTAL INTEGER FUNCTION str2bool(str) RESULT(out)  !--- Result: 0/1 for .FALSE./.TRUE., -1 if not a valid boolean
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: ierr
  LOGICAL :: lout
  READ(str,*,IOSTAT=ierr) lout
  out = -HUGE(1)
  IF(ierr /= 0) THEN
    IF(ANY(['.false.', 'false  ', 'no     ', 'f      ', 'n      '] == strLower(str))) out = 0
    IF(ANY(['.true. ', 'true   ', 'yes    ', 't      ', 'y      '] == strLower(str))) out = 1
  ELSE
    out = 0; IF(lout) out = 1
  END IF
END FUNCTION str2bool
!==============================================================================================================================
ELEMENTAL INTEGER FUNCTION str2int(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: ierr
  READ(str,*,IOSTAT=ierr) out
  IF(ierr/=0) out = -HUGE(1)
END FUNCTION str2int
!==============================================================================================================================
ELEMENTAL REAL(KIND=REAL32) FUNCTION str2real(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: ierr
  READ(str,*,IOSTAT=ierr) out
  IF(ierr/=0) out = -HUGE(1._REAL32)
END FUNCTION str2real
!==============================================================================================================================
ELEMENTAL REAL(KIND=REAL64) FUNCTION str2dble(str) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: str
  INTEGER :: ierr
  READ(str,*,IOSTAT=ierr) out
  IF(ierr/=0) out = -HUGE(1._REAL64)
END FUNCTION str2dble
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION bool2str(b) RESULT(out)
  IMPLICIT NONE
  LOGICAL, INTENT(IN) :: b
  WRITE(out,*)b
  out = ADJUSTL(out)
END FUNCTION bool2str
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION int2str(i, nDigits) RESULT(out)
  IMPLICIT NONE
  INTEGER,           INTENT(IN) :: i
  INTEGER, OPTIONAL, INTENT(IN) :: nDigits
!------------------------------------------------------------------------------------------------------------------------------
  WRITE(out,*)i
  out = ADJUSTL(out)
  IF(.NOT.PRESENT(nDigits)) RETURN
  IF(nDigits > LEN_TRIM(out)) out = REPEAT('0', nDigits - LEN_TRIM(out))//TRIM(out)
END FUNCTION int2str
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION real2str(r,fmt) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL32),          INTENT(IN) :: r
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: fmt
!------------------------------------------------------------------------------------------------------------------------------
  IF(     PRESENT(fmt)) WRITE(out,fmt)r
  IF(.NOT.PRESENT(fmt)) WRITE(out, * )r
  out = ADJUSTL(out)
END FUNCTION real2str
!==============================================================================================================================
ELEMENTAL CHARACTER(LEN=maxlen) FUNCTION dble2str(d,fmt) RESULT(out)
  IMPLICIT NONE
  REAL(KIND=REAL64),          INTENT(IN) :: d
  CHARACTER(LEN=*), OPTIONAL, INTENT(IN) :: fmt
!------------------------------------------------------------------------------------------------------------------------------
  IF(     PRESENT(fmt)) WRITE(out,fmt)d
  IF(.NOT.PRESENT(fmt)) WRITE(out, * )d
  out = ADJUSTL(out)
END FUNCTION dble2str
!==============================================================================================================================
ELEMENTAL SUBROUTINE cleanZeros(s)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(INOUT) :: s
  INTEGER :: ls, ix, i
  IF(is_numeric(s)) THEN
    ls = LEN_TRIM(s)
    ix = MAX(INDEX(s,'E'),INDEX(s,'e'),INDEX(s,'D'),INDEX(s,'d'))
    IF(ix == 0) THEN
      DO ix = ls,1,-1; IF(s(ix:ix) /= '0') EXIT; END DO; s=s(1:ix+1)
    ELSE IF(INDEX(s,'.')/=0) THEN
      i = ix-1; DO WHILE(s(i:i) == '0'); i = i-1; END DO; s=s(1:i)//s(ix:ls)
    END IF
  END IF
END SUBROUTINE cleanZeros
!==============================================================================================================================


!==============================================================================================================================
FUNCTION addQuotes_1(s) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)  :: s
  CHARACTER(LEN=:), ALLOCATABLE :: out
  IF(needQuotes(s)) THEN; out = "'"//TRIM(s)//"'"; ELSE; out = s; END IF
END FUNCTION addQuotes_1
!==============================================================================================================================
FUNCTION addQuotes_m(s) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN)  :: s(:)
  CHARACTER(LEN=:), ALLOCATABLE :: out(:)
!------------------------------------------------------------------------------------------------------------------------------
  INTEGER :: k, n
  n = MAXVAL(LEN_TRIM(s), MASK=.TRUE.)
  ALLOCATE(CHARACTER(LEN=n) :: out(SIZE(s)))
  DO k=1,SIZE(s)
    IF(needQuotes(s(k))) THEN; out(k) = "'"//TRIM(s(k))//"'"; ELSE; out(k) = s(k); END IF
  END DO
END FUNCTION addQuotes_m
!==============================================================================================================================
ELEMENTAL LOGICAL FUNCTION needQuotes(s) RESULT(out)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(IN) :: s
  CHARACTER(LEN=1) :: b, e
!------------------------------------------------------------------------------------------------------------------------------
  out = .TRUE.; IF(TRIM(s) == '') RETURN
  b = s(1:1); e = s(MAX(1,LEN_TRIM(s)):MAX(1,LEN_TRIM(s)))
  out = .NOT.is_numeric(s) .AND. (b /= "'" .OR. e /= "'") .AND. ( b /= '"' .OR. e /= '"')
END FUNCTION needQuotes
!==============================================================================================================================


!==============================================================================================================================
!=== DISPLAY "<message>: the following <items> are <reason>" FOLLOWED BY THE LIST OF <str> FOR WHICH <lerr>==T. ===============
!==============================================================================================================================
LOGICAL FUNCTION checkList(str, lerr, message, items, reason, nmax) RESULT(out)
  IMPLICIT NONE
! Purpose: Messages in case a list contains wrong elements (indicated by lerr boolean vector).
! Note:    Return value "out" is .TRUE. if there are errors (ie at least one element of "lerr" is TRUE).
  CHARACTER(LEN=*),   INTENT(IN)  :: str(:)
  LOGICAL,            INTENT(IN)  :: lerr(:)
  CHARACTER(LEN=*),   INTENT(IN)  :: message, items, reason
  INTEGER,  OPTIONAL, INTENT(IN)  :: nmax
!------------------------------------------------------------------------------------------------------------------------------
  CHARACTER(LEN=maxlen), ALLOCATABLE :: s(:)
  INTEGER :: i, nmx
  nmx = 256; IF(PRESENT(nmax)) nmx=nmax
  out = ANY(lerr); IF(.NOT.out) RETURN
  CALL msg(TRIM(message)//': the following '//TRIM(items)//' are '//TRIM(reason)//':')
  s = strStackm(PACK(str, MASK=lerr), ', ',nmx)
  DO i=1,SIZE(s,DIM=1); CALL msg(s(i)); END DO
END FUNCTION checkList
!==============================================================================================================================


!==============================================================================================================================
!=== Remove comment in line "str", ie all the characters from the first "#" sign found in "str". ==============================
!==============================================================================================================================
SUBROUTINE removeComment(str)
  IMPLICIT NONE
  CHARACTER(LEN=*), INTENT(INOUT) :: str
  INTEGER :: ix
  ix = INDEX(str,'# '); IF(ix /= 0) str = str(1:ix-1)//REPEAT(' ',LEN(str)-ix+1)
END SUBROUTINE removeComment
!==============================================================================================================================


END MODULE strings_mod
