#!/bin/bash

set -eu
. lmdz_env.sh

#####################################################################
# Authors :
# F. Hourdin, frederic.hourdin@lmd.ipsl.fr
# Modified A. Sima, adriana.sima@lmd.ipsl.fr
# Rewritten 2024 A. Barral
#
# This is the main user script of LMDZ_Setup. It defines basic options for the simulations, and runs setup.sh.
# Settings such as model and simulation paths are set in lmdz_env.sh. You must modify it to set the current LMDZ_Setup path as <root_dir>.
# Expert options are set directly in setup.sh via <define_expert_options>.
#
#####################################################################

#===========================================================
# 1. Model setup
#===========================================================

# Available version :
# -------------------
# on https://lmdz.lmd.jussieu.fr/pub/src_archives/testing/
# version="20250210.trunk -unstable" if using an unstable version (expert)
version="20241018.trunk"
version="20250327.trunk"

svn="" 
# CONTACT LMDZ Team :
# 	email: poihl@listes.lmd.ipsl.fr
# 	Mattermost: https://mattermost.lmd.ipsl.fr/lmdz/channels/installation-et-tutoriels

# Grid number of points IMxJMxLM
resol="144x142x79"

## Using XIOS for IOs: "-xios" (enabled) / "" (disabled)
xios=""

# Using or not the Cosp simulator: "-cosp" (enabled) / "" (disabled)
cosp=""

# Choice of physics: "lmd" (phylmd) / "new" (phynew) / "lmdiso" (isotopes)
lmd_phys="lmd"

#===========================================================
# 2. Simulation setup
#===========================================================

# Initial state and Boundary conditions
# init=   1: to create a new start in INIT
#         0: to read start files in INIT
#       SIM: to read start files from previous simulation SIM0, /!\ SIM0 must be in the same folder as the new SIM
# (limit.nc and aerosols forcing are put in ./LIMIT)
init=1

# climato=1 : Climatological SSTs with 360-day calendar
#        =0 : interannual SSTs with true (ie gregorian) calendar 
climato=1

# Nudging: Can only be activated with climato=0 and freq=mo
# "-nudging" (enabled) / "" (disabled)
nudging=""
#If using nudging, then check DEF/guide.def :
#  - ok_guide will be automatically set to "y" by setup.sh
#  - You may want to check/customize the nudging parameters

# Length of elementary simulations yr (year) or mo (month)
freq="yr"

# Initial/final month for simulation
# If $init=1, the INIT file will be called start.200001.nc,
# but the data correspond in fact to another day.
# NB : the run stops in the BEGINNING of mthend (test "next=stopsim")
mthini=200001
mthend=200501

# Expert options passed to setup.sh
rad=rrtm        # Radiation oldrad/rrtm/ecrad
netcdf=0        # netcddf 0(from system)/1 reinstall/dir. containing nectdf*
aerosols=clim   # n (no) / clim 
veget=CMIP6     # Orchidee version : CMIP6 / 7983 (orch2.2 ) / 8758 (orch4)
save_pub=$LMDZD # Will save files downloaded by wget on $LMDZD (could be 0 or 1)

# For rapid tests
# resol=32x32x39 ; veget=none ; aerosols=n ; rad=oldrad ; freq=mo ; mthend=200101

#-----------------------------------------------------------
# Output files, frequencies, levels 
#   If you use IOIPSL (option xios=""), you may want to choose and customize DEF/config.def. 
#     A few versions are available as DEF/config.def_*; config.def is a copy of config.def_default 
#     See phys_out_filekeys, phys_out_filelevels, phys_out_filetimesteps, and supplementary variables
#   If you use XIOS (option xios="-xios"), check/modify DEF/XMLfiles*/file*xml 
#   In both cases, the default output is "histday".
#-----------------------------------------------------------
# Grid characteristics (regular, zoomed)
#   You may want to choose and customize DEF/gcm.def 
#   A few versions are available as DEF/gcm.def_*; 
#      The default, for regular grid 144x142x79, corresponds to _iperiod7 
#      (Expert : TEMPORARILY : setup.sh forces use of gcm.def_zNAfrica_BiJe for aerosols=spla)

#===========================================================
# 3. Example of calling setup.sh in loop
#===========================================================

# By default, the series is done on one or more of the various versions of physiq.def 
# to be chosen among DEF/PHYS/physiq.def_* , and listed in "physics" :
# NOTE : automatically choosing DEF/PHYS/physiq.def_NPiso for isotopes
physics="NPv6.3"
if [[ $lmd_phys = "lmdiso" ]]; then physics="NPiso"; fi

if [[ $(echo "$physics" |wc -w) -gt 1 && $init = 1 ]]; then echo "!=!=! TO LOOP OVER MULTIPLE physics [$physics], YOU NEED init =/= 1 AND start* + limit* files ALREADY AVAILABLE IN THIS FOLDER !=!=!"; exit 1; fi

for phys in $physics; do
  deffile="DEF/PHYS/physiq.def_$phys"
  if [[ -f $deffile ]]; then
     cp -p $deffile DEF/physiq.def

    # name of simulation : can be changed to user's convenience
    # (Default: name=$phys to loop on different physics)
    name="$phys"

    # launching setup.sh with the options defined in this main.sh. 
    # Remember: some other options are only available in lmdz_env.sh and setup.sh.
    # shellcheck disable=SC2086
    ./setup.sh -v "$version" -d "$resol" -mthini "$mthini" -mthend "$mthend" -init "$init" -climato "$climato" -f "$freq" -p "$lmd_phys" -name "$name" $cosp $xios $nudging -r "$svn" -rad $rad -netcdf $netcdf -aerosols $aerosols -veget $veget -save_pub $save_pub
  else
    echo "File $deffile inexistent"; exit 1
  fi
done

