#!/bin/bash
# This script sets up and launches a (series of) simulation(s).
# RECOMMENDATION: use main.sh to drive it (do not run it directly)

set -eu

#---------------------------------------------------------------------------
function define_expert_options() { # Expert-level choices
#---------------------------------------------------------------------------

  # optim: either "" or "-debug" to compile in debug mode
  #       (slower but better diagnosis of segfaults)
  optim=""

  # "n" or "y". If testmode="y", then simulations run for a single day
  # per period.
  # NOTE: you must set mthend accordingly !
  testmode="n"

  # Radiative code: "oldrad" / "rrtm" / "ecrad"
  rad="rrtm"

  #   !!! STRONG recommendation : experiments with DIFFERENT Orchidee or
  #   aerosol options should be performed in DIFFERENT LMDZ_Setup folders
  #   !!! (especially as they may need different initial files)

  # AEROSOLS : "n" (=no)/"clim" (=average 1995-2014) / "spla" (interactive)
  # (WARNING : if you first run the scripts with aerosols=n, then you want
  # to change to =clim, you must remove the INIT and LIMIT folders and
  # rerun main.sh with init=1 in order to create aerosol files

  aerosols="clim"

  # SURFACE/VEGETATION SCHEME
  # - "none" : bucket scheme
  #   CMIP6 : orchidee CMIP6 version
  #   7983/7994 : orchidee svn release 7983 (orch2.2) and 7994 (last tested)
  #   If you need other orch versions, and also require XIOS, you'll need
  #                    to create the appropriate files in DEF/XMLfilesOR...
  veget="CMIP6"

  # New snow scheme INLANDSIS! "y" / "n"
  # Activates INLANDSIS compilation
  #  to be done : treatment of specific restart and def file
  inlandsis="n"

  # netcdf: 0 (use existing library) / 1 (recompile netcdf, slow)
  netcdf=0

  # --->>> ALSO PAY ATTENTION TO OUTPUT files, frequency, level ------------
  #   With IOIPSL : Choose your config.def among versions available in DEF,
  #     copy it as config.def (the copy is done automatically for "iso"),
  #     edit it and set up output files, frequencies and levels.
  #     NB : For aerosols=spla, output level minimum 4 is required to outpu
  #     the specific variables.
  #     For aerosols=n, aerosol flags will automatically be set to "n".
  #   With XIOS : adjust DEF/XMLfiles*/file*xml
}


#---------------------------------------------------------------------------
function enable_platform() {  # Sed platform-specific headers
#---------------------------------------------------------------------------
  local file="$1"
  local platform

  case ${hostname:0:5} in
    jean-) platform="JZ";;
    spiri) platform="SP";;
    adast) platform="ADS";;
    *) echo "Warning: $hostname is not a known job platform (ignore if running locally)"; return 0;;
  esac

  sed -i'' -e "s/^#@$platform//" "$file"
}

#---------------------------------------------------------------------------
function load_install_lib() { #Fetch and source install_lmdz.sh to get myget
#---------------------------------------------------------------------------
  if [[ ! -f "install_lmdz.sh" ]]; then
    wget --no-cache "http://svn.lmd.jussieu.fr/LMDZ/BOL/script_install/install_lmdz.sh"
  fi
  # shellcheck disable=SC1090
  source <(sed 's/function \(.*\) {/function installlmdz_\1 {/g' install_lmdz.sh)  # source with a namespace for functions
  function myget { installlmdz_myget "$@"; }
}

#---------------------------------------------------------------------------
function set_default_params() { # Default value of script parameters
#---------------------------------------------------------------------------

  SIM=$(basename "$local")CTL  # name
  phylmd="lmd" #option -p $phylmd for makelmdz

  cosp="n"  # COSP
  xios="n"  #XIOS

  # Nudging :
  ok_guide="n"
  # With nudging, use real calendar (climato=0) and monthly integrations
  climato=1
  freq="mo"  # frequence mensuelle mo ou annuelle yr

  # NB : the run stops in the BEGINNING of mthend (test "next=stopsim")
  mthini=200001
  mthend=200501
  resol="144x142x79"

  version="20230412.trunk"
  svn=""

  init=1

  LIMIT="LIMIT"

}

#---------------------------------------------------------------------------
function read_cmdline_args() {
#---------------------------------------------------------------------------
  while (($# > 0)); do
    case $1 in
      "-h") cat <<........fin
       setup.sh can be launched/driven by main.sh
             setup.sh [-v version] [-r svn_release] [-init INIT] [-d 96x95x79] [-f mo] [-nudging]
             -v "version" like 20150828.trunk; see https://lmdz.lmd.jussieu.fr/Distrib/LISMOI.trunk (default <$version>)
             -r "svn_release"        either the svn release number or "last" (default <$svn>)
             -d IMxJMxLM             to run in resolution IM x JM x LM (default <$resol>)
             -install                pour installer et compiler le modele
             -f mo/yr                pour tourner en mensuel ou annuel (default <$freq>)
             -rad RAD                Radiation
             -netcdf  0/1/DIR        Netcdf installation
             -aerosols n/clim/spla   Aerosols : none / climatological / interactive with SPLA
             -veget CMIP6            CMIP6 version of rchidee
                    7983             orch2.2 
                    7994             trunk
             "INIT"                  1: creates INIT and LIMIT
                                     0: reads from INIT and LIMIT
                                     SIMU: reads from preexisting simulation SIMU and LIMIT (default <$init>)
             -nudging                to run with nudging. Nudging files must be created independently
             -p                      the physics to use (default <$phylmd>)
             -name                   install folder name (default <$SIM>)
             Other options available (see "options" section in the script)
........fin
        exit;;
      "-v") version="$2"; shift; shift;;
      "-r") svn=$2; shift; shift;;
      "-d") resol=$2; shift; shift;;
      "-f") freq=$2; shift; shift;;
      "-p") phylmd=$2; shift; shift;;
      "-name") SIM=$2; shift; shift;;
      "-rad") rad=$2; shift; shift;;
      "-netcdf") netcdf=$2; shift; shift;;
      "-aerosols") aerosols=$2; shift; shift;;
      "-veget") veget=$2; shift; shift;;
      "-cosp") cosp=y; shift;;
      "-xios") xios=y; shift;;
      "-init") init=$2; shift; shift;;
      "-nudging") ok_guide=y; shift;;
      "-climato") climato=$2; shift; shift;;
      "-mthini") mthini=$2; shift; shift;;
      "-mthend") mthend=$2; shift; shift;;
      *) echo "unexpected $1"; $0 -h; exit
    esac
  done

  # Initialisation
  if [[ $init = 1 || $init = 0 ]]; then
    INIT="INIT"
  else
    INIT=$init
  fi

  case $rad in
    oldrad) iflag_rrtm=0; NSW=2;;
    rrtm)   iflag_rrtm=1; NSW=6;;
    ecrad)  iflag_rrtm=2; NSW=6;;
  esac

  im=$(echo "$resol" | cut -dx -f1)
  jm=$(echo "$resol" | cut -dx -f2)
  lm=$(echo "$resol" | cut -dx -f3)

  yearini=$(echo "$mthini" | cut -c-4)
  if [[ $freq = yr ]]; then stopsim=$(echo "$mthend" | cut -c-4); else stopsim=$mthend; fi

  if [[ -d $SIM ]]; then
     echo "La simulation $SIM existe deja. Il est conseillé d'arrêter et de vérifier."
     echo "Si vous êtes sûr de vous, vous pouvez la prolonger. Voulez vous la prolonger ? (y/n)"
     read -r ans
     if [[ $ans != y ]]; then exit 1; fi
  fi

  ######################################################################
  # Choix du nombre de processeurs
  # NOTES :
  # omp=8 by default (for Jean-Zay must be a divisor of 40 procs/node)
  # omp=1 : required for SPLA (omp parallelism not ready)
  # omp=2 for veget=CMIP6+XIOS beacause of a bug in ORCHIDEE/src_xml/xios_orchidee.f90
  ######################################################################
  (( mpi = ( jm + 1 ) / 2 ))
  omp=8
  if [[ $aerosols = "spla" ]]; then omp=1; fi
  if [[ $veget = "CMIP6" && $xios = "y" ]]; then omp=2; fi
  if [[ $mpi -gt $NB_MPI_MAX ]]; then mpi=$NB_MPI_MAX; fi
  if [[ $omp -gt $NB_OMP_MAX ]]; then omp=$NB_OMP_MAX; fi

  # Compute how many mpi per node (required e.g. for Adastra)
  mpi_per_node=0
  if [[ $NB_CORE_PER_NODE_MAX -gt 0 ]]; then
    local N_omp_mt=1
    if [[ $omp -gt 1 ]]; then (( N_omp_mt = omp / N_HYPERTHREADING )); fi  # take into account hyperthreading
    (( mpi_per_node = NB_CORE_PER_NODE_MAX / N_omp_mt ))
    if [[ mpi_per_node -gt mpi ]]; then mpi_per_node=$mpi; fi
  fi

  echo "Total MPI=$mpi (PER NODE=$mpi_per_node), OMP=$omp"
}

#---------------------------------------------------------------------------
function ensure_correct_option_combinations() {
  # AVOID COMBINATIONS OF OPTIONS THAT DON'T WORK in user choices
#---------------------------------------------------------------------------
  if [[ $ok_guide = y && $climato = 1 ]]; then
     echo "STOP: Running nudged simulations with climatological SSTs is not planned. Change <climato> to <0> or modify the setup (experts)"; exit 1
  fi

  if [[ $climato = 0 && $freq = "yr" ]]; then
     echo "STOP: Running simulations with interannual SSTs is possible only month by month and a true calendar."
     echo "Change <climato> to <1> or <freq> to <mo> or modify setup.sh (experts)"; exit 1
  fi


  # (Temporary) Constraints for aerosols=spla :
  # --> resolution 128x88x79 and rad=rrtm
  if [[ $aerosols = "spla" && ( ${im}x${jm} != "128x88" || $freq != mo ) ]]; then
    echo 'STOP: For now, <aerosols=spla> requires <resol=128x88x79>,'
    echo 'and uses the zoomed grid from gcm.def_zNAfrica_BiJe,'
    echo 'for which forcing & initial files are available'
    echo "Right now resol=<$resol>"
    echo 'Should be run in monthly mode'
    exit 1
  fi

  if [[ $rad = "ecrad" && $phylmd != "lmd" ]]; then
    echo "<rad=ecrad> is only supported for <phy=phylmd> here"  # (Amaury) I added this check because we fetch ecrad data from libf/phylmd/ecrad/data only.
  fi
}

#---------------------------------------------------------------------------
function install_model() {
#---------------------------------------------------------------------------
  mkdir -p "$LMDZD"

  local ins_xios ins_cosp ins_aero ins_inlandsis
  if [[ $xios = "y" ]]; then ins_xios="-xios"; else ins_xios=""; fi
  if [[ $cosp = "y" ]]; then ins_cosp="-cosp v1"; else ins_cosp=""; fi
  if [[ $aerosols = "spla" ]]; then ins_aero="-spla"; else ins_aero=""; fi
  if [[ $inlandsis = "y" ]]; then ins_inlandsis="-inlandsis"; else ins_inlandsis=""; fi
  if [[ -n $svn ]]; then svnopt="-r $svn"; else svnopt=""; fi

  version_name=LMDZ$(echo "$version" | sed -e 's/-v//g' -e 's/-unstable//' -e 's/-r/r/' -e 's/ //g')
  LMDZname="${version_name}${svn}OR$veget${ins_xios}"
  MODEL="$LMDZD/$LMDZname/modipsl/modeles/LMDZ"

  if [[ -d $MODEL ]]; then echo "Found existing install at MODEL=$MODEL"; fi
  echo "Installing model"
  cd "$LMDZD"
  cp "$local/install_lmdz.sh" .
  chmod +x install_lmdz.sh
  local make_j=8
  # We launch using $MPICMD, except if it's using mpirun (no srun equivalent for bash script) => if supported, the compilation runs in a cluster job
  jobcmd="\"$RUNBASHCMD $make_j\""
  if [[ ${hostname:0:5} = "jean-" ]]; then jobcmd="\"$RUNBASHCMD $make_j --partition=compil\""; fi  # On JeanZay: compile on the <compil> partition
  if [[ $(echo "$RUNBASHCMD" | cut -c -4) = "bash" ]]; then
     jobcmd="bash"
  fi
  if [[ $PARALLEL == 0 ]] ; then parallel= ; else parallel="-parallel mpi_omp" ; fi
  echo "./install_lmdz.sh -noclean $optim -v $version $svnopt -d $resol -rad $rad -bench 0 $parallel $ins_cosp $ins_xios $ins_aero $ins_inlandsis -name $LMDZname -veget $veget -netcdf $netcdf -arch $ARCH -make_j $make_j -jobcmd $jobcmd" >> install_lmdz_options.$$.sh
  chmod +x install_lmdz_options.$$.sh
  echo "Running install_lmdz_options.$$.sh"
  set -o pipefail
    gcm=$MODEL/$(./install_lmdz_options.$$.sh | tee /dev/tty | tail -n 1 | sed -n "s:.* executable is \(.*\.e\).*:\1:p")
  set +o pipefail
  mv install_lmdz.sh install_lmdz.$$.sh
  cd "$local"
}

#---------------------------------------------------------------------------
function setup_def() {  # modify various .def in ./DEF (+ xios xml )
#---------------------------------------------------------------------------
  cd "$local"

  # Utilisation des .def_iso pour LMDZ-ISOtopes
  if [[ $phylmd = "lmdiso" ]]; then
    for file_iso in $(ls DEF | grep _iso); do
      cp DEF/"$file_iso" DEF/"${file_iso%%_iso}"
    done
  fi

  ######################################################################
  # Choix de la grille verticale
  ######################################################################
  if [ ! -f "DEF/L$lm.def" ]; then
    echo "STOP: Résolution verticale non prévue - créer un fichier DEF/L$lm.def"; exit 1
  else
    sed -i'' -e "s/INCLUDEDEF=L.*.def/INCLUDEDEF=L$lm.def/" DEF/run.def
  fi

  ######################################################################
  # Changements dans les fichiers DEF/*def
  # (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $SIMRUNDIR)
  ######################################################################

  # Choix du fichier tracer.def coherent avec l'option "aerosols"
  #  NOTE : Le nouveau tracer.def_nospla par defaut n'inclut pas Rn-Pb;
  #  si on les veut, il faut utiliser ci-dessous; a la place, tracer_RN_PB.def
  #---------------------------------------------------------------------
  # NB : Les traceurs absents de start* files sont initialises=0 dans le code
  if [[ $aerosols = "spla" ]]; then
    # nouveau tracer.def (remplace "traceur.def")
    cp DEF/tracer.def_spla DEF/tracer.def
  elif [[ $phylmd = "lmdiso" ]]; then
    # déjà copié si 'y'
    cp DEF/tracer.def_nospla DEF/tracer.def
  fi

  # TEMPORAIREMENT pour spla on force l'utilisation de gcm.def_zNAfrica_BiJe (avec resolution 128x88x79)
  #----------------------------------------------------------------------
  if [[ $aerosols = "spla" ]]; then
      cp DEF/gcm.def_zNAfrica_BiJe DEF/gcm.def; fi

  # Inscription du choix ok_guide dans DEF/guide.def
  #---------------------------------------------------------------------
  sed -i'' -e 's/ok_guide=.*.$/ok_guide='$ok_guide'/' DEF/guide.def

  # Inscription du type de calendrier (qui est fonction de $climato) dans DEF/run.def
  #---------------------------------------------------------------------
  # NB Contrairement a ce qui est ecrit dans les fichiers run.def standard,
  # dans ce tutorial le choix earth_365d n'est pas disponible, et earth_366d s'appelle gregorian
  if [[ $climato = 0 ]]; then calend="gregorian"; else calend="earth_360d"; fi
  sed -i'' -e 's/calend=.*.$/calend='$calend'/' DEF/run.def

  # Changements dans config.def (pre-choisi, et regle pour output si utilisation avec IOIPSL)
  #   cf options veget, aerosols, cosp, xios
  #---------------------------------------------------------------------
  if [[ $veget = "none" ]]; then  VEGET="n"; else VEGET="y"; fi
  sed -i'' -e 's/VEGET=.*.$/VEGET='$VEGET'/' DEF/config.def

  ok_cdnc=y ; ok_ade=y ; ok_aie=y ; ok_alw=y
  case $aerosols in
    clim) flag_aerosol=6 ;;
    spla) flag_aerosol=1 ;;
    n)    flag_aerosol=0 ; ok_cdnc=n ; ok_ade=n ; ok_aie=n ; ok_alw=n ;;
    *)    echo Option aerosols=$aerosols not available ; exit 1
  esac
  sed -i'' -e 's/^ok_cdnc=.*.$/ok_cdnc='$ok_cdnc'/' \
           -e 's/^ok_ade=.*.$/ok_ade='$ok_ade'/' \
           -e 's/^ok_aie=.*.$/ok_aie='$ok_aie'/' \
           -e 's/^ok_alw=.*.$/ok_alw='$ok_alw'/' \
           -e 's/^flag_aerosol=.*.$/flag_aerosol='$flag_aerosol'/' DEF/config.def
  pwd

  # COSP : ok_cosp desactive COSP si on a compile avec; il ne l'active pas si on a compile sans
  sed -i'' -e 's/ok_cosp.*.$/ok_cosp='$cosp'/' DEF/config.def
  if [[ $cosp = "y" ]]; then \cp -f "$MODEL"/DefLists/cosp*.txt "$local"/DEF/; fi

  # Sorties LMDZ en fonction de l'option "xios"
  sed -i'' -e 's/ok_all_xml=.*.$/ok_all_xml='$xios'/' DEF/config.def

  # Ajuster physiq.def en fonction de radiative code (default: values for rad=rrtm)
  #   Pour isotopes=y , mettre iflag_ice_thermo=0 au lieu de 1
  #---------------------------------------------------------------------
  sed -i'' -e "s/iflag_rrtm=.*.$/iflag_rrtm=$iflag_rrtm/" -e "s/NSW=.*.$/NSW=$NSW/" DEF/physiq.def
  sed -i"" -e "s:directory_name.*$:directory_name=\"$MODEL/libf/phylmd/ecrad/data\",:" DEF/namelist_ecrad

  if [[ $phylmd = "lmdiso" ]]; then iflag_ice_thermo=0; else iflag_ice_thermo=1; fi
  sed -i -e 's/iflag_ice_thermo=.*.$/iflag_ice_thermo='$iflag_ice_thermo'/' DEF/physiq.def

  # Choix de orchidee.def en fonction de orchidee_rev; modification pour xios
  #  NOTE separate orchidee_pft.def file for ORCHIDEE trunk post-CMIP6
  #---------------------------------------------------------------------
  orchidee_def=orchidee.def_6.1
  orchidee_pft_def=""
  if [[ $veget = "7983" ]]; then
    orchidee_def=orchidee.def_6.2work
  elif [[ $veget = "7994" ]]; then
    orchidee_def=orchidee.def_6.4work
    orchidee_pft_def=orchidee_pft.def_6.4work
    if ! grep "INCLUDEDEF=orchidee_pft.def" DEF/run.def; then
      sed -i'' -e 's/INCLUDEDEF=orchidee.def/INCLUDEDEF=orchidee.def\nINCLUDEDEF=orchidee_pft.def/' DEF/run.def; fi
  fi
  cp -f DEF/$orchidee_def DEF/orchidee.def
  if [[ $orchidee_pft_def != "" ]]; then cp -f DEF/$orchidee_pft_def DEF/orchidee_pft.def; fi

  # Only for veget=CMIP6 it is still possible to use IOIPSL; newer versions of orchidee.def have XIOS_ORCHIDEE_OK = y
  sed -i'' -e 's/XIOS_ORCHIDEE_OK =.*.$/XIOS_ORCHIDEE_OK = '$xios'/' DEF/orchidee.def

  ######################################################################
  # Si on tourne avec XIOS, mise a jour des fichiers context et field* dans XMLfilesLMDZ
  # (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $SIMRUNDIR)
  ######################################################################
  if [[ $xios = y ]]; then
    cp -f "$MODEL"/DefLists/context_lmdz.xml "$local"/DEF/XMLfilesLMDZ/.
    cp -f "$MODEL"/DefLists/field_def_lmdz.xml "$local"/DEF/XMLfilesLMDZ/.
    if [[ $cosp = y ]]; then cp -f "$MODEL"/DefLists/field_def_cosp1.xml "$local"/DEF/XMLfilesLMDZ/.; fi
  fi
}

#---------------------------------------------------------------------------
function setup_ce0l() { # Verification de l'existance de l'état initial, compilation eventuelle pour sa creation
#---------------------------------------------------------------------------
  if [[ ! -d $INIT ]]; then
    if [[ $init = 0 ]]; then
      echo "STOP: Récuperer les repertoires $INIT ou lancer avec option -init"; exit 1
    else
      # Compile ce0l
      cd "$MODEL"
      sed -e 's/gcm$/ce0l/' compile.sh > compile_ce0l.sh; chmod +x compile_ce0l.sh
      echo "Compiling ce0l"
      if ! ./compile_ce0l.sh &> ce0l.log; then echo "STOP: ce0l compilation failed, see $MODEL/ce0l.log"; exit 1; fi
      echo "Compiled ce0l"
      ce0l=${gcm/gcm/ce0l}

      cd "$local"
    fi
  elif [[ $init = 1 ]]; then
    echo "STOP: Vous essayez d initialiser le modele mais $INIT existe deja"; exit 1
  fi
}

#---------------------------------------------------------------------------
function setup_simu() {
#---------------------------------------------------------------------------
  #SIMRUNTOPDIR="$SIMRUNBASEDIR/$(basename "$local")"
  SIMRUNTOPDIR="$SIMRUNBASEDIR"
  SIMRUNDIR=$SIMRUNTOPDIR
  mkdir -p "$SIMRUNDIR"
  cd "$SIMRUNDIR"
  echo "La simulation s'exécutera sur $SIMRUNDIR"

  #####################################################################
  # Creation du repertoire $SIM s'il n'existe pas deja
  #####################################################################
  if [[ ! -d $local/$SIM ]]; then
    mkdir "$local/$SIM"
    cd "$local"

    # Edit reb.sh
    cp reb.sh "$local/$SIM/reb.sh"; chmod +x "$local/$SIM/reb.sh"
    sed -i'' -e "s:^rebuild=.*.$:rebuild=$LMDZD/$LMDZname/modipsl/bin/rebuild:" "$local/$SIM/reb.sh"
    enable_platform "$local/$SIM/reb.sh"

    # Copy .def
    cp lmdz_env.sh "$local/$SIM/"
    mkdir "$local/$SIM/DEF"; cp DEF/*def DEF/namelis* "$local/$SIM/DEF/"
    #Pour XIOS, respectivement COSP, copier aussi les fichiers *xml / *txt
    if [[ $cosp = "y" ]]; then cp DEF/cosp*txt "$local/$SIM/DEF/"; fi
    if [[ $xios = "y" ]]; then
       cp DEF/XMLfilesLMDZ/*xml "$local/$SIM/DEF/"
       if [[ $veget != 'none' ]]; then cp DEF/XMLfilesOR$veget/*xml "$local/$SIM/DEF/"; fi
    fi
    chmod u+w "$local/$SIM"/DEF/*

    # Gestion du calendrier
    #######################
    cd "$SIM"
    sed -i'' -e "s/anneeref=.*.$/anneeref=$yearini/" DEF/run.def
    if [[ $freq = "yr" ]]; then date=$yearini; else date=$mthini; fi
    echo "$date a faire" >| etat

    # Recuperation des fichiers : executable initiaux et forcages
    #############################################################
    echo "date: $date"
    for f in start startphy; do
      inf=../$INIT/$f.$date.nc
      if [[ -f $inf || $init = 1 ]]; then ln -s "$inf" ./; else echo "STOP: $inf missing"; exit ; fi;
    done
    for f in sechiba stomate; do
      if [[ -f ../$INIT/start_$f.$date.nc ]]; then ln -sf "../$INIT/start_$f.$date.nc" .; fi
    done
    cp "$gcm" gcm.e
  fi
  cd "$local"/..

  #####################################################################
  echo "Modification du script de lancement"
  #####################################################################
  local cput
  if [[ $freq = "yr" ]]; then cput="04:00:00"; else cput="01:00:00"; fi
  local isotopes="n"
  if [[ $phylmd = "lmdiso" ]]; then isotopes="y"; fi
  sed -e "s/NOM_SIMU/$SIM/" \
  -e "s/time=.*.$/time=$cput/" \
  -e "s/ntasks=.*.$/ntasks=$mpi/" \
  -e "s/ntasks-per-node=.*.$/ntasks-per-node=$mpi_per_node/" \
  -e "s/cpus-per-task=.*.$/cpus-per-task=$omp/" \
  -e "s/nthreads=.*./nthreads=$omp/" \
  -e "s/MAINDIR=.*.$/MAINDIR=$(basename "$local")/" \
  -e "s:STORED=.*.*:STORED=$(dirname "$local"):" \
  -e "s:SCRATCHD=.*.*:SCRATCHD=$SIMRUNBASEDIR:" \
  -e "s/stopsim=.*.$/stopsim=$stopsim/" \
  -e "s/^veget=.*.$/veget=$veget/" \
  -e "s/^aerosols=.*.$/aerosols=$aerosols/" \
  -e "s/^isotopes=.*.$/isotopes=$isotopes/" \
  -e "s/^climato=.*.$/climato=$climato/" \
  -e "s/^ok_guide=.*.$/ok_guide=$ok_guide/" \
  "$local/script_SIMU" >| "$SIMRUNDIR/tmp_$SIM"

  enable_platform "$SIMRUNDIR/tmp_$SIM"

  if [[ $testmode = "y" ]]; then
    sed -i'' -e "s/time=.*.$/time=00:10:00/" -e "s/#nday=1/nday=1/" -e "s/#@TESTQ//" "$SIMRUNTOPDIR/tmp_$SIM"
  fi
}

#----------------------------------------------------------------------
function list_yrmth() {
#----------------------------------------------------------------------
    local liste=""
    local mthini=$1
    local mthend=$2
    local mo_=$mthini
    while [[ $mo_ != $mthend ]] ; do
        liste="$liste $mo_"
        yr__=$( echo $mo_ | cut -c 1-4 )
        mo__=$( echo $mo_ | cut -c 5-6 )
	echo YM $yr__ $mo__
        if [ $(( $mo__ +1 )) == 12 ] ; then
           yr__=$(( $yr__ + 1 )) ; mo_=$yr__$mo__
        else
           mo_=$(( $mo_ + 1 ))
        fi
    done
    echo $liste
    
}

#----------------------------------------------------------------------
increment_month() {
#----------------------------------------------------------------------
  local year=$( echo $1 | cut -c 1-4 )
  local month=$(( $1 - ${year}00 ))
  if [ $(( $month - 12 )) -eq 0 ]; then
    year=$((year + 1))
    month=1
  else
    month=$((month + 1))
  fi
  if [[ $month -le 9 ]] ; then month=0$month ; fi
  echo $year$month
}

#----------------------------------------------------------------------
function fetch_simu_init_files() {
#----------------------------------------------------------------------
  #####################################################################
  echo "Recuperation eventuelle de certains fichiers sur http:lmdz.lmd.jussieu.fr/pub"
  #####################################################################

  #  ORCHIDEE input files
  get_input_files wget_pub Orchidee

  # Files for aerosols and chemistry
  if [[ $aerosols = "clim" ]]; then
       get_input_files wget_pub AerChem ; fi

  # Files for SPLA
  if [[ $aerosols = "spla" ]]; then
       get_input_files wget_pub SPLA_WA/emissions
       mo_=$mthini
       while [[ $mo_ != $mthend ]] ; do
       # Code horrible duplique a cause de 08 (FH)
	   local yr__=$( echo $mo_ | cut -c 1-4 )
           local mo__=$(( $mo_ - ${yr__}00 ))
	   if [[ $mo__ -le 9 ]] ; then mo__=0$mo__ ; fi
           for var in u10m v10m u v ; do
               echo wget_pub 3DInputData/SPLA_WA/ERA5/$yr__/$mo__ $var.nc
               wget_pub 3DInputData/SPLA_WA/ERA5/$yr__/$mo__ $var.nc
           done
           mo_=$( increment_month $mo_ )
       done
  fi
}

#----------------------------------------------------------------------
function run_sim_or_init() {
#----------------------------------------------------------------------
  cd "$local"

  if [[ $init = 1 ]]; then
    #####################################################################
    echo "Creation de l etat initial"
    #####################################################################

    # Creation du repertoire INIT et mise en place de liens logiques vers les starts
    # en anticipation de leur création :
    mkdir "$local/$INIT"; cd "$local/$INIT"
    for an in $mthini $yearini; do for f in start startphy; do ln -s "$f.nc" "$f.$an.nc"; done; done

    # Creation du repertoire INIT temporaire et rapatriement des fichiers necessaires
    if [[ -d $SIMRUNDIR/$INIT ]]; then mv "$SIMRUNDIR/$INIT" "$SIMRUNDIR/$INIT$$"; fi
    mkdir "$SIMRUNDIR/$INIT"; cp -r "$local/DEF" "$SIMRUNDIR/$INIT/"
    cd "$SIMRUNDIR/$INIT"; cp DEF/*.def .; cp "$local/lmdz_env.sh" .
    if [[ $xios = "y" ]]; then
      cp DEF/XMLfilesLMDZ/*xml .
      if [[ $veget != 'none' ]]; then cp DEF/XMLfilesOR$veget/*xml .; fi
    fi
    sed -e "s/anneeref=.*.$/anneeref=$yearini/" DEF/run.def >| run.def

    #-------------------------------------------------------------------
    # Fichiers Limit
    #-------------------------------------------------------------------
    local yrs suf
    if [[ $climato = 0 ]]; then
      # calend est choisi plus haut dans "Changements dans les fichiers DEF/*def" et ecrit dans $MAINDIR/DEF/run.def
      yrini=$(echo "$mthini" | cut -c-4)
      yrend=$(echo "$mthend" | cut -c-4)
      yrs=""; yr=$yrini
      while [[ $yr -le $yrend ]]; do yrs="$yrs $yr"; (( yr = yr + 1 )); done
      suf="360x180_"
    else
      yrs=2000
      suf="1x1_clim"
    fi

    liste_get="Albedo.nc Relief.nc Rugos.nc landiceref.nc"
    for yr in $yrs; do
       if [[ $climato = 0 ]]; then sufyr=$suf$yr; else sufyr=$suf; fi
       liste_get="$liste_get amipbc_sic_$sufyr.nc amipbc_sst_$sufyr.nc"
    done
    echo LISTEGET "$liste_get"
    for file in $liste_get; do
       wget_pub 3DInputData/Limit $file
       ln_from_pub  3DInputData/Limit $file
    done
    #-------------------------------------------------------------------
    # Initial state ECDYN
    #-------------------------------------------------------------------
    wget_pub 3DInputData/Init ECDYN.nc
    ln_from_pub 3DInputData/Init ECDYN.nc
    ln -sf ECDYN.nc ECPHY.nc

    # Creation du script d'initialisation
    cat << ...eod >| tmp
#!/bin/bash
#@JZ#JeanZay
#@JZ#SBATCH --job-name=Init         # nom du job
#@JZ#SBATCH --ntasks=1              # Nombre de processus MPI
#@JZ#SBATCH --cpus-per-task=1      # nombre de threads OpenMP
#@JZ# /!\ Attention, la ligne suivante est trompeuse mais dans le vocabulaire
#@JZ# de Slurm "multithread" fait bien référence à l'hyperthreading.
#@JZ#SBATCH --hint=nomultithread    # 1 thread par coeur physique (pas d'hyperthreading)
#@JZ#SBATCH --time=00:10:00         # Temps d'exécution maximum demandé (HH:MM:SS)
#@JZ#SBATCH --output=Init%j.out     # Nom du fichier de sortie
#@JZ#SBATCH --error=Init%j.out      # Nom du fichier d'erreur (ici commun avec la sortie)
#@JZ# To submit to dev queue; "time" (above) must be max 2h
#@JZ#TESTQ#SBATCH --qos=qos_cpu-dev
#@SP#Spirit
#@SP#SBATCH --job-name=Init
#@SP#SBATCH --ntasks=1
#@SP#SBATCH --cpus-per-task=1
#@SP#SBATCH --hint=nomultithread
#@SP#SBATCH --time=00:10:00
#@SP#SBATCH --output=Init%j.out
#@SP#SBATCH --error=Init%j.out
#@ADS#Adastra
#@ADS#SBATCH --job-name=Init
#@ADS#SBATCH --ntasks=1
#@ADS#SBATCH --cpus-per-task=1
#@ADS#SBATCH --nodes=1
#@ADS#SBATCH --hint=nomultithread
#@ADS#SBATCH --time=00:10:00
#@ADS#SBATCH --output=Init%j.out
#@ADS#SBATCH --error=Init%j.out

set -eu

# ANCIEN MULTI STEP  case \${LOADL_STEP_NAME} in

# ANCIEN MULTI STEP   init )

if [ ! -f lmdz_env.sh ]; then echo "manque fichier lmdz_env.sh"; ls; exit 1; fi
. lmdz_env.sh
ulimit -s unlimited
cd $SIMRUNDIR/$INIT
echo "Executable : $ce0l"
for yr in $yrs; do
  if [ $climato = 0 ]; then sufyr=$suf\$yr; else sufyr=$suf; fi
  ln -sf amipbc_sic_\$sufyr.nc amipbc_sic_1x1.nc
  ln -sf amipbc_sst_\$sufyr.nc amipbc_sst_1x1.nc
  sed -e 's/anneeref=.*.$/anneeref='\$yr'/' DEF/run.def >| run.def
  echo Starting initialisation
  # Runing ce0l.e
  OMP_NUM_THREADS=1 $MPICMD 1 $ce0l  # ce0l requires MPI=OMP=1
  if [ $climato = 0 ]; then mv limit.nc limit.\$yr.nc; fi
done
# ANCIEN MULTI STEP   ;;

# ANCIEN MULTI STEP   interp )
if [[ $aerosols = clim ]]; then
  cp $local/interp_aerosols.sh .; chmod +x interp_aerosols.sh
  # Les aerosols de l'annee 2000 ont été remplacés par "9999" qui pointe vers un fichier moyen sur 1995-2014
  #for year in 2000 1850; do  ./interp_aerosols.sh \$year; done
  #mv aerosols.2000.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
  for year in 9999 1850; do ./interp_aerosols.sh \$year; done
  mv aerosols.9999.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
fi

for f in sta* gri*nc; do cp \$f $local/$INIT/\$f; done
if [[ $climato = 1 ]]; then cp limit.nc $local/$INIT/limit.nc; fi
mkdir -p $local/$LIMIT
for f in limit*.nc ; do cp \$f $local/$LIMIT/\$f; done
if [ $aerosols = clim ]; then  for f in aerosols[.0-9]*nc; do cp \$f $local/$LIMIT/\$f; done; fi
#
cd $SIMRUNDIR
...eod
    if [[ $ok_guide != "y" ]]; then # Running first simulation automatically except for nudging
      cat << ...eod >> tmp
      echo "Submitting job tmp_$SIM"
      #echo "\$SUBMITCMD tmp_$SIM"
      #\$SUBMITCMD tmp_$SIM
      echo submitcmd tmp_$SIM
      submitcmd tmp_$SIM
...eod
    fi
    cat << ...eod >> tmp
  # ANCIEN MULTI STEP   esac
...eod
    enable_platform tmp
    echo "#################################################################"
    #echo "Submitting initialisation job <$SUBMITCMD tmp> from $(pwd)"
    echo "Submitting initialisation job <submitcmd tmp> from $(pwd)"
    chmod +x tmp
    #$SUBMITCMD tmp
    submitcmd tmp
    echo "#################################################################"

  else #case [ $init != 1 ]
     cd "$SIMRUNDIR"
     echo "################################################################"
     echo "Submitting job tmp_$SIM"
     #echo "$SUBMITCMD tmp_$SIM"
     #$SUBMITCMD "tmp_$SIM"
     echo "submitcmd tmp_$SIM"
     submitcmd "tmp_$SIM"
     echo '################################################################'
  fi
}

#---------------------------------------------------------------------------
function message_post_submit() {
#---------------------------------------------------------------------------
  if [[ $ok_guide = "y" && $init = 1 ]]; then
    cd "$local"
    enable_platform era2gcm_tuto.sh
    echo "Once initialisation is finished, you have to create nudging files"
    echo "Edit era2gcm_tuto.sh and set the desired parameters in section <User choices>"
    echo "Make sure you have acces to the chosen ERA files, and the required modules are loaded, then run : ./era2gcm_tuto.sh"
    if [[ $aerosols = "spla" ]]; then
      echo "Your aerosol choice is <spla>, so you need ERA 10m-winds interpolated on LMDZ grid. Use script era2gcm_uv10m.sh"
    fi
  else
    echo "Si tout se passe bien, vous avez initialisé et lancé automatiquement la simulation."
    echo "Le job qui a été lancé se trouve sur $SIMRUNTOPDIR/tmp_$SIM"
  fi
}

#---------------------------------------------------------------------------
function setup_and_load_lmdz_env() {
#---------------------------------------------------------------------------
  if [[ ! -f .lmdz_setup_root_dir ]]; then echo "STOP: setup.sh is not located in the root dir ??!!"; exit 1; fi
  # sed root_dir in lmdz_env.sh
  sed -i'' "s<root_dir=.*<root_dir=$local<" lmdz_env.sh

  # Set up the appropriate environment
  source lmdz_env.sh
}

local=$(pwd)

setup_and_load_lmdz_env
load_install_lib
define_expert_options
set_default_params
read_cmdline_args "$@"
ensure_correct_option_combinations
install_model
setup_def
setup_ce0l
setup_simu
fetch_simu_init_files
run_sim_or_init
message_post_submit
