#!/bin/bash
# This script sets up and launches a (series of) simulation(s).
# RECOMMENDATION: use main.sh to drive it (do not run it directly)

set -eu

# EXPERT-LEVEL CHOICES, only available in setup.sh, not in main.sh :
####################################################################
function define_expert_options {
  #NOTE : $optim  is part of $LMDZname (see definition of LMDZname below),
  #   so that running with "-debug" needs to (re)install the model (install=-install in main.sh)
  #   If you have already installed the model, and only want to recompile the gcm in debug mode,
  #   add "-debug" to the compilation options in the compile.sh script created in the present folder, and run it (./compile.sh gcm)
  #   ALTERNATIVELY : you can remove $optim in the definition of "LMDZname" below
  optim=""
  #optim="-debug"

  #NOTE : "testmode=y" is for running in test mode :
  # the simulation job "tmp_$SIM" (created on $SCRATCHDIR) is automatically changed for a short run,
  # and submitted to the Jean-Zay "test" queue :
  #  a/ time=00:10:00  (run time limit = 10 min; for test queue, it must be 30 min maximum)
  #  b/ nday=1 (this line, forcing nday to 1 day, is "hidden" in script_SIMU, by default it is commented out)
  #  c/ #SBATCH --qos=qos_cpu-dev (this line is in script_SIMU, by default it is commented out)
  #If you want to change "time", or "nday", but still run in "test" mode, modify the "if [ $debug = 1 ]...; fi" further below.
  testmode=n

  # Radiative code : "oldrad" or "rrtm" or "ecrad"
  rad="rrtm"

  # AEROSOLS : n (=no) / clim (=average 1995-2014) / spla (interactive dust and sea salt)
  # (WARNING : if you first run the scripts with aerosols=n, then you want to change to =clim ,
  #   you'll need to remove the INIT and LIMIT folders that have been created, then run main.sh with init=1
  #   in order to re-run the initialisation job, which downloads the aerosol files and interpolates them)
  aerosols="clim"

  # SURFACE/VEGETATION SCHEME
  # It is controlled by the single variable "veget" which can have the following values
  # - "none": bucket scheme
  # - "CMIP6" or "orch2.0": orchidee version used in CMIP exercise, rev 5661
  # - "orch2.2": orchidee version used in IPSLCM7 configuration
  # - number: orchidee version number : only rev 7983 on branch _2_2, and 7994 on trunk, are available
  veget="CMIP6"
  #AS : If you have installed the model with a given "veget" option, and you want to change it :
  #   --> RECOMMENDED : re-install the model from scratch in a new TEST_PROD folder
  #   TODO Amaury: check w/ Adriana what the use case is for lmdz_setup here
  #   --> EXPERT : If you want to keep your previous installation, and not go through the install_lmdz.sh routine,
  #    you will need to (a) install the proper version of ORCHIDEE in the modipsl/modeles directory, and
  #                     (b) set the "veget" options (opt_veget, orchidee_rev, suforch) to their proper values
  #                        in the file $LMDZD/modipsl/modeles/surface_env
  # (NB : no need to initialise these vars here:
  #     surface_env is either (re)created when (re)installing the model (run with install=-install),
  #     or it already exists (install="").

  # New snow scheme INLANDSIS (y/n)
  # (this flag activates INLANDSIS compilation;
  # not yet done : treatment of specific restart and def file)
  inlandsis="n"

  # --->>> ALSO PAY ATTENTION TO OUTPUT files, frequency, level -------------
  #   With IOIPSL : Choose your config.def among the versions available in DEF,
  #     copy it as config.def (the copy is done automatically for "iso"),
  #     edit it and set up output files, frequencies and levels.
  #     NB : For aerosols=spla, output level minimum 4 is required to output the specific variables.
  #          For aerosols=n, the corresponding flags will automatically be set to "n".
  #   With XIOS : adjust DEF/XMLfiles*/file*xml
}

# /!\ DO NOT EDIT BELOW UNLESS YOU KNOW WHAT YOU ARE DOING /!\

function enable_groupe {  # In job scripts, sed platform-specific headers
  local file="$1"
  local platform

  case ${hostname:0:5} in
    jean-) platform="JZ";;
    spiri) platform="SP";;
    adast) platform="ADS";;
    *) echo "Warning: $hostname is not a known job platform (ignore if running locally)"; return 0;;
  esac

  sed -i'' -e "s/^#@$platform//" "$file"
}

function load_install_lib {
  # Fetch and source install_lmdz.sh to get `myget`
  if [[ ! -f "install_lmdz.sh" ]]; then
    wget "http://svn.lmd.jussieu.fr/LMDZ/BOL/script_install_amaury/install_lmdz.sh" # TODO change to normal branch once ready
  fi
  # shellcheck disable=SC1090
  source <(sed 's/function \(.*\) {/function installlmdz_\1 {/g' install_lmdz.sh)  # source with a namespace for functions
  function myget { installlmdz_myget "$@"; }
}

function set_default_params {
  # Default value of script parameters
  SIM=$(basename "$local")CTL  # name
  phylmd="lmd" #option -p $phylmd for makelmdz

  cosp="n"  # COSP
  xios="n"  #XIOS

  # Nudging :
  ok_guide="n"
  # With nudging, use real calendar (climato=0) and monthly integrations
  climato=1
  freq="mo"  # frequence mensuelle mo ou annuelle yr

  # NB : the run stops in the BEGINNING of mthend (test "next=stopsim")
  mthini=200001
  mthend=200501
  resol="144x142x79"

  version="20230412.trunk"
  svn=""

  init=1

  LIMIT="LIMIT"

  ######################################################################
  # Choix du nombre de processeurs
  # NOTES :
  # omp=8 by default (for Jean-Zay must be a divisor of 40 procs/node), but we need
  #   omp=1 for SPLA (only MPI parallelisation)
  #   omp=2 for veget=CMIP6 beacause of a bug in ORCHIDEE/src_xml/xios_orchidee.f90
  ######################################################################
  jm=$(echo "$resol" | cut -dx -f2)
  (( mpi = ( jm + 1 ) / 2 ))
  omp=8
  if [[ $aerosols = "spla" ]]; then omp=1; fi
  if [[ $veget = "CMIP6" && $xios = "y" ]]; then omp=2; fi
  if [[ $mpi -gt $NB_MPI_MAX ]]; then mpi=$NB_MPI_MAX; fi
  if [[ $omp -gt $NB_OMP_MAX ]]; then omp=$NB_OMP_MAX; fi

  case $rad in
    oldrad) iflag_rrtm=0; NSW=2;;
    rrtm)   iflag_rrtm=1; NSW=6;;
    ecrad)  iflag_rrtm=2; NSW=6;;
  esac
}

function read_cmdline_args {
  while (($# > 0)); do
    case $1 in
      "-h") cat <<........fin
       setup.sh can be launched/driven by main.sh; some options can only be specified in setup.sh (ex: veget, aerosols).
             setup.sh [-v version] [-r svn_release] [-init INIT] [-d 96x95x79] [-f mo] [-nudging]
             -v "version" like 20150828.trunk; see https://lmdz.lmd.jussieu.fr/Distrib/LISMOI.trunk (default <$version>)
             -r "svn_release"      either the svn release number or "last" (default <$svn>)
             -d IMxJMxLM        to run in resolution IM x JM x LM (default <$resol>)
             -install           pour installer et compiler le modele
             -f mo/yr           pour tourner en mensuel ou annuel (default <$freq>)
             "INIT"             1: creates INIT and LIMIT
                                0: reads from INIT and LIMIT
                                SIMU: reads from preexisting simulation SIMU and LIMIT (default <$init>)
             -nudging           to run with nudging. Nudging files must be created independently
             -p                 the physics to use (default <$phylmd>)
             -name              install folder name (default <$SIM>)
             Other options available (see "options" section in the script)
........fin
        exit;;
      "-v") version="$2"; shift; shift;;
      "-r") svn=$2; shift; shift;;
      "-d") resol=$2; shift; shift;;
      "-f") freq=$2; shift; shift;;
      "-p") phylmd=$2; shift; shift;;
      "-name") SIM=$2; shift; shift;;
      "-cosp") cosp=y; shift;;
      "-xios") xios=y; shift;;
      "-init") init=$2; shift; shift;;
      "-nudging") ok_guide=y; shift;;
      "-climato") climato=$2; shift; shift;;
      "-mthini") mthini=$2; shift; shift;;
      "-mthend") mthend=$2; shift; shift;;
      *) echo "unexpected $1"; $0 -h; exit
    esac
  done

  # Initialisation
  if [[ $init = 1 || $init = 0 ]]; then
    INIT="INIT"
  else
    INIT=$init
  fi

  yearini=$(echo "$mthini" | cut -c-4)
  if [[ $freq = yr ]]; then stopsim=$(echo "$mthend" | cut -c-4); else stopsim=$mthend; fi

  if [[ -d $SIM ]]; then
     echo "La simulation $SIM existe deja. Il est conseillé d'arrêter et de vérifier."
     echo "Si vous êtes sûr de vous, vous pouvez la prolonger. Voulez vous la prolonger ? (y/n)"
     read -r ans
     if [[ $ans != y ]]; then exit 1; fi
  fi
}

function ensure_correct_option_combinations {
  # AVOID COMBINATIONS OF OPTIONS THAT DON'T WORK in user choices
  if [[ $ok_guide = y && $climato = 1 ]]; then
     echo "STOP: Running nudged simulations with climatological SSTs is not planned. Change <climato> to <0> or modify the setup (experts)"; exit 1
  fi

  if [[ $climato = 0 && $freq = yr ]]; then
     echo "STOP: Running simulations with interannual SSTs is possible only month by month and a true calendar."
     echo "Change <climato> to <1> or <freq> to <mo> or modify setup.sh (experts)"; exit 1
  fi


  # (Temporary) Constraints for aerosols=spla :
  # --> resolution 128x88x79 and rad=rrtm
  if [[ $aerosols = 1 && $resol != "128x88x79" ]]; then
    echo 'STOP: For now, <aerosols=spla> requires <resol=128x88x79>, and uses the zoomed grid from gcm.def_zNAfrica_BiJe, for which forcing & initial files are available'
    echo "Right now resol=<$resol>"
    exit 1
  fi
}

function install_model {
  mkdir -p "$LMDZD"

  local ins_xios ins_cosp ins_aero ins_inlandsis
  if [[ $xios = "y" ]]; then ins_xios="-xios"; else ins_xios=""; fi
  if [[ $cosp = "y" ]]; then ins_cosp="-cosp v1"; else ins_cosp=""; fi
  if [[ $aerosols = "spla" ]]; then ins_aero="-spla"; else ins_aero=""; fi
  if [[ $inlandsis = "y" ]]; then ins_inlandsis="-inlandsis"; else ins_inlandsis=""; fi
  if [[ -n $svn ]]; then svnopt="-r $svn"; else svnopt=""; fi

  version_name=LMDZ$(echo "$version" | sed -e 's/-v//g' -e 's/-unstable//' -e 's/-r/r/' -e 's/ //g')
  LMDZname="${version_name}${svn}OR$veget${ins_xios}"
  MODEL="$LMDZD/$LMDZname/modipsl/modeles/LMDZ"

  if [[ -d $MODEL ]]; then echo "Found existing install at MODEL=$MODEL"; fi
  echo "Installing model"
  cd "$LMDZD"
  cp "$local/install_lmdz.sh" .
  chmod +x install_lmdz.sh
  echo "./install_lmdz.sh -noclean $optim -v $version $svnopt -d $resol -rad $rad -bench 0 -parallel mpi_omp $ins_cosp $ins_xios $ins_aero $ins_inlandsis -name $LMDZname -veget $veget -netcdf 0 -arch $ARCH" >> install_lmdz_options.$$.sh
  chmod +x install_lmdz_options.$$.sh
  echo "Running install_lmdz_options.$$.sh"
  set -o pipefail
  gcm=$MODEL/$(./install_lmdz_options.$$.sh | tee /dev/tty | tail -n 1 | sed -n "s:.* executable is \(.*\.e\).*:\1:p")
  set +o pipefail
  mv install_lmdz.sh install_lmdz.$$.sh
  cd "$local"
}

function setup_def {  # modify various .def in ./DEF (+ xios xml as needed)
  cd "$local"

  # Utilisation des .def_iso pour LMDZ-ISOtopes
  if [[ $phylmd = "lmdiso" ]]; then
    for file_iso in $(ls DEF | grep _iso); do
      cp DEF/"$file_iso" DEF/"${file_iso%%_iso}"
    done
  fi

  ######################################################################
  # Choix de la grille verticale
  ######################################################################
  lm=$(echo "$resol" | cut -dx -f3)
  if [ ! -f "DEF/L$lm.def" ]; then
    echo "STOP: Résolution verticale non prévue - créer un fichier DEF/L$lm.def"; exit 1
  else
    sed -i'' -e "s/INCLUDEDEF=L.*.def/INCLUDEDEF=L$lm.def/" DEF/run.def
  fi

  ######################################################################
  # Changements dans les fichiers DEF/*def
  # (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $SIMRUNDIR)
  ######################################################################

  # Choix du fichier traceur.def coherent avec l'option "aerosols"
  #  NOTE : Le nouveau tracer.def_nospla par defaut n'inclut pas Rn-Pb;
  #		si on les veut, il faut utiliser ci-dessous; a la place, tracer_RN_PB.def
  #---------------------------------------------------------------------
  # NB: Si on change de traceur.def (entre spla et nospla), il faut refaire l'etape "initialisation" (ce0l)?
  # Normalement les traceurs absents de start* files sont initialises=0 dans le code; verifier pour spla !
  if [[ $aerosols = "spla" ]]; then
    # ancien traceur pour l instant
    cp DEF/traceur.def_spla DEF/traceur.def
  elif [[ $phylmd = "lmdiso" ]]; then
    # nouveau traceur , déjà copié si 'y'
    cp DEF/tracer.def_nospla DEF/tracer.def
  fi

  # TEMPORAIREMENT pour spla on force l'utilisation de gcm.def_zNAfrica_BiJe (avec resolution 128x88x79)
  #----------------------------------------------------------------------
  if [[ $aerosols = spla ]]; then cp DEF/gcm.def_zNAfrica_BiJe DEF/gcm.def; fi

  # Inscription du choix ok_guide dans DEF/guide.def
  #---------------------------------------------------------------------
  sed -i'' -e 's/ok_guide=.*.$/ok_guide='$ok_guide'/' DEF/guide.def

  # Inscription du type de calendrier (qui est fonction de $climato) dans DEF/run.def
  #---------------------------------------------------------------------
  # NB Contrairement a ce qui est ecrit dans les fichiers run.def standard,
  # dans ce tutorial le choix earth_365d n'est pas disponible, et earth_366d s'appelle gregorian
  if [[ $climato = 0 ]]; then calend="gregorian"; else calend="earth_360d"; fi
  sed -i'' -e 's/calend=.*.$/calend='$calend'/' DEF/run.def

  # Changements dans config.def (pre-choisi, et regle pour output si utilisation avec IOIPSL)
  #   cf options veget, aerosols, cosp, xios
  #---------------------------------------------------------------------
  if [[ $veget = none ]]; then  VEGET="n"; else VEGET="y"; fi
  sed -i'' -e 's/VEGET=.*.$/VEGET='$VEGET'/' DEF/config.def

  if [[ $aerosols = "n" ]]; then
    # set flag_aerosols=0 and flags ok_ade&co=n
    sed -i'' -e 's/^ok_cdnc=.*.$/ok_cdnc=n/' -e 's/^ok_ade=.*.$/ok_ade=n/' -e 's/^ok_aie=.*.$/ok_aie=n/' -e 's/^ok_alw=.*.$/ok_alw=n/' -e 's/^flag_aerosol=.*.$/flag_aerosol=0/' DEF/config.def
  fi

  # COSP : ok_cosp desactive COSP si on a compile avec; il ne l'active pas si on a compile sans
  sed -i'' -e 's/ok_cosp.*.$/ok_cosp='$cosp'/' DEF/config.def
  if [[ $cosp = "y" ]]; then \cp -f "$MODEL"/DefLists/cosp*.txt "$local"/DEF/; fi

  # Sorties LMDZ en fonction de l'option "xios"
  sed -i'' -e 's/ok_all_xml=.*.$/ok_all_xml='$xios'/' DEF/config.def

  # Ajuster physiq.def en fonction de radiative code (default: values for rad=rrtm)
  #   Pour isotopes=y , mettre iflag_ice_thermo=0 au lieu de 1
  #---------------------------------------------------------------------
  sed -i'' -e 's/iflag_rrtm=.*.$/iflag_rrtm='"$iflag_rrtm"'/' -e 's/NSW=.*.$/NSW='"$NSW"'/' DEF/physiq.def
  sed -i'' -e 's:directory_name.*$:directory_name="'"$local"'/ecrad_data",:' DEF/namelist_ecrad

  if [[ $phylmd = "lmdiso" ]]; then iflag_ice_thermo=0; else iflag_ice_thermo=1; fi
  sed -i -e 's/iflag_ice_thermo=.*.$/iflag_ice_thermo='$iflag_ice_thermo'/' DEF/physiq.def

  # Choix de orchidee.def en fonction de orchidee_rev; modification pour xios
  #  NOTE separate orchidee_pft.def file for ORCHIDEE trunk post-CMIP6
  #---------------------------------------------------------------------
  orchidee_def=orchidee.def_6.1
  orchidee_pft_def=""
  if [[ $veget = "7983" ]]; then
    orchidee_def=orchidee.def_6.2work
  elif [[ $veget = "7994" ]]; then
    orchidee_def=orchidee.def_6.4work
    orchidee_pft_def=orchidee_pft.def_6.4work
    if ! grep "INCLUDEDEF=orchidee_pft.def" DEF/run.def; then
      sed -i'' -e 's/INCLUDEDEF=orchidee.def/INCLUDEDEF=orchidee.def\nINCLUDEDEF=orchidee_pft.def/' DEF/run.def; fi
  fi
  cp -f DEF/$orchidee_def DEF/orchidee.def
  if [[ $orchidee_pft_def != "" ]]; then cp -f DEF/$orchidee_pft_def DEF/orchidee_pft.def; fi

  # Only for veget=CMIP6 it is still possible to use IOIPSL; newer versions of orchidee.def have XIOS_ORCHIDEE_OK = y
  sed -i'' -e 's/XIOS_ORCHIDEE_OK =.*.$/XIOS_ORCHIDEE_OK = '$xios'/' DEF/orchidee.def

  ######################################################################
  # Si on tourne avec XIOS, mise a jour des fichiers context et field* dans XMLfilesLMDZ
  # (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $SIMRUNDIR)
  ######################################################################
  if [[ $xios = y ]]; then
    cp -f "$MODEL"/DefLists/context_lmdz.xml "$local"/DEF/XMLfilesLMDZ/.
    cp -f "$MODEL"/DefLists/field_def_lmdz.xml "$local"/DEF/XMLfilesLMDZ/.
    if [[ $cosp = y ]]; then cp -f "$MODEL"/DefLists/field_def_cosp1.xml "$local"/DEF/XMLfilesLMDZ/.; fi
  fi
}

function setup_ce0l { # Verification de l'existance de l'état initial, compilation eventuelle pour sa creation
  if [[ ! -d $INIT ]]; then
    if [[ $init = 0 ]]; then
      echo "STOP: Récuperer les repertoires $INIT ou lancer avec option -init"; exit 1
    else
      # Compile ce0l
      cd "$MODEL"
      sed -e 's/gcm$/ce0l/' compile.sh > compile_ce0l.sh; chmod +x compile_ce0l.sh
      echo "Compiling ce0l"
      if ! ./compile_ce0l.sh &> ce0l.log; then echo "STOP: ce0l compilation failed, see $MODEL/ce0l.log"; exit 1; fi
      echo "Compiled ce0l"
      ce0l=${gcm/gcm/ce0l}

      cd "$local"
    fi
  elif [[ $init = 1 ]]; then
    echo "STOP: Vous essayez d initialiser le modele mais $INIT existe deja"; exit 1
  fi
}

function setup_simu {
  SIMRUNTOPDIR="$SIMRUNBASEDIR/$(basename "$local")"
  SIMRUNDIR=$SIMRUNTOPDIR
  mkdir -p "$SIMRUNDIR"
  cd "$SIMRUNDIR"
  echo "La simulation s'exécutera sur $SIMRUNDIR"

  #####################################################################
  # Creation du repertoire $SIM s'il n'existe pas deja
  #####################################################################
  if [[ ! -d $local/$SIM ]]; then
    mkdir "$local/$SIM"
    cd "$local"

    # Edit reb.sh
    cp reb.sh "$local/$SIM/reb.sh"; chmod +x "$local/$SIM/reb.sh"
    sed -i'' -e "s:^rebuild=.*.$:rebuild=$LMDZD/$LMDZname/modipsl/bin/rebuild:" "$local/$SIM/reb.sh"
    enable_groupe "$local/$SIM/reb.sh"

    # Copy .def
    cp lmdz_env.sh "$local/$SIM/"
    mkdir "$local/$SIM/DEF"; cp DEF/*def DEF/namelis* "$local/$SIM/DEF/"
    #Pour XIOS, respectivement COSP, copier aussi les fichiers *xml / *txt
    if [[ $cosp = "y" ]]; then cp DEF/cosp*txt "$local/$SIM/DEF/"; fi
    if [[ $xios = "y" ]]; then
       cp DEF/XMLfilesLMDZ/*xml "$local/$SIM/DEF/"
       if [[ $veget != 'none' ]]; then cp DEF/XMLfilesOR$veget/*xml "$local/$SIM/DEF/"; fi
    fi
    chmod u+w "$local/$SIM"/DEF/*

    # Gestion du calendrier
    #######################
    cd "$SIM"
    sed -i'' -e "s/anneeref=.*.$/anneeref=$yearini/" DEF/run.def
    if [[ $freq = "yr" ]]; then date=$yearini; else date=$mthini; fi
    echo "$date a faire" >| etat

    # Recuperation des fichiers : executable initiaux et forcages
    #############################################################
    echo "date: $date"
    for f in start startphy; do
      inf=../$INIT/$f.$date.nc
      if [[ -f $inf || $init = 1 ]]; then ln -s "$inf" ./; else echo "STOP: $inf missing"; exit ; fi;
    done
    for f in sechiba stomate; do
      if [[ -f ../$INIT/start_$f.$date.nc ]]; then ln -sf "../$INIT/start_$f.$date.nc" .; fi
    done
    cp "$gcm" gcm.e
  fi
  cd "$local"/..

  #####################################################################
  echo "Modification du script de lancement"
  #####################################################################
  local cput
  if [[ $freq = "yr" ]]; then cput="04:00:00"; else cput="01:00:00"; fi
  local isotopes="n"
  if [[ $phylmd = "lmdiso" ]]; then isotopes="y"; fi
  sed -e "s/NOM_SIMU/$SIM/" \
  -e "s/time=.*.$/time=$cput/" \
  -e "s/ntasks=.*.$/ntasks=$mpi/" \
  -e "s/cpus-per-task=.*.$/cpus-per-task=$omp/" \
  -e "s/nthreads=.*./nthreads=$omp/" \
  -e "s/MAINDIR=.*.$/MAINDIR=$(basename "$local")/" \
  -e "s:STORED=.*.*:STORED=$(dirname $local):" \
  -e "s:SCRATCHD=.*.*:SCRATCHD=$SIMRUNBASEDIR:" \
  -e "s/stopsim=.*.$/stopsim=$stopsim/" \
  -e "s/^veget=.*.$/veget=$veget/" \
  -e "s/^aerosols=.*.$/aerosols=$aerosols/" \
  -e "s/^isotopes=.*.$/isotopes=$isotopes/" \
  -e "s/^climato=.*.$/climato=$climato/" \
  -e "s/^ok_guide=.*.$/ok_guide=$ok_guide/" \
  "$local/script_SIMU" >| "$SIMRUNDIR/tmp_$SIM"

  enable_groupe "$SIMRUNDIR/tmp_$SIM"

  if [[ $testmode = "y" ]]; then
    sed -i'' -e "s/time=.*.$/time=00:30:00/" -e "s/#nday=1/nday=1/" -e "s/#@TESTQ//" "$SIMRUNTOPDIR/tmp_$SIM"
  fi
}

function fetch_simu_init_files {
  #####################################################################
  echo "Recuperation eventuelle de certains fichiers sur $LMDZ_INIT"
  #####################################################################
  mkdir -p "$LMDZ_INIT"

  #-------------------------------------------------------------------
  # Fichiers ORCHIDEE
  #-------------------------------------------------------------------
  get="myget 3DInputData/Orchidee/"
  cd "$LMDZ_INIT";
  for file in "PFTmap_IPCC_2000.nc" "cartepente2d_15min.nc" "routing.nc" "routing_simple.nc" "lai2D.nc" "soils_param.nc" "woodharvest_2000.nc" "PFTmap_15PFT.v1_2000.nc"; do
    if [[ ! -f $file ]]; then ${get}$file; fi
  done
  cd - > /dev/null
  # Additionnal files needed for ORCHIDEE trunk post-CMIP6
  if [[ $veget = 7994 && ! -f $LMDZ_INIT/soil_bulk_and_ph.nc ]]; then
    cd "$LMDZ_INIT"
    ${get}soil_bulk_and_ph.nc; ${get}NITROGEN_for_ORtrunk.tar
    tar -xvf NITROGEN_for_ORtrunk.tar; cd - > /dev/null
  fi
  #-------------------------------------------------------------------
  # Fichiers aerosols/chimie
  #-------------------------------------------------------------------
  if [[ $aerosols = "clim" ]]; then
    get="myget 3DInputData/AerChem/"
    #liste_get="aerosols1850_from_inca.nc aerosols2000_from_inca.nc"
    #aerosols9999_from_inca.nc est un lien vers aerosols1995_2014_ensavg_from_inca.nc
    cd "$LMDZ_INIT"
    for file in "aerosols1850_from_inca.nc" "aerosols9999_from_inca.nc"; do
      if [[ ! -f $file ]]; then ${get}$file; fi
    done
    cd - > /dev/null
  fi
  # For SPLA
  #-------------------
  # Dans ${LMDZ_INIT} on cree folder SPLA_Init et dedans le INITIAL
  # Pour l'instant on copie là-dedans de chez Binta les fichiers a la res zoomNaf;
  # plus tard on y recupererea des fichiers a haute resolution reguliere depuis http:/LMDZ,
  # a regrider ensuite par un script interp_fichiers_spla.sh (comme pour aerosols="clim")
  # Les fichiers (regrides, sauf SOILSPEC.data utilise tel quel) seront mis dans $MAINDIR/INPUT_SPLA (equivalent de INPUT_DUST)
  if [[ $aerosols = "spla" ]]; then
    mkdir -p "${LMDZ_INIT}/SPLA_Init"; mkdir -p "$LMDZ_INIT/SPLA_Init/INITIAL"
    get="cp -p /gpfsstore/rech/gzi/rgzi027/ergon/BIBIAERO/INPUTS_DUST/INITIAL/"
    liste_get="wth.dat cly.dat donnees_lisa.nc SOILSPEC.data \
                 carbon_emissions.nc sulphur_emissions_antro.nc  \
                 sulphur_emissions_nat.nc  sulphur_emissions_volc.nc"

    cd "$LMDZ_INIT/SPLA_Init/INITIAL"
    for file in $liste_get; do
      if [[ ! -f $file ]]; then ${get}$file .; fi
    done
    cd - > /dev/null
    ###
    #Cas particulier des fichiers mensuels dust.nc :A DECIDER :
    #C'est entre le cas des aerosols.clim= 1 seul fichier, annuel,
    # et le cas des vents guidage, pré-interpolés avec era2gcm pour toute la periode, dans MAINDIR/GUIDE.
    # On pourrait (a)demander de precalculer dust.nc aussi, dans MAINDIR/INPUT_SPLA - avec un script à adapter de Jeronimo. Rien a mettre dans SPLA_Init alors.
    # Ou (b) fournir dans SPLA_Init les 12 mois d'un dust.nc climato (an 2006 pour nous ici) à la res EC, et faire juste le regrid vers MAINDIR/INPUT_SPLA
    #ICI pour l'instant je copie les fichiers de chez Binta (repositoire==http...) dans LMDZ_INIT/SPLA_Init, avec test sur mois 01

    if [[ ! -d ${LMDZ_INIT}/SPLA_Init/PERIOD0001 ]]; then
      if [[ ${hostname:0:5} != "jean-" ]]; then echo "PERIOD001 (aerosols=spla) IS ONLY AVAILABLE ON JEANZAY FOR NOW, CONTACT SUPPORT"; exit 1; fi
      cp -pr /gpfsstore/rech/gzi/rgzi027/ergon/BIBIAERO/INPUTS_DUST/PERIOD* "$LMDZ_INIT/SPLA_Init/.";
    fi
   #A la fin on doit avoir dans SPLA_Init les fichiers initiaux dans INITIAL plus les repertoires PERIOD00MM contenant dust.nc
   #Cela doit se retrouver dans script_SIMU qui les copie dans le repertoire de run sur $SCRATCH
  fi
  #-------------------------------------------------------------------
  # Fichiers Init
  #-------------------------------------------------------------------
  get="myget 3DInputData/Init/"
  liste_get="alb_bg_modisopt_2D_ESA_v2.nc reftemp.nc"
  cd "$LMDZ_INIT"
  for file in $liste_get; do
    if [[ ! -f $LMDZ_INIT/$file ]]; then ${get}$file; fi
  done
  cd - > /dev/null
}

function run_sim_or_init {
  cd "$local"

  if [[ $init = 1 ]]; then
    #####################################################################
    echo "Creation de l etat initial"
    #####################################################################

    # Creation du repertoire INIT et mise en place de liens logiques vers les starts
    # en anticipation de leur création :
    mkdir "$local/$INIT"; cd "$local/$INIT"
    for an in $mthini $yearini; do for f in start startphy; do ln -s "$f.nc" "$f.$an.nc"; done; done

    # Creation du repertoire INIT temporaire et rapatriement des fichiers necessaires
    if [[ -d $SIMRUNDIR/$INIT ]]; then mv "$SIMRUNDIR/$INIT" "$SIMRUNDIR/$INIT$$"; fi
    mkdir "$SIMRUNDIR/$INIT"; cp -r "$local/DEF" "$SIMRUNDIR/$INIT/"
    cd "$SIMRUNDIR/$INIT"; cp DEF/*.def .; cp "$local/lmdz_env.sh" .
    if [[ $xios = "y" ]]; then
      cp DEF/XMLfilesLMDZ/*xml .
      if [[ $veget != 'none' ]]; then cp DEF/XMLfilesOR$veget/*xml .; fi
    fi
    sed -e "s/anneeref=.*.$/anneeref=$yearini/" DEF/run.def >| run.def

    #-------------------------------------------------------------------
    # Fichiers Limit
    #-------------------------------------------------------------------
    local yrs suf
    if [[ $climato = 0 ]]; then
      # calend est choisi plus haut dans "Changements dans les fichiers DEF/*def" et ecrit dans $MAINDIR/DEF/run.def
      yrini=$(echo "$mthini" | cut -c-4)
      yrend=$(echo "$mthend" | cut -c-4)
      yrs=""; yr=$yrini
      while [[ $yr -le $yrend ]]; do yrs="$yrs $yr"; (( yr = yr + 1 )); done
      suf="360x180_"
    else
      yrs=2000
      suf="1x1_clim"
    fi

    get="myget 3DInputData/Limit/"
    liste_get="Albedo.nc Relief.nc Rugos.nc landiceref.nc"
    for yr in $yrs; do
       if [[ $climato = 0 ]]; then sufyr=$suf$yr; else sufyr=$suf; fi
       liste_get="$liste_get amipbc_sic_$sufyr.nc amipbc_sst_$sufyr.nc"
    done
    echo LISTEGET "$liste_get"
    for file in $liste_get; do
      cd "$LMDZ_INIT"
      if [[ ! -f $LMDZ_INIT/$file ]]; then ${get}$file; fi
      cd - > /dev/null
      ln -s "$LMDZ_INIT/$file" .
    done
    #-------------------------------------------------------------------
    # ECDYN
    #-------------------------------------------------------------------
    get="myget 3DInputData/Init/"
    if [[ ! -f $LMDZ_INIT/ECDYN.nc ]]; then cd "$LMDZ_INIT"; ${get}ECDYN.nc; cd - > /dev/null; fi
    ln -s "$LMDZ_INIT"/ECDYN.nc .
    ln -sf ECDYN.nc ECPHY.nc

    # Creation du script d'initialisation
    cat << ...eod >| tmp
#!/bin/bash
#@JZ#JeanZay
#@JZ#SBATCH --job-name=Init         # nom du job
#@JZ#SBATCH --ntasks=1              # Nombre de processus MPI
#@JZ#SBATCH --cpus-per-task=16      # nombre de threads OpenMP
#@JZ# /!\ Attention, la ligne suivante est trompeuse mais dans le vocabulaire
#@JZ# de Slurm "multithread" fait bien référence à l'hyperthreading.
#@JZ#SBATCH --hint=nomultithread    # 1 thread par coeur physique (pas d'hyperthreading)
#@JZ#SBATCH --time=00:10:00         # Temps d'exécution maximum demandé (HH:MM:SS)
#@JZ#SBATCH --output=Init%j.out     # Nom du fichier de sortie
#@JZ#SBATCH --error=Init%j.out      # Nom du fichier d'erreur (ici commun avec la sortie)
#@JZ# To submit to dev queue; "time" (above) must be max 2h
#@JZ# #SBATCH --qos=qos_cpu-dev
#@SP#Spirit
#@SP#SBATCH --job-name=Init
#@SP#SBATCH --ntasks=1
#@SP#SBATCH --cpus-per-task=16
#@SP#SBATCH --hint=nomultithread
#@SP#SBATCH --time=00:10:00
#@SP#SBATCH --output=Init%j.out
#@SP#SBATCH --error=Init%j.out
#@ADS#Adastra
#@ADS#SBATCH --job-name=Init
#@ADS#SBATCH --ntasks=1
#@ADS#SBATCH --cpus-per-task=16
#@ADS#SBATCH --hint=nomultithread
#@ADS#SBATCH --time=00:10:00
#@ADS#SBATCH --output=Init%j.out
#@ADS#SBATCH --error=Init%j.out

# ANCIEN MULTI STEP  case \${LOADL_STEP_NAME} in

# ANCIEN MULTI STEP   init )

if [ ! -f lmdz_env.sh ]; then echo "manque fichier lmdz_env.sh"; ls; exit 1; fi
. lmdz_env.sh
ulimit -s unlimited
export OMP_STACKSIZE=800M
export OMP_NUM_THREADS=1
cd $SIMRUNDIR/$INIT
echo "Executable : $ce0l"
for yr in $yrs; do
  if [ $climato = 0 ]; then sufyr=$suf\$yr; else sufyr=$suf; fi
  ln -sf amipbc_sic_\$sufyr.nc amipbc_sic_1x1.nc
  ln -sf amipbc_sst_\$sufyr.nc amipbc_sst_1x1.nc
  sed -e 's/anneeref=.*.$/anneeref='\$yr'/' DEF/run.def >| run.def
  echo Starting initialisation
  $RUNCMD 1 $ce0l
  if [ $climato = 0 ]; then mv limit.nc limit.\$yr.nc; fi
done
# ANCIEN MULTI STEP   ;;

# ANCIEN MULTI STEP   interp )
if [[ $aerosols = clim ]]; then
  cp $local/interp_aerosols.sh .; chmod +x interp_aerosols.sh
  # Les aerosols de l'annee 2000 ont été remplacés par "9999" qui pointe vers un fichier moyen sur 1995-2014
  #for year in 2000 1850; do  ./interp_aerosols.sh \$year; done
  #mv aerosols.2000.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
  for year in 9999 1850; do ./interp_aerosols.sh \$year; done
  mv aerosols.9999.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
fi

# AS : S'il etait possible d'automatiser l'interpolation de l'input SPLA, ce serait a lancer ici
#en attendant, on passe au paragraphe suivant où on copie les fichiers à la res ZoomNaf depuis $LMDZ_INIT/SPLA_Init
#if [ $aerosols = spla ]; then
#cp $local/futur script interp_aerosols_SPLA.sh .; chmod +x interp_aerosols_SPLA.sh
#for file in...; do  ./interp_aerosols_SPLA.sh \$year; done
#etc etc etc ...
#fi

# Copy initial and forcing files in $local/$INIT and $local/$LIMIT; also in $local/INPUT_SPLA if $aerosols=spla
for f in sta* limit.nc gri*nc; do cp \$f $local/$INIT/\$f; done
mkdir -p $local/$LIMIT
for f in limit*.nc ; do cp \$f $local/$LIMIT/\$f; done
if [ $aerosols = clim ]; then  for f in aerosols[.0-9]*nc; do cp \$f $local/$LIMIT/\$f; done; fi
#
if [ $aerosols = spla ]; then
 #mkdir -p $local/INPUT_SPLA; pour l'instant on copie $LMDZ_INIT/SPLA_Init en block
 if [ ! -d $local/INPUT_SPLA ]; then cp -pr $LMDZ_INIT/SPLA_Init $local/INPUT_SPLA; fi
fi
cd $SIMRUNDIR
...eod
    if [[ $ok_guide != "y" ]]; then # Running first simulation automatically except for nudging
      cat << ...eod >> tmp
       $SUBMITCMD tmp_$SIM
...eod
    fi
    cat << ...eod >> tmp
  # ANCIEN MULTI STEP   esac
...eod
    enable_groupe tmp
    echo "###############################################################################"
    echo "Submitting initialisation job <$SUBMITCMD tmp> from $(pwd)"
    chmod +x tmp
    $SUBMITCMD tmp
    echo "###############################################################################"

  else #case [ $init != 1 ]
     cd "$SIMRUNDIR"
     echo "###############################################################################"
     echo "Submitting job tmp_$SIM"
     echo "$SUBMITCMD tmp_$SIM"
     $SUBMITCMD "tmp_$SIM"
     echo '###############################################################################'
  fi
}

# Set up the appropriate environment
source lmdz_env.sh

# TODO Amaury: commented-out for now // /!\ check w/ Adriana: "groupe@cpu" never appears in seasonal.sh...
## Pour les post-traitements
#sed -i'' -e "s/groupe@cpu/$groupe@cpu/" seasonal.sh
local=$(pwd)

load_install_lib
define_expert_options
set_default_params
read_cmdline_args "$@"
ensure_correct_option_combinations
install_model
setup_def
setup_ce0l
setup_simu
fetch_simu_init_files
run_sim_or_init

exit 1


# COMPILATION OPTIONS depending on the user choices and LMDZ revision number $mysvn
#-------------------------------------------------------
if [[ $rad = "ecrad" && ! -d ecrad_data ]]; then
    cd "$local"; wget https://lmdz.lmd.jussieu.fr/pub/3DInputData/ecrad/ecrad_data.tar; tar xvf ecrad_data.tar; cd - > /dev/null
fi

if [[ $rad = "ecrad" && $aerosols != "n" && $mysvn -lt 4489 ]] ; then echo "Les aerosols tropospheriques ne sont pas pris en charge par ecrad avant LMDZ rev 4489, ici rev est $mysvn"; exit 1; fi

# EXIT after (install and) compilation if we are on jean-zay-pp instead of jean-zay !
hostname=`hostname`
if [ ${hostname:0:11} = "jean-zay-pp" ]; then
 echo "You are on jean-zay-pp, here you only can install and compile the model, not run it"
 echo "If you want to run a simulation, log in to jean-zay and launch main.sh again, without install"
 echo `date`
 exit
fi
















if [ $ok_guide = y -a $init = 1 ]; then
   echo Once initialisation is finished, you have to create nudging files
   echo Edit era2gcm.sh and set the desired parameters in section "User choices"
   echo Make sure you have acces to the chosen ERA files, and the required modules are load
   echo Then run : ./era2gcm.sh
   if [ "$aerosols" = "spla" ]; then
     echo Your aerosol choice is "spla", so you need ERA 10m-winds interpolated on LMDZ grid
     echo Use script era2gcm_uv10m.sh  
   fi
else
   echo Si tout se passe bien, vous avez initialise et lance automatiquement
   echo la simulation.
   echo Si vous voulez modifier les caracteristiques du job, comme le temps
   echo max ou le nombre de proc, il se trouve sur
   echo $SIMRUNTOPDIR/$MAINDIR/tmp_$SIM
fi

###############################################################################
# At the end, print on screen the compilation command, and also in a "compile.sh" script

echo "To recompile the model :"
echo "run the compile${sufiso}.sh script created in the present folder: ./compile${sufiso}.sh gcm " 
