#!/bin/bash

set -eu

#####################################################################
# This script manages the call to setup.sh
#
# Authors :
# F. Hourdin, frederic.hourdin@lmd.ipsl.fr
# Modified A. Sima, adriana.sima@lmd.ipsl.fr
#
# Some options can only be changed directly in lmdz_env.sh or setup.sh
# --> In setup.sh :
# * Orchidee version, to be defined through "veget" option
#   default : OR-CMIP6
# * Aerosol forcing, to be defined through "aerosols" option :
#   n (no, default) / clim (an2000)/ spla (interactive dust and sea salt)
#   !!! STRONG recommendation : experiments with DIFFERENT Orchidee or aerosol options
#   !!!   should be performed in DIFFERENT TEST_PROD folders 
#   !!!  (especially as they may need different initial files) 
#  * Use or not of the RRTM radiation code : rrtm=true/false
#  * Compiling options : debug, use or not of the fcm makefile
#
# --> In lmdz_env.sh :
# * Model & configuration setup
#    If you install the present tutorial_prod package in $STORE/your_folder instead of $STORE
#       modify STORED variable accordingly
#    The model will be installed in $LMDZD directory defined in lmdz_env.sh
#      Default : LMDZD=$WORK; can be changed in LMDZD=$WORK/Your_directory
#    To use a model already installed (and compiled), by you or someone else (check the acces!):
#       set LMDZD=path_to_model_to_use
#       You'll also need to (re)define "LMDZname" accordingly, in setup.sh 
#####################################################################

#===========================================================
# 1. Model setup
#===========================================================

# Version of the tar file on https://lmdz.lmd.jussieu.fr/pub/src
# Last "testing" version, thoroughly checked by the LMDZ team : contains LMDZ rev 4729 (2023-10-22)
# TODO move to expert-level settings ?
version="20240308.trunk" #!! DON'T CHANGE IT WITHOUT CHECKING WITH LMDZ TEAM !!
svn="" 			 #!! DON'T CHANGE IT WITHOUT CHECKING WITH LMDZ TEAM !!
	# CONTACT LMDZ Team :
	# 	email: poihl@listes.lmd.ipsl.fr
	# 	Mattermost: https://mattermost.lmd.ipsl.fr/lmdz/channels/installation-et-tutoriels

# Grid number of points IMxJMxLM
resol="144x142x79"

# Using XIOS for IOs: "-xios" for enabled, else ""
xios=""
# TODO Amaury: je pensais que l'idée de main.sh c'était de mettre des paramètres qu'on peut changer en recompilant uniquement lmdz. Or ce n'est pas le cas d'XIOS...

# Using or not the Cosp simulator: "-cosp" for enabled, else ""
cosp=""

# Choice of physics (default : "lmd" for "phylmd" )
#   Examples : lmd_phys="new" to compile with phynew instead of phylmd,
#              lmd_phys="lmdiso" to run with Isotopes 
lmd_phys="lmd"

#===========================================================
# 2. Simulation setup
#===========================================================

# Initial state and Boundary conditions
# init=   1: to create a new start in INIT
#         0: to read start files in INIT
#       SIM: to read start files from previous simulation SIM0, /!\ SIM0 must be in the same folder as the new SIM
# (limit.nc and aerosols forcing are put in ./LIMIT)
init=1

# climato=1 : Climatological SSTs with 360-day calendar
#        =0 : interannual SSTs with true (ie gregorian) calendar 
climato=1

# Nudging: Can only be activated with climato=0 and freq=mo
# ="-nudging" for enabled, else ""
nudging=""
#If using nudging, then check DEF/guide.def :
#  - ok_guide will be automatically set to "y" by setup.sh
#  - You may want to check/customize the nudging parameters

# Length of elementary simulations yr (year) or mo (month)
freq="yr"

# Initial/final month for simulation
# If $init=1, the INIT file will be called start.200001.nc,
# but the data correspond in fact to another day.
# NB : the run stops in the BEGINNING of mthend (test "next=stopsim")
mthini=200001
mthend=200501

#-----------------------------------------------------------
# Output files, frequencies, levels 
#   If you use IOIPSL (option xios=""), you may want to choose and customize DEF/config.def. 
#     A few versions are available as DEF/config.def_*; config.def is a copy of config.def_default 
#     See phys_out_filekeys, phys_out_filelevels, phys_out_filetimesteps, and supplementary variables
#   If you use XIOS (option xios="-xios"), check/modify DEF/XMLfiles*/file*xml 
#   In both cases, the default output is "histday".
#-----------------------------------------------------------
# Grid characteristics (regular, zoomed)
#   You may want to choose and customize DEF/gcm.def 
#   A few versions are available as DEF/gcm.def_*; 
#      The default, for regular grid 144x142x79, corresponds to _iperiod7 
#      (Expert : TEMPORARILY : setup.sh forces use of gcm.def_zNAfrica_BiJe for aerosols=spla)

#===========================================================
# 3. Example of calling setup.sh in loop
#===========================================================

# By default, the series is done on one or more of the various versions of physiq.def 
# to be chosen among DEF/PHYS/physiq.def_* , and listed in "physics" :
# NOTE : automatically choosing DEF/PHYS/physiq.def_NPiso for isotopes
physics="NPv6.3"
if [[ $lmd_phys = "lmdiso" ]]; then physics="NPiso"; fi

if [[ $(echo "$physics" |wc -w) -gt 1 && $init = 1 ]]; then echo "!=!=! TO LOOP OVER MULTIPLE physics [$physics], YOU NEED init =/= 1 AND start* + limit* files ALREADY AVAILABLE IN THIS FOLDER !=!=!"; exit 1; fi

for phys in $physics; do
  deffile="DEF/PHYS/physiq.def_$phys"
  if [[ -f $deffile ]]; then
     cp -p $deffile DEF/physiq.def

    # name of simulation : can be changed to user's convenience
    # (Default: name=$phys to loop on different physics)
    name="$phys"

    # launching setup.sh with the options defined in this main.sh. 
    # Remember: some other options are only available in lmdz_env.sh and setup.sh.
    # shellcheck disable=SC2086
    ./setup.sh -v "$version" -d "$resol" -mthini "$mthini" -mthend "$mthend" -init "$init" -climato "$climato" -f "$freq" -p "$lmd_phys" -name "$name" $cosp $xios $nudging $svn
  else
    echo "File $deffile inexistent"; exit 1
  fi
done

