#!/bin/bash
# This script sets up and launches a (series of) simulation(s).
# RECOMMENDATION: use main.sh to drive it (do not run it directly)

set -eu

# EXPERT-LEVEL CHOICES, only available in setup.sh, not in main.sh :
####################################################################
function define_expert_options {
  #NOTE : $optim  is part of $LMDZname (see definition of LMDZname below),
  #   so that running with "-debug" needs to (re)install the model (install=-install in main.sh)
  #   If you have already installed the model, and only want to recompile the gcm in debug mode,
  #   add "-debug" to the compilation options in the compile.sh script created in the present folder, and run it (./compile.sh gcm)
  #   ALTERNATIVELY : you can remove $optim in the definition of "LMDZname" below
  optim=""
  #optim="-debug"
  # TODO Amaury: quelle diff avec l'option fcm="debug" ???

  #NOTE : "testmode=y" is for running in test mode :
  # the simulation job "tmp_$SIM" (created on $SCRATCHDIR) is automatically changed for a short run,
  # and submitted to the Jean-Zay "test" queue :
  #  a/ time=00:10:00  (run time limit = 10 min; for test queue, it must be 30 min maximum)
  #  b/ nday=1 (this line, forcing nday to 1 day, is "hidden" in script_SIMU, by default it is commented out)
  #  c/ #SBATCH --qos=qos_cpu-dev (this line is in script_SIMU, by default it is commented out)
  #If you want to change "time", or "nday", but still run in "test" mode, modify the "if [ $debug = 1 ]...; fi" further below.
  testmode=n

  # Radiative code : "oldrad" or "rrtm" or "ecrad"
  rad="rrtm"

  # AEROSOLS : n (=no) / clim (=average 1995-2014) / spla (interactive dust and sea salt)
  # (WARNING : if you first run the scripts with aerosols=n, then you want to change to =clim ,
  #   you'll need to remove the INIT and LIMIT folders that have been created, then run main.sh with init=1
  #   in order to re-run the initialisation job, which downloads the aerosol files and interpolates them)
  aerosols="clim"

  # SURFACE/VEGETATION SCHEME
  # It is controlled by the single variable "veget" which can have the following values
  # - "none": bucket scheme
  # - "CMIP6" or "orch2.0": orchidee version used in CMIP exercise, rev 5661
  # - "orch2.2": orchidee version used in IPSLCM7 configuration
  # - number: orchidee version number : only rev 7983 on branch _2_2, and 7994 on trunk, are available
  veget="CMIP6"
  #AS : If you have installed the model with a given "veget" option, and you want to change it :
  #   --> RECOMMENDED : re-install the model from scratch in a new TEST_PROD folder
  #   TODO Amaury: check w/ Adriana what the use case is for lmdz_setup here
  #   --> EXPERT : If you want to keep your previous installation, and not go through the install_lmdz.sh routine,
  #    you will need to (a) install the proper version of ORCHIDEE in the modipsl/modeles directory, and
  #                     (b) set the "veget" options (opt_veget, orchidee_rev, suforch) to their proper values
  #                        in the file $LMDZD/modipsl/modeles/surface_env
  # (NB : no need to initialise these vars here:
  #     surface_env is either (re)created when (re)installing the model (run with install=-install),
  #     or it already exists (install="").

  # New snow scheme INLANDSIS (y/n)
  # (this flag activates INLANDSIS compilation;
  # not yet done : treatment of specific restart and def file)
  inlandsis="n"

  # COMPILATION options : "", "fcm", "debug"
  fcm=""

  # --->>> ALSO PAY ATTENTION TO OUTPUT files, frequency, level -------------
  #   With IOIPSL : Choose your config.def among the versions available in DEF,
  #     copy it as config.def (the copy is done automatically for "iso"),
  #     edit it and set up output files, frequencies and levels.
  #     NB : For aerosols=spla, output level minimum 4 is required to output the specific variables.
  #          For aerosols=n, the corresponding flags will automatically be set to "n".
  #   With XIOS : adjust DEF/XMLfiles*/file*xml
}

# /!\ DO NOT EDIT BELOW UNLESS YOU KNOW WHAT YOU ARE DOING /!\

function load_install_lib {
  # Fetch and source install_lmdz.sh
  if [[ ! -f "lib_install_lmdz.sh" ]]; then
    wget "http://svn.lmd.jussieu.fr/LMDZ/BOL/script_install_amaury/install_lmdz.sh" # TODO change to normal branch once ready
    mv install_lmdz.sh lib_install_lmdz.sh  # rename to avoid misleading users as to which script to run
    sed -i'' "2i# This is a library used in the installation, DO NOT RUN IT YOURSELF unless you're an expert !" lib_install_lmdz.sh
  fi
  source lib_install_lmdz.sh
}
load_install_lib  # Do not move later, as it would overwrite some functions

function set_default_params {
  # Default value of script parameters
  SIM=$(basename "$local")CTL  # name
  phylmd="lmd" #option -p $phylmd for makelmdz

  cosp="n"  # COSP
  xios="n"  #XIOS

  # Nudging :
  ok_guide="n"
  # With nudging, use real calendar (climato=0) and monthly integrations
  climato=1
  freq="mo"  # frequence mensuelle mo ou annuelle yr

  # NB : the run stops in the BEGINNING of mthend (test "next=stopsim")
  mthini=200001
  mthend=200501
  resol="144x142x79"

  version="20230412.trunk"
  svn=""

  install=0
  init=1

  LIMIT="LIMIT"
}

function read_cmdline_args {
  while (($# > 0)); do
    case $1 in
      "-h") cat <<........fin
       setup.sh can be launched/driven by main.sh; some options can only be specified in setup.sh (ex: veget, aerosols).
             setup.sh [-v version] [-r svn_release] [-init INIT] [-d 96x95x79] [-f mo] [-nudging]
             "version" like 20150828.trunk; see https://lmdz.lmd.jussieu.fr/Distrib/LISMOI.trunk (default <$version>)
             "svn_release"      either the svn release number or "last" (default <$svn>)
             -d IMxJMxLM        to run in resolution IM x JM x LM (default <$resol>)
             -install           pour installer et compiler le modele
             -f mo/yr           pour tourner en mensuel ou annuel (default <$freq>)
             "INIT"             1: creates INIT and LIMIT
                                0: reads from INIT and LIMIT
                                SIMU: reads from preexisting simulation SIMU and LIMIT (default <$init>)
             -nudging           to run with nudging. Nudging files must be created independently
             -p                 the physics to use (default <$phylmd>)
             -name              install folder name (default <$SIM>)
             Other options available (see "options" section in the script)
........fin
        exit;;
      "-v") version="$2"; shift; shift;;
      "-r") svn=$2; shift; shift;;
      "-d") resol=$2; shift; shift;;
      "-f") freq=$2; shift; shift;;
      "-p") phylmd=$2; shift; shift;;
      "-install") install=1; shift;;
      "-name") SIM=$2; shift; shift;;
      "-cosp") cosp=y; shift;;
      "-xios") xios=y; shift;;
      "-init") init=$2; shift; shift;;
      "-nudging") ok_guide=y; shift;;
      "-climato") climato=$2; shift; shift;;
      "-mthini") mthini=$2; shift; shift;;
      "-mthend") mthend=$2; shift; shift;;
      *) $0 -h; exit
    esac
  done

  # Isotopes : Compile and run with isotopes if lmdz_phys="lmdiso" in main.sh
  if [[ $phylmd = "lmdiso" ]]; then isotopes="y"; else isotopes="n"; fi

  # Initialisation
  if [[ $init = 1 || $init = 0 ]]; then
    INIT="INIT"
  else
    INIT=$init
  fi

  yearini=$(echo "$mthini" | cut -c-4)
  if [[ $freq = yr ]]; then stopsim=$(echo "$mthend" | cut -c-4); else stopsim=$mthend; fi

  if [[ $svn = "" ]]; then svnopt=""; else svnopt="-r $svn"; fi

  if [[ -d $SIM ]]; then
     echo "La simulation $SIM existe deja. Il est conseillé d'arrêter et de vérifier."
     echo "Si vous êtes sûr de vous, vous pouvez la prolonger. Voulez vous la prolonger ? (y/n)"
     read -r ans
     if [[ $ans != y ]]; then exit 1; fi
  fi

  if [[ $cosp = y ]]; then ins_cosp="-cosp v1"; else ins_cosp=""; fi

  if [[ $xios = y ]]; then ins_xios="-xios"; else ins_xios=""; fi
}

function ensure_correct_option_combinations {
  # AVOID COMBINATIONS OF OPTIONS THAT DON'T WORK in user choices

  if [[ $ok_guide = y && $climato = 1 ]]; then
     echo "Running nudged simulations with climatological SSTs is not planned. Change <climato> to <0> or modify the setup (experts)"; exit 1
  fi

  if [[ $climato = 0 && $freq = yr ]]; then
     echo "Running simulations with interannual SSTs is possible only month by month and a true calendar."
     echo "Change <climato> to <1> or <freq> to <mo> or modify setup.sh (experts)"; exit 1
  fi

  # STOP if trying to use both ORCHIDEE and Isotopes :
  if [[ $isotopes = "y" && $veget != "NONE" ]]; then
    echo "STOP: You cannot run LMDZ with ORCHIDEE and ISOtopes at the same time; either change <veget> option to NONE, or isotopes to <n> in setup.sh"; exit 1
  fi

  # STOP if trying to use both SPLA and Isotopes :
  if [[ $isotopes = "y" && $aerosols = "spla" ]]; then
    echo "STOP: You cannot run LMDZ with Isotopes and aerosols=spla at the same time; change <aerosols> option to <n> or <clim> in setup.sh"; exit 1
  fi

  # (Temporary) STOP if trying to use Isotopes with XIOS :
  # TODO Amaury: check if still relevant
  if [[ $isotopes = "y" && $xios = "y" ]]; then
    echo "STOP: Isotopes cannont yet be run with XIOS; change <xios> option in main.sh"; exit 1
  fi

  # (Temporary) Constraints for aerosols=spla :
  # --> resolution 128x88x79 and rad=rrtm
  if [[ $aerosols = spla && $resol != 128x88x79 ]]; then
    echo 'For now, <aerosols=spla> requires <resol=128x88x79>, and uses the zoomed grid from gcm.def_zNAfrica_BiJe, for which forcing & initial files are available'
    echo "Right now aerosols=<$aerosols>, resol=<$resol>"
    exit 1
  fi

  if [[ $aerosols = "spla" && $rad != "rrtm" ]]; then
    echo "STOP: For the time being, <aerosols=spla> requires <rad=rrtm>"; exit 1
  fi
}

# TODO Amaury: commented-out for now
## Set up the appropriate environment
##------------------------------------
#sed -i -e "s/hostname=.*.$/hostname=$(hostname)/" lmdz_env.sh
#. lmdz_env.sh
#module list

# TODO Amaury: commented-out for now
## Pour les post-traitements
#sed -i'' -e "s/groupe@cpu/$groupe@cpu/" seasonal.sh  # TODO Amaury: $groupe jamais défini, à virer ?
local=$(pwd)

define_expert_options
set_default_params
read_cmdline_args "$@"
ensure_correct_option_combinations

exit 1







########################################################################
# Recuperation ou compilation du GCM
########################################################################

if [ ! -d $LMDZD ]; then mkdir -p $LMDZD; fi
version_name=LMDZ`echo $version | sed -e 's/-v//g' -e 's/-unstable//' -e 's/-r/r/' -e 's/ //g'`
LMDZname=${version_name}${svn}${optim}OR${veget}${ins_xios}
MODEL=$LMDZD/$LMDZname/modipsl/modeles/LMDZ

if [ $install = 1 ]; then
   if [ -d $MODEL ]; then
	   echo $MODEL
      echo "Vous essayez d installer une version deja presente"
      exit
   else
      echo On installe le modele
      cd $LMDZD
      echo wget https://lmdz.lmd.jussieu.fr/pub/install_lmdz.sh
      \rm -f install_lmdz.sh
      wget --no-cache https://lmdz.lmd.jussieu.fr/pub/install_lmdz.sh
      # Si la ligne ci-dessus "wget https://lmdz.lmd..." ne marche pas pour "certificat expired", utiliser :  
      #wget https://web.lmd.jussieu.fr/~lmdz/pub/install_lmdz.sh
      chmod +x install_lmdz.sh
      echo "./install_lmdz.sh $fcm $optim -v $version $svnopt -d $resol -rad $rad -bench 0 -parallel mpi_omp $ins_cosp $ins_xios -name $LMDZname -env_file $local/lmdz_env.sh -veget $veget" >> $local/install_lmdz_options.$$
      ./install_lmdz.sh $fcm $optim -v $version $svnopt -d $resol -rad $rad -bench 0 -parallel mpi_omp $ins_cosp $ins_xios -name $LMDZname -env_file $local/lmdz_env.sh -veget $veget
      mv install_lmdz.sh install_lmdz.$$
      if grep "Not Found" $LMDZname/get.log >/dev/null; then echo "version=$version is NOT A TESTING VERSION, Ask LMDZ Team"; exit; fi
      cd $local
   fi
else
   if [ ! -d $MODEL ]; then
      echo Le model $MODEL n est pas installe
      echo relancer avec install=1 si vous voulez l installer automatiquement
      exit
   fi
fi

# Determine svn revision number, which will determine compilation options, executable names etc
# If svn not available, will use the svn writen in $MODEL/Readm*md
# For old version it assumes that it is before 4185 (the version
# for which the test was introduced
mysvn=`svnversion $MODEL | egrep -o "[0-9]+" 2>/dev/null`  
if [ "$mysvn" = "" ]; then mysvn=`grep 'Revision: [0-9]' $LMDZD/$LMDZname/Read*.md | awk ' { print $2 } ' 2>/dev/null`; fi
if [ "$mysvn" = "" ]; then mysvn=4190; fi
echo mysvn $mysvn

# For mysvn >= 4532, "module load netcdf95" must be added in lmdz_env.sh for Jean-Zay if it's not there;
# lmdz_env.sh must be sourced again, and copied again as $MODEL/arch/arch-X64_JEANZAY.env
# (first copy is done by install_lmdz.sh launched with option "-env lmdz_env.sh" )
if [ ${hostname:0:5} = "jean-" -a $mysvn -ge 4532 -a ! "`grep netcdf95 lmdz_env.sh`" ]; then
   cp -p lmdz_env.sh lmdz_env.orig
   sed -i'' -e 's/module load nco/module load netcdf95\n                module load nco/' lmdz_env.sh
   cp -p lmdz_env.sh $MODEL/arch/arch-X64_JEANZAY.env
   . lmdz_env.sh
fi

# COMPILATION OPTIONS depending on the user choices and LMDZ revision number $mysvn
#-------------------------------------------------------
if [ "$ARCH" = "" ]; then arch=; else arch="-arch $ARCH"; fi

if [ $cosp = y ]; then opt_cosp="-cosp true"; else opt_cosp=""; fi
   # makelmdz* uses "opt_cosp", and for v1 (default) needs "-cosp true/false"  
   # When cosp v2 will be validated, we'll have opt_cosp=ins_cosp="-cosp v2" 
   # (option cosp2 in makelmdz* to be ignored; it is a transitory version from v1 to v2)

if [ $xios = y ]; then opt_xios="-io xios"; else opt_xios=""; fi

case $rad in
    oldrad) iflag_rrtm=0; NSW=2; opt_rad="";;
    rrtm)   iflag_rrtm=1; NSW=6; if [ $mysvn -le 4185 ]; then opt_rad="-rrtm true"; else opt_rad="-rad rrtm"; fi;;
    ecrad)  iflag_rrtm=2; NSW=6; opt_rad="-rad ecrad";;
    *) echo Only oldrad rrtm ecrad for rad option; exit 
esac
if [ "$rad" = "ecrad" -a ! -d ecrad_data ]; then
    cd $local; wget https://lmdz.lmd.jussieu.fr/pub/3DInputData/ecrad/ecrad_data.tar; tar xvf ecrad_data.tar; cd -
fi
if [ "$rad" = "ecrad" -a "$aerosols" != "n" -a "$mysvn" -lt 4489 ] ; then echo Les aerosols tropospheriques ne sont pas pris en charge par ecrad avant LMDZ rev 4489, ici rev est $mysvn; exit; fi


# Aerosols : also define the associated suffix in gcm name
if [ $aerosols = spla ]; then
  opt_aer=' -dust true ';  sufaer='_spla'
else
  opt_aer=""            ;  sufaer=''
fi

if [ $inlandsis = y ]; then
   opt_inlandsis="-inlandsis true"
else
   opt_inlandsis=""
fi

# Isotopes : also define the associated suffix in gcm name
if [ $isotopes = y ]; then
  opt_isotopes="-isotopes true"; sufiso="_iso"
else
  opt_isotopes=""              ; sufiso=""
fi

# Source the surface_env file created by install_lmdz.sh, to get the "veget"-related variables
#--------------------------------------------------------------------------------------
# surface_env is produced by install_lmdz in $MODEL=$LMDZD/$LMDZname/modipsl/modeles
# Every model installation will have an unique orchidee_rev, 
#   to be changed by hand if a new Orchidee rev is installed by hand in modipsl/models
. $LMDZD/$LMDZname/modipsl/modeles/surface_env
echo LMDZname $LMDZD/$LMDZname/modipsl/modeles/surface_env
echo veget=$veget
echo opt_veget=$opt_veget
echo orchidee_rev=$orchidee_rev
echo suforch=$suforch


#######################################################################
# Compilation du modele
#######################################################################
# The gcm name defined below is used to check if the executable exists
# It is also used when compiling without fcm : the output of makelmdz is gcm.e, and is renamed as $gcm
#
# NOTE : Some compilation options do not appear in the gcm name (xios, cosp, ...).
#   (Also : LMDZ rev before 4185 included, did not contain radiative-code suffix; starting r4186, it includes oldrad/rrtm/ecrad )
#   So if you change those options and you want to recompile,
#   setup.sh will not detect the change when testing if [ ! -f $gcm ], 
#   You'll need to modify and run the compile.sh script created in the present folder, and run it (./compile.sh gcm) 

# compile_opt_iso is used for gcm only; ce0l cannot be compiled with isotopes yet (phylmd instead of phylmdiso is required)
if [ $isotopes = y ]; then
    compile_opt_iso="$optim -p $phylmd $opt_cosp $opt_xios $opt_rad $arch -d $resol $opt_veget $opt_aer $opt_inlandsis $opt_isotopes -mem -parallel mpi_omp"  
    if [ $mysvn -le 4185 ]; then 
      suffix_iso=_${resol}_phy${phylmd}_para_mem${suforch}${sufaer}${sufiso}
    else
      suffix_iso=_${resol}_phy${phylmd}_${rad}_para_mem${suforch}${sufaer}${sufiso}
    fi
    phylmd="lmd"
fi

# compile_opt is used for gcm if isotopes=n, and for ce0l always
compile_opt="$optim -p $phylmd $opt_cosp $opt_xios $opt_rad $arch -d $resol $opt_veget $opt_aer $opt_inlandsis -mem -parallel mpi_omp"

if [ $mysvn -le 4185 ]; then suffix=_${resol}_phy${phylmd}_para_mem${suforch}${sufaer};
                         else suffix=_${resol}_phy${phylmd}_${rad}_para_mem${suforch}${sufaer}; fi

# use an intermediate variable to use either suffix or suffix_iso
strsuffix=suffix${sufiso}
gcm=$MODEL/bin/gcm${!strsuffix}.e

echo $mysvn
echo $gcm

if [ ! -f $gcm ]; then
   echo Le model $gcm n existe pas
   echo il va se compiler automatiquement sur $MODEL
   sleep 10

   if [ "$fcm" = "-nofcm" ]; then
       compile="./makelmdz $compile_opt \$1; mkdir -p bin; \mv -f \${prog}.e $gcm"
       compile_iso="./makelmdz $compile_opt_iso \$1; mkdir -p bin; \mv -f \${prog}.e $gcm"
   else
      # makelmdz_fcm directly produces executable named $gcm, depending on the compilation options
       compile="./makelmdz_fcm $compile_opt -j 2 \$1"
       compile_iso="./makelmdz_fcm $compile_opt_iso -j 2 \$1"
   fi

   #NB on est dans $local, qui est $STORE/$MAINDIR
   echo $compile${sufiso}

   #create compile.sh; 
   # if isotopes = y , it will still be used for compiling ce0l
   cat << ...eod >| compile.sh
   cd $MODEL
   prog=\$1
   ${compile}
...eod
   pwd
   ls -l compile.sh
   chmod +x compile.sh

   #create compile_iso.sh for compiling gcm_ _iso.e
   if [ $isotopes = y ]; then
   cat << ...eod >| compile_iso.sh
   cd $MODEL
   prog=\$1
   ${compile_iso}
...eod
   pwd
   ls -l compile_iso.sh
   chmod +x compile_iso.sh
   fi

   echo Compilation de LMDZ, suivre avancement dans lmdz.log
   ./compile${sufiso}.sh gcm > lmdz.log 2>&1

   if [ ! -f $gcm ]; then echo la compilation a echoue; exit; fi
   cd $local
fi


######################################################################
# Choix du nombre de processeurs
# NOTES : 
# omp=8 by default (for Jean-Zay must be a divisor of 40 procs/node), but we need 
#   omp=1 for SPLA (only MPI parallelisation)
#   omp=2 for veget=CMIP6 beacause of a bug in ORCHIDEE/src_xml/xios_orchidee.f90
######################################################################
jm=`echo $resol | cut -dx -f2`
(( mpi = ( $jm + 1 ) / 2 ))
omp=8
if [ $aerosols = spla ]; then omp=1; fi
if [ $veget = CMIP6 -a $xios = y ]; then omp=2; fi
if [ $mpi -gt $nb_mpi_max ]; then mpi=$nb_mpi_max; fi
if [ $omp -gt $nb_omp_max ]; then omp=$nb_omp_max; fi

######################################################################
# Utilisation des .def_iso pour LMDZ-ISOtopes
######################################################################
if [ $isotopes = y ]; then
  for file_iso in $( ls DEF | grep _iso)
  do
    cp DEF/$file_iso DEF/${file_iso%%_iso}
  done
fi

######################################################################
# Choix de la grille verticale
######################################################################
if [ ! -d DEF ]; then echo Recuperer un repertoire DEF; exit; fi
lm=`echo $resol | cut -dx -f3`
if [ ! -f DEF/L$lm.def ]; then echo resolution verticale non prevue
                                 echo creer un fichier DEF/L$lm.def
   exit
else
   sed -i -e 's/INCLUDEDEF=L.*.def/INCLUDEDEF=L'$lm'.def/' DEF/run.def
fi

######################################################################
# Changements dans les fichiers DEF/*def
# (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $WRK)
######################################################################

# Choix du fichier traceur.def coherent avec l'option "aerosols"
#  NOTE : Le nouveau tracer.def_nospla par defaut n'inclut pas Rn-Pb; 
#		si on les veut, il faut utiliser ci-dessous; a la place, tracer_RN_PB.def
#---------------------------------------------------------------------
# NB: Si on change de traceur.def (entre spla et nospla), il faut refaire l'etape "initialisation" (ce0l)? 
# Normalement les traceurs absents de start* files sont initialises=0 dans le code; verifier pour spla !  
if [ $aerosols = spla ]; then 
   # ancien traceur pour l instant
   cp DEF/traceur.def_spla   DEF/traceur.def  
elif [ $isotopes = n ]; then
   # nouveau traceur 
   # déjà copié si 'y'
   cp DEF/tracer.def_nospla DEF/tracer.def  
fi

# TEMPORAIREMENT pour spla on force l'utilisation de gcm.def_zNAfrica_BiJe (avec resolution 128x88x79)
#----------------------------------------------------------------------
if [ $aerosols = spla ]; then cp DEF/gcm.def_zNAfrica_BiJe DEF/gcm.def; fi

# Inscription du choix ok_guide dans DEF/guide.def
#---------------------------------------------------------------------
sed -i -e 's/ok_guide=.*.$/ok_guide='$ok_guide'/' DEF/guide.def

# Inscription du type de calendrier (qui est fonction de $climato) dans DEF/run.def
#---------------------------------------------------------------------
# NB Contrairement a ce qui est ecrit dans les fichiers run.def standard, 
# dans ce tutorial le choix earth_365d n'est pas disponible, et earth_366d s'appelle gregorian
if [ $climato = 0 ]; then calend=gregorian; else calend=earth_360d; fi
sed -i -e 's/calend=.*.$/calend='$calend'/' DEF/run.def

# Changements dans config.def (pre-choisi, et regle pour output si utilisation avec IOIPSL) 
#   cf options veget, aerosols, cosp, xios
#---------------------------------------------------------------------
if [ $veget = NONE ]; then  VEGET=n; else VEGET=y; fi
sed -i -e 's/VEGET=.*.$/VEGET='$VEGET'/' DEF/config.def

if [ $aerosols = n ]; then
  # set flag_aerosols=0 and flags ok_ade&co=n
  sed -i'' -e 's/^ok_cdnc=.*.$/ok_cdnc=n/' -e 's/^ok_ade=.*.$/ok_ade=n/' -e 's/^ok_aie=.*.$/ok_aie=n/' -e 's/^ok_alw=.*.$/ok_alw=n/' -e 's/^flag_aerosol=.*.$/flag_aerosol=0/' DEF/config.def
fi

# COSP : ok_cosp desactive COSP si on a compile avec; il ne l'active pas si on a compile sans 
sed -i -e 's/ok_cosp.*.$/ok_cosp='$cosp'/' DEF/config.def
if [ $cosp = y ]; then \cp -f $MODEL/DefLists/cosp*.txt $local/DEF/; fi

# Sorties LMDZ en fonction de l'option "xios"
sed -i'' -e 's/ok_all_xml=.*.$/ok_all_xml='$xios'/' DEF/config.def

# Ajuster physiq.def en fonction de radiative code (default: values for rad=rrtm)
#   Pour isotopes=y , mettre iflag_ice_thermo=0 au lieu de 1
#---------------------------------------------------------------------
sed -i'' -e 's/iflag_rrtm=.*.$/iflag_rrtm='$iflag_rrtm'/' -e 's/NSW=.*.$/NSW='$NSW'/' DEF/physiq.def
pwd
sed -i'' -e 's:directory_name.*$:directory_name="'$local'/ecrad_data",:' DEF/namelist_ecrad

if [ $isotopes = y ]; then iflag_ice_thermo=0; else iflag_ice_thermo=1; fi
sed -i -e 's/iflag_ice_thermo=.*.$/iflag_ice_thermo='$iflag_ice_thermo'/' DEF/physiq.def

# Choix de orchidee.def en fonction de orchidee_rev; modification pour xios
#  NOTE separate orchidee_pft.def file for ORCHIDEE trunk post-CMIP6  
#---------------------------------------------------------------------
orchidee_def=orchidee.def_6.1
orchidee_pft_def=""
if [ $veget = "7983" ]; then 
  orchidee_def=orchidee.def_6.2work 
elif [ $veget = "7994" ]; then 
  orchidee_def=orchidee.def_6.4work
  orchidee_pft_def=orchidee_pft.def_6.4work
  if ! grep "INCLUDEDEF=orchidee_pft.def" DEF/run.def; then
    sed -i'' -e 's/INCLUDEDEF=orchidee.def/INCLUDEDEF=orchidee.def\nINCLUDEDEF=orchidee_pft.def/' DEF/run.def; fi
fi
\cp -f DEF/$orchidee_def DEF/orchidee.def
if [ "$orchidee_pft_def" != "" ]; then \cp -f DEF/$orchidee_pft_def DEF/orchidee_pft.def; fi

# Only for veget=CMIP6 it is still possible to use IOIPSL; newer versions of orchidee.def have XIOS_ORCHIDEE_OK = y
sed -i'' -e 's/XIOS_ORCHIDEE_OK =.*.$/XIOS_ORCHIDEE_OK = '$xios'/' DEF/orchidee.def


######################################################################
# Si on tourne avec XIOS, mise a jour des fichiers context et field* dans XMLfilesLMDZ
# (ils vont se repercuter dans les repertoires de simulation $local/$SIM et de run $WRK)
######################################################################
if [ $xios = y ]; then
  \cp -f $MODEL/DefLists/context_lmdz.xml $local/DEF/XMLfilesLMDZ/.
  \cp -f $MODEL/DefLists/field_def_lmdz.xml $local/DEF/XMLfilesLMDZ/.
  if [ $cosp = y ]; then \cp -f $MODEL/DefLists/field_def_cosp1.xml $local/DEF/XMLfilesLMDZ/.; fi
fi


######################################################################
# Verification de l'existance de l'etat initial et compilation eventuelle
# pour sa creation
######################################################################
if [ ! -d $INIT ]; then 
   if [ $init = 0 ]; then
      echo  Recuperer les repertoires $INIT ou lancer avec option -init; exit
   else
      ce0l=$MODEL/bin/ce0l${suffix}.e
      echo $ce0l

      # Test if $ce0l exists (it may be the case if you use a model configuration already installed and compiled)
      if [ ! -f $ce0l ]; then
       echo L executable $ce0l n existe pas
       echo il va se compiler automatiquement sur $MODEL
       sleep 10

       $local/compile.sh ce0l
     fi

     if [ ! -f $ce0l ]; then echo la compilation de $ce0l a echoue; exit; fi

      cd $local
   fi
elif [ $init = 1 ]; then
   echo Vous essayez d initialiser le modele mais $INIT existe deja
   exit
fi

MAINDIR=`basename \`pwd\``

# EXIT after (install and) compilation if we are on jean-zay-pp instead of jean-zay !
hostname=`hostname`
if [ ${hostname:0:11} = "jean-zay-pp" ]; then
 echo "You are on jean-zay-pp, here you only can install and compile the model, not run it" 
 echo "If you want to run a simulation, log in to jean-zay and launch main.sh again, without install"
 echo `date`
 exit
fi

######################################################################
# On cree sur le SCRATCHD un repertoire de travail avec le meme
# nom que le repertoire local
######################################################################
. ./lmdz_env.sh
WRK=$SCRATCHD/$MAINDIR
mkdir -p $WRK
cd $WRK
echo La simulation s executera sur $SCRATCHD/$MAINDIR

#####################################################################
# Creation du repertoire $SIM s'il n'existe pas deja
#####################################################################

if [ ! -d $local/$SIM ]; then
mkdir $local/$SIM
cd $local
sed -e 's:^rebuild=.*.$:rebuild='$LMDZD/$LMDZname/modipsl/bin/rebuild':' -e 's/groupe@cpu/'$groupe'@cpu/' reb.sh  >| $local/$SIM/reb.sh; chmod +x $local/$SIM/reb.sh
cp lmdz_env.sh $local/$SIM/
mkdir $local/$SIM/DEF; cp DEF/*def DEF/namelis* $local/$SIM/DEF/
#Pour XIOS, respectivement COSP, copier aussi les fichiers *xml / *txt
if [ $cosp = y ]; then cp DEF/cosp*txt $local/$SIM/DEF/; fi
if [ $xios = y ]; then
   cp DEF/XMLfilesLMDZ/*xml $local/$SIM/DEF/
   if [ $veget != 'NONE' ]; then cp DEF/XMLfilesOR$veget/*xml $local/$SIM/DEF/; fi
fi
chmod u+w $local/$SIM/DEF/*

# Gestion du calendrier
#######################
sed -e 's/anneeref=.*.$/anneeref='$yearini'/' \
    DEF/run.def >| $local/$SIM/DEF/run.def
cd $SIM
if [ "$freq" = "yr" ]; then date=$yearini; else date=$mthini; fi
echo $date a faire >| etat

# Recuperation des fichiers : executable initiaux et forcages
#############################################################
echo $date 
for f in start startphy; do inf=../$INIT/$f.$date.nc; if [ -f $inf -o $init = 1 ]; then ln -s $inf ./; else echo $inf inexistant; exit; fi; done
for f in sechiba stomate; do if [ -f ../$INIT/start_$f.$date.nc ]; then ln -sf  ../$INIT/start_$f.$date.nc .; fi; done
cp $gcm gcm.e
fi  # fin de " if [ ! -d $local/$SIM ]"
cd $local/..

# Choix du "time limit" pour le job
###################################
if [ "$freq" = "yr" ]; then cput=04:00:00; else cput=01:00:00; fi

#####################################################################
echo Modification du script de lancement
###################################################################/$S
sed -e 's/stopsim=.*.$/stopsim='$stopsim'/' -e 's/^veget=.*.$/veget='$veget'/' -e 's/orchidee_rev=.*.$/orchidee_rev='$orchidee_rev'/' -e 's/^aerosols=.*.$/aerosols='$aerosols'/' -e 's/^isotopes=.*.$/isotopes='$isotopes'/' -e 's/NOM_SIMU/'$SIM'/' -e 's/time=.*.$/time='$cput'/' -e 's/MAINDIR=.*.$/MAINDIR='$MAINDIR'/' -e 's:STORED=.*.*:STORED='$STORED':'  -e 's:SCRATCHD=.*.*:SCRATCHD='$SCRATCHD':' -e 's/ntasks=.*.$/ntasks='$mpi'/' -e 's/cpus-per-task=.*.$/cpus-per-task='$omp'/' -e 's/nthreads=.*./nthreads='$omp'/' -e 's/^climato=.*.$/climato='$climato'/' -e 's/^ok_guide=.*.$/ok_guide='$ok_guide'/' -e 's/groupe@cpu/'$groupe'@cpu/' $local/script_SIMU >| $WRK/tmp_$SIM

if [ "$testmode" = "y" ]; then
  sed -i -e 's/time=.*.$/time=00:30:00/' -e 's/#nday=1/nday=1/' -e 's/#[[:space:]]#SBATCH[[:space:]]--qos=qos_cpu-dev/#SBATCH --qos=qos_cpu-dev/' $WRK/tmp_$SIM
fi

if [ $climato = 0 ]; then
# calend est choisi plus haut dans "Changements dans les fichiers DEF/*def" et ecrit dans $MAINDIR/DEF/run.def
yrini=`echo $mthini | cut -c-4`
yrend=`echo $mthend | cut -c-4`
yrs=""; yr=$yrini; while [ $yr -le $yrend ]; do yrs="$yrs $yr"; (( yr = $yr + 1 )); done
suf=360x180_
else
yrs=2000
suf=1x1_clim
fi


#####################################################################
   echo Recuperation eventuelle de certains fichiers sur LMDZ_Init
#####################################################################

if [ ! -d $LMDZ_Init ]; then mkdir $LMDZ_Init; fi

#------------------------------------------------------------------- 
# Fichiers ORCHIDEE
#------------------------------------------------------------------- 
get="myget 3DInputData/Orchidee/"
liste_get="PFTmap_IPCC_2000.nc cartepente2d_15min.nc "
liste_get+="routing.nc routing_simple.nc lai2D.nc soils_param.nc "
liste_get+="woodharvest_2000.nc PFTmap_15PFT.v1_2000.nc"
for file in $liste_get; do 
  if [ ! -f $LMDZ_Init/$file ]; then cd $LMDZ_Init; ${get}$file; cd -; fi 
done
# Additionnal files needed for ORCHIDEE trunk post-CMIP6
if [ $veget = 7994 -a ! -f $LMDZ_Init/soil_bulk_and_ph.nc ]; then
  cd $LMDZ_Init 
  ${get}soil_bulk_and_ph.nc; ${get}NITROGEN_for_ORtrunk.tar
  tar -xvf NITROGEN_for_ORtrunk.tar; cd - 
fi
#------------------------------------------------------------------- 
# Fichiers aerosols/chimie
#------------------------------------------------------------------- 
if [ $aerosols = clim ]; then
  get="myget 3DInputData/AerChem/"
  #liste_get="aerosols1850_from_inca.nc aerosols2000_from_inca.nc" 
  #aerosols9999_from_inca.nc est un lien vers aerosols1995_2014_ensavg_from_inca.nc
  liste_get="aerosols1850_from_inca.nc aerosols9999_from_inca.nc"
  for file in $liste_get; do 
    if [ ! -f $LMDZ_Init/$file ]; then cd $LMDZ_Init; ${get}$file; cd -; fi
  done
fi

# For SPLA
#-------------------
# Dans ${LMDZ_Init} on cree folder SPLA_Init et dedans le INITIAL 
# Pour l'instant on copie là-dedans de chez Binta les fichiers a la res zoomNaf;
# plus tard on y recupererea des fichiers a haute resolution reguliere depuis http:/LMDZ, 
# a regrider ensuite par un script interp_fichiers_spla.sh (comme pour aerosols="clim") 
# Les fichiers (regrides, sauf SOILSPEC.data utilise tel quel) seront mis dans $MAINDIR/INPUT_SPLA (equivalent de INPUT_DUST)
if [ $aerosols = spla ]; then
  if [ ! -d ${LMDZ_Init}/SPLA_Init ]; then mkdir ${LMDZ_Init}/SPLA_Init; mkdir $LMDZ_Init/SPLA_Init/INITIAL; fi
  get="cp -p /gpfsstore/rech/gzi/rgzi027/ergon/BIBIAERO/INPUTS_DUST/INITIAL/"
  liste_get="wth.dat cly.dat donnees_lisa.nc SOILSPEC.data \
               carbon_emissions.nc sulphur_emissions_antro.nc  \
               sulphur_emissions_nat.nc  sulphur_emissions_volc.nc" 

  for file in $liste_get; do
    if [ ! -f $LMDZ_Init/SPLA_Init/INITIAL/$file ]; then cd $LMDZ_Init/SPLA_Init/INITIAL; ${get}$file .; cd -; fi
  done
###
  #Cas particulier des fichiers mensuels dust.nc :A DECIDER : 
  #C'est entre le cas des aerosols.clim= 1 seul fichier, annuel, 
  # et le cas des vents guidage, pré-interpolés avec era2gcm pour toute la periode, dans MAINDIR/GUIDE. 
  # On pourrait (a)demander de precalculer dust.nc aussi, dans MAINDIR/INPUT_SPLA - avec un script à adapter de Jeronimo. Rien a mettre dans SPLA_Init alors. 
  # Ou (b) fournir dans SPLA_Init les 12 mois d'un dust.nc climato (an 2006 pour nous ici) à la res EC, et faire juste le regrid vers MAINDIR/INPUT_SPLA
  #ICI pour l'instant je copie les fichiers de chez Binta (repositoire==http...) dans LMDZ_Init/SPLA_Init, avec test sur mois 01

  if [ ! -d ${LMDZ_Init}/SPLA_Init/PERIOD0001 ]; then cp -pr /gpfsstore/rech/gzi/rgzi027/ergon/BIBIAERO/INPUTS_DUST/PERIOD* $LMDZ_Init/SPLA_Init/.; fi

 #A la fin on doit avoir dans SPLA_Init les fichiers initiaux dans INITIAL plus les repertoires PERIOD00MM contenant dust.nc 
 #Cela doit se retrouver dans script_SIMU qui les copie dans le repertoire de run sur $SCRATCH

fi #$aerosols = spla


#------------------------------------------------------------------- 
# Fichiers Init
#------------------------------------------------------------------- 
get="myget 3DInputData/Init/"
liste_get="alb_bg_modisopt_2D_ESA_v2.nc reftemp.nc"
for file in $liste_get; do 
  if [ ! -f $LMDZ_Init/$file ]; then cd $LMDZ_Init; ${get}$file; cd -; fi
done

cd $local



if [ $init = 1 ]; then
   #####################################################################
      echo Creation de l etat initial
   #####################################################################

   # Creation du repertoire INIT et mise en place de liens logiques vers les starts
   # en anticipation de leur création :
   mkdir $local/$INIT; cd $local/$INIT
   for an in $mthini $yearini; do for f in start startphy; do ln -s $f.nc $f.$an.nc; done; done

   # Creation du repertoire INIT temporaire et rapatriement des fichiers necessaires
   if [ -d $WRK/$INIT ]; then mv $WRK/$INIT $WRK/$INIT$$; fi
   mkdir $WRK/$INIT;  cp -r $local/DEF $WRK/$INIT/
   cd $WRK/$INIT; cp DEF/*.def .; cp $local/lmdz_env.sh .
   if [ $xios = y ]; then 
       cp DEF/XMLfilesLMDZ/*xml .  
       if [ $veget != 'NONE' ]; then cp DEF/XMLfilesOR$veget/*xml .; fi 
   fi 
   sed -e 's/anneeref=.*.$/anneeref='$yearini'/' DEF/run.def >| run.def

#------------------------------------------------------------------- 
# Fichiers Limit
#------------------------------------------------------------------- 
   get="myget 3DInputData/Limit/"
   liste_get="Albedo.nc Relief.nc Rugos.nc landiceref.nc"
   for yr in $yrs; do
       if [ $climato = 0 ]; then sufyr=$suf$yr; else sufyr=$suf; fi
       liste_get="$liste_get  amipbc_sic_$sufyr.nc amipbc_sst_$sufyr.nc"
   done
   echo LISTEGET $liste_get
   for file in $liste_get; do
     if [ ! -f $LMDZ_Init/$file ]; then cd $LMDZ_Init; ${get}$file; cd -; fi
     ln -s $LMDZ_Init/$file
   done
#------------------------------------------------------------------- 
# ECDYN
#------------------------------------------------------------------- 
   get="myget 3DInputData/Init/"
   if [ ! -f $LMDZ_Init/ECDYN.nc ]; then cd $LMDZ_Init; ${get}ECDYN.nc; cd -; fi
   ln -s $LMDZ_Init/ECDYN.nc
   ln -sf ECDYN.nc ECPHY.nc


   # Creation du script d'initialisation
   cat << ...eod >| tmp
#!/bin/bash

#SBATCH --job-name=Init         # nom du job
#SBATCH -A "$groupe"@cpu
#SBATCH --ntasks=1             # Nombre de processus MPI
#SBATCH --cpus-per-task=16     # nombre de threads OpenMP
# /!\ Attention, la ligne suivante est trompeuse mais dans le vocabulaire
# de Slurm "multithread" fait bien rÃ©fÃ©rence Ã  l'hyperthreading.
#SBATCH --hint=nomultithread   # 1 thread par coeur physique (pas d'hyperthreading)
#SBATCH --time=00:10:00            # Temps dâ€™exÃ©cution maximum demandÃ© (HH:MM:SS)
#SBATCH --output=Init%j.out     # Nom du fichier de sortie
#SBATCH --error=Init%j.out      # Nom du fichier d'erreur (ici commun avec la sortie)
# To submit to dev queue; "time" (above) must be max 2h
# #SBATCH --qos=qos_cpu-dev

# ANCIEN MULTI STEP  case \${LOADL_STEP_NAME} in

# ANCIEN MULTI STEP   init )

   if [ ! -f lmdz_env.sh ]; then echo manque fichier lmdz_env.sh; ls; exit; fi
   . lmdz_env.sh
   ulimit -s unlimited
   export OMP_STACKSIZE=800M
   export OMP_NUM_THREADS=1
   cd $WRK/$INIT
   echo Executable : $ce0l
   for yr in $yrs; do 
      if [ $climato = 0 ]; then sufyr=$suf\$yr; else sufyr=$suf; fi
      ln -sf amipbc_sic_\$sufyr.nc amipbc_sic_1x1.nc
      ln -sf amipbc_sst_\$sufyr.nc amipbc_sst_1x1.nc
      sed -e 's/anneeref=.*.$/anneeref='\$yr'/' DEF/run.def >| run.def
      echo Starting initialisation
      $run 1 $ce0l
      if [ $climato = 0 ]; then mv limit.nc limit.\$yr.nc; fi
   done
# ANCIEN MULTI STEP   ;;

# ANCIEN MULTI STEP   interp )
   if [ $aerosols = clim ]; then
    cp $local/interp_aerosols.sh .; chmod +x interp_aerosols.sh
    # Les aerosols de l'annee 2000 ont ete remplaces par "9999" qui pointe vers un fichier moyen sur 1995-2014
    #for year in 2000 1850; do  ./interp_aerosols.sh \$year; done
    #mv aerosols.2000.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
    for year in 9999 1850; do ./interp_aerosols.sh \$year; done
    mv aerosols.9999.nc aerosols.clim.nc; mv aerosols.1850.nc aerosols.nat.nc
   fi

   # AS : S'il etait possible d'automatiser l'interpolation de l'input SPLA, ce serait a lancer ici
   #en attendant, on passe au paragraphe suivant où on copie les fichiers à la res ZoomNaf depuis $LMDZ_Init/SPLA_Init
   #if [ $aerosols = spla ]; then
    #cp $local/futur script interp_aerosols_SPLA.sh .; chmod +x interp_aerosols_SPLA.sh
    #for file in...; do  ./interp_aerosols_SPLA.sh \$year; done
    #etc etc etc ...
   #fi

# Copy initial and forcing files in $local/$INIT and $local/$LIMIT; also in $local/INPUT_SPLA if $aerosols=spla
   for f in sta* limit.nc gri*nc; do cp \$f $local/$INIT/\$f; done
   mkdir -p $local/$LIMIT
   for f in limit*.nc ; do cp \$f $local/$LIMIT/\$f; done
   if [ $aerosols = clim ]; then  for f in aerosols[.0-9]*nc; do cp \$f $local/$LIMIT/\$f; done; fi
   # 
   if [ $aerosols = spla ]; then 
     #mkdir -p $local/INPUT_SPLA; pour l'instant on copie $LMDZ_Init/SPLA_Init en block
     if [ ! -d $local/INPUT_SPLA ]; then cp -pr $LMDZ_Init/SPLA_Init $local/INPUT_SPLA; fi
   fi
   cd $WRK
...eod
   if [ $ok_guide != y ]; then # Running first simulation automatically except for nudging
      cat << ...eod >> tmp
         $submit tmp_$SIM
...eod
   fi
   cat << ...eod >> tmp
# ANCIEN MULTI STEP   esac
...eod

   echo '###############################################################################'
   echo Submitting initialisation job
pwd
   $submit tmp
   echo '###############################################################################'

else
#case [ $init != 1 ] 

   cd $WRK
   echo '###############################################################################'
   echo Submitting job tmp_$SIM
   echo $submit tmp_$SIM
   $submit tmp_$SIM
   echo '###############################################################################'
fi


if [ $ok_guide = y -a $init = 1 ]; then
   echo Once initialisation is finished, you have to create nudging files
   echo Edit era2gcm.sh and set the desired parameters in section "User choices"
   echo Make sure you have acces to the chosen ERA files, and the required modules are load
   echo Then run : ./era2gcm.sh
   if [ "$aerosols" = "spla" ]; then
     echo Your aerosol choice is "spla", so you need ERA 10m-winds interpolated on LMDZ grid
     echo Use script era2gcm_uv10m.sh  
   fi
else
   echo Si tout se passe bien, vous avez initialise et lance automatiquement
   echo la simulation.
   echo Si vous voulez modifier les caracteristiques du job, comme le temps
   echo max ou le nombre de proc, il se trouve sur
   echo $SCRATCHD/$MAINDIR/tmp_$SIM
fi

###############################################################################
# At the end, print on screen the compilation command, and also in a "compile.sh" script

echo "To recompile the model :"
echo "run the compile${sufiso}.sh script created in the present folder: ./compile${sufiso}.sh gcm " 
