#!/bin/bash
set -eu  # error on command failure, and on unset variables
export LC_ALL=C  # standardize awk format

###########################################################################
# Author : Laurent Fairhead et Frédéric Hourdin
# Usage  : install_lmdz.sh -help
#
# bash installation script of the LMDZ model on different computer types :
# Linux PC, "mesocentre" (IPSL-UPMC, IPSL-X), super-computer (IDRIS)
#
# The model is downloaded in the following directory tree
# $MODEL/modipsl/modeles/...
# using the "modipsl" infrastructure created by the "IPSL"
# for coupled (atmosphere/ocean/vegetation/chemistry) climate modeling
# activities.
# Here we only download atmospheric (LMDZ) and vegetation (ORCHIDEE)
# components.
#
# The sources of the models can be found in the "modeles" directory.
# In the present case, LMDZ, ORCHIDEE, and IOIPSL or XIOS (handling of
# input-outputs using the NetCDF library).
#
# The script downloads various source files (including a version of NetCDF)
# and utilities, compiles the model, and runs a test simulation in a
# minimal configuration.
#
# Prerequisites : pgf90/gfortran, bash or ksh, wget , gunzip, tar, ...
#
# Modif 18/11/2011
#    changes for option real 8.
#      We compile with -r8 (or equivalent) and -DNC_DOUBLE for the GCM
#      but with -r4 for netcdf. Variable real must be set to
#      r4 or r8 at the beginning of the script below.
#
###########################################################################

### Functions
# TODO add make_j parallel argument -> check what is used for libigcm

function myget { # Get a file from LMDZ repository
  local url=$1

  local filename
  filename=$(basename "$url")

  wget --no-check-certificate -nv "http://lmdz.lmd.jussieu.fr/pub/$url"
  if [[ $filename =~ .*\.(ba)?sh ]]; then chmod +x "$filename"; fi
}

function set_default_params {
    # Valeur par défaut des parametres
    svn=""
    version="20231022.trunk"

    netcdf=1
    bench=1
    pcmac=0 # default: not on a Mac
    SCM=0
    veget="none"
    grid_resolution="32x32x39"
    physiq=""

    xios_branch="2.6"
    xios_rev="2568"

    parallel="none"
    trusting="testing"
    MODEL=""

    with_xios=0
    opt_makelmdz_xios=""

    rad="rrtm"

    compile_with_fcm=1

    cosp="none"
    opt_cosp=""

    make_j=8

    # Check if on a Mac
    if [[ $(uname) = "Darwin" ]]; then
        pcmac=1
        export MAKE="make"
    fi

    optim_debug=""
    arch="local"

    local mpi_file
    mpi_file=$(readlink -f "$(which mpif90)")
    path_mpi=$(dirname "$mpi_file")
    root_mpi=$(dirname "$path_mpi")

    arch="local-gfortran"
    arch_dir=""
}

function read_cmdline_args {
    while (($# > 0)); do
        case $1 in
            "-h") cat <<........fin
        $0 [ -v version ] [ -r svn_release ]
               [ -parallel PARA ] [ -d GRID_RESOLUTION ] [ -bench 0/1 ]
               [-name LOCAL_MODEL_NAME] [-gprof] [-opt_makelmdz] [-rad RADIATIF]

        -v       "version" like 20150828.trunk, see http://www.lmd.jussieu.fr/~lmdz/Distrib/LISMOI.trunk (default <$version>)

        -r       "svn_release" : either the svn release number or "last" (default <$svn>)

        -parallel parallel support: mpi, omp, mpi_omp (mpi with openMP) or none (default: <$parallel>)

        -d        "grid resolution": should be among the available benchs if -bench 1 (valid values: 48x36x19, 48x36x39) (default : <$grid_resolution>)

        -bench     activating the bench or not (0/1) (default: <$bench>)

        -testing/unstable

        -name      name of the folder to install to (default <$MODEL>)

        -netcdf    0, 1 or PATH. 0: do not download NetCDF, look for it in standard locations (/usr);  1: download and compile NetCDF; PATH: full path to an existing installed NetCDF library (default: <$netcdf>)

        -xios      use (download and compile) the XIOS library (will compile the parallel NetCDF4-HDF5 library) (requires to also have -parallel mpi_omp)

        -cosp       to run with cospv1 or cospv2 [none/v1/v2] (default <$cosp>)

        -rad        radiative code: oldrad, rrtm or ecrad (default <$rad>)

        -nofcm      to compile without fcm

        -SCM        install 1D version automatically

        -debug      compile everything in debug mode

        -physiq     to choose which physics package to use (default <$physiq>)

        -veget      surface/vegetation scheme treatment controlled by the single variable veget which can have the following values: none: bucket scheme (default); CMIP6: orchidee version used in CMIP exercise, rev 5661; number: orchidee version number (default $veget)

        -arch       name of the arch to use (default <$arch>)

        -arch_dir   where to find the arch files (default <$arch_dir>)

        -make_j     number of processes to parallelize installations (default <$make_j>)

........fin
                  exit 1;;
            "-v") version=$2; shift; shift;;
            "-r") svn=$2; shift; shift;;
            "-d") grid_resolution=$2; shift; shift;;
            "-unstable"|"-testing") trusting=$(echo "$1" | cut -c2-); shift;;
            "-cosp") cosp=$2
                     case $cosp in
                         "none"|"v1"|"v2") cosp=$2; shift; shift;;
                         *) echo "Only none v1 v2 for cosp option"; exit 1
                     esac;;
            "-nofcm") compile_with_fcm=0; echo "This option will be reactivated soon (promesse du 8dec2022)"; exit 1;  shift;;
            "-SCM") SCM=1; shift;;
            "-rad") rad=$2
                    case $rad in
                        "oldrad"|"rrtm"|"ecrad") rad=$2; shift; shift;;
                        *) echo "Only oldrad rrtm ecrad for rad option"; exit 1
                    esac;;
            "-parallel") parallel=$2
                         case $parallel in
                             "none"|"mpi"|"omp"|"mpi_omp") parallel=$2; shift; shift;;
                             *) echo "Only none mpi omp mpi_omp for the parallel option"; exit 1
                         esac;;
            "-bench") bench=$2; shift; shift;;
            "-debug") optim_debug=-debug; shift;;
            "-name") MODEL=$2; shift; shift;;
            "-netcdf") netcdf=$2; shift; shift;;
            "-physiq") physiq=$2; shift; shift;;
            "-xios") with_xios=1; shift;;
            "-arch") arch=$2; shift; shift;;
            "-arch_dir") arch_dir=$2; shift; shift;;
            "-veget") veget=$2; shift; shift;;
            "-make_j") make_j=$2; shift; shift;;
            *)  bash install_lmdz.sh -h; exit 1
        esac
    done

    # Option de compilation pour Cosp
    case $cosp in
        v1) opt_cosp="-cosp true";;
        v2) opt_cosp="-cospv2 true";;
        *) opt_cosp=""
    esac

    # Check on veget version
    if [[ $veget != 'none' && $veget != "CMIP6" ]]; then
        re='^[0-9]+$'
        if ! [[ $veget =~ $re ]]; then
            echo 'Valeur de l option veget non valable'; exit 1
        fi
    fi

    #Define veget-related suffix for gcm name
    if [[ $veget = 'none' ]]; then
        suff_orc=''
        #For use with tutorial, orchidee_rev is also defined (will be
        #written in surface_env at the end of the script)
        orchidee_rev=''
    else
        suff_orc='_orch'
    fi


    if [[ $parallel = "none" ]]; then
        suff_exe='_seq'
    else
        suff_exe='_para_mem'
    fi

    #Chemin pour placer le modele
    if [[ $MODEL = "" ]]; then MODEL="./LMDZ$version$svn$optim_debug"; fi

    ## if also compiling XIOS, parallel must be mpi_omp
    if [[ $with_xios = 1 && $parallel != "mpi_omp" ]]; then
        echo "Error, you must set -parallel mpi_omp if you want XIOS"; exit 1
    fi

    if [[ $with_xios = 1 ]]; then
        opt_makelmdz_xios="-io xios"
    fi

    if [[ $cosp = "v2" && $with_xios = 0 ]]; then
        echo "Error, Cospv2 cannot run without Xios"; exit 1
    fi

    if [[ $arch_dir = "" ]]; then
        arch_dir="$MODEL/modipsl/config/IPSLCM7/ARCH/"
    fi
    arch_dir=$(readlink -f "$arch_dir")  # full path
}

function check_available_software {
    echo "################################################################"
    echo "Check if required software is available"
    echo "################################################################"
    for logiciel in wget tar gzip make gcc cmake m4 c++; do
        if [[ $(which "$logiciel") = "" ]]; then
            echo "You must first install $logiciel on your system"; exit 1
        fi
    done

    cat <<eod > tt.f90
print*,'coucou'
end
eod
    $compiler tt.f90 || a.out
    ./a.out >| tt
    if [[ $(< tt sed -e 's/ //g' ) != "coucou" ]]; then
        echo "problem installing with compiler $compiler"; exit 1
    fi
    \rm tt a.out tt.f90
}

function download_model {
    mkdir -p "$MODEL"
    MODEL=$(readlink -f "$MODEL"); echo "$MODEL"  # absolute path

    rm -rf "$MODEL/modipsl"

    echo "##########################################################"
    echo "Download a slightly modified version of  LMDZ"
    echo "##########################################################"
    cd "$MODEL"
    getlog="$(pwd)/get.log"
    echo "logfile : $getlog"
    myget "src_archives/$trusting/modipsl.$version.tar.gz" >> get.log 2>&1
    echo "install_lmdz.sh wget_OK $(date)"
    gunzip "modipsl.$version.tar.gz" >> get.log 2>&1
    tar xf "modipsl.$version.tar" >> get.log 2>&1
    \rm "modipsl.$version.tar"

    if [[ $svn != "" ]]; then
        mysvn=$svn
    else
        mysvn="$(grep 'Revision: [0-9]' "$MODEL"/Read*.md | awk ' { print $2 } ' 2>/dev/null)"
    fi
    echo "SVN revision used: $mysvn"
}

function download_model_AMAURY {  # TODO
    MODEL=$(readlink -f "$MODEL"); echo "$MODEL"  # absolute path

    if [[ $svn != "" ]]; then
        mysvn=$svn
    else
        mysvn="$(grep 'Revision: [0-9]' "$MODEL"/Read*.md | awk ' { print $2 } ' 2>/dev/null)"
    fi
    echo "SVN revision used: $mysvn"
}

function install_arch {
    if [[ ! -d $arch_dir ]]; then
        echo "Error: no arch dir $arch_dir"; exit 1
    elif [[ ! -f $arch_dir/arch-$arch.path ]]; then
        echo "Error: no arch-$arch.path in $arch_dir"; exit 1
    fi

    local fcm_path
    if [[ -f $arch_dir/arch-$arch.fcm ]]; then
        fcm_path="$arch_dir/arch-$arch.fcm"
    else
        fcm_path="$MODEL/modipsl/modeles/LMDZ/arch/arch-$arch.fcm"
        if [[ ! -f $fcm_path ]]; then
            echo "Error: no $fcm_path"; exit 1
        fi
    fi

    # check compiler
    compiler=$(< "$fcm_path" grep "%COMPILER" | sed -e "s/%COMPILER\s*//")

    # Copy arch to different components
    # TODO check that this doesn't error if the source and destination are the same
    for dir in "IOIPSL" "ORCHIDEE" "LMDZ" "XIOS"; do
        \cp -f "$arch_dir/arch-$arch.fcm" "$MODEL/modipsl/modeles/$dir/arch"
        \cp -f "$arch_dir/arch-$arch.path" "$MODEL/modipsl/modeles/$dir/arch"
    done

    # TODO TEMP compat w/ old orch version - make a proper case later
    sed -i'' -e "s/-I//" -e "s/-L//" "$MODEL/modipsl/modeles/ORCHIDEE/arch/arch-$arch.path"  # /!\ we only replace first occurence on purpose

    # Adapt for XIOS, which uses different naming conventions
    # TODO >:c ugly fdefault-real-8 fix
    sed -i'' -e "s/%COMPILER/%FCOMPILER/" -e "s/%LINK/%LINKER/" -e "s/-fdefault-real-8//" "$MODEL/modipsl/modeles/XIOS/arch/arch-$arch.fcm"
}

function install_netcdf {
    echo "Installing Netcdf"

    if [[ $netcdf = 0 ]]; then
        ncdfdir=$(nc-config --prefix)
    else
        cd "$MODEL"

        case $compiler in
          gfortran) opt1="-compiler gnu"; opt2="-CC gcc -FC gfortran -CXX g++";;
          ifort)  opt1="-compiler intel"; opt2="-CC icc -FC ifort -CXX icpc";;
          pgf90)  opt1="-compiler pgf90"; opt2="-CC pgcc -FC pgf90 -CXX pgCC";;
          *)      echo "unexpected compiler $compiler for netcdf"; exit 1
        esac

        case $with_xios in
            0) script_install_netcdf="install_netcdf4_hdf5_seq.bash"
               ncdfdir="netcdf4_hdf5_seq"
               opt_="$opt1";;
            1) script_install_netcdf="install_netcdf4_hdf5.bash"
               ncdfdir="netcdf4_hdf5"
               opt_="$opt2 -MPI $root_mpi";;
            *) echo "with_xios=$with_xios, should be 0 or 1"; exit 1
        esac
        if [[ $netcdf = 1 ]]; then
           ncdfdir="$MODEL/$ncdfdir"
        else
           mkdir -p "$netcdf"; ncdfdir="$netcdf/$ncdfdir"
        fi

        echo "Repertoire netcdf $ncdfdir"
        if [[ ! -d $ncdfdir ]]; then
            netcdflog=$(pwd)/netcdf.log
            echo "----------------------------------------------------------"
            echo "Compiling the Netcdf library"
            echo "----------------------------------------------------------"
            echo "log file : $netcdflog"
            myget script_install/$script_install_netcdf >> "$netcdflog" 2>&1
            chmod u=rwx $script_install_netcdf
            # shellcheck disable=SC2086
            ./$script_install_netcdf -prefix "$ncdfdir" $opt_ >> "$netcdflog" 2>&1
        fi

        #----------------------------------------------------------------------------
        # LF rajout d'une verrue, pour une raison non encore expliquee,
        # la librairie est parfois rangée dans lib64 et non dans lib
        # par certains compilateurs
        if [[ ! -e lib && -d lib64 ]]; then ln -s lib64 lib; fi
        #----------------------------------------------------------------------------

        echo "install_lmdz.sh netcdf_OK $(date)"
    fi

    cat >test_netcdf90.f90 <<EOF
    use netcdf
    print *, "NetCDF library version: ", nf90_inq_libvers()
    end
EOF

    if $compiler -I"$ncdfdir"/include test_netcdf90.f90 -L"$ncdfdir"/lib -lnetcdff \
              -lnetcdf -Wl,-rpath="$ncdfdir"/lib && ./a.out
    then
        \rm test_netcdf90.f90 a.out
    else
        cat <<EOF
Failed test program using NetCDF-Fortran. You can:
- check that you have NetCDF-Fortran installed in your system
- or specify an installation directory with option -netcdf of install_lmdz.sh
- or download and compile NetCDF-Fortran with option -netcdf 1 of nstall_lmdz.sh
EOF
        exit 1
    fi

    # Compile NetCDF95
    cd "$MODEL/modipsl/modeles/LMD"*
    echo "Installing NetCDF95"
    cd "$MODEL"
    myget src_archives/netcdf/NetCDF95-0.3.tar.gz
    tar -xf NetCDF95-0.3.tar.gz
    \rm NetCDF95-0.3.tar.gz
    cd NetCDF95-0.3
    mkdir -p build && cd build
    netCDF_INCLUDE_DIR=$(nc-config --includedir) netCDF_LIBRARY=$(nc-config --libdir) cmake .. -DCMAKE_BUILD_TYPE=Release -DCMAKE_PREFIX_PATH="$ncdfdir" -DCMAKE_INSTALL_PREFIX="$MODEL/modipsl"
    make install
}

function install_IOIPSL {  # via fcm
    cd "$MODEL/modipsl/modeles/IOIPSL"
    ioipsllog="$(pwd)/ioipsl.log"
    echo "Compiling IOIPSL, the interface library with Netcdf $(date) (log: $ioipsllog)"

    if ! ./makeioipsl_fcm -arch "$arch" -job 8 > "$ioipsllog" 2>&1; then
        echo "IOIPSL compile failed, exiting"; exit 1
    fi
    cp lib/* ../../lib
    echo "IOIPSL compiled $(date)"

    IOIPSL_LIBDIR_="$MODEL/modipsl/modeles/IOIPSL/lib"
    IOIPSL_INCDIR_="$MODEL/modipsl/modeles/IOIPSL/inc"
}

function install_XIOS {
    if [[ $with_xios = 1 ]]; then
        cd "$MODEL/modipsl/modeles"
        xioslog="$(pwd)/XIOS/xios.log"
        echo "##########################################################"
        echo "Compiling XIOS (log $xioslog) $(date)"
        echo "##########################################################"

        # Download XIOS
        case $xios_branch in
            "trunk")
            xios_http="http://forge.ipsl.jussieu.fr/ioserver/svn/XIOS/trunk";;
            "2.5")
            xios_http="http://forge.ipsl.jussieu.fr/ioserver/svn/XIOS/branchs/xios-2.5";;
            "2.6")
            xios_http="http://forge.ipsl.jussieu.fr/ioserver/svn/XIOS2/branches/xios-2.6";;
            *) echo "XIOS: Error, bad argument for -branch ! Did not expect $xios_branch"; exit;;
        esac
        
        cd "$MODEL/modipsl/modeles"
        set +e; svn co --revision $xios_rev $xios_http XIOS; set -e

        cd XIOS
        if ! ./make_xios --arch "$arch" --job 8 > "$xioslog" 2>&1; then
            echo "XIOS compilation failed, exiting"; exit 1
        fi

        echo "Compiled XIOS $(date)"
    fi

    XIOS_LIBDIR_="$MODEL/modipsl/modeles/XIOS/lib"
    XIOS_INCDIR_="$MODEL/modipsl/modeles/XIOS/inc"
}

function get_orchidee_version {  # Set / Check ORCHIDEE version
    if [[ $veget = "none" ]]; then
        veget_version="false"
    elif [[ $veget = "CMIP6" ]]; then
        veget_version=orchidee2.0
    else # specific orchidee revision newer than CMIP6, on 2_1 or 2_2 branches
        veget_version=orchidee2.1
        if [[ $veget -lt 4465 ]]; then
            echo 'ORCHIDEE version must be >=4465, exiting'
            exit 1
        fi

        set +e
        svn upgrade
        local orcbranch
        orcbranch=$(svn log -v -q svn://forge.ipsl.jussieu.fr/orchidee/ -r "$veget" |grep ORCHIDEE |head -1| sed -e 's:ORCHIDEE/.*$:ORCHIDEE:' | awk '{print $2}')
        echo "IF YOU INSTALL ORCHIDEE THE VERY FIRST TIME, ASK for PASSWORD at orchidee-help@listes.ipsl.fr"
        svn switch -r "$veget" --accept theirs-full "svn://forge.ipsl.jussieu.fr/orchidee/$orcbranch"
        svn log -r "$veget" | grep "$veget"
        if ! svn log -r "$veget" | grep "$veget"; then
            echo 'Cannot update ORCHIDEE as not on the right branch for ORCHIDEE'
            exit 1
        fi
        svn update -r "$veget"
        set -e
    fi
}

function compile_orchidee {
    get_orchidee_version
    ORCHPATH=""


    if [[ $veget != "none" ]]; then
        cd "$MODEL/modipsl/modeles/ORCHIDEE"
        ORCHPATH=$(pwd)

        orchideelog="$(pwd)/orchidee.log"
        echo "Compiling ORCHIDEE, the continental surface model (log $orchideelog) $(date)"

        local xios_orchid opt_orc
        if [[ $with_xios = 1 ]]; then
            xios_orchid="-xios";
        else
            xios_orchid="-noxios"
        fi
        if [[ $optim_debug = "-debug" ]]; then
            opt_orc="-debug";
        else
            opt_orc="-prod"
        fi

        # TODO check that this works with old orchidee versions (cf. -d tools)
        local varenv="IOIPSL_LIBDIR_=$IOIPSL_LIBDIR_ IOIPSL_INCDIR_=$IOIPSL_INCDIR_ ORCHPATH=$ORCHPATH"
        if [[ $parallel = "none" ]]; then
            echo "$varenv ./makeorchidee_fcm $xios_orchid $opt_orc -parallel none -arch $arch -j $make_j" > compile.sh
        else
            if [[ ! -d src_parallel ]]; then
               echo "Orchidee version too old for parallel support"; exit 1
            fi
            {
                echo "$varenv ./makeorchidee_fcm -j $make_j -clean $xios_orchid $opt_orc -parallel $parallel -arch $arch"
                echo "$varenv ./makeorchidee_fcm -j $make_j $xios_orchid $opt_orc -parallel $parallel -arch $arch"
            } > compile.sh
        fi
        chmod +x compile.sh
        echo "Compiling ORCHIDEE using $(\cat compile.sh)"
        if ! ./compile.sh > "$orchideelog" 2>&1; then
            echo "ORCHIDEE compilation failed, exiting"; exit 1
        fi
        echo "Compiled ORCHIDEE $(date)"
    fi

    ORCH_LIBDIR_="$MODEL/modipsl/modeles/ORCHIDEE/lib"
}

function get_lmdz_version {
    LMDZPATH=$(readlink -f "$MODEL/modipsl/modeles/LMD"*)
    cd "$LMDZPATH"
    lmdzlog="$(pwd)/lmdz.log"

    set +e
    svn upgrade
    if [[ $svn = "last" ]]; then
        svnopt=""
    else
        svnopt="-r $svn"
    fi
    if [[ $svn != "" ]]; then
        if svn info | grep -q 'https:'; then svn switch --relocate https://svn.lmd.jussieu.fr/LMDZ http://svn.lmd.jussieu.fr/LMDZ; fi
        svn update "$svnopt"
    fi
    mysvn=$(svnversion . | grep -E -o "[0-9]+" 2>/dev/null)
    set -e

    if [[ $mysvn = "" ]]; then mysvn=$(grep 'Revision: [0-9]' "$MODEL"/Read*.md | awk ' { print $2 } ' 2>/dev/null); fi
    if [[ $mysvn = "" ]]; then mysvn=4190; fi
}

function compile_lmdz {
    get_lmdz_version

    exe_name="bin/gcm_${grid_resolution}_phylmd_${rad}${suff_exe}${suff_orc}.e"

    local opt_rad
    case $rad in
        oldrad) iflag_rrtm=0; NSW=2; opt_rad="";;
        rrtm)   iflag_rrtm=1; NSW=6
            if [[ $mysvn -le 4185 ]]; then
                opt_rad="-rrtm true"
            else
                opt_rad="-rad rrtm"
            fi;;
        ecrad)  iflag_rrtm=2; NSW=6; opt_rad="-rad ecrad";;
        *) echo "Only oldrad rrtm ecrad for rad option"; exit 1
    esac
    if [[ $mysvn -le 4185 && $rad = "ecrad" ]]; then
        echo "ecrad only available for LMDZ rev starting with 4186 "; exit 1
    fi

    # Compile
    local varenv="IOIPSL_LIBDIR_=$IOIPSL_LIBDIR_ IOIPSL_INCDIR_=$IOIPSL_INCDIR_ ORCH_LIBDIR_=$ORCH_LIBDIR_ ORCHPATH=$ORCHPATH XIOS_INCDIR_=$XIOS_INCDIR_ XIOS_LIBDIR_=$XIOS_LIBDIR_"
    makelmdz="makelmdz_fcm $optim_debug -arch $arch -j $make_j"
    if [[ $parallel = "none" ]]; then
        echo "$varenv ./$makelmdz $opt_rad $opt_cosp -d ${grid_resolution} -v $veget_version gcm " > compile.sh
    else
        echo "$varenv ./$makelmdz $opt_rad $opt_cosp $opt_makelmdz_xios -d ${grid_resolution} -v $veget_version -mem -parallel $parallel gcm" > compile.sh
    fi
    echo "Compiling lmdz using $(\cat compile.sh) (log: $lmdzlog) $(date)"
    chmod +x ./compile.sh
    if ! ./compile.sh > "$lmdzlog" 2>&1; then
        echo "LMDZ compilation failed, exiting"; exit 1
    fi
    echo "Finished LMDZ compilation $(date)"

    # Check executable
    # TODO $mysev -ge 4186 => other name convention ! check original install
    if [[ ! -f $exe_name ]]; then
        echo "Compilation failed, can't find the executable"; exit 1
    else
        echo "Compilation successfull, the executable is $exe_name $(date)"
    fi
}

function run_bench {
    cd "$MODEL/modipsl/modeles/LMDZ"*

    if [[ $bench = "tuto" ]]; then
        myget "Training/tutorial.tar"; tar xf tutorial.tar; cd TUTORIAL
        ./init.sh
    elif [[ $bench = 1 ]]; then
        \rm -rf "BENCH${grid_resolution}"
        local bench=bench_lmdz_${grid_resolution}
        myget "3DBenchs/$bench.tar.gz"
        tar xf "$bench.tar.gz"

        if [[ $cosp = "v1" || $cosp = "v2" ]]; then
            cd "BENCH${grid_resolution}"
            # copier les fichiers namelist input et output our COSP
            cp ../DefLists/cosp*_input_nl.txt .
            cp ../DefLists/cosp*_output_nl.txt .
            # Activer la cles ok_cosp pour tourner avec COSP
            sed -e 's@ok_cosp=n@ok_cosp=y@' config.def > tmp
            \mv -f tmp config.def
            cd ..
        fi

        if [[ -n "$physiq" ]]; then
            cd "BENCH${grid_resolution}"
            if [[ -f "physiq.def_${physiq}" ]]; then
                cp "physiq.def_${physiq}" physiq.def
                echo "using physiq.def_${physiq}"
            else
                echo "using standard physiq.def"
            fi
            cd ..
        else
            echo "using standard physiq.def"
        fi

        if [[ $with_xios = 1 ]]; then
            cd "BENCH${grid_resolution}"
            cp ../DefLists/iodef.xml .
            cp ../DefLists/context_lmdz.xml .
            cp ../DefLists/field_def_lmdz.xml .
            # A raffiner par la suite
            echo "A FAIRE : Copier les *xml en fonction de l option cosp"
            cp ../DefLists/field_def_cosp*.xml .
            cp ../DefLists/file_def_hist*xml .
            # adapt iodef.xml to use attached mode
            sed -e 's@"using_server" type="bool">true@"using_server" type="bool">false@' \
                iodef.xml > tmp
            \mv -f tmp iodef.xml

            # and convert all the enabled="_AUTO_" (for libIGCM) to enabled=.FALSE.
            # except for histday
            for histfile in file_def_hist*xml
            do
                if [[ "$histfile" = "file_def_histday_lmdz.xml" ]]; then
                    sed -e 's@enabled="_AUTO_"@type="one_file" enabled=".TRUE."@' \
                        "$histfile" > tmp
                    \mv -f tmp "$histfile"
                    sed -e 's@output_level="_AUTO_"@output_level="5"@' "$histfile" \
                        > tmp
                    \mv -f tmp "$histfile"
                    sed -e 's@compression_level="2"@compression_level="0"@' \
                        "$histfile" > tmp
                    \mv -f tmp "$histfile"
                else
                    sed -e 's@enabled="_AUTO_"@type="one_file" enabled=".FALSE."@' \
                        "$histfile" > tmp
                    \mv -f tmp "$histfile"
                fi
            done
            # and add option "ok_all_xml=y" in config.def
            echo "### XIOS outputs" >> config.def
            echo 'ok_all_xml=.true.' >> config.def

            #activer les sorties pour Cosp
            if [[ "$cosp" = "v1" ]]; then
                sed -i'' -e 's@enabled=".FALSE."@enabled=".TRUE."@' \
                         -e 's@output_level="_AUTO_"@output_level="5"@' \
                         -e 's@compression_level="2"@compression_level="0"@' \
                         file_def_histdayCOSP_lmdz.xml
            fi
            if [[ "$cosp" = "v2" ]]; then
                sed -e 's@compression_level="2"@compression_level="0"@' file_def_histdayCOSPv2_lmdz.xml
                for type_ in hf day mth; do
                    file=file_def_hist${type_}COSP
                    sed -i'' -e 's@src="./'${file}'_lmdz.xml"@src="./'${file}'v2_lmdz.xml"@' context_lmdz.xml
                done
                sed -i '' -e 's@field_def_cosp1.xml@field_def_cospv2.xml@' field_def_lmdz.xml
            fi

            cd ..
        fi

        # Cas Bench avec ecrad
        if [[ $rad = "ecrad" ]]; then
            cd "BENCH${grid_resolution}"
            cp  ../DefLists/namelist_ecrad .
            cp -r ../libf/phylmd/ecrad/data .
            cd ..
        fi

        # Adjusting bench physiq.def to radiative code chosen
        cd "BENCH${grid_resolution}"
        sed -e 's/iflag_rrtm=.*.$/iflag_rrtm='$iflag_rrtm'/' \
            -e 's/NSW=.*.$/NSW='$NSW'/' physiq.def > tmpdef
        \mv tmpdef physiq.def
        cd ..

        cp "$exe_name" "BENCH${grid_resolution}/gcm.e"
        cd "BENCH${grid_resolution}"

        if [[ ${parallel:0:3} = "mpi" ]]; then
            # Lancement avec deux procs mpi et 2 openMP
            echo "export OMP_STACKSIZE=800M" > bench.sh
            if [[ "${parallel:4:3}" = "omp" ]]; then
                echo "export OMP_NUM_THREADS=2" >> bench.sh
            fi
            if [[ "$cosp" = "v1" || "$cosp" = "v2" ]]; then
                echo "ulimit -s 200000" >> bench.sh
            else
                echo "ulimit -s unlimited" >> bench.sh
            fi
            echo "mpirun -np 2 gcm.e > listing  2>&1" >> bench.sh
            # Add rebuild, using reb.sh if it is there
            cat <<EOF >> bench.sh
    if [[ -f reb.sh ]]; then
      ./reb.sh histday; ./reb.sh histmth; ./reb.sh histhf;
      ./reb.sh histins; ./reb.sh stomate_history;
      ./reb.sh sechiba_history; ./reb.sh sechiba_out_2
    fi
EOF
        else
            echo "./gcm.e > listing  2>&1" > bench.sh
        fi
        # Getting orchidee stuff
        if [[ $veget = 'CMIP6' ]]; then
            echo 'myget 3DBenchs/BENCHCMIP6.tar.gz'
            myget 3DBenchs/BENCHCMIP6.tar.gz
            tar xvzf BENCHCMIP6.tar.gz
            sed -e "s:VEGET=n:VEGET=y:" config.def > tmp
            mv -f tmp config.def
            if [[ $with_xios = 1 ]]; then
                cp ../../ORCHIDEE/src_xml/context_orchidee.xml .
                echo '<context id="orchidee" src="./context_orchidee.xml"/>' > add.tmp
                cp ../../ORCHIDEE/src_xml/field_def_orchidee.xml .
                cp ../../ORCHIDEE/src_xml/file_def_orchidee.xml .
                cp ../../ORCHIDEE/src_xml/file_def_input_orchidee.xml .
                if [[ -f ../../ORCHIDEE/src_xml/context_input_orchidee.xml ]]; then
                       cp ../../ORCHIDEE/src_xml/context_input_orchidee.xml .
                       echo '<context id="orchidee" src="./context_input_orchidee.xml"/>' >> add.tmp
                fi
                sed -e '/id="LMDZ"/r add.tmp' iodef.xml > tmp
                mv tmp iodef.xml
                sed -e'{/sechiba1/ s/enabled="_AUTO_"/type="one_file" enabled=".TRUE."/}' \
                    file_def_orchidee.xml > tmp
                \mv -f tmp file_def_orchidee.xml
                sed -e 's@enabled="_AUTO_"@type="one_file" enabled=".FALSE."@' \
                    file_def_orchidee.xml > tmp
                \mv -f tmp file_def_orchidee.xml
                sed -e 's@output_level="_AUTO_"@output_level="1"@' \
                    file_def_orchidee.xml > tmp
                \mv -f tmp file_def_orchidee.xml
                sed -e 's@output_freq="_AUTO_"@output_freq="1d"@' \
                    file_def_orchidee.xml > tmp
                \mv -f tmp file_def_orchidee.xml
                sed -e 's@compression_level="4"@compression_level="0"@' \
                    file_def_orchidee.xml > tmp
                \mv -f tmp file_def_orchidee.xml
                sed -e 's@XIOS_ORCHIDEE_OK = n@XIOS_ORCHIDEE_OK = y@' \
                    orchidee.def > tmp
                \mv -f tmp orchidee.def
            fi
        fi

        if [[ -f ../arch.env ]]; then source ../arch.env; fi

        echo "EXECUTION DU BENCH"
        date
        if (! ./bench.sh > out.bench 2>&1) || ! (tail -n 1 listing | grep "Everything is cool"); then
            tail listing
            echo "Bench FAILED, exiting"; exit 1
        fi
        date
        tail listing
    fi

    # 1D case
    if [[ $SCM = 1 ]]; then
        cd "$MODEL"
        myget 1D/1D.tar.gz
        tar xf 1D.tar.gz
        cd 1D
        if [[ $rad = "oldrad" ]]; then
            sed -i'' -e 's/^rad=.*$/rad=oldrad/' run.sh
            sed -i'' -e 's/^rad=.*$/rad=oldrad/' bin/compile
        fi
        echo "Running 1D/run.sh, log in $(pwd)/run1d.log"
        ./run.sh > "$(pwd)/run1d.log" 2>&1
    fi
}

echo "install_lmdz.sh DEBUT $(date)"

set_default_params
read_cmdline_args "$@"
#download_model
download_model_AMAURY
install_arch
check_available_software
install_netcdf
install_IOIPSL
install_XIOS
compile_orchidee
compile_lmdz
run_bench