module lscp_ini_mod

implicit none

  ! PARAMETERS for lscp:
  !--------------------
 
  REAL RCPD, RLSTT, RLVTT, RLMLT, RVTMP2, RTT, RD, RG
  !$OMP THREADPRIVATE(RCPD, RLSTT, RLVTT, RLMLT, RVTMP2, RTT, RD, RG)

  REAL, SAVE :: seuil_neb=0.001                 ! cloud fraction threshold: a cloud really exists when exceeded
  !$OMP THREADPRIVATE(seuil_neb)

  INTEGER, SAVE :: niter_lscp=5                 ! number of iterations to calculate autoconversion to precipitation
  !$OMP THREADPRIVATE(niter_lscp)

  INTEGER,SAVE :: iflag_evap_prec=1             ! precipitation evaporation flag. 0: nothing, 1: "old way", 
                                                ! 2: Max cloud fraction above to calculate the max of reevaporation
                                                ! 4: LTP'method i.e. evaporation in the clear-sky fraction of the mesh only
  !$OMP THREADPRIVATE(iflag_evap_prec)

  REAL t_coup                                   ! temperature threshold which determines the phase 
  PARAMETER (t_coup=234.0)                      ! for which the saturation vapor pressure is calculated

  REAL DDT0                                     ! iteration parameter 
  PARAMETER (DDT0=.01)

  REAL ztfondue                                 ! parameter to calculate melting fraction of precipitation
  PARAMETER (ztfondue=278.15)

  REAL, SAVE    :: rain_int_min=0.001           ! Minimum local rain intensity [mm/s] before the decrease in associated precipitation fraction 
  !$OMP THREADPRIVATE(rain_int_min)

  REAL, SAVE :: a_tr_sca(4)                     ! Variables for tracers temporary: alpha parameter for scavenging, 4 possible scavenging processes
  !$OMP THREADPRIVATE(a_tr_sca)
  
  INTEGER, SAVE ::  iflag_mpc_bl=0              ! flag to activate boundary layer mixed phase cloud param
  !$OMP THREADPRIVATE(iflag_mpc_bl)
  
  LOGICAL, SAVE :: ok_radocond_snow=.false.       ! take into account the mass of ice precip in the cloud ice content seen by radiation
  !$OMP THREADPRIVATE(ok_radocond_snow)

  REAL, SAVE :: t_glace_min=258.0                ! lower-bound temperature parameter for cloud phase determination
  !$OMP THREADPRIVATE(t_glace_min)

  REAL, SAVE :: t_glace_max=273.15               ! upper-bound temperature parameter for cloud phase determination
  !$OMP THREADPRIVATE(t_glace_max)

  REAL, SAVE :: exposant_glace=1.0               ! parameter for cloud phase determination
  !$OMP THREADPRIVATE(exposant_glace)

  INTEGER, SAVE :: iflag_vice=0                  ! which expression for ice crystall fall velocity
  !$OMP THREADPRIVATE(iflag_vice)

  INTEGER, SAVE :: iflag_t_glace=0               ! which expression for cloud phase partitioning
  !$OMP THREADPRIVATE(iflag_t_glace)

  INTEGER, SAVE :: iflag_cloudth_vert=0          ! option for determining cloud fraction and content in convective boundary layers
  !$OMP THREADPRIVATE(iflag_cloudth_vert)

  INTEGER, SAVE :: iflag_gammasat=0              ! which threshold for homogeneous nucleation below -40oC
  !$OMP THREADPRIVATE(iflag_gammasat)

  INTEGER, SAVE :: iflag_rain_incloud_vol=0      ! use of volume cloud fraction for rain autoconversion 
  !$OMP THREADPRIVATE(iflag_rain_incloud_vol)

  INTEGER, SAVE :: iflag_bergeron=0              ! bergeron effect for liquid precipitation treatment  
  !$OMP THREADPRIVATE(iflag_bergeron)

  INTEGER, SAVE :: iflag_fisrtilp_qsat=0         ! qsat adjustment (iterative) during autoconversion
  !$OMP THREADPRIVATE(iflag_fisrtilp_qsat)

  INTEGER, SAVE :: iflag_pdf=0                   ! type of subgrid scale qtot pdf
  !$OMP THREADPRIVATE(iflag_pdf)

  INTEGER, SAVE :: iflag_autoconversion=0        ! autoconversion option
  !$OMP THREADPRIVATE(iflag_autoconversion)

  LOGICAL, SAVE :: reevap_ice=.false.            ! no liquid precip for T< threshold
  !$OMP THREADPRIVATE(reevap_ice)

  REAL, SAVE :: cld_lc_lsc=2.6e-4                ! liquid autoconversion coefficient, stratiform rain
  !$OMP THREADPRIVATE(cld_lc_lsc)

  REAL, SAVE :: cld_lc_con=2.6e-4                ! liquid autoconversion coefficient, convective rain
  !$OMP THREADPRIVATE(cld_lc_con)

  REAL, SAVE :: cld_tau_lsc=3600.                ! liquid autoconversion timescale, stratiform rain
  !$OMP THREADPRIVATE(cld_tau_lsc)

  REAL, SAVE :: cld_tau_con=3600.                ! liquid autoconversion timescale, convective rain
  !$OMP THREADPRIVATE(cld_tau_con)

  REAL, SAVE :: cld_expo_lsc=2.                  ! liquid autoconversion threshold exponent, stratiform rain
  !$OMP THREADPRIVATE(cld_expo_lsc)

  REAL, SAVE :: cld_expo_con=2.                  ! liquid autoconversion threshold exponent, convective rain
  !$OMP THREADPRIVATE(cld_expo_con)

  REAL, SAVE :: ffallv_lsc=1.                    ! tuning coefficient crystal fall velocity, stratiform 
  !$OMP THREADPRIVATE(ffallv_lsc)

  REAL, SAVE :: ffallv_con=1.                    ! tuning coefficient crystal fall velocity, convective
  !$OMP THREADPRIVATE(ffallv_con)

  REAL, SAVE :: coef_eva=2e-5                    ! tuning coefficient liquid precip evaporation
  !$OMP THREADPRIVATE(coef_eva)

  REAL, SAVE :: coef_eva_i                       ! tuning coefficient ice precip sublimation
  !$OMP THREADPRIVATE(coef_eva_i)

  REAL cice_velo                                 ! factor in the ice fall velocity formulation
  PARAMETER (cice_velo=1.645)

  REAL dice_velo                                 ! exponent in the ice fall velocity formulation
  PARAMETER (dice_velo=0.16)

  REAL, SAVE :: dist_liq=300.                    ! typical deph of cloud-top liquid layer in mpcs
  !$OMP THREADPRIVATE(dist_liq)

  REAL, SAVE    :: tresh_cl=0.0                  ! cloud fraction threshold for cloud top search
  !$OMP THREADPRIVATE(tresh_cl)

CONTAINS

SUBROUTINE lscp_ini(dtime,ok_ice_sursat, RCPD_in, RLSTT_in, RLVTT_in, RLMLT_in, &
                    RVTMP2_in, RTT_in,RD_in,RG_in)


   USE ioipsl_getin_p_mod, ONLY : getin_p
   USE print_control_mod, ONLY: prt_level, lunout
   USE ice_sursat_mod, ONLY: ice_sursat_init

   REAL, INTENT(IN)      :: dtime
   LOGICAL, INTENT(IN)   :: ok_ice_sursat  

   REAL, INTENT(IN)      :: RCPD_in, RLSTT_in, RLVTT_in, RLMLT_in
   REAL, INTENT(IN)      ::  RVTMP2_in, RTT_in, RD_in, RG_in
   character (len=20) :: modname='lscp_ini_mod'
   character (len=80) :: abort_message


    RG=RG_in
    RD=RD_in
    RCPD=RCPD_in
    RLVTT=RLVTT_in
    RLSTT=RLSTT_in
    RLMLT=RLMLT_in
    RTT=RTT_in
    RG=RG_in
    RVTMP2=RVTMP2_in



    CALL getin_p('niter_lscp',niter_lscp)
    CALL getin_p('iflag_evap_prec',iflag_evap_prec)
    CALL getin_p('seuil_neb',seuil_neb)
    CALL getin_p('rain_int_min',rain_int_min) 
    CALL getin_p('iflag_mpc_bl',iflag_mpc_bl)
    CALL getin_p('ok_radocond_snow',ok_radocond_snow) 
    CALL getin_p('t_glace_max',t_glace_max)
    CALL getin_p('t_glace_min',t_glace_min)
    CALL getin_p('exposant_glace',exposant_glace) 
    CALL getin_p('iflag_vice',iflag_vice)
    CALL getin_p('iflag_t_glace',iflag_t_glace)
    CALL getin_p('iflag_cloudth_vert',iflag_cloudth_vert)
    CALL getin_p('iflag_gammasat',iflag_gammasat)
    CALL getin_p('iflag_rain_incloud_vol',iflag_rain_incloud_vol)
    CALL getin_p('iflag_bergeron',iflag_bergeron)
    CALL getin_p('iflag_fisrtilp_qsat',iflag_fisrtilp_qsat)
    CALL getin_p('iflag_pdf',iflag_pdf)
    CALL getin_p('reevap_ice',reevap_ice)
    CALL getin_p('cld_lc_lsc',cld_lc_lsc)
    CALL getin_p('cld_lc_con',cld_lc_con)
    CALL getin_p('cld_tau_lsc',cld_tau_lsc)
    CALL getin_p('cld_tau_con',cld_tau_con)
    CALL getin_p('cld_expo_lsc',cld_expo_lsc)
    CALL getin_p('cld_expo_con',cld_expo_con)
    CALL getin_p('ffallv_lsc',ffallv_lsc)
    CALL getin_p('ffallv_lsc',ffallv_con)
    CALL getin_p('coef_eva',coef_eva)
    coef_eva_i=coef_eva
    CALL getin_p('coef_eva_i',coef_eva_i)
    CALL getin_p('iflag_autoconversion',iflag_autoconversion)
    CALL getin_p('dist_liq',dist_liq)
    CALL getin_p('tresh_cl',tresh_cl)




    WRITE(lunout,*) 'lscp, niter_lscp:', niter_lscp
    WRITE(lunout,*) 'lscp, iflag_evap_prec:', iflag_evap_prec
    WRITE(lunout,*) 'lscp, seuil_neb:', seuil_neb
    WRITE(lunout,*) 'lscp, rain_int_min:', rain_int_min
    WRITE(lunout,*) 'lscp, iflag_mpc_bl:', iflag_mpc_bl
    WRITE(lunout,*) 'lscp, ok_radocond_snow:', ok_radocond_snow
    WRITE(lunout,*) 'lscp, t_glace_max:', t_glace_max
    WRITE(lunout,*) 'lscp, t_glace_min:', t_glace_min
    WRITE(lunout,*) 'lscp, exposant_glace:', exposant_glace
    WRITE(lunout,*) 'lscp, iflag_vice:', iflag_vice
    WRITE(lunout,*) 'lscp, iflag_t_glace:', iflag_t_glace
    WRITE(lunout,*) 'lscp, iflag_cloudth_vert:', iflag_cloudth_vert
    WRITE(lunout,*) 'lscp, iflag_gammasat:', iflag_gammasat
    WRITE(lunout,*) 'lscp, iflag_rain_incloud_vol:', iflag_rain_incloud_vol
    WRITE(lunout,*) 'lscp, iflag_bergeron:', iflag_bergeron
    WRITE(lunout,*) 'lscp, iflag_fisrtilp_qsat:', iflag_fisrtilp_qsat
    WRITE(lunout,*) 'lscp, iflag_pdf', iflag_pdf
    WRITE(lunout,*) 'lscp, reevap_ice', reevap_ice
    WRITE(lunout,*) 'lscp, cld_lc_lsc', cld_lc_lsc
    WRITE(lunout,*) 'lscp, cld_lc_con', cld_lc_con
    WRITE(lunout,*) 'lscp, cld_tau_lsc', cld_tau_lsc
    WRITE(lunout,*) 'lscp, cld_tau_con', cld_tau_con
    WRITE(lunout,*) 'lscp, cld_expo_lsc', cld_expo_lsc
    WRITE(lunout,*) 'lscp, cld_expo_con', cld_expo_con
    WRITE(lunout,*) 'lscp, ffallv_lsc', ffallv_lsc
    WRITE(lunout,*) 'lscp, ffallv_con', ffallv_con
    WRITE(lunout,*) 'lscp, coef_eva', coef_eva
    WRITE(lunout,*) 'lscp, coef_eva_i', coef_eva_i
    WRITE(lunout,*) 'lscp, iflag_autoconversion', iflag_autoconversion
    WRITE(lunout,*) 'lscp, dist_liq', dist_liq
    WRITE(lunout,*) 'lscp, tresh_cl', tresh_cl





    ! check for precipitation sub-time steps
    IF (ABS(dtime/REAL(niter_lscp)-360.0).GT.0.001) THEN
        WRITE(lunout,*) 'lscp: it is not expected, see Z.X.Li', dtime
        WRITE(lunout,*) 'I would prefer a 6 min sub-timestep'
    ENDIF

    ! check consistency between numerical resolution of autoconversion 
    ! and other options
   
    IF (iflag_autoconversion .EQ. 2) THEN
        IF ((iflag_vice .NE. 0) .OR. (niter_lscp .GT. 1)) THEN
           abort_message = 'in lscp, iflag_autoconversion=2 requires iflag_vice=0 and niter_lscp=1'
           CALL abort_physic (modname,abort_message,1) 
        ENDIF 
    ENDIF


    !AA Temporary initialisation
    a_tr_sca(1) = -0.5
    a_tr_sca(2) = -0.5
    a_tr_sca(3) = -0.5
    a_tr_sca(4) = -0.5
    
    IF (ok_ice_sursat) CALL ice_sursat_init()



end subroutine lscp_ini

end module lscp_ini_mod
