#ifdef ISO
! $Id: $

MODULE isotopes_mod
   USE strings_mod,  ONLY: msg, real2str, int2str, bool2str, maxlen, strIdx, strStack
   USE infotrac_phy, ONLY: isoName
   IMPLICIT NONE
   INTERFACE get_in; MODULE PROCEDURE getinp_s, getinp_i, getinp_r, getinp_l;  END INTERFACE get_in
   SAVE

  !--- Contains all isotopic variables + their initialization
  !--- Isotopes-specific routines are in isotopes_routines_mod to avoid circular dependencies with isotopes_verif_mod.

   !--- Isotopes indices (in [1,niso] ; non-existing => 0 index)
   INTEGER, SAVE :: iso_eau, iso_HDO, iso_O18, iso_O17, iso_HTO
!$OMP THREADPRIVATE(iso_eau, iso_HDO, iso_O18, iso_O17, iso_HTO)

   INTEGER, SAVE :: ntracisoOR
!$OMP THREADPRIVATE(ntracisoOR)

   !--- Variables not depending on isotopes
   REAL,    SAVE :: pxtmelt, pxtice, pxtmin, pxtmax
!$OMP THREADPRIVATE(pxtmelt, pxtice, pxtmin, pxtmax)
   REAL,    SAVE :: tdifexp, tv0cin, thumxt1
!$OMP THREADPRIVATE(tdifexp, tv0cin, thumxt1)
   INTEGER, SAVE :: ntot 
!$OMP THREADPRIVATE(ntot)
   REAL,    SAVE :: h_land_ice
!$OMP THREADPRIVATE(h_land_ice)
   REAL,    SAVE :: P_veg
!$OMP THREADPRIVATE(P_veg)
   REAL,    SAVE :: musi, lambda_sursat
!$OMP THREADPRIVATE(musi, lambda_sursat)
   REAL,    SAVE :: Kd
!$OMP THREADPRIVATE(Kd)
   REAL,    SAVE :: rh_cste_surf_cond, T_cste_surf_cond
!$OMP THREADPRIVATE(rh_cste_surf_cond, T_cste_surf_cond)
   LOGICAL, SAVE :: bidouille_anti_divergence    ! T: regularly, xteau <- q to avoid slow drifts
!$OMP THREADPRIVATE(bidouille_anti_divergence)
   LOGICAL, SAVE :: essai_convergence            ! F: as in LMDZ without isotopes (bad for isotopes)
!$OMP THREADPRIVATE(essai_convergence)
   INTEGER, SAVE :: initialisation_iso           ! 0: file ; 1: R=0 ; 2: R=distill. Rayleigh ; 3: R=Rsmow
!$OMP THREADPRIVATE(initialisation_iso)
   INTEGER, SAVE :: modif_SST                    ! 0: default ; 1: modified SST ; 2, 3: SST profiles
!$OMP THREADPRIVATE(modif_SST)
   REAL,    SAVE :: deltaTtest                   ! Uniform modification of the SST
!$OMP THREADPRIVATE(deltaTtest)
   INTEGER, SAVE :: modif_sic                    ! Holes in the Sea Ice
!$OMP THREADPRIVATE(modif_sic)
   REAL,    SAVE :: deltasic                     ! Minimal holes fraction
!$OMP THREADPRIVATE(deltasic)
   REAL,    SAVE :: deltaTtestpoles
!$OMP THREADPRIVATE(deltaTtestpoles)
   REAL,    SAVE :: sstlatcrit, dsstlatcrit
!$OMP THREADPRIVATE(sstlatcrit, dsstlatcrit)
   REAL,    SAVE :: deltaO18_oce
!$OMP THREADPRIVATE(deltaO18_oce)
   INTEGER, SAVE :: albedo_prescrit              ! 0: default ; 1: constant albedo
!$OMP THREADPRIVATE(albedo_prescrit)
   REAL,    SAVE :: lon_min_albedo, lon_max_albedo, lat_min_albedo, lat_max_albedo
!$OMP THREADPRIVATE(lon_min_albedo, lon_max_albedo, lat_min_albedo, lat_max_albedo)
   REAL,    SAVE :: deltaP_BL,tdifexp_sol
!$OMP THREADPRIVATE(deltaP_BL,tdifexp_sol)
   INTEGER, SAVE :: ruissellement_pluie, alphak_stewart
!$OMP THREADPRIVATE(ruissellement_pluie, alphak_stewart)
   INTEGER, SAVE :: calendrier_guide
!$OMP THREADPRIVATE(calendrier_guide)
   INTEGER, SAVE :: cste_surf_cond
!$OMP THREADPRIVATE(cste_surf_cond)
   REAL,    SAVE :: mixlen
!$OMP THREADPRIVATE(mixlen)
   INTEGER, SAVE :: evap_cont_cste
!$OMP THREADPRIVATE(evap_cont_cste)
   REAL,    SAVE :: deltaO18_evap_cont, d_evap_cont
!$OMP THREADPRIVATE(deltaO18_evap_cont, d_evap_cont)
   INTEGER, SAVE :: nudge_qsol, region_nudge_qsol
!$OMP THREADPRIVATE(nudge_qsol, region_nudge_qsol)
   INTEGER, SAVE :: nlevmaxO17
!$OMP THREADPRIVATE(nlevmaxO17)
   INTEGER, SAVE :: no_pce
!$OMP THREADPRIVATE(no_pce)
   REAL,    SAVE :: A_satlim
!$OMP THREADPRIVATE(A_satlim)
   INTEGER, SAVE :: ok_restrict_A_satlim, modif_ratqs
!$OMP THREADPRIVATE(ok_restrict_A_satlim, modif_ratqs)
   REAL,    SAVE :: Pcrit_ratqs, ratqsbasnew
!$OMP THREADPRIVATE(Pcrit_ratqs, ratqsbasnew)
   REAL,    SAVE :: fac_modif_evaoce
!$OMP THREADPRIVATE(fac_modif_evaoce)
   INTEGER, SAVE :: ok_bidouille_wake
!$OMP THREADPRIVATE(ok_bidouille_wake)
   LOGICAL, SAVE :: cond_temp_env
!$OMP THREADPRIVATE(cond_temp_env)

   !--- Vectors of length "niso"
   REAL, ALLOCATABLE, DIMENSION(:), SAVE :: &
                    tnat, toce, tcorr, tdifrel
!$OMP THREADPRIVATE(tnat, toce, tcorr, tdifrel)
   REAL, ALLOCATABLE, DIMENSION(:), SAVE :: &
                    talph1, talph2, talph3, talps1, talps2
!$OMP THREADPRIVATE(talph1, talph2, talph3, talps1, talps2)
   REAL, ALLOCATABLE, DIMENSION(:), SAVE :: &
                    tkcin0, tkcin1, tkcin2
!$OMP THREADPRIVATE(tkcin0, tkcin1, tkcin2)
   REAL, ALLOCATABLE, DIMENSION(:), SAVE :: &
                    alpha_liq_sol, Rdefault, Rmethox
!$OMP THREADPRIVATE(alpha_liq_sol, Rdefault, Rmethox)
   REAL, SAVE ::    fac_coeff_eq17_liq, fac_coeff_eq17_ice
!$OMP THREADPRIVATE(fac_coeff_eq17_liq, fac_coeff_eq17_ice)

   !--- Negligible lower thresholds: no need to check for absurd values under these lower limits
   REAL, PARAMETER :: &
      ridicule      = 1e-12,              & ! For mixing ratios
      ridicule_rain = 1e-8,               & ! For rain fluxes (rain, zrfl...) in kg/s <-> 1e-3 mm/day
      ridicule_evap = ridicule_rain*1e-2, & ! For evaporations                in kg/s <-> 1e-3 mm/day
      ridicule_qsol = ridicule_rain,      & ! For qsol                        in kg <-> 1e-8 kg
      ridicule_snow = ridicule_qsol         ! For snow                        in kg <-> 1e-8 kg
   REAL, PARAMETER :: expb_max = 30.0

   !--- Specific to HTO:
   LOGICAL, SAVE :: ok_prod_nucl_tritium    !--- TRUE => HTO production by nuclear tests
!$OMP THREADPRIVATE(ok_prod_nucl_tritium)
   INTEGER, PARAMETER :: nessai = 486
   INTEGER, DIMENSION(nessai), SAVE :: &
                    day_nucl, month_nucl, year_nucl
!$OMP THREADPRIVATE(day_nucl, month_nucl, year_nucl)
   REAL,    DIMENSION(nessai), SAVE :: &
                    lat_nucl, lon_nucl, zmin_nucl, zmax_nucl, HTO_nucl
!$OMP THREADPRIVATE(lat_nucl, lon_nucl, zmin_nucl, zmax_nucl, HTO_nucl)
 
 
CONTAINS

SUBROUTINE iso_init()
   USE infotrac_phy,       ONLY: ntiso, niso, getKey
    USE strings_mod,       ONLY: maxlen
   IMPLICIT NONE

   !=== Local variables:
   INTEGER :: ixt

   !--- H2[18]O reference
   REAL :: fac_enrichoce18, alpha_liq_sol_O18, &
           talph1_O18, talph2_O18, talph3_O18, talps1_O18, talps2_O18, &
           tkcin0_O18, tkcin1_O18, tkcin2_O18, tdifrel_O18  

   !--- For H2[17]O
   REAL    :: fac_kcin, pente_MWL
      
   !--- Sensitivity tests
   LOGICAL, PARAMETER ::   ok_nocinsfc = .FALSE. ! if T: no kinetic effect in sfc evap
   LOGICAL, PARAMETER ::   ok_nocinsat = .FALSE. ! if T: no sursaturation effect for ice
   LOGICAL, PARAMETER :: Rdefault_smow = .FALSE. ! if T: Rdefault=smow; if F: nul

   !--- For [3]H
   INTEGER :: iessai

   CHARACTER(LEN=maxlen) :: modname, sxt
   REAL, ALLOCATABLE :: tmp(:)

   modname = 'iso_init'
   CALL msg('219: entree', modname)

   !--------------------------------------------------------------
   ! General:
   !--------------------------------------------------------------

   !--- Check number of isotopes
   CALL msg('64: niso = '//TRIM(int2str(niso)), modname)

   !--- Init de ntracisoOR: on ecrasera en cas de traceurs de tagging isotopiques
   !                     (nzone>0) si complications avec ORCHIDEE
   ntracisoOR = ntiso  

   !--- Type of water isotopes:
   iso_eau = strIdx(isoName, 'H216O'); CALL msg('iso_eau='//int2str(iso_eau), modname)
   iso_HDO = strIdx(isoName, 'HDO');   CALL msg('iso_HDO='//int2str(iso_HDO), modname)
   iso_O18 = strIdx(isoName, 'H218O'); CALL msg('iso_O18='//int2str(iso_O18), modname)
   iso_O17 = strIdx(isoName, 'H217O'); CALL msg('iso_O17='//int2str(iso_O17), modname)
   iso_HTO = strIdx(isoName, 'HTO');   CALL msg('iso_HTO='//int2str(iso_HTO), modname)

   !--- Initialiaation: reading the isotopic parameters.
   CALL get_in('lambda',     lambda_sursat, 0.004)
   CALL get_in('thumxt1',    thumxt1,       0.75*1.2)
   CALL get_in('ntot',       ntot,          20,  .FALSE.)
   CALL get_in('h_land_ice', h_land_ice,    20., .FALSE.)
   CALL get_in('P_veg',      P_veg,         1.0, .FALSE.)
   CALL get_in('bidouille_anti_divergence', bidouille_anti_divergence, .FALSE.)
   CALL get_in('essai_convergence',         essai_convergence,         .FALSE.)
   CALL get_in('initialisation_iso',        initialisation_iso,        0)

!  IF(nzone>0 .AND. initialisation_iso==0) &
!      CALL get_in('initialisation_isotrac',initialisation_isotrac)
   CALL get_in('modif_sst',      modif_sst,         0)
   CALL get_in('deltaTtest',     deltaTtest,      0.0)     !--- For modif_sst>=1
   CALL get_in('deltaTtestpoles',deltaTtestpoles, 0.0)     !--- For modif_sst>=2
   CALL get_in( 'sstlatcrit',    sstlatcrit,     30.0)     !--- For modif_sst>=3
   CALL get_in('dsstlatcrit',   dsstlatcrit,      0.0)     !--- For modif_sst>=3
#ifdef ISOVERIF
   CALL msg('iso_init 270:  sstlatcrit='//real2str( sstlatcrit), modname, sstlatcrit < 0.0) !--- For modif_sst>=2
   CALL msg('iso_init 279: dsstlatcrit='//real2str(dsstlatcrit), modname, sstlatcrit < 0.0) !--- For modif_sst>=3
   IF(modif_sst >= 2 .AND. sstlatcrit < 0.0) STOP
#endif              

   CALL get_in('modif_sic', modif_sic,  0)
   IF(modif_sic >= 1) &
   CALL get_in('deltasic',  deltasic, 0.1)

   CALL get_in('albedo_prescrit', albedo_prescrit, 0)
   IF(albedo_prescrit == 1) THEN
      CALL get_in('lon_min_albedo', lon_min_albedo, -200.)
      CALL get_in('lon_max_albedo', lon_max_albedo,  200.)
      CALL get_in('lat_min_albedo', lat_min_albedo, -100.)
      CALL get_in('lat_max_albedo', lat_max_albedo,  100.)
   END IF
   deltaO18_oce=0.0
   CALL get_in('deltaP_BL',           deltaP_BL,     10.0)
   CALL get_in('ruissellement_pluie', ruissellement_pluie, 0)
   CALL get_in('alphak_stewart',      alphak_stewart,      1)
   CALL get_in('tdifexp_sol',         tdifexp_sol,      0.67)
   CALL get_in('calendrier_guide',    calendrier_guide,    0)
   CALL get_in('cste_surf_cond',      cste_surf_cond,      0)
   CALL get_in('mixlen',              mixlen,           35.0)
   CALL get_in('evap_cont_cste',      evap_cont_cste,      0)
   CALL get_in('deltaO18_evap_cont',  deltaO18_evap_cont,0.0)
   CALL get_in('d_evap_cont',         d_evap_cont,       0.0)
   CALL get_in('nudge_qsol',          nudge_qsol,          0)
   CALL get_in('region_nudge_qsol',   region_nudge_qsol,   1)
   nlevmaxO17 = 50
   CALL msg('nlevmaxO17='//TRIM(int2str(nlevmaxO17)))
   CALL get_in('no_pce',   no_pce,     0)
   CALL get_in('A_satlim', A_satlim, 1.0)
   CALL get_in('ok_restrict_A_satlim', ok_restrict_A_satlim, 0)
#ifdef ISOVERIF
   CALL msg(' 315: A_satlim='//real2str(A_satlim), modname, A_satlim > 1.0)
   IF(A_satlim > 1.0) STOP
#endif
!  CALL get_in('slope_limiterxy',   slope_limiterxy,  2.0) 
!  CALL get_in('slope_limiterz',    slope_limiterz,   2.0) 
   CALL get_in('modif_ratqs',       modif_ratqs,        0)
   CALL get_in('Pcrit_ratqs',       Pcrit_ratqs,    500.0)
   CALL get_in('ratqsbasnew',       ratqsbasnew,     0.05)
   CALL get_in('fac_modif_evaoce',  fac_modif_evaoce, 1.0)
   CALL get_in('ok_bidouille_wake', ok_bidouille_wake,  0)
   ! si oui, la temperature de cond est celle de l'environnement, pour eviter
   ! bugs quand temperature dans ascendances convs est mal calculee
   CALL get_in('cond_temp_env',        cond_temp_env,        .FALSE.)
   IF(ANY(isoName == 'HTO')) &
   CALL get_in('ok_prod_nucl_tritium', ok_prod_nucl_tritium, .FALSE., .FALSE.)

   !--------------------------------------------------------------
   ! Parameters that do not depend on the nature of water isotopes:
   !--------------------------------------------------------------
   ! -- temperature at which ice condensate starts to form (valeur ECHAM?):
   pxtmelt = 273.15

   ! -- temperature at which all condensate is ice:
   pxtice  = 273.15-10.0

   !- -- test PHASE
!   pxtmelt = 273.15 - 10.0
!   pxtice  = 273.15 - 30.0

   ! -- minimum temperature to calculate fractionation coeff
   pxtmin = 273.15 - 120.0   ! On ne calcule qu'au dessus de -120°C
   pxtmax = 273.15 +  60.0   ! On ne calcule qu'au dessus de +60°C
   !    Remarque: les coeffs ont ete mesures seulement jusq'à -40!

   ! -- a constant for alpha_eff for equilibrium below cloud base:
   tdifexp = 0.58
   tv0cin  = 7.0

   ! facteurs lambda et mu dans Si=musi-lambda*T
   musi=1.0
   if (ok_nocinsat) lambda_sursat = 0.0          ! no sursaturation effect

   ! diffusion dans le sol
   Kd=2.5e-9 ! m2/s   

   ! cas où cste_surf_cond: on met rhs ou/et Ts cste pour voir
   rh_cste_surf_cond = 0.6
    T_cste_surf_cond = 288.0
   
   CALL msg('iso_O18, iso_HDO, iso_eau = '//TRIM(strStack(int2str([iso_O18, iso_HDO, iso_eau]))), modname)

   !--------------------------------------------------------------
   ! Parameters that depend on the nature of water isotopes:
   !--------------------------------------------------------------
   IF(getKey('tnat',    tnat,    isoName)) CALL abort_physic(modname, 'can''t get tnat',    1)
   IF(getKey('toce',    toce,    isoName)) CALL abort_physic(modname, 'can''t get toce',    1)
   IF(getKey('tcorr',   tcorr,   isoName)) CALL abort_physic(modname, 'can''t get tcorr',   1)
   IF(getKey('talph1',  talph1,  isoName)) CALL abort_physic(modname, 'can''t get talph1',  1)
   IF(getKey('talph2',  talph2,  isoName)) CALL abort_physic(modname, 'can''t get talph2',  1)
   IF(getKey('talph3',  talph3,  isoName)) CALL abort_physic(modname, 'can''t get talph3',  1)
   IF(getKey('talps1',  talps1,  isoName)) CALL abort_physic(modname, 'can''t get talps1',  1)
   IF(getKey('talps2',  talps2,  isoName)) CALL abort_physic(modname, 'can''t get talps2',  1)
   IF(getKey('tkcin0',  tkcin0,  isoName)) CALL abort_physic(modname, 'can''t get tkcin0',  1)
   IF(getKey('tkcin1',  tkcin1,  isoName)) CALL abort_physic(modname, 'can''t get tkcin1',  1)
   IF(getKey('tkcin2',  tkcin2,  isoName)) CALL abort_physic(modname, 'can''t get tkcin2',  1)
   IF(getKey('tdifrel', tdifrel, isoName)) CALL abort_physic(modname, 'can''t get tdifrel', 1)
   IF(getKey('alpha_liq_sol', alpha_liq_sol, isoName)) CALL abort_physic(modname, 'can''t get alpha_liq_sol',  1)
   IF(getKey('Rdefault',Rdefault,isoName)) CALL abort_physic(modname, 'can''t get Rdefault',1)
   IF(getKey('Rmethox', Rmethox, isoName)) CALL abort_physic(modname, 'can''t get Rmethox', 1)

   IF(.NOT.Rdefault_smow) then
        Rdefault(:) = 0.0
        if (iso_eau.gt.0) Rdefault(iso_eau) = 1.0 ! correction Camille 30 mars 2023
   ENDIF
   write(*,*) 'Rdefault=',Rdefault

   !--- Sensitivity test: no kinetic effect in sfc evaporation
   IF(ok_nocinsfc) THEN
      tkcin0(1:niso) = 0.0
      tkcin1(1:niso) = 0.0
      tkcin2(1:niso) = 0.0
   END IF

   CALL msg('285: verif initialisation:', modname)
   DO ixt=1,niso
      sxt=int2str(ixt)
      CALL msg(' * isoName('//TRIM(sxt)//') = <'//TRIM(isoName(ixt))//'>',  modname)
      CALL msg(  '    tnat('//TRIM(sxt)//') = '//TRIM(real2str(tnat(ixt))), modname)
!     CALL msg('    alpha_liq_sol('//TRIM(sxt)//') = '//TRIM(real2str(alpha_liq_sol(ixt))), modname)
!     CALL msg(        '   tkcin0('//TRIM(sxt)//') = '//TRIM(real2str(tkcin0(ixt))),        modname)
!     CALL msg(       '   tdifrel('//TRIM(sxt)//') = '//TRIM(real2str(tdifrel(ixt))),       modname)
   END DO
   CALL msg('69:     lambda = '//TRIM(real2str(lambda_sursat)), modname)
   CALL msg('69:    thumxt1 = '//TRIM(real2str(thumxt1)),       modname)
   CALL msg('69: h_land_ice = '//TRIM(real2str(h_land_ice)),    modname)
   CALL msg('69:      P_veg = '//TRIM(real2str(P_veg)),         modname)

END SUBROUTINE iso_init


SUBROUTINE getinp_s(nam, val, def, lDisp)
   USE ioipsl_getincom, ONLY: getin
   USE mod_phys_lmdz_mpi_data, ONLY :  is_mpi_root
   USE mod_phys_lmdz_omp_data, ONLY :  is_omp_root
   USE mod_phys_lmdz_transfert_para, ONLY : bcast
   CHARACTER(LEN=*),           INTENT(IN)    :: nam
   CHARACTER(LEN=*),           INTENT(INOUT) :: val
   CHARACTER(LEN=*), OPTIONAL, INTENT(IN)    :: def
   LOGICAL,          OPTIONAL, INTENT(IN)    :: lDisp
   LOGICAL :: lD
!$OMP BARRIER
   IF(is_mpi_root.AND.is_omp_root) THEN
      IF(PRESENT(def)) val=def; CALL getin(nam,val)
      lD=.TRUE.; IF(PRESENT(lDisp)) lD=lDisp
      IF(lD) CALL msg(TRIM(nam)//' = '//TRIM(val))
  END IF
  CALL bcast(val)
END SUBROUTINE getinp_s

SUBROUTINE getinp_i(nam, val, def, lDisp)
   USE ioipsl_getincom, ONLY: getin
   USE mod_phys_lmdz_mpi_data, ONLY :  is_mpi_root
   USE mod_phys_lmdz_omp_data, ONLY :  is_omp_root
   USE mod_phys_lmdz_transfert_para, ONLY : bcast
   CHARACTER(LEN=*),  INTENT(IN)    :: nam
   INTEGER,           INTENT(INOUT) :: val
   INTEGER, OPTIONAL, INTENT(IN)    :: def
   LOGICAL, OPTIONAL, INTENT(IN)    :: lDisp
   LOGICAL :: lD
!$OMP BARRIER
   IF(is_mpi_root.AND.is_omp_root) THEN
      IF(PRESENT(def)) val=def; CALL getin(nam,val)
      lD=.TRUE.; IF(PRESENT(lDisp)) lD=lDisp
      IF(lD) CALL msg(TRIM(nam)//' = '//TRIM(int2str(val)))
  END IF
  CALL bcast(val)
END SUBROUTINE getinp_i

SUBROUTINE getinp_r(nam, val, def, lDisp)
   USE ioipsl_getincom, ONLY: getin
   USE mod_phys_lmdz_mpi_data, ONLY :  is_mpi_root
   USE mod_phys_lmdz_omp_data, ONLY :  is_omp_root
   USE mod_phys_lmdz_transfert_para, ONLY : bcast
   CHARACTER(LEN=*),  INTENT(IN)    :: nam
   REAL,              INTENT(INOUT) :: val
   REAL,    OPTIONAL, INTENT(IN)    :: def
   LOGICAL, OPTIONAL, INTENT(IN)    :: lDisp
   LOGICAL :: lD
!$OMP BARRIER
   IF(is_mpi_root.AND.is_omp_root) THEN
      IF(PRESENT(def)) val=def; CALL getin(nam,val)
      lD=.TRUE.; IF(PRESENT(lDisp)) lD=lDisp
      IF(lD) CALL msg(TRIM(nam)//' = '//TRIM(real2str(val)))
  END IF
  CALL bcast(val)
END SUBROUTINE getinp_r

SUBROUTINE getinp_l(nam, val, def, lDisp)
   USE ioipsl_getincom, ONLY: getin
   USE mod_phys_lmdz_mpi_data, ONLY :  is_mpi_root
   USE mod_phys_lmdz_omp_data, ONLY :  is_omp_root
   USE mod_phys_lmdz_transfert_para, ONLY : bcast
   CHARACTER(LEN=*),  INTENT(IN)    :: nam
   LOGICAL,           INTENT(INOUT) :: val
   LOGICAL, OPTIONAL, INTENT(IN)    :: def
   LOGICAL, OPTIONAL, INTENT(IN)    :: lDisp
   LOGICAL :: lD
!$OMP BARRIER
   IF(is_mpi_root.AND.is_omp_root) THEN
      IF(PRESENT(def)) val=def; CALL getin(nam,val)
      lD=.TRUE.; IF(PRESENT(lDisp)) lD=lDisp
      IF(lD) CALL msg(TRIM(nam)//' = '//TRIM(bool2str(val)))
  END IF
  CALL bcast(val)
END SUBROUTINE getinp_l

END MODULE isotopes_mod
#endif


