module lscp_ini_mod

implicit none

  ! PARAMETERS for lscp:
  !--------------------
  
  REAL, SAVE :: seuil_neb=0.001                 ! cloud fraction threshold: a cloud really exists when exceeded
  !$OMP THREADPRIVATE(seuil_neb)

  INTEGER, SAVE :: ninter=5                     ! number of iterations to calculate autoconversion to precipitation
  !$OMP THREADPRIVATE(ninter)

  INTEGER,SAVE :: iflag_evap_prec=1             ! precipitation evaporation flag. 0: nothing, 1: "old way", 
                                                ! 2: Max cloud fraction above to calculate the max of reevaporation
                                                ! 4: LTP'method i.e. evaporation in the clear-sky fraction of the mesh only
  !$OMP THREADPRIVATE(iflag_evap_prec)

  REAL t_coup                                   ! temperature threshold which determines the phase 
  PARAMETER (t_coup=234.0)                      ! for which the saturation vapor pressure is calculated

  REAL DDT0                                     ! iteration parameter 
  PARAMETER (DDT0=.01)

  REAL ztfondue                                 ! parameter to calculate melting fraction of precipitation
  PARAMETER (ztfondue=278.15)

  REAL, SAVE    :: rain_int_min=0.001           ! Minimum local rain intensity [mm/s] before the decrease in associated precipitation fraction 
  !$OMP THREADPRIVATE(rain_int_min)

  REAL, SAVE :: a_tr_sca(4)                     ! Variables for tracers temporary: alpha parameter for scavenging, 4 possible scavenging processes
  !$OMP THREADPRIVATE(a_tr_sca)
  
  INTEGER, SAVE ::  iflag_mpc_bl=0              ! flag to activate boundary layer mixed phase cloud param
  !$OMP THREADPRIVATE(iflag_mpc_bl)
  
  LOGICAL, SAVE :: ok_radocond_snow=.false.       ! take into account the mass of ice precip in the cloud ice content seen by radiation
  !$OMP THREADPRIVATE(ok_radocond_snow)


CONTAINS

SUBROUTINE lscp_ini(dtime,ok_ice_sursat)


   USE ioipsl_getin_p_mod, ONLY : getin_p
   USE print_control_mod, ONLY: prt_level, lunout
   USE ice_sursat_mod, ONLY: ice_sursat_init

   REAL, INTENT(IN)      :: dtime
   LOGICAL, INTENT(IN)   :: ok_ice_sursat  

    CALL getin_p('ninter',ninter)
    CALL getin_p('iflag_evap_prec',iflag_evap_prec)
    CALL getin_p('seuil_neb',seuil_neb)
    CALL getin_p('rain_int_min',rain_int_min) 
    CALL getin_p('iflag_mpc_bl',iflag_mpc_bl)
    CALL getin_p('ok_radocond_snow',ok_radocond_snow)         
    WRITE(lunout,*) 'lscp, ninter:', ninter
    WRITE(lunout,*) 'lscp, iflag_evap_prec:', iflag_evap_prec
    WRITE(lunout,*) 'lscp, seuil_neb:', seuil_neb
    WRITE(lunout,*) 'lscp, rain_int_min:', rain_int_min
    WRITE(lunout,*) 'lscp, iflag_mpc_bl:', iflag_mpc_bl
    WRITE(lunout,*) 'lscp, ok_radocond_snow:', ok_radocond_snow
    
    ! check for precipitation sub-time steps
    IF (ABS(dtime/REAL(ninter)-360.0).GT.0.001) THEN
        WRITE(lunout,*) 'lscp: it is not expected, see Z.X.Li', dtime
        WRITE(lunout,*) 'I would prefer a 6 min sub-timestep'
    ENDIF


    !AA Temporary initialisation
    a_tr_sca(1) = -0.5
    a_tr_sca(2) = -0.5
    a_tr_sca(3) = -0.5
    a_tr_sca(4) = -0.5
    
    IF (ok_ice_sursat) CALL ice_sursat_init()



end subroutine lscp_ini

end module lscp_ini_mod
