!$Id: infotrac.F90 4368 2022-12-05 23:01:16Z lguez $
!
MODULE infotrac

   USE       strings_mod, ONLY: msg, fmsg, maxlen, cat, dispTable, int2str, bool2str, strStack, strParse
   USE readTracFiles_mod, ONLY: trac_type, readTracersFiles, tracers, setGeneration, itZonIso, nbIso, tran0, delPhase, &
                        getKey, isot_type, readIsotopesFile, isotope, maxTableWidth, iqIsoPha, ntiso, ixIso, addPhase, &
                   indexUpdate, isoSelect, isoPhas, isoZone, isoName, isoKeys, iH2O, isoCheck, nphas, nzone, niso
   IMPLICIT NONE

   PRIVATE

   !=== FOR TRACERS:
   PUBLIC :: init_infotrac                                 !--- Initialization of the tracers
   PUBLIC :: tracers, type_trac, types_trac                !--- Full tracers database, tracers type keyword
   PUBLIC :: nqtot,   nbtr,   nqo,   nqCO2,   nqtottr      !--- Main dimensions
   PUBLIC :: conv_flg, pbl_flg                             !--- Convection & boundary layer activation keys

   !=== FOR ISOTOPES: General
   PUBLIC :: isot_type, nbIso                              !--- Derived type, full isotopes families database + nb of families
   PUBLIC :: isoSelect, ixIso                              !--- Isotopes family selection tool + selected family index
   !=== FOR ISOTOPES: Specific to water
   PUBLIC :: iH2O                                          !--- H2O isotopes class index
   PUBLIC :: min_qParent, min_qMass, min_ratio             !--- Min. values for various isotopic quantities
   !=== FOR ISOTOPES: Depending on the selected isotopes family
   PUBLIC :: isotope, isoKeys                              !--- Selected isotopes database + associated keys (cf. getKey)
   PUBLIC :: isoName, isoZone, isoPhas                     !--- Isotopes and tagging zones names, phases
   PUBLIC :: niso,    nzone,   nphas,   ntiso              !---  " " numbers + isotopes & tagging tracers number
   PUBLIC :: itZonIso                                      !--- idx "it" (in "isoName(1:niso)") = function(tagging idx, isotope idx)
   PUBLIC :: iqIsoPha                                      !--- idx "iq" (in "qx") = function(isotope idx, phase idx) + aliases
   PUBLIC :: isoCheck                                      !--- Run isotopes checking routines
   !=== FOR BOTH TRACERS AND ISOTOPES
   PUBLIC :: getKey                                        !--- Get a key from "tracers" or "isotope"

!=== CONVENTIONS FOR TRACERS NUMBERS:
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  | water in different |    water tagging      |  water isotopes | other tracers | additional tracers moments |
!  | phases: H2O_[gls]  |      isotopes         |                 |               |  for higher order schemes  |
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  |                    |                       |                 |               |                            |
!  |<--     nqo      -->|<-- nqo*niso* nzone -->|<-- nqo*niso  -->|<--  nbtr   -->|<--        (nmom)        -->|         
!  |                    |                                         |                                            |
!  |                    |<-- nqo*niso*(nzone+1)  =   nqo*ntiso -->|<--    nqtottr = nbtr + nmom             -->|
!  |                                                                              = nqtot - nqo*(ntiso+1)      |
!  |                                                                                                           |
!  |<--                        nqtrue  =  nbtr + nqo*(ntiso+1)                 -->|                            |
!  |                                                                                                           |
!  |<--                        nqtot   =  nqtrue + nmom                                                     -->|
!  |                                                                                                           |
!  |-----------------------------------------------------------------------------------------------------------|
!  NOTES FOR THIS TABLE:
!  * Used "niso", "nzone" and "ntiso" are components of "isotopes(ip)" for water (isotopes(ip)%parent == 'H2O'),
!    since water is so far the sole tracers family, except passive CO2, removed from the main tracers table.
!  * For water, "nqo" is equal to the more general field "isotopes(ip)%nphas".
!  * "niso", "nzone", "ntiso", "nphas" are defined for other isotopic tracers families, if any.
!
!=== DERIVED TYPE EMBEDDING MOST OF THE TRACERS-RELATED QUANTITIES (LENGTH: nqtot)
!    Each entry is accessible using "%" sign.
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  |  entry      | Meaning                                              | Former name | Possible values        |
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  | name        | Name (short)                                         | tname       |                        |
!  | gen0Name    | Name of the 1st generation ancestor                  | /           |                        |
!  | parent      | Name of the parent                                   | /           |                        |
!  | longName    | Long name (with adv. scheme suffix) for outputs      | ttext       |                        |
!  | type        | Type (so far: tracer or tag)                         | /           | tracer,tag             |
!  | phase       | Phases list ("g"as / "l"iquid / "s"olid)             | /           | [g][l][s]              |
!  | component   | Name(s) of the merged/cumulated section(s)           | /           | coma-separated names   |
!  | iGeneration | Generation (>=1)                                     | /           |                        |
!  | iqParent    | Index of the parent tracer                           | iqpere      | 1:nqtot                |
!  | iqDescen    | Indexes of the childs       (all generations)        | iqfils      | 1:nqtot                |
!  | nqDescen    | Number of the descendants   (all generations)        | nqdesc      | 1:nqtot                |
!  | nqChildren  | Number of childs            (1st generation only)    | nqfils      | 1:nqtot                |
!  | keys        | key/val pairs accessible with "getKey" routine       | /           |                        |
!  | iadv        | Advection scheme number                              | iadv        | 1,2,10-20(exc.15,19),30|
!  | isAdvected  | advected tracers flag (.TRUE. if iadv >= 0)          | /           | nqtrue  .TRUE. values  |
!  | isInPhysics | tracers not extracted from the main table in physics | /           | nqtottr .TRUE. values  |
!  | iso_iGroup  | Isotopes group index in isotopes(:)                  | /           | 1:nbIso                |
!  | iso_iName   | Isotope  name  index in isotopes(iso_iGroup)%trac(:) | iso_indnum  | 1:niso                 |
!  | iso_iZone   | Isotope  zone  index in isotopes(iso_iGroup)%zone(:) | zone_num    | 1:nzone                |
!  | iso_iPhas   | Isotope  phase index in isotopes(iso_iGroup)%phas(:) | phase_num   | 1:nphas                |
!  +-------------+------------------------------------------------------+-------------+------------------------+
!
!=== DERIVED TYPE EMBEDDING MOST OF THE ISOTOPES-RELATED QUANTITIES (LENGTH: nbIso, NUMBER OF ISOTOPES FAMILIES)
!    Each entry is accessible using "%" sign.
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  |  entry | length | Meaning                                          |    Former name     | Possible values |
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  | parent          | Parent tracer (isotopes family name)             |                    |                 |
!  | keys   | niso   | Isotopes keys/values pairs list + number         |                    |                 |
!  | trac   | ntiso  | Isotopes + tagging tracers list + number         | / | ntraciso       |                 |
!  | zone   | nzone  | Geographic tagging zones   list + number         | / | ntraceurs_zone |                 |
!  | phase  | nphas  | Phases                     list + number         |                    | [g][l][s], 1:3  |
!  | iqIsoPha        | Index in "qx"           = f(name(1:ntiso)),phas) | iqiso              | 1:nqtot         |
!  | itZonIso        | Index in "trac(1:ntiso)"= f(zone, name(1:niso))  | index_trac         | 1:ntiso         |
!  +-----------------+--------------------------------------------------+--------------------+-----------------+

   REAL, PARAMETER :: min_qParent = 1.e-30, min_qMass = 1.e-18, min_ratio = 1.e-16 ! MVals et CRisi

   !=== DIMENSIONS OF THE TRACERS TABLES AND OTHER SCALAR VARIABLES
   INTEGER,               SAVE :: nqtot,  &                     !--- Tracers nb in dynamics (incl. higher moments + H2O)
                                  nbtr,   &                     !--- Tracers nb in physics  (excl. higher moments + H2O)
                                  nqo,    &                     !--- Number of water phases
                                  nqtottr, &                    !--- Number of tracers passed to phytrac (TO BE DELETED ?)
                                  nqCO2                         !--- Number of tracers of CO2  (ThL)
   CHARACTER(LEN=maxlen), SAVE :: type_trac                     !--- Keyword for tracers type(s)
   CHARACTER(LEN=maxlen), SAVE, ALLOCATABLE :: types_trac(:)    !--- Keyword for tracers type(s), parsed version

   !=== VARIABLES FOR INCA
   INTEGER,               SAVE, ALLOCATABLE :: conv_flg(:), &   !--- Convection     activation ; needed for INCA        (nbtr)
                                                pbl_flg(:)      !--- Boundary layer activation ; needed for INCA        (nbtr)

CONTAINS

SUBROUTINE init_infotrac
   USE control_mod, ONLY: planet_type
#ifdef REPROBUS
   USE CHEM_REP,    ONLY: Init_chem_rep_trac
#endif
   IMPLICIT NONE
!==============================================================================================================================
!
!   Auteur:  P. Le Van /L. Fairhead/F.Hourdin
!   -------
!
!   Modifications:
!   --------------
!   05/94: F.Forget      Modif special traceur
!   02/02: M-A Filiberti Lecture de traceur.def
!   01/22: D. Cugnet     Nouveaux tracer.def et tracer_*.def + encapsulation (types trac_type et isot_type)
!
!   Objet:
!   ------
!   GCM LMD nouvelle grille
!
!==============================================================================================================================
!   ... modification de l'integration de q ( 26/04/94 ) ....
!------------------------------------------------------------------------------------------------------------------------------
! Declarations:
   INCLUDE "dimensions.h"
   INCLUDE "iniprint.h"

!------------------------------------------------------------------------------------------------------------------------------
! Local variables
   INTEGER, ALLOCATABLE :: hadv(:), vadv(:)                          !--- Horizontal/vertical transport scheme number
#ifdef INCA
   INTEGER, ALLOCATABLE :: had (:), hadv_inca(:), conv_flg_inca(:), &!--- Variables specific to INCA
                           vad (:), vadv_inca(:),  pbl_flg_inca(:)
   CHARACTER(LEN=8), ALLOCATABLE :: solsym_inca(:)                   !--- Tracers names for INCA
   INTEGER :: nqINCA
#endif
   CHARACTER(LEN=2)      ::   suff(9)                                !--- Suffixes for schemes of order 3 or 4 (Prather)
   CHARACTER(LEN=3)      :: descrq(30)                               !--- Advection scheme description tags
   CHARACTER(LEN=maxlen) :: msg1                                     !--- String for messages
   INTEGER :: fType                                                  !--- Tracers description file type ; 0: none
                                                                     !--- 1/2/3: "traceur.def"/"tracer.def"/"tracer_*.def"
   INTEGER :: nqtrue                                                 !--- Tracers nb from tracer.def (no higher order moments)
   INTEGER :: iad                                                    !--- Advection scheme number
   INTEGER :: ic, iq, jq, it, nt, im, nm, iz, k                      !--- Indexes and temporary variables
   LOGICAL :: lerr, ll, lRepr
   CHARACTER(LEN=1) :: p
   TYPE(trac_type), ALLOCATABLE, TARGET :: ttr(:)
   TYPE(trac_type), POINTER             :: t1, t(:)
   INTEGER :: ierr

   CHARACTER(LEN=*), PARAMETER :: modname="init_infotrac"
!------------------------------------------------------------------------------------------------------------------------------
! Initialization :
!------------------------------------------------------------------------------------------------------------------------------
   suff          = ['x ','y ','z ','xx','xy','xz','yy','yz','zz']
   descrq( 1: 2) = ['LMV','BAK']
   descrq(10:20) = ['VL1','VLP','FH1','FH2','VLH','   ','PPM','PPS','PPP','   ','SLP']
   descrq(30)    =  'PRA'
    
   CALL msg('type_trac = "'//TRIM(type_trac)//'"', modname)
   IF(strParse(type_trac, '|', types_trac, n=nt)) CALL abort_gcm(modname,'can''t parse "type_trac = '//TRIM(type_trac)//'"',1)

   !---------------------------------------------------------------------------------------------------------------------------
   DO it = 1, nt                                                          !--- nt>1=> "type_trac": coma-separated keywords list
   !---------------------------------------------------------------------------------------------------------------------------
      !--- MESSAGE ABOUT THE CHOSEN CONFIGURATION
      msg1 = 'For type_trac = "'//TRIM(types_trac(it))//'":'
      SELECT CASE(types_trac(it))
         CASE('inca'); CALL msg(TRIM(msg1)//' coupling with INCA chemistry model',        modname)
         CASE('inco'); CALL msg(TRIM(msg1)//' coupling jointly with INCA and CO2 cycle',  modname)
         CASE('repr'); CALL msg(TRIM(msg1)//' coupling with REPROBUS chemistry model',    modname)
         CASE('co2i'); CALL msg(TRIM(msg1)//' you have chosen to run with CO2 cycle',     modname)
         CASE('coag'); CALL msg(TRIM(msg1)//' tracers are treated for COAGULATION tests', modname)
         CASE('lmdz'); CALL msg(TRIM(msg1)//' tracers are treated in LMDZ only',          modname)
         CASE DEFAULT; CALL abort_gcm(modname,'type_trac='//TRIM(types_trac(it))//' not possible yet.',1)
      END SELECT

      !--- COHERENCE TEST BETWEEN "type_trac" AND PREPROCESSING KEYS
      SELECT CASE(types_trac(it))
         CASE('inca', 'inco')
#ifndef INCA
            CALL abort_gcm(modname, 'You must add cpp key INCA and compile with INCA code', 1)
#endif
         CASE('repr')
#ifndef REPROBUS
            CALL abort_gcm(modname, 'You must add cpp key REPROBUS and compile with REPROBUS code', 1)
#endif
         CASE('coag')
#ifndef CPP_StratAer
            CALL abort_gcm(modname, 'You must add cpp key StratAer and compile with StratAer code', 1)
#endif
      END SELECT

   !---------------------------------------------------------------------------------------------------------------------------
   END DO
   !---------------------------------------------------------------------------------------------------------------------------

   nqCO2 = COUNT( [ANY(types_trac == 'inco') .OR. (ANY(types_trac == 'co2i') .AND. ANY(types_trac == 'inca'))] )

!==============================================================================================================================
! 1) Get the numbers of: true (first order only) tracers "nqtrue", water tracers "nqo" (vapor/liquid/solid)
!==============================================================================================================================
   lRepr = ANY(types_trac(:) == 'repr')
   IF(readTracersFiles(type_trac, fType, lRepr)) CALL abort_gcm(modname, 'problem with tracers file(s)',1)
   !---------------------------------------------------------------------------------------------------------------------------
   IF(fType == 0) CALL abort_gcm(modname, 'Missing tracers file: "traceur.def", "tracer.def" or "tracer_<keyword>.def file.',1)
   !---------------------------------------------------------------------------------------------------------------------------
   IF(fType == 1 .AND. ANY(['inca','inco'] == type_trac)) THEN       !=== FOUND OLD STYLE INCA "traceur.def" (single type_trac)
   !---------------------------------------------------------------------------------------------------------------------------
#ifdef INCA
      nqo = SIZE(tracers) - nqCO2
      CALL Init_chem_inca_trac(nqINCA)                               !--- Get nqINCA from INCA
      nbtr = nqINCA + nqCO2                                          !--- Number of tracers passed to phytrac
      nqtrue = nbtr + nqo                                            !--- Total number of "true" tracers
      IF(ALL([2,3] /= nqo)) CALL abort_gcm(modname, 'Only 2 or 3 water phases allowed ; found nqo='//TRIM(int2str(nqo)), 1)
      ALLOCATE(hadv(nqtrue), hadv_inca(nqINCA), conv_flg_inca(nqINCA), solsym_inca(nqINCA))
      ALLOCATE(vadv(nqtrue), vadv_inca(nqINCA),  pbl_flg_inca(nqINCA))
      CALL init_transport(solsym_inca, conv_flg_inca, pbl_flg_inca, hadv_inca, vadv_inca)
      ALLOCATE(ttr(nqtrue))
      ttr(1:nqo+nqCO2)                    = tracers
      ttr(1    :      nqo   )%component   = 'lmdz'
      ttr(1+nqo:nqCO2+nqo   )%component   = 'co2i'
      ttr(1+nqo+nqCO2:nqtrue)%component   = 'inca'
      ttr(1+nqo      :nqtrue)%name        = [('CO2     ', k=1, nqCO2), solsym_inca]
      ttr(1+nqo+nqCO2:nqtrue)%parent      = tran0
      ttr(1+nqo+nqCO2:nqtrue)%phase       = 'g'
      lerr = getKey('hadv', had, ky=tracers(:)%keys)
      lerr = getKey('vadv', vad, ky=tracers(:)%keys)
      hadv(1:nqo) = had(:); hadv(nqo+1:nqtrue) = hadv_inca
      vadv(1:nqo) = vad(:); vadv(nqo+1:nqtrue) = vadv_inca
      CALL MOVE_ALLOC(FROM=ttr, TO=tracers)
      CALL setGeneration(tracers)                                    !--- SET FIELDS %iGeneration, %gen0Name
      DEALLOCATE(had, hadv_inca, vad, vadv_inca, conv_flg_inca, pbl_flg_inca, solsym_inca)
#endif
   !---------------------------------------------------------------------------------------------------------------------------
   ELSE                                                              !=== OTHER CASES (OLD OR NEW FORMAT, NO INCA MODULE)
   !---------------------------------------------------------------------------------------------------------------------------
      nqo    =        COUNT(delPhase(tracers(:)%name)     == 'H2O' &
                               .AND. tracers(:)%component == 'lmdz') !--- Number of water phases
      nqtrue = SIZE(tracers)                                         !--- Total number of "true" tracers
      nbtr   = nqtrue-COUNT(delPhase(tracers(:)%gen0Name) == 'H2O' &
                               .AND. tracers(:)%component == 'lmdz') !--- Number of tracers passed to phytrac
#ifdef INCA
      nqINCA = COUNT(tracers(:)%component == 'inca')
#endif
      lerr = getKey('hadv', hadv, ky=tracers(:)%keys)
      lerr = getKey('vadv', vadv, ky=tracers(:)%keys)
   !---------------------------------------------------------------------------------------------------------------------------
   END IF
   !---------------------------------------------------------------------------------------------------------------------------

   !--- Transfert the number of tracers to Reprobus
#ifdef REPROBUS
   CALL Init_chem_rep_trac(nbtr, nqo, tracers(:)%name)
#endif

!==============================================================================================================================
! 2) Calculate nqtot, number of tracers needed (greater if advection schemes 20 or 30 have been chosen).
!==============================================================================================================================
   DO iq = 1, nqtrue
      IF( hadv(iq)<20 .OR. (ANY(hadv(iq)==[20,30]) .AND. hadv(iq)==vadv(iq)) ) CYCLE
      WRITE(msg1,'("The choice hadv=",i0,", vadv=",i0,a)')hadv(iq),vadv(iq),' for "'//TRIM(tracers(iq)%name)//'" is not available'
      CALL abort_gcm(modname, TRIM(msg1), 1)
   END DO
   nqtot =    COUNT( hadv< 20 .AND. vadv< 20 ) &                     !--- No additional tracer
         +  4*COUNT( hadv==20 .AND. vadv==20 ) &                     !--- 3  additional tracers
         + 10*COUNT( hadv==30 .AND. vadv==30 )                       !--- 9  additional tracers

   !--- More tracers due to the choice of advection scheme => assign total number of tracers
   IF( nqtot /= nqtrue ) THEN
      CALL msg('The choice of advection scheme for one or more tracers makes it necessary to add tracers')
      CALL msg('The number of true tracers is '//TRIM(int2str(nqtrue)))
      CALL msg('The total number of tracers needed is '//TRIM(int2str(nqtot)))
   END IF

!==============================================================================================================================
! 3) Determine the advection scheme choice for water and tracers "iadv" and the fields long name, isAdvected.
!     iadv = 1    "LMDZ-specific humidity transport" (for H2O vapour)          LMV
!     iadv = 2    backward                           (for H2O liquid)          BAK
!     iadv = 14   Van-Leer + specific humidity, modified by Francis Codron     VLH
!     iadv = 10   Van-Leer (chosen for vapour and liquid water)                VL1
!     iadv = 11   Van-Leer for hadv and PPM version (Monotonic) for vadv       VLP
!     iadv = 12   Frederic Hourdin I                                           FH1
!     iadv = 13   Frederic Hourdin II                                          FH2
!     iadv = 16   Monotonic         PPM (Collela & Woodward 1984)              PPM
!     iadv = 17   Semi-monotonic    PPM (overshoots allowed)                   PPS
!     iadv = 18   Definite positive PPM (overshoots and undershoots allowed)   PPP
!     iadv = 20   Slopes                                                       SLP
!     iadv = 30   Prather                                                      PRA
!
!        In array q(ij,l,iq) : iq = 1/2[/3]    for vapour/liquid[/ice] water
!        And optionaly:        iq = 3[4],nqtot for other tracers
!==============================================================================================================================
   ALLOCATE(ttr(nqtot))
   jq = nqtrue+1; tracers(:)%iadv = -1
   DO iq = 1, nqtrue
      t1 => tracers(iq)

      !--- VERIFY THE CHOICE OF ADVECTION SCHEME
      iad = -1
      IF(hadv(iq)     ==    vadv(iq)    ) iad = hadv(iq)
      IF(hadv(iq)==10 .AND. vadv(iq)==16) iad = 11
      WRITE(msg1,'("Bad choice of advection scheme for ",a,": hadv = ",i0,", vadv = ",i0)')TRIM(t1%name), hadv(iq), vadv(iq)
      IF(iad == -1) CALL abort_gcm(modname, msg1, 1)

      !--- SET FIELDS %longName, %iadv, %isAdvected, %isInPhysics
      t1%longName   = t1%name; IF(iad > 0) t1%longName=TRIM(t1%name)//descrq(iad)
      t1%iadv       = iad
      t1%isAdvected = iad >= 0
      t1%isInPhysics= delPhase(t1%gen0Name) /= 'H2O' &
                          .OR. t1%component /= 'lmdz' !=== OTHER EXCEPTIONS TO BE ADDED: CO2i, SURSATURATED WATER CLOUD...
      ttr(iq)       = t1

      !--- DEFINE THE HIGHER ORDER TRACERS, IF ANY
      nm = 0
      IF(iad == 20) nm = 3                                           !--- 2nd order scheme
      IF(iad == 30) nm = 9                                           !--- 3rd order scheme
      IF(nm == 0) CYCLE                                              !--- No higher moments
      ttr(jq+1:jq+nm)             = t1
      ttr(jq+1:jq+nm)%name        = [(TRIM(t1%name)    //'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%parent      = [(TRIM(t1%parent)  //'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%longName    = [(TRIM(t1%longName)//'-'//TRIM(suff(im)), im=1, nm) ]
      ttr(jq+1:jq+nm)%iadv        = [(-iad,    im=1, nm) ]
      ttr(jq+1:jq+nm)%isAdvected  = [(.FALSE., im=1, nm) ]
      jq = jq + nm
   END DO
   DEALLOCATE(hadv, vadv)
   CALL MOVE_ALLOC(FROM=ttr, TO=tracers)

   !--- SET FIELDS %iqParent, %nqChildren, %iGeneration, %iqDescen, %nqDescen
   CALL indexUpdate(tracers)

   !=== TEST ADVECTION SCHEME
   DO iq=1,nqtot ; t1 => tracers(iq); iad = t1%iadv

      !--- ONLY TESTED VALUES FOR TRACERS FOR NOW:               iadv = 14, 10 (and 0 for non-transported tracers)
      IF(ALL([10,14,0] /= iad)) &
         CALL abort_gcm(modname, 'Not tested for iadv='//TRIM(int2str(iad))//' ; 10 or 14 only are allowed !', 1)

      !--- ONLY TESTED VALUES FOR PARENTS HAVING CHILDS FOR NOW: iadv = 14, 10 (PARENTS: TRACERS OF GENERATION 1)
      IF(ALL([10,14] /= iad) .AND. t1%iGeneration == 1 .AND. ANY(tracers(:)%iGeneration > 1)) &
         CALL abort_gcm(modname, 'iadv='//TRIM(int2str(iad))//' not implemented for parents ; 10 or 14 only are allowed !', 1)

      !--- ONLY TESTED VALUES FOR CHILDS FOR NOW:                iadv = 10     (CHILDS:  TRACERS OF GENERATION GREATER THAN 1)
      IF(fmsg('WARNING ! iadv='//TRIM(int2str(iad))//' not implemented for childs. Setting iadv=10 for "'//TRIM(t1%name)//'"',&
         modname, iad /= 10 .AND. t1%iGeneration > 1)) t1%iadv = 10

      !--- ONLY VALID SCHEME NUMBER FOR WATER VAPOUR:            iadv = 14
      ll = t1%name /= addPhase('H2O','g')
      IF(fmsg('WARNING ! iadv=14 is valid for water vapour only. Setting iadv=10 for "'//TRIM(t1%name)//'".', &
         modname, iad == 14 .AND. ll))                 t1%iadv = 10
   END DO

   !=== READ PHYSICAL PARAMETERS FOR ISOTOPES ; DONE HERE BECAUSE dynetat0 AND iniacademic NEED "tnat" AND "alpha_ideal"
   niso = 0; nzone = 0; nphas = nqo; ntiso = 0; isoCheck = .FALSE.
   IF(readIsotopesFile()) CALL abort_gcm(modname, 'Problem when reading isotopes parameters', 1)

   !--- Convection / boundary layer activation for all tracers
   ALLOCATE(conv_flg(nbtr)); conv_flg(1:nbtr) = 1
   ALLOCATE( pbl_flg(nbtr));  pbl_flg(1:nbtr) = 1

   !--- Note: nqtottr can differ from nbtr when nmom/=0
   nqtottr = nqtot - COUNT(delPhase(tracers%gen0Name) == 'H2O' .AND. tracers%component == 'lmdz')
   IF(COUNT(tracers%iso_iName == 0) - COUNT(delPhase(tracers%name) == 'H2O' .AND. tracers%component == 'lmdz') /= nqtottr) &
      CALL abort_gcm(modname, 'pb dans le calcul de nqtottr', 1)

   !=== DISPLAY THE RESULTS
   IF(prt_level > 1) THEN
      CALL msg('nqo    = '//TRIM(int2str(nqo)),    modname)
      CALL msg('nbtr   = '//TRIM(int2str(nbtr)),   modname)
      CALL msg('nqtrue = '//TRIM(int2str(nqtrue)), modname)
      CALL msg('nqtot  = '//TRIM(int2str(nqtot)),  modname)
      CALL msg('niso   = '//TRIM(int2str(niso)),   modname)
      CALL msg('ntiso  = '//TRIM(int2str(ntiso)),  modname)
#ifdef INCA
      CALL msg('nqCO2  = '//TRIM(int2str(nqCO2)),  modname)
      CALL msg('nqINCA = '//TRIM(int2str(nqINCA)), modname)
#endif
   END IF
   t => tracers
   CALL msg('Information stored in infotrac :', modname)
   IF(dispTable('isssssssssiiiiiiiii', &
      ['iq    ', 'name  ', 'lName ', 'gen0N ', 'parent', 'type  ', 'phase ', 'compon', 'isPhy ', 'isAdv ', &
       'iadv  ', 'iGen  ', 'iqPar ', 'nqDes ', 'nqChld', 'iGroup', 'iName ', 'iZone ', 'iPhase'],          &
      cat(t%name, t%longName, t%gen0Name, t%parent, t%type, t%phase, t%component, bool2str(t%isInPhysics), &
                                                                                  bool2str(t%isAdvected)), &
      cat([(iq, iq=1, nqtot)], t%iadv, t%iGeneration, t%iqParent, t%nqDescen, t%nqChildren, t%iso_iGroup,  &
                  t%iso_iName, t%iso_iZone, t%iso_iPhase), nColMax=maxTableWidth, nHead=2, sub=modname))   &
      CALL abort_gcm(modname, "problem with the tracers table content", 1)
   IF(niso > 0) THEN
      CALL msg('Where, for isotopes family "'//TRIM(isotope%parent)//'":', modname)
      CALL msg('  isoKeys%name = '//strStack(isoKeys%name), modname)
      CALL msg('  isoName = '//strStack(isoName),      modname)
      CALL msg('  isoZone = '//strStack(isoZone),      modname)
      CALL msg('  isoPhas = '//TRIM(isoPhas),          modname)
   ELSE
      CALL msg('No isotopes identified.', modname)
   END IF
   CALL msg('end', modname)

END SUBROUTINE init_infotrac

END MODULE infotrac
