
! $Id: $

MODULE infotrac_phy

   USE       strings_mod, ONLY: msg, maxlen, strStack, strHead, strParse, strIdx, int2str
   USE readTracFiles_mod, ONLY: trac_type, isot_type, keys_type, delPhase, getKey, tnom_iso => newH2OIso

   IMPLICIT NONE

   PRIVATE

   !=== FOR TRACERS:
   PUBLIC :: init_infotrac_phy                             !--- Initialization of the tracers
   PUBLIC :: tracers, type_trac, types_trac                !--- Full tracers database, tracers type keyword
   PUBLIC :: nqtot,   nbtr,   nqo,   nqCO2,   nqtottr      !--- Main dimensions
   PUBLIC :: conv_flg, pbl_flg                             !--- Convection & boundary layer activation keys
#ifdef CPP_StratAer
   PUBLIC :: nbtr_bin, nbtr_sulgas                         !--- Number of aerosols bins and sulfur gases for StratAer model
   PUBLIC :: id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat
#endif

   !=== FOR ISOTOPES: General
   PUBLIC :: isotopes,  nbIso                              !--- Derived type, full isotopes families database + nb of families
   PUBLIC :: isoSelect, ixIso                              !--- Isotopes family selection tool + selected family index
   !=== FOR ISOTOPES: Specific to water
   PUBLIC :: iH2O                                          !--- H2O isotopes index
   !=== FOR ISOTOPES: Depending on the selected isotopes family
   PUBLIC :: isotope, isoKeys                              !--- Selected isotopes database + associated keys (cf. getKey)
   PUBLIC :: isoName, isoZone, isoPhas                     !--- Isotopes and tagging zones names, phases
   PUBLIC :: niso,    nzone,   nphas,   ntiso              !---  " " numbers + isotopes & tagging tracers number
   PUBLIC :: itZonIso                                      !--- iq = function(tagging zone idx, isotope idx)
   PUBLIC :: iqIsoPha                                      !--- idx of tagging tracer in iName = function(isotope idx, phase idx)
   PUBLIC :: isoCheck                                      !--- Run isotopes checking routines
   !=== FOR BOTH TRACERS AND ISOTOPES
   PUBLIC :: getKey                                        !--- Get a key from "tracers" or "isotope"

   INTERFACE isoSelect; MODULE PROCEDURE isoSelectByIndex, isoSelectByName; END INTERFACE isoSelect

!=== CONVENTIONS FOR TRACERS NUMBERS:
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  | water in different |    water tagging      |  water isotopes | other tracers | additional tracers moments |
!  | phases: H2O_[gls]  |      isotopes         |                 |               |  for higher order schemes  |
!  |--------------------+-----------------------+-----------------+---------------+----------------------------|
!  |                    |                       |                 |               |                            |
!  |<--     nqo      -->|<-- nqo*niso* nzone -->|<-- nqo*niso  -->|<--  nbtr   -->|<--        (nmom)        -->|         
!  |                    |                                         |                                            |
!  |                    |<-- nqo*niso*(nzone+1)  =   nqo*ntiso -->|<--    nqtottr = nbtr + nmom             -->|
!  |                                                                              = nqtot - nqo*(ntiso+1)      |
!  |                                                                                                           |
!  |<--                        nqtrue  =  nbtr + nqo*(ntiso+1)                 -->|                            |
!  |                                                                                                           |
!  |<--                        nqtot   =  nqtrue + nmom                                                     -->|
!  |                                                                                                           |
!  |-----------------------------------------------------------------------------------------------------------|
!  NOTES FOR THIS TABLE:
!  * Used "niso", "nzone" and "ntiso" are components of "isotopes(ip)" for water (isotopes(ip)%parent == 'H2O'),
!    since water is so far the sole tracers family, except passive CO2, removed from the main tracers table.
!  * For water, "nqo" is equal to the more general field "isotopes(ip)%nphas".
!  * "niso", "nzone", "ntiso", "nphas" are defined for other isotopic tracers families, if any.
!
!=== DERIVED TYPE EMBEDDING MOST OF THE TRACERS-RELATED QUANTITIES (LENGTH: nqtot)
!    Each entry is accessible using "%" sign.
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  |  entry      | Meaning                                              | Former name | Possible values        |
!  |-------------+------------------------------------------------------+-------------+------------------------+
!  | name        | Name (short)                                         | tname       |                        |
!  | gen0Name    | Name of the 1st generation ancestor                  | /           |                        |
!  | parent      | Name of the parent                                   | /           |                        |
!  | longName    | Long name (with adv. scheme suffix) for outputs      | ttext       |                        |
!  | type        | Type (so far: tracer or tag)                         | /           | tracer,tag             |
!  | phase       | Phases list ("g"as / "l"iquid / "s"olid)             | /           | [g][l][s]              |
!  | component   | Name(s) of the merged/cumulated section(s)           | /           | coma-separated names   |
!  | iadv        | Advection scheme number                              | iadv        | 1-20,30 exc. 3-9,15,19 |
!  | iGeneration | Generation (>=1)                                     | /           |                        |
!  | isAdvected  | advected tracers flag (.TRUE. if iadv >= 0)          | /           | nqtrue  .TRUE. values  |
!  | isInPhysics | tracers not extracted from the main table in physics | /           | nqtottr .TRUE. values  |
!  | iqParent    | Index of the parent tracer                           | iqpere      | 1:nqtot                |
!  | iqDescen    | Indexes of the childs       (all generations)        | iqfils      | 1:nqtot                |
!  | nqDescen    | Number of the descendants   (all generations)        | nqdesc      | 1:nqtot                |
!  | nqChilds    | Number of childs            (1st generation only)    | nqfils      | 1:nqtot                |
!  | iso_iGroup  | Isotopes group index in isotopes(:)                  | /           | 1:nbIso                |
!  | iso_iName   | Isotope  name  index in isotopes(iso_iGroup)%trac(:) | iso_indnum  | 1:niso                 |
!  | iso_iZone   | Isotope  zone  index in isotopes(iso_iGroup)%zone(:) | zone_num    | 1:nzone                |
!  | iso_iPhas   | Isotope  phase index in isotopes(iso_iGroup)%phas(:) | phase_num   | 1:nphas                |
!  | keys        | key/val pairs accessible with "getKey" routine       | /           |                        |
!  +-------------+------------------------------------------------------+-------------+------------------------+
!
!=== DERIVED TYPE EMBEDDING MOST OF THE ISOTOPES-RELATED QUANTITIES (LENGTH: nbIso, NUMBER OF ISOTOPES FAMILIES)
!    Each entry is accessible using "%" sign.
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  |  entry | length | Meaning                                          |    Former name     | Possible values |
!  |-----------------+--------------------------------------------------+--------------------+-----------------+
!  | parent          | Parent tracer (isotopes family name)             |                    |                 |
!  | keys   | niso   | Isotopes keys/values pairs list + number         |                    |                 |
!  | trac   | ntiso  | Isotopes + tagging tracers list + number         | / | ntraciso       |                 |
!  | zone   | nzone  | Geographic tagging zones   list + number         | / | ntraceurs_zone |                 |
!  | phase  | nphas  | Phases                     list + number         |                    | [g][l][s], 1:3  |
!  | iqIsoPha        | Index in "qx"           = f(name(1:ntiso)),phas) | iqiso              | 1:nqtot         |
!  | itZonIso        | Index in "trac(1:ntiso)"= f(zone, name(1:niso))  | index_trac         | 1:ntiso         |
!  +-----------------+--------------------------------------------------+--------------------+-----------------+

   !=== DIMENSIONS OF THE TRACERS TABLES AND OTHER SCALAR VARIABLES
   INTEGER,                 SAVE :: nqtot,  &                   !--- Tracers nb in dynamics (incl. higher moments + H2O)
                                    nbtr,   &                   !--- Tracers nb in physics  (excl. higher moments + H2O)
                                    nqo,    &                   !--- Number of water phases
                                    nbIso,  &                   !--- Number of available isotopes family
                                    nqtottr, &                  !--- Number of tracers passed to phytrac (TO BE DELETED ?)
                                    nqCO2                       !--- Number of tracers of CO2  (ThL)
   CHARACTER(LEN=maxlen),   SAVE :: type_trac                   !--- Keyword for tracers type
   CHARACTER(LEN=maxlen),   SAVE, ALLOCATABLE :: types_trac(:)  !--- Keyword for tracers type
!$OMP THREADPRIVATE(nqtot, nbtr, nqo, nbIso, nqtottr, nqCO2, type_trac, types_trac)

   !=== DERIVED TYPES EMBEDDING MOST INFORMATIONS ABOUT TRACERS AND ISOTOPES FAMILIES
   TYPE(trac_type), TARGET, SAVE, ALLOCATABLE ::  tracers(:)    !=== TRACERS DESCRIPTORS VECTOR
   TYPE(isot_type), TARGET, SAVE, ALLOCATABLE :: isotopes(:)    !=== ISOTOPES PARAMETERS VECTOR
!$OMP THREADPRIVATE(tracers, isotopes)

   !=== ALIASES FOR CURRENTLY SELECTED ISOTOPES FAMILY OF VARIABLES EMBEDDED IN THE VECTOR "isotopes"
   TYPE(isot_type),         SAVE, POINTER :: isotope            !--- CURRENTLY SELECTED ISOTOPES FAMILY DESCRIPTOR
   INTEGER,                 SAVE          :: ixIso, iH2O        !--- Index of the selected isotopes family and H2O family
   LOGICAL,                 SAVE          :: isoCheck           !--- Flag to trigger the checking routines
   TYPE(keys_type),         SAVE, POINTER :: isoKeys(:)         !--- ONE SET OF KEYS FOR EACH ISOTOPE (LISTED IN isoName)
   CHARACTER(LEN=maxlen),   SAVE, POINTER :: isoName(:),   &    !--- ISOTOPES NAMES FOR THE CURRENTLY SELECTED FAMILY
                                             isoZone(:),   &    !--- TAGGING ZONES  FOR THE CURRENTLY SELECTED FAMILY
                                             isoPhas            !--- USED PHASES    FOR THE CURRENTLY SELECTED FAMILY
   INTEGER,                 SAVE          ::  niso, nzone, &    !--- NUMBER OF ISOTOPES, TAGGING ZONES AND PHASES
                                             nphas, ntiso       !--- NUMBER OF PHASES AND ISOTOPES + ISOTOPIC TAGGING TRACERS
   INTEGER,                 SAVE, POINTER ::itZonIso(:,:), &    !--- INDEX IN "isoTrac" AS f(tagging zone idx,  isotope idx)
                                            iqIsoPha(:,:)       !--- INDEX IN "qx"      AS f(isotopic tracer idx, phase idx)
!$OMP THREADPRIVATE(isotope, ixIso,iH2O, isoCheck, isoKeys, isoName,isoZone,isoPhas, niso,nzone,nphas,ntiso, itZonIso,iqIsoPha)

   !=== VARIABLES FOR ISOTOPES INITIALIZATION AND FOR INCA
   INTEGER,          SAVE,    ALLOCATABLE ::conv_flg(:),  &     !--- Convection     activation ; needed for INCA        (nbtr)
                                             pbl_flg(:)         !--- Boundary layer activation ; needed for INCA        (nbtr)
!$OMP THREADPRIVATE(conv_flg, pbl_flg)

#ifdef CPP_StratAer
  !=== SPECIFIC TO STRATOSPHERIC AEROSOLS (CK/OB)
  INTEGER, SAVE ::  nbtr_bin, nbtr_sulgas         !--- number of aerosols bins and sulfur gases for StratAer model
!$OMP THREADPRIVATE(nbtr_bin, nbtr_sulgas)
  INTEGER, SAVE ::  id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat
!$OMP THREADPRIVATE(id_OCS_strat, id_SO2_strat, id_H2SO4_strat, id_BIN01_strat, id_TEST_strat)
#endif

CONTAINS

SUBROUTINE init_infotrac_phy(type_trac_, tracers_, isotopes_, nqtottr_, nqCO2_, pbl_flg_, conv_flg_)

   USE print_control_mod, ONLY: prt_level, lunout

   IMPLICIT NONE
   CHARACTER(LEN=*),INTENT(IN) :: type_trac_
   TYPE(trac_type), INTENT(IN) ::  tracers_(:)
   TYPE(isot_type), INTENT(IN) :: isotopes_(:)
   INTEGER,         INTENT(IN) :: nqtottr_
   INTEGER,         INTENT(IN) :: nqCO2_
   INTEGER,         INTENT(IN) :: conv_flg_(:)
   INTEGER,         INTENT(IN) ::  pbl_flg_(:)

   INTEGER :: iq, ixt
#ifdef CPP_StratAer
   CHARACTER(LEN=maxlen), ALLOCATABLE :: tnames(:)
#endif
   CHARACTER(LEN=maxlen) :: modname="init_infotrac_phy"

   type_trac = type_trac_
   IF(strParse(type_trac, '|', types_trac)) CALL abort_physic(modname,'can''t parse "type_trac = '//TRIM(type_trac)//'"',1)
   tracers   = tracers_
   isotopes  = isotopes_
   nqtottr   = nqtottr_
   nqCO2     = nqCO2_
   pbl_flg   =  pbl_flg_
   conv_flg  = conv_flg_
   nqtot     = SIZE(tracers_)
   nqo       = COUNT(delPhase(tracers%name)=='H2O' .AND. tracers%iGeneration==0 .AND. tracers%component=='lmdz')
   nbtr      = SIZE(conv_flg)
   nbIso     = SIZE(isotopes_)

   !=== Determine selected isotopes class related quantities:
   !    ixIso, isotope, niso,isoKeys, ntiso,isoName, nzone,isoZone, nphas,isoPhas, itZonIso, iqIsoPha, isoCheck
   IF(.NOT.isoSelect('H2O')) iH2O = ixIso
   IF(prt_level > 1) THEN
      CALL msg('nqtot   = '//TRIM(int2str(nqtot)),   modname)
      CALL msg('nbtr    = '//TRIM(int2str(nbtr )),   modname)
      CALL msg('nqo     = '//TRIM(int2str(nqo  )),   modname)
      CALL msg('niso    = '//TRIM(int2str(niso )),   modname)
      CALL msg('ntiso   = '//TRIM(int2str(ntiso)),   modname)
      CALL msg('nqtottr = '//TRIM(int2str(nqtottr)), modname)
      CALL msg('nqCO2   = '//TRIM(int2str(nqCO2)),   modname)
   END IF

#ifdef CPP_StratAer
   IF (ANY(types_trac == 'coag')) THEN
      nbtr_bin    = COUNT([(tracers(iq)%name(1:3)=='BIN', iq=1, nqtot)])
      nbtr_sulgas = COUNT([(tracers(iq)%name(1:3)=='GAS', iq=1, nqtot)])
      tnames = PACK(tracers(:)%name, MASK=tracers(:)%isInPhysics)
      id_BIN01_strat = strIdx(tnames, 'BIN01'   )
      id_OCS_strat   = strIdx(tnames, 'GASOCS'  )
      id_SO2_strat   = strIdx(tnames, 'GASSO2'  )
      id_H2SO4_strat = strIdx(tnames, 'GASH2SO4')
      id_TEST_strat  = strIdx(tnames, 'GASTEST' )
      CALL msg('nbtr_bin       ='//TRIM(int2str(nbtr_bin      )), modname)
      CALL msg('nbtr_sulgas    ='//TRIM(int2str(nbtr_sulgas   )), modname)
      CALL msg('id_BIN01_strat ='//TRIM(int2str(id_BIN01_strat)), modname)
      CALL msg('id_OCS_strat   ='//TRIM(int2str(id_OCS_strat  )), modname)
      CALL msg('id_SO2_strat   ='//TRIM(int2str(id_SO2_strat  )), modname)
      CALL msg('id_H2SO4_strat ='//TRIM(int2str(id_H2SO4_strat)), modname)
      CALL msg('id_TEST_strat  ='//TRIM(int2str(id_TEST_strat )), modname)
   END IF
#endif
#ifdef ISOVERIF
   CALL msg('iso_iName = '//strStack(int2str(PACK(tracers(:)%iso_iName, MASK=tracers(:)%iso_iGroup==iH2O))), modname)
#endif

END SUBROUTINE init_infotrac_phy


!==============================================================================================================================
!=== THE ROUTINE isoSelect IS USED TO SWITCH FROM AN ISOTOPE FAMILY TO ANOTHER: ISOTOPES DEPENDENT PARAMETERS ARE UPDATED
!     Single generic "isoSelect" routine, using the predefined index of the parent (fast version) or its name (first call).
!==============================================================================================================================
LOGICAL FUNCTION isoSelectByName(iName, lVerbose) RESULT(lerr)
   IMPLICIT NONE
   CHARACTER(LEN=*),  INTENT(IN)  :: iName
   LOGICAL, OPTIONAL, INTENT(IN) :: lVerbose
   INTEGER :: iIso
   LOGICAL :: lV
   lV = .FALSE.; IF(PRESENT(lVerbose)) lV = lVerbose
   iIso = strIdx(isotopes(:)%parent, iName)
   lerr = iIso == 0
   IF(lerr) THEN
      niso = 0; ntiso = 0; nzone=0; nphas=nqo; isoCheck=.FALSE.
      CALL msg('no isotope family named "'//TRIM(iName)//'"', ll=lV)
      RETURN
   END IF
   lerr = isoSelectByIndex(iIso, lV)
END FUNCTION isoSelectByName
!==============================================================================================================================
LOGICAL FUNCTION isoSelectByIndex(iIso, lVerbose) RESULT(lerr)
   IMPLICIT NONE
   INTEGER,           INTENT(IN) :: iIso
   LOGICAL, OPTIONAL, INTENT(IN) :: lVerbose
   LOGICAL :: lv
   lv = .FALSE.; IF(PRESENT(lVerbose)) lv = lVerbose
   lerr = .FALSE.
   IF(iIso == ixIso) RETURN                                      !--- Nothing to do if the index is already OK
   lerr = iIso<=0 .OR. iIso>nbIso
   CALL msg('Inconsistent isotopes family index '//TRIM(int2str(iIso))//': should be > 0 and <= '//TRIM(int2str(nbIso))//'"',&
            ll=lerr .AND. lV)
   IF(lerr) RETURN
   ixIso = iIso                                                  !--- Update currently selected family index
   isotope  => isotopes(ixIso)                                   !--- Select corresponding component
   isoKeys  => isotope%keys;     niso     = isotope%niso
   isoName  => isotope%trac;     ntiso    = isotope%ntiso
   isoZone  => isotope%zone;     nzone    = isotope%nzone
   isoPhas  => isotope%phase;    nphas    = isotope%nphas
   itZonIso => isotope%itZonIso; isoCheck = isotope%check
   iqIsoPha => isotope%iqIsoPha
END FUNCTION isoSelectByIndex
!==============================================================================================================================


END MODULE infotrac_phy
