SUBROUTINE dynetat0(fichnom,vcov,ucov,teta,q,masse,ps,phis,time)
!
!-------------------------------------------------------------------------------
! Authors: P. Le Van , L.Fairhead
!-------------------------------------------------------------------------------
! Purpose: Initial state reading.
!-------------------------------------------------------------------------------
  USE infotrac,    ONLY: nqtot, tracers, niso, iqIsoPha, tnat, alpha_ideal, iH2O
  USE strings_mod, ONLY: maxlen, msg, strStack, real2str, int2str
  USE netcdf,      ONLY: NF90_OPEN,  NF90_NOWRITE, NF90_INQ_VARID, &
                         NF90_CLOSE, NF90_GET_VAR, NF90_NoErr
  USE readTracFiles_mod, ONLY: new2oldName
  USE control_mod, ONLY: planet_type
  USE assert_eq_m, ONLY: assert_eq
  USE comvert_mod, ONLY: pa,preff
  USE comconst_mod, ONLY: cpp, daysec, dtvr, g, im, jm, kappa, lllm, omeg, rad
  USE logic_mod, ONLY: fxyhypb, ysinus
  USE serre_mod, ONLY: clon, clat, grossismx, grossismy
  USE temps_mod, ONLY: annee_ref, day_ini, day_ref, itau_dyn, start_time
  USE ener_mod, ONLY: etot0,ptot0,ztot0,stot0,ang0

  IMPLICIT NONE
  include "dimensions.h"
  include "paramet.h"
  include "comgeom2.h"
  include "description.h"
  include "iniprint.h"
!===============================================================================
! Arguments:
  CHARACTER(LEN=*), INTENT(IN) :: fichnom          !--- FILE NAME
  REAL, INTENT(OUT) ::  vcov(iip1,jjm, llm)        !--- V COVARIANT WIND
  REAL, INTENT(OUT) ::  ucov(iip1,jjp1,llm)        !--- U COVARIANT WIND
  REAL, INTENT(OUT) ::  teta(iip1,jjp1,llm)        !--- POTENTIAL TEMP.
  REAL, INTENT(OUT) ::     q(iip1,jjp1,llm,nqtot)  !--- TRACERS
  REAL, INTENT(OUT) :: masse(iip1,jjp1,llm)        !--- MASS PER CELL
  REAL, INTENT(OUT) ::    ps(iip1,jjp1)            !--- GROUND PRESSURE
  REAL, INTENT(OUT) ::  phis(iip1,jjp1)            !--- GEOPOTENTIAL
!===============================================================================
! Local variables:
  CHARACTER(LEN=maxlen) :: mesg, var, modname, oldVar
  INTEGER, PARAMETER :: length=100
  INTEGER :: iq, fID, vID, idecal, iqParent, iName, iZone, iPhase
  REAL    :: time, tab_cntrl(length)               !--- RUN PARAMS TABLE
  LOGICAL :: lOldHNO3, ll
!-------------------------------------------------------------------------------
  modname="dynetat0"

!--- Initial state file opening
  var=fichnom
  CALL err(NF90_OPEN(var,NF90_NOWRITE,fID),"open",var)
  CALL get_var1("controle",tab_cntrl)

!!! AS: idecal is a hack to be able to read planeto starts...
!!!     .... while keeping everything OK for LMDZ EARTH
  IF(planet_type=="generic") THEN
    CALL msg('NOTE NOTE NOTE : Planeto-like start files', modname)
    idecal = 4
    annee_ref  = 2000
  ELSE
    CALL msg('NOTE NOTE NOTE : Earth-like start files', modname)
    idecal = 5
    annee_ref  = tab_cntrl(5)
  END IF
  im         = tab_cntrl(1)
  jm         = tab_cntrl(2)
  lllm       = tab_cntrl(3)
  day_ref    = tab_cntrl(4)
  rad        = tab_cntrl(idecal+1)
  omeg       = tab_cntrl(idecal+2)
  g          = tab_cntrl(idecal+3)
  cpp        = tab_cntrl(idecal+4)
  kappa      = tab_cntrl(idecal+5)
  daysec     = tab_cntrl(idecal+6)
  dtvr       = tab_cntrl(idecal+7)
  etot0      = tab_cntrl(idecal+8)
  ptot0      = tab_cntrl(idecal+9)
  ztot0      = tab_cntrl(idecal+10)
  stot0      = tab_cntrl(idecal+11)
  ang0       = tab_cntrl(idecal+12)
  pa         = tab_cntrl(idecal+13)
  preff      = tab_cntrl(idecal+14)
!
  clon       = tab_cntrl(idecal+15)
  clat       = tab_cntrl(idecal+16)
  grossismx  = tab_cntrl(idecal+17)
  grossismy  = tab_cntrl(idecal+18)
!
  IF ( tab_cntrl(idecal+19)==1. )  THEN
    fxyhypb  = .TRUE.
!   dzoomx   = tab_cntrl(25)
!   dzoomy   = tab_cntrl(26)
!   taux     = tab_cntrl(28)
!   tauy     = tab_cntrl(29)
  ELSE
    fxyhypb = .FALSE.
    ysinus  = tab_cntrl(idecal+22)==1.
  END IF

  day_ini    = tab_cntrl(30)
  itau_dyn   = tab_cntrl(31)
  start_time = tab_cntrl(32)

!-------------------------------------------------------------------------------
  CALL msg('rad, omeg, g, cpp, kappa = '//TRIM(strStack(real2str([rad,omeg,g,cpp,kappa]))), modname)
  CALL check_dim(im,iim,'im','im')
  CALL check_dim(jm,jjm,'jm','jm')
  CALL check_dim(lllm,llm,'lm','lllm')
  CALL get_var1("rlonu",rlonu)
  CALL get_var1("rlatu",rlatu)
  CALL get_var1("rlonv",rlonv)
  CALL get_var1("rlatv",rlatv)
  CALL get_var2("cu"   ,cu)
  CALL get_var2("cv"   ,cv)
  CALL get_var2("aire" ,aire)
  var="temps"
  IF(NF90_INQ_VARID(fID,var,vID)/=NF90_NoErr) THEN
    CALL msg('missing field <temps> ; trying with <Time>', modname)
    var="Time"
    CALL err(NF90_INQ_VARID(fID,var,vID),"inq",var)
  END IF
  CALL err(NF90_GET_VAR(fID,vID,time),"get",var)
  CALL get_var2("phisinit",phis)
  CALL get_var3("ucov",ucov)
  CALL get_var3("vcov",vcov)
  CALL get_var3("teta",teta)
  CALL get_var3("masse",masse)
  CALL get_var2("ps",ps)

!--- Tracers
  ll=.FALSE.
#ifdef REPROBUS
  !--- Old convention: HNO3 = g+l and HNO3_g = g ; new convention: HNO3tot = g+l and HNO3 = g
  ll = NF90_INQ_VARID(fID,var,vID), "inq", 'HNO3tot') /= NF90_NoErr
#endif
  DO iq=1,nqtot
    var = tracers(iq)%name
    oldVar = new2oldName(var)
    lOldHNO3 = ll .AND. (var == 'HNO3' .OR. var == 'HNO3_g')
    !--------------------------------------------------------------------------------------------------------------------------
    IF(NF90_INQ_VARID(fID, var, vID) == NF90_NoErr .AND. .NOT.lOldHNO3) THEN             !=== REGULAR CASE
      CALL err(NF90_GET_VAR(fID,vID,q(:,:,:,iq)),"get",var)
    !--------------------------------------------------------------------------------------------------------------------------
    ELSE IF(NF90_INQ_VARID(fID, oldVar, vID) == NF90_NoErr) THEN                         !=== OLD NAME
      CALL msg('Tracer <'//TRIM(var)//'> is missing => initialized to <'//TRIM(oldVar)//'>', modname)
      CALL err(NF90_GET_VAR(fID,vID,q(:,:,:,iq)),"get",oldVar)
    !--------------------------------------------------------------------------------------------------------------------------
#ifdef REPROBUS
    ELSE IF(NF90_INQ_VARID(fID, 'HNO3_g', vID) == NF90_NoErr .AND. lOldHNO3 .AND. var == 'HNO3'   ) THEN
      CALL msg('Tracer <HNO3_g> is used for <HNO3> (old convention)', modname)           !=== REPROBUS: HNO3_g INSTEAD OF HNO3
      CALL err(NF90_GET_VAR(fID,vID,q(:,:,:,iq)),"get",var)
    ELSE IF(NF90_INQ_VARID(fID, 'HNO3',   vID) == NF90_NoErr .AND. lOldHNO3 .AND. var == 'HNO3tot') THEN
      CALL msg('Tracer <HNO3> is used for <HNO3tot> (old convention)', modname)          !=== REPROBUS: HNO3 INSTEAD OF HNO3tot
      CALL err(NF90_GET_VAR(fID,vID,q(:,:,:,iq)),"get",var)
#endif
    !--------------------------------------------------------------------------------------------------------------------------
#ifdef INCA
    ELSE IF(NF90_INQ_VARID(fID, 'OX',   vID) == NF90_NoErr .AND. var == 'O3') THEN       !=== INCA: OX INSTEAD OF O3
      CALL msg('Tracer <O3> is missing => initialized to <OX>', modname)
      CALL err(NF90_GET_VAR(fID,vID,q(:,:,:,iq)),"get",'OX')
    !--------------------------------------------------------------------------------------------------------------------------
#endif
    ELSE IF(tracers(iq)%iso_iGroup == iH2O .AND. niso > 0) THEN                          !=== WATER ISOTOPES
      iName    = tracers(iq)%iso_iName
      iPhase   = tracers(iq)%iso_iPhase
      iqParent = tracers(iq)%iqParent
      IF(tracers(iq)%iso_iZone == 0) THEN
         CALL msg('Tracer <'//TRIM(var)//'> is missing => initialized with a simplified Rayleigh distillation law.', modname)
         q(:,:,:,iq) = q(:,:,:,iqParent)*tnat(iName)*(q(:,:,:,iqParent)/30.e-3)**(alpha_ideal(iName)-1.)
      ELSE
         CALL msg('Tracer <'//TRIM(var)//'> is missing => initialized to its parent isotope concentration.', modname)
         q(:,:,:,iq) = q(:,:,:,iqIsoPha(iName,iPhase))
      END IF
    !--------------------------------------------------------------------------------------------------------------------------
    ELSE                                                                                 !=== MISSING: SET TO 0
      CALL msg('Tracer <'//TRIM(var)//'> is missing => initialized to zero', modname)
      q(:,:,:,iq)=0.
    !--------------------------------------------------------------------------------------------------------------------------
    END IF
  END DO

  CALL err(NF90_CLOSE(fID),"close",fichnom)
  day_ini=day_ini+INT(time)
  time=time-INT(time)


  CONTAINS


SUBROUTINE check_dim(n1,n2,str1,str2)
  INTEGER,          INTENT(IN) :: n1, n2
  CHARACTER(LEN=*), INTENT(IN) :: str1, str2
  CHARACTER(LEN=maxlen) :: s1, s2
  IF(n1/=n2) CALL abort_gcm(TRIM(modname), 'value of "'//TRIM(str1)//'" = '//TRIM(int2str(n1))// &
   ' read in starting file differs from gcm value of "'//TRIM(str2)//'" = '//TRIM(int2str(n2)), 1)
END SUBROUTINE check_dim


SUBROUTINE get_var1(var,v)
  CHARACTER(LEN=*), INTENT(IN)  :: var
  REAL,             INTENT(OUT) :: v(:)
  CALL err(NF90_INQ_VARID(fID,var,vID),"inq",var)
  CALL err(NF90_GET_VAR(fID,vID,v),"get",var)
END SUBROUTINE get_var1


SUBROUTINE get_var2(var,v)
  CHARACTER(LEN=*), INTENT(IN)  :: var
  REAL,             INTENT(OUT) :: v(:,:)
  CALL err(NF90_INQ_VARID(fID,var,vID),"inq",var)
  CALL err(NF90_GET_VAR(fID,vID,v),"get",var)
END SUBROUTINE get_var2


SUBROUTINE get_var3(var,v)
  CHARACTER(LEN=*), INTENT(IN)  :: var
  REAL,             INTENT(OUT) :: v(:,:,:)
  CALL err(NF90_INQ_VARID(fID,var,vID),"inq",var)
  CALL err(NF90_GET_VAR(fID,vID,v),"get",var)
END SUBROUTINE get_var3


SUBROUTINE err(ierr,typ,nam)
  INTEGER,          INTENT(IN) :: ierr   !--- NetCDF ERROR CODE
  CHARACTER(LEN=*), INTENT(IN) :: typ    !--- TYPE OF OPERATION
  CHARACTER(LEN=*), INTENT(IN) :: nam    !--- FIELD/FILE NAME
  IF(ierr==NF90_NoERR) RETURN
  SELECT CASE(typ)
    CASE('inq');   mesg="Field <"//TRIM(nam)//"> is missing"
    CASE('get');   mesg="Reading failed for <"//TRIM(nam)//">"
    CASE('open');  mesg="File opening failed for <"//TRIM(nam)//">"
    CASE('close'); mesg="File closing failed for <"//TRIM(nam)//">"
  END SELECT
  CALL ABORT_gcm(TRIM(modname),TRIM(mesg),ierr)
END SUBROUTINE err

END SUBROUTINE dynetat0
