!
! $Id: aeropt_5wv_rrtm.F90 2211 2015-02-23 11:58:32Z oboucher $
!

SUBROUTINE AEROPT_5WV_RRTM(&
   pdel, m_allaer, delt, &
   RHcl, ai, flag_aerosol, &
   pplay, t_seri, &
   tausum, tau )

  USE DIMPHY
  USE aero_mod
  USE phys_local_var_mod, only: od550aer,od865aer,ec550aer,od550lt1aer

  !
  !    Yves Balkanski le 12 avril 2006
  !    Celine Deandreis
  !    Anne Cozic  Avril 2009
  !    a partir d'une sous-routine de Johannes Quaas pour les sulfates
  !    Olivier Boucher mars 2014 pour adaptation RRTM
  !
  ! Refractive indices for seasalt come from Shettle and Fenn (1979)
  !
  ! Refractive indices from water come from Hale and Querry (1973)
  !
  ! Refractive indices from Ammonium Sulfate Toon and Pollack (1976)
  !
  ! Refractive indices for Dust, internal mixture of minerals coated with 1.5% hematite 
  ! by Volume (Balkanski et al., 2006)
  !
  ! Refractive indices for POM: Kinne (pers. Communication 
  !
  ! Refractive index for BC from Shettle and Fenn (1979)
  !
  ! Shettle, E. P., & Fenn, R. W. (1979), Models for the aerosols of the lower atmosphere and 
  ! the effects of humidity variations on their optical properties, U.S. Air Force Geophysics 
  ! Laboratory Rept. AFGL-TR-79-0214, Hanscomb Air Force Base, MA.
  !
  ! Hale, G. M. and M. R. Querry, Optical constants of water in the 200-nm to 200-m 
  ! wavelength region, Appl. Opt., 12, 555-563, 1973.
  !
  ! Toon, O. B. and J. B. Pollack, The optical constants of several atmospheric aerosol species:
  ! Ammonium sulfate, aluminum oxide, and sodium chloride, J. Geohys. Res., 81, 5733-5748,
  ! 1976.
  !
  ! Balkanski, Y., M. Schulz, T. Claquin And O. Boucher, Reevaluation of mineral aerosol 
  ! radiative forcings suggests a better agreement with satellite and AERONET data, Atmospheric 
  ! Chemistry and Physics Discussions., 6, pp 8383-8419, 2006.
  !
  IMPLICIT NONE
  INCLUDE "YOMCST.h"
  !
  ! Input arguments:
  !
  REAL, DIMENSION(klon,klev), INTENT(in)   :: pdel
  REAL, INTENT(in)                         :: delt
  REAL, DIMENSION(klon,klev,naero_tot), INTENT(in) :: m_allaer
  REAL, DIMENSION(klon,klev), INTENT(in)   :: RHcl     ! humidite relative ciel clair
  INTEGER,INTENT(in)                       :: flag_aerosol
  REAL, DIMENSION(klon,klev), INTENT(in)   :: pplay
  REAL, DIMENSION(klon,klev), INTENT(in)   :: t_seri
  !
  ! Output arguments:
  !
  REAL, DIMENSION(klon), INTENT(out)                      :: ai      ! POLDER aerosol index 
  REAL, DIMENSION(klon,nwave,naero_tot), INTENT(out)      :: tausum
  REAL, DIMENSION(klon,klev,nwave,naero_tot), INTENT(out) :: tau

  !
  ! Local
  !
  INTEGER, PARAMETER :: las = nwave
  LOGICAL :: soluble
  
  INTEGER :: i, k, m, aerindex
  INTEGER :: spsol, spinsol, la
  INTEGER :: RH_num(klon,klev)
  INTEGER, PARAMETER :: la443 = 1
  INTEGER, PARAMETER :: la550 = 2
  INTEGER, PARAMETER :: la670 = 3
  INTEGER, PARAMETER :: la765 = 4
  INTEGER, PARAMETER :: la865 = 5
  INTEGER, PARAMETER :: nbre_RH=12
  INTEGER, PARAMETER :: naero_soluble=7   !  1- BC soluble; 2- POM soluble; 3- SO4 acc.
                                          !  4- SO4 coarse; 5 seasalt super-C; 6 seasalt coarse; 7 seasalt acc.
  INTEGER, PARAMETER :: naero_insoluble=3 !  1- Dust; 2- BC insoluble; 3- POM insoluble

  REAL :: zrho

  REAL, PARAMETER :: RH_tab(nbre_RH)=(/0.,10.,20.,30.,40.,50.,60.,70.,80.,85.,90.,95./)
  REAL, PARAMETER :: RH_MAX=95.
  REAL :: delta(klon,klev), rh(klon,klev)
  REAL :: tau_ae5wv_int   ! Intermediate computation of epaisseur optique aerosol
  REAL :: od670aer(klon)  ! epaisseur optique aerosol extinction 670 nm
  REAL :: fac
  REAL :: zdp1(klon,klev) 
  INTEGER, ALLOCATABLE, DIMENSION(:)  :: aerosol_name
  INTEGER :: nb_aer, itau
  LOGICAL :: ok_itau
  
  REAL :: dh(KLON,KLEV)
  
   ! Soluble components 1- BC soluble; 2- POM soluble; 3- SO4 acc.; 4- SO4 coarse; 5 seasalt super-coarse; 6 seasalt coarse; 7 seasalt acc.
  REAL :: alpha_aers_5wv(nbre_RH,las,naero_soluble)   ! Ext. coeff. ** m2/g 
   ! Insoluble components 1- Dust: 2- BC; 3- POM
  REAL :: alpha_aeri_5wv(las,naero_insoluble)         ! Ext. coeff. ** m2/g 

  REAL, DIMENSION(klon,klev,naero_tot) :: mass_temp
  
  !
  ! Proprietes optiques
  !
  REAL :: fact_RH(nbre_RH)
  LOGICAL :: used_tau(naero_tot)
  INTEGER :: n

! From here on we look at the optical parameters at 5 wavelengths:  
! 443nm, 550, 670, 765 and 865 nm 
!                                   le 12 AVRIL 2006 
!  
 DATA alpha_aers_5wv/ & 
                                ! bc soluble 
       7.930,7.930,7.930,7.930,7.930,7.930,     & 
       7.930,7.930,10.893,12.618,14.550,16.613, & 
       7.658,7.658,7.658,7.658,7.658,7.658,     & 
       7.658,7.658,10.351,11.879,13.642,15.510, & 
       7.195,7.195,7.195,7.195,7.195,7.195,     & 
       7.195,7.195,9.551,10.847,12.381,13.994,  & 
       6.736,6.736,6.736,6.736,6.736,6.736,     & 
       6.736,6.736,8.818,9.938,11.283,12.687,   & 
       6.277,6.277,6.277,6.277,6.277,6.277,     & 
       6.277,6.277,8.123,9.094,10.275,11.501,   & 
                                ! pom soluble 
       6.676,6.676,6.676,6.676,6.710,6.934,   & 
       7.141,7.569,8.034,8.529,9.456,10.511,  & 
       5.109,5.109,5.109,5.109,5.189,5.535,   & 
       5.960,6.852,8.008,9.712,12.897,19.676, & 
       3.718,3.718,3.718,3.718,3.779,4.042,   & 
       4.364,5.052,5.956,7.314,9.896,15.688,  & 
       2.849,2.849,2.849,2.849,2.897,3.107,   & 
       3.365,3.916,4.649,5.760,7.900,12.863,  & 
       2.229,2.229,2.229,2.229,2.268,2.437,   & 
       2.645,3.095,3.692,4.608,6.391,10.633,  & 
                                ! Sulfate (Accumulation) 
       5.751,6.215,6.690,7.024,7.599,8.195,      & 
       9.156,10.355,12.660,14.823,18.908,24.508, & 
       4.320,4.675,5.052,5.375,5.787,6.274,      & 
       7.066,8.083,10.088,12.003,15.697,21.133,  & 
       3.079,3.351,3.639,3.886,4.205,4.584,      & 
       5.206,6.019,7.648,9.234,12.391,17.220,    & 
       2.336,2.552,2.781,2.979,3.236,3.540,      & 
       4.046,4.711,6.056,7.388,10.093,14.313,    & 
       1.777,1.949,2.134,2.292,2.503,2.751,      & 
       3.166,3.712,4.828,5.949,8.264,11.922,     & 
                                ! Sulfate (Coarse) 
       5.751,6.215,6.690,7.024,7.599,8.195,      & 
       9.156,10.355,12.660,14.823,18.908,24.508, & 
       4.320,4.675,5.052,5.375,5.787,6.274,      & 
       7.066,8.083,10.088,12.003,15.697,21.133,  & 
       3.079,3.351,3.639,3.886,4.205,4.584,      & 
       5.206,6.019,7.648,9.234,12.391,17.220,    & 
       2.336,2.552,2.781,2.979,3.236,3.540,      & 
       4.046,4.711,6.056,7.388,10.093,14.313,    & 
       1.777,1.949,2.134,2.292,2.503,2.751,      & 
       3.166,3.712,4.828,5.949,8.264,11.922,     & 
                           ! seasalt seasalt Super Coarse Soluble (SS)
        0.218, 0.272, 0.293, 0.316, 0.343, 0.380, &
        0.429, 0.501, 0.636, 0.755, 0.967, 1.495, & 
        0.221, 0.275, 0.297, 0.320, 0.348, 0.383, &
        0.432, 0.509, 0.640, 0.759, 0.972, 1.510, &
        0.224, 0.279, 0.301, 0.324, 0.352, 0.388, &
        0.438, 0.514, 0.647, 0.768, 0.985, 1.514, &
        0.227, 0.282, 0.303, 0.327, 0.356, 0.392, &
        0.441, 0.518, 0.652, 0.770, 0.987, 1.529, &
        0.230, 0.285, 0.306, 0.330, 0.359, 0.396, &
        0.446, 0.522, 0.656, 0.777, 0.993, 1.539, &
                           ! seasalt seasalt Coarse Soluble (CS)      
        0.578, 0.706, 0.756, 0.809, 0.876, 0.964, &
        1.081, 1.256, 1.577, 1.858, 2.366, 3.613, &
        0.598, 0.725, 0.779, 0.833, 0.898, 0.990, &
        1.109, 1.290, 1.609, 1.889, 2.398, 3.682, &
        0.619, 0.750, 0.802, 0.857, 0.927, 1.022, &
        1.141, 1.328, 1.648, 1.939, 2.455, 3.729, &
        0.633, 0.767, 0.820, 0.879, 0.948, 1.044, &
        1.167, 1.353, 1.683, 1.969, 2.491, 3.785, & 
        0.648, 0.785, 0.838, 0.896, 0.967, 1.066, &
        1.192, 1.381, 1.714, 2.006, 2.531, 3.836, &
                           ! seasalt seasalt Accumulation Soluble (AS)
        4.432, 5.899, 6.505, 7.166, 7.964, 7.962, &
        9.232,11.257,14.979,18.337,24.223,37.811, &
        3.298, 4.569, 5.110, 5.709, 6.446, 6.268, &
        7.396, 9.246,12.787,16.113,22.197,37.136, &
        2.340, 3.358, 3.803, 4.303, 4.928, 4.696, &
        5.629, 7.198,10.308,13.342,19.120,34.296, &
        1.789, 2.626, 2.999, 3.422, 3.955, 3.730, &
        4.519, 5.864, 8.593,11.319,16.653,31.331, &
        1.359, 2.037, 2.343, 2.693, 3.139, 2.940, &
        3.596, 4.729, 7.076, 9.469,14.266,28.043 /

  DATA alpha_aeri_5wv/ &
                                 ! dust insoluble 
        0.759, 0.770, 0.775, 0.775, 0.772, & 
                                 !!jb bc insoluble 
        11.536,10.033, 8.422, 7.234, 6.270, & 
                                 ! pom insoluble 
        5.042, 3.101, 1.890, 1.294, 0.934/ 
  ! 
  ! Initialisations
  ai(:) = 0.
  tausum(:,:,:) = 0.

  DO k=1, klev
    DO i=1, klon
      zrho=pplay(i,k)/t_seri(i,k)/RD                  ! kg/m3
      dh(i,k)=pdel(i,k)/(RG*zrho)
!CDIR UNROLL=naero_spc
      mass_temp(i,k,:) = m_allaer(i,k,:) / zrho / 1.e+9
      zdp1(i,k)=pdel(i,k)/(RG*delt)     ! air mass auxiliary  variable --> zdp1 [kg/(m^2 *s)]
    ENDDO
  ENDDO

  IF (flag_aerosol .EQ. 1) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M_phy
     aerosol_name(2) = id_CSSO4M_phy
  ELSEIF (flag_aerosol .EQ. 2) THEN
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASBCM_phy
     aerosol_name(2) = id_AIBCM_phy
  ELSEIF (flag_aerosol .EQ. 3) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASPOMM_phy
     aerosol_name(2) = id_AIPOMM_phy
  ELSEIF (flag_aerosol .EQ. 4) THEN 
     nb_aer = 3
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CSSSM_phy
     aerosol_name(2) = id_SSSSM_phy
     aerosol_name(3) = id_ASSSM_phy
  ELSEIF (flag_aerosol .EQ. 5) THEN 
     nb_aer = 1
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CIDUSTM_phy
  ELSEIF (flag_aerosol .EQ. 6) THEN 
     nb_aer = 10
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M_phy      
     aerosol_name(2) = id_ASBCM_phy
     aerosol_name(3) = id_AIBCM_phy
     aerosol_name(4) = id_ASPOMM_phy
     aerosol_name(5) = id_AIPOMM_phy
     aerosol_name(6) = id_CSSSM_phy
     aerosol_name(7) = id_SSSSM_phy
     aerosol_name(8) = id_ASSSM_phy
     aerosol_name(9) = id_CIDUSTM_phy
     aerosol_name(10) = id_CSSO4M_phy
  ENDIF

  ! 
  ! Loop over modes, use of precalculated nmd and corresponding sigma
  !    loop over wavelengths
  !    for each mass species in mode
  !      interpolate from Sext to retrieve Sext_at_gridpoint_per_species
  !      compute optical_thickness_at_gridpoint_per_species
  !
  ! Calculations that need to be done since we are not in the subroutines INCA
  !      

!CDIR ON_ADB(RH_tab)
!CDIR ON_ADB(fact_RH)
!CDIR NOVECTOR
  DO n=1,nbre_RH-1
    fact_RH(n)=1./(RH_tab(n+1)-RH_tab(n))
  ENDDO
   
  DO k=1, KLEV
!CDIR ON_ADB(RH_tab)
!CDIR ON_ADB(fact_RH)
    DO i=1, KLON
      rh(i,k)=MIN(RHcl(i,k)*100.,RH_MAX)
      RH_num(i,k) = INT( rh(i,k)/10. + 1.)
      IF (rh(i,k).GT.85.) RH_num(i,k)=10
      IF (rh(i,k).GT.90.) RH_num(i,k)=11
      delta(i,k)=(rh(i,k)-RH_tab(RH_num(i,k)))*fact_RH(RH_num(i,k))
    ENDDO
  ENDDO

!CDIR SHORTLOOP  
  used_tau(:)=.FALSE.
    
  DO m=1,nb_aer   ! tau is only computed for each mass    
    fac=1.0
    IF (aerosol_name(m).EQ.id_ASBCM_phy) THEN
        soluble=.TRUE.
        spsol=1
    ELSEIF (aerosol_name(m).EQ.id_ASPOMM_phy) THEN 
        soluble=.TRUE.
        spsol=2 
    ELSEIF (aerosol_name(m).EQ.id_ASSO4M_phy) THEN
        soluble=.TRUE.
        spsol=3
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
    ELSEIF (aerosol_name(m).EQ.id_CSSO4M_phy) THEN
        soluble=.TRUE.
        spsol=4
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
    ELSEIF (aerosol_name(m).EQ.id_SSSSM_phy) THEN 
        soluble=.TRUE.
        spsol=5
    ELSEIF (aerosol_name(m).EQ.id_CSSSM_phy) THEN 
        soluble=.TRUE.
        spsol=6
    ELSEIF (aerosol_name(m).EQ.id_ASSSM_phy) THEN
        soluble=.TRUE.
        spsol=7
    ELSEIF (aerosol_name(m).EQ.id_CIDUSTM_phy) THEN 
        soluble=.FALSE.
        spinsol=1
    ELSEIF  (aerosol_name(m).EQ.id_AIBCM_phy) THEN 
        soluble=.FALSE.
        spinsol=2
    ELSEIF (aerosol_name(m).EQ.id_AIPOMM_phy) THEN 
        soluble=.FALSE.
        spinsol=3
    ELSE 
        CYCLE
    ENDIF

    IF (soluble) then
      used_tau(spsol)=.TRUE.
    ELSE
      used_tau(naero_soluble+spinsol)=.TRUE.
    ENDIF

    aerindex=aerosol_name(m)

    DO la=1,las

      IF (soluble) THEN            ! For soluble aerosol

          DO k=1, KLEV
            DO i=1, KLON
              tau_ae5wv_int = alpha_aers_5wv(RH_num(i,k),la,spsol)+DELTA(i,k)* &
                             (alpha_aers_5wv(RH_num(i,k)+1,la,spsol) - & 
                              alpha_aers_5wv(RH_num(i,k),la,spsol))
              tau(i,k,la,aerindex) = mass_temp(i,k,aerindex)*1000.*zdp1(i,k)*   &
                                    tau_ae5wv_int*delt*fac
              tausum(i,la,aerindex)=tausum(i,la,aerindex)+tau(i,k,la,aerindex)
            ENDDO
          ENDDO
 
      ELSE                         ! For insoluble aerosol

        DO k=1, KLEV
          DO i=1, KLON
            tau_ae5wv_int = alpha_aeri_5wv(la,spinsol)
            tau(i,k,la,aerindex) = mass_temp(i,k,aerindex)*1000.*zdp1(i,k)* &
                                   tau_ae5wv_int*delt*fac
            tausum(i,la,aerindex)= tausum(i,la,aerindex)+tau(i,k,la,aerindex)
          ENDDO
        ENDDO

      ENDIF

    ENDDO   ! Boucle sur les longueurs d'onde
  ENDDO     ! Boucle sur les masses de traceurs

  DO m=1,naero_tot
    IF (.NOT.used_tau(m)) tau(:,:,:,m)=0.
  ENDDO  

  DO i=1, klon
     od550aer(i)=0.
     DO m=1,naero_tot
        od550aer(i)=od550aer(i)+tausum(i,la550,m)
     END DO
  END DO

  DO i=1, klon
     od670aer(i)=0.
     DO m=1,naero_tot
        od670aer(i)=od670aer(i)+tausum(i,la670,m)
     END DO
  END DO

  DO i=1, klon
     od865aer(i)=0.
     DO m=1,naero_tot
        od865aer(i)=od865aer(i)+tausum(i,la865,m)
     END DO
  END DO

  DO i=1, klon
     DO k=1, KLEV
        ec550aer(i,k)=0.
        DO m=1,naero_tot
           ec550aer(i,k)=ec550aer(i,k)+tau(i,k,la550,m)/dh(i,k)
        END DO
     END DO
  END DO
  
  DO i=1, klon
    ai(i)=-LOG(MAX(od670aer(i),1.e-8)/MAX(od865aer(i),1.e-8))/LOG(670./865.)
  ENDDO

  od550lt1aer(:)=tausum(:,la550,id_ASSO4M_phy)+tausum(:,la550,id_ASBCM_phy) +tausum(:,la550,id_AIBCM_phy)+ &
                 tausum(:,la550,id_ASPOMM_phy)+tausum(:,la550,id_AIPOMM_phy)+tausum(:,la550,id_ASSSM_phy)+ &
                 0.03*tausum(:,la550,id_CSSSM_phy)+0.4*tausum(:,la550,id_CIDUSTM_phy)

  DEALLOCATE(aerosol_name) 
  
END SUBROUTINE AEROPT_5WV_RRTM
