!
! $Id: aeropt_5wv.F90 1716 2013-01-22 13:36:30Z acozic $
!

SUBROUTINE AEROPT_5WV_RRTM(&
   pdel, m_allaer, delt, &
   RHcl, ai, flag_aerosol, &
   pplay, t_seri, &
   tausum, tau )

  USE DIMPHY
  USE aero_mod
  USE phys_local_var_mod, only: od550aer,od865aer,ec550aer,od550lt1aer

  !
  !    Yves Balkanski le 12 avril 2006
  !    Celine Deandreis
  !    Anne Cozic  Avril 2009
  !    a partir d'une sous-routine de Johannes Quaas pour les sulfates
  !    Olivier Boucher mars 2014 pour adaptation RRTM
  !
  ! Refractive indices for seasalt come from Shettle and Fenn (1979)
  !
  ! Refractive indices from water come from Hale and Querry (1973)
  !
  ! Refractive indices from Ammonium Sulfate Toon and Pollack (1976)
  !
  ! Refractive indices for Dust, internal mixture of minerals coated with 1.5% hematite 
  ! by Volume (Balkanski et al., 2006)
  !
  ! Refractive indices for POM: Kinne (pers. Communication 
  !
  ! Refractive index for BC from Shettle and Fenn (1979)
  !
  ! Shettle, E. P., & Fenn, R. W. (1979), Models for the aerosols of the lower atmosphere and 
  ! the effects of humidity variations on their optical properties, U.S. Air Force Geophysics 
  ! Laboratory Rept. AFGL-TR-79-0214, Hanscomb Air Force Base, MA.
  !
  ! Hale, G. M. and M. R. Querry, Optical constants of water in the 200-nm to 200-m 
  ! wavelength region, Appl. Opt., 12, 555-563, 1973.
  !
  ! Toon, O. B. and J. B. Pollack, The optical constants of several atmospheric aerosol species:
  ! Ammonium sulfate, aluminum oxide, and sodium chloride, J. Geohys. Res., 81, 5733-5748,
  ! 1976.
  !
  ! Balkanski, Y., M. Schulz, T. Claquin And O. Boucher, Reevaluation of mineral aerosol 
  ! radiative forcings suggests a better agreement with satellite and AERONET data, Atmospheric 
  ! Chemistry and Physics Discussions., 6, pp 8383-8419, 2006.
  !
  IMPLICIT NONE
  INCLUDE "YOMCST.h"
  !
  ! Input arguments:
  !
  REAL, DIMENSION(klon,klev), INTENT(in)   :: pdel
  REAL, INTENT(in)                         :: delt
  REAL, DIMENSION(klon,klev,naero_spc), INTENT(in) :: m_allaer
  REAL, DIMENSION(klon,klev), INTENT(in)   :: RHcl     ! humidite relative ciel clair
  INTEGER,INTENT(in)                       :: flag_aerosol
  REAL, DIMENSION(klon,klev), INTENT(in)   :: pplay
  REAL, DIMENSION(klon,klev), INTENT(in)   :: t_seri
  !
  ! Output arguments:
  !
  REAL, DIMENSION(klon), INTENT(out)                      :: ai      ! POLDER aerosol index 
  REAL, DIMENSION(klon,nwave,naero_tot), INTENT(out)      :: tausum
  REAL, DIMENSION(klon,klev,nwave,naero_tot), INTENT(out) :: tau

  !
  ! Local
  !
  INTEGER, PARAMETER :: las = nwave
  LOGICAL :: soluble
  
  INTEGER :: i, k, m
  INTEGER :: spsol, spinsol, la
  INTEGER :: RH_num(klon,klev)
  INTEGER, PARAMETER :: la443 = 1
  INTEGER, PARAMETER :: la550 = 2
  INTEGER, PARAMETER :: la670 = 3
  INTEGER, PARAMETER :: la765 = 4
  INTEGER, PARAMETER :: la865 = 5
  INTEGER, PARAMETER :: nbre_RH=12
  INTEGER, PARAMETER :: naero_soluble=7   !  1- BC soluble; 2- POM soluble; 3- SO4 acc.
                                          !  4- SO4 coarse; 5 seasalt super-C; 6 seasalt coarse; 7 seasalt acc.
  INTEGER, PARAMETER :: naero_insoluble=3 !  1- Dust; 2- BC insoluble; 3- POM insoluble

  REAL :: zrho

  REAL, PARAMETER :: RH_tab(nbre_RH)=(/0.,10.,20.,30.,40.,50.,60.,70.,80.,85.,90.,95./)
  REAL, PARAMETER :: RH_MAX=95.
  REAL :: delta(klon,klev), rh(klon,klev)
  REAL :: tau_ae5wv_int   ! Intermediate computation of epaisseur optique aerosol
  REAL :: od670aer(klon)  ! epaisseur optique aerosol extinction 670 nm
  REAL :: fac
  REAL :: zdp1(klon,klev) 
  INTEGER, ALLOCATABLE, DIMENSION(:)  :: aerosol_name
  INTEGER :: nb_aer
  
  REAL :: dh(KLON,KLEV)
  
   ! Soluble components 1- BC soluble; 2- POM soluble; 3- SO4 acc.; 4- SO4 coarse; 5 seasalt super-coarse; 6 seasalt coarse; 7 seasalt acc.
  REAL :: alpha_aers_5wv(nbre_RH,las,naero_soluble)   ! Ext. coeff. ** m2/g 
   ! Insoluble components 1- Dust: 2- BC; 3- POM
  REAL :: alpha_aeri_5wv(las,naero_insoluble)         ! Ext. coeff. ** m2/g 

  REAL, DIMENSION(klon,klev,naero_spc) :: mass_temp
  
  !
  ! Proprietes optiques
  !
  REAL :: fact_RH(nbre_RH)
  LOGICAL :: used_tau(naero_tot)
  INTEGER :: n
  
! From here on we look at the optical parameters at 5 wavelengths:  
! 443nm, 550, 670, 765 and 865 nm 
!                                   le 12 AVRIL 2006 
!  
 DATA alpha_aers_5wv/ & 
                                ! bc soluble 
       7.930,7.930,7.930,7.930,7.930,7.930,     & 
       7.930,7.930,10.893,12.618,14.550,16.613, & 
       7.658,7.658,7.658,7.658,7.658,7.658,     & 
       7.658,7.658,10.351,11.879,13.642,15.510, & 
       7.195,7.195,7.195,7.195,7.195,7.195,     & 
       7.195,7.195,9.551,10.847,12.381,13.994,  & 
       6.736,6.736,6.736,6.736,6.736,6.736,     & 
       6.736,6.736,8.818,9.938,11.283,12.687,   & 
       6.277,6.277,6.277,6.277,6.277,6.277,     & 
       6.277,6.277,8.123,9.094,10.275,11.501,   & 
                                ! pom soluble 
       6.676,6.676,6.676,6.676,6.710,6.934,   & 
       7.141,7.569,8.034,8.529,9.456,10.511,  & 
       5.109,5.109,5.109,5.109,5.189,5.535,   & 
       5.960,6.852,8.008,9.712,12.897,19.676, & 
       3.718,3.718,3.718,3.718,3.779,4.042,   & 
       4.364,5.052,5.956,7.314,9.896,15.688,  & 
       2.849,2.849,2.849,2.849,2.897,3.107,   & 
       3.365,3.916,4.649,5.760,7.900,12.863,  & 
       2.229,2.229,2.229,2.229,2.268,2.437,   & 
       2.645,3.095,3.692,4.608,6.391,10.633,  & 
                                ! Sulfate (Accumulation) 
       5.751,6.215,6.690,7.024,7.599,8.195,      & 
       9.156,10.355,12.660,14.823,18.908,24.508, & 
       4.320,4.675,5.052,5.375,5.787,6.274,      & 
       7.066,8.083,10.088,12.003,15.697,21.133,  & 
       3.079,3.351,3.639,3.886,4.205,4.584,      & 
       5.206,6.019,7.648,9.234,12.391,17.220,    & 
       2.336,2.552,2.781,2.979,3.236,3.540,      & 
       4.046,4.711,6.056,7.388,10.093,14.313,    & 
       1.777,1.949,2.134,2.292,2.503,2.751,      & 
       3.166,3.712,4.828,5.949,8.264,11.922,     & 
                                ! Sulfate (Coarse) 
       5.751,6.215,6.690,7.024,7.599,8.195,      & 
       9.156,10.355,12.660,14.823,18.908,24.508, & 
       4.320,4.675,5.052,5.375,5.787,6.274,      & 
       7.066,8.083,10.088,12.003,15.697,21.133,  & 
       3.079,3.351,3.639,3.886,4.205,4.584,      & 
       5.206,6.019,7.648,9.234,12.391,17.220,    & 
       2.336,2.552,2.781,2.979,3.236,3.540,      & 
       4.046,4.711,6.056,7.388,10.093,14.313,    & 
       1.777,1.949,2.134,2.292,2.503,2.751,      & 
       3.166,3.712,4.828,5.949,8.264,11.922,     & 
                           ! seasalt seasalt Super Coarse Soluble (SS)
        0.218, 0.240, 0.262, 0.311, 0.391, 0.452, &
        0.544, 0.614, 0.886, 1.018, 1.260, 2.336, &
        0.221, 0.243, 0.266, 0.315, 0.396, 0.455, &
        0.551, 0.622, 0.896, 1.029, 1.270, 2.352, &
        0.224, 0.246, 0.270, 0.319, 0.400, 0.461, &
        0.558, 0.626, 0.904, 1.039, 1.278, 2.362, &
        0.227, 0.249, 0.272, 0.322, 0.404, 0.466, &
        0.560, 0.632, 0.907, 1.041, 1.284, 2.375, &
        0.230, 0.252, 0.276, 0.325, 0.408, 0.469, &
        0.567, 0.636, 0.917, 1.052, 1.293, 2.383, &
                           ! seasalt seasalt Coarse Soluble (CS)      
        0.578, 0.632, 0.691, 0.814, 1.017, 1.171, &
        1.407, 1.585, 2.264, 2.594, 3.197, 5.882, &
        0.598, 0.654, 0.711, 0.836, 1.041, 1.201, &
        1.445, 1.622, 2.313, 2.647, 3.256, 5.961, &
        0.619, 0.676, 0.736, 0.862, 1.072, 1.234, &
        1.484, 1.659, 2.361, 2.698, 3.304, 6.014, &
        0.633, 0.692, 0.754, 0.884, 1.095, 1.262, &
        1.510, 1.693, 2.395, 2.734, 3.352, 6.090, &
        0.648, 0.708, 0.771, 0.902, 1.119, 1.288, &
        1.545, 1.725, 2.441, 2.784, 3.402, 6.144, &
                           ! seasalt seasalt Accumulation Soluble (AS)
        4.432, 5.160, 5.940, 7.646,10.520,11.464, &
       14.686,16.949,25.977,30.118,37.327,65.333, &
        3.298, 3.916, 4.594, 6.127, 8.845, 9.446, &
       12.511,14.710,24.073,28.546,36.578,69.050, &
        2.340, 2.826, 3.371, 4.638, 6.992, 7.360, &
       10.047,12.016,20.909,25.349,33.605,69.554, &
        1.789, 2.185, 2.633, 3.698, 5.733, 5.994, &
        8.345,10.094,18.293,22.511,30.548,67.717, &
        1.359, 1.676, 2.040, 2.916, 4.637, 4.829, &
        6.843, 8.364,15.723,19.614,27.195,64.359 /

  DATA alpha_aeri_5wv/ &
                                 ! dust insoluble 
        0.759, 0.770, 0.775, 0.775, 0.772, & 
                                 !!jb bc insoluble 
        11.536,10.033, 8.422, 7.234, 6.270, & 
                                 ! pom insoluble 
        5.042, 3.101, 1.890, 1.294, 0.934/ 
  ! 
  ! Initialisations
  ai(:) = 0.
  tausum(:,:,:) = 0.

  DO k=1, klev
    DO i=1, klon
      zrho=pplay(i,k)/t_seri(i,k)/RD                  ! kg/m3
      dh(i,k)=pdel(i,k)/(RG*zrho)
!CDIR UNROLL=naero_spc
      mass_temp(i,k,:) = m_allaer(i,k,:) / zrho / 1.e+9
      zdp1(i,k)=pdel(i,k)/(RG*delt)     ! air mass auxiliary  variable --> zdp1 [kg/(m^2 *s)]
    ENDDO
  ENDDO

  IF (flag_aerosol .EQ. 1) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M
     aerosol_name(2) = id_CSSO4M
  ELSEIF (flag_aerosol .EQ. 2) THEN
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASBCM
     aerosol_name(2) = id_AIBCM
  ELSEIF (flag_aerosol .EQ. 3) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASPOMM
     aerosol_name(2) = id_AIPOMM
  ELSEIF (flag_aerosol .EQ. 4) THEN 
     nb_aer = 3
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CSSSM
     aerosol_name(2) = id_SSSSM
     aerosol_name(3) = id_ASSSM
  ELSEIF (flag_aerosol .EQ. 5) THEN 
     nb_aer = 1
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CIDUSTM
  ELSEIF (flag_aerosol .EQ. 6) THEN 
     nb_aer = 10
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M      
     aerosol_name(2) = id_ASBCM
     aerosol_name(3) = id_AIBCM
     aerosol_name(4) = id_ASPOMM
     aerosol_name(5) = id_AIPOMM
     aerosol_name(6) = id_CSSSM
     aerosol_name(7) = id_SSSSM
     aerosol_name(8) = id_ASSSM
     aerosol_name(9) = id_CIDUSTM
     aerosol_name(10) = id_CSSO4M
  ENDIF

  ! 
  ! Loop over modes, use of precalculated nmd and corresponding sigma
  !    loop over wavelengths
  !    for each mass species in mode
  !      interpolate from Sext to retrieve Sext_at_gridpoint_per_species
  !      compute optical_thickness_at_gridpoint_per_species
  !
  ! Calculations that need to be done since we are not in the subroutines INCA
  !      

!CDIR ON_ADB(RH_tab)
!CDIR ON_ADB(fact_RH)
!CDIR NOVECTOR
  DO n=1,nbre_RH-1
    fact_RH(n)=1./(RH_tab(n+1)-RH_tab(n))
  ENDDO
   
  DO k=1, KLEV
!CDIR ON_ADB(RH_tab)
!CDIR ON_ADB(fact_RH)
    DO i=1, KLON
      rh(i,k)=MIN(RHcl(i,k)*100.,RH_MAX)
      RH_num(i,k) = INT( rh(i,k)/10. + 1.)
      IF (rh(i,k).GT.85.) RH_num(i,k)=10
      IF (rh(i,k).GT.90.) RH_num(i,k)=11
      delta(i,k)=(rh(i,k)-RH_tab(RH_num(i,k)))*fact_RH(RH_num(i,k))
    ENDDO
  ENDDO

!CDIR SHORTLOOP  
  used_tau(:)=.FALSE.
    
  DO m=1,nb_aer   ! tau is only computed for each mass    
    fac=1.0
    IF (aerosol_name(m).EQ.id_ASBCM) THEN
        soluble=.TRUE.
        spsol=1
    ELSEIF (aerosol_name(m).EQ.id_ASPOMM) THEN 
        soluble=.TRUE.
        spsol=2 
    ELSEIF (aerosol_name(m).EQ.id_ASSO4M) THEN
        soluble=.TRUE.
        spsol=3
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
    ELSEIF (aerosol_name(m).EQ.id_CSSO4M) THEN
        soluble=.TRUE.
        spsol=4
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
    ELSEIF (aerosol_name(m).EQ.id_SSSSM) THEN 
        soluble=.TRUE.
        spsol=5
    ELSEIF (aerosol_name(m).EQ.id_CSSSM) THEN 
        soluble=.TRUE.
        spsol=6
    ELSEIF (aerosol_name(m).EQ.id_ASSSM) THEN
        soluble=.TRUE.
        spsol=7
    ELSEIF (aerosol_name(m).EQ.id_CIDUSTM) THEN 
        soluble=.FALSE.
        spinsol=1
    ELSEIF  (aerosol_name(m).EQ.id_AIBCM) THEN 
        soluble=.FALSE.
        spinsol=2
    ELSEIF (aerosol_name(m).EQ.id_AIPOMM) THEN 
        soluble=.FALSE.
        spinsol=3
    ELSE 
        CYCLE
    ENDIF

    IF (soluble) then
      used_tau(spsol)=.TRUE.
    ELSE
      used_tau(naero_soluble+spinsol)=.TRUE.
    ENDIF

    DO la=1,las

      IF (soluble) THEN            ! For soluble aerosol

          DO k=1, KLEV
            DO i=1, KLON
              tau_ae5wv_int = alpha_aers_5wv(RH_num(i,k),la,spsol)+DELTA(i,k)* &
                             (alpha_aers_5wv(RH_num(i,k)+1,la,spsol) - & 
                              alpha_aers_5wv(RH_num(i,k),la,spsol))
              tau(i,k,la,spsol) = mass_temp(i,k,spsol)*1000.*zdp1(i,k)   &
                                 *tau_ae5wv_int*delt*fac
              tausum(i,la,spsol)=tausum(i,la,spsol)+tau(i,k,la,spsol)
            ENDDO
          ENDDO

      ELSE                         ! For insoluble aerosol

        DO k=1, KLEV
          DO i=1, KLON
            tau_ae5wv_int = alpha_aeri_5wv(la,spinsol)
            tau(i,k,la,naero_soluble+spinsol) = mass_temp(i,k,naero_soluble+spinsol)*1000.*zdp1(i,k)* &
                                                tau_ae5wv_int*delt*fac
            tausum(i,la,naero_soluble+spinsol)= tausum(i,la,naero_soluble+spinsol)  &
                                               +tau(i,k,la,naero_soluble+spinsol)
          ENDDO
        ENDDO

      ENDIF

    ENDDO   ! Boucle sur les longueurs d'onde
  ENDDO     ! Boucle sur les masses de traceurs

  DO m=1,naero_tot
    IF (.NOT.used_tau(m)) tau(:,:,:,m)=0.
  ENDDO  

  DO i=1, klon
     od550aer(i)=0.
     DO m=1,naero_tot
        od550aer(i)=od550aer(i)+tausum(i,la550,m)
     END DO
  END DO

  DO i=1, klon
     od670aer(i)=0.
     DO m=1,naero_tot
        od670aer(i)=od670aer(i)+tausum(i,la670,m)
     END DO
  END DO

  DO i=1, klon
     od865aer(i)=0.
     DO m=1,naero_tot
        od865aer(i)=od865aer(i)+tausum(i,la865,m)
     END DO
  END DO

  DO i=1, klon
     DO k=1, KLEV
        ec550aer(i,k)=0.
        DO m=1,naero_tot
           ec550aer(i,k)=ec550aer(i,k)+tau(i,k,la550,m)/dh(i,k)
        END DO
     END DO
  END DO
  
  DO i=1, klon
    ai(i)=-LOG(MAX(od670aer(i),1.e-8)/MAX(od865aer(i),1.e-8))/LOG(670./865.)
  ENDDO

  od550lt1aer(:)=tausum(:,la550,id_ASSO4M)+tausum(:,la550,id_ASBCM) +tausum(:,la550,id_AIBCM)+ &
                 tausum(:,la550,id_ASPOMM)+tausum(:,la550,id_AIPOMM)+tausum(:,la550,id_ASSSM)+ &
                 0.03*tausum(:,la550,id_CSSSM)+0.4*tausum(:,la550,id_CIDUSTM)

  DEALLOCATE(aerosol_name) 
  
END SUBROUTINE AEROPT_5WV_RRTM
