!
! $Id: aeropt_6bands.F90 1716 2013-01-22 13:36:30Z acozic $
!
SUBROUTINE AEROPT_6BANDS_RRTM ( &
     pdel, m_allaer, delt, RHcl, &
     tau_allaer, piz_allaer, &
     cg_allaer, m_allaer_pi, &
     flag_aerosol, pplay, t_seri )

  USE dimphy
  USE aero_mod
  USE phys_local_var_mod, only: absvisaer

  !    Yves Balkanski le 12 avril 2006
  !    Celine Deandreis
  !    Anne Cozic Avril 2009
  !    a partir d'une sous-routine de Johannes Quaas pour les sulfates
  !    Olivier Boucher fvrier 2014 pour passage  RRTM
  !    a partir des proprits optiques fournies par Yves Balkanski
  !
  IMPLICIT NONE

  INCLUDE "YOMCST.h"
  INCLUDE "iniprint.h"
  INCLUDE "clesphys.h"

  !
  ! Input arguments:
  !
  REAL, DIMENSION(klon,klev),     INTENT(in)  :: pdel
  REAL,                           INTENT(in)  :: delt
  REAL, DIMENSION(klon,klev,naero_spc),   INTENT(in)  :: m_allaer
  REAL, DIMENSION(klon,klev,naero_spc),   INTENT(in)  :: m_allaer_pi
  REAL, DIMENSION(klon,klev),     INTENT(in)  :: RHcl       ! humidite relative ciel clair
  INTEGER,                        INTENT(in)  :: flag_aerosol
  REAL, DIMENSION(klon,klev),     INTENT(in)  :: pplay
  REAL, DIMENSION(klon,klev),     INTENT(in)  :: t_seri
  !
  ! Output arguments:
  ! 1= total aerosols 
  ! 2= natural aerosols
  !
  REAL, DIMENSION(klon,klev,2,nbands_rrtm), INTENT(out) :: tau_allaer ! epaisseur optique aerosol
  REAL, DIMENSION(klon,klev,2,nbands_rrtm), INTENT(out) :: piz_allaer ! single scattering albedo aerosol
  REAL, DIMENSION(klon,klev,2,nbands_rrtm), INTENT(out) :: cg_allaer  ! asymmetry parameter aerosol

  !
  ! Local
  !
  LOGICAL ::  soluble
  INTEGER :: i, k,n, inu, m
  INTEGER :: spsol, spinsol
  INTEGER :: RH_num(klon,klev)

  INTEGER, PARAMETER :: nb_level=19 ! number of vertical levels in DATA

  INTEGER, PARAMETER :: naero_soluble=7    ! 1- BC soluble; 2- POM soluble; 3- SO4. acc. 4- SO4 coarse
                                           ! 5- seasalt super coarse  6- seasalt coarse   7- seasalt acc.
  INTEGER, PARAMETER :: naero_insoluble=3  ! 1- Dust; 2- BC insoluble; 3- POM insoluble

  INTEGER, PARAMETER :: nbre_RH=12
  REAL,PARAMETER :: RH_tab(nbre_RH)=(/0.,10.,20.,30.,40.,50.,60.,70.,80.,85.,90.,95./)
  REAL, PARAMETER :: RH_MAX=95.
  REAL :: delta(klon,klev), rh(klon,klev)
  REAL :: tau_ae2b_int   ! Intermediate computation of epaisseur optique aerosol
  REAL :: piz_ae2b_int   ! Intermediate computation of Single scattering albedo
  REAL :: cg_ae2b_int    ! Intermediate computation of Assymetry parameter
  REAL :: Fact_RH(nbre_RH)
  REAL :: zrho
  REAL :: fac
  REAL :: zdp1(klon,klev) 
  INTEGER, ALLOCATABLE, DIMENSION(:)   :: aerosol_name
  INTEGER :: nb_aer

  REAL, DIMENSION(klon,klev,naero_spc) :: mass_temp
  REAL, DIMENSION(klon,klev,naero_spc) :: mass_temp_pi
  REAL, DIMENSION(klon,klev,naero_tot,nbands_rrtm) ::  tau_ae
  REAL, DIMENSION(klon,klev,naero_tot,nbands_rrtm) ::  tau_ae_pi
  REAL, DIMENSION(klon,klev,naero_tot,nbands_rrtm) ::  piz_ae
  REAL, DIMENSION(klon,klev,naero_tot,nbands_rrtm) ::  cg_ae


  !
  ! Proprietes optiques
  !
  REAL:: alpha_aers_6bands(nbre_RH,nbands_rrtm,naero_soluble)   !--unit m2/g SO4
  REAL:: alpha_aeri_6bands(nbands_rrtm,naero_insoluble)
  REAL:: cg_aers_6bands(nbre_RH,nbands_rrtm,naero_soluble)      !--unit 
  REAL:: cg_aeri_6bands(nbands_rrtm,naero_insoluble)
  REAL:: piz_aers_6bands(nbre_RH,nbands_rrtm,naero_soluble)     !-- unit
  REAL:: piz_aeri_6bands(nbands_rrtm,naero_insoluble)        !-- unit

  INTEGER :: id
  LOGICAL :: used_aer(naero_tot)
  REAL :: tmp_var, tmp_var_pi

!***************************************************************************
!--the order of the soluble   species has to follow the spsol   index below
!--the order of the insoluble species has to follow the spinsol index below

  DATA alpha_aers_6bands/  & 
       ! bc soluble AS
  6.497, 6.497, 6.497, 6.497, 6.497, 7.160, 7.875, 9.356,10.811,10.974,11.149,12.734, &
  6.497, 6.497, 6.497, 6.497, 6.497, 7.160, 7.875, 9.356,10.811,10.974,11.149,12.734, &
  5.900, 5.900, 5.900, 5.900, 5.900, 6.502, 7.151, 8.496, 9.818, 9.965,10.124,11.564, &
  4.284, 4.284, 4.284, 4.284, 4.284, 4.721, 5.193, 6.169, 7.129, 7.236, 7.352, 8.397, &
  2.163, 2.163, 2.163, 2.163, 2.163, 2.384, 2.622, 3.115, 3.600, 3.654, 3.712, 4.240, &
  0.966, 0.966, 0.966, 0.966, 0.966, 1.065, 1.171, 1.392, 1.608, 1.632, 1.658, 1.894, &
       ! pom soluble AS
  6.443, 6.443, 6.443, 6.443, 6.443, 7.100, 7.809, 9.277,10.721,10.882,11.056,12.628, &
  6.443, 6.443, 6.443, 6.443, 6.443, 7.100, 7.809, 9.277,10.721,10.882,11.056,12.628, &
  4.381, 4.381, 4.381, 4.381, 4.381, 4.828, 5.310, 6.309, 7.290, 7.400, 7.518, 8.587, &
  1.846, 1.846, 1.846, 1.846, 1.846, 2.034, 2.237, 2.658, 3.072, 3.118, 3.168, 3.618, &
  0.377, 0.377, 0.377, 0.377, 0.377, 0.415, 0.456, 0.542, 0.627, 0.636, 0.646, 0.738, &
  0.052, 0.052, 0.052, 0.052, 0.052, 0.057, 0.063, 0.075, 0.087, 0.088, 0.089, 0.102, &
       ! sulfate AS   
  6.554, 6.554, 6.554, 7.223, 7.931, 8.665, 9.438,10.736,14.275,17.755,17.755,31.722, &
  6.554, 6.554, 6.554, 7.223, 7.931, 8.665, 9.438,10.736,14.275,17.755,17.755,31.722, &
  4.381, 4.381, 4.381, 4.828, 5.301, 5.792, 6.309, 7.176, 9.542,11.868,11.868,21.204, &
  1.727, 1.727, 1.727, 1.903, 2.090, 2.283, 2.487, 2.829, 3.762, 4.679, 4.679, 8.359, &
  0.312, 0.312, 0.312, 0.344, 0.378, 0.413, 0.450, 0.511, 0.680, 0.846, 0.846, 1.511, &
  0.121, 0.121, 0.121, 0.134, 0.147, 0.161, 0.175, 0.199, 0.264, 0.329, 0.329, 0.588, &
       ! sulfate coarse CS
  0.693, 0.693, 0.693, 0.764, 0.839, 0.917, 0.999, 1.136, 1.510, 1.879, 1.879, 3.356, &
  0.693, 0.693, 0.693, 0.764, 0.839, 0.917, 0.999, 1.136, 1.510, 1.879, 1.879, 3.356, &
  0.715, 0.715, 0.715, 0.788, 0.865, 0.945, 1.029, 1.171, 1.557, 1.936, 1.936, 3.459, &
  0.736, 0.736, 0.736, 0.811, 0.891, 0.973, 1.060, 1.206, 1.603, 1.994, 1.994, 3.563, &
  0.711, 0.711, 0.711, 0.783, 0.860, 0.939, 1.023, 1.164, 1.548, 1.925, 1.925, 3.439, &
  0.602, 0.602, 0.602, 0.664, 0.729, 0.796, 0.867, 0.986, 1.312, 1.631, 1.631, 2.915, &
       ! seasalt seasalt Super Coarse Soluble (SS)
  0.214, 0.235, 0.257, 0.305, 0.385, 0.444, 0.537, 0.606, 0.876, 1.006, 1.243, 2.313, &
  0.217, 0.238, 0.261, 0.309, 0.389, 0.448, 0.543, 0.612, 0.882, 1.015, 1.255, 2.327, &
  0.221, 0.243, 0.266, 0.315, 0.396, 0.456, 0.552, 0.621, 0.895, 1.028, 1.269, 2.351, &
  0.230, 0.253, 0.276, 0.326, 0.408, 0.469, 0.568, 0.638, 0.918, 1.053, 1.298, 2.389, &
  0.250, 0.273, 0.298, 0.349, 0.435, 0.501, 0.601, 0.674, 0.960, 1.098, 1.349, 2.466, &
  0.279, 0.307, 0.336, 0.396, 0.493, 0.566, 0.679, 0.760, 1.072, 1.221, 1.486, 2.652, &
       ! seasalt seasalt Coarse Soluble (CS)      
  0.550, 0.605, 0.661, 0.782, 0.980, 1.131, 1.362, 1.536, 2.204, 2.528, 3.115, 5.773, &
  0.568, 0.622, 0.681, 0.802, 1.004, 1.155, 1.393, 1.568, 2.246, 2.574, 3.174, 5.846, &
  0.599, 0.655, 0.714, 0.838, 1.044, 1.202, 1.446, 1.621, 2.312, 2.645, 3.249, 5.950, &
  0.647, 0.708, 0.771, 0.904, 1.121, 1.290, 1.546, 1.731, 2.447, 2.791, 3.416, 6.166, &
  0.663, 0.735, 0.809, 0.964, 1.215, 1.386, 1.677, 1.883, 2.681, 3.059, 3.733, 6.647, &
  0.479, 0.549, 0.623, 0.787, 1.065, 1.174, 1.491, 1.717, 2.648, 3.094, 3.901, 7.382, &
       ! seasalt seasalt Accumulation Soluble (AS)
  6.128, 6.867, 7.624, 9.184,11.602,13.194,15.955,17.858,24.856,28.034,33.562,56.971, &
  5.080, 5.837, 6.636, 8.347,11.144,12.321,15.463,17.649,26.080,29.896,36.491,62.414, &
  3.290, 3.899, 4.567, 6.075, 8.744, 9.355,12.369,14.532,23.747,28.159,36.101,68.531, &
  1.389, 1.706, 2.070, 2.942, 4.643, 4.846, 6.836, 8.334,15.546,19.346,26.738,62.978, &
  0.309, 0.394, 0.495, 0.754, 1.312, 1.377, 2.087, 2.654, 5.732, 7.538,11.368,35.212, &
  0.021, 0.028, 0.036, 0.059, 0.114, 0.126, 0.208, 0.279, 0.718, 1.009, 1.691, 7.271  /

  DATA alpha_aeri_6bands/  & 
       ! dust insoluble CI
  0.751, 0.751, 0.769, 0.772, 0.672, 0.437, &
       ! bc insoluble AI
  6.497, 6.497, 5.900, 4.284, 2.163, 0.966, &
       ! pom insoluble AI
  6.443, 6.443, 4.381, 1.846, 0.377, 0.052 /

  DATA cg_aers_6bands/ &
       ! bc soluble AS
  0.721, 0.721, 0.721, 0.729, 0.735, 0.741, 0.746, 0.754, 0.762, 0.766, 0.769, 0.775, &
  0.721, 0.721, 0.721, 0.729, 0.735, 0.741, 0.746, 0.754, 0.762, 0.766, 0.769, 0.775, &
  0.643, 0.643, 0.643, 0.654, 0.662, 0.670, 0.677, 0.688, 0.698, 0.704, 0.707, 0.715, &
  0.513, 0.513, 0.513, 0.522, 0.530, 0.536, 0.542, 0.552, 0.560, 0.565, 0.568, 0.575, &
  0.321, 0.321, 0.321, 0.323, 0.325, 0.327, 0.328, 0.331, 0.333, 0.334, 0.335, 0.337, &
  0.153, 0.153, 0.153, 0.149, 0.145, 0.142, 0.139, 0.135, 0.130, 0.128, 0.127, 0.123, &
       ! pom soluble AS
  0.687, 0.687, 0.687, 0.687, 0.687, 0.700, 0.710, 0.726, 0.736, 0.737, 0.738, 0.745, &
  0.687, 0.687, 0.687, 0.687, 0.687, 0.700, 0.710, 0.726, 0.736, 0.737, 0.738, 0.745, &
  0.658, 0.658, 0.658, 0.658, 0.658, 0.667, 0.674, 0.685, 0.692, 0.692, 0.693, 0.698, &
  0.564, 0.564, 0.564, 0.564, 0.564, 0.566, 0.568, 0.571, 0.573, 0.573, 0.573, 0.574, &
  0.363, 0.363, 0.363, 0.363, 0.363, 0.360, 0.357, 0.352, 0.350, 0.349, 0.349, 0.347, &
  0.142, 0.142, 0.142, 0.142, 0.142, 0.139, 0.137, 0.133, 0.131, 0.131, 0.130, 0.129, &
       ! sulfate AS 
  0.675, 0.675, 0.675, 0.689, 0.701, 0.711, 0.720, 0.735, 0.748, 0.756, 0.760, 0.771, &
  0.675, 0.675, 0.675, 0.689, 0.701, 0.711, 0.720, 0.735, 0.748, 0.756, 0.760, 0.771, &
  0.653, 0.653, 0.653, 0.662, 0.670, 0.676, 0.683, 0.692, 0.701, 0.706, 0.709, 0.716, &
  0.563, 0.563, 0.563, 0.565, 0.567, 0.569, 0.570, 0.573, 0.575, 0.576, 0.577, 0.579, &
  0.362, 0.362, 0.362, 0.359, 0.356, 0.354, 0.352, 0.348, 0.345, 0.343, 0.342, 0.340, &
  0.137, 0.137, 0.137, 0.135, 0.133, 0.132, 0.130, 0.128, 0.126, 0.125, 0.124, 0.122, &
       ! sulfate coarse CS
  0.803, 0.803, 0.803, 0.792, 0.783, 0.776, 0.769, 0.758, 0.747, 0.742, 0.738, 0.730, &
  0.803, 0.803, 0.803, 0.792, 0.783, 0.776, 0.769, 0.758, 0.747, 0.742, 0.738, 0.730, &
  0.799, 0.799, 0.799, 0.787, 0.777, 0.768, 0.760, 0.747, 0.736, 0.729, 0.725, 0.716, &
  0.797, 0.797, 0.797, 0.782, 0.770, 0.760, 0.750, 0.735, 0.722, 0.714, 0.709, 0.698, &
  0.810, 0.810, 0.810, 0.794, 0.781, 0.770, 0.759, 0.743, 0.728, 0.719, 0.714, 0.702, &
  0.803, 0.803, 0.803, 0.790, 0.779, 0.770, 0.762, 0.748, 0.736, 0.729, 0.725, 0.715, &
       ! seasalt seasalt Super Coarse Soluble (SS)
  0.797, 0.799, 0.800, 0.803, 0.805, 0.823, 0.827, 0.830, 0.836, 0.838, 0.840, 0.845, &
  0.788, 0.790, 0.792, 0.795, 0.798, 0.817, 0.821, 0.824, 0.830, 0.832, 0.835, 0.841, &
  0.773, 0.776, 0.777, 0.782, 0.786, 0.805, 0.811, 0.814, 0.821, 0.824, 0.829, 0.835, &
  0.746, 0.748, 0.752, 0.757, 0.764, 0.786, 0.792, 0.797, 0.806, 0.810, 0.815, 0.826, &
  0.706, 0.710, 0.713, 0.718, 0.726, 0.752, 0.760, 0.766, 0.778, 0.783, 0.790, 0.806, &
  0.681, 0.681, 0.681, 0.683, 0.685, 0.723, 0.728, 0.733, 0.740, 0.744, 0.751, 0.767, &
       ! seasalt seasalt Coarse Soluble (CS)       
  0.756, 0.759, 0.761, 0.766, 0.772, 0.792, 0.799, 0.802, 0.813, 0.816, 0.820, 0.828, &
  0.736, 0.740, 0.742, 0.749, 0.756, 0.778, 0.785, 0.789, 0.800, 0.804, 0.810, 0.821, &
  0.712, 0.714, 0.717, 0.724, 0.732, 0.757, 0.765, 0.770, 0.783, 0.788, 0.794, 0.809, &
  0.690, 0.691, 0.693, 0.697, 0.703, 0.734, 0.742, 0.746, 0.758, 0.763, 0.769, 0.788, &
  0.682, 0.682, 0.682, 0.683, 0.684, 0.725, 0.729, 0.733, 0.739, 0.742, 0.746, 0.759, &
  0.669, 0.671, 0.672, 0.675, 0.678, 0.721, 0.727, 0.732, 0.737, 0.740, 0.743, 0.745, &
       ! seasalt seasalt Accumulation Soluble (AS)
  0.694, 0.694, 0.693, 0.690, 0.685, 0.733, 0.733, 0.735, 0.729, 0.728, 0.727, 0.732, &
  0.685, 0.688, 0.690, 0.692, 0.693, 0.738, 0.742, 0.746, 0.745, 0.744, 0.742, 0.731, &
  0.650, 0.656, 0.662, 0.672, 0.681, 0.722, 0.733, 0.740, 0.750, 0.753, 0.755, 0.748, &
  0.561, 0.573, 0.584, 0.603, 0.626, 0.661, 0.681, 0.693, 0.721, 0.730, 0.741, 0.758, &
  0.392, 0.408, 0.424, 0.453, 0.490, 0.517, 0.548, 0.567, 0.619, 0.637, 0.662, 0.716, &
  0.144, 0.155, 0.167, 0.190, 0.223, 0.237, 0.268, 0.287, 0.353, 0.379, 0.418, 0.530  /

  DATA cg_aeri_6bands/ &
       ! dust insoluble CI
  0.718, 0.718, 0.699, 0.661, 0.676, 0.670, &
       ! bc insoluble AI
  0.721, 0.721, 0.643, 0.513, 0.321, 0.153, &
       ! pom insoluble AI
  0.687, 0.687, 0.658, 0.564, 0.363, 0.142  /

  DATA piz_aers_6bands/&
       ! bc soluble AS
  0.460, 0.460, 0.460, 0.460, 0.460, 0.534, 0.594, 0.688, 0.748, 0.754, 0.760, 0.803, &
  0.460, 0.460, 0.460, 0.460, 0.460, 0.534, 0.594, 0.688, 0.748, 0.754, 0.760, 0.803, &
  0.445, 0.445, 0.445, 0.445, 0.445, 0.521, 0.583, 0.679, 0.741, 0.747, 0.753, 0.798, &
  0.394, 0.394, 0.394, 0.394, 0.394, 0.477, 0.545, 0.649, 0.718, 0.724, 0.730, 0.779, &
  0.267, 0.267, 0.267, 0.267, 0.267, 0.365, 0.446, 0.571, 0.652, 0.660, 0.667, 0.725, &
  0.121, 0.121, 0.121, 0.121, 0.121, 0.139, 0.155, 0.178, 0.193, 0.195, 0.196, 0.207, &
       ! pom soluble AS
  0.973, 0.973, 0.973, 0.973, 0.973, 0.977, 0.980, 0.984, 0.987, 0.988, 0.988, 0.990, &
  0.973, 0.973, 0.973, 0.973, 0.973, 0.977, 0.980, 0.984, 0.987, 0.988, 0.988, 0.990, &
  0.972, 0.972, 0.972, 0.972, 0.972, 0.976, 0.979, 0.984, 0.987, 0.987, 0.988, 0.990, &
  0.940, 0.940, 0.940, 0.940, 0.940, 0.948, 0.955, 0.965, 0.972, 0.973, 0.973, 0.978, &
  0.816, 0.816, 0.816, 0.816, 0.816, 0.839, 0.859, 0.888, 0.908, 0.910, 0.911, 0.925, &
  0.663, 0.663, 0.663, 0.663, 0.663, 0.607, 0.562, 0.492, 0.446, 0.441, 0.437, 0.404, &
       ! sulfate AS
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.988, 0.988, 0.988, 0.989, 0.990, 0.990, 0.991, 0.992, 0.993, 0.993, 0.994, 0.994, &
  0.256, 0.256, 0.256, 0.263, 0.268, 0.273, 0.277, 0.284, 0.290, 0.294, 0.296, 0.301, &
       ! sulfate coarse CS
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, 0.996, &
  0.877, 0.877, 0.877, 0.873, 0.870, 0.867, 0.864, 0.860, 0.856, 0.854, 0.852, 0.849, &
       ! seasalt seasalt Super Coarse Soluble (SS)
  0.983, 0.982, 0.982, 0.982, 0.981, 0.992, 0.993, 0.994, 0.995, 0.996, 0.997, 0.998, &
  0.984, 0.983, 0.983, 0.983, 0.982, 0.992, 0.993, 0.995, 0.996, 0.996, 0.997, 0.998, &
  0.986, 0.985, 0.985, 0.985, 0.984, 0.993, 0.994, 0.995, 0.996, 0.997, 0.997, 0.998, &
  0.989, 0.989, 0.988, 0.988, 0.987, 0.995, 0.995, 0.996, 0.997, 0.997, 0.998, 0.999, &
  0.994, 0.993, 0.993, 0.992, 0.992, 0.997, 0.997, 0.997, 0.998, 0.998, 0.998, 0.999, &
  0.997, 0.997, 0.997, 0.997, 0.996, 0.999, 0.999, 0.999, 0.999, 0.999, 0.999, 0.999, &
       ! seasalt seasalt Coarse Soluble (CS)      
  0.988, 0.988, 0.988, 0.987, 0.986, 0.994, 0.995, 0.996, 0.996, 0.997, 0.998, 0.998, &
  0.990, 0.990, 0.990, 0.989, 0.988, 0.995, 0.996, 0.996, 0.997, 0.997, 0.998, 0.999, &
  0.993, 0.993, 0.992, 0.992, 0.991, 0.996, 0.997, 0.997, 0.998, 0.998, 0.998, 0.999, &
  0.996, 0.995, 0.995, 0.994, 0.994, 0.998, 0.998, 0.998, 0.998, 0.999, 0.999, 0.999, &
  0.998, 0.997, 0.997, 0.997, 0.997, 0.999, 0.999, 0.999, 0.999, 0.999, 0.999, 0.999, &
  0.999, 0.999, 0.998, 0.998, 0.998, 0.999, 0.999, 1.000, 1.000, 1.000, 1.000, 1.000, &
       ! seasalt seasalt Accumulation Soluble (AS)
  0.999, 0.999, 0.998, 0.998, 0.998, 0.999, 0.999, 0.999, 0.999, 1.000, 1.000, 1.000, &
  0.999, 0.999, 0.999, 0.999, 0.999, 0.999, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.999, 0.999, 0.999, 0.999, 0.999, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.999, 0.999, 0.999, 0.999, 0.999, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.997, 0.997, 0.998, 0.998, 0.998, 0.999, 0.999, 1.000, 1.000, 1.000, 1.000, 1.000, &
  0.985, 0.987, 0.988, 0.991, 0.993, 0.997, 0.998, 0.998, 0.999, 0.999, 1.000, 1.000  /

  DATA piz_aeri_6bands/ &
       ! dust insoluble CI
  0.954, 0.954, 0.965, 0.981, 0.996, 0.990, &
       ! bc insoluble AI
  0.460, 0.460, 0.445, 0.394, 0.267, 0.121, &
       ! pom insoluble AI
  0.973, 0.973, 0.972, 0.940, 0.816, 0.663 /

!----BEGINNING OF CALCULATIONS

  spsol = 0
  spinsol = 0 
  if(NSW.NE.nbands_rrtm) then
     print *,'Erreur NSW doit etre egal a 6 pour cette routine'
     stop
  endif

  DO k=1, klev
    DO i=1, klon
      zrho=pplay(i,k)/t_seri(i,k)/RD                  ! kg/m3
!CDIR UNROLL=naero_spc
      mass_temp(i,k,:) = m_allaer(i,k,:) / zrho / 1.e+9
!CDIR UNROLL=naero_spc
      mass_temp_pi(i,k,:) = m_allaer_pi(i,k,:) / zrho / 1.e+9
      zdp1(i,k)=pdel(i,k)/(RG*delt)      ! air mass auxiliary  variable --> zdp1 [kg/(m^2 *s)]
    ENDDO
  ENDDO

  IF (flag_aerosol .EQ. 1) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M
     aerosol_name(2) = id_CSSO4M
  ELSEIF (flag_aerosol .EQ. 2) THEN
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASBCM
     aerosol_name(2) = id_AIBCM
  ELSEIF (flag_aerosol .EQ. 3) THEN 
     nb_aer = 2
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASPOMM
     aerosol_name(2) = id_AIPOMM
  ELSEIF (flag_aerosol .EQ. 4) THEN 
     nb_aer = 3
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CSSSM
     aerosol_name(2) = id_SSSSM
     aerosol_name(3) = id_ASSSM
  ELSEIF (flag_aerosol .EQ. 5) THEN 
     nb_aer = 1
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_CIDUSTM
  ELSEIF (flag_aerosol .EQ. 6) THEN 
     nb_aer = 10
     ALLOCATE (aerosol_name(nb_aer)) 
     aerosol_name(1) = id_ASSO4M      
     aerosol_name(2) = id_ASBCM
     aerosol_name(3) = id_AIBCM
     aerosol_name(4) = id_ASPOMM
     aerosol_name(5) = id_AIPOMM
     aerosol_name(6) = id_CSSSM
     aerosol_name(7) = id_SSSSM
     aerosol_name(8) = id_ASSSM
     aerosol_name(9) = id_CIDUSTM
     aerosol_name(10)= id_CSSO4M
  ENDIF

  !
  ! loop over modes, use of precalculated nmd and corresponding sigma
  !    loop over wavelengths
  !    for each mass species in mode
  !      interpolate from Sext to retrieve Sext_at_gridpoint_per_species
  !      compute optical_thickness_at_gridpoint_per_species

!!CDIR ON_ADB(RH_tab)
!CDIR ON_ADB(fact_RH)
!CDIR SHORTLOOP
  DO n=1,nbre_RH-1
    fact_RH(n)=1./(RH_tab(n+1)-RH_tab(n))
  ENDDO
   
  DO k=1, KLEV
!CDIR ON_ADB(fact_RH)
    DO i=1, KLON
      rh(i,k)=MIN(RHcl(i,k)*100.,RH_MAX)
      RH_num(i,k) = INT( rh(i,k)/10. + 1.)
      IF (rh(i,k).GT.85.) RH_num(i,k)=10
      IF (rh(i,k).GT.90.) RH_num(i,k)=11
      delta(i,k)=(rh(i,k)-RH_tab(RH_num(i,k)))*fact_RH(RH_num(i,k))
    ENDDO
  ENDDO

  used_aer(:)=.FALSE.
    
  DO m=1,nb_aer   ! tau is only computed for each mass
     fac=1.0
     IF (aerosol_name(m).EQ.id_ASBCM) THEN
        soluble=.TRUE.
        spsol=1
     ELSEIF (aerosol_name(m).EQ.id_ASPOMM) THEN 
        soluble=.TRUE.
        spsol=2 
     ELSEIF (aerosol_name(m).EQ.id_ASSO4M) THEN 
        soluble=.TRUE.
        spsol=3
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
     ELSEIF  (aerosol_name(m).EQ.id_CSSO4M) THEN
        soluble=.TRUE.
        spsol=4
        fac=1.375    ! (NH4)2-SO4/SO4 132/96 mass conversion factor for OD
     ELSEIF (aerosol_name(m).EQ.id_SSSSM) THEN 
         soluble=.TRUE.
         spsol=5
     ELSEIF (aerosol_name(m).EQ.id_CSSSM) THEN 
         soluble=.TRUE.
         spsol=6
     ELSEIF (aerosol_name(m).EQ.id_ASSSM) THEN
         soluble=.TRUE.
         spsol=7
     ELSEIF (aerosol_name(m).EQ.id_CIDUSTM) THEN 
         soluble=.FALSE.
         spinsol=1
     ELSEIF  (aerosol_name(m).EQ.id_AIBCM) THEN 
         soluble=.FALSE.
         spinsol=2
     ELSEIF (aerosol_name(m).EQ.id_AIPOMM) THEN 
         soluble=.FALSE.
         spinsol=3
     ELSE 
         CYCLE
     ENDIF

    id=aerosol_name(m)
    used_aer(id)=.TRUE.

    IF (soluble) THEN

       DO k=1, KLEV
         DO i=1, KLON
           tmp_var=mass_temp(i,k,spsol)*1000.*zdp1(i,k)*delt*fac
           tmp_var_pi=mass_temp_pi(i,k,spsol)*1000.*zdp1(i,k)*delt*fac

           DO inu=1,NSW

             tau_ae2b_int= alpha_aers_6bands(RH_num(i,k),inu,spsol)+ & 
                           delta(i,k)* (alpha_aers_6bands(RH_num(i,k)+1,inu,spsol) - & 
                           alpha_aers_6bands(RH_num(i,k),inu,spsol))
                   
             piz_ae2b_int = piz_aers_6bands(RH_num(i,k),inu,spsol) + & 
                            delta(i,k)* (piz_aers_6bands(RH_num(i,k)+1,inu,spsol) - & 
                            piz_aers_6bands(RH_num(i,k),inu,spsol))
                   
             cg_ae2b_int = cg_aers_6bands(RH_num(i,k),inu,spsol) + & 
                           delta(i,k)* (cg_aers_6bands(RH_num(i,k)+1,inu,spsol) - & 
                           cg_aers_6bands(RH_num(i,k),inu,spsol))

             tau_ae(i,k,id,inu)    = tmp_var*tau_ae2b_int
             tau_ae_pi(i,k,id,inu) = tmp_var_pi* tau_ae2b_int
             piz_ae(i,k,id,inu)    = piz_ae2b_int
             cg_ae(i,k,id,inu)     = cg_ae2b_int
                      
           ENDDO
         ENDDO
       ENDDO
        
     ELSE    ! For all aerosol insoluble components

       DO k=1, KLEV
         DO i=1, KLON
           tmp_var=mass_temp(i,k,naero_soluble+ spinsol)*1000.*zdp1(i,k)*delt*fac
           tmp_var_pi=mass_temp_pi(i,k,naero_soluble+spinsol)*1000.*zdp1(i,k)*delt*fac

           DO inu=1,NSW
             tau_ae2b_int = alpha_aeri_6bands(inu,spinsol)
             piz_ae2b_int = piz_aeri_6bands(inu,spinsol)
             cg_ae2b_int = cg_aeri_6bands(inu,spinsol) 

             tau_ae(i,k,id,inu) = tmp_var*tau_ae2b_int
             tau_ae_pi(i,k,id,inu) = tmp_var_pi*tau_ae2b_int
             piz_ae(i,k,id,inu) = piz_ae2b_int
             cg_ae(i,k,id,inu)= cg_ae2b_int
           ENDDO
         ENDDO
       ENDDO

     ENDIF ! soluble / insoluble

  ENDDO  ! nb_aer  

  DO m=1,naero_tot
    IF (.NOT. used_aer(m)) THEN
      tau_ae(:,:,m,:)=0.
      tau_ae_pi(:,:,m,:)=0.
      piz_ae(:,:,m,:)=0.
      cg_ae(:,:,m,:)=0.
    ENDIF
  ENDDO

  DO inu=1, NSW 
     DO k=1, KLEV
       DO i=1, KLON
!--anthropogenic aerosol
         tau_allaer(i,k,2,inu)=tau_ae(i,k,id_ASSO4M,inu)+tau_ae(i,k,id_CSSO4M,inu)+ &
                               tau_ae(i,k,id_ASBCM,inu)+tau_ae(i,k,id_AIBCM,inu)+   &
                               tau_ae(i,k,id_ASPOMM,inu)+tau_ae(i,k,id_AIPOMM,inu)+ &
                               tau_ae(i,k,id_ASSSM,inu)+tau_ae(i,k,id_CSSSM,inu)+   &
                               tau_ae(i,k,id_SSSSM,inu)+ tau_ae(i,k,id_CIDUSTM,inu)
         tau_allaer(i,k,2,inu)=MAX(tau_allaer(i,k,2,inu),1e-5)

         piz_allaer(i,k,2,inu)=(tau_ae(i,k,id_ASSO4M,inu)*piz_ae(i,k,id_ASSO4M,inu)+ &
                                tau_ae(i,k,id_CSSO4M,inu)*piz_ae(i,k,id_CSSO4M,inu)+ &
                                tau_ae(i,k,id_ASBCM,inu)*piz_ae(i,k,id_ASBCM,inu)+ &
                                tau_ae(i,k,id_AIBCM,inu)*piz_ae(i,k,id_AIBCM,inu)+ &
                                tau_ae(i,k,id_ASPOMM,inu)*piz_ae(i,k,id_ASPOMM,inu)+ &
                                tau_ae(i,k,id_AIPOMM,inu)*piz_ae(i,k,id_AIPOMM,inu)+ &
                                tau_ae(i,k,id_ASSSM,inu)*piz_ae(i,k,id_ASSSM,inu)+ &
                                tau_ae(i,k,id_CSSSM,inu)*piz_ae(i,k,id_CSSSM,inu)+ &
                                tau_ae(i,k,id_SSSSM,inu)*piz_ae(i,k,id_SSSSM,inu)+ &
                                tau_ae(i,k,id_CIDUSTM,inu)*piz_ae(i,k,id_CIDUSTM,inu)) &
                                /tau_allaer(i,k,2,inu)
         piz_allaer(i,k,2,inu)=MAX(piz_allaer(i,k,2,inu),0.1)

         cg_allaer(i,k,2,inu)=(tau_ae(i,k,id_ASSO4M,inu)*piz_ae(i,k,id_ASSO4M,inu)*cg_ae(i,k,id_ASSO4M,inu)+ &
                               tau_ae(i,k,id_CSSO4M,inu)*piz_ae(i,k,id_CSSO4M,inu)*cg_ae(i,k,id_CSSO4M,inu)+ &
                               tau_ae(i,k,id_ASBCM,inu)*piz_ae(i,k,id_ASBCM,inu)*cg_ae(i,k,id_ASBCM,inu)+ &
                               tau_ae(i,k,id_AIBCM,inu)*piz_ae(i,k,id_AIBCM,inu)*cg_ae(i,k,id_AIBCM,inu)+ &
                               tau_ae(i,k,id_ASPOMM,inu)*piz_ae(i,k,id_ASPOMM,inu)*cg_ae(i,k,id_ASPOMM,inu)+ &
                               tau_ae(i,k,id_AIPOMM,inu)*piz_ae(i,k,id_AIPOMM,inu)*cg_ae(i,k,id_AIPOMM,inu)+ &
                               tau_ae(i,k,id_ASSSM,inu)*piz_ae(i,k,id_ASSSM,inu)*cg_ae(i,k,id_ASSSM,inu)+ &
                               tau_ae(i,k,id_CSSSM,inu)*piz_ae(i,k,id_CSSSM,inu)*cg_ae(i,k,id_CSSSM,inu)+ &
                               tau_ae(i,k,id_SSSSM,inu)*piz_ae(i,k,id_SSSSM,inu)*cg_ae(i,k,id_SSSSM,inu)+ &
                               tau_ae(i,k,id_CIDUSTM,inu)*piz_ae(i,k,id_CIDUSTM,inu)*cg_ae(i,k,id_CIDUSTM,inu))/ &
                               (tau_allaer(i,k,2,inu)*piz_allaer(i,k,2,inu))

!--natural aerosol
         tau_allaer(i,k,1,inu)=tau_ae_pi(i,k,id_ASSO4M,inu)+tau_ae_pi(i,k,id_CSSO4M,inu)+ &
                               tau_ae_pi(i,k,id_ASBCM,inu)+tau_ae_pi(i,k,id_AIBCM,inu)+   &
                               tau_ae_pi(i,k,id_ASPOMM,inu)+tau_ae_pi(i,k,id_AIPOMM,inu)+ &
                               tau_ae_pi(i,k,id_ASSSM,inu)+tau_ae_pi(i,k,id_CSSSM,inu)+   &
                               tau_ae_pi(i,k,id_SSSSM,inu)+ tau_ae_pi(i,k,id_CIDUSTM,inu)
         tau_allaer(i,k,1,inu)=MAX(tau_allaer(i,k,1,inu),1e-5)

         piz_allaer(i,k,1,inu)=(tau_ae_pi(i,k,id_ASSO4M,inu)*piz_ae(i,k,id_ASSO4M,inu)+ &
                                tau_ae_pi(i,k,id_CSSO4M,inu)*piz_ae(i,k,id_CSSO4M,inu)+ &
                                tau_ae_pi(i,k,id_ASBCM,inu)*piz_ae(i,k,id_ASBCM,inu)+ &
                                tau_ae_pi(i,k,id_AIBCM,inu)*piz_ae(i,k,id_AIBCM,inu)+ &
                                tau_ae_pi(i,k,id_ASPOMM,inu)*piz_ae(i,k,id_ASPOMM,inu)+ &
                                tau_ae_pi(i,k,id_AIPOMM,inu)*piz_ae(i,k,id_AIPOMM,inu)+ &
                                tau_ae_pi(i,k,id_ASSSM,inu)*piz_ae(i,k,id_ASSSM,inu)+ &
                                tau_ae_pi(i,k,id_CSSSM,inu)*piz_ae(i,k,id_CSSSM,inu)+ &
                                tau_ae_pi(i,k,id_SSSSM,inu)*piz_ae(i,k,id_SSSSM,inu)+ &
                                tau_ae_pi(i,k,id_CIDUSTM,inu)*piz_ae(i,k,id_CIDUSTM,inu)) &
                                /tau_allaer(i,k,1,inu)
         piz_allaer(i,k,1,inu)=MAX(piz_allaer(i,k,1,inu),0.1)

         cg_allaer(i,k,1,inu)=(tau_ae_pi(i,k,id_ASSO4M,inu)*piz_ae(i,k,id_ASSO4M,inu)*cg_ae(i,k,id_ASSO4M,inu)+ &
                               tau_ae_pi(i,k,id_CSSO4M,inu)*piz_ae(i,k,id_CSSO4M,inu)*cg_ae(i,k,id_CSSO4M,inu)+ &
                               tau_ae_pi(i,k,id_ASBCM,inu)*piz_ae(i,k,id_ASBCM,inu)*cg_ae(i,k,id_ASBCM,inu)+ &
                               tau_ae_pi(i,k,id_AIBCM,inu)*piz_ae(i,k,id_AIBCM,inu)*cg_ae(i,k,id_AIBCM,inu)+ &
                               tau_ae_pi(i,k,id_ASPOMM,inu)*piz_ae(i,k,id_ASPOMM,inu)*cg_ae(i,k,id_ASPOMM,inu)+ &
                               tau_ae_pi(i,k,id_AIPOMM,inu)*piz_ae(i,k,id_AIPOMM,inu)*cg_ae(i,k,id_AIPOMM,inu)+ &
                               tau_ae_pi(i,k,id_ASSSM,inu)*piz_ae(i,k,id_ASSSM,inu)*cg_ae(i,k,id_ASSSM,inu)+ &
                               tau_ae_pi(i,k,id_CSSSM,inu)*piz_ae(i,k,id_CSSSM,inu)*cg_ae(i,k,id_CSSSM,inu)+ &
                               tau_ae_pi(i,k,id_SSSSM,inu)*piz_ae(i,k,id_SSSSM,inu)*cg_ae(i,k,id_SSSSM,inu)+ &
                               tau_ae_pi(i,k,id_CIDUSTM,inu)*piz_ae(i,k,id_CIDUSTM,inu)*cg_ae(i,k,id_CIDUSTM,inu))/ &
                               (tau_allaer(i,k,1,inu)*piz_allaer(i,k,1,inu))

        ENDDO
      ENDDO
    ENDDO
   
!--???????
  inu=1
  DO i=1, KLON
     absvisaer(i)=SUM((1-piz_allaer(i,:,:,inu))*tau_allaer(i,:,:,inu))
  END DO	

  DEALLOCATE(aerosol_name) 

END SUBROUTINE AEROPT_6BANDS_RRTM
