#!/bin/bash

###########################################################################
# Author : Frdric Hourdin/LMD/hourdin@lmd.jussieu.fr
# Usage  : install.sh
#
# bash installation script of the LMDZ model on a Linux PC.
# the model is downloaded in the following direcory tree
# $MODEL/modipsl/modeles/...
# using the "modipsl" infrastructure created by the "IPSL"
# for coupled (atmosphere/ocean/vegetation/chemistry) climate modeling
# activities.
# Here we only download atmospheric (LMDZ) and vegetation (ORCHIDEE)
# components.
#
# The sources of the models can be found in the "modeles" directory.
# In the present case, LMDZ5, ORCHIDEE and IOIPSL (handling of input-outputs
# using the NetCDF library.
#
# The script downloads various source files (including a version of NetCDF)
# and utilities, compiles the model, and runs a test simulation in a
# munimal configuration.
#
# Prerequisites : g95/pgf90/gfortran, ksh, wget , gunzip, tar, ... 
#
# Modif 18/11/2011
#    changes for option real 8.
#      We comopile with -r8 (or equivalent) and -DNC_DOUBLE for the GCM
#      but with -r4 for netcdf. Variable real must be set to 
#      r4 or r8 at the beginning of the script below.
#
###########################################################################

echo '################################################################'
echo  Choice of installation options 
echo '################################################################'


#real=r4
real=r8

# WARNING !!!! For versions before october 2009, use
# install.v2.sh instead of install.sh

version=20130716.trunk
version=testing

#Chemin pour placer le modele
MODEL=./LMDZ$version

getlmdzor=1
netcdf=1   #  1 for automatic installation
           #  0 for no installation
           #  /.../../netcdf-4.0.1 if wanting to link with an already
           #  compiled netcdf library (implies to check option compatibility)
check_linux=1
ioipsl=1
veget=1
bench=1
pclinux=1
compilo=gfortran # compilo=pgf90 or g95 or gfortran or ifort sur PC linux


#####################################################################
# Test for old gfortran compilers
if [ $compilo = gfortran ] ; then
   gfortranv=`gfortran --version | \
   head -1 | awk ' { print $NF } ' | awk -F. ' { print $1 * 10 + $2 } '`
   if [ $gfortranv -le 43 ] ; then
       echo ERROR : Your gfortran compiler is too old
       echo 'Please choose a new one (g95, ifort) and change the line'
       echo compilo=xxx
       echo in the install.sh script and rerun it
       exit
   fi
fi
#####################################################################



## compile_with_fcm=1 : use makelmdz_fcm, possible a of version 20111103.trunk (LMDZ5/trunk rev 1578)
## compile_with_fcm=0 : use makelmdz
compile_with_fcm=0

OPTPREC=""
echo '################################################################'
echo  Choix des options de compilation
echo '################################################################'

if [ $compilo = g95 ] ; then
   if [ $real = r8 ] ; then OPTPREC="-r8 -DNC_DOUBLE" ; fi
   OPTIM='-i4 -O3'
elif [ $compilo = gfortran ] ; then
   if [ $real = r8 ] ; then OPTPREC="-fdefault-real-8 -DNC_DOUBLE" ; fi
   OPTIM='-O3'
elif [ $compilo = pgf90 ] ; then
   if [ $real = r8 ] ; then OPTPREC="-r8 -DNC_DOUBLE" ; fi
   OPTIM='-O2 -Munroll -Mnoframe -Mautoinline -Mcache_align'
   # with pgf90, compile with fcm
   compile_with_fcm=1
else 
   # ifort
   if [ $real = r8 ] ; then OPTPREC="-real-size 64 -DNC_DOUBLE" ; fi
   OPTIM="-O2 -fp-model strict -ip -align all "
   # with ifort, compile with fcm
   compile_with_fcm=1
fi
OPTIMGCM="$OPTIM $OPTPREC"

# choose the resolution for the bench runs
# grid_resolution= 32x24x11 or 48x36x19 for tests (test without ORCHIDEE)
#                  96x71x19  standard configuration
grid_resolution=48x36x19

hostname=`hostname`

##########################################################################
# If installing on know machines such as IBM x3750 (Ada)
# at IDRIS, don't check for available software and don"t install netcdf
if [ ${hostname:0:5} = ada33 ] ; then
  netcdf=0 # no need to recompile netcdf, alreday available
  check_linux=0
  pclinux=0
  ioipsl=0 # no need to recompile ioipsl, already available
  #netcdf="/smplocal/pub/NetCDF/4.1.3"
  compilo="ifort"
  fmod='module '
  if [ $real = r8 ] ; then OPTPREC="-real-size 64 -DNC_DOUBLE" ; fi
  OPTIM="-O2 -fp-model strict -ip -axAVX,SSE4.2 -align all "
  OPTIMGCM="$OPTIM $OPTPREC"
fi
##########################################################################


mkdir -p $MODEL
echo $MODEL
MODEL=`( cd $MODEL ; pwd )` # to get absolute path, if necessary



# Option -fendian=big is only to be used with ARPEGE1D.
# The -r8 should probably be avoided if running on 32 bit machines
# Option r8 is not mandatory and generates larger executables.
# It is however mandatory if using ARPEGE1D
# Better optimization options might be a better choice (e.g. -O3)


echo '################################################################'
if [ "$check_linux" = 1 ] ; then
echo   Check if required software is available
echo '################################################################'

#### Ehouarn: test if ksh and/or bash are available
use_shell="ksh" # default: use ksh
if [ "`which ksh`" = "" ] ; then
  echo "no ksh ... we will use bash"
  use_shell="bash"
  if [ "`which bash`" = "" ] ; then
    echo "ksh (or bash) needed!! Install it!"
  fi
fi


for logiciel in csh wget tar gzip make $compilo gcc ; do
if [ "`which $logiciel`" = "" ] ; then
echo You must first install $logiciel on your system
exit
fi
done

if [ $pclinux = 1 ] ; then
cd $MODEL
cat <<eod> tt.f90
print*,'coucou'
end
eod
$compilo tt.f90 -o a.out
./a.out >| tt
if [ "`cat tt | sed -e 's/ //g' `" != "coucou" ] ; then
echo problem installing with compiler $compilo ; exit ; fi
\rm tt a.out tt.f90
fi
fi

###########################################################################



if [ $getlmdzor = 1 ] ; then
echo '##########################################################'
echo  Download a slightly modified version of  LMDZ
echo '##########################################################'
cd $MODEL
wget http://www.lmd.jussieu.fr/~lmdz/DistribG95/modipsl.$version.tar.gz
gunzip modipsl.$version.tar.gz
tar xvf modipsl.$version.tar
\rm modipsl.$version.tar

# We download LMDZ and make some modifications to make it
#compatible with $compilo
# and we use an old stable but robust and well tested version of ORCHIDEE
# That version of ORCHIDEE can be obtained using
# wget http://www.lmd.jussieu.fr/~lmdz/DistribG95/getlmdzor.x
fi

echo OK1

if [ $netcdf = 1 ] ; then
echo '##########################################################'
echo Compiling the Netcdf library
echo '##########################################################'
cd $MODEL
wget http://www.lmd.jussieu.fr/~lmdz/DistribG95/netcdf-4.0.1.tar.gz
gunzip netcdf-4.0.1.tar.gz
tar xvf netcdf-4.0.1.tar
\rm -f netcdf-4.0.1.tar

cd netcdf-4.0.1

OPTIMNC=$OPTIM
if [ $compilo = g95 ] ; then
# Set the appropriate compilation options
   export FC=g95
   export F90=g95
   export F90FLAGS=" -cpp -ffree-form $OPTIMNC"
   export FFLAGS=" -cpp $OPTIMNC"
   export CPPFLAGS=-Df2cFortran
   export CC=gcc
   export CXX=g++
elif [ $compilo = gfortran ] ; then
   export FC=gfortran
   export F90=gfortran
   export F90FLAGS=" -ffree-form $OPTIMNC"
   export FFLAGS=" $OPTIMNC"
   export CPPFLAGS=
   export CC=gcc
   export CXX=g++
elif [ $compilo = pgf90 ] ; then
   export CPPFLAGS="-DNDEBUG -DpgiFortran"
   export CC=pgcc
   export CFLAGS="-Msignextend"
   export CXX=pgCC
   export CXXFLAGS="-Msignextend"
   export FC=pgf90
   export FFLAGS="$OPTIMNC"
   export F90=pgf90
   export F90FLAGS="$OPTIMNC"
elif [ $compilo = ifort ] ; then
   export CPP="icc -E"
   export F77=ifort
   export FFLAGS="-O2 -ip -fpic -mcmodel=large"
   export F90=ifort
   export FCFLAGS="-O2 -ip -fpic -mcmodel=large"
   export CC=icc
   export CFLAGS="-O2 -ip -fpic -mcmodel=large"
   export CXX=icpc
   export CXXFLAGS="-O2 -ip -fpic -mcmodel=large"
else
   echo unexpected compiler $compilo ; exit
fi

localdir=`pwd -P`
./configure --prefix=$localdir
make check
make install
fi # of if [ $netcdf = 1 ]

#=======================================================================================
echo OK2 ioipsl=$ioipsl
echo '##########################################################'
echo 'Installing MODIPSL, the installation package manager for the '
echo 'IPSL models and tools'
echo '##########################################################'

if [ $netcdf = 0 -o $netcdf = 1 ] ; then
ncdfdir=$MODEL/netcdf-4.0.1
else
ncdfdir=$netcdf
fi

if [ $ioipsl = 1 ] ; then
  cd $MODEL/modipsl
  \rm -r lib/*

  cd util

  if [ $compilo = pgf90 ] ; then 
    fmod='module '
  elif [ $compilo = g95 ] ; then
    fmod='fmod='
  elif [ $compilo = ifort ] ; then
    fmod='module '
  else # gfortran
    fmod='I '
  fi
  cp AA_make.gdef AA_make.orig
  F_C="$compilo -c " ; if [ $compilo = gfortran ] ; then F_C="$compilo -c -cpp " ; fi
  sed -e 's/^\#.*.g95.*.\#.*.$/\#/' AA_make.gdef > tmp
  sed -e "s:F_L = g95:F_L = $compilo:" -e "s:F_C = g95 -c:F_C = $F_C": \
  -e 's/g95.*.w_w.*.(F_D)/g95      w_w = '"$OPTIMGCM"'/' \
  -e 's:g95.*.NCDF_INC.*.$:g95      NCDF_INC= '"$ncdfdir"'/include:' \
  -e 's:g95.*.NCDF_LIB.*.$:g95      NCDF_LIB= -L'"$ncdfdir"'/lib -lnetcdf:' \
  -e "s:-fmod=:-$fmod:" -e 's/-fno-second-underscore//' \
  -e 's:#-Q- g95      M_K = gmake:#-Q- g95      M_K = make:' \
  tmp >| AA_make.gdef


  if [ "$use_shell" = "ksh" ] ; then
    if [ "$pclinux" = 1 ] ; then
       ./ins_make -t g95 # We use lines for g95 even for the other compilers
    fi
  else # bash
    sed -e s:/bin/ksh:/bin/bash:g ins_make > ins_make.bash
    chmod u=rwx ins_make.bash
    if [ "$pclinux" = 1 ] ; then
    ./ins_make.bash -t g95 # We use lines for g95 even for the other compilers
    else
    ./ins_make.bash
    fi
  fi # of if [ "$use_shell" = "ksh" ]

  echo '##########################################################'
  echo 'Compiling IOIPSL, the interface library with Netcdf'
  echo '##########################################################'

  cd $MODEL/modipsl/modeles/IOIPSL/src
  if [ "$use_shell" = "bash" ] ; then
    cp Makefile Makefile.ksh
    sed -e s:/bin/ksh:/bin/bash:g Makefile.ksh > Makefile
  fi
  if [ "$pclinux" = 1 ] ; then
    # Build IOIPSL modules and library
    if [ $compilo = g95 ] ; then
       cp restcom.f90 restcom.f90.orig
       sed -e 's:cmode=or(NF90_NOCLOBBER,NF90_64BIT_OFFSET):cmode=NF90_NOCLOBBER:' restcom.f90.orig > restcom.f90  
    fi
    make clean
    make
    if [ $compilo = gfortran ] ; then # copy module files to lib
      cp -f *.mod ../../../lib
    fi
    # Build IOIPSL tools (ie: "rebuild", if present)
    if [ -f $MODEL/modipsl/modeles/IOIPSL/tools/rebuild ] ; then
      cd $MODEL/modipsl/modeles/IOIPSL/tools
      # adapt Makefile & rebuild script if in bash
      if [ "$use_shell" = "bash" ] ; then
        cp Makefile Makefile.ksh
        sed -e s:/bin/ksh:/bin/bash:g Makefile.ksh > Makefile
        cp rebuild rebuild.ksh
        sed -e 's:/bin/ksh:/bin/bash:g' \
            -e 's:print -u2:echo:g' \
            -e 's:print:echo:g' rebuild.ksh > rebuild
      fi
      make clean
      make
    fi
  fi # of if [ "$pclinux" = 1 ] 

else # of if [ $ioipsl = 1 ]
  if [ ${hostname:0:5} = ada33 ] ; then
    cd $MODEL/modipsl
    cd util

    cp AA_make.gdef AA_make.orig
    sed -e 's/^\#.*.g95.*.\#.*.$/\#/' AA_make.gdef > tmp
    sed -e "s:F_L = g95:F_L = $compilo:" -e "s:F_C = g95 -c:F_C = $compilo -c": \
    -e 's/g95.*.w_w.*.(F_D)/g95      w_w = '"$OPTIMGCM"'/' \
    -e 's:g95.*.NCDF_INC.*.$:g95      NCDF_INC= -I/smplocal/pub/HDF5/1.8.9/seq/include -I/smplocal/pub/NetCDF/4.1.3/include:' \
    -e 's:g95.*.NCDF_LIB.*.$:g95      NCDF_LIB= -L/smplocal/pub/NetCDF/4.1.3/lib -lnetcdff -lnetcdf:' \
    -e "s:-fmod=:-$fmod:" -e 's/-fno-second-underscore//' \
    -e 's:#-Q- g95      M_K = gmake:#-Q- g95      M_K = make:' \
    tmp >| AA_make.gdef

    ./ins_make -t g95 # We use lines for g95 even for the other compilers

    # on Ada, IOIPSL is already installed in ~rpsl035/IOIPSL_PLUS
    # so link it to current settings
    cd $MODEL/modipsl/modeles/
    \rm -r -f IOIPSL
    ln -s ~rpsl035/IOIPSL_PLUS IOIPSL
    cd ..
    ln -s ~rpsl035/IOIPSL_PLUS/modipsl/bin/* bin/
    ln -s ~rpsl035/IOIPSL_PLUS/modipsl/lib/* lib/

  fi # of if [ ${hostname:0:5} = ada33 ]
fi # of if [ $ioipsl = 1 ]

#============================================================================
if [ "$veget" = 1 ] ; then
  echo '########################################################'
  echo 'Compiling ORCHIDEE, the continental surfaces model '
  echo '########################################################'
  cd $MODEL/modipsl/modeles/ORCHIDEE
  echo OKpwd ; pwd
  if [ -d src_parallel ] ; then
     liste_src="parallel parameters global stomate sechiba driver"
  else
     # Obsolete, for ORCHIDEE_beton only
     liste_src="parameters stomate sechiba "
     # A trick to compile ORCHIDEE depending on if we are using real*4 or real*8
     cd src_parameters ; \cp reqdprec.$real reqdprec.f90 ; cd ..
  fi
  echo liste_strc $liste_src

  for d in $liste_src ; do src_d=src_$d
      echo src_d $src_d
      echo ls ; ls
      if [ ! -d $src_d ] ; then echo Problem orchidee : no $src_d ; exit ; fi
      cd $src_d ; \rm -f *.mod make ; make clean
      make ; if [ $compilo = gfortran ] ; then cp -f *.mod ../../../lib ; fi
      cd ..
  done
fi # of if [ "$veget" = 1 ]

#============================================================================
# Ehouarn: it may be directory LMDZ4 or LMDZ5 depending on tar file...
if [ -d $MODEL/modipsl/modeles/LMDZ[45] ] ; then
  echo '##########################################################'
  echo 'Compiling LMDZ'
  echo '##########################################################'
  cd $MODEL/modipsl/modeles/LMDZ[45]
else
  echo "ERROR: No LMDZ4 (or LMDZ5) directory !!!"
  exit
fi

#============================================================================
# Traitement momentanne a cause d'un bug dans makegcm
cp create_make_gcm create_make_gcm.orig
nl=`sed -n -e /PROGRAM/= create_make_gcm.orig | tail -1`
sed -e "$nl s/      PROGRA/PROGRA/" create_make_gcm.orig >| create_make_gcm

#mv -f tmp crea
if [ "$pclinux" = 1 ] ; then
  if [ $compilo = gfortran ] ; then
sed \
-e 's:\#setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:\#setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's:setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's/set FC_LINUX.*.$/set FC_LINUX='$compilo'/' \
-e 's/g95/gfortran/g' \
-e 's/-fmod=/-I/g' \
-e 's/-fno-second-underscore//' -e 's/-fstatic//' \
-e 's/-lparallel//' \
-e 's/-lnetcdff//g' \
-e 's/-lorglob//' \
-e 's/-ffixed-form//' -e 's/-ffree-form//' \
-e 's/set OPT_LINUX.*.$/set OPT_LINUX=\"'"$OPTIMGCM"\"'/' makegcm.orig >| makegcm
  elif [ $compilo = ifort ] ; then
sed \
-e 's:\#setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:\#setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's:setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's/set FC_LINUX.*.$/set FC_LINUX='$compilo'/' \
-e 's/g95/ifort/g' \
-e 's/-fmod=/-module /g' \
-e 's/-fno-second-underscore//' -e 's/-fstatic//' \
-e 's/-lparallel//' \
-e 's/-lnetcdff//g' \
-e 's/-lorglob//' \
-e 's/-ffixed-form//' -e 's/-ffree-form//' \
-e 's/set OPT_LINUX.*.$/set OPT_LINUX=\"'"$OPTIMGCM"\"'/' makegcm.orig >| makegcm
  else # g95 or pgf90
sed \
-e 's:\#setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:\#setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's:setenv NCDFINC.*.$:setenv NCDFINC '"$ncdfdir"'/include:' \
-e 's:setenv NCDFLIB.*.$:setenv NCDFLIB '"$ncdfdir"'/lib:' \
-e 's/set FC_LINUX.*.$/set FC_LINUX='$compilo'/' \
-e 's/-fno-second-underscore//' -e 's/-fstatic//' \
-e 's/-lparallel//' \
-e 's/-lnetcdff//g' \
-e 's/-lorglob//' \
-e 's/-ffixed-form//' -e 's/-ffree-form//' \
-e 's/set OPT_LINUX.*.$/set OPT_LINUX=\"'"$OPTIMGCM"\"'/' makegcm.orig >| makegcm
  fi
else
sed \
-e 's/-lparallel//' \
-e 's/-lorglob//' \
-e 's/-lsxorglob//' \
-e 's/-lsxparallel//' \
-e 's/-lsxioipsl/-lioipsl/g' \
makegcm.orig >| makegcm
fi

chmod +x makegcm

###########################################################
# For those who want to use fcm to compile via :
#  makelmdz_fcm -arch local .....
############################################################

if [ "$pclinux" = 1 ] ; then
# create local 'arch' files (if on Linux PC):
cd arch
# arch-local.path file
echo "NETCDF_LIBDIR=\"-L${ncdfdir}/lib -lnetcdf\"" > arch-local.path
echo "NETCDF_INCDIR=-I${ncdfdir}/include" >> arch-local.path
echo 'IOIPSL_INCDIR=$LMDGCM/../../lib' >> arch-local.path
echo 'IOIPSL_LIBDIR=$LMDGCM/../../lib' >> arch-local.path
echo 'ORCH_INCDIR=$LMDGCM/../../lib' >> arch-local.path
echo 'ORCH_LIBDIR=$LMDGCM/../../lib' >> arch-local.path
# arch-local.fcm file (adapted from arch-linux-32bit.fcm)
if [ $compilo = g95 ] ; then
sed -e s:"%COMPILER            pgf95":"%COMPILER            g95":1 \
    -e s:"%LINK                pgf95":"%LINK                g95":1 \
    -e s:"%PROD_FFLAGS         -fast":"%PROD_FFLAGS         $OPTIM":1 \
    -e s:"%DEV_FFLAGS          -g -O1":"%DEV_FFLAGS          -g -O1 -Wall":1 \
    -e s:"%DEBUG_FFLAGS        -g -O0 -Kieee -Ktrap=fp -Mbounds":"%DEBUG_FFLAGS        -g -O0 -Wall -ftrace=full -fbounds-check -freal=nan":1 \
    -e s:"%BASE_FFLAGS         ":"%BASE_FFLAGS         $OPTPREC":1 \
   arch-linux-32bit.fcm > arch-local.fcm
   if [ $real = r8 ] ; then
     sed -e s:"%FPP_DEF             ":"%FPP_DEF             NC_DOUBLE":1 \
     arch-local.fcm > arch-local.fcm.new
     mv -f arch-local.fcm.new arch-local.fcm
   fi
elif [ $compilo = gfortran ] ; then
sed -e s:"%COMPILER            pgf95":"%COMPILER            gfortran":1 \
    -e s:"%LINK                pgf95":"%LINK                gfortran":1 \
    -e s:"%PROD_FFLAGS         -fast":"%PROD_FFLAGS         $OPTIM":1 \
    -e s:"%DEV_FFLAGS          -g -O1":"%DEV_FFLAGS          -Wall -fbounds-check":1 \
    -e s:"%DEBUG_FFLAGS        -g -O0 -Kieee -Ktrap=fp -Mbounds":"%DEBUG_FFLAGS        -g3 -Wall -fbounds-check -ffpe-trap=invalid,zero,overflow -O0 -fstack-protector-all":1 \
    -e s:"%BASE_FFLAGS         ":"%BASE_FFLAGS         $OPTPREC":1 \
   arch-linux-32bit.fcm > arch-local.fcm
   if [ $real = r8 ] ; then
     sed -e s:"%FPP_DEF             ":"%FPP_DEF             NC_DOUBLE":1 \
     arch-local.fcm > arch-local.fcm.new
     mv -f arch-local.fcm.new arch-local.fcm
   fi
elif [ $compilo = pgf90 ] ; then
sed -e s:"-Wl,-Bstatic -L/usr/lib/gcc-lib/i386-linux/2.95.2":" ":1 \
    -e s:"%PROD_FFLAGS         -fast":"%PROD_FFLAGS         $OPTIM":1 \
    -e s:"%BASE_FFLAGS         ":"%BASE_FFLAGS         $OPTPREC":1 \
   arch-linux-32bit.fcm > arch-local.fcm
   if [ $real = r8 ] ; then
     sed -e s:"%FPP_DEF             ":"%FPP_DEF             NC_DOUBLE":1 \
     arch-local.fcm > arch-local.fcm.new
     mv -f arch-local.fcm.new arch-local.fcm
   fi
elif [ $compilo = ifort ] ; then
sed -e s:"%COMPILER            pgf95":"%COMPILER            ifort":1 \
    -e s:"%LINK                pgf95":"%LINK                ifort":1 \
    -e s:"-Wl,-Bstatic -L/usr/lib/gcc-lib/i386-linux/2.95.2":" ":1 \
    -e s:"%PROD_FFLAGS         -fast":"%PROD_FFLAGS         $OPTIM":1 \
    -e s:"%BASE_FFLAGS         ":"%BASE_FFLAGS         $OPTPREC":1 \
    -e s:"%DEV_FFLAGS          -g -O1":"%DEV_FFLAGS          -p -g -O2 -traceback -fp-stack-check -ftrapuv -check":1 \
    -e s:"%DEBUG_FFLAGS        -g -O0 -Kieee -Ktrap=fp -Mbounds":"%DEBUG_FFLAGS        -g -no-ftz -traceback -ftrapuv -fp-stack-check -check":1 \
   arch-linux-32bit.fcm > arch-local.fcm
   if [ $real = r8 ] ; then
     sed -e s:"%FPP_DEF             ":"%FPP_DEF             NC_DOUBLE":1 \
     arch-local.fcm > arch-local.fcm.new
     mv -f arch-local.fcm.new arch-local.fcm
   fi
else
   echo Unexpected compiler $compilo ; exit
fi # of if [ $compilo = g95 ] elif [ $compilo = pgf90 ]
cd ..
### Adapt "bld.cfg" (add the shell):
whereisthatshell=$(which ${use_shell})
echo "bld::tool::SHELL   $whereisthatshell" >> bld.cfg

fi # of if [ "$pclinux" = 1 ]


cd $MODEL/modipsl/modeles/LMDZ?

### Modify makelmdz_fcm and makelmdz to use ORCHIDEE in the bench:
### remove liborglob.a and libparallel.a
cp makelmdz_fcm makelmdz_fcm.orig
sed -e "s/-l\${LIBPREFIX}parallel//g" \
sed -e "s/-l\${LIBPREFIX}orglob//g" \
    makelmdz_fcm.orig > makelmdz_fcm
cp makelmdz makelmdz.orig
sed -e "s/-l\${LIBPREFIX}parallel//g" \
sed -e "s/-l\${LIBPREFIX}orglob//g" \
    makelmdz.orig > makelmdz


##################################################################
# Compile LMDZ
##################################################################
ok_veget=false
if [ "$veget" = 1 ] ; then $ok_veget = true ; fi
if [ $compile_with_fcm = 1 ] ; then makelmdz=makelmdz_fcm ; else makelmdz=makelmdz ; fi
if [ "$pclinux" = 1 ] ; then
	./$makelmdz -d ${grid_resolution} -arch local -v $ok_veget gcm
else
   # we are on Ada
    ./$makelmdz -d ${grid_resolution} -arch X64_ADA -v $ok_veget gcm
fi

if [ -f gcm.e ] || [ -f bin/gcm_${grid_resolution}_phylmd_seq_orch.e ] || [ -f bin/gcm_${grid_resolution}_phylmd_seq.e ] ; then
echo '##########################################################'
echo 'Compilation successfull !! '
echo '##########################################################'
else
echo 'Compilation failed !!'
exit
fi

##################################################################
# Below, we run a benchmark test (if bench=0)
##################################################################
if [ $bench = 0 ] ; then
		exit
fi

echo '##########################################################'
echo ' Running a test run '
echo '##########################################################'

\rm -r BENCH${grid_resolution}
bench=bench_lmdz_${grid_resolution}
wget http://www.lmd.jussieu.fr/~lmdz/DistribG95/$bench.tar.gz
gunzip $bench.tar.gz
tar xvf $bench.tar

if [ -f gcm.e ] ; then 
    cp gcm.e BENCH${grid_resolution}/
elif [ -f bin/gcm_${grid_resolution}_phylmd_seq_orch.e ] ; then
    cp bin/gcm_${grid_resolution}_phylmd_seq_orch.e  BENCH${grid_resolution}/gcm.e
elif [ -f bin/gcm_${grid_resolution}_phylmd_seq.e ] ; then
    cp bin/gcm_${grid_resolution}_phylmd_seq.e  BENCH${grid_resolution}/gcm.e
else
    echo "No gcm.e found"
    exit
fi

cd BENCH${grid_resolution}
./bench.sh > bench.out  2>&1

echo '##########################################################'
echo ' Bench results '
echo '##########################################################'

cat ./bench.out

echo '##########################################################'
echo 'Simulation finished in' `pwd`
echo 'You may re-run it with : cd ' `pwd` ' ; gcm.e'
echo 'or ./bench.sh'
echo '##########################################################'
