MODULE radiative_sw

#include "use_logging.h"

  IMPLICIT NONE
  SAVE

  PRIVATE

  PUBLIC :: sw

CONTAINS

  PURE SUBROUTINE monGATHER(n,a,b,index)
    INTEGER, INTENT(IN) ::  n,index(n)
    REAL, INTENT(IN) :: b(n)
    REAL, INTENT(OUT) :: a(n)
    INTEGER :: i

    DO i=1,n
       a(i)=b(index(i))
    END DO
  END SUBROUTINE monGATHER

  PURE subroutine monscatter(ngrid, n,a,index,b)
    INTEGER, INTENT(IN) :: ngrid, n,index(n)
    REAL, INTENT(IN)    :: b(n)
    REAL, INTENT(OUT)   :: a(ngrid)
    INTEGER :: i
    IF(n<ngrid) THEN
       a(:)=0.
       DO i=1,n
          a(index(i))=b(i)
       END DO
    ELSE
       a(:)=b(:)
    END IF
  end subroutine monscatter

  SUBROUTINE sw(ngrid,nlayer,ldiurn, coefvis,albedo, &
       &        plevel,ps_rad,pmu,pfract,psolarf0, &
       &        fsrfvis,dtsw, lverbose, lwrite)
    USE phys_const, ONLY : cpp, g
    USE writefield_mod, ONLY : writefield

    !=======================================================================
    !
    !   Rayonnement solaire en atmosphere non diffusante avec un
    !   coefficient d absoprption gris.
    !
    !=======================================================================
    !
    !   declarations:
    !   -------------
    !
    !
    !   arguments:
    !   ----------
    !
    INTEGER, INTENT(IN) :: ngrid,nlayer
    LOGICAL, INTENT(IN) :: ldiurn, lverbose, lwrite
    REAL, INTENT(IN)    :: albedo(ngrid), coefvis
    REAL, INTENT(IN)    :: pmu(ngrid), pfract(ngrid)
    REAL, INTENT(IN)    :: plevel(ngrid,nlayer+1), ps_rad
    REAL, INTENT(IN)    :: psolarf0
    REAL, INTENT(OUT)   :: fsrfvis(ngrid),dtsw(ngrid,nlayer)

    REAL zalb(ngrid),zmu(ngrid),zfract(ngrid)
    REAL zplev(ngrid,nlayer+1)
    REAL zflux(ngrid),zdtsw(ngrid,nlayer)

    INTEGER ig,l,nlevel,index(ngrid),ncount,igout
    REAL ztrdir(ngrid,nlayer+1), & ! transmission coef for downward flux
         ztrref(ngrid,nlayer+1)    ! transmission coef for upward flux
    REAL zfsrfref(ngrid)           ! upward flux at surface
    REAL z1(ngrid)
    REAL zu(ngrid,nlayer+1)
    REAL tau0

    REAL :: flux_in(ngrid), &             ! incoming solar flux
         &  flux_ref(ngrid), &            ! flux reflected by surface
         &  flux_down(ngrid, nlayer+1), & ! downward flux
         &  flux_up(ngrid, nlayer+1)      ! upward flux
    REAL :: buf1(ngrid), buf2(ngrid, nlayer+1) ! buffers for output

    !-----------------------------------------------------------------------
    !   1. initialisations:
    !   -------------------

    nlevel=nlayer+1

    !-----------------------------------------------------------------------
    !   Definitions des tableaux locaux pour les points ensoleilles:
    !   ------------------------------------------------------------

    IF (ldiurn) THEN
       ncount=0
       DO ig=1,ngrid
          index(ig)=0
       ENDDO
       DO ig=1,ngrid
          IF(pfract(ig).GT.1.e-6) THEN
             ncount=ncount+1
             index(ncount)=ig
          ENDIF
       ENDDO
       CALL monGATHER(ncount,zfract,pfract,index)
       CALL monGATHER(ncount,zmu,pmu,index)
       CALL monGATHER(ncount,zalb,albedo,index)
       DO l=1,nlevel
          CALL monGATHER(ncount,zplev(1,l),plevel(1,l),index)
       ENDDO
    ELSE
       ncount=ngrid
       zfract(:)=pfract(:)
       zmu(:)=pmu(:)
       zalb(:)=albedo(:)
       zplev(:,:)=plevel(:,:)
    ENDIF

    !-----------------------------------------------------------------------
    !   calcul des profondeurs optiques integres depuis p=0:
    !   ----------------------------------------------------

    tau0=-.5*log(coefvis)

    ! calcul de la partie homogene de l opacite
    tau0=tau0/ps_rad
    DO l=1,nlayer+1
       DO ig=1,ncount
          zu(ig,l)=tau0*zplev(ig,l)
       ENDDO
    ENDDO

    !-----------------------------------------------------------------------
    !   2. calcul de la transmission depuis le sommet de l atmosphere:
    !   -----------------------------------------------------------

    DO ig=1,ncount
       flux_in(ig) = psolarf0*zfract(ig)*zmu(ig)
    ENDDO

    DO l=1,nlevel
       DO ig=1,ncount
          ztrdir(ig,l)=exp(-zu(ig,l)/zmu(ig)) ! transmission coefficient
          flux_down(ig,l) = flux_in(ig)*ztrdir(ig,l)
       ENDDO
    ENDDO

    IF (lverbose) THEN
       igout=ncount/2+1
       WRITELOG(*,*)
       WRITELOG(*,*) 'Diagnostique des transmission dans le spectre solaire'
       WRITELOG(*,*) 'zfract, zmu, zalb'
       WRITELOG(*,*) zfract(igout), zmu(igout), zalb(igout)
       WRITELOG(*,*) 'Pression, quantite d abs, transmission'
       DO l=1,nlayer+1
          WRITELOG(*,*) zplev(igout,l),zu(igout,l),ztrdir(igout,l)
       ENDDO
    ENDIF

    !-----------------------------------------------------------------------
    !   3. taux de chauffage, ray. solaire direct:
    !   ------------------------------------------

    DO l=1,nlayer
       DO ig=1,ncount
          ! m.cp.dT = dflux/dz
          ! m = -(dp/dz)/g
          ! flux = ztrdir * psolarf0*zfract*zmu
          IF(.FALSE.) THEN
             zdtsw(ig,l)=g*psolarf0*zfract(ig)*zmu(ig)       &
                  &     *(ztrdir(ig,l+1)-ztrdir(ig,l))     &
                  &     /(cpp*(zplev(ig,l)-zplev(ig,l+1)))
          ELSE
             zdtsw(ig,l)=(g/cpp) &
                  &     * (flux_down(ig,l+1)-flux_down(ig,l)) &
                  &     / (zplev(ig,l)-zplev(ig,l+1))
          END IF
       ENDDO
    ENDDO
    IF (lverbose) THEN
       WRITELOG(*,*)
       WRITELOG(*,*) 'Diagnostique des taux de chauffage solaires:'
       WRITELOG(*,*) ' 1 taux de chauffage lie au ray. solaire  direct'
       DO l=1,nlayer
          WRITELOG(*,*) zdtsw(igout,l)
       ENDDO
    ENDIF


    !-----------------------------------------------------------------------
    !   4. calcul du flux solaire arrivant sur le sol:
    !   ----------------------------------------------

    DO ig=1,ncount
       z1(ig)=zfract(ig)*zmu(ig)*psolarf0*ztrdir(ig,1) ! total (down)
       zflux(ig)=(1.-zalb(ig))*z1(ig)                  ! absorbed (net)
       zfsrfref(ig)=    zalb(ig)*z1(ig)                ! reflected (up)
    ENDDO
    IF (lverbose) THEN
       WRITELOG(*,*)
       WRITELOG(*,*) 'Diagnostique des taux de chauffage solaires:'
       WRITELOG(*,*) ' 2 flux solaire net incident sur le sol'
       WRITELOG(*,*) zflux(igout)
    ENDIF


    !-----------------------------------------------------------------------
    !   5.calcul des transmissions depuis le sol, cas diffus:
    !   ------------------------------------------------------

    DO l=1,nlevel
       DO ig=1,ncount
          ztrref(ig,l)=exp(-(zu(ig,1)-zu(ig,l))*1.66)
       ENDDO
    ENDDO

    IF (lverbose) THEN
       WRITELOG(*,*)
       WRITELOG(*,*) 'Diagnostique des taux de chauffage solaires'
       WRITELOG(*,*) ' 3 transmission avec les sol'
       WRITELOG(*,*) 'niveau     transmission'
       DO l=1,nlevel
          WRITELOG(*,*) l,ztrref(igout,l)
       ENDDO
    ENDIF

    !-----------------------------------------------------------------------
    !   6.ajout a l echauffement de la contribution du ray. sol. reflechit:
    !   -------------------------------------------------------------------

    DO l=1,nlayer
       DO ig=1,ncount
          zdtsw(ig,l)=zdtsw(ig,l)+ &
               g*zfsrfref(ig)*(ztrref(ig,l+1)-ztrref(ig,l))/ &
               (cpp*(zplev(ig,l+1)-zplev(ig,l)))
       ENDDO
    ENDDO

    !-----------------------------------------------------------------------
    !   10. sorties eventuelles:
    !   ------------------------

    IF (lverbose) THEN
       WRITELOG(*,*)
       WRITELOG(*,*) 'Diagnostique des taux de chauffage solaires:'
       WRITELOG(*,*) ' 3 taux de chauffage total'
       DO l=1,nlayer
          WRITELOG(*,*) zdtsw(igout,l)
       ENDDO
    ENDIF

    CALL monscatter(ngrid,ncount,fsrfvis,index,zflux)
    DO l=1,nlayer
       CALL monscatter(ngrid,ncount,dtsw(:,l),index,zdtsw(:,l))
    ENDDO

    IF(lwrite) THEN
       CALL monscatter(ngrid, ncount,buf1,index,flux_in)
       CALL writefield('swtop','SW down TOA','W/m2',buf1)

       DO l=1,nlayer+1
          CALL monscatter(ngrid,ncount,buf2(:,l),index,flux_down(:,l))
       ENDDO
       CALL writefield('swflux_down','Downwards SW flux','W/m2',buf2)

    END IF

    LOG_INFO('rad_sw')

  END SUBROUTINE sw

END MODULE radiative_sw
